#include <xvinclude.h>

/*
 * This example creates a display containing an image, and underneath it,
 * a colorcell object and a pseudocolor object.  All three objects are 
 * initialized to use the moon image.
 *
 * Then, an event handler is added on the image for button press events,
 * so that the user may click on the image to select a pixel, and then
 * change the color of that pixel using the pseudocolor scrollbars;  the
 * colorcell object will reflect the pixel value and it's current color.
 *
 * Thus, you can set the colors of pixels in the image one by one.
 */

void  image_handler   PROTO((xvobject, kaddr, XEvent *, Boolean *));

typedef struct _ColorObjects {

    xvobject pseudo; 
    xvobject colorcell; 

    } ColorObjects;

void main(
    int  argc,
    char **argv,
    char **envp)
{
	char         *filename = "image:moon";
	xvobject     image, pseudo, colorcell, parent;
	ColorObjects *colorobjs;

        /* initialize Khoros program */
        khoros_initialize(argc, argv, envp, "ENVISION");

	/* initialize the xvwidgets lib */
        if (!xvw_initialize(XVW_MENUS_XVFORMS))
        {
           kerror(NULL, "main", "unable to open display");
           kexit(KEXIT_FAILURE);
        }

	/* create a common parent for image, colorcell, & pseudocolor objects */
        parent = xvw_create_manager(NULL, "manager");

	/* create an image object to display "image:moon" */
	image = xvw_create_image(parent, "image");
	xvw_set_attribute(image, XVW_IMAGE_IMAGEFILE, filename);

	/* create colorcell object below image using info from "image:moon" */
	colorcell = xvw_create_colorcell(parent, "colorcell");
	xvw_set_attributes(colorcell, 
		           XVW_COLOR_COLORFILE, filename,
		           XVW_WIDTH,           30,
		           XVW_HEIGHT,          30,
			   XVW_BELOW,           image,
			   NULL);

	/*
	 * create pseudocolor object to the right of the colorcell
         * object; don't need the palette this time, so turn it off.
	 */
        pseudo = xvw_create_pseudo(parent, "pseudo");
        xvw_set_attributes(pseudo,
		           XVW_COLOR_COLORFILE,    "image:moon",
		           XVW_RIGHT_OF,	    colorcell,
	                   XVW_BELOW,               image,
			   XVW_PSEUDO_SHOW_PALETTE, FALSE,
		           NULL);
	
	/* initialize client_data struct with pseudo & colorcell objects */
	colorobjs = (ColorObjects *) kcalloc(1, sizeof (ColorObjects));
	colorobjs->pseudo = pseudo;	
	colorobjs->colorcell = colorcell;	

	/* add event handler to image object on button press */
	xvw_add_event(image, ButtonPressMask, image_handler, colorobjs);

	/* display & run */
        xvf_run_form();
}

/*
 *  event handler to update the colors in the image according to
 *  the latest setting of the pseudo scrollbars.
 */
void  image_handler(
    xvobject image,
    kaddr    client_data,
    XEvent   *event,
    Boolean  *dispatch)
{
	double indx;
	int red, green, blue;
	ColorObjects *colorobjs = (ColorObjects *) client_data;

	/*
	 *  get the value of the pixel that the user clicked on with the mouse
	 */
	xvw_get_attribute(image, XVW_IMAGE_VALUE, &indx);

	/*
	 *  clear the pseudocolor list, so that the next time the 
         *  scrollbars are used, only this pixel value will be changed.
	 */
	xvw_set_attribute(colorobjs->pseudo, 
			  XVW_PSEUDO_CLEAR, TRUE);

	/*
	 *  set the colorcell object to reflect the value of this pixel
	 */
	xvw_set_attribute(colorobjs->colorcell, 
			  XVW_COLORCELL_INDEX, (int) indx);

	/*
	 * add this pixel value to the pseudocolor list, so that chaning
         * the pseudo scrollbars will change the color of this pixel.
	 */
	xvw_set_attribute(colorobjs->pseudo, XVW_PSEUDO_ADD, (int) indx);

	/*
	 *  make this pixel value initially red, so it's real obvious
         *  that it got selected.
	 */
	xvw_set_attributes(colorobjs->pseudo,
		XVW_PSEUDO_REDVAL,    255,
		XVW_PSEUDO_GREENVAL,  0,
		XVW_PSEUDO_BLUEVAL,   0,
		NULL);
}
