#include <xvinclude.h>

/*
 * This example demonstrates how the colorcell object can be used to
 * interactively change colors of pixels in an image.
 *
 * An image is created, with "restore" and "quit" buttons to the upper left,
 * and a colorcell object to the upper right.  
 *
 * As the user moves the pointer across the image, an event handler gets the 
 * value of the pixel at that point and adds it to the list of indexes used 
 * by the colorcell (the value of the last pixel is deleted, so that the 
 * colorcell list contains only one index at any given time).  The RGB values 
 * being displayed for that pixel are then obtained from the colorcell, and 
 * are reversed with a simple (255 - x) algorithm.  The new RGB values are 
 * then set on the colorcell.  This procedure has the effect of "highlighting"
 * all the pixels in the image that have the same value as the pixel under the 
 * pointer.
 *
 * The "restore" button restores the original colors of the image, and
 * the "quit" button quits the program.
 */

void  set_colorcell	PROTO((xvobject, kaddr, XEvent *, int *));
void  restore_colorcell	PROTO((xvobject, kaddr, kaddr));
void  quit_cb          	PROTO((xvobject, kaddr, kaddr));
void  destroy_cb       	PROTO((xvobject, kaddr, kaddr));

xvobject manager, image, colorcell, restore, quit;

void main(
     int  argc,
     char **argv,
     char **envp)
{
	char *filename = "image:kitten";

        /* initialize Khoros program */
        khoros_initialize(argc, argv, envp, "ENVISION");

        /* initialize the xvwidgets lib */
	if (!xvw_initialize(XVW_MENUS_XVFORMS))
	{
	   kerror(NULL, "main", "unable to open display");
	   kexit(KEXIT_FAILURE);
	}

        /* Create a manager to parent the image and colorcell objects  */
	manager = xvw_create_manager(NULL, "manager");

	/*
	 *  create the colorcell object, and associate it with the image.
	 *  set XVW_COLORCELL_RESTORE_ONDELETE to TRUE, so that when a pixel
	 *  is deleted from the colorcell list, that pixel will revert to
	 *  its original color.
	 */
	colorcell = xvw_create_colorcell(manager, "colorcell");
	xvw_set_attributes(colorcell,
		XVW_COLOR_COLORFILE, filename,
		XVW_BELOW,	     NULL,
		XVW_LEFT_OF,	     NULL,
		XVW_WIDTH,           35,
		XVW_HEIGHT,          35,
		XVW_COLORCELL_RESTORE_ONDELETE, TRUE,
		NULL);

	/* create an image object to display the image */
	image = xvw_create_image(manager, "image");
	xvw_set_attributes(image,
		XVW_IMAGE_IMAGEFILE, filename,
		XVW_BELOW,	     colorcell,
		NULL);

	/* add event handler to reverse RGB values of pixel under cursor */
	xvw_add_event(image, PointerMotionMask, set_colorcell, colorcell);

	/* add a restore button to restore colors of pixels */
	restore = xvw_create_button(manager, "restore");
	xvw_set_attributes(restore, 
			   XVW_LABEL,   "restore",
			   XVW_RIGHT_OF, NULL,
			   NULL);
	xvw_add_callback(restore, XVW_BUTTON_SELECT,
		         restore_colorcell, colorcell);

	/* add a quit button */
        quit = xvw_create_button(manager, "quit");
        xvw_set_attributes(quit,
                           XVW_LABEL,   "quit",
                           XVW_RIGHT_OF, restore,
                           NULL);

        xvw_add_callback(quit, XVW_BUTTON_SELECT,
                         quit_cb, manager);

#if 0
{
	xvobject destroy;
        destroy = xvw_create_button(manager, "destroy");
        xvw_set_attributes(destroy,
                           XVW_LABEL,   "destroy",
                           XVW_RIGHT_OF, quit,
                           NULL);

        xvw_add_callback(destroy, XVW_BUTTON_SELECT,
                         destroy_cb, NULL);
}
#endif

	/* display & run */
	xvf_run_form();
}


/*
 *  event handler to highlight pixels in the image by 
 *  reversing the colorcell RGB values 
 */
void  set_colorcell(
    xvobject object, 
    kaddr    client_data,  
    XEvent   *event,
    int      *dispatch)
{
	int      new_pixel, redval, greenval, blueval;
	double   tmp_double;
	xvobject colorcell = (xvobject) client_data;
	static   int last_pixel = -1;

	/*
         *  get value of pixel under cursor
         */
        xvw_get_attribute(object, XVW_IMAGE_VALUE, &tmp_double);
	new_pixel = (int) tmp_double;
	if (last_pixel == new_pixel) return;

	/*
	 * disassociate colorcell with last pixel by deleting 
	 * the last index from the colorcell list
	 */
	xvw_set_attribute(colorcell, XVW_COLORCELL_DELETE, last_pixel);
	
	/*
	 * associate colorcell with the new pixel by adding
	 * the new index to the colorcell list
	 */
	xvw_set_attribute(colorcell, XVW_COLORCELL_ADD, new_pixel);

	/*
         *  get original RGB values for new pixel by inquiring the colorcell
         */
        xvw_get_attributes(colorcell,
                           XVW_COLORCELL_REDVAL,   &redval,
                           XVW_COLORCELL_GREENVAL, &greenval,
                           XVW_COLORCELL_BLUEVAL,  &blueval,
                           NULL);
	/*
	 * invert the original RGB values
	 */
        redval   = 255 - redval;
        greenval = 255 - greenval;
        blueval  = 255 - blueval;
	
	/*
	 *  set new "highlighted" RGB values for pixel
	 */
	xvw_set_attributes(colorcell,
                           XVW_COLORCELL_REDVAL,   redval,
                           XVW_COLORCELL_GREENVAL, greenval,
                           XVW_COLORCELL_BLUEVAL,  blueval,
                           NULL);
	/*
	 * the new pixel will be the last pixel next time around
	 */
	last_pixel = new_pixel;
}

/*
 *  callback to restore colorcell
 */
void  restore_colorcell(
   xvobject object,
   kaddr    client_data,
   kaddr    call_data)
{
	xvobject colorcell = (xvobject) client_data;

	/* 
	 * restore image to its original colors
         */
	xvw_set_attribute(colorcell, XVW_COLORCELL_RESTORE, TRUE);

}


/*
 *  callback to destroy colorcell
 */
void  destroy_cb(
   xvobject object,
   kaddr    client_data,
   kaddr    call_data)
{
	xvw_destroy(colorcell);
	xvw_destroy(image);
	xvw_destroy(restore);
}

/*
 *  callback to quit program
 */
void  quit_cb(
   xvobject object,
   kaddr    client_data,
   kaddr    call_data)
{
        xvobject manager = (xvobject) client_data;
	xvw_destroy(manager);
}
