#include <xvinclude.h>

/*
 * This program creates a window with an area object, containing a
 * 2D axis object and an indicator object.  The indicator object displays 
 * its world coordinate position within the area object. The internal menuform 
 * for the indicator is displayed automatically, and may be used to set 
 * attributes of the indicator.
 *
 * An event handler is installed which allows you to interactively
 * move the indicator by "grabbing" it with the left mouse button and moving
 * it around inside the area object; it will update the world coordinate value
 * that it displays as it is moved.  The 2D axis object serves to provide a
 * reference as to the world coordinate system.
 */
void move_indicator PROTO((xvobject, kaddr, XEvent *, int *));

void main(
   int  argc,
   char *argv[],
   char *envp[])
{
        xvobject parent, area, axis, indicator;

	/* initialize khoros program */
	khoros_initialize(argc, argv, envp, "ENVISION");

	/* initialize the xvwidgets lib */
        if (!xvw_initialize(XVW_MENUS_XVFORMS))
        {
           kerror(NULL, "main", "unable to open display");
           kexit(KEXIT_FAILURE);
        }
	
	/*
	 *  create a manager backplane, 300x300 (pixels)
	 */
	parent = xvw_create_manager(NULL, "parent");
        xvw_set_attributes(parent,
		           XVW_WIDTH,      300,  
		           XVW_HEIGHT,     300, 
		           NULL);

        /*
         * create an area object to provide a coordinated WC view
	 * for the 2D axis object and the indicator object
         */
        area = xvw_create_area(parent, "area");
        xvw_set_attributes(area,
                     XVW_AREA_DISPLAY_DATE,     FALSE,
                     XVW_AREA_DISPLAY_TITLE,    TRUE,
                     NULL );

        /*
         *  the 2D axis system will display the world coordinates
         */
        axis = xvw_create_axis2d(area, "axis coordinated");
        xvw_set_attributes(axis,
                     XVW_AREA_ATTACH,             axis,
                     XVW_GRAPHICS_VIEWPORT_MIN_X, 0.1,
                     XVW_GRAPHICS_VIEWPORT_MIN_Y, 0.1,
                     XVW_GRAPHICS_VIEWPORT_MAX_X, 0.9,
                     XVW_GRAPHICS_VIEWPORT_MAX_Y, 0.9,
                     NULL);

	/*
	 *  create the indicator object;  set the X & Y position values, 
         *  and set the foreground color. set menuable to TRUE so user can
         *  bring up internal menuform and set attributes of indicator.
	 */
        indicator = xvw_create_indicator(area, "indicator");
        xvw_set_attributes(indicator,
                XVW_AREA_ATTACH,          axis,
		XVW_INDICATOR_XPOS_VALUE, 0.5, 
		XVW_INDICATOR_YPOS_VALUE, 0.5, 
	        XVW_MENUABLE,             TRUE,
	        XVW_FOREGROUND_COLOR,     "pink",
		NULL);

	/* display internal menuform for indicator */
	xvw_activate_menu(indicator);


	/* add event handler to area object to allow user to move indicator */
	xvw_add_event(area, ButtonMotionMask, move_indicator, indicator);

	/* display and run the program */
        xvf_run_form();
}

/*
 *  This is the event handler which moves the indicator on ButtonMotion events.
 */
void move_indicator(
    xvobject object,
    kaddr    client_data,
    XEvent   *event,
    int      *dispatch)
{
	int      id, dcx, dcy;
	Coord    coord;
	xvobject indicator = (xvobject) client_data;

	/*
	 * event structure holds the (x,y) position of mouse in device coords.
	 */
	dcx = event->xbutton.x;
	dcy = event->xbutton.y;

	/*
	 * convert device coordinates of screen into the
	 * world coordinates that are used by the indicator
	 */
	id = (int) (xvw_widget(indicator));
	X2D_convert_point_dc_to_wc(id, (Real) dcx, (Real) dcy, &coord);

	/*
	 * don't let user move indicator out of bounds
	 */
	if ((coord.x < 0.0) || (coord.x > 1.0) || 
	    (coord.y < 0.0) || (coord.y > 1.0))
	    return;

	/*
	 * move indicator to new world coordinate position
	 */
	xvw_set_attributes(indicator, 
		XVW_INDICATOR_XPOS_VALUE, coord.x,
                XVW_INDICATOR_YPOS_VALUE, coord.y,
                NULL);

}
