 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>       	Generalized Routines
   >>>>
   >>>>  Private:
   >>>>   Static:
   >>>> 	  destroy_object() 
   >>>> 	  realize_parent() 
   >>>> 	  manage_child() 
   >>>>   Public:
   >>>>           xvw_widget()
   >>>>           xvw_window()
   >>>>           xvw_rootwindow()
   >>>>           xvw_toplevel()
   >>>>           xvw_visual()
   >>>>           xvw_colormap()
   >>>>           xvw_object()
   >>>>           xvw_name()
   >>>>           xvw_create()
   >>>>           xvw_recreate()
   >>>>           xvw_parent()
   >>>>           xvw_destroy()
   >>>>           xvw_place()
   >>>>           xvw_duplicate()
   >>>>           xvw_unmap()
   >>>>           xvw_map()
   >>>>           xvw_manage()
   >>>>           xvw_unmanage()
   >>>>           xvw_sensitive()
   >>>>           xvw_refresh()
   >>>>           xvw_geometry()
   >>>>           xvw_children()
   >>>>           xvw_numchildren()
   >>>>           xvw_translate_coords()
   >>>>           xvw_font()
   >>>>           xvw_fontname()
   >>>>           xvw_sort()
   >>>>
   >>>>           xvw_check_realized()
   >>>>           xvw_check_sensitive()
   >>>>           xvw_check_mapped()
   >>>>           xvw_check_visible()
   >>>>           xvw_check_managed()
   >>>>           xvw_check_menuactive()
   >>>>           xvw_check_menuexist()
   >>>>           xvw_check_toplevel()
   >>>>           xvw_check_localhost()
   >>>>           xvw_check_subclass()
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<  */

#include "internals.h"
#include <xvobjects/ManagerP.h>
#include <xvobjects/ManagerObjP.h>


static klist *object_list = NULL;


/*-----------------------------------------------------------
|
|  Routine Name: destroy_object
|
|       Purpose: This routine will be called when a child is being deleted
|
|         Input: object     - not used
|                client_data - the child to be destroyed
|                callData   - not used
|
|        Output:
|	Returns:
|
|    Written By: Mark Young
|          Date: Jun 15, 1992 9:13
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void destroy_object(
   Widget    widget,
   XtPointer client_data,
   XtPointer callData)
{
	klist    *entry;
	xvobject object;

	if (!widget)
	   return;

	if ((entry = klist_locate(object_list, (kaddr) widget)) != NULL)
	{
	   kinfo(KDEBUG, "Destroy object '%s' ... ", XtName(widget));
	   if ((object = (xvobject) entry->client_data) != NULL)
	   {
	      if (xvw_check_menuactive(object))
	         xvw_inactivate_menu(object);

	      xvw_call_callback(object, XVW_DESTROY);
	      xvw_destroy_events(object);
	      xvw_destroy_callbacks(object);
	      xvw_destroy_timeouts(object);
/*
	      xvw_destroy_actions(object);
	      xvw_destroy_detectfiles(object);
	      xvw_destroy_detectfids(object);
	      kfree(object);
 */
	   }
	   object_list = klist_delete(object_list, (kaddr) widget);
	   kinfo(KDEBUG, "done\n");
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: realize_parent
|
|       Purpose: This routine will realize a object's parent. 
|
|         Input: client_data - the parent to be realized
|                id         - not used
|
|        Output:
|	Returns:
|
|    Written By: Mark Young
|          Date: Jun 15, 1992 9:13
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void realize_parent(
   XtPointer    client_data,
   XtIntervalId *id)
{
        xvobject parent = (xvobject) client_data;

	int  initialize_state;

        if (!parent)
	   return;

	xvw_get_attribute(parent, XVW_SHELL_INITIAL_STATE, &initialize_state);
	if (initialize_state == NormalState)
           XtRealizeWidget(xvw_widget(parent));
}

/*-----------------------------------------------------------
|
|  Routine Name: manage_child
|
|       Purpose: This routine will be called to manage a child.
|
|         Input: client_data - the child to be managed
|                id         - not used
|
|        Output:
|	Returns:
|
|    Written By: Mark Young
|          Date: Jun 15, 1992 9:13
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void manage_child(
   XtPointer    client_data,
   XtIntervalId *id)
{
        xvobject child = (xvobject) client_data;

        if (!klist_locate_clientdata(object_list, (kaddr) child) ||
	    !child || !xvw_widget(child) || xvw_check_managed(child))
	{
	   return;
	}
        xvw_manage(child);
}


/************************************************************
*
*  Routine Name: xvw_widget - get the widget (or gadget) associated with an 
*		              object
*
*       Purpose: Returns the widget (or gadget) associated with
*		 the specified GUI or visual object.
*
*         Input: object - object for which to return the widget (or gadget)
*
*        Output:
*	Returns: The widget (or gadget) associated with the GUI or visual object
*                on success, NULL on failure
*  Restrictions:
*    Written By: Mark Young
*          Date: Nov 13, 1992
*      Verified: 
*  Side Effects: 
* Modifications:
*
*******************************************************************/

Widget xvw_widget(
   xvobject object)
{
	if (object && (!widgetdefined(object) || XtIsObject((Widget) object)))
	  return((Widget) object);

	return(!object ? NULL : object->widget);
}

/************************************************************
*
*  Routine Name: xvw_window - get the window associated with an object
*
*       Purpose: Returns the window associated with a GUI or visual object.
*
*         Input: object - object to get the window
*
*        Output:
*	Returns: The window associated with the object or gadget
*                on success, NULL on failure
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Jun 15, 1992
*      Verified: 
*  Side Effects: 
* Modifications:
*
*******************************************************************/

Window xvw_window(
   xvobject object)
{
	Widget widget = xvw_widget(object);

	if (!widget)
	   return(NONE);

	do
	{
	   if (XtIsWidget(widget))
	      return(XtWindow(widget));
	} while ((widget = XtParent(widget)) != NULL);

	return(NONE);
}

/************************************************************
*
*  Routine Name: xvw_rootwindow - get the root window associated with
*				  an object
*
*       Purpose: Returns the root window associated with the display
*		 on which the GUI or visual object appears.  If the object 
*		 is passed in as NULL, then it returns the rootwindow of 
*		 the default display. 
*
*         Input: object - object to get the rootwindow from
*        Output:
*	Returns: The rootwindow associated with the object
*                on success, NULL on failure
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Jun 15, 1992
*      Verified: 
*  Side Effects:
* Modifications:
*
*******************************************************************/

Window xvw_rootwindow(
   xvobject object)
{
	Display *display  = xvw_display(object);
	int     screennum = xvw_screennum(object);

	return(RootWindow(display, screennum));
}

/************************************************************
*
*  Routine Name: xvw_toplevel - get the toplevel object of an object
*
*       Purpose: Returns the toplevel object associated with a particular 
*		 with an object. 
*
*         Input: object - object for which to retrieve the toplevel
*        Output:
*	Returns: The toplevel associated with the object
*                on success, NULL on failure
*  Restrictions:
*    Written By: Mark Young
*          Date: Jun 15, 1992
*      Verified: 
*  Side Effects:
* Modifications:
*
*******************************************************************/

xvobject xvw_toplevel(
   xvobject object)
{
	Widget widget = xvw_widget(object);

	if (!widget)
	   return(NULL);

	while (XtParent(widget) != NULL)
	   widget = XtParent(widget);

	return(xvw_object(widget));
}

/************************************************************
*
*  Routine Name: xvw_visual - get the visual associated with
*			      an object
*
*       Purpose: Returns the visual associated with the object.
*		 If the object is NULL then it returns the visual
*		 associated with the default display. 
*
*         Input: object - object to get the visual from
*        Output:
*	Returns: The default visual associated with the object or gadget
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Jun 15, 1992
*      Verified: 
*  Side Effects:
* Modifications:
*
*******************************************************************/

Visual *xvw_visual(
   xvobject object)
{
	Display *display  = xvw_display(object);
	int     screennum = xvw_screennum(object);

	return(DefaultVisual(display, screennum));
}

/************************************************************
*
*  Routine Name: xvw_colormap - get the colormap associated with
*                               a object
*
*       Purpose: Returns the colormap associated with the object.
*		 If the object is passed as NULL, it will return 
*		 the colormap associated with the default display. 
*
*         Input: object - object to for which get the colormap; pass
*			  NULL to get the default colormap.
*        Output:
*	Returns: The colormap associated with the object (or the 
*	         default colormap) on success,  NULL on failure
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Jun 15, 1992
*      Verified: 
*  Side Effects:
* Modifications:
*
*******************************************************************/

Colormap xvw_colormap(
   xvobject object)
{
	Display *display  = xvw_display(object);
	int     screennum = xvw_screennum(object);

	return(DefaultColormap(display, screennum));
}

/************************************************************
*
*  Routine Name: xvw_name - get the name of the object
*
*       Purpose: Returns the name associated with the object.
*
*         Input: object - object for which to get the name
*
*        Output:
*	Returns: The name associated with the object
*                on success, NULL on failure
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Feb 16, 1993
*      Verified: 
*  Side Effects: 
* Modifications:
*
*******************************************************************/

char *xvw_name(
   xvobject object)
{
	Widget widget = xvw_widget(object);

	return(!widget ? NULL : XtName(widget));
}

/************************************************************
*
*  Routine Name: xvw_class - get the class of the object
*
*       Purpose: Returns the class associated with the object.
*
*         Input: object - object for which return the class
*
*        Output:
*	Returns: The class associated with the object
*                on success, NULL on failure
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Feb 16, 1993
*      Verified: 
*  Side Effects:
* Modifications:
*
*******************************************************************/

xvclass xvw_class(
   xvobject object)
{
	Widget widget = xvw_widget(object);

	return(!widget ? NULL : XtClass(widget));
}

/************************************************************
*
*  Routine Name: xvw_object - get the object associated with a particular widget
*
*       Purpose: Returns the GUI or visual object associated with 
*		 the specified widget.  If no object is associated with
*		 the widget, then an object is created, the widget is assigned
*		 to the new object.
*
*         Input: widget - widget for which to return the GUI or visual object
*
*        Output:
*	Returns: The object associated with the widget
*                on success, NULL on failure
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Feb 16, 1993
*      Verified: 
*  Side Effects: 
* Modifications:
*
*******************************************************************/

xvobject xvw_object(
   Widget widget)
{
	klist    *entry;
	xvobject object = NULL;

	if (!widget || !widgetdefined(widget))
	   return(NULL);

	if ((entry = klist_locate(object_list, widget)) == NULL &&
	    (xvw_widget((xvobject) widget) == widget)		&&
	    (widget->core.being_destroyed  == FALSE))
	{
	   if ((object = (xvobject) kcalloc(1,
			sizeof(struct _xvobject))) == NULL)
	   {
              kerror("xvwidgets", "xvw_object",
                     "Could not allocate xvobject structure for widget '%s'.",
		     XtName(widget));
              return(NULL);
	   }
	   object_list = klist_add(object_list, widget, object);
	   object->widget = widget;
	   XtAddCallback(widget, XtNdestroyCallback, destroy_object, NULL);
	}
	else if (entry)
	   object = (xvobject) entry->client_data;

	return(object);
}

/************************************************************
*
*  Routine Name: xvw_create - create a new object
*
*       Purpose: Creates and returns a new GUI or visual object 
*		 of the desired object class. Note that use of this 
*		 generalized routine is not as encouraged as 
*		 use of the appropriate specific object creation routine.
*		 For example, if you would like to create a button object, 
*		 it is generally easier to use \fIxvw_create_button()\fP.
*
*         Input: parent      - the parent object; pass NULL if a default
*		               toplevel object is to be created automatically
*		               to contain the object.
*		 transient_parent - only used if the parent object passed as
*	                       NULL, TRUE will cause the default toplevel to 
*			       be created as a transient shell while FALSE will
*			       cause the default toplevel to be created as
*			       an application shell.
*		 managed     - TRUE if the object should be managed upon 
*			       creation, FALSE if the object should only be
*			       instaniated
*		 name	     - the object name
*		 class       - the class of the object
*
*        Output:
*	Returns: The newly created GUI or visual object on success,
*		 NULL on failure
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Jun 15, 1992
*      Verified: 
*  Side Effects: 
* Modifications:
*
*******************************************************************/

xvobject xvw_create(
   xvobject    parent,
   int	       transient_parent,
   int	       managed,
   kstring     name,
   xvclass     class)
{
	xvclass      tmp;
	XtAppContext context;
	Widget       widget, pwidget;
	xvobject     object, toplevel = NULL;
	Boolean      realize_toplevel = FALSE;


        if (parent == NULL)
        {
           /*
            *  Create the toplevel application shell object if a parent
            *  is not provided.
            */
	   realize_toplevel = TRUE;
	   if (transient_parent == TRUE)
              toplevel = xvw_create_transient_shell(name, NULL, NULL);
	   else
              toplevel = xvw_create_application_shell(name, NULL, NULL);

	   for (tmp = class; tmp != NULL; tmp = tmp->core_class.superclass)
	      if (tmp == ManagerWidgetClass) break;

	   if (tmp != ManagerWidgetClass)
	   {
	      parent = xvw_create(toplevel, FALSE, TRUE, "object_back",
				  ManagerWidgetClass);
	      xvw_set_attributes(parent, XVW_WIDTH, 512, XVW_HEIGHT, 512, NULL);
	   }
	   else parent = toplevel;
        }

        /*
         *  Create the desired object now that we are sure that a parent
	 *  exists in which it will be placed.
         */
	pwidget = xvw_widget(parent);
	context = xvw_appcontext(parent);
	if (managed == TRUE && (XtIsComposite(pwidget) || XtIsShell(pwidget)))
	{
	   if (class != InputOnlyWidgetClass && xvw_check_realized(parent))
	   {
	      widget = XtCreateWidget(name, class, pwidget, NULL, 0);
	      object = xvw_object(widget);
	      XtAppAddTimeOut(context, 0, manage_child, (XtPointer) object);
	   }
	   else
	      widget = XtCreateManagedWidget(name, class, pwidget, NULL, 0);
	}
	else
	{
	   widget = XtCreateWidget(name, class, pwidget, NULL, 0);
	   if (managed == TRUE && xvw_check_realized(parent))
	      xvw_realize(xvw_object(widget));
	}

	/*
	 *  Realize the parent if we were the ones that created it.
	 */
        if (realize_toplevel == TRUE)
        {
	   XtAppAddTimeOut(context, 0, realize_parent, (XtPointer) toplevel);
	   object = xvw_object(widget);
	   object->created_toplevel = TRUE;
        }
	return(xvw_object(widget));
}

/************************************************************
*
*  Routine Name: xvw_recreate - re-create an object
*
*       Purpose: Re-creates the specified GUI or visual object 
*		 using the specified object class.  All the attributes 
*		 of the original object are copied and used as
*		 initial values for the new object.
*
*         Input: object  - the object to be recreated; cannot be NULL
*		 class   - the class of object to be created
*        Output:
*	Returns: TRUE (1) on success, FALSE (0) on failure
*
*  Restrictions:
*    Written By: Mark Young and John Salas
*          Date: Feb 23, 1993
*      Verified: 
*  Side Effects: 
* Modifications:
*
*******************************************************************/

int xvw_recreate(
   xvobject    object,
   xvclass class)
{
	Widget widget = xvw_widget(object);
	char   *name = xvw_name(object);

	int num;
	Arg args[KLENGTH];
	xvobject new, parent = xvw_parent(object);


	/*
	 *  Sanity check
	 */
	if (!object || !widget)
	   return(FALSE);

	/*
	 *  Copy the object's attributes then recreate it with using the
	 *  specified widget class and the old widget class.
	 */
	xvw_copy_attributes(object, args, &num);
	new = xvw_create(parent, FALSE, TRUE, kstrdup(name), class);
	if (new == NULL)
           return(FALSE);
 
	/*
	 *  Gotta delete the widget from the list, otherwise our object 
	 *  is gonna be freed.  The object gets freed when the widget
	 *  is destroyed.
	 */
	object_list = klist_delete(object_list, widget);
	XtUnmanageChild(widget);
	XtDestroyWidget(widget);

	widget = xvw_widget(new); kfree(new);
	object_list = klist_delete(object_list, widget);
	object_list = klist_add(object_list, widget, object);
	object->widget = widget;
	XtSetValues(widget, args, (Cardinal) num);
	return(TRUE);
}

/************************************************************
*
*  Routine Name: xvw_duplicate - duplicate an object
*
*       Purpose: Duplicates the specified GUI or visual object.
*
*         Input: object - the object to be duplicated
*        Output:
*	Returns: The duplicated object on success, NULL on failure.
*  Restrictions:
*    Written By: Mark Young and John Salas
*          Date: Feb 23, 1993
*      Verified: 
*  Side Effects: 
* Modifications:
*
*******************************************************************/

xvobject xvw_duplicate(
   xvobject    object)
{
	Widget widget = xvw_widget(object);
	char   *name = xvw_name(object);

	int num;
	Arg args[KLENGTH];
	xvobject new, parent = xvw_parent(object);


	/*
	 *  Sanity check
	 */
	if (!object || !widget)
	   return(FALSE);

	/*
	 *  Copy the object's attributes then recreate it with using the
	 *  specified widget class and the old widget class.
	 */
	xvw_copy_attributes(object, args, &num);
	new = xvw_create(parent, FALSE, TRUE, kstrdup(name), xvw_class(object));
	if (new == NULL)
	   return(NULL);

	XtSetValues(xvw_widget(new), args, (Cardinal) num);
	return(new);
}

/************************************************************
*
*  Routine Name: xvw_parent - get the parent of an object 
*
*       Purpose: Returns the parent of the specified GUI or visual object.
*
*         Input: object - the object for which to return the parent.
*
*        Output:
*	Returns: The parent of the object on success, NULL on failure
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Jun 15, 1992
*      Verified: 
*  Side Effects: 
* Modifications:
*
*******************************************************************/

xvobject xvw_parent(
   xvobject object)
{
	Widget widget = xvw_widget(object);

	return(!widget ? NULL : xvw_object(XtParent(widget)));
}

/************************************************************
*
*  Routine Name: xvw_destroy - destroy an object
*
*       Purpose: Destroy the visual or GUI object specified.
*		 Not only does this destroy the windows used
*		 by the object, it also frees all memory associated
*		 with the object.  You may never refer to the
*		 object again, after making this call. Any children
*                of the object will also be recursively destroyed. 
*		 The parent of the object, however, will be left
*                intact.
*
*         Input: object - the object to be destroyed
*
*        Output:
*	Returns:
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Jun 15, 1992
*      Verified: 
*  Side Effects: 
* Modifications:
*
*******************************************************************/

void xvw_destroy(
   xvobject object)
{
	Widget widget, parent;


	if ((parent = xvw_widget(xvw_parent(object))) != NULL &&
	     parent->core.being_destroyed)
	{
	   return;
	}
	else if ((widget = xvw_widget(object)) == NULL)
	   return;
	else if (xvw_check_toplevel(object) == TRUE)
	   xvw_remove_toplevel(object);

        if (klist_locate(object_list, widget) != NULL)
	{
	   if (object->created_toplevel == FALSE)
	      XtDestroyWidget(widget);
	   else
	      xvw_destroy(xvw_toplevel(object));
	}
}


/************************************************************
*
*  Routine Name: xvw_place - place an object on the screen
*
*       Purpose: This utility is used for placing a created object on the
*		 display.   This is used most often with custom compound 
*		 objects that are to be popped up after \fIxvf_run_form()\fP
*		 has already been called.
*
*         Input: object - the object to be destroyed
*		 placement - the object which the object should be placed
*			     relative to.  If NULL then the object is centered
*			     around the cursor.
*        Output:
*	Returns:
*  Restrictions:
*    Written By: Mark Young & Danielle Argiro
*          Date: Jun 15, 1992
*      Verified: 
*  Side Effects: 
* Modifications:
*
*******************************************************************/

void xvw_place(
   xvobject object,
   xvobject placement)
{
	Window   rw, cw;
	unsigned int mask;
	Display  *display;
	XWindowAttributes  xwa, pwa;
	int      rx, ry, wx, wy, x, y;
	xvobject toplevel = xvw_toplevel(object);
	Widget   widget = xvw_widget(toplevel);
	Position xpos1, xpos2, ypos1, ypos2;

	display = xvw_display(toplevel);

	XtSetMappedWhenManaged(widget, FALSE);
	if (!XtWindowOfObject(widget))
	{
	   XtRealizeWidget(widget);
	   xvw_sync(FALSE);
	}
	XQueryPointer(display, XDefaultRootWindow(display), &rw, &cw,
		 &rx, &ry, &wx, &wy, &mask);

	if (!placement)
	{
	   XGetWindowAttributes(display, xvw_window(toplevel), &xwa);

	   x = rx - xwa.width/2;
	   y = ry - xwa.height/2;
	}
	else if (toplevel == xvw_toplevel(placement))
	{
	   XtTranslateCoords(widget, 0, 0, &xpos1, &ypos1);
	   XtTranslateCoords(xvw_widget(placement), 0, 0, &xpos2, &ypos2);
	   XGetWindowAttributes(display, xvw_window(object), &xwa);
	   XGetWindowAttributes(display, xvw_window(placement), &pwa);

	   x = rx - (xpos2 - xpos1) - pwa.width + 3;
	   y = ry - (ypos2 - ypos1) - pwa.height/2;
	   xvw_set_attribute(toplevel, XVW_SHELL_WIN_GRAVITY, StaticGravity);
	}
	else
	{
	   XtTranslateCoords(widget, 0, 0, &xpos1, &ypos1);
	   XtTranslateCoords(xvw_widget(placement), 0, 0, &xpos2, &ypos2);
	   XGetWindowAttributes(display, xvw_window(object), &xwa);
	   XGetWindowAttributes(display, xvw_window(placement), &pwa);

	   x = xpos2 + ((pwa.width - xwa.width)/2);
	   y = ypos2 + ((pwa.height - xwa.height)/2);
	}

	if ((x + xwa.width) > XDisplayWidth(display, xvw_screennum(toplevel)))
	{
	   xvw_set_attribute(toplevel, XVW_SHELL_WIN_GRAVITY, StaticGravity);
	   x = XDisplayWidth(display, xvw_screennum(toplevel)) - xwa.width - 10;
	}

	if ((y + xwa.height) > XDisplayHeight(display, xvw_screennum(NULL)))
	{
	   xvw_set_attribute(toplevel, XVW_SHELL_WIN_GRAVITY, StaticGravity);
	   y = XDisplayHeight(display,xvw_screennum(toplevel)) - xwa.height -10;
	}

	if (x < 0) x = 0;
	if (y < 0) y = 0;

	xvw_set_attributes(toplevel, XVW_XPOSITION, x, XVW_YPOSITION, y, NULL);
	xvw_map(toplevel);
	xvw_raise(toplevel);
}

/************************************************************
*
*  Routine Name: xvw_unmap - unmap an object
*
*       Purpose: Unmap the GUI or visual object specified, causing
*		 it to "disappear".
*
*         Input: object - the object to be unmapped
*
*        Output:
*	Returns:
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Nov 06, 1992
*      Verified: 
*  Side Effects: 
* Modifications:
*
*******************************************************************/

void xvw_unmap(
   xvobject object)
{
	if (!object)
	   return;

	if (xvw_check_subclass(object, ManagerGadgetClass) == TRUE)
	   xvw_set_attribute(object, XVW_MAPPED, FALSE);
	else if ((xvw_check_managed(object) || xvw_check_toplevel(object)) &&
	    xvw_check_realized(object))
	{
	   XtUnmapWidget(xvw_widget(object));
	}
	else if (xvw_check_toplevel(object))
	   xvw_set_attribute(object, XVW_SHELL_INITIAL_STATE, WithdrawnState);
	else
	   xvw_set_attribute(object, XVW_MAP_WHEN_MANAGED, FALSE);
	
}

/************************************************************
*
*  Routine Name: xvw_map - map an object
*
*       Purpose: Map the specified GUI or visual object,
*		 causing it to be displayed.
*
*         Input: object - the object to be mapped
*
*        Output:
*	Returns:
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Nov 06, 1992  10:21
*      Verified: 
*  Side Effects: 
* Modifications:
*
*******************************************************************/

void xvw_map(
   xvobject object)
{
	if (!object)
	   return;

	if (xvw_check_subclass(object, ManagerGadgetClass) == TRUE)
	   xvw_set_attribute(object, XVW_MAPPED, TRUE);
	else if ((xvw_check_managed(object) || xvw_check_toplevel(object)) &&
		 xvw_check_realized(object))
	{
	   XtMapWidget(xvw_widget(object));
	}
	else if (xvw_check_toplevel(object))
	{
	   XtAppContext context = xvw_appcontext(object);
	   xvw_set_attribute(object, XVW_SHELL_INITIAL_STATE, NormalState);
	   XtAppAddTimeOut(context, 0, realize_parent, (XtPointer) object);
	}
	else
	   xvw_set_attribute(object, XVW_MAP_WHEN_MANAGED, TRUE);
}

/************************************************************
*
*  Routine Name: xvw_unmanage - unmanage an object
*
*       Purpose: Causes the specified GUI or visual object to have its 
*		 geometry unmanaged by its parent. Unmanaging an unmapped
*		 object means that it does not use resources of the X server, 
*		 freeing them for use by other objects.  Furthermore, 
*		 if the unmapped object does not require management by 
*		 its parent, the calling application will be able to manage
*		 currently mapped objects more efficiently.  Note that an 
*		 object cannot be made visible if it is not managed. 
*
*         Input: object - the object to be unmanaged
*
*        Output:
*	Returns:
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Jun 14, 1993
*      Verified: 
*  Side Effects: 
* Modifications:
*
*******************************************************************/

void xvw_unmanage(
   xvobject object)
{
	if (object) XtUnmanageChild(xvw_widget(object));
}

/************************************************************
*
*  Routine Name: xvw_manage - manage an object
*
*       Purpose: Causes the specified GUI or visual object to have its 
*		 geometry managed by its parent. Note that an object cannot 
*		 be made visible (mapped) until it is managed;  however, by
*		 default most objects are managed at the time when they 
*                are created.  The panicon object is an exception to this rule.
*
*         Input: object - the object to be managed
*
*        Output:
*	Returns:
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Nov 06, 1992
*      Verified: 
*  Side Effects: 
* Modifications:
*
*******************************************************************/

void xvw_manage(
   xvobject object)
{
	if (object) XtManageChild(xvw_widget(object));
}

/************************************************************
*
*  Routine Name: xvw_raise - raise an object
*
*       Purpose: Raises a GUI or visual object with respect to other
*                siblings in the visual stacking order.  If the object
*                overlaps another sibling object, this call will cause
*                the object to obscure the sibling.
*
*         Input: object - the object to be raised
*
*        Output:
*	Returns:
*  Restrictions:
*    Written By: Mark Young
*          Date: Oct 20, 1993
*      Verified: 
*  Side Effects: 
* Modifications:
*
*******************************************************************/

void xvw_raise(
   xvobject object)
{
	if (xvw_check_realized(object))
	   XRaiseWindow(xvw_display(object), xvw_window(object));
}

/************************************************************
*
*  Routine Name: xvw_lower - lower an object
*
*       Purpose: Lowers a GUI or visual object with respect to other
*		 siblings in the visual stacking order.  If the object
*		 overlaps another sibling object, this call will cause
*		 the object to be obscured by the sibling.
*
*         Input: object - the object to be lowered
*        Output:
*	Returns:
*  Restrictions:
*    Written By: Mark Young
*          Date: Oct 20, 1993
*      Verified: 
*  Side Effects: 
* Modifications:
*
*******************************************************************/

void xvw_lower(
   xvobject object)
{
	if (xvw_check_realized(object))
	   XLowerWindow(xvw_display(object), xvw_window(object));
}

/************************************************************
*
*  Routine Name: xvw_realize - realize an object
*
*       Purpose: Realizes a GUI or visual object.  The realization
*                process involves creation of the actual (displayable)
*                parts of the object, including any window(s) that may
*                be associated with the object.  
*
*                An object cannot be made visible (mapped) until it is 
*                realized;  however, by default most objects are realized 
*                at the time when they are created. 
*
*         Input: object - the object to be realized
*
*        Output:
*	Returns:
*
*  Restrictions:
*    Written By: Danielle Argiro & Mark Young
*          Date: Nov 06, 1992
*      Verified: 
*  Side Effects: 
* Modifications:
*
*******************************************************************/

void xvw_realize(
   xvobject object)
{
	if (!xvw_check_realized(object))
	   XtRealizeWidget(xvw_widget(object));

}

/************************************************************
*
*  Routine Name: xvw_unrealize - un-realize an object
*
*       Purpose: When an object is \fIrealized\fP, it has its windows
*                created on the display, and its final initializations
*                done. This routine allows you to un-realize an object.
*		 This destroys the windows associated with an object
*		 and its descentants, but does not destroy the instance
*		 of the object itself;  the object can be still be
*		 re-realized and used again at a later date.
*
*         Input: object - the object to be un-realized
*
*        Output:
*	Returns:
*
*  Restrictions:
*    Written By: Danielle Argiro & Mark Young
*          Date: Nov 06, 1992
*      Verified:
*  Side Effects: 
* Modifications:
*
*******************************************************************/

void xvw_unrealize(
   xvobject object)
{
	if (xvw_check_realized(object))
	   XtUnrealizeWidget(xvw_widget(object));

}

/************************************************************
*
*  Routine Name: xvw_sensitive - sensitize or de-sensitize an object
*
*       Purpose: Sensitizes or de-sensitizes a GUI or visual object.
*		 When an object is made insensitive, it will ignore all 
*		 attempts at user input; it will also appear "dim" or 
*		 "stippled" to indicate its insensitive state (a current 
*		 limitation restricts this capability to widgets only; 
*		 insensitive gadgets will not accept input, but they cannot
*		 provide a visual cue to reflect their insensitive state).  
*		 All objects are normally sensitive.
*
*         Input: object    - the object to be affected
*                sensitive - FALSE if object is to be de-sensitized.
*			     TRUE if object is to be re-sensitized,
*        Output:
*	Returns:
*
*  Restrictions:
*    Written By: Danielle Argiro & Mark Young
*          Date: Nov 06, 1992
*      Verified: 
*  Side Effects: 
* Modifications:
*
*******************************************************************/

void xvw_sensitive(
   xvobject object,
   int    sensitive)
{
	XtSetSensitive(xvw_widget(object), sensitive);
}

/************************************************************
*
*  Routine Name: xvw_refresh - refreshes an object
*
*       Purpose: Refreshes a GUI or visual object.
*
*         Input: object    - the object to be refreshed
*        Output:
*	Returns:
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Nov 06, 1993
*      Verified: 
*  Side Effects: 
* Modifications:
*
*******************************************************************/

void xvw_refresh(
   xvobject object)
{
	if (xvw_check_realized(object))
	   XClearArea(xvw_display(object),xvw_window(object), 0, 0, 0, 0, TRUE);
}

/************************************************************
*
*  Routine Name: xvw_geometry - get the geometry of an object
*
*       Purpose: Returns the current geometry of the specified object,
*		 including (x,y) position with respect to the parent,
*		 width, height, and border width.  All geometry values
*		 are given in \fIpixels\fP.
*
*         Input: object - object for which to return geometry
*
*        Output: x            - x position of the object in pixels, 
*                               with respect to its parent
*		 y            - y position of the object in pixels
*                               with respect to its parent
*	   	 width        - width of the object in pixels
*		 height       - height of the object in pixels
*		 border_width - border width of the object in pixels
*	
*	Returns: TRUE (1) on success, FALSE (0) on failure
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Jun 15, 1992
*      Verified: 
*  Side Effects: 
* Modifications:
*
*******************************************************************/

int xvw_geometry(
   xvobject object,
   int	    *x,
   int	    *y,
   unsigned int	*width,
   unsigned int	*height,
   unsigned int	*border_width)
{
	Position  xpos, ypos;
	Dimension wdim, hdim, bwdim;


	if (!ManagerGeometry(xvw_widget(object), &xpos, &ypos, &wdim, &hdim,
				&bwdim))
	{
	   return(FALSE);
	}
	if (x) *x = (int) xpos;
	if (y) *y = (int) ypos;
	if (width)  *width  = (int) wdim;
	if (height) *height = (int) hdim;
	if (border_width) *border_width = (int) bwdim;
	return(TRUE);
}

/************************************************************
*
*  Routine Name: xvw_children - get the children of an object
*
*       Purpose: Returns an array of xvobjects containing the children 
*		 of the object specified.  If desired, an object class
*		 may be specified so that only children of the specified 
*		 class are returned.  
*
*         Input: object - object for which to find children
*		 class  - the desired class of the children, 
*			  or NULL for all children
*
*        Output: num_children - the number of children being returned
*				in the xvobject array
*	Returns: An array of children upon success, NULL on failure
*  Restrictions:
*    Written By: Mark Young
*          Date: Nov 27, 1993
*      Verified: 
*  Side Effects: The array in which the children returned is allocated,
*		 and must be freed by the caller.
* Modifications:
*
*******************************************************************/

xvobject *xvw_children(
   xvobject object,
   xvclass  class,
   int	    *num_children)
{
	xvobject child;
	klist    *list;
	int      i, num = 0, matched = 0;
	xvobject *temp = NULL, *children = NULL;


	/*
	 *  get all the children...
	 */
	if (object)
	{
	   xvw_get_attributes(object,
		XVW_NUM_CHILDREN, &num,
		XVW_CHILDREN,     &temp,
		NULL);

	   for (i = 0; i < num; i++)
	   {
	      child = xvw_object((Widget) temp[i]);
	      if (!class || xvw_check_subclass(child, class))
	      {
	         children = (xvobject *) karray_add((char **) children,
			(kaddr) child, matched++);
	      }
	   }
	}
	else
	{
	   list = object_list;
	   while (list != NULL)
	   {
	      child = (xvobject) klist_clientdata(list);
	      if (!class || xvw_check_subclass(child, class))
	      {
	         children = (xvobject *) karray_add((char **) children,
			(kaddr) child, matched++);
	      }
	      list = klist_next(list);
	   }
	}
	if (num_children) *num_children = matched;
	return(children);
}

/************************************************************
*
*  Routine Name: xvw_numchildren - get the number of children of an 
*				   object
*
*       Purpose: Returns the number of children of the object.  
*		 If desired, an object class may be specified, so that 
*		 only children of that class are counted.
*
*         Input: object - object to for which to check number of children
*		 class  - the desired class of the children, or NULL for all
*			  children
*
*        Output: 
*	Returns: Returns the number of children found
*  Restrictions:
*    Written By: Mark Young
*          Date: Nov 27, 1993
*      Verified: 
*  Side Effects:
* Modifications:
*
*******************************************************************/

int xvw_numchildren(
   xvobject    object,
   xvclass class)
{
	xvobject child, *temp = NULL;
	int      i, num = 0, matched = 0;


	/*
	 *  get all the children...
	 */
	xvw_get_attributes(object,
		XVW_NUM_CHILDREN, &num,
		XVW_CHILDREN,     &temp,
		NULL);

	for (i = 0; i < num; i++)
	{
	   child = xvw_object((Widget) temp[i]);
	   if ((!class && class != InputOnlyWidgetClass) ||
		class == xvw_class(child))
	      matched++;
	}
	return(matched);
}

/************************************************************
*
*  Routine Name: xvw_translate_coords - translates the coordinates of an 
*				        object to those of the root window
*
*       Purpose: Returns the device coordinates of a object with respect
*		 to those of the root window.  If x and y are both passed
*		 as 0, then the device coordinates returned are with respect
*		 to the upper left hand corner of the root window.  If x and
*		 y are passed as greater than 0, then the device coordinates
*		 returned will be relative to the (x, y) position specified.
*		 Note that it is an error to specify an x value larger then
*		 the screen width, or a y value larger than the screen height.
*
*         Input: object - the object for which to get device coordinates
*		 x      - the relative horizontal offset from the 
*			  upper left hand corner of the root window 
*		 y      - the relative vertical offset from the 
*			  upper left hand corner of the root window 
*        Output: xpos   - the x position relative to (x,y) coordinate 
*			  of the root window specified
*                ypos   - the y position relative to the (x,y) coordinate
*			  of the root window specified
*	Returns: TRUE (1) on success, FALSE (0) on failure
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Jan 2, 1994
*      Verified: 
*  Side Effects: 
* Modifications:
*
*******************************************************************/

int xvw_translate_coords(
    xvobject object,
    int      x,
    int      y,
    int      *xpos,
    int      *ypos)
{
	Position xtmp, ytmp;
	Widget widget = xvw_widget(object);

	if (!widget)
	   return(FALSE);

	XtTranslateCoords(widget, x, y, &xtmp, &ytmp);
	if (xpos) *xpos = xtmp;
	if (ypos) *ypos = ytmp;
	return(TRUE);
}

/************************************************************
*
*  Routine Name: xvw_font - return the font being used by a object
*
*       Purpose: Returns the display font being used
*                by an object;  in other words, this is the setting
*                of the XVW_FONT attribute of the object.
*                If the object specified does not have the XVW_FONTE
*                attribute set, the font being used by its parent will
*                be returned.  Recursion up the inheritance tree
*                will be done until a setting for the font is found.
*
*         Input:  object - object to get the font struct from
*        Output: 
*       Returns: The font struct associated with the object
*		 on success, NULL on failure
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Jan 21, 1993
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

XFontStruct *xvw_font(
   xvobject object)
{
	XFontStruct *fontstruct = NULL;

	if (!object)
	{
	   errno = KCALL;
	   kerror("xvwidgets", "xvw_font",
		  "object passed in may not be NULL");
	   return(NULL);
	}

	(void) xvw_get_attribute(object, XVW_FONT, &fontstruct);
	return(fontstruct ? fontstruct : xvw_font(xvw_parent(object)));
}

/************************************************************
*
*  Routine Name: xvw_fontname - return the font name being used by an object
*
*       Purpose: Returns the name of the font being used
*		 by an object;  in other words, this is the setting
*		 of the XVW_FONTNAME attribute of the object.
*		 If the object specified does not have the XVW_FONTNAME
*		 attribute set, the font being used by its parent will
*		 be returned.  Recursion up the inheritance tree
*		 will be done until a setting for the font name is found.
*
*         Input: object - object for which to query font name
*        Output:
*       Returns: The fontname associated with the object
*                on success, NULL on failure
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Jan 21, 1993
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

char *xvw_fontname(
   xvobject object)
{
	char *fontname = NULL;

	if (!object)
	{
	   errno = KCALL;
	   kerror("xvwidgets", "xvw_fontname",
		  "object passed in may not be NULL");
	   return(NULL);
	}

        (void) xvw_get_attribute(object, XVW_FONTNAME, &fontname);
        return(fontname ? fontname : xvw_fontname(xvw_parent(object)));
}

/************************************************************
*
*  Routine Name: xvw_sort - sort a list of objects
*
*       Purpose: This routine will sort a list of GUI and visual objects,  
*		 first by their Y position, and then by their X position.
*
*         Input: objects - array of GUI and/or visual objects to be sorted
*		 num	 - the number of objects in the array
*        Output: objects - the sorted array
*       Returns: TRUE (1) on success, FALSE (0) on failure
*  Restrictions:
*    Written By:
*          Date:
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int xvw_sort(
*                !  xvobject *object,
*                !  int      num)
*
*************************************************************/
/* ARGSUSED */
int xvw_sort(
   xvobject *objects,
   int	    num)
{
	if (!objects)
	{
	   errno = KCALL;
	   kerror("xvwidgets", "xvw_sort", "objects passed in may not be NULL");
	   return(FALSE);
	}
	return(FALSE);
}

/************************************************************
*
*  Routine Name: xvw_check_realized - see if an object is realized
*
*       Purpose: Indicates whether or not an object has been realized.
*		 When an object is \fIrealized\fP, it has its windows
*		 created on the display, and its final initializations
*		 done. The realization of a GUI or visual object is 
*		 delayed until \fIxvf_run_form()\fP is called (if the
*		 object is created before the call to \fIxvf_run_form()\fP),
*		 or until software control flow is diverted back to the 
*		 \fIxvf_run_form()\fP loop (f the object is created after 
*		 the call to \fIxvf_run_form()\fP).
*
*         Input: object - the object to be checked
*
*        Output:
*	Returns: TRUE if the object has been realized, 
*		 FALSE otherwise
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Jan 18, 1993
*      Verified: 
*  Side Effects: 
* Modifications:
*
*******************************************************************/

int xvw_check_realized(
   xvobject object)
{
	Widget widget = xvw_widget(object);

	if (!widget)
	   return(FALSE);

	return(!XtWindowOfObject(xvw_widget(object)) ? FALSE : TRUE);
}


/************************************************************
*
*  Routine Name: xvw_check_sensitive - see if an object is sensitive
*
*       Purpose: Indicates whether or not an object is currently sensitive.
*		 Note that a GUI or visual object may be made insensitive
*		 using \fIxvw_sensitive()\fP. When an object is made 
*		 insensitive, it will ignore all attempts at user input;
*		 it will also appear "dim" or "stippled" to indicate its
*		 insensitive state (a current limitation restricts this
*		 capability to widgets only; insensitive gadgets will not
*		 accept input, but they cannot provide a visual cue to reflect
*		 their insensitive state).  All objects are normally sensitive.
*
*         Input: object - the object to check
*
*        Output:
*	Returns: TRUE if the object is sensitive, 
*		 FALSE otherwise.
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Jan 18, 1993
*      Verified: 
*  Side Effects: 
* Modifications:
*
*******************************************************************/

int xvw_check_sensitive(
   xvobject object)
{
	Widget widget = xvw_widget(object);

	if (!widget)
	   return(FALSE);

	return((int) XtIsSensitive(widget));
}

/************************************************************
*
*  Routine Name: xvw_check_mapped - see if an object is mapped
*
*       Purpose: Indicates whether or not an object is currently mapped.
*
*         Input: object - the object to check
*
*        Output:
*	Returns: TRUE if the object is mapped,
*		 FALSE otherwise
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Jan 18, 1993
*      Verified: 
*  Side Effects: 
* Modifications:
*
*******************************************************************/

int xvw_check_mapped(
   xvobject object)
{
	XWindowAttributes xwa;
	XvwManagerGadget  manobj;
	Widget widget = xvw_widget(object);


	if (!widget || (!XtIsManaged(widget) && XtParent(widget)))
	   return(FALSE);

	if (xvw_check_subclass(object, ManagerGadgetClass))
	{
	   manobj = (XvwManagerGadget) widget;
	   return((int) manobj->manager.mapped);
	}
	else if (!XtIsWidget(widget))
	{
	   return(TRUE);
	}
	else if (!XtWindowOfObject(widget) ||
		 !XGetWindowAttributes(XtDisplay(widget),XtWindow(widget),&xwa))
	{
	   return(FALSE);
	}
	return(xwa.map_state != IsUnmapped);
}

/************************************************************
*
*  Routine Name: xvw_check_visible - see if an object is visible
*
*       Purpose: Indicates whether or not an object is currently visible.
*
*         Input: object - the object to check
*
*        Output:
*	Returns: TRUE if the object is visible,
*		 FALSE otherwise
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Jun 03, 1994
*      Verified: 
*  Side Effects: 
* Modifications:
*
*******************************************************************/

int xvw_check_visible(
   xvobject object)
{
	XWindowAttributes xwa;
	XvwManagerGadget  manobj;
	Widget widget = xvw_widget(object);


	if (!widget || !XtWindowOfObject(widget) ||
	    (!XtIsManaged(widget) && XtParent(widget)))
	   return(FALSE);

	if (xvw_check_subclass(object, ManagerGadgetClass))
	{
	   manobj = (XvwManagerGadget) widget;
	   if (!manobj->manager.mapped || !manobj->manager.realized)
	      return(FALSE);

	   widget = XtParent(widget);
	}
	else if (!XtIsWidget(widget))
	   widget = XtParent(widget);

	XSync(XtDisplay(widget), FALSE);
	if (!XGetWindowAttributes(XtDisplay(widget), XtWindow(widget), &xwa))
	   return(FALSE);

	return(xwa.map_state == IsViewable);
}

/************************************************************
*
*  Routine Name: xvw_check_managed - see if an object is managed
*
*       Purpose: Indicates whether or not an object is currently
*		 having its geometry managed by its parent. Note
*		 that an object cannot be made visible until it
*		 is managed.
*
*         Input: object - the object to check
*
*        Output:
*	Returns: TRUE if the object is being managed,
*		 FALSE otherwise
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Jan 18, 1993
*      Verified: 
*  Side Effects: 
* Modifications:
*
*******************************************************************/

int xvw_check_managed(
   xvobject object)
{
	Widget widget = xvw_widget(object);

	if (!widget)
	   return(FALSE);

	return((int) XtIsManaged(widget));
}


/************************************************************
*
*  Routine Name: xvw_check_menuactive - see if an object's internal 
*                                       menuform is displayed
*
*       Purpose: Indicates whether or not the internal menuform for
*		 a GUI or visual object is currently being displayed.
*
*         Input: object - the GUI or visual object to check
*
*        Output:
*	Returns: TRUE if the internal menuform for the object is
*		 currently displayed, FALSE otherwise
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Jan 18, 1993
*      Verified: 
*  Side Effects: 
* Modifications:
*
*******************************************************************/

int xvw_check_menuactive(
   xvobject object)
{
	return((int) xvw_activated_menu(object));
}

/************************************************************
*
*  Routine Name: xvw_check_menuexist - check if an object has an 
*	                               internal menuform
*
*       Purpose: Checks whether or not a GUI or visual object
*		 has an internal menuform associated with it.
*
*         Input: object - the object for which to check for
*			  internal menuforms
*        Output:
*	Returns: TRUE if the object has an internal menuform associated
*		 with it, FALSE otherwise.
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Jan 18, 1993
*      Verified: 
*  Side Effects: 
* Modifications:
*
*******************************************************************/

int xvw_check_menuexist(
   xvobject object)
{
	return((int) xvw_get_menuinfo(object, XVW_MENU_FORMFILE, NULL));
}

/************************************************************
*
*  Routine Name: xvw_check_toplevel - see if object specified is a toplevel, 
*				      or see if a toplevel exists
*
*       Purpose: If the object is specified as non-NULL, returns whether
*                or not the object is a toplevel object.  If NULL is passed
*		 for the object parameter, returns whether or not any 
*		 toplevel objects exist.
*
*         Input: object - the object to check, or NULL to check
*			  for any toplevels
*        Output:
*	Returns: TRUE if the object specified is a toplevel object
*		 (or if any toplevel objects exist), FALSE otherwise.
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Jun 28, 1993
*      Verified: 
*  Side Effects: 
* Modifications:
*
*******************************************************************/

int xvw_check_toplevel(
   xvobject object)
{
	if (!object)
	   return(xvw_toplevels != NULL);
	else
	   return(xvw_parent(object) == NULL);
}

/************************************************************
*
*  Routine Name: xvw_check_localhost - see if the object is on the local 
*                                      display
*
*       Purpose: When passed a GUI or visual object, this routine 
*		 returns TRUE if the object appears on the local display.
*		 If passed NULL for the object, the routine will indicate
*                whether or not the default display is local.  This is used 
*                to determine whether certain visual objects can use some 
*                of the fast server extensions (eg, MIT shared memory) that 
*                are only available if the client process and display are 
*                located on the same local host.
*
*         Input: object - the object for to check, or NULL to check 
*                         the default display
*        Output:
*	Returns: TRUE to indicate the local display, FALSE to indicate
*		 a remote display
*  Restrictions:
*    Written By: Mark Young
*          Date: July 19, 1993
*      Verified: 
*  Side Effects: 
* Modifications:
*
*******************************************************************/

int xvw_check_localhost(
   xvobject object)
{
	char temp[KLENGTH], *display_name = DisplayString(xvw_display(object));


	if (kstrncmp(display_name, ":", kstrlen(":")) == 0	  ||
	    kstrncmp(display_name, "unix", kstrlen("unix")) == 0  ||
	    kstrncmp(display_name, "localhost", kstrlen("localhost")) == 0)
	{
	   return(TRUE);
	}
	else if (!kgethostname(NULL, temp, KLENGTH) &&
		 kstrncmp(display_name, temp, kstrlen(temp)) == 0)
	{
	   return(TRUE);
	}
	return(FALSE);
}

/************************************************************
*
*  Routine Name: xvw_check_subclass - check the subclass of an object
*
*       Purpose: Checks if the object has the specified subclass.
*		 Example1: if the object passed in was a zoom object,
*		 and the class given was ImageClass, the routine
*		 would return TRUE since the zoom object is subclassed
*		 off the image object.  Example2: if the object passed
*		 in was a zoom object, and the class given was ColorClass,
*		 the routine would return TRUE, since the zoom object
*		 is subclassed off the image object and the image object
*		 is subclassed off the color object.  Example3: if the
*		 object passed in was a zoom object, and the class given
*		 was PanIconClass, the routine would return FALSE, since
*		 the PanIconClass does not appear in the inheritance tree
*		 of the zoom object.
*
*         Input: object - object for which to check the subclass
*		 class  - the class to be checked for
*        Output:
*	Returns: TRUE if the object has the specified subclass,
*		 FALSE otherwise
*  Restrictions:
*    Written By: Mark Young
*          Date: Nov 29, 1993
*      Verified: 
*  Side Effects:
* Modifications:
*
*******************************************************************/

int xvw_check_subclass(
   xvobject object,
   xvclass  class)
{
	Widget widget = xvw_widget(object);

	return(!widget ? FALSE : XtIsSubclass(widget, class));
}
