 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            General Extent extraction Utility Routines
   >>>>
   >>>>   Static:
   >>>>		   rectangle_handler
   >>>>		   circle_handler
   >>>>		   ellipse_handler
   >>>>		   line_handler
   >>>>		   polyline_handler
   >>>>		   freehand_handler
   >>>>  Private:
   >>>>   Public:
   >>>>		   xvw_getextent_rectangle()
   >>>>		   xvw_getextent_circle()
   >>>>		   xvw_getextent_ellipse()
   >>>>		   xvw_getextent_line()
   >>>>		   xvw_getextent_polyline()
   >>>>		   xvw_getextent_freehand()
   >>>>
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"

#define KRECTANGLE 1
#define KCIRCLE    2
#define KELLIPSE   3
#define KLINE      4
#define KPOLYLINE  5
#define KPAINT     6

static int    maxpts = 0;
static XPoint *pts = NULL;
static unsigned int   numpts;
static int extracted, erase_selection, xpos1, xpos2,
	   ypos1, ypos2;

/*-----------------------------------------------------------
|
|  Routine Name: rectangle_handler - interactive rectangle extent handler
|
|       Purpose: interactively getting extents for a rectangle for a given
|		 object, by drawing a rubber-band extent.
|
|         Input: object	     - object in which the event was placed
|		 client_data - the GC to be used
|		 event       - the event that triggered this handler
|
|    Written By: Mark Young
|          Date: Jul 30, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void rectangle_handler(
    xvobject object,
    kaddr    client_data,
    XEvent   *event,
    int	     *dispatch)
{
	GC gc = (GC) client_data;
	Display *display = xvw_display(object);
	Window  window;

	int      x, y;
	unsigned int w, h;


	x = xpos2;
	y = ypos2;
	*dispatch = FALSE;
	if (event->type == MotionNotify)
	{
	   if (xpos1 == -1 && ypos1 == -1)
	   {
	      x = xpos1 = event->xmotion.x;
	      y = ypos1 = event->xmotion.y;
	   }
	   else
	   {
	      xpos2 = event->xmotion.x;
	      ypos2 = event->xmotion.y;
	   }
	}
	else if (event->type == ButtonPress)
	{
	   if (event->xbutton.button == Button3)
	   {
	      extracted = TRUE;
	      return;
	   }
	   xpos1 = event->xbutton.x;
	   ypos1 = event->xbutton.y;
	}
	else if (event->type == ButtonRelease)
	{
	   xpos2 = event->xbutton.x;
	   ypos2 = event->xbutton.y;
	   extracted = TRUE;
	   if (erase_selection == FALSE)
	      return;
	}

	if (xvw_check_subclass(object, InputOnlyWidgetClass) == TRUE)
	   window = xvw_window(xvw_parent(object));
	else
	   window = xvw_window(object);

        if (event->type != ButtonPress && x != -1 && y != -1)
        {
           /*
            *  Erase the old rectangle.
            */
           w = kabs(xpos1 - x); h = kabs(ypos1 - y);
           x = kmin(xpos1,  x); y = kmin(ypos1,  y);
           XDrawRectangle(display, window, gc, x, y, w, h);
        }
 
        if (extracted == FALSE && xpos2 != -1 && ypos2 != -1)
        {
           /*
            *  Draw the new rectangle.
            */
           x = kmin(xpos1,  xpos2); y = kmin(ypos1,  ypos2);
           w = kabs(xpos1 - xpos2); h = kabs(ypos1 - ypos2);
           XDrawRectangle(display, window, gc, x, y, w, h);
        }
}

/*-----------------------------------------------------------
|
|  Routine Name: circle_handler - interactive circle extent handler
|
|       Purpose: interactively getting extents for a circle for a given
|		 object, by drawing a rubber-band extent.
|
|         Input: object	     - object in which the event was placed
|		 client_data - the GC to be used
|		 event       - the event that triggered this handler
|
|    Written By: Mark Young
|          Date: Jul 30, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void circle_handler(
    xvobject object,
    kaddr    client_data,
    XEvent   *event,
    int	     *dispatch)
{
	GC gc = (GC) client_data;
	Display *display = xvw_display(object);
	Window  window;

	int      x, y, rx, ry;
	unsigned int w, h, radius;


	x = xpos2;
	y = ypos2;
	*dispatch = FALSE;
	if (event->type == MotionNotify)
	{
	   if (xpos1 == -1 && ypos1 == -1)
	   {
	      x = xpos1 = event->xmotion.x;
	      y = ypos1 = event->xmotion.y;
	   }
	   else
	   {
	      xpos2 = event->xmotion.x;
	      ypos2 = event->xmotion.y;
	   }
	}
	else if (event->type == ButtonPress)
	{
	   if (event->xbutton.button == Button3)
	   {
	      extracted = TRUE;
	      return;
	   }
	   xpos1 = event->xbutton.x;
	   ypos1 = event->xbutton.y;
	}
	else if (event->type == ButtonRelease)
	{
	   xpos2 = event->xbutton.x;
	   ypos2 = event->xbutton.y;
	   extracted = TRUE;
	   if (erase_selection == FALSE)
	      return;
	}

	if (xvw_check_subclass(object, InputOnlyWidgetClass) == TRUE)
	   window = xvw_window(xvw_parent(object));
	else
	   window = xvw_window(object);

        if (event->type != ButtonPress && x != -1 && y != -1)
        {
           /*
            *  Erase the old circle.
            */
	   rx = xpos1 - x; ry = ypos1 - y;
	   radius = ksqrt((double) (rx*rx + ry*ry));
	   x = xpos1 - radius; y = ypos1 - radius;
	   w = h = radius * 2;
           XDrawArc(display, window, gc, x, y, w, h, 0, 23040);
        }
 
        if (extracted == FALSE && xpos2 != -1 && ypos2 != -1)
        {
           /*
            *  Draw the new circle.
            */
	   rx = xpos1 - xpos2; ry = ypos1 - ypos2;
	   radius = ksqrt((double) (rx*rx + ry*ry));
	   x = xpos1 - radius; y = ypos1 - radius;
	   w = h = radius * 2;
           XDrawArc(display, window, gc, x, y, w, h, 0, 23040);
        }
}

/*-----------------------------------------------------------
|
|  Routine Name: ellipse_handler - interactive ellipse extent handler
|
|       Purpose: interactively getting extents for a elliptical for a given
|		 object, by drawing a rubber-band extent.
|
|         Input: object	     - object in which the event was placed
|		 client_data - the GC to be used
|		 event       - the event that triggered this handler
|
|    Written By: Mark Young
|          Date: Jul 30, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void ellipse_handler(
    xvobject object,
    kaddr    client_data,
    XEvent   *event,
    int	     *dispatch)
{
	GC gc = (GC) client_data;
	Display *display = xvw_display(object);
	Window  window;

	int       x, y;
	unsigned int w, h;

	x = xpos2;
	y = ypos2;
	*dispatch = FALSE;
	if (event->type == MotionNotify)
	{
	   if (xpos1 == -1 && ypos1 == -1)
	   {
	      x = xpos1 = event->xmotion.x;
	      y = ypos1 = event->xmotion.y;
	   }
	   else
	   {
	      xpos2 = event->xmotion.x;
	      ypos2 = event->xmotion.y;
	   }
	}
	else if (event->type == ButtonPress)
	{
	   if (event->xbutton.button == Button3)
	   {
	      extracted = TRUE;
	      return;
	   }
	   xpos1 = event->xbutton.x;
	   ypos1 = event->xbutton.y;
	}
	else if (event->type == ButtonRelease)
	{
	   xpos2 = event->xbutton.x;
	   ypos2 = event->xbutton.y;
	   extracted = TRUE;
	   if (erase_selection == FALSE)
	      return;
	}

	if (xvw_check_subclass(object, InputOnlyWidgetClass) == TRUE)
	   window = xvw_window(xvw_parent(object));
	else
	   window = xvw_window(object);

        if (event->type != ButtonPress && x != -1 && y != -1)
        {
           /*
            *  Erase the old ellipse.
            */
           w = kabs(xpos1 - x)*2; h = kabs(ypos1 - y)*2;
	   x = (xpos1 < x) ? xpos1 - w/2 : x;
	   y = (ypos1 < y) ? ypos1 - h/2 : y;
           XDrawArc(display, window, gc, x, y, w, h, 0, 23040);
        }
 
        if (extracted == FALSE && xpos2 != -1 && ypos2 != -1)
        {
           /*
            *  Draw the new ellipse.
            */
           w = kabs(xpos1 - xpos2)*2;
	   h = kabs(ypos1 - ypos2)*2;
	   x = (xpos1 < xpos2) ? xpos1 - w/2 : xpos2;
	   y = (ypos1 < ypos2) ? ypos1 - h/2 : ypos2;
           XDrawArc(display, window, gc, x, y, w, h, 0, 23040);
        }
}

/*-----------------------------------------------------------
|
|  Routine Name: line_handler - interactive line extent handler
|
|       Purpose: interactively getting extents for a line for a given
|		 object, by drawing a rubber-band extent.
|
|         Input: object	     - object in which the event was placed
|		 client_data - the GC to be used
|		 event       - the event that triggered this handler
|
|    Written By: Mark Young
|          Date: Jul 30, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void line_handler(
    xvobject object,
    kaddr    client_data,
    XEvent   *event,
    int	     *dispatch)
{
	GC gc = (GC) client_data;
	Display *display = xvw_display(object);
	Window  window;

	int x, y;

	x = xpos2;
	y = ypos2;
	*dispatch = FALSE;
	if (event->type == MotionNotify)
	{
	   if (xpos1 == -1 && ypos1 == -1)
	   {
	      x = xpos1 = event->xmotion.x;
	      y = ypos1 = event->xmotion.y;
	   }
	   else
	   {
	      xpos2 = event->xmotion.x;
	      ypos2 = event->xmotion.y;
	   }
	}
	else if (event->type == ButtonPress)
	{
	   if (event->xbutton.button == Button3)
	   {
	      extracted = TRUE;
	      return;
	   }
	   xpos1 = event->xbutton.x;
	   ypos1 = event->xbutton.y;
	}
	else if (event->type == ButtonRelease)
	{
	   xpos2 = event->xbutton.x;
	   ypos2 = event->xbutton.y;
	   extracted = TRUE;
	   if (erase_selection == FALSE)
	      return;
	}

	if (xvw_check_subclass(object, InputOnlyWidgetClass) == TRUE)
	   window = xvw_window(xvw_parent(object));
	else
	   window = xvw_window(object);

        if (event->type != ButtonPress && x != -1 && y != -1)
        {
           /*
            *  Erase the old line.
            */
	   XDrawLine(display, window, gc, xpos1, ypos1, x, y);
        }
 
        if (extracted == FALSE && xpos2 != -1 && ypos2 != -1)
        {
           /*
            *  Draw the new line.
            */
	   XDrawLine(display, window, gc, xpos1, ypos1, xpos2,
			ypos2);
        }
}

/*-----------------------------------------------------------
|
|  Routine Name: polyline_handler - interactive polyline extent handler
|
|       Purpose: interactively getting extents for a polyline for a given
|		 object, by drawing a rubber-band extent.
|
|         Input: object	     - object in which the event was placed
|		 client_data - the GC to be used
|		 event       - the event that triggered this handler
|
|    Written By: Mark Young
|          Date: Jul 30, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void polyline_handler(
    xvobject object,
    kaddr    client_data,
    XEvent   *event,
    int	     *dispatch)
{
	GC gc = (GC) client_data;
	Display *display = xvw_display(object);
	Window  window;

	int i, x, y;
	static int done = FALSE;

	x = xpos2;
	y = ypos2;
	*dispatch = FALSE;
	if (xvw_check_subclass(object, InputOnlyWidgetClass) == TRUE)
	   window = xvw_window(xvw_parent(object));
	else
	   window = xvw_window(object);

	if (numpts == 0)
	   done = FALSE;

	if (numpts >= maxpts)
	{
	   maxpts += KLENGTH;
	   pts = (XPoint *) krealloc(pts, sizeof(XPoint) * maxpts);
	}

	if (event->type == MotionNotify)
	{
	   if (numpts > 0 || event->xbutton.state == 0)
	   {
	      xpos2 = event->xmotion.x;
	      ypos2 = event->xmotion.y;
	   }
	   else
	   {
	      pts[numpts].x = event->xbutton.x;
	      pts[numpts].y = event->xbutton.y;
	      numpts++;
	   }
	}
	else if (event->type == ButtonPress)
	{
	   if (event->xbutton.button == Button3)
	   {
	      extracted = TRUE;
	      return;
	   }
	   pts[numpts].x = event->xbutton.x;
	   pts[numpts].y = event->xbutton.y;

	   if (numpts > 0 &&
	       kabs(pts[numpts].x - pts[numpts-1].x) < 5 &&
	       kabs(pts[numpts].y - pts[numpts-1].y) < 5)
	   {
	      done = TRUE;
	   }
	   else numpts++;
	}
	else if (event->type == ButtonRelease && done == TRUE)
	{
	   extracted = TRUE;
	   if (erase_selection == FALSE)
	      return;

	   for (i = 1; i < numpts; i++)
	   {
	      XDrawLine(display, window, gc, pts[i-1].x, pts[i-1].y,
			pts[i].x, pts[i].y);
	   }
	}

        if (numpts > 0)
        {
           /*
            *  Erase the old line.
            */
	   xpos1 = pts[numpts-1].x; ypos1 = pts[numpts-1].y;
	   if (x != -1 && y != -1)
	      XDrawLine(display, window, gc, xpos1, ypos1, x, y);

           /*
            *  Draw the new line.
            */
	   if (extracted == FALSE && xpos2 != -1 && ypos2 != -1)
	   {
	      XDrawLine(display, window, gc, xpos1, ypos1,
			xpos2, ypos2);
	   }
        }
}

/*-----------------------------------------------------------
|
|  Routine Name: freehand_handler - interactive paints a freehand extent handler
|
|       Purpose: interactively getting extents for a freehand for a given
|		 object, by drawing a set of points freehand
|
|         Input: object	     - object in which the event was placed
|		 client_data - the GC to be used
|		 event       - the event that triggered this handler
|
|    Written By: Mark Young
|          Date: Jul 30, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void freehand_handler(
    xvobject object,
    kaddr    client_data,
    XEvent   *event,
    int	     *dispatch)
{
	GC gc = (GC) client_data;
	Display *display = xvw_display(object);
	Window  window;

	int i, x, y;

	*dispatch = FALSE;
	if (xvw_check_subclass(object, InputOnlyWidgetClass) == TRUE)
	   window = xvw_window(xvw_parent(object));
	else
	   window = xvw_window(object);

	if (event->type == ButtonRelease && numpts == 0)
	   return;

	if (numpts >= maxpts)
	{
	   maxpts += KLENGTH;
	   pts = (XPoint *) krealloc(pts, sizeof(XPoint) * maxpts);
	}

	if (event->type == MotionNotify)/* && numpts > 0)*/
	{
	   x = event->xmotion.x;
	   y = event->xmotion.y;
	}
	else if (event->type == ButtonPress || event->type == ButtonRelease)
	{
	   if (event->xbutton.button == Button3)
	   {
	      extracted = TRUE;
	      return;
	   }
	   x = event->xbutton.x;
	   y = event->xbutton.y;
	}
	else
	   return;

	if (numpts > 0)
	{
	   if (pts[numpts-1].x != x || pts[numpts-1].y != y)
	   {
	      XDrawLine(display,window,gc,pts[numpts-1].x,pts[numpts-1].y,x,y);
	      pts[numpts].x = x;
              pts[numpts++].y = y;
	   }
	}
	else
	{
	   pts[numpts].x = x;
           pts[numpts++].y = y;
	}

	if (event->type == ButtonRelease && numpts > 1)
	{
	   extracted = TRUE;
	   if (erase_selection == FALSE)
	      return;

	   for (i = 1; erase_selection && i < numpts; i++)
	      XDrawLine(display, window, gc, pts[i-1].x, pts[i-1].y,
			pts[i].x, pts[i].y);
	}
}


/*-----------------------------------------------------------
|
|  Routine Name: getextent - driver for setting up the get extent routine
|
|       Purpose: initializes getting the different types of extents.
|
|         Input: object - the object to get the extent for
|		 erase  - whether to erase the final selection or leave it
|		 type   - the type of extent to extract for
|    Written By: Mark Young
|          Date: Jul 30, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static int getextent(
   xvobject object,
   int      erase,
   int	    type)
{
	GC         gc;
	XGCValues  values;
	unsigned   long mask;
	Widget     widget;
	Cursor     cursor = None;
	static     int inuse = FALSE;
	kfunc_void extent_handler = NULL;


	if (object == NULL)
	   return(FALSE);

	inuse = TRUE;
	if (xvw_check_subclass(object, InputOnlyWidgetClass) == TRUE)
	   widget = xvw_widget(xvw_parent(object));
	else
	   widget = xvw_widget(object);

        /* set up the graphics context to be used with graphics windows */
        values.line_width     = 2;
        values.function       = GXinvert;
        values.subwindow_mode = IncludeInferiors;
        mask = GCLineWidth | GCFunction | GCSubwindowMode;
        gc = XtGetGC(widget, mask, &values);

        /*
         *  Add the event handler to rubberband on the object.
         */
	if (type == KPOLYLINE)
           mask = ButtonPressMask | ButtonReleaseMask | PointerMotionMask;
	else
           mask = ButtonPressMask | ButtonReleaseMask | ButtonMotionMask;

	switch (type)
	{
	   case KRECTANGLE:
		extent_handler = rectangle_handler;
		break;
	   case KCIRCLE:
		extent_handler = circle_handler;
		break;
	   case KELLIPSE:
		extent_handler = ellipse_handler;
		break;
	   case KLINE:
		extent_handler = line_handler;
		break;
	   case KPOLYLINE:
		extent_handler = polyline_handler;
		break;
	   case KPAINT:
		extent_handler = freehand_handler;
		break;
	}
	xvw_insert_event(object, (int) mask, extent_handler, gc, KLIST_HEAD);

	/*
	 *  Save and then change the cursor
	 */
	xvw_busy(xvw_toplevel(object), FALSE);
	xvw_get_attribute(object, XVW_CURSOR, &cursor);
	xvw_set_attribute(object, XVW_CURSORNAME, "hand1");

	/*
	 *  Process events until the portion of the screen has been extracted
	 */
	erase_selection = erase;
	extracted = FALSE;
	while (extracted == FALSE)
	   xvw_process_event();

	/*
	 *  Remove the handler and free the gc
	 */
	inuse = FALSE;
	xvw_set_attribute(object, XVW_CURSOR, cursor);
	xvw_remove_event(object, (int) mask, extent_handler, gc);
	XtReleaseGC(widget, gc);
	return(TRUE);
}


/************************************************************
*
*  Routine Name: xvw_getextent_rectangle - returns the rectangular extent of
*					   an object
*
*       Purpose: interactively extracts a rectangular extents of an object.
*
*         Input: object - the object in which to get extents, if NULL then
*			  from the RootWindow.
*		 erase  - whether to erase the final selection or leave it
*        Output: x - the starting x position
*		 y - the starting y position
*		 w - the width of the extent
*		 h - the height of the extent
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions:
*    Written By: Mark Young & Danielle Argiro
*          Date: Dec 17, 1994
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int xvw_getextent_rectangle(
   xvobject object,
   int      erase,
   int          *x,
   int          *y,
   unsigned int *w,
   unsigned int *h)
{
	xpos1 = xpos2 = ypos1 = ypos2 = -1;
	if (getextent(object, erase, KRECTANGLE) == FALSE)
	   return(FALSE);

	if (xpos2 == -1) xpos2 = xpos1;
	if (ypos2 == -1) ypos2 = ypos1;
	if (x) *x = kmin(xpos1,  xpos2);
	if (y) *y = kmin(ypos1,  ypos2);
	if (w) *w = kabs(xpos1 - xpos2);
	if (h) *h = kabs(ypos1 - ypos2);

	if ((xpos1 == -1 && ypos1 == -1) || (*w == 0 && *h == 0))
	   return(FALSE);

	return(TRUE);
}

/************************************************************
*
*  Routine Name: xvw_getextent_circle - returns the circular extent of
*					   an object
*
*       Purpose: interactively extracts a circular extents of an object.
*
*         Input: object - the object in which to get extents, if NULL then
*			  from the RootWindow.
*		 erase  - whether to erase the final selection or leave it
*        Output: x - the center x position
*		 y - the center y position
*		 radius - the radius of the circle
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions:
*    Written By: Mark Young & Danielle Argiro
*          Date: Dec 17, 1994
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int xvw_getextent_circle(
   xvobject object,
   int      erase,
   int          *x,
   int          *y,
   unsigned int *radius)
{
	int rx, ry;

	xpos1 = xpos2 = ypos1 = ypos2 = -1;
	if (getextent(object, erase, KCIRCLE) == FALSE)
	   return(FALSE);

	if (x) *x = xpos1;
	if (y) *y = ypos1;
        rx = xpos1 - xpos2; ry = ypos1 - ypos2;
	if (radius) *radius = ksqrt((double) (rx*rx + ry*ry));

	if ((xpos1 == -1 && ypos1 == -1) || (*radius == 0))
	   return(FALSE);

	return(TRUE);
}

/************************************************************
*
*  Routine Name: xvw_getextent_ellipse - returns the elliptical extent of
*					   an object
*
*       Purpose: interactively extracts a elliptical extents of an object.
*
*         Input: object - the object in which to get extents, if NULL then
*			  from the RootWindow.
*		 erase  - whether to erase the final selection or leave it
*        Output: x - the center x position
*		 y - the center y position
*		 wradius - the width radius of the ellipse
*		 hradius - the height radius of the ellipse
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions:
*    Written By: Mark Young & Danielle Argiro
*          Date: Dec 17, 1994
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int xvw_getextent_ellipse(
   xvobject object,
   int      erase,
   int          *x,
   int          *y,
   unsigned int *wradius,
   unsigned int *hradius)
{
	xpos1 = xpos2 = ypos1 = ypos2 = -1;
	if (getextent(object, erase, KELLIPSE) == FALSE)
	   return(FALSE);

	if (x) *x = xpos1;
	if (y) *y = ypos1;
	if (wradius) *wradius = kabs(xpos1 - xpos2);
	if (hradius) *hradius = kabs(ypos1 - ypos2);

	if ((xpos1 == -1 && ypos1 == -1) || (*wradius == 0 && *hradius == 0))
	   return(FALSE);

	return(TRUE);
}

/************************************************************
*
*  Routine Name: xvw_getextent_line - returns the linear extent of an object
*
*       Purpose: interactively extracts a linear extent of an object.
*
*         Input: object - the object in which to get extents, if NULL then
*			  from the RootWindow.
*		 erase  - whether to erase the final selection or leave it
*        Output: xbegin - the x begin position
*		 ybegin - the y begin position
*		 xend   - the x end position
*		 yend   - the x end position
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions:
*    Written By: Mark Young & Danielle Argiro
*          Date: Dec 17, 1994
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int xvw_getextent_line(
   xvobject object,
   int      erase,
   int      *xbegin,
   int      *ybegin,
   int      *xend,
   int      *yend)
{
	xpos1 = xpos2 = ypos1 = ypos2 = -1;
	if (getextent(object, erase, KLINE) == FALSE)
	   return(FALSE);

	if (xpos2 == -1) xpos2 = xpos1;
	if (ypos2 == -1) ypos2 = ypos1;
	if (xend)   *xend   = xpos2;
	if (yend)   *yend   = ypos2;
	if (xbegin) *xbegin = xpos1;
	if (ybegin) *ybegin = ypos1;

	if ((xpos1 == -1 && ypos1 == -1) || (xpos1 == xpos2 && ypos1 == ypos2))
	   return(FALSE);

	return(TRUE);
}

/************************************************************
*
*  Routine Name: xvw_getextent_polyline - returns the polylne extent of an
*					  object
*
*       Purpose: interactively extracts a polyline extent of an object.
*
*         Input: object - the object in which to get extents, if NULL then
*			  from the RootWindow.
*		 erase  - whether to erase the final selection or leave it
*        Output: num - the number of points returned
*       Returns: malloc'ed XPoints on success, NULL on failure
*
*  Restrictions:
*    Written By: Mark Young & Danielle Argiro
*          Date: Dec 17, 1994
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

XPoint *xvw_getextent_polyline(
   xvobject object,
   int      erase,
   int      *num)
{
	XPoint *points;

	numpts = 0;
	xpos1 = xpos2 = ypos1 = ypos2 = -1;
	if (getextent(object, erase, KPOLYLINE) == FALSE)
	   return(FALSE);

	if (numpts <= 1)
	{
	   if (num) *num = 0;
	   return(NULL);
	}

	if (num) *num = numpts;
	points = (XPoint *) kdupalloc(pts, sizeof(XPoint) * numpts);
	return(points);
}

/************************************************************
*
*  Routine Name: xvw_getextent_freehand - returns the freehand draw extent of an
*					  object
*
*       Purpose: interactively extracts a set of points that the user freehand
*		 draws in which to extent an object.
*
*         Input: object - the object in which to get extents, if NULL then
*			  from the RootWindow.
*		 erase  - whether to erase the final selection or leave it
*        Output: num - the number of points returned
*       Returns: malloc'ed XPoints on success, NULL on failure
*
*  Restrictions:
*    Written By: Mark Young & Danielle Argiro
*          Date: Dec 17, 1994
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

XPoint *xvw_getextent_freehand(
   xvobject object,
   int	    erase,
   int      *num)
{
	XPoint *points;

	numpts = 0;
	xpos1 = xpos2 = ypos1 = ypos2 = -1;
	if (getextent(object, erase, KPAINT) == FALSE)
	   return(FALSE);

	if (numpts <= 1)
	{
	   if (num) *num = 0;
	   return(NULL);
	}

	if (num) *num = numpts;
	points = (XPoint *) kdupalloc(pts, sizeof(XPoint) * numpts);
	return(points);
}
