 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>         Attribute Utilities
   >>>>
   >>>>  Private:
   >>>>   Static:
   >>>>             xvw_set_arg()
   >>>>             xvw_get_arg()
   >>>>             xvw_arguments()
   >>>>   Public:
   >>>>             xvw_vset_attributes()
   >>>>             xvw_vget_attributes()
   >>>>             xvw_set_attributes()
   >>>>             xvw_get_attributes()
   >>>>             xvw_format_attribute()
   >>>>             xvw_init_attributes()
   >>>>             xvw_define_attributes()
   >>>>             xvw_define_constraints()
   >>>>
   >>>>             xvw_call_attrcallback()
   >>>>             xvw_add_attrcallback()
   >>>>             xvw_remove_attrcallback()
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"

static char temp_stack[KLENGTH];
static char *tstack = temp_stack;

/*-----------------------------------------------------------
|
|  Routine Name: xvw_find_subpart - given an attribute return the object
|				   it belongs to
|
|       Purpose: This routine handles subparts specified thru the actual
|		 attribute.
|         Input: object    - the object which the attribute was initially
|			     intended
|                attribute - the attribute which we are getting the object for
|	 Output: subname   - the return subname
|       Returns: return the object or NULL on error
|
|    Written By: Mark Young
|          Date: Nov 28, 1994
|
------------------------------------------------------------*/
/*ARGSUSED*/
static xvobject xvw_find_subpart(
   xvobject object,
   char     *attribute,
   char	    *subname)
{
	xvobject subpart = object;
	char   temp[KLENGTH], *indx, *name = attribute;


	while ((indx = kstrchr(name, '.')) != NULL)
	{
	   kstring_ncopy(name, indx - name, temp);
	   if (!xvw_get_attributes(subpart, temp, &subpart, NULL))
	   {
	      kerror("xvwidgets", "xvw_arguments", "Unknown \
attribute '%s'.  Unable to continue setting or getting the rest of the \
attributes list.", temp);
	      return(NULL);
	   }
	   name = indx+1;
	}
	kstrcpy(subname, name);
	return(subpart);
}

/*-----------------------------------------------------------
|
|  Routine Name: xvw_set_arg - set an argument
|
|       Purpose: This routine is used to set an attribute from
|		 the variable argument list and place it into
|		 the arg list.
|
|         Input: object - the object in which the attribute is associated
|		 name   - the attribute in which describes the attribute
|		 list   - the variable argument list in which to
|		 indx   - the index into the arg structure
|		 maxargs - the max number of entries for the arg structure
|		 args   - the arg structure that contains the entry
|
|        Output: none
|       Returns: TRUE (1) on success, FALSE (0) on failure of conversion or
|		 getting of the attribute, ERROR (-1) if the attribute does
|		 not exist
|
|    Written By: Mark Young
|          Date: Nov 11, 1992
| Modifications:
|
------------------------------------------------------------*/
/*ARGSUSED*/
static int xvw_set_arg(
   xvobject  object,
   kstring  name,
   kva_list *list,
   int     *indx,
   int     maxargs,
   Arg     args[])
{
	Arg	  *arg;
	XtArgVal  value;
	float     fvalue;
	double    dvalue;
	xvobject  subpart;
	char	  subname[KLENGTH], *attribute;

	XtPointer addr;
	XrmValue  from, to;
	Widget widget, parent;
	xvclass wclass, pclass;
	CompiledAttribute *temp;
	XrmQuark  quark, widres, xvwres;


	if ((subpart = xvw_find_subpart(object, name, subname)) != object)
	{
	   args = NULL;
	   indx = NULL;
	}
	quark     = XrmStringToQuark(subname);
	attribute = XrmQuarkToString(quark);
	widget = xvw_widget(subpart);
	parent = XtParent(widget);

	/*
	 *  Check to see if we found the compiled attribute and that the
	 *  xv class and object class types are not NULL Quarks.
	 */
	wclass = XtClass(widget);
	pclass = (parent && ((int) parent != -1)) ? XtClass(parent) : NULL;
	if ((temp = xvw_findcompiled(wclass, pclass, quark)) == NULL ||
	    (widres(temp) == NULLQUARK))
	{
	   return(ERROR);
	}
	xvwres = xvwres(temp);
	widres = widres(temp);

	if (temp->setroutine != NULL)
	{
	   if (args != NULL && *indx > 0)
	   {
	      XtSetValues(widget, args, (Cardinal) *indx);
	      *indx = 0;
	   }

	   if (xvwres == XtQfloat)
	   {
	      fvalue = (float) kva_arg(*list, double);
	      return(temp->setroutine(subpart, attribute, &fvalue));
	   }
	   else if (xvwres == XtQdouble)
	   {
	      dvalue = kva_arg((*list), double);
	      return(temp->setroutine(subpart, attribute, &dvalue));
	   }
	   else
	   {
	      value = kva_arg((*list), XtArgVal);
	      return(temp->setroutine(subpart, attribute, &value));
	   }
	}

	/*
	 *  Retrieve the value from the argument list.  If the value is
	 *  a floating point value then retrieve the value which is stored
	 *  as a double and then store it in "value".
	 */
	if (xvwres == XtQfloat)
	{
	   fvalue = (float) kva_arg((*list), double);
	   addr = (XtPointer) &fvalue;
	}
	else if (xvwres == XtQdouble)
	{
	   dvalue = (double) kva_arg((*list), double);
	   addr = (XtPointer) &dvalue;
	}
	else if (xvwres == XtQstring || xvwres == XtQfilename)
	{
	   addr = (XtPointer) kva_arg((*list), kstring);
	}
	else
	{
	   value = kva_arg((*list), XtArgVal);
	   addr = (XtPointer) &value;
	}

	/*
	 *  If the xv attribute class is not the same as that of the objects
	 *  then go ahead and call the converter to convert it.  If it turns
	 *  out that no such converter exists a warning will be printed...
	 */
	if (xvwres != widres)
	{
	   to.size = 0;
	   to.addr = NULL;
	   from.addr = addr;
	   from.size = sizeof(from.addr);
	   XtConvert(widget, XrmQuarkToString(xvwres), &from,
		     XrmQuarkToString(widres), &to);

	   if (widres == XtQfloat)
	   {
	      kmemcpy(&fvalue, to.addr, to.size);
	      addr = (XtPointer) &fvalue;
	   }
	   else if (widres == XtQdouble)
	   {
	      kmemcpy(&dvalue, to.addr, to.size);
	      addr = (XtPointer) &dvalue;
	   }
	   else if (widres == XtQstring || widres == XtQfilename)
	   {
	      addr = (XtPointer) to.addr;
	   }
	   else
	   {
	      kmemcpy(&value, to.addr, to.size);
	      addr = (XtPointer) &value;
	   }
	}


	if (widname(temp) != xvwname(temp))
	{
	   name = XrmQuarkToString(widname(temp));
	   if ((subpart = xvw_find_subpart(object, name, subname)) != object)
	   {
	      args = NULL;
	      indx = NULL;
	      widget = xvw_widget(subpart);
	   }
	   quark     = XrmStringToQuark(subname);
	   attribute = XrmQuarkToString(quark);
	}
	arg = (args != NULL) ? &args[*indx] : NULL;

	/*
	 *  Make sure that we don't extend past the maximum number of
	 *  allowable arguments.  If so then the answer is easy.  All
	 *  we need to do is call XtVaSetValues() in order to do a variable
	 *  arg XtSetValues().
	 */
	if (arg == NULL || *indx == maxargs)
	{
	   if (widres == XtQfloat && sizeof(float) > sizeof(XtArgVal))
	      value = (XtArgVal) &fvalue;
	   else if (widres == XtQfloat && sizeof(float) <= sizeof(XtArgVal))
	      kmemcpy(&value, &fvalue, sizeof(float));
	   else if (widres == XtQdouble && sizeof(double) > sizeof(XtArgVal))
	      value = (XtArgVal) &dvalue;
	   else if (widres == XtQdouble && sizeof(double) <= sizeof(XtArgVal))
	      kmemcpy(&value, &dvalue, sizeof(double));
	   else if (widres == XtQstring || widres == XtQfilename)
	      value = (XtArgVal) addr;

	   XtVaSetValues(widget, attribute, value, NULL);
	   xvw_call_attrcallback(subpart, attribute);
	   return(TRUE);
	}

	/*
	 *  If we are to store the argument into the "arg" array, then we
	 *  may need to store the address or the actual value, depending
	 *  on whether size of the type being stored is larger than the
	 *  XtArgVal (typically a long).  If so then we store the address
	 *  rather than the value.
	 */
	if (widres == XtQfloat)
	{
	   if (sizeof(float) > sizeof(XtArgVal))
	   {
	      kmemcpy(tstack, addr, sizeof(float));
	      value = (XtArgVal) tstack;
	      tstack += sizeof(float);
	   }
	   else kmemcpy(&value, &fvalue, sizeof(float));
	}
	else if (widres == XtQdouble)
	{
	   if (sizeof(double) > sizeof(XtArgVal))
	   {
	      kmemcpy(tstack, addr, sizeof(double));
	      value = (XtArgVal) tstack;
	      tstack += sizeof(double);
	   }
	   else kmemcpy(&value, &dvalue, sizeof(double));
	}
	else if (widres == XtQstring || widres == XtQfilename)
	   value = (XtArgVal) addr;

	XtSetArg(*arg, attribute, value); (*indx)++;
	xvw_call_attrcallback(subpart, attribute);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: xvw_get_arg - get an argument
|
|       Purpose: This routine is used to get an attribute from
|		 the variable argument list and place it into
|		 the arg list.
|
|         Input: object - the object in which the attribute is associated
|		 name   - the attribute in which describes the attribute
|		 list   - the variable argument list in which to
|		 indx   - the index into the arg structure
|		 maxargs - the max number of entries for the arg structure
|		 args   - the arg structure that contains the address
|
|        Output: none
|       Returns: TRUE (1) on success, FALSE (0) on failure of conversion or
|		 getting of the attribute, ERROR (-1) if the attribute does
|		 not exist
|
|    Written By: Mark Young
|          Date: Nov 11, 1992
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static int xvw_get_arg(
   xvobject  object,
   kstring   name,
   kva_list *list,
   int      *indx,
   int       maxargs,
   Arg       args[],
   int	    *junk)
{
	Arg	  *arg;
	XtArgVal  *addr;
	XtPointer value;
	double	  dvalue;
	xvobject  subpart;
	char	  subname[KLENGTH], *attribute;

	XrmValue  from, to;
	Widget  widget, parent;
	xvclass wclass, pclass;
	CompiledAttribute *temp;
	XrmQuark  quark, xvwres, widres;


	if ((subpart = xvw_find_subpart(object, name, subname)) != object)
	{
	   args = NULL;
	   indx = NULL;
	}
	widget = xvw_widget(subpart);
	parent = XtParent(widget);
	quark     = XrmStringToQuark(subname);
	attribute = XrmQuarkToString(quark);

	/*
	 *  Check to see if we found the compiled attribute and that the
	 *  xv class and object class types are not NULL Quarks.
	 */
	wclass = XtClass(widget);
	pclass = (parent && ((int) parent != -1)) ? XtClass(parent) : NULL;
	if ((temp = xvw_findcompiled(wclass, pclass, quark)) == NULL ||
	    (widres(temp) == NULLQUARK))
	{
	   return(ERROR);
	}
	xvwres = xvwres(temp);
	widres = widres(temp);

	if (temp->getroutine != NULL)
	{
	   addr = (XtArgVal *) kva_arg((*list), XtArgVal *);
	   return(temp->getroutine(subpart, attribute, addr));
	}

	/*
	 *  Retrieve the value from the argument list.  If the value is
	 *  a floating point value then retrieve the value which is stored
	 *  as a double and then store it in "value".
	 */
	addr = (XtArgVal *) kva_arg((*list), XtArgVal *);

	/*
	 *  If the xv attribute class is not the same as that of the objects
	 *  then go ahead and call the converter to convert it.  If it turns
	 *  out that no such converter exists a warning will be printed...
	 */
	if (widname(temp) != xvwname(temp))
	{
	   name = XrmQuarkToString(widname(temp));
	   if ((subpart = xvw_find_subpart(object, name, subname)) != object)
	   {
	      args = NULL;
	      indx = NULL;
	      widget = xvw_widget(subpart);
	   }
	   quark     = XrmStringToQuark(subname);
	   attribute = XrmQuarkToString(quark);
	}
	arg = (args != NULL) ? &args[*indx] : NULL;

	if (xvwres != widres)
	{
	   to.size   = sizeof(addr);
	   to.addr   = (char *) addr;

           if (widres == XtQdouble)
           {
	      from.addr = (char *) &dvalue;
	      from.size = sizeof(from.addr);
	      XtVaGetValues(widget, attribute, &dvalue, NULL);
	   }
	   else
           {
	      from.addr = (char *) &value;
	      from.size = sizeof(from.addr);
	      XtVaGetValues(widget, attribute, &value, NULL);
	   }
	   XtConvert(widget, XrmQuarkToString(widres), &from,
		     XrmQuarkToString(xvwres), &to);
	   kmemcpy(addr, to.addr, to.size);
	}
	else if (arg == NULL || *indx == maxargs)
	{
	   /*
	    *  Make sure that we don't extend past the maximum number of
	    *  allowable arguments.
	    */
	   XtVaGetValues(widget, attribute, addr, NULL);
	}
	else
	{
	   XtSetArg(*arg, attribute, addr); (*indx)++;
	}
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: xvw_arguments - general set/get argument routine
|
|       Purpose: This routine is used to race thru the variable
|		 argument list.  Calling either the xvw_set_arg()
|		 or xvw_get_arg() routine to actually set or get
|		 attributes for the argument list. 
|
|         Input: object - the object that the attributes are associated
|		 list   - the variable argument list
|		 args   - the Xt argument structure in which the
|			  values will be stored
|		 maxargs - maximum number of arguments
|		 routine - the routine to call (ie. xvw_set_arg or xvw_get_arg)
|
|        Output: arg - the Xt arg structure in which we set the attribute
|		 numargs - the max number of arguments in args array
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: Nov 11, 1992
| Modifications:
|
------------------------------------------------------------*/

static int xvw_arguments(
   xvobject  object,
   kva_list *list,
   Arg     *args,
   int     maxargs,
   int     (*routine)(),
   int     *numargs)
{
	String  name;
	int	val, i = 0, ret_val = TRUE;

	/*
	 *  Sanity check to see if there is a object to get and set the
	 *  arguments for.
	 */
	if (!object)
	   return(FALSE);

	if (numargs != NULL)
	   *numargs = 0;

	while ((name = kva_arg(*list, kstring)) != NULL)
	{
	    /*
	     *  Search the object's private attributes
	     */
	    if ((val = routine(object, name, list, &i, maxargs, args)) != ERROR)
	    {
	       if (val == FALSE) ret_val = FALSE;
	       continue;
	    }
	    else
	    {
	       kerror("xvwidgets", "xvw_arguments",
			"Unknown attribute '%s'.  Unable to continue with \
setting or getting the rest of the attributes list.", name);
	       if (numargs != NULL) *numargs = i;
	       return(FALSE);
	    }
	}

	if (numargs != NULL) *numargs = i;
	return(ret_val);
}


/************************************************************
*
*  Routine Name: xvw_vset_attributes - set attributes on an object
*                                      (variable argument list)
*
*       Purpose: Sets a variable number of attributes associated
*                with a GUI or visual object.  The difference between
*		 xvw_set_attributes and this routine is that
*		 xvw_vset_attributes takes an existing variable
*		 argument list (kva_list).
*
*         Input: object   - the object on which to set the attributes
*                kva_list - variable argument list, must be in the form:\f(CW
*		 !  XVW_ATTRIBUTE_1,   value1,
*		 !  XVW_ATTRIBUTE_2,   value2,
*		 !  XVW_ATTRIBUTE_3,   value3,
*                !  :
*                !  :
*		 !  XVW_ATTRIBUTE_N,   valueN,
*		 !  NULL\fP
*
*        Output:
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Nov 11, 1992
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: int xvw_vset_attributes(
*                !  xvobject object,
*                !  kva_list list)
*
*************************************************************/

/* VARARGS */
int xvw_vset_attributes(
   xvobject object,
   kva_list list)
{
	char	 *tmp;
	int	 num, value;
	Arg	 args[MAX_ARGS];

	/*
	 *  Sanity check...
	 */
	if (!object)
	{
	   kerror("xvwidgets", "xvw_vset_attributes", "xvobject is NULL. \
The object argument for xvobject's set attributes function needs to be an \
active (non-null) object");
	   return(FALSE);
	}

	/*
	 *  Grab the args off the variable argument list and set them into
	 *  the arguments array.
	 */
	tmp = tstack;
	value = xvw_arguments(object, &list, args, MAX_ARGS, xvw_set_arg, &num);
	if (num > 0) XtSetValues(xvw_widget(object), args, (Cardinal) num);
	tstack = tmp;

	/*
	 *  Update the menu in case it has been activated
	 */
	xvw_update_menu(object);
	return(value);
}

/************************************************************
*
*  Routine Name: xvw_vget_attributes - get attributes from an object 
*					(variable argument list)
*
*       Purpose: Gets a variable number of attributes associated
*                with a GUI or visual object.  The difference between
*                xvw_get_attributes and this routine is that
*                xvw_vget_attributes takes an existing variable
*                argument list (kva_list).
*
*         Input: object   - the object for which to get the attributes
*                kva_list - variable argument list, must be in the form:\f(CW
*		 !    XVW_ATTRIBUTE_1,   pointer1,
*		 !    XVW_ATTRIBUTE_2,   pointer2,
*		 !    XVW_ATTRIBUTE_3,   pointer3,
*                !    :
*                !    :
*		 !    XVW_ATTRIBUTE_N,   pointerN,
*		 !    NULL\fP
*        Output:
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Mark Young
*          Date: Nov 11, 1992
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: int xvw_vget_attributes(
*                !  xvobject object,
*                !  kva_list list)
*
*************************************************************/

/* VARARGS */
int xvw_vget_attributes(
   xvobject object,
   kva_list list)
{
	int	 num, value;
	Arg	 args[MAX_ARGS];

	/*
	 *  Sanity check...
	 */
	if (!object)
	{
	   kerror("xvwidgets", "xvw_vget_attributes", "xvobject is NULL. \
The object argument for xvobject's get attributes function needs to be an \
active (non-null) object");
	   return(FALSE);
	}

	/*
	 *  Grab the args off the variable argument list and set them into
	 *  the arguments array.
	 */
	value = xvw_arguments(object, &list, args, MAX_ARGS, xvw_get_arg, &num);
	if (num > 0) XtGetValues(xvw_widget(object), args, (Cardinal) num);

	return(value);
}

/************************************************************
*
*  Routine Name: xvw_set_attributes - set attributes on an object
*                                     (variable argument list)
*
*       Purpose: Sets a variable number of attributes associated
*                with a GUI or visual object.
*
*         Input: object   - the object on which to set the attributes
*                kvalist  - variable argument list, must be in the form:\f(CW
*		 !    XVW_ATTRIBUTE_1,   value1,
*		 !    XVW_ATTRIBUTE_2,   value2,
*		 !    XVW_ATTRIBUTE_3,   value3,
*                !    :
*                !    :
*		 !    XVW_ATTRIBUTE_N,   valueN,
*		 !    NULL\fP
*
*        Output:
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Nov 11, 1992
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: int xvw_set_attributes(
*                !  xvobject object,
*                !  kvalist)
*
*************************************************************/

/* VARARGS */
int xvw_set_attributes(
   xvobject object,
   kvalist)
{
	kva_list list;


	/*
	 *  Set the argument list to start after the object argument, and
	 *  then call xvw_vset_attributes() to do the actual work...
	 */
	kva_start(list, object);
	return(xvw_vset_attributes(object, list));
}

/************************************************************
*
*  Routine Name: xvw_get_attributes - get attributes from an object
*                                     (variable argument list)
*
*       Purpose: Gets a variable number of attributes associated
*                with a GUI or visual object.
*
*         Input: object   - the object for which to get the attributes
*                kvalist  - variable argument list, must be in the form:\f(CW
*		 !    XVW_ATTRIBUTE_1,   pointer1,
*		 !    XVW_ATTRIBUTE_2,   pointer2,
*		 !    XVW_ATTRIBUTE_3,   pointer3,
*                !    :
*                !    :
*		 !    XVW_ATTRIBUTE_N,   pointerN,
*		 !    NULL\fP
*        Output:
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Mark Young
*          Date: Nov 11, 1992
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: int xvw_get_attributes(
*                !  xvobject object,
*                !  kvalist)
*
*************************************************************/

/* VARARGS */
int xvw_get_attributes(
   xvobject object,
   kvalist)
{
	kva_list list;


	/*
	 *  Set the argument list to start after the object argument, and
	 *  then call xvw_vget_attributes() to do the actual work...
	 */
	kva_start(list, object);
	return(xvw_vget_attributes(object, list));
}

/************************************************************
*
*  Routine Name: xvw_format_attribute - formatted set attribute 
*                                       (variable argument list)
*
*       Purpose: This routine sets a single attribute for a visual or GUI
*		 object.  First the attribute's format and arguments are
*		 passed to the kvsprintf().  Afterwards the attribute
*		 and value is passed to xvw_set_attribute() in order
*		 pass the attribute and formatted string to the visual
*		 or GUI object.
*
*         Input: object    - the object on which to set the attributes
*                attribute - the attribute name
*                format    - the format to be passed to kvsprintf()
*                kvalist  - variable argument list, which are the argument
*			     values to be passed to kvsprintf().
*        Output:
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: a 40K temporary format buffer is used, so any formatting of
*		 strings larger than this may result in a bus error.
*    Written By: Mark Young
*          Date: Jan 30, 1994
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: int xvw_format_attribute(
*                !  xvobject object,
*                !  char     *attribute,
*                !  char     *format,
*                !  kvalist)
*
*************************************************************/

/* VARARGS */
int xvw_format_attribute(
   xvobject object,
   char     *attribute,
   char     *format,
   kvalist)
{
	kva_list list;
	char	 temp[KLENGTH*20];


	/*
	 *  Set the argument list to start after the object argument, and
	 *  then call xvw_vset_attributes() to do the actual work...
	 */
	kva_start(list, format);
	if (!format || kvsprintf(temp, format, list) <= 0)
	{
           kerror("xvwidgets", "xvw_format_attribute", "Error in formatting \
object '%s' with a format of '%s'.  Unable to continue with setting \
attribute '%s'.", xvw_name(object), format, attribute);
           return(FALSE);
	}
	return(xvw_set_attribute(object, attribute, temp));
}

/************************************************************
*
*  Routine Name: xvw_init_attributes - initialize the attributes to be 
*                                      associated with a object class
*
*       Purpose: Allows the advanced GUI/visual object programmer to 
*                associate default attributes with a new object class.  
*                The constraint attributes are used by constraint objects.
*
*         Input: widget_class - the object class to associate the
*				attributes and constraint attributes
*                attributes   - the object attribute structure
*                num_attributes - the number of object attributes
*                attributes   - the constraint object attribute structure
*                num_attributes - the number of constraint object attributes
*		 menuform     - the internal menu associated with a glyph
*
*        Output:
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Mark Young
*          Date: Nov 10, 1992
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int xvw_init_attributes(
   xvclass     widget_class,
   xvattribute *attributes,
   int         num_attributes,
   xvattribute *constraints,
   int         num_constraints,
   char        *menuform)
{
	if (widget_class == NULL)
	{
	   kerror("xvwidgets", "xvw_init_attributes",
		  "xvobject Class is set to NULL therefore no compiled \
attributes can be retrieved");
	   return(FALSE);
	}

	xvw_addcompiled_attributes(widget_class, attributes,
	       		num_attributes);
	xvw_addcompiled_constraints(widget_class, constraints,
	       		num_constraints);
	xvw_set_menuinfo((xvobject) widget_class, XVW_MENU_FORMFILE, &menuform);
	return(TRUE);
}

/************************************************************
*
*  Routine Name: xvw_define_attributes - add user defined attributes
*
*       Purpose: Allows the advanced application programmer to add
*		 attributes for a particular object class.  This allows 
*                the addition of attributes to a object class or if NULL 
*                then a general set of attributes.  When the "set" or 
*                "get" attribute routines are called then the corresponding 
*                routine is called.
*
*         Input: widget_class - the widget class for which to define the 
*                               attributes
*                kvalist - variable argument list, in the form:\f(CW
*		 !  XVW_ATTRIBUTE_1, resource_type, setroutine, getroutine,
*		 !  XVW_ATTRIBUTE_2, resource_type, setroutine, getroutine,
*		 !  XVW_ATTRIBUTE_3, resource_type, setroutine, getroutine,
*		 !  NULL\fP
*
*        Output:
*       Returns: TRUE (1) on success, FALSE (0) on failure
*
*  Restrictions: Restrictions on data or input as applicable
*    Written By: Mark Young
*          Date: Nov 11, 1992
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

/* VARARGS */
int xvw_define_attributes(
   xvclass widget_class,
   kvalist)
{
	kva_list   list;
	String	  name, resource_type;
	kfunc_int setroutine, getroutine;


	/*
	 *  Set the argument list to start after the object argument
	 */
	kva_start(list, widget_class);

	while ((name = kva_arg(list, kstring)) != NULL)
	{
	   resource_type = kva_arg(list, kstring);
	   setroutine    = kva_arg(list, kfunc_int);
	   getroutine    = kva_arg(list, kfunc_int);
	   xvw_definecompiled_attributes(widget_class, name, resource_type,
			setroutine, getroutine);
	}

	/*
	 *  Terminate the argument list
	 */
	kva_end(list);
	return(TRUE);
}

/************************************************************
*
*  Routine Name: xvw_define_constraints - add user defined constraint attributes
*
*       Purpose: Allows the advanced application programmer to add
*		 attributes for a particular object class.  This allows 
*                the addition of attributes to a object class or if NULL 
*                then a general set of attributes.  When the "set" or 
*                "get" attribute routines are called then the corresponding 
*                routine is called.
*
*         Input: widget_class - the widget class for which to define the
*				constraint attributes
*                kvalist - variable argument list, in the form:\f(CW
*		 !  XVW_ATTRIBUTE_1, resource_type, setroutine, getroutine,
*		 !  XVW_ATTRIBUTE_2, resource_type, setroutine, getroutine,
*		 !  XVW_ATTRIBUTE_3, resource_type, setroutine, getroutine,
*		 !  NULL\fP
*
*        Output:
*       Returns: TRUE (1) on success, FALSE (0) on failure
*
*  Restrictions: Restrictions on data or input as applicable
*    Written By: Mark Young
*          Date: Dec 06, 1994
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

/* VARARGS */
int xvw_define_constraints(
   xvclass widget_class,
   kvalist)
{
	kva_list   list;
	String	  name, resource_type;
	kfunc_int setroutine, getroutine;


	/*
	 *  Set the argument list to start after the object argument
	 */
	kva_start(list, widget_class);

	while ((name = kva_arg(list, kstring)) != NULL)
	{
	   resource_type = kva_arg(list, kstring);
	   setroutine    = kva_arg(list, kfunc_int);
	   getroutine    = kva_arg(list, kfunc_int);
	   xvw_definecompiled_constraints(widget_class, name, resource_type,
			setroutine, getroutine);
	}

	/*
	 *  Terminate the argument list
	 */
	kva_end(list);
	return(TRUE);
}


/*
 *  We need to keep track of all the gadget events handlers.  We would rather
 *  simply add it to the gadget's event table, but since there is no way
 *  for our dispatch routine to dig into the gadget's event table.  The
 *  definition of XtEventTable is hidden inside the Xt library.  This is
 *  unfortunate, since it will greatly complicate what should be an easy
 *  extension. ;-(
 */
typedef struct {
	XrmQuark    name;
	XtPointer   client_data;
	kfunc_void  callback;
} HandlerStruct;

static klist *handlerlist = NULL;

/*-------------------------------------------------------------
|
|  Routine Name: xvw_add_attrcallback - add an attribute callback on an object
|
|       Purpose: Adds a callback to an object that will be fired if
|		 the specified attribute has its value changed.
|		 For example, an attribute callback might be added
|		 to a plot3D object that would be fired if the 
|		 XVW_PLOT3D_PLOTTYPE attribute had its value changed.
|
|         Input: object      - the object on which to add the callback
|		 name        - the name of the attribute for which callback 
|			       is to be added
|		 callback    - the callback to be added
|		 client_data - client_data to pass to callback
|
|        Output: 
|       Returns: TRUE (1) on success, FALSE (0) on failure
|
|  Restrictions: Restrictions on data or input as applicable
|    Written By: Mark Young
|          Date: Nov 11, 1992
|      Verified:
|  Side Effects:
| Modifications:
|
-------------------------------------------------------------*/

/* 
 * NOTE: interface needs to change to take an object, all three routines
 *       need testing, then header needs to change to PUBLIC.
 */
int xvw_add_attrcallback(
   xvclass widget_class,
   char        *name,
   void        (*callback)(),
   kaddr       client_data)
{
	HandlerStruct *entry;
	klist	*temp, *namelist;
	XrmQuark quark = XrmStringToQuark(name);

	/*
	 *
	 */
	if ((temp = klist_locate(handlerlist, widget_class)) == NULL)
	{
	   handlerlist = klist_add(handlerlist, widget_class, NULL);
	   temp = klist_locate(handlerlist, widget_class);
	}

	if ((namelist = klist_locate(temp, (kaddr) quark)) == NULL)
	{
	   if ((entry = (HandlerStruct *) kcalloc(1,
			sizeof(HandlerStruct))) == NULL)
	   {
	      kerror("xvwidgets", "xvw_add_attrcallback",
		     "Cannot allocate memory for xvw_detect structure");
	      return(FALSE);
	   }
	   temp->client_data = (kaddr) klist_add(temp->client_data,
					(kaddr) quark, entry);
	}
	else
	   entry = (HandlerStruct *) namelist->client_data;

	entry->name       = quark;
	entry->callback   = callback;
	entry->client_data = client_data;
	return(TRUE);
}

/*------------------------------------------------------------
|
|  Routine Name: xvw_call_attrcallback - call an attribute callback for an
|				         object
|
|       Purpose: Calls attribute callbacks that are currently installed
|		 on an object
|
|         Input: object - the object for which to call installed
|			  attribute callbacks
|		 name   - name of the attribute for which to call
|		          installed attribute callbacks
|
|        Output:
|       Returns: TRUE (1) on success, FALSE (0) on failure
|
|  Restrictions: Restrictions on data or input as applicable
|    Written By: Mark Young
|          Date: Nov 11, 1992
|      Verified:
|  Side Effects:
| Modifications:
|
-------------------------------------------------------------*/

/*
 * NOTE: interface needs to change to take an object, all three routines
 *       need testing, then header needs to change to PUBLIC.
 */
int xvw_call_attrcallback(
   xvobject object,
   char   *name)
{
	HandlerStruct *entry;
	klist	*temp, *namelist = NULL;
	XrmQuark quark = XrmStringToQuark(name);
	xvclass widget_class = xvw_class(object);

	/*
	 *
	 */
	if ((temp = klist_locate(handlerlist, widget_class)) != NULL)
           namelist = klist_locate(temp->client_data, (kaddr) quark);

	if (!namelist && (temp = klist_locate(handlerlist, NULL)) != NULL)
           namelist = klist_locate(temp->client_data, (kaddr) quark);

	if (namelist != NULL)
	{
	   entry = (HandlerStruct *) namelist->client_data;
	   if (entry->callback)
	      entry->callback(object, name, entry->client_data);

	   return(TRUE);
	}
	return(FALSE);
}

/*-------------------------------------------------------------
|
|  Routine Name: xvw_remove_attrcallback - remove attribute callbacks currently
|					   installed on an object
|
|       Purpose: Removes any attribute callbacks that are currently
|		 installed on an object
|
|         Input: object - the object from which to remove attribute callbacks
|		 name   - name of attribute for which attribute callbacks
|		          are installed
|
|        Output:
|       Returns: TRUE (1) on success, FALSE (0) on failure
|
|  Restrictions: Restrictions on data or input as applicable
|    Written By: Mark Young
|          Date: Nov 11, 1992
|      Verified:
|  Side Effects:
| Modifications:
|
|-----------------------------------------------------------*/

/*
 * NOTE: interface needs to change to take an object and the attribute, 
 *       all three routines need testing, then header needs to change to PUBLIC.
 */
/* ARGSUSED */
int xvw_remove_attrcallback(
   xvclass widget_class)
{
	return(FALSE);
}
