 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>                Display xvobject                         <<<<
   >>>>  Private:                                             <<<<
   >>>>   Static:                                             <<<<
   >>>>                xvu_update_display_contents()          <<<<
   >>>>                xvu_quit_display()                     <<<<
   >>>>   Public:                                             <<<<
   >>>>                xvu_create_display()		      <<<<
   >>>>                xvu_update_display()	              <<<<
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"	

static int  xvu_display_count = 0;

static void xvu_quit_display            PROTO((xvobject, kaddr, kaddr));
static int  xvu_update_display_contents PROTO((int, char *, kaddr));

/************************************************************
*
*  Routine Name: xvu_create_display - display ascii file
*
*       Purpose: Creates a large read-only text object in which an ascii
*		 file is displayed, until the user clicks on "Quit".
*		 Will not block events from the application program.
*                If desired, the contents of the text object may be
*                updated automatically as the file is changed.
*		 A new file may be displayed in the text object 
*		 with the use of xvu_update_display().
*
*         Input: filename - Character string containing the full path 
*			    to the ascii file to be displayed.  The 
*                           filename may be specified using a tilda (~).
*
*                display_label - A short label for top of object. Passing 
*                                NULL gives default of "Display File".
*                                Often, the filename is used as the label.
*
*                update_time - If zero (0), the contents of the text object 
*                              will NOT be updated if the file is changed.
*                              If non-zero, this is the time (in seconds) 
*                              specifying how often the file is to be checked
*                              for automatic update of the object contents.
*
*        Output: None.
*	Returns: On success, returns a pointer to a DisplayWid_Data 
*                structure that may be passed to xvu_update_display().
*                Returns NULL on failure.
*
*  Restrictions: none
*    Written By: Danielle Argiro
*          Date: Jul 16, 1992 13:50
*      Verified:
*  Side Effects:
* Modifications: Converted from Khoros 1.0 (DA)
*
*************************************************************/

#define DISPLAY_WIDTH  65
#define DISPLAY_HEIGHT 20

DisplayWid_Data *xvu_create_display(
   char  *filename,
   char  *display_label,
   float update_time)
{
	xvobject        quit;	  
	char            name[KLENGTH];
	float           width, height;
	DisplayWid_Data *display_data;

	display_data = (DisplayWid_Data *) kcalloc(1, sizeof(DisplayWid_Data));

	if (filename == NULL)
	{
	    errno = KNULL_PARAMETER;
	    kerror("xvutils", "xvu_create_display","'filename' cannot be NULL");
	    return(NULL);
	}

	if (display_label == NULL)
	    display_label = kstring_copy("Display File", NULL);

	/*
	 *  make sure that specified file is available
	 */
	if (kaccess(filename, R_OK) != 0)
	{
	    kerror("xvutils", "xvu_create_display",
	           "Non-existent or unreadable file %s", filename); 
	    return(NULL);
	}
	display_data->filename = kstring_copy(filename, NULL);

	if (update_time < 0)
	{
	    errno = KINVALID_PARAMETER;
	    kerror("xvutils","xvu_create_display","'update_time' must be >= 0");
	    return(NULL);
	}

	/*
	 * create identifying object name
	 */
	ksprintf(name, "%d_display", xvu_display_count++);

	/* 
	 *  create the display's toplevel object, and
	 *  add it to the list of toplevels
	 *  that is used with journal playback.
 	 */
	display_data->toplevel = xvw_create_transient_shell(name, NULL, NULL);

	/*
	 * set update time
	 */
	display_data->update_time = update_time;

	/* 
	 * create the backplane object 
	 */
	width  = (float) DISPLAY_WIDTH;
	height = (float) DISPLAY_HEIGHT;
	display_data->back = xvw_create_manager(display_data->toplevel, "back");
	xvw_set_attributes(display_data->back,
		XVW_CHAR_WIDTH, width,
		XVW_CHAR_HEIGHT, height,
		NULL);
	
        /*
         * create the label object
         */
	width = (float) DISPLAY_WIDTH - 13.0;

        display_data->label = xvw_create_label(display_data->back, "label");
        xvw_set_attributes(display_data->label,
		XVW_LABEL,             display_label,      /* label         */
		XVW_CHAR_WIDTH,        width,              /* set width     */
                XVW_MAP_WHEN_MANAGED,  TRUE,               /* mapped        */
		XVW_LABEL_JUSTIFY,     KLABEL_JUSTIFY_CENTER, /* centered   */
                NULL);

        /*
         * create the quit button
         */
        quit = xvw_create_button(display_data->back, "quit");
        xvw_set_attributes(quit, 
		XVW_LABEL,           "Quit",            /* button label   */
		XVW_LEFT_OF,         NULL,              /* upper R corner */
		XVW_BELOW,           NULL,              /* upper R corner */
		XVW_CHAR_WIDTH,      6.0, 		/* set width      */
		XVW_CHAR_HEIGHT,     1.0, 		/* set height     */
		NULL);
   	xvw_add_callback(quit, XVW_BUTTON_SELECT, 
		 	 xvu_quit_display, display_data);

	
	display_data->text = xvw_create_textdisplay(display_data->back,"display_text");
        xvw_set_attributes(display_data->text,
		XVW_TEXTDISPLAY_FILENAME, display_data->filename,
		XVW_BELOW,		  quit,		/* below "quit"  */
		XVW_TACK_EDGE,		  KMANAGER_TACK_ALL,
		NULL);


	/*
	 * map display object & return
	 */
	xvw_add_protocol(display_data->toplevel, "WM_DELETE_WINDOW", 
			 xvu_quit_display, (kaddr) display_data);

	if (update_time > 0)
	    xvw_add_detectfile(NULL, display_data->filename, update_time, 
			       xvu_update_display_contents, 
			      (kaddr) display_data); 

	xvw_realize(display_data->toplevel);
	return(display_data);
}


/************************************************************
*
*  Routine Name: xvu_update_display - change display to another file
*
*       Purpose: Changes the file that is currently displayed 
*                in a display object that was created with 
*                xvu_create_display().
*
*         Input: display_data - the DisplayWid_Data structure that was
*			        passed back to the application by 
*				xvu_create_display.  
*
*		 filename  - Character string containing the full path
*                            to the ascii file to be displayed.  The
*                            filename may be specified using a tilda (~).
*
*                update_time - If zero (0), the contents of the text object 
*                              will NOT be updated if the file is changed.
*                              If non-zero, this is the time (in seconds) 
*                              specifying how often the file is to be checked
*                              for automatic update of the object contents.
*
*        Output:
*	Returns: TRUE on success, FALSE on failure
*  Restrictions: none
*    Written By: Danielle Argiro
*          Date: Jul 16, 1992 13:50
*      Verified:
*  Side Effects:
* Modifications: Converted from Khoros 1.0 (DA)
*
*************************************************************/

int xvu_update_display(
   DisplayWid_Data *display_data,
   char            *filename,
   float           update_time)
{
	char  temp[KLENGTH];

	/*
	 *  make sure that specified file is available
	 */
	if (kaccess(filename, R_OK) != 0)
	{
	    ksprintf(temp, "Unable to find file '%s' to display", filename);
	    errno = KINVALID_FILE;
	    kerror("xvutils", "xvu_update_display", temp);
	    return(FALSE);
	}

	if (update_time < 0)
	{
	    errno = KINVALID_PARAMETER;
	    kerror("xvutils", "xvu_create_display", 
		    "'update_time' must be >= 0");
	    return(FALSE);
	}

	if (display_data->filename != NULL) 
	{
	    xvw_remove_detectfile(NULL, display_data->filename, 
				  xvu_update_display_contents,
				  (kaddr) display_data);
	    kfree (display_data->filename);
	}
	display_data->filename    = kstring_copy(filename, NULL);
	display_data->update_time = update_time;

	
	if (update_time > 0)
	    xvw_add_detectfile(NULL, display_data->filename, 
			       update_time, xvu_update_display_contents, 
			       (kaddr) display_data); 

	xvu_update_display_contents(0, display_data->filename, display_data);
	return(TRUE);
}

/*--------------------------------------------------------------
|
|  Routine Name: xvu_update_display_contents()
|
|       Purpose: Input handler which does the actual re-displaying of the 
|                contents of the display object. Is also called
|                automatically when the contents of the file being
|                displayed register a change with xvw_detect_file().  
|
|         Input: id          - Detect id; not used
|                filename    - name of file being displayed; not used
|                client_data - stores the display_data structure
|        Output: Returns TRUE to make xvw_detect_file happy.
|          Date: Jul 8, 1992
|    Written By: Danielle Argiro
| Modifications: Converted from Khoros 1.0 (DA)
|
--------------------------------------------------------------------*/
/* ARGSUSED */
static int xvu_update_display_contents(
   int   id,
   char  *filename,
   kaddr client_data)
{
	DisplayWid_Data *display_data = (DisplayWid_Data *) client_data;
	
	/* update text with changed file contents */
        xvw_set_attributes(display_data->text, XVW_TEXTDISPLAY_FILENAME,
                           display_data->filename,  NULL);
	XFlush(xvw_display(display_data->text));
	return(TRUE);
}

/*--------------------------------------------------------------
|
|  Routine Name: xvu_quit_display()
|
|       Purpose: Quits the display object
|
|         Input: object      - the quit button
|                client_data - holds the DisplayWid_Data structure
|                call_data   - unused
|
|        Output: none
|          Date: Jul 8, 1992
|    Written By: Danielle Argiro
| Modifications: Converted from Khoros 1.0 (DA)
|
-------------------------------------------------------------*/
/* ARGSUSED */
static void xvu_quit_display(
   xvobject object,
   kaddr  client_data,
   kaddr  call_data)
{
	DisplayWid_Data *display_data = (DisplayWid_Data *) client_data;

	xvw_remove_protocol(display_data->toplevel, "WM_DELETE_WINDOW", 
			    xvu_quit_display, display_data);

	xvw_remove_detectfile(NULL, display_data->filename, 
			      xvu_update_display_contents, 
			      (kaddr) display_data);

	xvw_destroy(display_data->label);
	xvw_destroy(display_data->text);
	xvw_destroy(display_data->back);
	xvw_destroy(display_data->toplevel);

	display_data->label= NULL;
	display_data->text= NULL;
	display_data->back= NULL;
	display_data->toplevel = NULL;
}
