 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            General Utility Routines
   >>>>
   >>>>   Static:
   >>>>		   place_handler
   >>>>  Private:
   >>>>   Public:
   >>>>            xvw_getposition
   >>>>            xvw_getobject
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"
#include <X11/cursorfont.h>


static int extracted, xpos, ypos;
static unsigned int width, height;

/*-----------------------------------------------------------
|
|  Routine Name: place_handler - place an object handler
|
|       Purpose: interactively getting extents or positioning an extent
|		 within a given object, by drawing a rubber-band rectangle.
|
|         Input: object	     - object in which the event was placed
|		 client_data - the GC to be used
|		 event       - the event that triggered this handler
|
|    Written By: Mark Young
|          Date: Jul 30, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
void place_handler(
    xvobject object,
    kaddr    client_data,
    XEvent   *event,
    int	     *dispatch)
{
	GC gc = (GC) client_data;
	Display *display = xvw_display(object);
	Window  window;

	int x, y;

	x = xpos;
	y = ypos;
	if (event->type == MotionNotify)
	{
	   xpos = event->xmotion.x;
	   ypos = event->xmotion.y;
	}
	else if (event->type == ButtonPress)
	{
	   xpos = event->xbutton.x;
	   ypos = event->xbutton.y;
	   extracted = TRUE;
	}

	if (xvw_check_subclass(object, InputOnlyWidgetClass) == TRUE)
	   window = xvw_window(xvw_parent(object));
	else
	   window = xvw_window(object);

        if (x != -1 && y != -1)
        {
           /*
            *  Erase the old rectangle.
            */
           XDrawRectangle(display, window, gc, x, y, width, height);
	   if (width > 1 && height > 1)
              XDrawRectangle(display, window, gc, x+1, y+1, width-2, height-2);
        }
 
        if (extracted == FALSE && xpos != -1 && ypos != -1)
        {
           /*
            *  Draw the new rectangle.
            */
           XDrawRectangle(display, window, gc, xpos, ypos, width, height);
	   if (width > 1 && height > 1)
	   {
              XDrawRectangle(display, window, gc, xpos+1, ypos+1, width-2,
			height-2);
	   }
        }
	*dispatch = FALSE;
}

/************************************************************
*
*  Routine Name: xvw_getposition - returns the position at which to place the
*				   object.
*
*       Purpose: Interactively prompts the user to place an object the object.
*		 This done by drawing a box which is moved around the screen
*		 until the user performs a ButtonPress.
*
*         Input: object - the object in which to get extents, if NULL then
*			  from the RootWindow.
*        Output: x - the starting x position
*		 y - the starting y position
*		 w - the width of the extent
*		 h - the height of the extent
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Nov 03, 1993
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int xvw_getposition(
   xvobject object,
   int          *x,
   int          *y)
{
	GC        gc;
	XGCValues values;
	unsigned  long mask;

	Widget    widget;
	Window    rw, cw;
	int       rx, ry;
	unsigned  int w, h;
	unsigned  int state;
	Cursor    cursor = None;
	xvobject  parent = xvw_parent(object);
	static    int inuse = FALSE;


	if (object == NULL)
	   return(FALSE);

	inuse = TRUE;
	xvw_geometry(object, NULL, NULL, &width, &height, NULL);
	widget = xvw_widget(parent);

        /* set up the graphics context to be used with graphics windows */
        values.function       = GXinvert;
        values.subwindow_mode = IncludeInferiors;
        mask = GCFunction  | GCSubwindowMode;
        gc = XtGetGC(widget, mask, &values);

        /*
         *  Add the event handler to rubberband on the object.
         */
        mask = ButtonPressMask | PointerMotionMask;
	xvw_insert_event(parent, (int) mask, place_handler, gc, KLIST_HEAD);

	/*
	 *  Draw the initial position...
	 */
	XQueryPointer(XtDisplay(widget), XtWindow(widget), &rw, &cw, &rx, &ry,
				&xpos, &ypos, &state);
	xvw_geometry(parent, NULL, NULL, &w, &h, NULL);
	if (xpos < 0) xpos = 0; else if (xpos + width > w)  xpos = w-width;
	if (ypos < 0) ypos = 0; else if (ypos + height > h) ypos = h-height;
        XDrawRectangle(XtDisplay(widget), XtWindow(widget), gc, xpos, ypos,
				width, height);
	if (width > 1 && height > 1)
	{
           XDrawRectangle(XtDisplay(widget), XtWindow(widget), gc, xpos+1,
				ypos+1, width-2, height-2);
	}

	/*
	 *  Save and then change the cursor
	 */
	xvw_busy(xvw_toplevel(object), FALSE);
	xvw_get_attribute(object, XVW_CURSOR, &cursor);
	xvw_set_attribute(object, XVW_CURSORNAME, "hand2");

	/*
	 *  Process events until the portion of the screen has been extracted
	 */
	extracted = FALSE;
	while (extracted == FALSE)
	   xvw_process_event();

	/*
	 *  Remove the handler and free the gc
	 */
	inuse = FALSE;
	xvw_set_attribute(object, XVW_CURSOR, cursor);
	xvw_remove_event(parent, (int) mask, place_handler, gc);
	XtReleaseGC(widget, gc);

	if (xpos == -1 && ypos == -1)
	   return(FALSE);

	if (x) *x = xpos;
	if (y) *y = ypos;
	return(TRUE);
}

/************************************************************
*
*  Routine Name: xvw_getobject - get the object associated with a window
*
*       Purpose: Retrieves a window from the screen as an xvobject so
*                that the xvobject can then be used within an application
*                that uses GUI & Visualization Services.
*
*                This routine can be used to allow the user to interactively 
*                select a window.  The idea here is that the application
*		 programmer may want to request that the user "pick" a
*		 window, which then may be used as needed in the form of
*                an xvobject.  For this method of operation, pass the 
*                the wname as NULL, and the window as NONE.  
*                The cursor will be changed 
*                to a cross, and events will be blocked until the user clicks
*                the mouse on a window.  That window will then be returned,
*                in xvobject form, to the calling routine.  Note that when
*                this approach is used, it is best to precede the call
*                to xvw_getobject() with a prompt about "Please choose a 
*                window", or it may appear to the user that the application
*                is hung.
*
*                Alternatively, this routine can be used to take a window 
*                specified by its name, and translate it into an 
*                xvobject.  For this method of operation,
*                pass the window name as wname, and NONE as the window ID.
*		 Being non-interactive, this method is better suited for
*		 obtaining the xvobject representation of a window when
*		 the window itself is already known.  Note that the name
*		 and ID of a window may be obtained using the X11 utility
*		 program, "xwininfo".  For instance, the following is
*		 from the xwininfo program:
*
*		 !        xwininfo: Window id: 0x1c0000d "xterm"
*
*		 In order to convert the above window into an xvobject,
*		 you would call xvw_getobject() with the following
*		 arguments:
*
*		 !        xvw_getobject(NULL, NONE, "xterm");
*		 !  - or -
*		 !        xvw_getobject(NULL, NONE, "0x1c0000d");
*
*		 Alternatively, this routine can be used to take a window
*		 specified by its actual ID, and translate it into an
*                xvobject.  For this method of operation, pass the window ID
*		 as window, and the parent and wname as NULL.  The following
*		 is an example given the window ID, specified by "window":
*
*                !         xvw_getobject(NULL, window, NULL);
*
*		 The parent object allows the programmer
*                to constrain the area in which the user is allowed
*                to interactively select the object.  If the parent
*                is NULL, then the area is constrainted to the rootwindow.
*
*                A window manager is an example of where xvw_getobject()
*                might be useful.  Part of the nature of a window manager is
*                to be able to reverse engineer existing windows, since
*                the user may actually start or switch window managers during
*                their X session.  So if this was not done then the user would
*                not be able to move, resize, or iconify the existing toplevel
*                windows.  By reverse engineering the toplevel windows as
*                xvobjects, the programmer would then have full access to
*                the xvobject manipulation routines, found in the xvwidget
*                library, which may prove useful in simplifying writing of
*                such an application.
*
*         Input: parent   - the object in which to get the subobject from.
*			    If NULL, then assume the rootwindow is the parent
*		 wname    - the wname or id of the window to get
*		 window   - the actual window to create as an object
*        Output:
*       Returns: The sub-object is successful or NULL otherwise
*
*  Restrictions: 
*    Written By: Mark Young
*          Date: Apr 01, 1995
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

xvobject xvw_getobject(
   xvobject parent,
   Window   window,
   char     *wname)
{
	XEvent   event;
	Cursor   cursor;
	xvobject object;
	Window   tempwindow;
	Screen   *screen  = xvw_screen(parent);
	Display  *display = xvw_display(parent);

        unsigned int num;
	int      i, grab;
        Window   dummy, *children;
        char     *string, *name = "GenericWindow";
	int      mask = PointerMotionMask|ButtonPressMask|ButtonReleaseMask;


	if (window == NONE && wname != NULL)
	{
	   window = strtol(wname, &string, 0);
	   if (string == wname || string[0] != '\0')
	   {
	      /*
	       *  Get the window relative to where we should grab the pointer.
	       *  If the parent is not NULL, then it should be relative to the
	       *  parent window, else relative to the rootwindow.
	       */
	      if ((tempwindow = xvw_window(parent)) == NONE)
	         tempwindow = xvw_rootwindow(parent);

	      if (!XQueryTree(display,tempwindow,&dummy,&dummy,&children,&num))
	         return(NULL);

	      for (i = 0, window = NONE; i < num && window == NONE; i++)
	      {
	         if (XFetchName(display, children[i], &string))
	         {
		    if (kstrcmp(wname, string) == 0)
		       window = children[i];
		    kfree(string);
	         }

	         if ((window = XmuClientWindow(display, children[i])) != NONE)
	         {
		    if (XFetchName(display, window, &string))
		    {
		       if (kstrcmp(wname, string) == 0)
			  window = children[i];
		       kfree(string);
		    }
	         }
	      }
	      kfree(children);
	   }
	}
	else if (window == NONE)
	{
	   /*
	    *  Get the winodw in which we will be selecting the window
	    */
	   if ((window = xvw_window(parent)) == NONE)
	      window = xvw_rootwindow(parent);

	   /*
	    *  Grab the pointer.
	    */
	   XSync(display, FALSE);
	   cursor = XCreateFontCursor(display, XC_crosshair);
	   grab = XGrabPointer(display, window, False, mask, GrabModeAsync,
			GrabModeAsync, window, cursor, CurrentTime);
           if (grab != GrabSuccess)
           {
              kerror(NULL,"xvw_getobject","Could not grab pointer for display");
              return(NULL);
           }

	   /*
	    * Get the entire window rather than a portion or subimage of one.
	    */
	   do
	   {
	      XNextEvent(display, &event);
	   } while(event.type != ButtonPress);
	   XUngrabPointer(display, CurrentTime);
	   XFreeCursor(display, cursor);

	   /*
	    *  Get the window.  If the subwindow is None then we should choose
	    *  the actual window.
	    */
	   if ((window = event.xbutton.subwindow) == None)
	      window = event.xany.window;
	}

        /*
         *  Create the invisible parent.  Shouldn't have to do this, but
         *  Xt and most widget set expect toplevel widgets to be subclassed
         *  off a shell widget.
         */
	if (parent == NULL)
           parent = xvw_create_application_shell(name, display, screen);
 
	/*
	 *  So we got the window, now create a GenericWindow object out of it.
	 */
	object = xvw_create(parent, FALSE,TRUE, name, GenericWindowWidgetClass);
	if (object == NULL)
	   return(NULL);

	xvw_set_attribute(object, XVW_GENERIC_WINDOW, window);
	return(object);
}
