/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>                 3D Drawing Utilities
   >>>>
   >>>>  Private:
   >>>>			X3D_draw_polyline()
   >>>>			X3D_draw_lineplot()
   >>>>			X3D_draw_wireframe()
   >>>>			X3D_draw_impulse()
   >>>>			X3D_draw_polymarker()
   >>>>			X3D_draw_colormarker()
   >>>>			X3D_draw_polygon()
   >>>>			X3D_draw_segments()
   >>>>			X3D_draw_line()
   >>>>			X3D_draw_rectangle()
   >>>>			X3D_draw_mesh()
   >>>>			X3D_draw_horizon()
   >>>>			X3D_draw_colormesh()
   >>>>			X3D_draw_colorline()
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "graphics.h"


/*-----------------------------------------------------------
|
|  Routine Name: draw_colorline 
|
|       Purpose: Draws a multi-colored line according to the
|		 different color levels.
|
|		 draw_colorline() is the real routine that computes
|		 the different colored line segments and draws them.
|		 This is done in a seperate routine in order to speed
|		 up the graphics processing by avoiding the costly
|		 setup time required for X3D_draw_colorline().
|		 draw_colorline() runs thru the coordinates finding
|		 the contour level indxes for a given line segment.
|		 If the levels are the same then the line is drawn
|		 with that color, otherwise the line is divided into
|		 segements for each contour level for the given line.
|
|         Input: graphics     - X3D graphics structure ID
|		 coords	      - coordinates that make up the line
|		 points	      -
|		 size	      - number of coordinates
|		 levels	      - level divisions at which each new color
|				will begin
|		 colors	      - pixel values (color) for each level
|		 color_offset -
|		 num	      - number of levels
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

static void draw_colorline(
   X3DGraphics *graphics,
   Coord       *coords,
   XPoint      *points,
   int         size,
   Real        *levels,
   XColor      *colors,
   int         color_offset,
   int         num)
{
	Coord    point;
	XPoint	 start, end;
	register Real x, y, z, w;
	int	 i, indx, indx1, indx2;

	for (i = 0; i < size; i++)
	{
	   _X3D_vector_mult(coords[i], graphics->imatrix, x, y, z, w);
	   coords[i].x = x; coords[i].y = y; coords[i].z = z;
	}
	indx1 = _X3D_find_level(coords[0].z, levels, num);
	indx = indx1+color_offset;
	X3D_draw[graphics->device].draw_color(graphics, &colors[indx], TRUE);

	for (i = 1; i < size; i++)
	{
	    indx2 = _X3D_find_level(coords[i].z, levels, num);
	    if (indx1 == indx2)
	    {
	       X3D_draw[graphics->device].line(graphics, points[i-1].x,
			points[i-1].y, points[i].x, points[i].y);
	    }
	    else
	    {
	       start = points[i -1];

	       if (indx2 < indx1)
	       {
	          while (indx1 != indx2)
	          {
		     /*
		      *  Compute the end point for 
		      */
		     point.x = kintercept(coords[i-1].x, coords[i].x, coords[i-1].z,
				      coords[i].z, levels[indx1]);
		     point.y = kintercept(coords[i-1].y, coords[i].y, coords[i-1].z,
				      coords[i].z, levels[indx1]);
		     point.z = levels[indx1];

		     if (_X3D_convert_wc_to_dc(graphics, graphics->matrix,
				&point, &end, 1))
		     {
			indx = indx1+color_offset;
			X3D_draw[graphics->device].draw_color(graphics,
				&colors[indx], TRUE);
	                X3D_draw[graphics->device].line(graphics, start.x,
			        start.y, end.x, end.y);
		     }

		     /*
		      *  Save the end of the segment as the start of the new
		      *  segment.
		      */
		     start = end;
		     indx1--;
	          }
		  indx = indx2+color_offset;
		  X3D_draw[graphics->device].draw_color(graphics,
				&colors[indx], TRUE);
	          X3D_draw[graphics->device].line(graphics, start.x,start.y,
			   points[i].x, points[i].y);
	       }
	       else
	       {
	          while (indx1 != indx2)
	          {
		     /*
		      *  Compute the end point for 
		      */
		     point.x = kintercept(coords[i-1].x, coords[i].x, coords[i-1].z,
				      coords[i].z, levels[indx1 +1]);
		     point.y = kintercept(coords[i-1].y, coords[i].y, coords[i-1].z,
				      coords[i].z, levels[indx1 +1]);
		     point.z = levels[indx1 +1];

		     if (_X3D_convert_wc_to_dc(graphics, graphics->matrix,
				&point, &end, 1))
		     {
			indx = indx1+color_offset;
			X3D_draw[graphics->device].draw_color(graphics,
				&colors[indx], TRUE);
	                X3D_draw[graphics->device].line(graphics, start.x,
			         start.y, end.x, end.y);
		     }

		     /*
		      *  Save the end of the segment as the start of the new
		      *  segment.
		      */
		     start = end;
		     indx1++;
	          }
		  indx = indx2+color_offset;
		  X3D_draw[graphics->device].draw_color(graphics,
				&colors[indx], TRUE);
	          X3D_draw[graphics->device].line(graphics, start.x,start.y,
			   points[i].x, points[i].y);
	       }
	    }
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: X3D_draw_polyline
|
|       Purpose:
|
|         Input: id     - X3D graphics structure ID
|                coords - array of world coordinate points
|                size   - number of points
|                fg     - color of polyline
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

void X3D_draw_polyline (
   int    id,
   Coord  *coords,
   int    size,
   XColor *fg)
{
	X3DGraphics	*graphics;
	Coord		*ncoords;
	XSegment	*segments;
	int		nsize, seg_num;


	if ((graphics = _X3D_get_graphics(id)) == NULL)
	{
	   kfprintf (kstderr,"X3D_draw_polyline:\n");
	   kfprintf (kstderr,"\t unknown graphics id\n");
	   return;
	}

	if ((ncoords = _X3D_clip_lines(graphics, coords, size, NULL, &nsize))
		== NULL)
	   return;

	seg_num = nsize/2;
	if (!(_X3D_convert_wc_to_dc_seg(graphics, graphics->matrix2, ncoords,
			segments, nsize)))
	{
	   return;
	}

        /*
	 *  Plot the desired polyline.
	 */
	X3D_draw[graphics->device].draw_color(graphics, fg, FALSE);
	X3D_draw[graphics->device].segments(graphics, segments, seg_num);

	if (ncoords != coords) kfree(ncoords);
        kfree(segments);
}

/*-----------------------------------------------------------
|
|  Routine Name: X3D_draw_lineplot
|
|       Purpose: 
|
|         Input: id       - X3D graphics structure ID
|		 coords	  - array of world coordinate points
|		 row_size - number of points in each polyline
|		 size	  - number of points
|		 fg	  - color of lines
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

void X3D_draw_lineplot(
   int    id,
   Coord  *coords,
   int    row_size,
   int    size,
   XColor *fg)
{
	X3DGraphics	*graphics;
	XSegment	*segments;
	Coord		*ncoords;
	int		i, j, col_size, nsize, seg_num;


	if ((graphics = _X3D_get_graphics(id)) == NULL)
	{
	   kfprintf (kstderr,"X3D_draw_lineplot:\n");
	   kfprintf (kstderr,"\t unknown graphics id\n");
	   return;
	}

        /*
	 *  see how many segments are to be plotted.  If there is only one
	 *  segment, then plot the whole thing.
	 */
	if (row_size == 1)
	   row_size = size; 

	col_size = size/row_size;

	if ((ncoords = (Coord *) kmalloc((unsigned) row_size*2*sizeof(Coord)))
		== NULL)
	{
	   kfprintf (kstderr,"X3D_draw_lineplot:\n");
	   kfprintf (kstderr,"\t not enough memory\n");
	   return;
	}

	if ((segments = (XSegment *) kmalloc(row_size * sizeof(XSegment)))
		== NULL)
	{
	   kfprintf (kstderr,"X3D_draw_wireframe:\n");
	   kfprintf (kstderr,"\t not enough memory\n");
	   return;
	}

	/*
	 *  Initialize the current segment size
	 */
	X3D_draw[graphics->device].draw_color(graphics, fg, FALSE);

        /* for each segment, plot a separate curve */
	for (i = 0; i < col_size; i++)
        {
	   j = i * row_size;
	   if (!_X3D_clip_lines(graphics, &coords[j], row_size,ncoords, &nsize))
	   {
	      continue;
	   }
	   seg_num = nsize/2;

	   if (!(_X3D_convert_wc_to_dc_seg(graphics, graphics->matrix2, ncoords,
			segments, nsize)))
	   {
	      kfree(segments);
	      return;
	   }
	   X3D_draw[graphics->device].segments(graphics, segments, seg_num);

	}
        kfree(ncoords);
	kfree(segments);
}

/*-----------------------------------------------------------
|
|  Routine Name: X3D_draw_wireframe
|
|       Purpose: Connects a series of polylines to make a 3D
|		 plot in a mesh pattern. This is done by drawing
|		 both lines along both the columns and rows.
|
|         Input: id	  - X3D graphics structure ID
|		 coords	  - array of world coordinate points
|		 row_size - number of points in each polyline
|		 size	  - number of points
|		 fg	  - color of wire frame
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

void X3D_draw_wireframe(
   int    id,
   Coord  *coords,
   int    row_size,
   int    size,
   XColor *fg)
{
	X3DGraphics	*graphics;
	XSegment	*segments;
	Coord		*columns, *ncoords;
	int		i, j, l, col_size, nsize, seg_num, seg_size;


	col_size = size/row_size;
	if (col_size == 1 || row_size == 1)
	{
	   X3D_draw_lineplot(id, coords, row_size, size, fg);
	   return;
	}

	if ((graphics = _X3D_get_graphics(id)) == NULL)
	{
	   kfprintf (kstderr,"X3D_draw_wireframe:\n");
	   kfprintf (kstderr,"\t unknown graphics id\n");
	   return;
	}

	if ((columns = (Coord *) kmalloc((unsigned)col_size*sizeof(Coord)))
		== NULL)
	{
	   kfprintf (kstderr,"X3D_draw_wireframe:\n");
	   kfprintf (kstderr,"\t not enough memory\n");
	   return;
	}

	seg_size = kmax(col_size, row_size);
	if ((segments = (XSegment *) kmalloc(seg_size * sizeof(XSegment)))
		== NULL)
	{
	   kfprintf (kstderr,"X3D_draw_wireframe:\n");
	   kfprintf (kstderr,"\t not enough memory\n");
	   return;
	}

	if ((ncoords = (Coord *) kmalloc(seg_size*2 * sizeof(Coord))) == NULL)
	{
	   kfprintf (kstderr,"X3D_draw_wireframe:\n");
	   kfprintf (kstderr,"\t not enough memory\n");
	   return;
	}

        /*
	 *  Draw the rows first
	 */
	X3D_draw[graphics->device].draw_color(graphics, fg, FALSE);
	for (i = 0; i < col_size; i++)
        {
	   j = i * row_size;
	   if (!_X3D_clip_lines(graphics, &coords[j], row_size,ncoords, &nsize))
	      continue;

	   seg_num = nsize/2;
	   if (!(_X3D_convert_wc_to_dc_seg(graphics, graphics->matrix2, ncoords,
		segments, nsize)))
	   {
	      kfree(segments);
	      return;
	   }
	   X3D_draw[graphics->device].segments(graphics, segments, seg_num);
	}

        /*
	 *  Draw the cols last
	 */
	for (i = 0; i < row_size; i++)
        {
	    for (j = 0; j < col_size; j++)
	    {
	        l = j * row_size + i;
		columns[j] = coords[l];
	    }

	   if (!_X3D_clip_lines(graphics, columns, col_size, ncoords, &nsize))
	      continue;

	   seg_num = nsize/2;
	   if (!(_X3D_convert_wc_to_dc_seg(graphics, graphics->matrix2, ncoords,
		segments, nsize)))
	   {
	      kfree(segments);
	      return;
	   }
	   X3D_draw[graphics->device].segments(graphics, segments, seg_num);
	}
        kfree(segments);
        kfree(columns);
}

/*-----------------------------------------------------------
|
|  Routine Name: X3D_draw_impulse
|
|       Purpose:
|
|         Input: id	- X3D graphics structure ID
|		 coords	- array of world coordinate points
|		 size	- number of points
|		 fg	- color of impulse
|
|        Output: none
|
|       Returns: nothing
|
|    Written By:
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

void X3D_draw_impulse (
   int    id,
   Coord  *coords,
   int    size,
   XColor *fg)
{
	X3DGraphics	*graphics;
        XSegment        *segments;
	Coord		*impulse;
	Real		base_value;
	int		i, j, seg_num, nsize;


	if ((graphics = _X3D_get_graphics(id)) == NULL)
	{
	   kfprintf (kstderr,"X3D_draw_discrete:\n");
	   kfprintf (kstderr,"\t unknown graphics id\n");
	   return;
	}

	if ((impulse = (Coord *) kmalloc ((unsigned) size*2 * sizeof(Coord)))
		== NULL)
	{
	   kfprintf (kstderr,"X3D_draw_discrete:\n");
	   kfprintf (kstderr,"\t not enough memory\n");
	   return;
	}

	if (graphics->wc_min.z < 0.0 && graphics->wc_max.z > 0.0)
	   base_value = 0.0;
	else
	   base_value = graphics->wc_min.z;

	i = j = 0;
	while (i < size)
	{
	   impulse[j] = impulse[j+1] = coords[i++];
	   impulse[j+1].z = base_value;
	   j += 2;
	}

	if (!_X3D_clip_segments(graphics, impulse, 2*size, impulse, &nsize))
	{
	   kfree(impulse);
	   return;
	}

	seg_num = nsize/2;
	if ((segments = (XSegment *) kmalloc ((unsigned) nsize * 
			 sizeof(XSegment))) == NULL)
	{
	   kfprintf (kstderr,"X3D_draw_discrete:\n");
	   kfprintf (kstderr,"\t not enough memory\n");
	   kfree(impulse);
	   kfree(segments);
	   return;
	}


	if (!(_X3D_convert_wc_to_dc_seg(graphics, graphics->matrix2, impulse,
			segments, nsize)))
	{
	   kfree(impulse);
	   kfree(segments);
	   return;
	}
	X3D_draw[graphics->device].draw_color(graphics, fg, FALSE);
	X3D_draw[graphics->device].segments(graphics, segments, seg_num);

	kfree(impulse);
	kfree(segments);
}

/*-----------------------------------------------------------
|
|  Routine Name: X3D_draw_polymarker
|
|       Purpose:
|
|         Input: id          - X3D graphics structure ID
|                coords      - array of world coordinate points
|                size        - number of points
|                marker_type - type of marker to display
|                fg          - color of polymarker
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date:
| Modifications:
|
------------------------------------------------------------*/

void X3D_draw_polymarker (
   int    id,
   Coord  *coords,
   int    size,
   int    marker_type,
   XColor *fg)
{
	X3DGraphics	*graphics;
	XPoint		*points;
	Coord		*ncoords;
	int		nsize, clip_code;


	if ((graphics = _X3D_get_graphics(id)) == NULL)
	{
	   kfprintf (kstderr,"X3D_draw_polyline:\n");
	   kfprintf (kstderr,"\t unknown graphics id\n");
	   return;
	}

	clip_code = X2D_CENTER;
	if ((ncoords = _X3D_clip_points(graphics, coords, size, clip_code,
		NULL, &nsize)) == NULL)
	{
	   return;
	}

	if ((points = (XPoint *) kmalloc ((unsigned) nsize * sizeof (XPoint)))
		== NULL)
	{
	   kfprintf (kstderr,"X3D_draw_marker:\n");
	   kfprintf (kstderr,"\t not enough memory\n");
	   return;
	}

	if(!(_X3D_convert_wc_to_dc(graphics, graphics->matrix2, ncoords, points,
			nsize)))
	{
	   kfree(points);
	   return;
	}
	X3D_draw[graphics->device].draw_color(graphics, fg, FALSE);
	_X2D_draw_markers(id, points, nsize, marker_type);

	if (coords != ncoords)
	   kfree(ncoords);
        kfree(points);
}

/*-----------------------------------------------------------
|
|  Routine Name: X3D_draw_colormarker
|
|       Purpose:
|
|         Input: id          - X3D graphics structure ID
|                coords      - array of world coordinate points
|                size        - number of points
|                marker_type - type of marker to display
|		 colors	     - color array
|		 num   	     - size of color array
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date:
| Modifications:
|
------------------------------------------------------------*/

void X3D_draw_colormarker (
   int    id,
   Coord  *coords,
   int    size,
   int    marker_type,
   XColor *colors,
   int    num,
   XColor *plotcolor)
{
	XColor		*fg;
	X3DGraphics	*graphics;
	XPoint		*points;
	Coord		*ncoords;
	int		i, nsize, clip_code;


	if ((graphics = _X3D_get_graphics(id)) == NULL)
	{
	   kfprintf (kstderr,"X3D_draw_polyline:\n");
	   kfprintf (kstderr,"\t unknown graphics id\n");
	   return;
	}

	clip_code = X2D_CENTER;
	if ((ncoords = _X3D_clip_points(graphics, coords, size, clip_code,
		NULL, &nsize)) == NULL)
	{
	   return;
	}

	if ((points = (XPoint *) kmalloc ((unsigned) nsize * sizeof (XPoint)))
		== NULL)
	{
	   kfprintf (kstderr,"X3D_draw_marker:\n");
	   kfprintf (kstderr,"\t not enough memory\n");
	   return;
	}

	if(!(_X3D_convert_wc_to_dc(graphics, graphics->matrix2, ncoords, points,
			nsize)))
	{
	   kfree(points);
	   return;
	}

	for (i = 0; i < size; i++)
	{
           if ( ncoords[i].d < num )
           {
             fg = &colors[ncoords[i].d];
           }
           else
           {
             fg = plotcolor;
           }
	   X3D_draw[graphics->device].draw_color(graphics, fg, FALSE);
	   _X2D_draw_markers(id, &points[i], 1, marker_type);
	}

	if (coords != ncoords)
	   kfree(ncoords);
        kfree(points);
}

/*-----------------------------------------------------------
|
|  Routine Name: X3D_draw_polygon
|
|       Purpose:
|
|         Input: id     - X3D graphics structure ID
|                coords - array of world coordinate points
|                size   - number of points
|                fg     - color of polygon
|                bg     - color of polygon's faces
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date:
| Modifications:
|
------------------------------------------------------------*/

void X3D_draw_polygon (
   int    id,
   Coord  *coords,
   int    size,
   XColor *fg,
   XColor *bg)
{
	X3DGraphics	*graphics;
	int		nsize;
	Coord		*ncoords;
	XPoint		*points;

	if ((graphics = _X3D_get_graphics(id)) == NULL)
	{
	   kfprintf (kstderr,"X3D_draw_polyline:\n");
	   kfprintf (kstderr,"\t unknown graphics id\n");
	   return;
	}

	if ((ncoords = _X3D_clip_polygon(graphics, coords, size, NULL,&nsize))
		== NULL)
	   return;

	if ((points = (XPoint *) kmalloc ((unsigned)(nsize+1)*sizeof (XPoint)))
		== NULL)
	{
	   kfprintf (kstderr,"X3D_draw_polygon:\n");
	   kfprintf (kstderr,"\t not enough memory\n");
	   return;
	}

	if (!(_X3D_convert_wc_to_dc(graphics, graphics->matrix2, ncoords,
			points, nsize)))
	{
	   kfree(points);
	   return;
	}

        /*
	 *  Fill the desired polygon.
	 */
	X3D_draw[graphics->device].fill_color(graphics, bg, FALSE);
	X3D_draw[graphics->device].fill_polygon(graphics, points, nsize,
			Complex, CoordModeOrigin);

	if (points[0].x != points[nsize-1].x || points[0].y !=
		points[nsize-1].y)
	{
	   points[nsize++] = points[0];
	}

        /*
	 *  Draw the desired polyline outline.
	 */
	X3D_draw[graphics->device].draw_color(graphics, fg, FALSE);
	X3D_draw[graphics->device].lines(graphics, points, nsize,
			CoordModeOrigin);

	if (ncoords != coords)
	   kfree(ncoords);
        kfree(points);
}

/*-----------------------------------------------------------
|
|  Routine Name: X3D_draw_segments
|
|       Purpose:
|
|         Input: id     - X3D graphics structure ID
|                coords - array of world coordinate points
|                size   - number of points
|                fg     - color of segments
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date:
| Modifications:
|
------------------------------------------------------------*/

void X3D_draw_segments (
   int    id,
   Coord  *coords,
   int    size,
   XColor *fg)
{
        X3DGraphics     *graphics;
	Coord		*ncoords;
        XSegment        *segments;
        int             num, nsize;


        if ((graphics = _X3D_get_graphics(id)) == NULL)
        {
           kfprintf (kstderr,"X3D_draw_segments:\n");
           kfprintf (kstderr,"\t unknown graphics id\n");
           return;
        }

	if ((ncoords = _X3D_clip_segments(graphics, coords, size,NULL,&nsize))
		== NULL)
	   return;

        num = nsize/2;
        if ((segments = (XSegment *) kmalloc ((unsigned) num *
                         sizeof (XSegment))) == NULL)
        {
           kfprintf (kstderr,"X3D_draw_segments:\n");
           kfprintf (kstderr,"\t not enough memory\n");
           return;
        }

        if(!(_X3D_convert_wc_to_dc_seg(graphics, graphics->matrix2, ncoords,
			segments, nsize)))
	{
	   kfree(segments);
	   return;
	}
	X3D_draw[graphics->device].draw_color(graphics, fg, FALSE);
        X3D_draw[graphics->device].segments(graphics, segments, num);

	if (ncoords != coords)
	   kfree(ncoords);
        kfree(segments);
}

/*-----------------------------------------------------------
|
|  Routine Name: X3D_draw_line
|
|       Purpose:
|
|         Input: id     - X3D graphics structure ID
|                coord1 - the beginning world coordinate point
|                coord2 - the end world coordinate point
|                fg     - color of line
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date:
| Modifications:
|
------------------------------------------------------------*/

void X3D_draw_line (
   int    id,
   Coord  coord1,
   Coord  coord2,
   XColor *fg)
{
        X3DGraphics     *graphics;
        XPoint          pt1, pt2;

        if ((graphics = _X3D_get_graphics(id)) == NULL)
        {
           kfprintf (kstderr,"X3D_draw_line:\n");
           kfprintf (kstderr,"\t unknown graphics id\n");
           return;
        }

        if(!(_X3D_convert_wc_to_dc(graphics, graphics->matrix, &coord1,
			&pt1, 1)))
	   return;
	
        if(!(_X3D_convert_wc_to_dc(graphics, graphics->matrix, &coord2,
			&pt2, 1)))
	   return;

	X3D_draw[graphics->device].draw_color(graphics, fg, FALSE);
        X3D_draw[graphics->device].line(graphics, pt1.x, pt1.y, pt2.x, pt2.y);
}

/*-----------------------------------------------------------
|
|  Routine Name: X3D_draw_rectangle
|
|       Purpose:
|
|         Input: id     - X3D graphics structure ID
|                coord  - the world coordinate upper left edge of rectangle
|                width  - the world coordinate width (x direction) of rectangle
|                height - the world coordinate height (z direction) of rectangle
|                depth  - the world coordinate depth (y direction) of rectangle
|                fg     - color of rectangle
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date:
| Modifications:
|
------------------------------------------------------------*/

void X3D_draw_rectangle (
   int    id,
   Coord  coord,
   Real   width,
   Real   height,
   Real   depth,
   XColor *fg)
{
	X3DGraphics	*graphics;
	Coord		coords[5];
	XPoint		point[5];


	if ((graphics = _X3D_get_graphics(id)) == NULL)
	{
	   kfprintf (kstderr,"X3D_draw_rectangle:\n");
	   kfprintf (kstderr,"\t unknown graphics id\n");
	   return;
	}

	coords[0] = coords[4] = coord;

	coords[1] = coords[0];
	coords[1].x += width;
	coords[1].y += depth;

	coords[2] = coords[1];
	coords[2].z += height;

	coords[3] = coords[1];
	coords[3].z += height;

	if(!(_X2D_convert_wc_to_dc(graphics, coords, point, 5)))
	   return;

	X3D_draw[graphics->device].draw_color(graphics, fg, FALSE);
	X3D_draw[graphics->device].lines(graphics, point, 5, CoordModeOrigin);
}

/*-----------------------------------------------------------
|
|  Routine Name: X3D_draw_mesh
|
|       Purpose: 
|
|         Input: id        - X3D graphics structure ID
|		 coords    - array of world coordinate points
|		 row_size  - number of points in each polyline
|		 size      - number of points
|		 fg        - color of mesh
|		 bg        - color of mesh face
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

void X3D_draw_mesh(
   int    id,
   Coord  *coords,
   int    row_size,
   int    size,
   XColor *fg,
   XColor *bg)
{
        X3DGraphics     *graphics;
	int		nsize;
	XPoint		points[10];
	Coord		ncoords[10], polygon[5];

	int	 corner;
	register int indx;
	int	 row, col, num_row, num_col, i, j;

	if ((graphics = _X3D_get_graphics(id)) == NULL)
	{
	   kfprintf (kstderr,"X3D_draw_mesh:\n");
	   kfprintf (kstderr,"\t unknown graphics id\n");
	   return;
	}
	
	num_col = row_size;
	num_row = size/num_col;
	row = num_row -1;
	col = num_col -1;

	/*  Compute which corner to start drawing from */
	corner = _X3D_find_corner(graphics, &coords[0], &coords[row_size-1],
			&coords[size-row_size], &coords[size-1]);

	if (corner == BACKLEFT)
	{
	    for (i = 0; i < col; i++)
	    {
	        for (j = row; j > 0; j--)
		{
		    indx = j * num_col + i;
		    polygon[0] =
		    polygon[4] = coords[indx];

		    polygon[1] = coords[++indx];

		    indx -= num_col;
		    polygon[2] = coords[indx];

		    polygon[3] = coords[--indx];

		    if (!_X3D_clip_polygon(graphics, polygon,5,ncoords, &nsize))
		       continue;

		    if (!(_X3D_convert_wc_to_dc(graphics, graphics->matrix2,
			ncoords, points, nsize)))
		       continue;

		    /*
		     *  Fill the desired polygon outline
		     */
		    X3D_draw[graphics->device].fill_color(graphics, bg, FALSE);
		    X3D_draw[graphics->device].fill_polygon(graphics, points,
				nsize, Nonconvex, CoordModeOrigin);

		    /*
		     *  Draw the desired polyline outline.
		     */
		   X3D_draw[graphics->device].draw_color(graphics, fg, FALSE);
		   X3D_draw[graphics->device].lines(graphics, points, nsize,
			CoordModeOrigin);
		}
	    }
	}
	else if (corner == BACKRIGHT)
	{
	    for (j = row; j > 0; j--)
	    {
	        for (i = col; i > 0; i--)
		{
	            indx = j * num_col + i;
		    polygon[0] =
		    polygon[4] = coords[indx];

		    polygon[1] = coords[--indx];

		    indx -= num_col;
		    polygon[2] = coords[indx];

		    polygon[3] = coords[++indx];

		    if (!_X3D_clip_polygon(graphics, polygon,5,ncoords, &nsize))
		       continue;

		    if (!(_X3D_convert_wc_to_dc(graphics, graphics->matrix2,
			ncoords, points, nsize)))
		       continue;

		    /*
		     *  Fill the desired polygon outline
		     */
		    X3D_draw[graphics->device].fill_color(graphics, bg, FALSE);
		    X3D_draw[graphics->device].fill_polygon(graphics, points,
				nsize, Nonconvex, CoordModeOrigin);

		    /*
		     *  Draw the desired polyline outline.
		     */
		   X3D_draw[graphics->device].draw_color(graphics, fg, FALSE);
		   X3D_draw[graphics->device].lines(graphics, points, nsize,
			CoordModeOrigin);
		}
	    }
	}
	else if (corner == FRONTRIGHT)
	{
	    for (i = col; i > 0; i--)
	    {
	        for (j = 0; j < row; j++)
		{
	            indx = j * num_col + i;
		    polygon[0] =
		    polygon[4] = coords[indx];

		    polygon[1] = coords[--indx];

		    indx += num_col;
		    polygon[2] = coords[indx];

		    polygon[3] = coords[++indx];

		    if (!_X3D_clip_polygon(graphics, polygon,5,ncoords, &nsize))
		       continue;

		    if (!(_X3D_convert_wc_to_dc(graphics, graphics->matrix2,
			ncoords, points, nsize)))
		       continue;

		    /*
		     *  Fill the desired polygon outline
		     */
		    X3D_draw[graphics->device].fill_color(graphics, bg, FALSE);
		    X3D_draw[graphics->device].fill_polygon(graphics, points,
				nsize, Nonconvex, CoordModeOrigin);

		    /*
		     *  Draw the desired polyline outline.
		     */
		   X3D_draw[graphics->device].draw_color(graphics, fg, FALSE);
		   X3D_draw[graphics->device].lines(graphics, points, nsize,
			CoordModeOrigin);
		}
	    }
	}
	else if (corner == FRONTLEFT)
	{
	    for (j = 0; j < row; j++)
	    {
	        for (i = 0; i < col; i++)
		{
	            indx = j * num_col + i;
		    polygon[0] =
		    polygon[4] = coords[indx];

		    polygon[1] = coords[++indx];

		    indx += num_col;
		    polygon[2] = coords[indx];

		    polygon[3] = coords[--indx];

		    if (!_X3D_clip_polygon(graphics, polygon,5,ncoords, &nsize))
		       continue;

		    if (!(_X3D_convert_wc_to_dc(graphics, graphics->matrix2,
			ncoords, points, nsize)))
		       continue;

		    /*
		     *  Fill the desired polygon outline
		     */
		    X3D_draw[graphics->device].fill_color(graphics, bg, FALSE);
		    X3D_draw[graphics->device].fill_polygon(graphics, points,
				nsize, Nonconvex, CoordModeOrigin);

		    /*
		     *  Draw the desired polyline outline.
		     */
		   X3D_draw[graphics->device].draw_color(graphics, fg, FALSE);
		   X3D_draw[graphics->device].lines(graphics, points, nsize,
			CoordModeOrigin);
		}
	    }
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: X3D_draw_horizon
|
|       Purpose: Draws a horizon plot.  This is done by finding
|		 opposite side of the plot and drawing each row
|		 from back to front.  Painter's algorithm is used
|		 to fill the row, thus obscuring the previous row.
|		 This is repeated until all rows have been drawn.
|
|		 Note:  We say row, but we also mean column.  We draw
|			rows or columns depending on which side of the
|			plot is furthest from the viewer.  Thus if the
|			back left corner is furthest then we would draw
|			each row, but if the front left corner is the
|			furthest then we draw the each column.
|
|         Input: id        - X3D graphics structure ID
|		 coords    - array of world coordinate points
|		 row_size  - number of points in each polyline
|		 size      - number of points
|		 fg	   - color of horizon
|		 bg	   - color of horizon surface
|		 wcmin	   - the minimum extent of the plot
|		 wcmax	   - the maximum extent of the plot
|
|        Output:
|       Returns:
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

void X3D_draw_horizon(
   int    id,
   Coord  *coords,
   int    rows,
   int    size,
   XColor *fg,
   XColor *bg,
   Coord  wcmin,
   Coord  wcmax)
{
        X3DGraphics     *graphics;
	register XPoint	*points;
	register Coord	*horizon, *ncoords;

	int	 corner;
	register int i, j, m;
	int	 nsize, cols, rsize = rows +2;

	if ((graphics = _X3D_get_graphics(id)) == NULL)
	{
	   kfprintf (kstderr,"X3D_draw_horizon:\n");
	   kfprintf (kstderr,"\t unknown graphics id\n");
	   return;
	}
	

	if ((horizon = (Coord *) kmalloc((unsigned) rsize * sizeof(Coord)))
		== NULL)
	{
	   kfprintf (kstderr,"X3D_draw_horizon:\n");
	   kfprintf (kstderr,"\t not enough memory\n");
	   return;
	}

	if ((ncoords = (Coord *) kmalloc((unsigned) 2*rsize * sizeof(Coord)))
		== NULL)
	{
	   kfprintf (kstderr,"X3D_draw_horizon:\n");
	   kfprintf (kstderr,"\t not enough memory\n");
	   kfree(horizon); return;
	}

	if ((points = (XPoint *) kmalloc((unsigned) 2*rsize * sizeof(XPoint)))
		== NULL)
	{
	   kfprintf (kstderr,"X3D_draw_horizon:\n");
	   kfprintf (kstderr,"\t not enough memory\n");
	   kfree(horizon); kfree(ncoords);
	   return;
	}

	/*  Compute which corner is closest to you */
	corner = _X3D_find_corner(graphics, &coords[0], &coords[rows-1],
			&coords[size-rows], &coords[size-1]);
	cols = size/rows;

	if (corner == BACKLEFT)
	{
	   horizon[0] = horizon[rows +1] = wcmin;
	   horizon[rows +1].x = wcmax.x;

	   for (i = 0; i < cols; i++)
	   {
	      m = (cols -i -1) * rows;
	      horizon[0].y = coords[m].y;

	      for (j = 1; j <= rows; j++)
		 horizon[j] = coords[m++];
	      horizon[rows +1].y = horizon[rows].y;

	      if (!_X3D_clip_polygon(graphics, horizon, rsize, ncoords, &nsize))
		 continue;

	      if (!(_X3D_convert_wc_to_dc(graphics, graphics->matrix2, ncoords,
			points, nsize)))
		 continue;

	      /*
	       *  Fill the desired polygon outline
	       */
	      X3D_draw[graphics->device].fill_color(graphics, bg, FALSE);
	      X3D_draw[graphics->device].fill_polygon(graphics, points, nsize,
				Nonconvex, CoordModeOrigin);

	      /*
	       *  Draw the desired polyline outline.
	       */
	      X3D_draw[graphics->device].draw_color(graphics, fg, FALSE);
	      X3D_draw[graphics->device].lines(graphics, points, nsize,
			CoordModeOrigin);
	   }
	}
	else if (corner == FRONTLEFT)
	{
	   horizon[0] = horizon[rows +1] = wcmin;
	   horizon[rows +1].x = wcmax.x;

	   for (i = 0; i < cols; i++)
	   {
	      m = i * rows;
	      horizon[0].y = coords[m].y;

	      for (j = 1; j <= rows; j++)
		 horizon[j] = coords[m++];
	      horizon[rows +1].y = horizon[rows].y;

	      if (!_X3D_clip_polygon(graphics, horizon, rsize, ncoords, &nsize))
		 continue;

	      if (!(_X3D_convert_wc_to_dc(graphics, graphics->matrix2, ncoords,
			points, nsize)))
		 continue;

	      /*
	       *  Fill the desired polygon outline
	       */
	      X3D_draw[graphics->device].fill_color(graphics, bg, FALSE);
	      X3D_draw[graphics->device].fill_polygon(graphics, points, nsize,
				Nonconvex, CoordModeOrigin);

	      /*
	       *  Draw the desired polyline outline.
	       */
	      X3D_draw[graphics->device].draw_color(graphics, fg, FALSE);
	      X3D_draw[graphics->device].lines(graphics, points, nsize,
			CoordModeOrigin);
	   }
	}
	else if (corner == BACKRIGHT)
	{
	   horizon[0] = horizon[rows +1] = wcmin;
	   horizon[0].x = wcmax.x;

	   for (i = 0; i < cols; i++)
	   {
	      m = (cols -i) * rows -1;
	      horizon[0].y = coords[m].y;

	      for (j = 1; j <= rows; j++)
		 horizon[j] = coords[m--];
	      horizon[rows +1].y = horizon[rows].y;

	      if (!_X3D_clip_polygon(graphics, horizon, rsize, ncoords, &nsize))
		 continue;

	      if (!(_X3D_convert_wc_to_dc(graphics, graphics->matrix2, ncoords,
			points, nsize)))
		 continue;

	      /*
	       *  Fill the desired polygon outline
	       */
	      X3D_draw[graphics->device].fill_color(graphics, bg, FALSE);
	      X3D_draw[graphics->device].fill_polygon(graphics, points, nsize,
				Nonconvex, CoordModeOrigin);

	      /*
	       *  Draw the desired polyline outline.
	       */
	      X3D_draw[graphics->device].draw_color(graphics, fg, FALSE);
	      X3D_draw[graphics->device].lines(graphics, points, nsize,
			CoordModeOrigin);
	   }
	}
	else if (corner == FRONTRIGHT)
	{
	   horizon[0] = horizon[rows +1] = wcmin;
	   horizon[0].x = wcmax.x;

	   for (i = 0; i < cols; i++)
	   {
	      m = (i +1) * rows -1;
	      horizon[0].y = coords[m].y;

	      for (j = 1; j <= rows; j++)
		 horizon[j] = coords[m--];
	      horizon[rows +1].y = horizon[rows].y;

	      if (!_X3D_clip_polygon(graphics, horizon, rsize, ncoords, &nsize))
		 continue;

	      if (!(_X3D_convert_wc_to_dc(graphics, graphics->matrix2, ncoords,
			points, nsize)))
		 continue;

	      /*
	       *  Fill the desired polygon outline
	       */
	      X3D_draw[graphics->device].fill_color(graphics, bg, FALSE);
	      X3D_draw[graphics->device].fill_polygon(graphics, points, nsize,
				Nonconvex, CoordModeOrigin);

	      /*
	       *  Draw the desired polyline outline.
	       */
	      X3D_draw[graphics->device].draw_color(graphics, fg, FALSE);
	      X3D_draw[graphics->device].lines(graphics, points, nsize,
			CoordModeOrigin);
	   }
	}
	kfree(points); kfree(horizon); kfree(ncoords);
}

/*-----------------------------------------------------------
|
|  Routine Name: X3D_draw_colormesh
|
|       Purpose: Draws a mesh plot using contour levels for
|		 the different color levels.  Surfaces are built
|		 exactly the same way that X3D_draw_mesh() does,
|		 accept the surfaces are left in 3D coordinate
|		 system.  The surface is then converted to device
|		 coordinates and the area is filled using the gc_fill
|		 color.  The surface is finally passed to the
|		 draw_colorline() routine, which draws the color
|		 border around the filled area.
|
|         Input: id           - X3D graphics structure ID
|		 coords       - array of world coordinate points
|		 row_size     - number of points in each polyline
|		 size         - number of points
|		 levels	      -
|		 colors	      -
|		 color_offset -
|		 num_levels   -
|		 bg	      - fill color
|		 wcmin	      - the minimum extent of the plot
|		 wcmax	      - the maximum extent of the plot
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

void X3D_draw_colormesh(
   int    id,
   Coord  *coords,
   int    row_size,
   int    size,
   Real   *levels,
   XColor *colors,
   int    color_offset,
   int    num_levels,
   XColor *bg,
   Coord  wcmin,
   Coord  wcmax)
{
        X3DGraphics     *graphics;
	Coord		polygon[5], ncoords[10];
	XPoint		points[10];

	int	 corner;
	register int indx;
	int	 free_levels = FALSE;
	int	 nsize, row, col, num_row, num_col, i, j;

	if ((graphics = _X3D_get_graphics(id)) == NULL)
	{
	   kfprintf (kstderr,"X3D_draw_colormesh:\n");
	   kfprintf (kstderr,"\t unknown graphics id\n");
	   return;
	}
	X3D_draw[graphics->device].fill_color(graphics, bg, FALSE);

	num_col = row_size;
	num_row = size/num_col;
	row = num_row -1;
	col = num_col -1;

	/*  Compute which corner to start drawing from */
	corner = _X3D_find_corner(graphics, &coords[0], &coords[row_size-1],
			&coords[size-row_size], &coords[size-1]);

	if (levels == NULL)
	{
	   free_levels = TRUE;
	   levels = X3D_get_contour_levels(wcmin.z, wcmax.z, 0.0, num_levels);
	   if (levels == NULL)
	      return;
	}


	if (corner == BACKLEFT)
	{
	   for (i = 0; i < col; i++)
	   {
	      for (j = row; j > 0; j--)
	      {
		 indx = j * num_col + i;
		 polygon[0] =
		 polygon[4] = coords[indx];
		 polygon[1] = coords[++indx];

		 indx -= num_col;
		 polygon[2] = coords[indx];
		 polygon[3] = coords[--indx];

		 if (!_X3D_clip_polygon(graphics, polygon, 5, ncoords, &nsize))
		    continue;

		 if (!(_X3D_convert_wc_to_dc(graphics, graphics->matrix2,
			ncoords, points, nsize)))
		    continue;

		 X3D_draw[graphics->device].fill_color(graphics, bg, FALSE);
		 X3D_draw[graphics->device].fill_polygon(graphics, points,
				nsize, Nonconvex, CoordModeOrigin);
		 draw_colorline(graphics, ncoords, points, nsize, levels,
				colors, color_offset, num_levels);
	      }
	   }
	}
	else if (corner == BACKRIGHT)
	{
	   for (j = row; j > 0; j--)
	   {
	      for (i = col; i > 0; i--)
	      {
	         indx = j * num_col + i;
		 polygon[0] =
		 polygon[4] = coords[indx];
		 polygon[1] = coords[--indx];

		 indx -= num_col;
		 polygon[2] = coords[indx];
		 polygon[3] = coords[++indx];

		 if (!_X3D_clip_polygon(graphics, polygon, 5, ncoords, &nsize))
		    continue;

		 if (!(_X3D_convert_wc_to_dc(graphics, graphics->matrix2,
			ncoords, points, nsize)))
		    continue;

		 X3D_draw[graphics->device].fill_color(graphics, bg, FALSE);
		 X3D_draw[graphics->device].fill_polygon(graphics, points,
				nsize, Nonconvex, CoordModeOrigin);
		 draw_colorline(graphics, ncoords, points, nsize, levels,
				colors, color_offset, num_levels);
	      }
	   }
	}
	else if (corner == FRONTRIGHT)
	{
	   for (i = col; i > 0; i--)
	   {
	      for (j = 0; j < row; j++)
	      {
	         indx = j * num_col + i;
		 polygon[0] =
		 polygon[4] = coords[indx];
		 polygon[1] = coords[--indx];

		 indx += num_col;
		 polygon[2] = coords[indx];
		 polygon[3] = coords[++indx];

		 if (!_X3D_clip_polygon(graphics, polygon, 5, ncoords, &nsize))
		    continue;

		 if (!(_X3D_convert_wc_to_dc(graphics, graphics->matrix2,
			ncoords, points, nsize)))
		    continue;

		 X3D_draw[graphics->device].fill_color(graphics, bg, FALSE);
		 X3D_draw[graphics->device].fill_polygon(graphics, points,
				nsize, Nonconvex, CoordModeOrigin);
		 draw_colorline(graphics, ncoords, points, nsize, levels,
				colors, color_offset, num_levels);
	      }
	   }
	}
	else if (corner == FRONTLEFT)
	{
	   for (j = 0; j < row; j++)
	   {
	      for (i = 0; i < col; i++)
	      {
	         indx = j * num_col + i;
		 polygon[0] =
		 polygon[4] = coords[indx];
		 polygon[1] = coords[++indx];

		 indx += num_col;
		 polygon[2] = coords[indx];
		 polygon[3] = coords[--indx];

		 if (!_X3D_clip_polygon(graphics, polygon, 5, ncoords, &nsize))
		    continue;

		 if (!(_X3D_convert_wc_to_dc(graphics, graphics->matrix2,
			ncoords, points, nsize)))
		    continue;

		 X3D_draw[graphics->device].fill_color(graphics, bg, FALSE);
		 X3D_draw[graphics->device].fill_polygon(graphics, points,
				nsize, Nonconvex, CoordModeOrigin);
		 draw_colorline(graphics, ncoords, points, nsize, levels,
				colors, color_offset, num_levels);
	      }
	   }
	}

	/*
	 *  Cleanup time.
	 */
	if (free_levels) kfree(levels);
}

/*-----------------------------------------------------------
|
|  Routine Name: X3D_draw_colorline
|
|       Purpose: Draws a multi-colored line according to the
|		 different color levels.  X3D_draw_colorline()
|		 is a frontend for the real routine static
|		 draw_colorline().
|
|		 X3D_draw_colorline() gets the graphics structure,
|		 perpares the contour levels by making sure they
|		 are sorted in ascending order, converts the world
|		 coordinates to device coordinates, then calls
|		 draw_colorline() to actually draw the multi-colored
|		 line.
|
|		 draw_colorline() is the real routine that computes
|		 the different colored line segments and draws them.
|
|         Input: id	      - X3D graphics structure ID
|		 coords	      - coordinates that make up the line
|		 size	      - number of coordinates
|		 levels	      - level divisions at which each new color
|				will begin
|		 colors	      - pixel values (color) for each level
|		 color_offset -
|		 num_levels   - number of levels
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

void X3D_draw_colorline(
   int    id,
   Coord  *coords,
   int    size,
   Real   *levels,
   XColor *colors,
   int    color_offset,
   int    num_levels)
{
	X3DGraphics	*graphics;
	XPoint		*points;

	if ((graphics = _X3D_get_graphics(id)) == NULL)
	{
	   kfprintf (kstderr,"X3D_draw_colorline:\n");
	   kfprintf (kstderr,"\t unknown graphics id\n");
	   return;
	}

	if ((points = (XPoint *) kmalloc ((unsigned) size * sizeof (XPoint)))
		== NULL)
	{
	   kfprintf (kstderr,"X3D_draw_colorline:\n");
	   kfprintf (kstderr,"\t not enough memory\n");
	   return;
	}

	if(!(_X3D_convert_wc_to_dc(graphics, graphics->matrix, coords, points,
			size)))
	{
	   kfree(points);
	   return;
	}

	/*
	 *  Sort contours in ascending order.
	 *
	 sort_contour(levels, &contours, colors, &colors, num_levels);
	 */

	/*
	 *  Call draw_colorline() to compute the different colored line
	 *  segments and draw them.
	draw_colorline(graphics, coords, points, size, levels, colors,
		       color_offset, num_levels);
	 */
	kfree(points);
}
