/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>		       Form  Utility Routines                 <<<<
   >>>>  Private:					      <<<<
   >>>>		       xvf_get_line_type()	              <<<<
   >>>>		       xvf_call_gui_callbacks()               <<<<
   >>>>		       xvf_call_extra_calls()                 <<<<
   >>>> 	       xvf_def_number_sign_digits()           <<<<
   >>>>		       xvf_set_optional_sensitivity()	      <<<<
   >>>>		       xvf_set_toggle_optional_sensitivity()  <<<<
   >>>>		       xvf_synchronize()	              <<<<
   >>>>		       xvf_switch_toggle_value()	      <<<<
   >>>>		       xvf_calculate_y_position()	      <<<<
   >>>>		       xvf_return_live_selection()            <<<<
   >>>>		       xvf_mark_modified()	              <<<<
   >>>>   Static:					      <<<<
   >>>>   Public:					      <<<<
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"

/*-----------------------------------------------------------
|
|  Routine Name: xvf_call_gui_callbacks
|
|       Purpose: Calls routines registered with xvf_add_gui_callback()
|
|         Input: call_list  - list of extra routines to call
|                attribute  - attribute whose change fired callback(s)
|                new_value  - new value of attribute
|        Output: none
|          Date: June 14, 1993
|    Written By: Danielle Argiro 
| Modifications: 
|
-------------------------------------------------------------*/
void xvf_call_gui_callbacks(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        new_value)
{
	int all_token, attribute_token;
        xvf_callback_info *call_info;
	klist *callback_ptr;
static  int in_progress = FALSE;

	/*
	 *  first, see if the "xvf_all_attributes" callback was added for 
         *  ANY attribute change, call the callback if it was
	 */
	if (in_progress) return;
	in_progress = TRUE;
	all_token = kstring_to_token("xvf_all_attributes");
	callback_ptr = klist_locate(kformstruct->callback_list, (kaddr) all_token);
	if (callback_ptr != NULL) 
	{
	    call_info = (xvf_callback_info *) callback_ptr->client_data;
	    (*call_info->routine) (kformstruct, "xvf_all_attributes", 
				   new_value, call_info->client_data);
	}

	/*
	 * next, see if any individual attribute callbacks were added,
	 * call any callbacks if they were.
	 */
	attribute_token = kstring_to_token(attribute);
	callback_ptr = klist_locate(kformstruct->callback_list, 
			 	        (kaddr) attribute_token);
	if (callback_ptr != NULL) 
	{
	    call_info = (xvf_callback_info *) callback_ptr->client_data;
	    (*call_info->routine) (kformstruct, attribute, new_value,
		 	               call_info->client_data);
	}
	in_progress = FALSE;
}

/*-----------------------------------------------------------
|
|  Routine Name: xvf_call_extra_calls
|
|       Purpose: Calls routines registered with xvf_add_extra_call()
|
|         Input: call_list - list of extra routines to call
|                location  - XVF_CALL_FIRST, XVF_CALL_LAST, 
|                            or XVF_CALL_SUBSTITUTE
|        Output: none
|          Date: Oct 28, 1992
|    Written By: Danielle Argiro
| Modifications:
|
-------------------------------------------------------------*/
int xvf_call_extra_calls(
   klist *call_list,
   int   location)
{
        xvf_callback_info *call_info;
	klist *call_list_ptr;
	int   status = FALSE;

	call_list_ptr = call_list;
        while (call_list_ptr != NULL)
        {
            call_info = (xvf_callback_info *) call_list_ptr->client_data;
            if (call_info->call_location == location)
	    {
               (*call_info->routine) (call_info->client_data);
		if (location == XVF_CALL_SUBSTITUTE)
		    status = TRUE;

	    }
            call_list_ptr = klist_next(call_list_ptr);
        }
	return(status);
}

/*-----------------------------------------------------------
|
|  Routine Name: xvf_set_optional_sensitivity
|
|       Purpose: When an optional selection is un-selected,
|                desensitizes all objects associated with the
|                selection (except the optional button).
|                Re-sensitizes selection when optional selection
|                is re-selected.
|
|         Input: selection - the optional selection
|                value     - TRUE if selection is to be sensitized, 
|                            FALSE if selection is to be de-sensitized
|        Output: none
|          Date: Jun 26, 1992
|    Written By: Danielle Argiro
| Modifications:
|
-----------------------------------------------------------*/
void xvf_set_optional_sensitivity(
   kselection *selection,
   int           value)
{
	kselection *toggle;

	if (!xvf_desensitize_optionals) return;

	if (selection->type == KUIS_TOGGLE)
	{
	    toggle = selection->toggle_next;
	    while (toggle != NULL)
	    {
	        xvw_sensitive(toggle->opt_object, value);
		toggle = toggle->next;
	    }
	    return;
	}

	if (selection->value_object != NULL)
	    xvw_sensitive(selection->value_object, value);
	if (selection->label_object != NULL)
	    xvw_sensitive(selection->label_object, value);
	if (selection->scroll != NULL)
	    xvw_sensitive(selection->scroll, value);

}

/*-----------------------------------------------------------
|
|  Routine Name: xvf_set_toggle_sensitivity
|
|       Purpose: When a toggle group  is un-selected,
|                desensitizes all objects associated with the
|                toggle group (except the optional button).
|                Re-sensitizes objects when optional selection
|                is re-selected.
|
|         Input: toggle_start - start of the toggle (-T line)
|        Output: none
|          Date: Nov 5, 1992
|    Written By: Danielle Argiro
| Modifications:
|
-----------------------------------------------------------*/
void xvf_set_toggle_optional_sensitivity(
   kselection *toggle_start)
{
        kselection *toggle;
        toggle = toggle_start->toggle_next;

        if (!xvf_desensitize_optionals) return;

        while (toggle != NULL)
        {
            xvf_set_optional_sensitivity(toggle, toggle_start->opt_selected);
            xvw_sensitive(toggle->opt_object, toggle_start->opt_selected);
            toggle = toggle->next;
        }
}
/*-----------------------------------------------------------
|
|  Routine Name: xvf_list_button_width
|
|       Purpose: Calculates width for list or cycle button such that
|                the button will accomodate the longest string in the
|                list (or cycle).
|
|         Input: labels - pointer to list of strings to be labels
|                num    - size of labels[]
|        Output: floating point width (in characters)
|          Date: Nov 19, 1992
|    Written By: Danielle Argiro
| Modifications:
|
-----------------------------------------------------------*/

float xvf_list_button_width(
   char **labels,
   int  num)
{
	int   i;
	float width;

	width = 0.0;
	for (i = 0; i < num; i++)
	{
	   if (((float) kstrlen(labels[i])) > width) 
		width = (float) kstrlen(labels[i]);
	}
	width += 3.0;
	if (width < 4.0) width = 4.0;
	return(width);
}

/*-----------------------------------------------------------
|
|  Routine Name: xvf_synchronize
|
|       Purpose: Calls xvf_synchronize; for use in dbx only
|
|         Input: none
|        Output: none
|          Date: Jun 26, 1992
|    Written By: Danielle Argiro
| Modifications:
|
-----------------------------------------------------------*/
int xvf_synchronize(void)
{

 	XSynchronize(xvw_display(NULL), TRUE);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: xvf_check_modified
|
|       Purpose: Sees if any of the selections in the selection list
|                have been modified
|         Input: sel_list - pointer to header of selection list
|        Output: none
|          Date: Nov 29, 1993
|    Written By: Danielle Argiro
| Modifications:
|
-----------------------------------------------------------*/
int xvf_check_modified(
    kselection *sel_list)
{
	kselection *selection;
	
	selection = sel_list;
	while (selection != NULL)
	{
	    if (selection->modified == TRUE)
		return(TRUE);

	    if ((selection->type == KUIS_MUTEXCL) ||
                (selection->type == KUIS_MUTINCL) ||
                (selection->type == KUIS_GROUP))
                if (xvf_check_modified(selection->group_next))
		    return(TRUE);

	    selection = selection->next;
	}
        return(FALSE);
}

/*-----------------------------------------------------------
|
|  Routine Name: xvf_clear_modified
|
|       Purpose: Clears all selections in the selection list to not modified
|         Input: sel_list - pointer to header of selection list
|        Output: none
|          Date: Aug 10, 1994
|    Written By: Danielle Argiro
| Modifications:
|
-----------------------------------------------------------*/
int xvf_clear_modified(
    kselection *sel_list)
{
        kselection *selection;

        selection = sel_list;
        while (selection != NULL)
        {
            selection->modified = FALSE;
	    if ((selection->type == KUIS_MUTEXCL) ||
		(selection->type == KUIS_MUTINCL) ||
		(selection->type == KUIS_GROUP))
	 	xvf_clear_modified(selection->group_next);
            selection = selection->next;
        }
        return(FALSE);
}


/*-----------------------------------------------------------
|
|  Routine Name: xvf_add_title
|
|       Purpose: Adds a title to a GUI item that has none
|         Input: kformstruct - generic kform_struct
|        Output: none
|          Date: July 23, 1992
|    Written By: Danielle Argiro
| Modifications:
|
-------------------------------------------------------------*/

int xvf_add_title(
   kform_struct *kformstruct,
   int          flag,
   float        xpos, 
   float        ypos,
   char         *title)
{
	kcontrol *control;

        if (kformstruct->type == KFORM)
	{
	    control = kformstruct->Formptr->master;
            control->label_object =
			xvf_create_label(control->back, xpos, ypos, NULL, 
				         title, "label", FALSE, TRUE);
            xvw_set_attribute(control->label_object, XVW_MENU_CLIENTDATA, 
			      kformstruct);
	}
        else if (kformstruct->type == KSUBFORM)
        {
            if (flag == KUIS_STARTSUBFORM)
	    {
                 kformstruct->Subformptr->label_object =
			xvf_create_label(kformstruct->Subformptr->back, 
				         xpos, ypos, NULL, title, "label", 
				         FALSE, TRUE);
		 xvw_set_attribute(kformstruct->Subformptr->label_object,
                                  XVW_MENU_CLIENTDATA,
                                  kformstruct);
	    }
            else
	    {
		errno = KINTERNAL;
	  	kerror("xvforms", "xvf_add_title",
		       "Can't add title to non-existent subform button");
		return(FALSE);
	    }
        }
        else if (kformstruct->type ==  KGUIDE)
        {
            if (flag == KUIS_STARTGUIDE)
	    {
		control = kformstruct->Guideptr->back_subform->guidepane;
                control->label_object = 
			xvf_create_label(control->back, xpos, ypos, NULL, 
				         title, "label", FALSE, TRUE);
                xvw_set_attribute(control->label_object, XVW_MENU_CLIENTDATA,
                                  kformstruct);
	    }
            else
            {
		errno = KINTERNAL;
                kerror("xvforms", "xvf_add_title",
                       "Can't add title to non-existent guide button");
                return(FALSE);
            }
        }
        else if (kformstruct->type == KPANE) 
        {
	    control = kformstruct->Controlptr;
            control->label_object = 
		xvf_create_label(control->back, xpos, ypos, 
				 NULL, title, "label", FALSE, TRUE);
                xvw_set_attribute(control->label_object, 
				  XVW_MENU_CLIENTDATA, kformstruct);
;
        }

        return(TRUE);
}

/*------------------------------------------------------------
|
|  Routine Name: xvf_add_opt_button
|
|       Purpose: Given an kformstruct representing a selection, create
|                the optional box object, install the appropriate callback
|                for the optional box, and re-position the label object 
|                to be to the right of optional box.
|
|         Input: kformstruct - kformstruct representing selection to have 
|                            optional box added
|        Output: none
|          Date: Nov 17, 1992
|    Written By: Danielle Argiro
| Modifications:
|
-------------------------------------------------------------*/
void xvf_add_opt_button(
   kform_struct *kformstruct)
{
	xvobject        opt_wid;
	kselection *selection;
	
	selection = kformstruct->Selptr;

	opt_wid = xvf_create_optional_button(selection->back);

	switch(selection->type)
	{
	    case KUIS_LOGICAL:
		 xvw_insert_callback(opt_wid, XVW_BUTTON_SELECT, FALSE,
                                  xvf_gen_opt_cb, (kaddr) kformstruct);
		 break;

	    case KUIS_FLAG:
            	 xvw_insert_callback(opt_wid, XVW_BUTTON_SELECT, FALSE,
                                  xvf_gen_opt_cb, (kaddr) kformstruct);
		 break;

	    case KUIS_CYCLE:
            	 xvw_insert_callback(opt_wid, XVW_BUTTON_SELECT, FALSE,
                                  xvf_gen_opt_cb, (kaddr) kformstruct);
		 break;

	    case KUIS_LIST:
            	 xvw_insert_callback(opt_wid, XVW_BUTTON_SELECT, FALSE,
                                  xvf_gen_opt_cb, (kaddr) kformstruct);
		 break;

	    case KUIS_TOGGLE:
            	 xvw_insert_callback(opt_wid, XVW_BUTTON_SELECT, FALSE,
                                  xvf_gen_opt_cb, (kaddr) kformstruct);
		 break;
	
	    default:
            	 xvw_insert_callback(opt_wid, XVW_BUTTON_SELECT, FALSE,
                                  xvf_gen_opt_cb, (kaddr) kformstruct);
		 break;
	}
	xvw_set_attribute(selection->label_object, XVW_RIGHT_OF, opt_wid); 

 	selection->opt_object = opt_wid;

	if (selection->opt_selected == 1)
	    xvw_reverse_colors(selection->opt_object, TRUE);
	    
}

/*------------------------------------------------------------
|
|  Routine Name: xvf_delete_opt_button
|
|       Purpose: Given an kformstruct representing a selection, delete
|                the optional box object, remove the appropriate callback
|                for the optional box, and re-position the label object 
|                to be to the far right.
|
|         Input: kformstruct - kformstruct representing selection to have 
|                            optional box deleted
|        Output: none
|          Date: Nov 17, 1992
|    Written By: Danielle Argiro
| Modifications:
|
-------------------------------------------------------------*/
void xvf_delete_opt_button(
   kform_struct *kformstruct)
{
	kselection *selection;
	
	selection = kformstruct->Selptr;

	/*
	 *  move label to far left
	 */
	xvw_set_attributes(selection->label_object, XVW_RIGHT_OF, NULL, NULL); 

	/*
	 *  take the callback off the opt button
	 */
	switch(selection->type)
	{
	    case KUIS_LOGICAL:
		 xvw_remove_callback(selection->opt_object, XVW_BUTTON_SELECT,
                                     xvf_gen_opt_cb, (kaddr) kformstruct);
		 break;

	    case KUIS_FLAG:
            	 xvw_remove_callback(selection->opt_object, XVW_BUTTON_SELECT,
                                     xvf_gen_opt_cb, (kaddr) kformstruct);
		 break;

	    case KUIS_CYCLE:
            	 xvw_remove_callback(selection->opt_object, XVW_BUTTON_SELECT,
                                     xvf_gen_opt_cb, (kaddr) kformstruct);
		 break;

	    case KUIS_LIST:
            	 xvw_remove_callback(selection->opt_object, XVW_BUTTON_SELECT,
                                     xvf_gen_opt_cb, (kaddr) kformstruct);
		 break;

	    case KUIS_TOGGLE:
            	 xvw_remove_callback(selection->opt_object, XVW_BUTTON_SELECT,
                                     xvf_gen_opt_cb, (kaddr) kformstruct);
		 break;
	
	    default:
            	 xvw_remove_callback(selection->opt_object, XVW_BUTTON_SELECT,
                                     xvf_gen_opt_cb, (kaddr) kformstruct);
		 break;
	}
	
	/*
	 * destroy the opt button
	 */
	xvw_destroy(selection->opt_object);
	selection->opt_object = NULL;
}


/*------------------------------------------------------------
|
|  Routine Name: xvf_delete_scrollbar
|
|       Purpose: Given an kformstruct representing an Integer,
|                Float, or Double selection, delete the scrollbar object, 
|                remove the callbacks for the scrollbar box, and re-position 
|                the cr pixmap (if one) to be to right of the text object.
|
|         Input: kformstruct - kformstruct representing selection to have 
|                            optional box deleted
|        Output: none
|          Date: May 10, 1993
|    Written By: Danielle Argiro
| Modifications:
|
-------------------------------------------------------------*/
void xvf_delete_scrollbar(
   kselection *selection)
{
	Line_Info line_info;
	float     text_width;

	/*
	 *  resize text object to use newly unused space
	 */
	kvf_clear_line_info(&line_info);
	kvf_gen_parse(selection->line, &line_info);
	text_width = xvf_calculate_remaining_width(line_info.width,
				line_info.optional, line_info.opt_sel, 
				line_info.title);
	xvw_set_attribute(selection->value_object, XVW_CHAR_WIDTH, text_width);
	kvf_free_line_info_strings(&line_info);

	/*
	 * move <cr> pixmap to just right of text object
	 */
        if (selection->pix_object != NULL)
	{
            xvw_set_attribute(selection->value_object,
                              XVW_LEFT_OF,  selection->pix_object);
	}
	else xvw_set_attribute(selection->value_object, XVW_LEFT_OF, NULL);
	
	xvw_set_attribute(selection->value_object, XVW_TACK_EDGE,
			KMANAGER_TACK_ALL);

	/*
	 *  destroy the scrollbar
	 */
	xvw_destroy(selection->scroll);
        selection->scroll = NULL;

}

/*-----------------------------------------------------------
|
|  Routine Name: xvf_calculate_y_position
|
|       Purpose: Calculates the Y position for a new selection -
|                simply puts it at the bottom of the form by
|                looking at the y position & height of all the
|                widgets on the backplane (as opposed to 
|                kvf_calculate_y_position, which uses the geometries
|                specified by the UIS lines).
|         Input: control - pane, guidepane, or master
|        Output: None
|       Returns: returns the floating point Y position
|    Written By: Danielle Argiro
|          Date: Tues Dec  8, 1992
| Modifications:
|
------------------------------------------------------------*/
float xvf_calculate_y_position(
   kcontrol *control)
{
        int      i, num;
        xvobject parent;
        xvobject *children;
        float    ypos,   max_y      = 0;
        float    height;
        float    final_ypos;

        /*
         *  find the selection with maximum Y position on control
         */
        parent = control->back;
        children = xvw_children(parent, NULL, &num);
        for (i = 0; i < num; i++)
        {
            if (xvw_class(children[i]) == InputOnlyWidgetClass)
               continue;

            xvw_get_attributes(children[i],
                               XVW_CHAR_YPOS,   &ypos,
                               XVW_CHAR_HEIGHT, &height,
                               NULL);
            if ((ypos + height) > max_y)
                max_y = ypos + height;
        }
        kfree(children);
        final_ypos = max_y;
        return(final_ypos);
}

/*-----------------------------------------------------------
|
|  Routine Name: xvf_return_live_selection
|
|       Purpose: When a selection is "live", this is the event
|                handler that is initiated when the user enters a
|                value into the text object and hits <cr>.  It updates
|                the form tree and calls xvf_run_done().
|
|         Input: object      - the text object of the live selection
|                client_data - passes the pointer to the selection
|        Output: none
|       Returns:
|    Written By: Danielle Argiro
|          Date: Jun 26, 1992
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

/* ARGSUSED */
void xvf_return_live_selection(
   xvobject object,
   kaddr    client_data,
   XEvent   *event)
{
        int        live;
	char       ch;
        kselection *selection = (kselection *) client_data;

        selection->modified = TRUE;

        /*
         *  prepare to return control to application
         */
        xvf_get_attribute(selection->back_kformstruct, XVF_LIVE, &live);

        if (live)
        {
            xvf_update_form_tree(selection);
            if (!(xvf_collect_check_data(selection->back_form)))
		return;
            xvf_set_attribute(selection->back_kformstruct, XVF_SELECTED, TRUE);
            xvf_current_subform = selection->back_subform;
            xvf_current_form    = selection->back_form;
            xvf_run_done(TRUE);
        }
        else
        {
            xvf_collect_value(selection);
        }

}

/*-----------------------------------------------------------
|
|  Routine Name: xvf_mark_modified
|
|       Purpose: When the menuform of a selection is used to change
|                an attribute of the selection, such as the title,
|                description, etc, this action handler is called
|                every time a key is released.  The only thing
|                it does is mark the selection as modified, so that
|                xvf_update_item() will update the selection according
|                to the change made in the menuform.
|
|                However, we have to protect against setting the flag on <cr>,
|                because xvf_update_live_selection is also installed on <cr>.
|                When <cr> is the key used, xvf_update_live_selection is called
|                first, then xvf_update_item (which resets modified to FALSE).
|                xvf_mark_modified is called last, causing modified to be set
|                back to TRUE again when it shouldn't be, causing the menuforms
|                not to update correctly when you click on another GUI item of
|                the same type (because xvf_update_menuform_selections sees that
|                modified is TRUE and skips over that gui attribute).
|
|         Input: object      - the text object of the live selection
|                client_data - passes the pointer to the selection
|                event       - event that invoked the action handler
|        Output: none
|       Returns:
|    Written By: Danielle Argiro
|          Date: Sept 23, 1993
| Modifications:
|
------------------------------------------------------------*/

/* ARGSUSED */
void xvf_mark_modified(
   xvobject object,
   kaddr    client_data,
   XEvent   *event)
{
        char ch;
        kselection *selection = (kselection *) client_data;

        if (XLookupString(&(event->xkey), &ch, 1, NULL, NULL)  == 0)
            return;
        if (ch == '\015') /* looking for <cr> in octal */
            return;

        selection->modified = TRUE;
}


void xvf_set_selections_sensitive(
    kselection *sel_list,
    int        sensitive)
{
	kselection *selection;
	xvobject   object;
	int        activate;

	/*
	 * when de-sensitizing selection list, don't want to desensitize 
	 * the quit button, or they won't be able to close the subform
 	 */
	if (sensitive == FALSE)
	{
	    selection = sel_list;
	    while (selection != NULL)
	    {
	        if (selection->type != KUIS_QUIT)
	        {
		    object = xvf_get_xvobject(selection->back_kformstruct, 
				              XVF_BACKPLANE, FALSE);
		    if (object != NULL) xvw_sensitive(object, FALSE);
	        }
	        selection = selection->next;
	    }
	}

	/*
	 * when re-sensitizing selection list, want to only re-sensitize
	 * those selections with activate = TRUE
	 */
	else
	{
	    selection = sel_list;
	    while (selection != NULL)
	    {
		kvf_get_attribute(selection->back_kformstruct,
                                  XVF_ACTIVATE, &activate);
		if (activate)
		{ 
		    object = xvf_get_xvobject(selection->back_kformstruct, 
				              XVF_BACKPLANE, FALSE);
		    if (object != NULL) xvw_sensitive(object, TRUE);
		}
	        selection = selection->next;
	    }
	}
}

