/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>   Graphical User Interface Creation Routines          <<<<
   >>>>                                                       <<<<
   >>>>  Private:                                             <<<<
   >>>>			xvf_create_mainform()		      <<<<
   >>>>			xvf_create_subform()		      <<<<
   >>>>			xvf_create_pane()		      <<<<
   >>>>                                                       <<<<
   >>>>   Static:                                             <<<<
   >>>>			xvf_create_guidepane()		      <<<<
   >>>>			xvf_create_selections()		      <<<<
   >>>>			xvf_create_submenu()		      <<<<
   >>>>			xvf_create_toggle()		      <<<<
   >>>>			xvf_create_group()		      <<<<
   >>>>			xvf_create_toggle()		      <<<<
   >>>>   Public:                                             <<<<
   >>>>			xvf_create_form()		      <<<<
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<  */

#include "internals.h"


static int xvf_create_guidepane PROTO((kcontrol *));
static int xvf_create_toggle    PROTO((kselection *, int));
static int xvf_create_group     PROTO((kselection *, xvobject, int));

/************************************************************
*
*  Routine Name: xvf_create_form - create and map GUI of xvroutine
*
*       Purpose: This is the main driver for the routines that creates
*		 and maps the forms of an xvroutine, both externally 
*                (the GUI made up of objects) and internally (the abstract 
*                data structure referred to as a form tree).  
*
*         Input: filename    - name of the UIS file
*		 glyph_type  - type of glyph associated with this form:
*                              ! NONE or SIMPLE
*		 callback    - optional callback routine for this form 
*                client_data - client data for callback routine
*	         x	     - X location at which to place the newly created
*                              GUI. If values of x and y are both negative 1, 
*                              placement of the GUI must be done manually.
*                y           - Y location at which to place the newly created
*                              GUI. If values of x and y are both negative 1,
*                              placement of the GUI must be done manually.
*                editable    - controls use of the menuforms by the user.
*                              !XVF_FULL_EDIT: if GUI is to be completely editable, eg, 
*                              !               as when creating a GUI with preview or guise.
*                              !XVF_PARTIAL_EDIT: for normal use where the GUI of the 
*                              !               application is to be editable by the user 
*                              !               in ways that will not affect the performance 
*                              !               of the program (for example, button titles 
*                              !               or location).
*                              !XVF_NO_EDIT:   will disable use of the menuforms altogether,
*                              !               so that the user will not be able to change 
*                              !               any aspect of the GUI.
*        Output: 
*	Returns: A pointer to the form tree on success, NULL on failure.
*  Restrictions: 
*    Written By: Danielle Argiro
*          Date: Jul 16, 1992
*      Verified:
*  Side Effects:
* Modifications: 
*
*************************************************************/

kform *xvf_create_form(
   char  *filename,
   int   glyph_type,
   kfunc_void callback,
   kaddr client_data,
   int   x,
   int   y,
   int   editable)
{
	kform *form;

	/*
	 *  Initialize the forms structures
	 */
	xvf_initialize();
	xvw_busy(NULL, TRUE);

	form = kvf_create_form(filename, glyph_type, callback, client_data);
	if (form == NULL)
	{
	    xvw_busy(NULL, FALSE);
	    return(NULL);
	}

	form->editable = editable;

	xvf_create_mainform(form, x, y);

	xvw_busy(NULL, FALSE);
	return(form);

}  /* end xvf_create_form */

/*------------------------------------------------------------
|
|  Routine Name: xvf_create_mainform
|
|       Purpose: Drives the creation of the actual GUI; 
|                assumes that the form tree has already been created.
|
|         Input: form     - pointer to the form tree
|        Output: Returns TRUE on success, FALSE on failure
|          Date: Mar 27, 1992
|    Written By: Danielle Argiro
| Modifications:
|
------------------------------------------------------------*/

int xvf_create_mainform(
   kform *form,
   int   x,
   int   y)
{
	ksubform   *subform;
	kselection *selection;
	Line_Info  line_info;

	kvf_set_context(KVF_GUI);

	/*
	 *  no master form => simply create single subform
	 */
	if (form->master == NULL) 
	{
	    xvf_create_subform(form->subform, x, y, TRUE);
	    return(TRUE);
	}
	else if (form->master->back != NULL)
	    return(TRUE);

	if (form->toplevel != NULL)
	    return(TRUE);

	/*
	 *  make sure that the toplevel object has a unique name
	 */
	kvf_clear_line_info(&line_info);
	kvf_parse_startform_line(form->control_line, &line_info);

	/*
	 * create toplevel object
	 */
	form->toplevel = xvw_create_application_shell(kprog_get_program(), 
						      NULL, NULL);

	if (line_info.title != NULL)
	    xvw_set_attribute(form->toplevel, XVW_SHELL_TITLE, line_info.title);

	if (x != -1) xvw_set_attribute(form->toplevel, XVW_SHELL_X, x);
	if (y != -1) xvw_set_attribute(form->toplevel, XVW_SHELL_Y, y);

	if (form->toplevel == NULL)
        {
	    errno = KWIDGET_CREATION;
	    kerror("xvforms", "xvf_create_mainform", 
		      "Couldn't create toplevel object for forms");
            return(FALSE);
	}

	/* 
	 * create backplane of the form 
         */
        form->master->back = xvf_create_backplane((kaddr) form, KUIS_STARTFORM,
					          form->toplevel, TRUE);
	if (form->master->back == NULL)
        {
	    errno = KWIDGET_CREATION;
	    kerror("xvforms", "xvf_create_mainform", 
		     "Couldn't create backplane for master form");
            return(FALSE);
        }
	
	/*
	 *  create all selections on master
	 */
	selection = form->master->sel_list;
	while (selection != NULL)
	{
	    xvf_create_selection(selection, form->master->back, KMASTER);
	    selection = selection->next;
	}


	/*
	 * create all subform buttons on master
	 */
	subform = form->master->subform_list;
	while (subform != NULL)
	{
	    if (subform->back_submenu == NULL)
	    {
	        switch (subform->type)
	        {
		case KUIS_SUBFORMBUTTON:
	             subform->button = xvf_create_subform_sel(subform, 
						form->master->back);
                     if (subform->button == NULL)
                     { 
			  xvf_object_creation_error_mesg("xvf_create_mainform",
                                                     "subform button");
                          return(FALSE);
                     }
		     break;

		default:
	    	     errno = KINTERNAL;
		     kerror("xvforms",  "xvf_create_mainform",
                            "Unexpected subform struct type");
                     return(FALSE);
	        }
	    }
	    subform = subform->next;
	}

	xvf_map_form(form);

	/*
	 *  highlight selections on master 
	 */
	subform = form->master->subform_list;
	while(subform != NULL)
	{
	     if (subform->selected) 
	     {
		  xvw_reverse_colors(subform->button, TRUE);
		  xvf_map_subform(subform);
	     }
	     subform = subform->next;
	}
	if (form->master != NULL)
	    xvf_highlight_objects(form->master->sel_list);


	kvf_free_line_info_strings(&line_info);

	return(TRUE);

} /* end xvf_create_mainform */


/*-----------------------------------------------------------
|
|  Routine Name: xvf_create_subform
|
|       Purpose: Drives the creation of the objects in a subform; 
|                assumes that the subform branch has already been created.
|
|         Input: subform  - pointer to this subform branch of the form tree
|                x, y     - locations for automatic GUI placement 
|                           (-1, -1) for manual placement
| 		 mapped   - TRUE if subform is to be mapped immediately.
|        Output: Returns TRUE on success, FALSE on failure
|
|          Date: Mar 27, 1992
|    Written By: Danielle Argiro
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

int xvf_create_subform(
   ksubform *subform,
   int      x,
   int      y,
   int      mapped)
{
        Line_Info   line_info;
	kguide      *guide;


	/*
	 *  make sure that subform has not already been created
	 */
	if (subform->back != NULL)
	   return(TRUE);

	/*
	 *  make sure that the toplevel object has a unique name
	 */
	/* parse the KUIS_STARTSUBFORM line */
        kvf_clear_line_info(&line_info);
        if (!(kvf_parse_startsubform_line(subform->control_line, &line_info)))
	{
	    kvf_free_line_info_strings(&line_info);
            return(FALSE);
	}

	/* 
	 * create the subform's toplevel form 
         */
        subform->toplevel = xvw_create_application_shell(kprog_get_program(),
						         NULL, NULL);
	xvw_set_attribute(subform->toplevel, XVW_MAP_WHEN_MANAGED, mapped);
	if (x != -1) xvw_set_attribute(subform->toplevel, XVW_SHELL_X, x);
	if (y != -1) xvw_set_attribute(subform->toplevel, XVW_SHELL_Y, y);

	if (line_info.title != NULL)
	    xvw_set_attribute(subform->toplevel, XVW_SHELL_TITLE, 
			      line_info.title);
 
	/* 
	 * create backplane for the subform 
         */
        subform->back = xvf_create_backplane((kaddr) subform, 
					    KUIS_STARTSUBFORM, 
					    subform->toplevel,
                                            mapped);
	kvf_free_line_info_strings(&line_info);
        if (subform->back == NULL)
        {
	    errno = KWIDGET_CREATION;
	    kerror("xvforms", "xvf_create_subform", 
		     "Couldn't create subform backplane");
            return(FALSE);
        }

	/* 
	 * need to set the correct guide
         */
	if (subform->guidepane != NULL) 
        {
	    /* 
	     * create the guide pane 
             */
	    if (!(xvf_create_guidepane(subform->guidepane)))
                return(FALSE);

	    /* 
	     * find the guide button that is selected on guide pane
	     */
	    guide = subform->guidepane->guide_list;
	    while ((guide->pane == NULL || guide->selected != TRUE) && 
		   (guide->next != NULL))
	        guide = guide->next;
	}
	else guide = subform->guide;

	/* 
	 * create the pane that is displayed by default 
         */
	if (guide != NULL)
	{
	    if (guide->selected) 
            {
		if (!(xvf_create_pane(guide->pane, TRUE)))
                        return(FALSE);
            }
	}

        /*
         *  Map the subform if mapped is set to TRUE, otherwise the subform,
         *  by default, will be left unmapped.
         */
        if (mapped == TRUE)
           xvf_map_subform(subform);

	/*
	 *  highlight objects on displayed pane
	 */
	 if (guide != NULL)
	 {
	     if (guide->selected)
	     {
		 if (guide->button != NULL)
	 	     xvw_reverse_colors(guide->button, TRUE);
	     }
	 }

	/*
	 *  highlight objects on subform
	 */
	if (subform->guidepane != NULL)
	     xvf_highlight_objects(subform->guidepane->sel_list);

	kvf_free_line_info_strings(&line_info);

	return(TRUE);

} /* end xvf_create_subform */

/*------------------------------------------------------------
|
|  Routine Name: xvf_create_guidepane
|
|       Purpose: Drives the creation of the objects in a guidepane; 
|                assumes that the guidepane control panel struct 
|                has already been created.
|
|         Input: guidepane - pointer to this guidepane
|        Output: Returns TRUE on success, FALSE on failure
|          Date: Mar 27, 1992
|    Written By: Danielle Argiro
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

static int xvf_create_guidepane(
   kcontrol *guidepane)
{
        Line_Info     line_info;
        kguide     *guide;
	kselection *selection;

	if (guidepane->back != NULL)
	    return(TRUE);

	/* parse the KUIS_STARTGUIDE line */
        kvf_clear_line_info(&line_info);
	if (!(kvf_parse_startguide_line(guidepane->control_line, &line_info)))
                    return(FALSE);

	/* 
	 * create the backplane for the guidepane 
         */
	guidepane->back = xvf_create_backplane((kaddr) guidepane, 
					 KUIS_STARTGUIDE, 
					 guidepane->back_subform->back, TRUE);
        if (guidepane->back == NULL)
            xvf_object_creation_error_mesg("xvf_create_guidepane", 
				       "guide backplane");
	/*
	 *  create all selections on guide pane
	 */
	selection = guidepane->sel_list;
	while (selection != NULL)
	{
	    xvf_create_selection(selection, guidepane->back, KGUIDEPANE);
	    selection = selection->next;
	}

        /*
	 * create all guide buttons on the guide pane
         */
	guide = guidepane->guide_list;
        while (guide != NULL)
        {
	    if (guide->back_submenu == NULL)
                guide->button = xvf_create_guide_sel(guide, guidepane->back);

            if (guide->button == NULL) 
	    {
                xvf_object_creation_error_mesg("xvf_create_guidepane", 
					   "guide button");
		return(FALSE);
	    }
	    guide = guide->next;
	}

	kvf_free_line_info_strings(&line_info);

	return(TRUE);

}  /* end xvf_create_guidepane */

/*-----------------------------------------------------------
|
|  Routine Name: xvf_create_pane
|
|       Purpose: Drives the creation of the objects in a pane; 
|                assumes that the pane control panel struct 
|                has already been created.
|
|         Input: pane   - pointer to this control panel struct
|                parent - parent of pane's backplane
|                mapped - TRUE if pane is to be mapped immediately
|
|        Output: Returns TRUE on success, FALSE on failure
|          Date: Mar 27, 1992
|    Written By: Danielle Argiro
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/


int xvf_create_pane(
   kcontrol *pane,
   int      mapped)
{
	Line_Info     line_info;
	kselection *selection;

	if (pane->back != NULL)
	    return(TRUE);
	
	if (pane->back_subform->back == NULL)
	{
	    return(xvf_create_subform(pane->back_subform, -1, -1, mapped));
	}

	/* parse the KUIS_STARTPANE line */
	kvf_clear_line_info(&line_info);
        if (!(kvf_parse_startpane_line(pane->control_line, &line_info)))
	{
	    kvf_free_line_info_strings(&line_info);
            return(FALSE);
	}

	/* 
	 * create backplane of the pane 
	 */
	pane->back = xvf_create_backplane((kaddr) pane, KUIS_STARTPANE,
					  pane->back_subform->back, 
					  mapped);
        if (pane->back == NULL)
        {
            xvf_object_creation_error_mesg("xvf_create_pane", "pane backplane");
            return(FALSE);
        }

	/*
	 *  if it is a single pane on a single subform, tack it to parent
	if (pane->back_subform->guidepane == NULL)
	    xvw_set_attribute(pane->back, XVW_TACK_EDGE, KMANAGER_TACK_ALL);
	 */

	/* 
	 * create all the selections on the pane 
	 */
	selection = pane->sel_list;
	while (selection != NULL)
        {
	     if (!(xvf_create_selection(selection, pane->back, KPANE)))
		return(FALSE);
	     selection = selection->next;
        }
	xvf_highlight_objects(pane->sel_list);

	kvf_free_line_info_strings(&line_info);
        return(TRUE);

} /* end xvf_create_pane */

/*-----------------------------------------------------------
|
|  Routine Name: xvf_create_selection
|
|       Purpose: Drives the creation of the objects in a selection; 
|                assumes that the selection node has already been created.
|
|         Input: selection     - pointer to selection node 
|		 parent        - parent object of this selection
|                location_flag - indicates location of selection on GUI;
|                                one of Master, GuidePane, or Pane
|
|        Output: Returns TRUE on success, FALSE on failure
|          Date: May 7, 1992
|    Written By: Danielle Argiro
| Modifications:
|
------------------------------------------------------------*/


int xvf_create_selection(
   kselection *selection,
   xvobject   parent,
   int        location_flag)
{
	if (selection->back_submenu == NULL)
	{
	    if (selection->back != NULL)
		return(TRUE);

	    switch (selection->type)
	    {
	    case KUIS_MASTERACTION:
		 selection->back = xvf_create_master_action_sel(selection,
						                parent);
                 if (selection->back == NULL) 
		 {
                     xvf_object_creation_error_mesg("xvf_create_selection", 
		    				    "subform action button");
		     return(FALSE);
		 }
		 selection->value_object = 
		 selection->label_object = selection->back;
		 break;
	
	    case KUIS_STARTSUBMENU:
		 selection->back = xvf_create_submenu_sel(selection, parent);
                 if (selection->back == NULL) 
		 {
                     xvf_object_creation_error_mesg("xvf_create_selection", 
						    "subform action button");
		     return(FALSE);
		 }
		 if (!(xvf_create_submenu(selection->submenu_next, 
					  location_flag)))
		     return(FALSE);

		 break;

            case KUIS_SUBFORMACTION:
		 selection->back = xvf_create_subform_action_sel(selection, 
						                 parent);
                 if (selection->back == NULL) 
		 {
                     xvf_object_creation_error_mesg("xvf_create_selection", 
						    "subform action button");
		     return(FALSE);
		 }
		 selection->value_object = 
		 selection->label_object = selection->back;
		 break;

	    case KUIS_INPUTFILE:
		 selection->back  = xvf_create_input_sel(selection, parent);
		 if (selection->back  == NULL) 
	         {
                     xvf_object_creation_error_mesg("xvf_create_selection", 
					            "input selection");
	             return (FALSE);
		 }
		 break;

	    case KUIS_OUTPUTFILE:
		 selection->back  = xvf_create_output_sel(selection, parent);
                 if (selection->back  == NULL) 
	         {
                     xvf_object_creation_error_mesg("xvf_create_selection", 
					            "output selection");
	             return (FALSE);
		 }
		 break;

	    case KUIS_STDIN:
		 selection->back  = xvf_create_std_sel(selection, parent);
                 if (selection->back  == NULL) 
	         {
                     xvf_object_creation_error_mesg("xvf_create_selection", 
					            "stdin selection");
	             return (FALSE);
		 }
		 break;

	    case KUIS_STDOUT:
		 selection->back  = xvf_create_std_sel(selection, parent);
                 if (selection->back  == NULL) 
	         {
                     xvf_object_creation_error_mesg("xvf_create_selection", 
					            "stdout selection");
	             return (FALSE);
		 }
		 break;

	    case KUIS_INTEGER:
                 selection->back  = xvf_create_int_sel(selection, parent);
                 if (selection->back == NULL) 
	         {
                     xvf_object_creation_error_mesg("xvf_create_selection", 
					            "integer selection");
	             return (FALSE);
		 }
		 break;

	    case KUIS_FLOAT:
		 selection->back = xvf_create_float_sel(selection, parent);
                 if (selection->back == NULL) 
	         {
                     xvf_object_creation_error_mesg("xvf_create_selection", 
					            "float selection");
	             return (FALSE);
		 }
		 break;

	    case KUIS_DOUBLE:
		 selection->back = xvf_create_double_sel(selection, parent);
                 if (selection->back == NULL) 
	         {
                     xvf_object_creation_error_mesg("xvf_create_selection", 
					            "double selection");
	             return (FALSE);
		 }
		 break;

	    case KUIS_STRING:
                 selection->back = xvf_create_string_sel(selection, parent);
                 if (selection->back == NULL) 
	         {
                     xvf_object_creation_error_mesg("xvf_create_selection", 
					            "string selection");
	             return (FALSE);
		 }
		 break;

	    case KUIS_STRINGLIST:
                 selection->back = xvf_create_stringlist_sel(selection, parent);
                 if (selection->back == NULL) 
	         {
                     xvf_object_creation_error_mesg("xvf_create_selection", 
					            "stringlist selection");
	             return (FALSE);
		 }
		 break;

            case KUIS_LOGICAL:
		 selection->back = xvf_create_logic_sel(selection, parent);
                 if (selection->back == NULL) 
	         {
                     xvf_object_creation_error_mesg("xvf_create_selection", 
					            "logical selection");
	             return (FALSE);
		 }
		 break;

            case KUIS_FLAG:
		 selection->back = xvf_create_flag_sel(selection, parent);
                 if (selection->back == NULL) 
	         {
                     xvf_object_creation_error_mesg("xvf_create_selection", 
					            "flag selection");
	             return (FALSE);
		 }
		 break;

	    case KUIS_ROUTINE:
                 selection->back = xvf_create_routine_sel(selection, 
						parent);
                 if (selection->back == NULL) 
	         {
                     xvf_object_creation_error_mesg("xvf_create_selection", 
					            "routine button");
	             return (FALSE);
		 }
		 selection->value_object = 
		 selection->label_object = selection->back;
		 break;

	    case KUIS_CYCLE:
		 selection->back = xvf_create_cycle_sel(selection, parent);
                 if (selection->back == NULL) 
	         {
                     xvf_object_creation_error_mesg("xvf_create_selection", 
					            "cycle selection");
	             return (FALSE);
		 }
		 break;

	    case KUIS_LIST:
		 selection->back = xvf_create_list_sel(selection, parent);
		 if (selection->back == NULL) 
	         {
                     xvf_object_creation_error_mesg("xvf_create_selection", 
					            "list selection");
	             return (FALSE);
		 }
		 break;

            case KUIS_DISPLAYLIST:
                 selection->back = xvf_create_displaylist_sel(selection, parent);
                 if (selection->back == NULL)
                 {
                     xvf_object_creation_error_mesg("xvf_create_selection",
                                                    "displaylist selection");
                     return (FALSE);
                 }
                 break;

	    case KUIS_TOGGLE:
                 selection->back = xvf_create_toggle_sel(selection, parent);
                 if (selection->back == NULL) 
	         {
                     xvf_object_creation_error_mesg("xvf_create_selection", 
					            "toggle backplane");
	             return (FALSE);
		 }
		 if (!(xvf_create_toggle(selection, location_flag)))
			return(FALSE);
		 break;

	    case KUIS_HELP:
                 selection->back = xvf_create_help_sel(selection, parent);
                 if (selection->back == NULL) 
	         {
                     xvf_object_creation_error_mesg("xvf_create_selection", 
					            "help button");
	             return (FALSE);
		 }
		 selection->value_object =
		 selection->label_object = selection->back;
	         break;

            case KUIS_WORKSPACE:
                 selection->back = xvf_create_workspace_sel(selection, parent);
                 if (selection->back == NULL) 
	         {
                     xvf_object_creation_error_mesg("xvf_create_selection", 
					            "workspace object");
	             return (FALSE);
		 }
		 break;

	    case KUIS_PANEACTION:
                 selection->back = xvf_create_pane_action_sel(selection, 
						              parent);
                 if (selection->back == NULL) 
	         {
                     xvf_object_creation_error_mesg("xvf_create_selection", 
					            "action button");
	             return (FALSE);
		 }
		 selection->value_object = 
		 selection->label_object = selection->back;
		 break;



            case KUIS_BLANK:
                 selection->back = xvf_create_blank_sel(selection, parent);
                 if (selection->back == NULL) 
	         {
                     xvf_object_creation_error_mesg("xvf_create_selection", 
					            "blank selection");
	             return (FALSE);
		 }
		 selection->value_object = 
		 selection->label_object = selection->back;
		 break;

	    case KUIS_QUIT:
                 selection->back = xvf_create_quit_sel(selection, parent, 
						       location_flag);
                 if (selection->back == NULL) 
	         {
                     xvf_object_creation_error_mesg("xvf_create_selection", 
					            "quit button");
	             return (FALSE);
		 }
		 selection->value_object = 
		 selection->label_object = selection->back;
		 break;

	    case KUIS_MUTEXCL:
	    case KUIS_MUTINCL:
	    case KUIS_GROUP:
		 if (!(xvf_create_group(selection, parent, location_flag)))
			return(FALSE);
		 break;
	    
	     default:
	    	     errno = KINTERNAL;
		     kerror("xvforms", "xvf_create_selection",
                            "Unexpected selection struct type\n");
		     return(FALSE);
	    }
	}
	return(TRUE);

}  /* end xvf_create_selection */

/*-----------------------------------------------------------
|
|  Routine Name: xvf_create_submenu
|
|       Purpose: Drives the creation of the objects in a submenu; 
|                assumes that the submenu list has already been created.
|
|         Input: kformstruct     - pointer to the first item on the submenu
|                location_flag - indicates location of submenu on GUI;
|                                one of Master, GuidePane, or Pane
|        Output: Returns TRUE on success, FALSE on failure
|          Date: Apr 30, 1992
|    Written By: Danielle Argiro
| Modifications:
|
------------------------------------------------------------*/
int xvf_create_submenu(
   kform_struct *kformstruct,
   int          location_flag)
{
	while (kformstruct != NULL)
	{
	    switch(kformstruct->flag)
	    {
		case KUIS_MASTERACTION:
		     kformstruct->Selptr->back = 
			xvf_create_master_action_sel(kformstruct->Selptr,
			  	  kformstruct->Selptr->back_submenu->back);
                     if (kformstruct->Selptr->back == NULL) 
		     {
                          xvf_object_creation_error_mesg("xvf_create_submenu", 
						     "master action button");
			  return(FALSE);
		     }
		     kformstruct->Selptr->value_object =
                     kformstruct->Selptr->label_object = 
					kformstruct->Selptr->back;
		     break;

		case KUIS_SUBFORMACTION:
		     kformstruct->Selptr->back = 
			xvf_create_subform_action_sel(kformstruct->Selptr,
                                   kformstruct->Selptr->back_submenu->back);
                     if (kformstruct->Selptr->back == NULL)
                     {
                          xvf_object_creation_error_mesg("xvf_create_submenu",
                                                     "subform action button");
                          return(FALSE);
                     }
		     kformstruct->Selptr->value_object =
                     kformstruct->Selptr->label_object = 
					kformstruct->Selptr->back;
                     break;

		case KUIS_PANEACTION:
		     kformstruct->Selptr->back = 
			xvf_create_pane_action_sel(kformstruct->Selptr,
                                  kformstruct->Selptr->back_submenu->back);
                     if (kformstruct->Selptr->back == NULL)
                     {
                          xvf_object_creation_error_mesg("xvf_create_submenu",
                                                     "pane action button");
                          return(FALSE);
                     }
		     kformstruct->Selptr->value_object =
                     kformstruct->Selptr->label_object = 
					kformstruct->Selptr->back;
                     break;

		case KUIS_BLANK:
		     kformstruct->Selptr->back = 
			xvf_create_blank_sel(kformstruct->Selptr,
                                  kformstruct->Selptr->back_submenu->back);
                     if (kformstruct->Selptr->back == NULL)
                     {
                          xvf_object_creation_error_mesg("xvf_create_submenu",
                                                     "blank selection");
                          return(FALSE);
                     }
		     kformstruct->Selptr->value_object =
                     kformstruct->Selptr->label_object = 
					kformstruct->Selptr->back;
                     break;

		case KUIS_HELP:
		     kformstruct->Selptr->back = 
			xvf_create_help_sel(kformstruct->Selptr,
                                  kformstruct->Selptr->back_submenu->back);
                     if (kformstruct->Selptr->back == NULL)
                     {
                          xvf_object_creation_error_mesg("xvf_create_submenu",
                                                     "help button");
                          return(FALSE);
                     }
		     kformstruct->Selptr->value_object =
                     kformstruct->Selptr->label_object = 
					kformstruct->Selptr->back;
		     break;
		     
		case KUIS_QUIT:
		     kformstruct->Selptr->back = 
			xvf_create_quit_sel(kformstruct->Selptr,
                                  kformstruct->Selptr->back_submenu->back,
			          location_flag);
                     if (kformstruct->Selptr->back == NULL)
                     {
                          xvf_object_creation_error_mesg("xvf_create_submenu",
                                                     "quit button");
                          return(FALSE);
                     }
		     kformstruct->Selptr->value_object =
                     kformstruct->Selptr->label_object = 
					kformstruct->Selptr->back;
		     break;

		case KUIS_ROUTINE:
		     kformstruct->Selptr->back = 
			xvf_create_routine_sel(kformstruct->Selptr,
                                  kformstruct->Selptr->back_submenu->back);
                     if (kformstruct->Selptr->back == NULL)
                     {
                          xvf_object_creation_error_mesg("xvf_create_submenu",
                                                     "routine button");
                          return(FALSE);
                     }
		     kformstruct->Selptr->value_object =
                     kformstruct->Selptr->label_object = 
					kformstruct->Selptr->back;
		     break;

		case KUIS_SUBFORMBUTTON:
                     kformstruct->Subformptr->button = 
			xvf_create_subform_sel(kformstruct->Subformptr,
			          kformstruct->Subformptr->back_submenu->back);
                     if (kformstruct->Subformptr->button == NULL)
                     {
                          xvf_object_creation_error_mesg("xvf_create_submenu",
                                                     "subform button");
                          return(FALSE);
                     }
                     break;

		case KUIS_GUIDEBUTTON:
                     kformstruct->Guideptr->button = 
			xvf_create_guide_sel(kformstruct->Guideptr,
			          kformstruct->Guideptr->back_submenu->back);
                     if (kformstruct->Guideptr->button == NULL)
                     {
                          xvf_object_creation_error_mesg("xvf_create_submenu",
                                                     "guide button");
                          return(FALSE);
                     }
                     break;

	    }
	    kformstruct = kformstruct->next;
	}
	return(TRUE);

}  /* end xvf_create_submenu */

/*------------------------------------------------------------
|
|  Routine Name: xvf_create_toggle
|
|       Purpose: Drives the creation of the objects in a toggle; 
|                assumes that the toggle list has already been created.
|
|         Input: toggle_start  - pointer to KUIS_TOGGLE selection 
|                location_flag - indicates location of toggle on GUI;
|                                one of Master, GuidePane, or Pane
|        Output: Returns TRUE on success, FALSE on failure
|          Date: Mar 27, 1992
|    Written By: Danielle Argiro
| Modifications: Converted from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

static int xvf_create_toggle(
   kselection *toggle_start,
   int           location_flag)
{
	kselection *toggle;

	toggle = toggle_start->toggle_next;
	while (toggle != NULL)
        {
	    if (!(xvf_create_selection(toggle, toggle_start->back, 
					  location_flag)))
		return(FALSE); 
	    toggle = toggle->next;
	}
	return(TRUE);

} /* end xvf_create_toggle */

/*-----------------------------------------------------------
|
|  Routine Name: xvf_create_group
|
|       Purpose: Drives the creation of the members in a mutually
|                exclusive group; assumes that the ME list has already 
|                been created.
|
|         Input: group_start     - pointer to KUIS_MUTEXCL/KUIS_MUTINCL selection 
|                location_flag   - indicates location of group on GUI;
|                                  one of Master, GuidePane, or Pane
|
|        Output: Returns TRUE on success, FALSE on failure
|          Date: Jun 10, 1992
|    Written By: Danielle Argiro
| Modifications:
|
------------------------------------------------------------*/

static int xvf_create_group(
   kselection *group_start,
   xvobject   parent,
   int        location_flag)
{
	kselection *group;

	group = group_start->group_next;
	while (group != NULL)
        {
	    if (group->group_next != NULL)
		xvf_create_group(group, parent, location_flag);

	    else if (!(xvf_create_selection(group, parent, location_flag)))
		return(FALSE); 
	    group = group->next;
	}
	return(TRUE);

}  /* end xvf_create_group */

