 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";

#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Khoros RAST Data Service Routines
   >>>>
   >>>>   Static:
   >>>>  Private:
   >>>>             rast_check()
   >>>>             rast_input()
   >>>>             rast_output()
   >>>>             _init()
   >>>>             rast_destroy()
   >>>>   Public:
   >>>>             None - no public should ever exist as these are
   >>>>                    internal routines only accessible via the
   >>>>                    DataServiceInformation *services[] structure.
   >>>>
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"


#if !defined(KRAST_DEF)
DataServiceInformation rast_format[] =
{NULL};

#else

#include "kdatafmt/rast.h"


static int rast_check PROTO((int));
static int rast_input PROTO((kobject, int, int));
static int rast_output PROTO((kobject, int, int));
static int rast_destroy PROTO((kobject));
static int rast_order PROTO((char *, int *));
static int rast_architecture PROTO((kobject));

DataServiceInformation rast_format[] =
{
   {
      "Sun Rasterfile File Format (rast)",
      "rast",
      rast_check,
      rast_input,
      rast_output,
      rast_destroy,
      NULL,
      NULL,
      rast_order,
      rast_architecture,
      NULL,
      NULL,
   }
};

/*
 *  Internal Resource Structure for the following data services
 *
 *              rast  -  Sun Rasterfile File Format Utilities (rast)
 */
typedef struct
{
   kfile *file;
   rast *image;
}
ResourceStruct;


/*-----------------------------------------------------------
|
|  Routine Name: _init - Creates an rast image
|
|       Purpose: This function is used to create an rast image.  Which
|                means that we use the current dimension values in order
|                to create a proper rast.
|
|         Input: data - initial sample of data
|                num  - number of bytes in the sample
|
|        Output: Returns: returns TRUE or FALSE if an error occurs
|
|    Written By: Mark Young
|          Date: Jul 13, 1992 15:50
| Modifications:
|
------------------------------------------------------------*/

static int
_init(kobject object)
{
   ResourceStruct *resources;


   /*
    * Create the image to the object file
    */
   resources = (ResourceStruct *) kcalloc(1, sizeof(ResourceStruct));

   if (resources == NULL)
      return (FALSE);

   _kdms_glue_set_resources(object, resources);
   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: rast_check - Checks to see if the file is
|                                an rast
|
|       Purpose: This function is used to check the first few bytes of
|                the stream to see if it is an rast.
|
|         Input: object - the rast object to be initialized
|
|        Output: Returns: returns TRUE or FALSE if an error occurs
|
|    Written By: Mark Young
|          Date: Jul 13, 1992 15:50
| Modifications:
|
------------------------------------------------------------*/

static int
rast_check(int fid)
{
   int machtype;
   int magic = 0;


   /*
    * Save the current machine type, then set it to read a Sun machine type.
    */
   machtype = kfile_getmachtype(fid);
   kfile_setmachtype(fid, KMACH_SPARC);

   /*
    * Read the magic identifier, then restore the machine type to the the
    * original incomming machine type.
    */
   kread_int(fid, &magic, 1);
   kfile_setmachtype(fid, machtype);

   /*
    * Return whether the magic identifier matches the Rasterfile Magic
    * define.  This will indicate whether we are a Sun Rasterfile or not.
    */
   return (magic == RAS_MAGIC);
}



/*-----------------------------------------------------------
|
|  Routine Name: rast_input - Reda a rast image
|
|       Purpose: This function is used to read in the image if the
|                supplied data is rast image.  The data was checked
|                by the rast_check() routine in which the rast
|                identifier indicated that the data is a valid rast.
|                Since the data was valid we can assume simply read
|                the header in and initialize that data segments.
|
|         Input: object - the rast object to be initialized
|
|        Output: Returns: returns TRUE or FALSE if an error occurs
|
|    Written By: Mark Young
|          Date: Jul 13, 1992 15:50
| Modifications:
|
------------------------------------------------------------*/

/*ARGSUSED */
static int
rast_input(kobject object, int fid, int flags)
{
   ResourceStruct *resources;
   rast *image;

   int type = KUBYTE;
   int i;
   int size[KDMS_MAX_DIM];
   int order[KDMS_MAX_DIM];
   int begin[KDMS_MAX_DIM];
   int end[KDMS_MAX_DIM];
   int num;

   if (!_kdms_initialized(object))
      _init(object);

   resources = (ResourceStruct *) _kdms_glue_get_resources(object);

   /*
    * Read the image from the object file and go ahead and set the rast image
    * within the resource structure.
    */
   image = resources->image = rast_fdread(fid);
   if (image == NULL)
   {
      _kdms_set_error(KDMS_EFMT_FAILURE);
      return (FALSE);
   }

   /*
    * Initialize the data segments according to the data found in the rast
    * structure.
    * 
    * The rast structure only contains value (image->data)
    */

   /* check the value data */
   if (image->ras_width != 0 && image->ras_height != 0)
   {
      if (image->ras_depth == 1)
      {
         type = KBIT;
         num = 1;
	 /* invert the values since bit 1's are black and bit 0's are white */
	 for (i = 0; i < ((image->ras_width * image->ras_height) >> 3); i++)
	    image->data[i] = ~image->data[i];
      }
      else
      {
         type = KUBYTE;
         num = image->ras_depth >> 3;
      }

      size[0] = num;
      size[1] = image->ras_width;
      size[2] = image->ras_height;
      size[3] = size[4] = 1;

      order[0] = KELEMENTS;
      order[1] = KWIDTH;
      order[2] = KHEIGHT;
      order[3] = KDEPTH;
      order[4] = KTIME;

      for (i = 0; i < KDMS_MAX_DIM; i++)
      {
	 end[i] = size[i] - 1;
	 begin[i] = 0;
      }
      
      /*
       * if the bit depth is 24 or 32, then things could be difficult.
       * for some unfathomable reason, data can be stored either BGR
       * or RGB, depending on the type field.  If its 32 bit, then the
       * first byte of each pixel is padding.
       */
      switch (num)
      {
	 case 1:
	    kdms_create_segment(object, KDMS_SEGMENT_VALUE);
	    kdms_set_attributes(object, KDMS_SEGMENT_VALUE,
				KDMS_DIMENSION, 5,
				KDMS_INDEX_ORDER, order,
				KDMS_SIZE, size,
				KDMS_DATA_TYPE, type,
				NULL);
	    kdms_put_data(object, KDMS_SEGMENT_VALUE, begin, end, image->data);
	    break;
	 case 3:
	    /*
	     * 	if the type is RGB, things are easy,  if not, then the data
	     * is probably stored BGR, in which case things are a hassle.
	     * Technically, we gotta check to see if conversion was from IFF
	     * or TIFF, in which case, things could be different, but this
	     * is good enough for now.
	     */
	    if (image->ras_type != RT_FORMAT_RGB)
	    {
	       unsigned char *data;
	       int num_pixels = size[1] * size[2];
	       if ((data = (unsigned char*)kmalloc((unsigned)(num_pixels * 3)))
		    == NULL)
		  return FALSE;
	       for (i = 0; i < num_pixels; i++)
	       {
		  data[3*i] = image->data[3*i + 2];
		  data[3*i + 1] = image->data[3*i + 1];
		  data[3*i + 2] = image->data[3*i];
	       }
	    
	       kdms_create_segment(object, KDMS_SEGMENT_VALUE);
	       kdms_set_attributes(object, KDMS_SEGMENT_VALUE,
				   KDMS_DIMENSION, 5,
				   KDMS_INDEX_ORDER, order,
				   KDMS_SIZE, size,
				   KDMS_DATA_TYPE, type,
				   NULL);
	       kdms_put_data(object, KDMS_SEGMENT_VALUE, begin, end, data);
	    }
	    else
	    {
	       kdms_create_segment(object, KDMS_SEGMENT_VALUE);
	       kdms_set_attributes(object, KDMS_SEGMENT_VALUE,
				   KDMS_DIMENSION, 5,
				   KDMS_INDEX_ORDER, order,
				   KDMS_SIZE, size,
				   KDMS_DATA_TYPE, type,
				   NULL);
	       kdms_put_data(object, KDMS_SEGMENT_VALUE, begin, end, 
			     image->data);
	    }	    
	    break;
	 case 4:
	    {
	       unsigned char *data;
	       int num_pixels = size[1] * size[2];
	       if ((data = (unsigned char*)kmalloc((unsigned)(num_pixels * 3)))
		   == NULL)
		  return FALSE;

	       if (image->ras_type != RT_FORMAT_RGB)
	       {
		  for (i = 0; i < num_pixels; i++)
		  {
		     data[3*i] = image->data[4*i + 3];
		     data[3*i + 1] = image->data[4*i + 2];
		     data[3*i + 2] = image->data[4*i + 1];
		  }
	       }
	       else
	       {
		  for (i = 0; i < num_pixels; i++)
		  {
		     data[3*i] = image->data[4*i + 1];
		     data[3*i + 1] = image->data[4*i + 2];
		     data[3*i + 2] = image->data[4*i + 3];
		  }
	       }
	       
	       kdms_create_segment(object, KDMS_SEGMENT_VALUE);
	       kdms_set_attributes(object, KDMS_SEGMENT_VALUE,
				   KDMS_DIMENSION, 5,
				   KDMS_INDEX_ORDER, order,
				   KDMS_SIZE, size,
				   KDMS_DATA_TYPE, type,
				   NULL);
	       kdms_put_data(object, KDMS_SEGMENT_VALUE, begin, end, data);
	    }
	    break;
	 default:
	    kerror("kdataman", "kdms_open", "Unrecognized pixel depth (%d) "
		   "in Sun Raster file.", image->ras_depth);
      }
      /*
       * it is vitally important that you include this step because if the
       * data set larger than the buffer threshold, then spurous flushes
       * will ensue that should not happen...it turns out that xvimages and
       * viffs will cope with that situation (though not very efficiently),
       * but file formats that buffer the whole thing in memory won't.
       */
      kdms_set_attribute(object, KDMS_SEGMENT_VALUE, KDMS_COUPLING,KUNCOUPLED);
   }

   if (image->ras_maptype != RMT_NONE && image->map)
   {
      if (image->ras_maptype == RMT_RAW)
         num = 1;
      else
         num = 3;

      size[0] = image->ras_maplength / num;
      size[1] = num;
      size[2] = size[3] = size[4] = 1;

      order[0] = KMAP_HEIGHT;
      order[1] = KMAP_WIDTH;
      order[2] = KMAP_ELEMENTS;
      order[3] = KMAP_DEPTH;
      order[4] = KMAP_TIME;

      for (i = 0; i < KDMS_MAX_DIM; i++)
      {
         end[i] = size[i] - 1;
         begin[i] = 0;
      }

      kdms_create_segment(object, KDMS_SEGMENT_MAP);
      kdms_set_attributes(object, KDMS_SEGMENT_MAP,
			  KDMS_DIMENSION, 5,
			  KDMS_INDEX_ORDER, order,
			  KDMS_SIZE, size,
			  KDMS_DATA_TYPE, type,
			  NULL);
      kdms_put_data(object, KDMS_SEGMENT_MAP, begin, end, image->map);

      /*
       * it is vitally important that you include this step because if the
       * data set larger than the buffer threshold, then spurous flushes
       * will ensue that should not happen...it turns out that xvimages and
       * viffs will cope with that situation (though not very efficiently),
       * but file formats that buffer the whole thing in memory won't.
       */
      kdms_set_attribute(object, KDMS_SEGMENT_MAP, KDMS_COUPLING,KUNCOUPLED);
   }

   /*
    * if the number of elements is three, or the map width is three,
    * then set the colorspace model to be RGB so that display programs
    * interpret this image properly.
    */
   if (num == 3)
   {
      if (!kdms_query_attribute_definition(KDMS_OBJECT, 
					   _INTERNAL_COLORSPACE))
	 kdms_define_attribute(KDMS_OBJECT, _INTERNAL_COLORSPACE, 1, 1, 
			       KINT, TRUE, TRUE, KCOLOR_COLORSPACE_DEFAULT);
      kdms_set_attribute(object, NULL, _INTERNAL_COLORSPACE, KRGB);
   }

   return (TRUE);
}



/*-----------------------------------------------------------
|
|  Routine Name: rast_output - Closes the rast image
|
|       Purpose: This function is used to close the rast image.  Which
|                means if the image was modified we then re-write image
|                back out.
|
|         Input: object - the rast object to be closed
|
|        Output: Returns: returns TRUE or FALSE if an error occurs
|
|    Written By: Mark Young
|          Date: Jul 13, 1992 15:50
| Modifications:
|
------------------------------------------------------------*/

/*ARGSUSED */
static int
rast_output(kobject object, int fid, int flags)
{
   ResourceStruct *resources;

   rast *image;
   int type;
   int *size;
   int order[KDMS_MAX_DIM];
   int width;
   int height;
   int depth;
   int begin[KDMS_MAX_DIM] = {0, 0, 0, 0, 0};
   int end[KDMS_MAX_DIM];
   int dim;
   int i;
   
   if (!_kdms_initialized(object))
      _init(object);

   resources = (ResourceStruct *) _kdms_glue_get_resources(object);

   if (resources->image == NULL)
   {
      resources->image = rast_create();
      if (!resources->image)
      {
         _kdms_traceback("rast_output");
         _kdms_set_error(KDMS_EFMT_FAILURE);
         return (FALSE);
      }
   }
   image = resources->image;

   /*
    * Initialize the image's data arrays according to the data found in the
    * data segment->
    * 
    * The rast structure only contains value (imagedata), maps, and location
    * data.
    */

   /*
    * check the value data
    */
   if (kdms_get_attributes(object, KDMS_SEGMENT_VALUE, 
			   KDMS_PHYSICAL_SIZE, &size,
			   KDMS_PHYSICAL_DATA_TYPE, &type,
			   KDMS_PHYSICAL_DIMENSION, &dim,
			   NULL))
   {
      /*
       * set the presentation layer to the physical layer.  Then get the
       * data. 
       */
      order[0] = KELEMENTS;
      order[1] = KWIDTH;
      order[2] = KHEIGHT;
      order[3] = KDEPTH;
      order[4] = KTIME;

      if (type != KBIT && type != KUBYTE)
	 type = KUBYTE;
      /* 
       * note that order is set *after* size...that way the old 
       * index order is honored 
       */
      kdms_set_attributes(object, KDMS_SEGMENT_VALUE, 
			  KDMS_COUPLING, KNONE,
			  KDMS_SIZE, size, 
			  KDMS_INDEX_ORDER, order,
			  KDMS_DATA_TYPE, type, 
			  KDMS_INTERPOLATE, KNONE,
			  KDMS_SCALING, KNONE,
			  NULL);
      kdms_get_attribute(object, KDMS_SEGMENT_VALUE, KDMS_SIZE, &size);
      
      end[0] = size[0] - 1;
      end[1] = size[1] - 1;
      end[2] = size[2] - 1;
      end[3] = end[4] = 0;
      
      image->data = kdms_get_data(object, KDMS_SEGMENT_VALUE, begin, end,
				  NULL);
      
      image->ras_width = (int)size[1];
      image->ras_height = (int)size[2];

      if (type == KBIT)
      {
         image->ras_type = RT_STANDARD;
         image->ras_depth = 1;
	 /* invert the values since bit 1's are black and bit 0's are white */
	 for (i = 0; i < ((image->ras_width * image->ras_height) >> 3); i++)
	    image->data[i] = ~image->data[i];
      }
      else
      {
         image->ras_type = (size[0] == 3) ? RT_FORMAT_RGB : RT_STANDARD;
         image->ras_depth = (size[0] << 3);
      }
   }
   else
      image->ras_type = RT_STANDARD;

   width = image->ras_width;
   height = image->ras_height;
   depth = image->ras_depth >> 3;

   if (image->ras_depth == 1)
   {
      image->ras_length = (width + 7) / 8;
      image->ras_length = (image->ras_length + image->ras_length % 2) * height;
   }
   else
      image->ras_length = (width * depth + (width * depth) % 2) * height;

   if (kdms_get_attributes(object, KDMS_SEGMENT_MAP, 
			   KDMS_PHYSICAL_SIZE, &size,
			   KDMS_PHYSICAL_DATA_TYPE, &type,
			   KDMS_PHYSICAL_DIMENSION, &dim,
			   NULL))
   {
      /*
       * set the presentation layer to the physical layer.  Then get the
       * data. 
       */
      order[0] = KMAP_WIDTH;
      order[1] = KMAP_HEIGHT;
      order[2] = KMAP_DEPTH;
      order[3] = KMAP_TIME;
      order[4] = KMAP_ELEMENTS;

      type = KUBYTE;
      
      /* 
       * note that order is set *after* size...that way the old 
       * index order is honored 
       */
      kdms_set_attributes(object, KDMS_SEGMENT_MAP, 
			  KDMS_COUPLING, KNONE,
			  KDMS_SIZE, size, 
			  KDMS_INDEX_ORDER, order,
			  KDMS_DATA_TYPE, type, 
			  KDMS_INTERPOLATE, KNONE,
			  KDMS_SCALING, KNONE,
			  NULL);
      kdms_get_attribute(object, KDMS_SEGMENT_MAP, KDMS_SIZE, &size);
      
      end[0] = size[0] - 1;
      end[1] = size[1] - 1;
      end[2] = end[3] = end[4] = 0;
      
      image->map = kdms_get_data(object, KDMS_SEGMENT_MAP, begin, end, NULL);
      
      image->ras_maplength = (int)size[1];

      if (size[1] == 1)
         image->ras_maptype = RMT_RAW;
      else
         image->ras_maptype = RMT_EQUAL_RGB;
   }
   else
      image->ras_maptype = RMT_NONE;

   /*
    * Write the rast structure out to the specified filename
    */
   return(rast_fdwrite(fid, image));
}

/*-----------------------------------------------------------
|
|  Routine Name: rast_destroy - Frees an rast image
|
|       Purpose: This function is used to create an rast image.  Which
|                means that we use the current dimension values in order
|                to create a proper rast.
|
|         Input: data - initial sample of data
|                num  - number of bytes in the sample
|
|        Output: Returns: returns TRUE or FALSE if an error occurs
|
|    Written By: Mark Young
|          Date: Jul 13, 1992 15:50
| Modifications:
|
------------------------------------------------------------*/

static int
rast_destroy(kobject object)
{
   ResourceStruct *resources;

   resources = (ResourceStruct *) _kdms_glue_get_resources(object);

   /*
    * free the raster structure
    */
   rast_free(resources->image);

   /*
    * free the resources
    */
   kfree(resources);

   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: rast_order
|
|       Purpose:
|
|         Input:
|
|        Output:
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Mar 12, 1993 13:12
| Modifications:
|
------------------------------------------------------------*/

static int
rast_order(char *seg, int *ord)
{
   if (kstring_to_token(seg) == kstring_to_token(KDMS_SEGMENT_MAP))
   {
      ord[0] = KMAP_HEIGHT;
      ord[1] = KMAP_WIDTH;
      ord[2] = KMAP_DEPTH;
      ord[3] = KMAP_TIME;
      ord[4] = KMAP_ELEMENTS;
   }
   return (TRUE);
}


/*-----------------------------------------------------------
|
|  Routine Name: rast_architecture
|
|       Purpose: Return the architecture that the data was
|                generated on in khoros-speak.
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Nov 08, 1993 11:35
| Modifications:
|
------------------------------------------------------------*/

static int
rast_architecture(kobject object)
{
   ResourceStruct *resources;

   /*
    * get the resource structure
    */
   resources = (ResourceStruct *) _kdms_glue_get_resources(object);

   if (resources == NULL)
      return (KMACH_LOCAL);
   else
      return (kfile_getmachtype(kfileno(resources->file)));

}

#endif /* KRAST_DEF */

/* don`t add after the endif */
