 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            X11 Bitmap Format Utilities
   >>>>
   >>>>	    The "xbm" format describes the conversion used for
   >>>>	    the format:
   >>>>
   >>>>			X11 Bitmap Format
   >>>>
   >>>>	    The format was orignially developed by the company/persons:
   >>>>
   >>>>			MIT Project Athena
   >>>>	    
   >>>>	    The following routines are used by the conversion routines
   >>>>	    to read and write the xbm files.  These routines are not a
   >>>>	    standard, just an a set of routines used by the Khoros routines
   >>>>	    to manipulate X11 Bitmap Format.
   >>>>
   >>>>
   >>>>  Private:
   >>>>   Static:
   >>>>   Public:
   >>>>             xbm_readheader()
   >>>>             xbm_read()
   >>>>             xbm_writeheader()
   >>>>             xbm_write()
   >>>>             xbm_free()
   >>>>             xbm_create()
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"
#include <kdatafmt/xbm.h>
#define  KDATAFMT "kdatafmt"


/************************************************************
*
*  Routine Name: xbm_readheader - reads a xbm header structure
*				      from the specified kfile id
*
*       Purpose: This routines reads a xbm header structure and it's
*		 data into the specified filename.  The routine
*		 uses the ktransport library which allows for
*		 remote reading of the xbm header.
*
*         Input: fid - the kfile id opened previously with kopen()
*        Output: none
*       Returns: image - explanation
*
*  Restrictions: 
*    Written By: Mark Young 
*          Date: Oct 07, 1993 13:41 
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

xbm *xbm_readheader(
   int	fid)
{
	kfile *file;
	xbm   *image;
	int   num, done = FALSE;
	char  line[KLENGTH], name[KLENGTH], *temp;


	/* 
	 * Sanity check to make sure that the image pointer is not NULL
	 */
	if ((image = (xbm *) kcalloc(1, sizeof(xbm))) == NULL)
	   return(NULL);

	/*
	 *   Initialize the width and height to be -1, so that
	 *   if the user tries to give the bitmap data before
	 *   defining the width or height we will know about it.
	 */
	image->width = image->height = -1;
	file = kfdopen(fid, "r");

	while (!done)
	{
	   if (!kfgets(line, KLENGTH, file))
	   {
	      kfree(image);
	      return(NULL);
	   }

	   if (ksscanf(line, "#define %s %d", name, &num) == 2)
	   {
	      if ((temp = kstrrchr(name,'_')) != NULL)
	      {
		 if (kstrcmp(temp+1,"width") == 0)
		    image->width = num;
		 else if (kstrcmp(temp+1,"height") == 0)
		    image->height = num;
		 else
		    continue;
 
		 *temp = '\0';
              }

	      if (image->width != -1 && image->height != -1)
		 done = TRUE;
           }
	}
	return(image);
}


/************************************************************
*
*  Routine Name: xbm_fdread - read a xbm structure from the
*				specified file descriptor
*
*       Purpose: This routines reads a xbm structure and it's
*		 data from the specified filename.  The routine
*		 uses the ktransport library which allows for
*		 remote reading of the data and it's header.
*
*         Input: fid - file descriptor in which we will be reading
*			    the xbm
*
*        Output: none
*
*       Returns: returns the newly read xbm or NULL upon failure
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Oct 07, 1993 13:41
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

xbm *xbm_fdread(int fid)
{
   xbm *image;
   kfile *file;
	
   unsigned char *data;
   int  i, size, num, done = FALSE;
   char line[KLENGTH], name[KLENGTH], *temp;

   /*
    *  Read the X11 Bitmap Format header
    */
   if ((image = xbm_readheader(fid)) == NULL)
      return(FALSE);

   /*
    *  Now read the data associated with the
    *  X11 Bitmap Format header
    */
   
   /*
    *   Initialize the width and height to be -1, so that
    *   if the user tries to give the bitmap data before
    *   defining the width or height we will know about it.
    */
   file = kfdopen(fid, "r");

   while (!done)
   {
      if (!kfgets(line, KLENGTH, file))
      {
	 kfree(image);
	 return(FALSE);
      }
      
      if (ksscanf(line, "static char %s", name) == 1)
      {
	 if ((temp = kstrrchr(name, '_')) != NULL)
	 {
	    if (kstrncmp(temp+1,"bits",4) == 0)
	       *temp = '\0';
	    else
	       continue;
	 }
	 
	 /*
	  *  Check to make sure that the width and height have been
	  *  previously defined.
	  */
	 if (image->width == -1 || image->height == -1)
	 {
	    kfree(image);
	    return(NULL);
	 }
	 
	 /*
	  *  Otherwise we need to create and read in the bitmap data.
	  */
	 size = ((image->width + 7)/8) * image->height;
	 image->data = (char *) kmalloc((unsigned) size);
	 for (i = 0, data = (unsigned char *) image->data; i < size; i++)
	 {
	    if (kfscanf(file," 0x%x%*[,};]", &num) == 1)
	    {
	       *data++ = (unsigned char) ~num;
	    }
	    else if (!kfgets(line, KLENGTH, file))
	    {
	       xbm_free(image);
	       return(NULL);
	    }
	    else
	    {
	       xbm_free(image);
	       return(NULL);
	    }
	 }
	 done = TRUE;
      }
   }

   return(image);
}

/************************************************************
*
*  Routine Name: xbm_read - read a xbm structure from the
*				specified filename
*
*       Purpose: This routines reads a xbm structure and it's
*		 data from the specified filename.  The routine
*		 uses the ktransport library which allows for
*		 remote reading of the data and it's header.
*
*         Input: filename - filename in which we will be reading
*			    the xbm
*
*        Output: none
*
*       Returns: returns the newly read xbm or NULL upon failure
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Oct 07, 1993 13:41
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/
   
xbm *xbm_read(
   char	*filename)
{
   xbm *image;
   int fid;
   
   /* 
    * Open the input file 
    */
   if ((fid = kopen(filename, O_RDONLY, 0644)) < 0)
      return(FALSE);

   /*
    *  Lock the transport file descriptor if the transport has
    *  data permanence.
    */
   if (kfile_getpermanence(filename) == TRUE)
      kflock(fid, KLOCK_SH);

   image = xbm_fdread(fid);
   
   kclose(fid);
   
   return image;
}

/************************************************************
*
*  Routine Name: xbm_write - write a xbm structure to the
*				 specified filename
*
*       Purpose: This routines writes a xbm structure and it's
*		 data into the specified filename.  The routine
*		 uses the ktransport library which allows for
*		 remote writing of the data.
*
*         Input: filename - the filename in which we will be writing
*			    the xbm image and associated data
*                image    - the xbm structure to be written
*        Output: none
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions:
*    Written By: Mark Young 
*          Date: Oct 07, 1993 13:41 
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int xbm_fdwrite(int fid, xbm *image)
{
   kfile *file;
   int   i, j;
   char  name[KLENGTH];
   unsigned char *data;
   
   file = kfdopen(fid,"w");
	
   /*
    *  Read the X11 Bitmap Format header
    */
   if (!xbm_writeheader(fid, image))
      return(FALSE);

   /*
    *  Now writer the data associated with the X11 Bitmap Format
    *  header
    */
   kbasename(kfile_filename(fid), name);
   kfprintf(file, "static char %s_bits[] = {\n", name);
   for (j = 0, data = (unsigned char *)image->data; j < image->height; j++)
   {
      kfprintf(file, "\t");
      for (i = 0; i < image->width; i++)
      {
	 kfprintf(file, "0x%1x, ", *data++);
      }
      kfprintf(file, "\n");
   }
   kfprintf(file, "};\n\n");

   return(TRUE);
}

int xbm_write(
   char	*filename,
   xbm	*image)
{
   int fid;
   int status;
   
   /* 
    * Sanity check to make sure that the image pointer is not NULL
    */
   if (image == NULL)
   {
      errno = KNULL_PARAMETER;
      return(FALSE);
   }

   /* 
    * Open the output file 
    */
   if ((fid = kopen(filename, O_WRONLY|O_TRUNC|O_CREAT, 0664)) < 0)
      return(FALSE);

   /*	
    *  Lock the transport file descriptor if the transport has
    *  data permanence.
    */
   if (kfile_getpermanence(filename) == TRUE)
      kflock(fid, KLOCK_EX);
   
   status = xbm_fdwrite(fid, image);
   
   kclose(fid);
   
   return status;
}


/************************************************************
*
*  Routine Name: xbm_writeheader - writes a xbm header structure
*				      from the specified kfile id
*
*       Purpose: This routines writes a xbm header structure and it's
*		 data into the specified filename.  The routine
*		 uses the ktransport library which allows for
*		 remote writing of the xbm header.
*
*         Input: fid - the kfile id opened previously with kopen()
*		 image - the image header to be written
*
*        Output: none
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Mark Young 
*          Date: Oct 07, 1993 13:41 
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int xbm_writeheader(
   int	fid,
   xbm	*image)
{
	kfile *file;
	char  name[KLENGTH];

	/*
	 *  Write the (xbm) header out
	 */
	file = kfdopen(fid, "w");
	kbasename(kfile_filename(fid), name);

	kfprintf(file, "#define %s_width   %d\n", name, image->width);
	kfprintf(file, "#define %s_height  %d\n", name, image->height);
	return(TRUE);
}


/************************************************************
*
*  Routine Name: xbm_free - frees a xbm structure and it's
*				associated data
*
*       Purpose: This routine frees an khoros xbm structure and
*		 it's associated data.
*
*         Input: image - a pointer to an khoros xbm structure that
*			 contains the image structure to be freed.
*
*        Output: none
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Mark Young 
*          Date: Oct 07, 1993 13:41
*      Verified:
*  Side Effects: Once this routine is called no further reference
*		 the image should be made.
* Modifications:
*
*************************************************************/

int xbm_free(
   xbm	*image)
{
	if (!image)
	   return(FALSE);

	/*
	 *  Free the data and all associated resources associated with
	 *  the image.
	 */
	kfree(image->data);

	/*
	 *  Free the xbm image structure itself, which was previously
	 */
	kfree(image);
	return(TRUE);
}


/************************************************************
*
*  Routine Name: xbm_create - creates a xbm structure and it's
*				associated data
*
*       Purpose: This routine creates an khoros xbm structure and
*		 it's associated data.
*
*         Input: image - a pointer to an khoros xbm structure that
*			 contains the image structure to be freed.
*
*        Output: none
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Mark Young 
*          Date: Oct 07, 1993 13:41
*      Verified:
*  Side Effects: 
* Modifications:
*
*************************************************************/

xbm *xbm_create(
   void)
{
	xbm *image;

	/* 
	 * Sanity check to make sure that the image pointer is not NULL
	 */
	if ((image = (xbm *) kcalloc(1, sizeof(xbm))) == NULL)
	   return(NULL);

	return(image);
}
