 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */
/* See $DATASERV/repos/copyright/Copyright for terms and conditions. */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>		  pbmplus (pnm) file utilities
   >>>>
   >>>>  Private:
   >>>>   Static:
   >>>>			read_whitespace()
   >>>>			read_pbmfile()
   >>>>			read_pgmfile()
   >>>>			read_ppmfile()
   >>>>   Public:
   >>>>			pnm_read()
   >>>>			pnm_write()
   >>>>			pnm_free()
   >>>>			pnm_create()
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"
#include <kdatafmt/pnm.h>
#define KDATAFMT "kdatafmt"

/*-----------------------------------------------------------
|
|  Routine Name: read_whitespace - eats whites space
|
|       Purpose: This routine eats white space using the ktransport
|		 library routines.
|
|         Input: file - the transport file structure
|
|        Output: none
|
|       Returns: none
|
|    Written By: Mark Young
|          Date: Sep 22, 1992 09:37
| Modifications: Converted from Khoros 1.0
|
------------------------------------------------------------*/

static void read_whitespace(
   kfile *file)
{
	int  c;
	char temp[KLENGTH];

	while ((c = kfgetc(file)) != EOF)
	{
	   if (!isspace(c) && c != '#')
	   {
	      kungetc(c, file);
	      break;
	   }
	   else if (c == '#')
	      (void) kfgets(temp, KLENGTH, file);
	}
}



/*-----------------------------------------------------------
|
|  Routine Name: read_pbmfile - reads a portable bitmap file
|
|       Purpose: This routine reads a portable bitmap file.
|
|         Input: file - the ktransport file structure in which
|			we will be reading the pbm data.
|
|        Output: pbm  - the pnm structure to filled out
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: Sep 22, 1992 10:04
| Modifications: Converted from Khoros 1.0
|
------------------------------------------------------------*/

static int read_pbmfile(
   kfile *file,
   pnm   *pbm)
{
	int    i, num_bytes, num;


	num_bytes = (pbm->width+7)/8*pbm->height;
	if ((pbm->data = (kaddr) kcalloc(1, (unsigned) num_bytes)) == NULL)
	   return(FALSE);

	read_whitespace(file);

	if (pbm->raw)
	{
	   if ((num = kfread(pbm->data, 1, num_bytes, file)) != num_bytes)
	   {
	      kfree(pbm->data);
	      return(FALSE);
	   }
	}
	else
	{
	   char *c = (char *)pbm->data;

	   for (i = 0; i < pbm->width * pbm->height; i++)
	   {
	      if (kfscanf(file," %d", &num) != 1)
	      {
		 kfree(pbm->data);
		 return(FALSE);
	      }
	      else
	      {
		 if (num == 1)
		    c[i/8] |= (1 << (i % 8));
		 else
		    c[i/8] &= ~(1 << (i % 8));
	      }
	   }
	}
	return(TRUE);
}



/*-----------------------------------------------------------
|
|  Routine Name: read_pgmfile - reads a portable greymap file
|
|       Purpose: This routine reads a portable greymap file.
|
|         Input: file - the ktransport file structure in which
|			we will be reading the pgm data.
|
|        Output: pgm  - the pnm structure to filled out
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: Sep 22, 1992 10:04
| Modifications: Converted from Khoros 1.0
|
------------------------------------------------------------*/

static int read_pgmfile(
   kfile *file,
   pnm   *pgm)
{
	int    i, num_bytes, num;
	int    *data;


	if (pgm->raw)
	   num_bytes = pgm->width*pgm->height;
	else
	   num_bytes = pgm->width*pgm->height*sizeof(int);

	if ((pgm->data = (kaddr) kcalloc(1, (unsigned) num_bytes)) == NULL)
	   return(FALSE);

        /*
         *  Get the maximum value
         */
	read_whitespace(file);

	if (kfscanf(file, " %d", &pgm->maxval) != 1)
	{
	   kfree(pgm->data);
           return(FALSE);
	}

	if (pgm->raw)
	{
	   if ((num = kfread(pgm->data, 1, num_bytes, file)) != num_bytes)
	   {
	      kfree(pgm->data);
	      return(FALSE);
	   }
	}
	else
	{
	   data = (int *) pgm->data;
	   for (i = 0; i < pgm->width * pgm->height; i++)
	   {
	      if (kfscanf(file," %d", &num) != 1)
	      {
		 kfree(pgm->data);
		 return(FALSE);
	      }
	      else
		 data[i] = num;
	   }
	}
	return(TRUE);
}



/*-----------------------------------------------------------
|
|  Routine Name: read_ppmfile - reads a portable pixmap file
|
|       Purpose: This routine reads a portable pixmap file.
|
|         Input: file - the ktransport file structure in which
|			we will be reading the ppm data.
|
|        Output: ppm  - the pnm structure to filled out
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: Sep 22, 1992 10:04
| Modifications: Converted from Khoros 1.0
|
------------------------------------------------------------*/

static int read_ppmfile(
   kfile *file,
   pnm   *ppm)
{
	int  i, num_bytes, num, *data;


	if (ppm->raw)
	   num_bytes = ppm->width*ppm->height*3;
	else
	   num_bytes = ppm->width*ppm->height*sizeof(int)*3;

	if ((ppm->data = (kaddr) kcalloc(1, (unsigned) num_bytes)) == NULL)
	   return(FALSE);

        /*
         *  Get the maximum value
         */
	read_whitespace(file);
	if (kfscanf(file, " %d", &ppm->maxval) != 1)
	{
	   errno = KINVALID_INPUT;
	   kfree(ppm->data);
           return(FALSE);
	}
	(void) kfgetc(file);	/* munch the next character of white space */

	if (ppm->raw)
	{
	   if ((num = kfread(ppm->data, 1, num_bytes, file)) != num_bytes)
	   {
	      kfree(ppm->data);
	      return(FALSE);
	   }
	}
	else
	{
	   data = (int *) ppm->data;
	   for (i = 0; i < ppm->width * ppm->height * 3; i++)
	   {
	      if (kfscanf(file," %d", &num) != 1)
	      {
		 kfree(ppm->data);
		 return(FALSE);
	      }
	      else
		 data[i] = num;
	   }
	}
	return(TRUE);
}



/************************************************************
*
*  Routine Name: pnm_fdread - reads a pnm structure from the specified
*			    file descriptor
*
*       Purpose: Load a portable anymap (bitmap,pixmap,greymap)
*		 (pnm) structure from the pnm file.
*
*         Input: fid - the file descriptor in which to read the pnm structure
*
*        Output: none
*
*       Returns: pnm structure on success, NULL otherwise
*
*  Restrictions: Restrictions on data or input as applicable
*    Written By: Mark Young
*          Date: Sep 22, 1992 14:50
*      Verified:
*  Side Effects: The structure and data should be freed using
*		 the pnm_free() routine
* Modifications: Converted from Khoros 1.0
*
*************************************************************/

pnm *pnm_fdread(int fid)
{
   pnm *image;
   char     line[KLENGTH];
   kfile *file = kfdopen(fid, "r");

   /*
    *  Malloc space for the pnm structure.
    */
   if ((image = (pnm *) kcalloc(1, sizeof(pnm))) == NULL)
      return(NULL);

   if (!kfgets(line, KLENGTH, file))
   {
      kfree(image);
      return(NULL);
   }

   if (kstrncmp(line, "P1", 2) == 0)
   {
      image->type = PBM;
      image->raw  = FALSE;
   }
   else if (kstrncmp(line, "P4", 2) == 0)
   {
      image->type = PBM;
      image->raw  = TRUE;
   }
   else if (kstrncmp(line, "P2", 2) == 0)
   {
      image->type = PGM;
      image->raw  = FALSE;
   }
   else if (kstrncmp(line, "P5", 2) == 0)
   {
      image->type = PGM;
      image->raw  = TRUE;
   }
   else if (kstrncmp(line, "P3", 2) == 0)
   {
      image->type = PPM;
      image->raw  = FALSE;
   }
   else if (kstrncmp(line, "P6", 2) == 0)
   {
      image->type = PPM;
      image->raw  = TRUE;
   }
   else
   {
      errno = KINVALID_FILE;
      kfree(image);
      return(NULL);
   }

   /*
    *  Get the image's width and height
    */
   read_whitespace(file);
   if (kfscanf(file, " %d", &image->width) != 1)
   {
      kfree(image);
      return(NULL);
   }
   
   read_whitespace(file);
   if (kfscanf(file, " %d", &image->height) != 1)
   {
      kfree(image);
      return(NULL);
   }
   
   /*
    * it is vitally important to data services that this is set
    */
   image->hdr_length = kftell(file) - 1;
   
   /*
    *  Call the appropriate routine to create the viff image
    */
   switch (image->type)
   {
      case PBM:
	 if (!read_pbmfile(file, image))
	 {
	    kfree(image);
	    return(NULL);
	 }
	 break;
	 
      case PGM:
	 if (!read_pgmfile(file, image))
	 {
	    kfree(image);
	    return(NULL);
	 }
	 break;
	 
      case PPM:
	 if (!read_ppmfile(file, image))
	 {
	    kfree(image);
	    return(NULL);
	 }
	 break;
   }
   return(image);
}

/************************************************************
*
*  Routine Name: pnm_read - reads a pnm structure from the specified
*			    filename
*
*       Purpose: Load a portable anymap (bitmap,pixmap,greymap)
*		 (pnm) structure from the pnm file.
*
*         Input: fid - the file descriptor in which to read the pnm structure
*
*        Output: none
*
*       Returns: pnm structure on success, NULL otherwise
*
*  Restrictions: 
*    Written By: Jeremy Worley
*          Date: Jan 05, 1995 13:48
*      Verified:
*  Side Effects: 
* Modifications: 
*
*************************************************************/

pnm *pnm_read(
   char *filename)
{
   int fid;
   pnm	 *image;

   if((fid = kopen(filename, KOPEN_RDONLY, 0)) < 0)
      return (NULL);
   
   image = pnm_fdread(fid);
   
   kclose(fid);
   
   return image;
}

/************************************************************
*
*  Routine Name: pnm_write - writes a pnm structure to a specified
*			     filename.
*
*       Purpose: Writes a portable anymap (bitmap,greymap,pixmap)
*		 (pnm) file from the pnm structure.
*
*         Input: filename - the file in which to write the pnm structure
*                image    - the pnm structure to be written
*
*        Output: none
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Mark Young
*          Date: Sep 22, 1992 14:41
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int pnm_fdwrite(int fid, pnm *image)
{
	kfile	 *file;
        char     line[KLENGTH];
	int	 i, j, size;


	/*
	 *  Open the output pnm file
	 */
	if ((file = kfdopen(fid, "w")) == NULL)
	   return(FALSE);

	if (image->type != PBM && image->type != PGM && image->type != PPM)
        {
	   errno = KINVALID_DATATYPE;
           return(FALSE);
        }

	/*
	 *  Write the header
	 */
	if (image->raw)
	{
	   if (image->type == PBM)
	      (void) kfprintf(file,"P4\n");
	   else if (image->type == PGM)
	      (void) kfprintf(file,"P5\n");
	   else if (image->type == PPM)
	      (void) kfprintf(file,"P6\n");
	}
	else
	{
	   if (image->type == PBM)
	      (void) kfprintf(file,"P1\n");
	   else if (image->type == PGM)
	      (void) kfprintf(file,"P2\n");
	   else if (image->type == PPM)
	      (void) kfprintf(file,"P3\n");
	}

        /*
         *  Write the image's width and height
         */
	(void) kfprintf(file,"%d %d\n", image->width, image->height);

	/*
	 *  Write the image's max value if the image is not a bitmap
	 */
	if (image->type == PGM || image->type == PPM)
	   (void) kfprintf(file,"%d\n", image->maxval);

	/*
	 * this is important to data services.
	 */
	image->hdr_length = kftell(file);

	/*
	 *  Check to see what format the image is to be written (raw or ascii)
	 */
	if (image->raw)
	{
	   int num_bytes;
	   
	   if (image->type == PBM)
	      num_bytes = ((image->width+7)/8) * image->height;
	   else if (image->type == PGM)
	      num_bytes = image->width * image->height;
	   else if (image->type == PPM)
	      num_bytes = image->width * image->height * 3;
	   else
	      return (FALSE);
	   
	   if ((kfwrite(image->data, 1, num_bytes, file)) != num_bytes)
	      return(FALSE);
	}
	else if (image->type == PBM)    /* writing ASCII PBM file */
	{
	   unsigned char *data;

	   data = (unsigned char *) image->data;
	   for (i = 0; i < image->width * image->height; i++)
	   {
	      if (data[i/8] & (unsigned char)(1 << (i % 8)))
		 (void) kfprintf(file, " 1");
	      else
		 (void) kfprintf(file, " 0");

	      if (i % 35 == 0)
		 (void) kfprintf(file,"\n");
	   }
	}
	else if (image->type == PGM)      /* writing ASCII PGM file */
	{
	   int *data;

	   data = (int *) image->data;
	   for (i = size = 0; i < image->width * image->height; i++)
	   {
	      (void) ksprintf(line," %3d", data[i]);
 
	      size += kstrlen(line);
	      if (size > 70)
	      {
		 (void) kfprintf(file,"\n");
		 size = kstrlen(line);
	      }
	      (void) kfprintf(file,"%s", line);
	   }
	   (void) kfprintf(file,"\n");
	}
	else                           /* writing ASCII PPM file */
	{
	   int *data;

	   data = (int *) image->data;
	   for (i = j = size = 0; i < image->width * image->height; i++)
	   {
	      (void) sprintf(line," %3d %3d %3d ",data[j],data[j+1],data[j+2]);
	      j += 3;

	      size += kstrlen(line);
	      if (size > 70)
	      {
		 (void) kfprintf(file,"\n");
		 size = kstrlen(line);
	      }
	      (void) kfprintf(file,"%s", line);
	   }
	   (void) kfprintf(file,"\n");
	}

	return(TRUE);
}

/************************************************************
*
*  Routine Name: pnm_write - writes a pnm structure to a specified
*			     filename.
*
*       Purpose: Writes a portable anymap (bitmap,greymap,pixmap)
*		 (pnm) file from the pnm structure.
*
*         Input: filename - the file in which to write the pnm structure
*                image    - the pnm structure to be written
*
*        Output: none
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Jeremy Worley
*          Date: Jan 05, 1995 13:48
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/
   
int pnm_write(
   char *filename,
   pnm  *image)
{
   int fid;
   int status;
   
   if ((fid = kopen(filename, KOPEN_WRONLY, 0644)) < 0)
      return FALSE;

   status = pnm_fdwrite(fid, image);
   
   kclose(fid);
   
   return status;
}

/************************************************************
*
*  Routine Name: pnm_free - frees a pnm structure and associated
*			    data.
*
*       Purpose: Frees a portable anymap (bitmap,greymap,pixmap)
*		 pnm structure and it's associated data.
*
*         Input: image   - the pnm structure to be freed
*
*        Output: none
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Mark Young
*          Date: Sep 22, 1992 14:41
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int pnm_free(
   pnm *image)
{
	/*
	 *  Sanity check to make sure that the pnm structure is not
	 *  NULL
	 */
	if (image == NULL)
	   return(FALSE);

	kfree(image->data); image->data = NULL; 
	kfree(image);
	return(TRUE);
}



/************************************************************
*
*  Routine Name: pnm_create - creates a pnm structure and allocates
*			      associated data.
*
*       Purpose: Creates a portable anymap (bitmap,greymap,pixmap)
*		 pnm structure and it's associated data.
*
*         Input: 
*
*        Output: none
*
*       Returns: the pnm structure or NULL 
*
*  Restrictions: 
*    Written By: Mark Young
*          Date: Sep 22, 1992 14:41
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

pnm *pnm_create(
   int width,
   int height,
   int type,
   int raw)
{
	pnm *image;
	int num_bytes;

	/*
	 *  Malloc space for the pnm structure.
	 */
	if ((image = (pnm *) kcalloc(1, sizeof(pnm))) == NULL)
	   return(NULL);

	image->width  = width;
	image->height = height;
	image->type   = type;
	image->raw    = raw;

	if ((num_bytes = image->width * image->height) > 0)
	{
	   if (image->raw == FALSE)
	      num_bytes *= sizeof(int);
	   if (image->type == PPM)
	      num_bytes *= 3;

	   if ((image->data = (kaddr) kcalloc(1, (unsigned) num_bytes)) == NULL)
	   {
	      kfree(image);
	      return(NULL);
	   }
	}
	return(image);
}
