 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */



/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Advanced Visualization System File Format Utilities
   >>>>
   >>>>	    The "avs" format describes the conversion used for
   >>>>	    the format:
   >>>>
   >>>>			Advanced Visualization System File Format
   >>>>
   >>>>	    The format was orignially developed by the company/persons:
   >>>>
   >>>>			AVS Inc.
   >>>>	    
   >>>>	    The following routines are used by the conversion routines
   >>>>	    to read and write the avs files.  These routines are not a
   >>>>	    standard, just an a set of routines used by the Khoros routines
   >>>>	    to manipulate Advanced Visualization System File Format.
   >>>>
   >>>>
   >>>>  Private:
   >>>>   Static:
   >>>>   Public:
   >>>>             avs_readheader()
   >>>>             avs_read()
   >>>>             avs_writeheader()
   >>>>             avs_write()
   >>>>             avs_free()
   >>>>             avs_create()
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"
#include <kdatafmt/avs.h>
#define  KDATAFMT "kdatafmt"


/************************************************************
*
*  Routine Name: avs_readheader - reads a avs header structure
*				      from the specified kfile id
*
*       Purpose: This routines reads a avs header structure and it's
*		 data into the specified filename.  The routine
*		 uses the ktransport library which allows for
*		 remote reading of the avs header.
*
*         Input: fid - the kfile id opened previously with kopen()
*        Output:
*       Returns: image - explanation
*
*  Restrictions: 
*    Written By: Mark Young 
*          Date: Nov 14, 1993  
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

avs *avs_readheader(
   int	fid)
{
	int size;
	avs *image;


	/* 
	 * Sanity check to make sure that the image pointer is not NULL
	 */
	if ((image = (avs *) kcalloc(1, sizeof(avs))) == NULL)
	   return(NULL);

	/*
	 *  Data is always stored in Sun format so we need to set the
	 *  descriptor machine type to be of type KMACH_SPARC.
	 */
	kfile_setmachtype(fid, KMACH_SPARC);
	if (kread_int(fid, &image->width, 2) != 2)
	{
	   kfree(image);
	   return(NULL);
	}

	size = klseek(fid, 0, SEEK_END);
	if ((size-8) != image->width*image->height*4)
	{
	   errno = KINVALID_FILE;
	   kfree(image);
	   return(NULL);
	}
	klseek(fid, 8, SEEK_SET);
	return(image);
}


/************************************************************
*
*  Routine Name: avs_fdread - read a avs structure from the
*				specified file descriptor
*
*       Purpose: This routines reads a avs structure and it's
*		 data from the specified file descriptor.  The routine
*		 uses the ktransport library which allows for
*		 remote reading of the data and it's header.
*
*         Input: filename - filename in which we will be reading
*			    the avs
*        Output:
*       Returns: returns the newly read avs or NULL upon failure
*
*  Restrictions:
*    Written By: Mark Young & Jeremy Worley
*          Date: Jan 05, 1995 09:38
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

avs *avs_fdread(int fid)
{
   avs *image;
   int size;

   /*
    *  Read the Advanced Visualization System File Format header
    */
   if ((image = avs_readheader(fid)) == NULL)
      return(FALSE);

   /*
    *  Now read the data associated with the
    *  Advanced Visualization System File Format header
    */
   size = image->width * image->height * 4;

   if ((image->data = kmalloc((unsigned) size)) == NULL)
      avs_free(image);
   
   kread(fid, (char *) image->data, image->width * image->height * 4);
   
   return(image);
}

/************************************************************
*
*  Routine Name: avs_read - read a avs structure from the
*				specified filename
*
*       Purpose: This routines reads a avs structure and it's
*		 data from the specified filename.  The routine
*		 uses the ktransport library which allows for
*		 remote reading of the data and it's header.
*
*         Input: filename - filename in which we will be reading
*			    the avs
*        Output:
*       Returns: returns the newly read avs or NULL upon failure
*
*  Restrictions:
*    Written By: Mark Young & Jeremy Worley
*          Date: Jan 05, 1995 09:39
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

avs *avs_read(
   char	*filename)
{
   int  fid;
   avs *image;
   
   /* 
    * Open the input file 
    */
   if ((fid = kinput(filename)) < 0)
      return(FALSE);
   
   /*
    *  Lock the transport file descriptor if the transport has
    *  data permanence.
    */
   if (kfile_getpermanence(filename) == TRUE)
      kflock(fid, KLOCK_SH);

   image = avs_fdread(fid);

   kclose(fid);

   return image;
}


/************************************************************
*
*  Routine Name: avs_fdwrite - write a avs structure to the
*				 specified file descriptor
*
*       Purpose: This routines writes a avs structure and it's
*		 data into the specified file descriptor.  The routine
*		 uses the ktransport library which allows for
*		 remote writing of the data.
*
*         Input: fid - the file descriptor in which we will be writing
*			    the avs image and associated data
*                image    - the avs structure to be written
*        Output:
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions:
*    Written By: Mark Young & Jeremy Worley
*          Date: Jan 05, 1995 09:45
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int avs_fdwrite(int fid, avs *image)
{
   int size;
   
   /*	
    *  Write the Advanced Visualization System File Format header
    */
   if (!avs_writeheader(fid, image))
   {
      kclose(fid);
      return(FALSE);
   }

   /*		
    *  Now write the data associated with the Advanced Visualization
    *  System File Format header
    */
   size = image->width*image->height*4;

   if ((kwrite(fid, (char *) image->data, size)) != size)
      return(FALSE);

   return TRUE;
}

/************************************************************
*
*  Routine Name: avs_write - write a avs structure to the
*				 specified filename
*
*       Purpose: This routines writes a avs structure and it's
*		 data into the specified filename.  The routine
*		 uses the ktransport library which allows for
*		 remote writing of the data.
*
*         Input: filename - the filename in which we will be writing
*			    the avs image and associated data
*                image    - the avs structure to be written
*        Output:
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions:
*    Written By: Mark Young & Jeremy Worley
*          Date: Jan 05, 1995 09:46
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int avs_write(
   char	*filename,
   avs	*image)
{
   int fid;
   int status;
   
   /* 
    * Sanity check to make sure that the image pointer is not NULL
    */
   if (image == NULL)
   {
      errno = KNULL_PARAMETER;
      return(FALSE);
   }
   
   /* 
    * Open the output file 
    */
   if ((fid = koutput(filename)) < 0)
      return(FALSE);

   /*
    *  Lock the transport file descriptor if the transport has
    *  data permanence.
    */
   if (kfile_getpermanence(filename) == TRUE)
      kflock(fid, KLOCK_EX);

   status = avs_fdwrite(fid, image);

   kclose(fid);

   return status;
}


/************************************************************
*
*  Routine Name: avs_writeheader - writes a avs header structure
*				      from the specified kfile id
*
*       Purpose: This routines writes a avs header structure and it's
*		 data into the specified filename.  The routine
*		 uses the ktransport library which allows for
*		 remote writing of the avs header.
*
*         Input: fid - the kfile id opened previously with kopen()
*		 image - the image header to be written
*        Output:
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Mark Young 
*          Date: Nov 14, 1993  
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int avs_writeheader(
   int		fid,
   avs	*image)
{

	/*
	 *  Write the (avs) header out
	 */

	/*
	 *  Data is always stored in Sun format so we need to set the
	 *  descriptor machine type to be of type KMACH_SPARC.
	 */
	kfile_setmachtype(fid, KMACH_SPARC);
	if (kwrite_int(fid, &image->width, 2) != 2)
	   return(FALSE);

	return(TRUE);
}


/************************************************************
*
*  Routine Name: avs_free - frees a avs structure and it's
*				associated data
*
*       Purpose: This routine frees an khoros avs structure and
*		 it's associated data.
*
*         Input: image - a pointer to an khoros avs structure that
*			 contains the image structure to be freed.
*        Output:
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Mark Young 
*          Date: Nov 14, 1993 
*      Verified:
*  Side Effects: Once this routine is called no further reference
*		 the image should be made.
* Modifications:
*
*************************************************************/

int avs_free(
   avs	*image)
{
	/*
	 *  Free the data and all associated resources associated with
	 *  the image.
	 */
	kfree(image->data);

	/*
	 *  Free the avs image structure itself, which was previously
	 */
	kfree(image);
	return(TRUE);
}


/************************************************************
*
*  Routine Name: avs_create - creates a avs structure and it's
*				associated data
*
*       Purpose: This routine creates an khoros avs structure and
*		 it's associated data.
*
*         Input: image - a pointer to an khoros avs structure that
*			 contains the image structure to be freed.
*
*        Output:
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Mark Young 
*          Date: Nov 14, 1993 
*      Verified:
*  Side Effects: 
* Modifications:
*
*************************************************************/

avs *avs_create(
   void)
{
	avs *image;

	/* 
	 * Sanity check to make sure that the image pointer is not NULL
	 */
	if ((image = (avs *) kcalloc(1, sizeof(avs))) == NULL)
	   return(NULL);

	return(image);
}
