/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>> 
   >>>> 	Library Routine for expression based generation
   >>>> 
   >>>>	Private: 
   >>>>	Static: 
   >>>>	Public: kapu_exprn_generate()
   >>>>		kapu_exprn_function()
   >>>> 
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"


/************************************************************
*
*  Routine Name: kapu_exprn_generate - expression based data generation
*
*       Purpose: This routine can be used to generate a data object based
*		 on a expression.  The expression 
*         Input: expression - the expression expressed in terms of f(w,h,d,t,e)
*		 wsize - the width of the output object being created
*		 hsize - the height of the output object being created
*		 dsize - the depth of the output object being created
*		 tsize - the size of the output object's time dimension
*		 esize - the output object's element dimension size
*		 wbeg  - the beginning width value
*		 hbeg  - the beginning height value
*		 dbeg  - the beginning depth value
*		 tbeg  - the beginning time value
*		 ebeg  - the beginning element value
*		 wend  - the ending width value
*		 hend  - the ending height value
*		 dend  - the ending depth value
*		 tend  - the ending time value
*		 eend  - the ending element value
*		 create_location - whether to create location data or not
*		 
*	 Output: out_obj - the output data object
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Jan 21, 1995
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int kapu_exprn_generate(
   char *expression,
   int  wsize,
   int  hsize,
   int  dsize,
   int  tsize,
   int  esize,
   double  wbeg,
   double  hbeg,
   double  dbeg,
   double  tbeg,
   double  ebeg,
   double  wend,
   double  hend,
   double  dend,
   double  tend,
   double  eend,
   int     create_location,
   kobject out_obj)
{
	double  *data;
	kaddr   function;
	kobject reference;
	char    temp[KLENGTH];
	register double *tmpdata;
	register double wval, hval, dval, tval, eval;
	register double wstep, hstep, dstep, tstep, estep;

	/*
	 *  Not sure why, but it's gotta be that you have to initialize
	 *  the size of the value segment..
	 */
	kpds_set_attribute(out_obj, KPDS_VALUE_SIZE,
		wsize, hsize, dsize, tsize, esize);

	/*
	 *  Create a reference so that we don't have side-effects on the
	 *  output's presentation attributes.
	 */
	reference = kpds_reference_object(out_obj);
	kpds_set_attributes(reference,
		KPDS_VALUE_DATA_TYPE,   KDOUBLE,
		KPDS_VALUE_POSITION,    0, 0, 0, 0, 0,
		KPDS_VALUE_SIZE,        wsize, hsize, dsize, tsize, esize,
		KPDS_VALUE_REGION_SIZE, wsize, hsize, dsize, tsize, esize,
		NULL);

        if ((data = kpds_get_data(reference, KPDS_VALUE_REGION, NULL)) == NULL)
        {
           (void) kpds_close_object(reference);
           kerror(NULL, "kapu_exprn_generate", "Failed to get data from \
output object.");
           return(FALSE);
        }
	tmpdata = data;
 
        /*
         *  Define the expression
         */
        ksprintf(temp, "f(w,h,d,t,e) = %s", expression);
        if (!(function = kexpr_compile_function(0, temp, NULL)))
	   return(FALSE);
 
	/*
	 *  Compute step size
	 */
	wstep = (wsize > 1) ? (wend-wbeg)/(wsize - 1) : (wend-wbeg);
	hstep = (hsize > 1) ? (hend-hbeg)/(hsize - 1) : (hend-hbeg);
	dstep = (dsize > 1) ? (dend-dbeg)/(dsize - 1) : (dend-dbeg);
	tstep = (tsize > 1) ? (tend-tbeg)/(tsize - 1) : (tend-tbeg);
	estep = (esize > 1) ? (eend-ebeg)/(esize - 1) : (eend-ebeg);
	if (wstep == 0.0) wstep = 1.0;
	if (hstep == 0.0) hstep = 1.0;
	if (dstep == 0.0) dstep = 1.0;
	if (tstep == 0.0) tstep = 1.0;
	if (estep == 0.0) estep = 1.0;

        /*
         *  Evaluate the expression
         */
        eval = ebeg;
        while (eval <= eend)
        {
          tval = tbeg;
          while (tval <= tend)
          {
             dval = dbeg;
             while (dval <= dend)
             {
                 hval = hbeg;
                 while (hval <= hend)
                 {
                     wval = wbeg;
                     while (wval <= wend)
                     {
                         kexpr_execute_function(function, KDOUBLE, tmpdata++,
                                NULL, wval, hval, dval, tval, eval);
                         wval += wstep;
                     }
                     hval += hstep;
                 }
                 dval += dstep;
             }
             tval += tstep;
          }
          eval += estep;
        }
        kpds_put_data(reference, KPDS_VALUE_REGION, data);
 
        /*
         *  Now create uniform location data
         */
	if (create_location)
	{
           kpds_set_attribute(reference, KPDS_LOCATION_GRID, KUNIFORM);
           kpds_create_location(reference);
           kpds_set_attributes(reference,
                KPDS_LOCATION_BEGIN, wbeg, hbeg, dbeg,
                KPDS_LOCATION_END,   wend, hend, dend,
                NULL);
	}
	kfree(data);
	kpds_close_object(reference);
	return(TRUE);
}

/************************************************************
*
*  Routine Name: kapu_exprn_function - object expression based generation
*
*       Purpose: This routine can be used to generate a data object based
*		 on a expression.  The expression 
*         Input: expression - the expression expressed in terms of f(i1,i2,...)
*		 in_objs  - array of input objects
*		 num_objs - the number of input objects
*	 Output: out_obj - the output data object
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Jan 21, 1995
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int kapu_exprn_function(
   char *expression,
   kobject *in_objs,
   int     num,
   kobject out_obj)
{
        kaddr   func;
        char    temp[KLENGTH];
        kobject *in, reference;
        int     i, j, w, h, d, t, e, wsize, hsize, dsize, tsize, esize;
        double  **idata, *odata, *tmpdata, wval, hval, dval, tval, eval;


	if (in_objs == NULL || num <= 0)
	{
	   return(FALSE);
	}

	in    = (kobject *) kcalloc((unsigned) num, sizeof(kobject));
	idata = (double **) kcalloc((unsigned) num, sizeof(double *));
        for (wsize = hsize = dsize = tsize = esize = i = 0; i < num; i++)
        {
           if (in_objs[i] != NULL &&
	       (in[i] = kpds_reference_object(in_objs[i])) != NULL)
           {
              kpds_get_attribute(in[i], KPDS_VALUE_SIZE, &w, &h, &d, &t, &e);
              wsize = kmax(w, wsize); hsize = kmax(h, hsize);
              dsize = kmax(d, dsize); tsize = kmax(t, tsize);
              esize = kmax(e, esize);
           }
           else
              idata[i] = NULL, in[i] = NULL;
        }
        reference = kpds_reference_object(out_obj);
        for (i = 0; i < num; i++)
        {
           if (in[i] == NULL) continue;
 
           kpds_set_attributes(in[i],
                KPDS_VALUE_DATA_TYPE, KDOUBLE,
                KPDS_VALUE_POSITION, 0, 0, 0, 0, 0,
                KPDS_VALUE_SIZE, wsize, hsize, dsize, tsize, esize,
                KPDS_VALUE_REGION_SIZE, wsize, hsize, dsize, tsize, esize,
                NULL);
           idata[i] = kpds_get_data(in[i], KPDS_VALUE_REGION, NULL);
        }
        kpds_set_attributes(reference,
                KPDS_VALUE_DATA_TYPE, KDOUBLE,
                KPDS_VALUE_POSITION, 0, 0, 0, 0, 0,
                KPDS_VALUE_SIZE, wsize, hsize, dsize, tsize, esize,
                KPDS_VALUE_REGION_SIZE, wsize, hsize, dsize, tsize, esize,
                NULL);

        if ((odata = kpds_get_data(reference, KPDS_VALUE_REGION, NULL)) == NULL)
        {
           kerror(NULL, "kgexprn", "Failed to get data from output object.");
           kexit(KEXIT_FAILURE);
        }
	tmpdata = odata;
 
        /*
         *  Define the expression
         */
        ksprintf(temp, "f(i1,i2,i3,i4,i5,i6,w,h,d,t,e) = %s", expression);
        func = kexpr_compile_function(0, temp, NULL);
 
        /*
         *  Evaluate the expression
         */
        eval = 0; j = 0;
        while (eval < esize)
        {
          tval = 0;
          while (tval < tsize)
          {
             dval = 0;
             while (dval < dsize)
             {
                 hval = 0;
                 while (hval < hsize)
                 {
                     wval = 0;
                     while (wval < wsize)
                     {
                         kexpr_execute_function(func, KDOUBLE, tmpdata++, NULL,
                                idata[0] ? idata[0][j] : 0.0,
                                idata[1] ? idata[1][j] : 0.0,
                                idata[2] ? idata[2][j] : 0.0,
                                idata[3] ? idata[3][j] : 0.0,
                                idata[4] ? idata[4][j] : 0.0,
                                idata[5] ? idata[5][j] : 0.0,
                                wval, hval, dval, tval, eval);
                         wval += 1; j++;
                     }
                     hval += 1;
                 }
                 dval += 1;
             }
             tval += 1;
          }
          eval += 1;
        }
        kpds_put_data(reference, KPDS_VALUE_REGION, odata);

        for (i = 0;  i < num; i++)
           kfree(idata[i]);

        kpds_close_object(reference);
        kfree(odata);
	return(TRUE);
}
