 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Polymophic Model Data Handlers
   >>>>
   >>>>   Static:
   >>>>  Private:
   >>>>                 kaps_offset_corners()
   >>>>                 kaps_get_primitive()
   >>>>                 kaps_put_primitive()
   >>>>                 kaps_copy_object()
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"

/*
 *    ================================================================== 
 *    Polymorphic Data Access Routines
 *    ==================================================================
 */

/*-----------------------------------------------------------
|
|  Routine Name: kaps_offset_corners
|
|       Purpose: Given a zero-based begin and end point
|		 defining the absolute size of a region,
|		 offset that region into space according
|		 to the position and offsets for a given
|		 segment.   A mask can be provided to determine
|		 which positional components are to be used.
|		 This routine uses zeros in the mask to 
|		 mark which positional components are 
|		 going to be used (this lets me use the 
|		 usage mask directly).
|
|         Input: obj     - the object to retrieve the position 
|			   and offset from
|		 segment - the segment to retrieve the position 
|			   and offset from
|		 mask    - an array indicating which positional
|		           components are to be used.  Zeros mark
|			   which positional components are to be used.
|			   If this is not present, then it defaults to use
|			   all.
|
|	         begin   - the begin point to be shifted
|	         end     - the end point to be shifted
|
|        Output: begin   - the begin point is shifted by the position
|			   and offset in place
|		 end     - the end point is shifted by the position
|			   and offset in place
|       Returns: none
|
|    Written By: Steve Kubica
|          Date: Apr 29, 1994 11:17
| Modifications:
|
------------------------------------------------------------*/
void
kaps_offset_corners(
   kobject obj,
   char   *segment,
   int    *mask,
   int    *begin,
   int    *end)
{
   int *pos = NULL;
   int *off = NULL;
   int  dim;
   int  i;
   int  total_offset;

   dim = kaps_get_segment_dimension(obj, segment);
   pos = kaps_get_segment_position(obj, segment);
   off = kaps_get_segment_offset(obj, segment);

   if (pos == NULL || off == NULL || mask == NULL)
      return;

   for (i = 0; i < dim; i++)
   {
      if (mask[i] == 0)
      {				/* zero's indicate what to use */
	 total_offset = 0;
         total_offset += (pos != NULL) ? pos[i] : 0;
	 total_offset += (off != NULL) ? off[i] : 0;

	 begin[i] += total_offset;
	 end[i]   += total_offset;
      }
   }

   return;
}

/*-----------------------------------------------------------
|
|  Routine Name: kaps_get_primitive
|
|       Purpose: This routine gets a primitive.  The autoincrementing
|		 functionality is invoked here.  This routine
|		 can handle all regularly defined primitives.  
|		 Region-based primitives have their own routine.
|
|         Input: obj  - the object to retrieve the primitive from
|		 prim - the primitive to retrieve
|		 data - pointer to allocated data with which to
|		        return the primitive, or NULL if we should
|		        perform the allocation.
|
|        Output: none
|
|       Returns: pointer to the primitive data on success,
|		 NULL otherwise
|
|    Written By: Steve Kubica
|          Date: Apr 29, 1994 11:17
| Modifications:
|
------------------------------------------------------------*/
kaddr
kaps_get_primitive(
   kobject         obj,
   kaps_primitive *prim,
   kaddr           data)
{
   static int begin[KDMS_MAX_DIM];
   static int end[KDMS_MAX_DIM];
   int *size;
   int  i;


   /* verify that the segment is present and get its size */
   if ((size = kaps_get_segment_size(obj, prim->segment)) == NULL)
      return NULL;
   
   /* figure out the begin and end points according to the primitive */
   for (i = 0; i < KDMS_MAX_DIM; i++)
   {
      begin[i] = 0;
      end[i]   = (prim->usage[i] == 1) ? (size[i] - 1) : 0;
   }

   /* increment the position for this segment */
   kaps_advance_segment_position(obj, prim->segment, begin, end, size, TRUE);

   /* move the begin and end to the appropriate position and offset */
   kaps_offset_corners(obj, prim->segment, prim->usage, begin, end);

   return (kdms_get_data(obj, prim->segment, begin, end, data));
}

/*-----------------------------------------------------------
|
|  Routine Name: kaps_put_primitive
|
|       Purpose: This routine gets a primitive.  The autoincrementing
|		 functionality is invoked here.  This routine
|		 can handle all regularly defined primitives.  
|		 Region-based primitives have their own routine.
|
|         Input: obj  - the object to put the primitive into
|		 prim - the primitive to store
|		 data - pointer to the primitive data being stored
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Steve Kubica
|          Date: Apr 29, 1994 11:17
| Modifications:
|
------------------------------------------------------------*/
int
kaps_put_primitive(
   kobject         obj,
   kaps_primitive *prim,
   kaddr           data)
{
   static int begin[KDMS_MAX_DIM];
   static int end[KDMS_MAX_DIM];
   int       *size;
   int        i;


   /* verify that the segment is present and get its size */
   if ((size = kaps_get_segment_size(obj, prim->segment)) == NULL)
      return FALSE;

   /* figure out the begin and end points according to the primitive */
   for (i = 0; i < KDMS_MAX_DIM; i++)
   {
      begin[i] = 0;
      end[i]   = (prim->usage[i] == 1) ? (size[i] - 1) : 0;
   }

   /* increment the position for this segment */
   kaps_advance_segment_position(obj, prim->segment, begin, end, size, FALSE);

   /* move the begin and end to the appropriate position and offset */
   kaps_offset_corners(obj, prim->segment, prim->usage, begin, end);

   return (kdms_put_data(obj, prim->segment, begin, end, data));
}

/*-----------------------------------------------------------
|
|  Routine Name: kaps_copy_object
|
|       Purpose: This routine will copy all of the 
|		 segments from one object to another.
|
|		 
|
|         Input: src_obj         - the object to copy from
|		 dest_obj        - the object to copy to
|		 copy_attributes - copy the segment attributes 
|		                   and object level attributes
|		 copy_data       - copy the segment data
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley and Steve Kubica
|          Date: Dec 01, 1993 09:00
| Modifications:
|
------------------------------------------------------------*/
int
kaps_copy_object(
   kobject src_obj,
   kobject dest_obj,
   int copy_attributes,
   int copy_data)
{
   char **segments = NULL;
   int i;
   int nsegments = 0;


   segments = kdms_get_segment_names(src_obj, &nsegments);

   for (i = 0; i < nsegments; i++)
   {

      /*  if the segment appears to exists in the src, then copy it */
      if (kdms_query_segment(src_obj, segments[i]))
      {
	 if (kstrcmp(segments[i], KPDS_SEGMENT_VALUE)==0)
	 {
	    if (copy_attributes)
	       if (!kpds_copy_value_attr(src_obj, dest_obj))
		  return FALSE;
	    
	    if (copy_data)
	       if (!kpds_copy_value_data(src_obj, dest_obj, TRUE))
		  return FALSE;
	 }
	 else if (kstrcmp(segments[i], KPDS_SEGMENT_MASK)==0)
	 {
	    if (copy_attributes)
	       if (!kpds_copy_mask_attr(src_obj, dest_obj))
		  return FALSE;
	    
	    if (copy_data)
	       if (!kpds_copy_mask_data(src_obj, dest_obj, TRUE))
		  return FALSE;
	 }
	 else if (kstrcmp(segments[i], KPDS_SEGMENT_MAP)==0)
	 {
	    if (!kaps_mapping_mode(src_obj))
	    {
	       if (copy_attributes)
		  if (!kpds_copy_map_attr(src_obj, dest_obj))
		     return FALSE;
	    
	       if (copy_data)
		  if (!kpds_copy_map_data(src_obj, dest_obj, TRUE))
		     return FALSE;
	    }
	 }
	 else if (kstrcmp(segments[i], KPDS_SEGMENT_LOCATION)== 0 ||
		  kstrcmp(segments[i], KPDS_SEGMENT_UNIFORM) == 0 ||
		  kstrcmp(segments[i], KPDS_SEGMENT_WIDTH)   == 0 ||
		  kstrcmp(segments[i], KPDS_SEGMENT_HEIGHT)  == 0 ||
		  kstrcmp(segments[i], KPDS_SEGMENT_DEPTH)   == 0)
	 {
	    if (copy_attributes)
	       if (!kpds_copy_location_attr(src_obj, dest_obj))
		  return FALSE;
	    
	    if (copy_data)
	       if (!kpds_copy_location_data(src_obj, dest_obj, TRUE))
		  return FALSE;
	 }
	 else if (kstrcmp(segments[i], KPDS_SEGMENT_TIME)==0)
	 {
	    if (copy_attributes)
	       if (!kpds_copy_time_attr(src_obj, dest_obj))
		  return FALSE;
	    
	    if (copy_data)
	       if (!kpds_copy_time_data(src_obj, dest_obj, TRUE))
		  return FALSE;
	 }
	 else
	 {
	    if (copy_attributes)
	       if (!kdut_copy_segment_attr(src_obj, dest_obj, segments[i]))
		  return FALSE;

	    if (copy_data)
	       if (!kdut_copy_segment_data(src_obj, dest_obj, segments[i]))
		  return FALSE;
	 }
      }
   }

   /* copy the object level attributes if desired */
   if (copy_attributes)
      if (!kdms_copy_segment_attributes(src_obj, NULL, dest_obj, NULL))
	 return FALSE;

   return TRUE;
}

/*-----------------------------------------------------------
|
|  Routine Name: kaps_copy_global_attributes
|
|       Purpose: This routine will copy all the global
|		 object-level attributes from one object 
|		 to another.
|
|         Input: src_obj         - the object to copy from
|		 dest_obj        - the object to copy to
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Steve Kubica
|          Date: Aug 19, 1994 10:26
| Modifications:
|
------------------------------------------------------------*/
int
kaps_copy_global_attributes(
   kobject src_obj,
   kobject dest_obj)
{
   if (!kdms_copy_segment_attributes(src_obj, NULL, dest_obj, NULL))
      return FALSE;

   return TRUE;
}

