 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Polymophic Model Attributes
   >>>>
   >>>>  Private:
   >>>>                 kaps_init()
   >>>>   Static:
   >>>>			_kaps_init_segment_tokens()
   >>>>             
   >>>>                 _axis_get()
   >>>>                 _axis_set()
   >>>>
   >>>>                 _position_get()
   >>>>                 _position_set()
   >>>>                 _position_match()
   >>>>                 _position_copy()
   >>>>                 _position_query()
   >>>>                 _position_print()
   >>>>
   >>>>                 _offset_get()
   >>>>                 _offset_set()
   >>>>                 _offset_match()
   >>>>                 _offset_copy()
   >>>>                 _offset_query()
   >>>>                 _offset_print()
   >>>>             
   >>>>                 _complex_convert_get()
   >>>>                 _complex_convert_set()
   >>>>                 _complex_convert_match()
   >>>>                 _complex_convert_copy()
   >>>>                 _complex_convert_query()
   >>>>                 _complex_convert_print()
   >>>>             
   >>>>                 _interpolate_get()
   >>>>                 _interpolate_set()
   >>>>                 _interpolate_match()
   >>>>                 _interpolate_copy()
   >>>>                 _interpolate_query()
   >>>>                 _interpolate_print()
   >>>>             
   >>>>                 _norm_min_get()
   >>>>                 _norm_min_set()
   >>>>                 _norm_min_match()
   >>>>                 _norm_min_copy()
   >>>>                 _norm_min_query()
   >>>>                 _norm_min_print()
   >>>>             
   >>>>                 _norm_max_get()
   >>>>                 _norm_max_set()
   >>>>                 _norm_max_match()
   >>>>                 _norm_max_copy()
   >>>>                 _norm_max_query()
   >>>>                 _norm_max_print()
   >>>>             
   >>>>                 _pad_value_get()
   >>>>                 _pad_value_set()
   >>>>                 _pad_value_match()
   >>>>                 _pad_value_copy()
   >>>>                 _pad_value_query()
   >>>>                 _pad_value_print()
   >>>>             
   >>>>                 _scale_factor_get()
   >>>>                 _scale_factor_set()
   >>>>                 _scale_factor_match()
   >>>>                 _scale_factor_copy()
   >>>>                 _scale_factor_query()
   >>>>                 _scale_factor_print()
   >>>>             
   >>>>                 _scale_offset_get()
   >>>>                 _scale_offset_set()
   >>>>                 _scale_offset_match()
   >>>>                 _scale_offset_copy()
   >>>>                 _scale_offset_query()
   >>>>                 _scale_offset_print()
   >>>>             
   >>>>                 _scaling_get()
   >>>>                 _scaling_set()
   >>>>                 _scaling_match()
   >>>>                 _scaling_copy()
   >>>>                 _scaling_query()
   >>>>                 _scaling_print()
   >>>>             
   >>>>                 _minmax_get()
   >>>>
   >>>>                 _loc_grid_get()
   >>>>                 _loc_grid_set()
   >>>>                 _loc_grid_match()
   >>>>                 _loc_grid_copy()
   >>>>                 _loc_grid_query()
   >>>>                 _loc_grid_print()
   >>>>   Public:
   >>>>             
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"

static int _polymorphic_model_initialized = FALSE;

char *_pds_seg_value    = KPDS_SEGMENT_VALUE;
char *_pds_seg_mask     = KPDS_SEGMENT_MASK;
char *_pds_seg_location = KPDS_SEGMENT_LOCATION;
char *_pds_seg_map      = KPDS_SEGMENT_MAP;
char *_pds_seg_time     = KPDS_SEGMENT_TIME;

/* -- global Token Representations of the Polymorphic Segments -- */
int value_token;
int mask_token;
int location_token;
int time_token;
int map_token;
int width_token;
int height_token;
int depth_token;

/* -- model size syncronization -- */
int sync_sizes = TRUE;

/*-----------------------------------------------------------
|
|  Routine Name: _kaps_init_segment_tokens
|
|       Purpose: This routine initializes the segment tokens
|		 representing the polymophic segments.  These
|		 tokens are used globally throughout the 
|		 application services code to 
|
|	 	 To use these, you need to extern the variables
|
|			extern int value_token;
|			extern int mask_token;
|			extern int location_token;
|			extern int time_token;
|			extern int map_token;
|			extern int width_token;
|			extern int height_token;
|			extern int depth_token;
|         Input: none
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Steve Kubica
|          Date: Jun 08, 1994 11:10
| Modifications:
|
------------------------------------------------------------*/
void
_kaps_init_segment_tokens(void)
{
   value_token    = kstring_to_token(KPDS_SEGMENT_VALUE);
   mask_token     = kstring_to_token(KPDS_SEGMENT_MASK);
   location_token = kstring_to_token(KPDS_SEGMENT_LOCATION);
   time_token     = kstring_to_token(KPDS_SEGMENT_TIME);
   map_token      = kstring_to_token(KPDS_SEGMENT_MAP);

   width_token    = kstring_to_token(KPDS_SEGMENT_WIDTH);
   height_token   = kstring_to_token(KPDS_SEGMENT_HEIGHT);
   depth_token    = kstring_to_token(KPDS_SEGMENT_DEPTH);

   return;
}

/*
 *    ================================================================== 
 *    Polymorphic Model Quasi Attribute Methods
 *    ==================================================================
 */
/*
 * Centerline's CLCC doesn't honor the ARGSUSED *standard* so this
 * pragma turns off warnings about unused arguments, which are
 * numerous in the next few routines...get the picture Centerline?!? 
 */
#ifdef __CLCC__
#pragma Warning_level (3)
#endif

/*********** ---------------------------------------------------------------
 ***********  KPDS_AXIS_ASSIGNMENT
 *********** --------------------------------------------------------------- */

/*-----------------------------------------------------------
|  Routine Name: (static) _axis_get
|       Purpose: get the axis assignment attribute
|    Written By: Steve Kubica
|          Date: Apr 28, 1994 17:39
------------------------------------------------------------*/
/* ARGSUSED */
static int
_axis_get(kobject obj, int assoc, int attr, kaddr clientData, kva_list *list)
{
   int  i;
   int *ord = NULL;
   int *tmp = NULL;


   /* 
    * the current axis assignment will be adequately described by
    * any one of the index orders of the value, mask, and location.  
    * try to get each of these in succession
    */
   if (kdms_query_segment(obj, KPDS_SEGMENT_VALUE))
   {
      if (!kdms_get_attribute(obj, KPDS_SEGMENT_VALUE, KDMS_INDEX_ORDER, &ord))
	 return FALSE;
   }
   else if (kdms_query_segment(obj, KPDS_SEGMENT_MASK))
   {
      if (!kdms_get_attribute(obj, KPDS_SEGMENT_MASK, KDMS_INDEX_ORDER, &ord))
	 return FALSE;
   }
   else if (kdms_query_segment(obj, KPDS_SEGMENT_LOCATION) && 
	    kaps_location_grid(obj) == KCURVILINEAR)
   {
      /* -- this is only good for curvilinear location -- */
      if (!kdms_get_attribute(obj, KPDS_SEGMENT_LOCATION, 
			           KDMS_INDEX_ORDER, &ord))
	 return FALSE;
   }
   else
   {
      /* errno = NOTHING TO GET - CREATE SOME VALUE OR SOMETHING! */
      return FALSE;
   }

   /* at this point, the ord array has what we want and we can return it */

   /* 
    *  the first 3 entries of _any_ order array are assignment for
    *  width, height, and, depth 
    */
   for (i = 0; i < 3; i++)	/* return the axis */
      if ((tmp = kva_arg(*list, int *)) != NULL)
	    (*tmp) = ord[i];

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _axis_set
|       Purpose: set the axis attribute
|    Written By: Steve Kubica
|          Date: Apr 28, 1994 17:39
------------------------------------------------------------*/
/* ARGSUSED */
static int
_axis_set(kobject obj, int assoc, int attr, kaddr clientData, kva_list *list)
{
   int newaxis[3];
   int nord[KDMS_MAX_DIM];
   int npos[KDMS_MAX_DIM];
   int noff[KDMS_MAX_DIM];

   int  i;
   int *ord = NULL;
   int *pos = NULL;
   int *off = NULL;


   /* -- pull the new axis assignement off the variable argument list -- */
   for (i = 0; i < 3; i++)
      newaxis[i] = kva_arg(*list, int);

   /* -- make sure that these are valid indices to be setting -- */
   if ((newaxis[0]!=KWIDTH && newaxis[0]!=KHEIGHT && newaxis[0]!=KDEPTH) ||
       (newaxis[1]!=KWIDTH && newaxis[1]!=KHEIGHT && newaxis[1]!=KDEPTH) ||
       (newaxis[2]!=KWIDTH && newaxis[2]!=KHEIGHT && newaxis[2]!=KDEPTH))
   {
      /* error = BOGUS AXIS */
      return FALSE;
   }

   /* -- make sure that there are no repeat indices -- */
   if (newaxis[0] == newaxis[1] ||
       newaxis[0] == newaxis[2] ||
       newaxis[1] == newaxis[2])
   {
      /* error = REPEAT AXIS */
      return FALSE;
   }

   /* 
    * the axis assignement will affect the value, mask and location.
    * try to go each of these segments and reassign the axis and
    * reorder their positions and offsets.  
    */

   /*
    *	Reorient the value segment
    */
   if (kdms_query_segment(obj, KPDS_SEGMENT_VALUE))
   {
      /* -- retrieve all the attributes which need adjusting -- */
      if (!kdms_get_attributes(obj, KPDS_SEGMENT_VALUE,
			       KDMS_INDEX_ORDER, &ord,
			       _INTERNAL_POSITION, &pos,
			       _INTERNAL_OFFSET, &off, 
			       NULL))
	 return FALSE;

      /* -- figure out the new index order -- */
      nord[KWID] = newaxis[KWID];
      nord[KHGT] = newaxis[KHGT];
      nord[KDEP] = newaxis[KDEP];
      nord[KTIM] = KTIME;
      nord[KELM] = KELEMENTS;

      /* -- reorder the position and offset -- */
      for (i = 0; i < KPDS_VALUE_DIM; i++)
      {
	 npos[i] = _kdms_get_order(nord[i], ord, pos, KPDS_VALUE_DIM, 0);
	 noff[i] = _kdms_get_order(nord[i], ord, off, KPDS_VALUE_DIM, 0);
      }

      /* -- assign the index order, position and offset -- */
      if (!kdms_set_attributes(obj, KPDS_SEGMENT_VALUE,
			       KDMS_INDEX_ORDER, nord,
			       _INTERNAL_POSITION, npos,
			       _INTERNAL_OFFSET, noff, 
			       NULL))
	 return FALSE;
   }

   /*
    *	Reorient the mask segment
    */
   if (kdms_query_segment(obj, KPDS_SEGMENT_MASK))
   {

      /* -- retrieve all the attributes which need adjusting -- */
      if (!kdms_get_attributes(obj, KPDS_SEGMENT_MASK,
			       KDMS_INDEX_ORDER, &ord,
			       _INTERNAL_POSITION, &pos,
			       _INTERNAL_OFFSET, &off, 
			       NULL))
	 return FALSE;

      /* -- figure out the new index order -- */
      nord[KWID] = newaxis[KWID];
      nord[KHGT] = newaxis[KHGT];
      nord[KDEP] = newaxis[KDEP];
      nord[KTIM] = KTIME;
      nord[KELM] = KELEMENTS;

      /* -- reorder the position and offset -- */
      for (i = 0; i < KPDS_VALUE_DIM; i++)
      {
	 npos[i] = _kdms_get_order(nord[i], ord, pos, KPDS_MASK_DIM, 0);
	 noff[i] = _kdms_get_order(nord[i], ord, off, KPDS_MASK_DIM, 0);
      }

      /* -- assign the index order, position and offset -- */
      if (!kdms_set_attributes(obj, KPDS_SEGMENT_MASK,
			       KDMS_INDEX_ORDER, nord,
			       _INTERNAL_POSITION, npos,
			       _INTERNAL_OFFSET, noff, 
			       NULL))
	 return FALSE;
   }

   /*
    *	Reorient the location segment
    */
   if (kdms_query_segment(obj, KPDS_SEGMENT_LOCATION))
   {
      int order[KPDS_LOCATION_DIM] = {KWIDTH, KHEIGHT, KDEPTH, KDIMENSION};
      double x, y, z;
      double nx, ny, nz;

      int grid;
   
      /* -- get the current grid type -- KRECTILINEAR or KCURVILINEAR -- */
      if (!kpds_get_attribute(obj, KPDS_LOCATION_GRID, &grid))
	 return(FALSE);

      if (grid == KUNIFORM)
      {
	 /* -- reorder the begin point -- */ 
	 if (!kpds_get_attribute(obj, KPDS_LOCATION_BEGIN, &x, &y, &z))
	    return FALSE;

	 nx = ((newaxis[KWID]==KDEPTH)  ? z :
	      ((newaxis[KWID]==KHEIGHT) ? y : x));

	 ny = ((newaxis[KHGT]==KDEPTH)  ? z :
	      ((newaxis[KHGT]==KHEIGHT) ? y : x));

	 nz = ((newaxis[KDEP]==KDEPTH)  ? z :
	      ((newaxis[KDEP]==KHEIGHT) ? y : x));

	 if (!kpds_set_attribute(obj, KPDS_LOCATION_BEGIN, nx, ny, nz))
	    return FALSE;

	 /* -- reorder the end point -- */ 
	 if (!kpds_get_attribute(obj, KPDS_LOCATION_END, &x, &y, &z))
	    return FALSE;

	 nx = ((newaxis[KWID]==KDEPTH)  ? z :
	      ((newaxis[KWID]==KHEIGHT) ? y : x));

	 ny = ((newaxis[KHGT]==KDEPTH)  ? z :
	      ((newaxis[KHGT]==KHEIGHT) ? y : x));

	 nz = ((newaxis[KDEP]==KDEPTH)  ? z :
	      ((newaxis[KDEP]==KHEIGHT) ? y : x));
	 
	 if (!kpds_set_attribute(obj, KPDS_LOCATION_END, nx, ny, nz))
	    return FALSE;
	    
	 /* -- retrieve all the attributes which need adjusting -- */
	 if (!kdms_get_attributes(obj, KPDS_SEGMENT_UNIFORM,
				  _INTERNAL_POSITION, &pos,
				  _INTERNAL_OFFSET,   &off, 
				  NULL))
	    return FALSE;

	 /* -- figure out the new index order -- */
	 nord[KWID] = newaxis[KWID];
	 nord[KHGT] = newaxis[KHGT];
	 nord[KDEP] = newaxis[KDEP];
	 nord[KDIM] = KDIMENSION;

	 /* -- reorder the position and offset -- */
	 for (i = 0; i < KPDS_LOCATION_DIM; i++)
	 {
	    npos[i] = _kdms_get_order(nord[i],order,pos,KPDS_LOCATION_DIM,0);
	    noff[i] = _kdms_get_order(nord[i],order,off,KPDS_LOCATION_DIM,0);
	 }

	 /* -- assign the index order, position and offset -- */
	 if (!kdms_set_attributes(obj, KPDS_SEGMENT_UNIFORM,
				  _INTERNAL_POSITION, npos,
				  _INTERNAL_OFFSET,   noff, 
				  NULL))
	    return FALSE;
				  
	 return TRUE;
      }
      else if (grid == KRECTILINEAR)
      {
	 char *names[3] = {"wid", "hgt", "dep"};

	 /* -- rename the width to a temporary name -- */
	 if (!kdms_rename_segment(obj, KPDS_SEGMENT_WIDTH, 
				  names[newaxis[KWID]-KWIDTH]))
	    return FALSE;

	 /* -- rename the height to a temporary name -- */
	 if (!kdms_rename_segment(obj, KPDS_SEGMENT_HEIGHT, 
				  names[newaxis[KHGT]-KWIDTH]))
	    return FALSE;

	 /* -- rename the depth to a temporary name -- */
	 if (!kdms_rename_segment(obj, KPDS_SEGMENT_DEPTH, 
				  names[newaxis[KDEP]-KWIDTH]))
	    return FALSE;

	 /* -- rename all the temporary names to their proper names -- */
	 if (!kdms_rename_segment(obj, names[KWID], KPDS_SEGMENT_WIDTH))
	    return FALSE;
	 if (!kdms_rename_segment(obj, names[KHGT], KPDS_SEGMENT_HEIGHT))
	    return FALSE;
	 if (!kdms_rename_segment(obj, names[KDEP], KPDS_SEGMENT_DEPTH))
	    return FALSE;

	 /* -- retrieve all the attributes which need adjusting -- */
	 if (!kdms_get_attributes(obj, KPDS_SEGMENT_WIDTH,
				  _INTERNAL_POSITION, &pos,
				  _INTERNAL_OFFSET,   &off, 
				  NULL))
	    return FALSE;

	 /* -- figure out the new index order -- */
	 nord[KWID] = newaxis[KWID];
	 nord[KHGT] = newaxis[KHGT];
	 nord[KDEP] = newaxis[KDEP];
	 nord[KDIM] = KDIMENSION;

	 /* -- reorder the position and offset -- */
	 for (i = 0; i < KPDS_LOCATION_DIM; i++)
	 {
	    npos[i] = _kdms_get_order(nord[i],order,pos,KPDS_LOCATION_DIM,0);
	    noff[i] = _kdms_get_order(nord[i],order,off,KPDS_LOCATION_DIM,0);
	 }

	 /* -- assign the index order, position and offset -- */
	 if (!kdms_set_attributes(obj, KPDS_SEGMENT_WIDTH,
				  _INTERNAL_POSITION, npos,
				  _INTERNAL_OFFSET,   noff, 
				  NULL))
	    return FALSE;
				  
	 return TRUE;
      }
      else if (grid == KCURVILINEAR)
      {
	 /* -- retrieve all the attributes which need adjusting -- */
	 if (!kdms_get_attributes(obj, KPDS_SEGMENT_LOCATION,
				  KDMS_INDEX_ORDER,   &ord,
				  _INTERNAL_POSITION, &pos,
				  _INTERNAL_OFFSET,   &off, 
				  NULL))
	    return FALSE;

	 /* -- figure out the new index order -- */
	 nord[KWID] = newaxis[KWID];
	 nord[KHGT] = newaxis[KHGT];
	 nord[KDEP] = newaxis[KDEP];
	 nord[KDIM] = KDIMENSION;

	 /* -- reorder the position and offset -- */
	 for (i = 0; i < KPDS_LOCATION_DIM; i++)
	 {
	    npos[i] = _kdms_get_order(nord[i],ord,pos,KPDS_LOCATION_DIM,0);
	    noff[i] = _kdms_get_order(nord[i],ord,off,KPDS_LOCATION_DIM,0);
	 }

	 /* -- assign the index order, position and offset -- */
	 if (!kdms_set_attributes(obj, KPDS_SEGMENT_LOCATION,
				  KDMS_INDEX_ORDER,   nord,
				  _INTERNAL_POSITION, npos,
				  _INTERNAL_OFFSET,   noff, 
				  NULL))
	    return FALSE;
      }
   }

   return TRUE;
}

/*********** ---------------------------------------------------------------
 ***********  KPDS_POSITION
 *********** --------------------------------------------------------------- */

/*-----------------------------------------------------------
|  Routine Name: (static) _position_get
|       Purpose: get the position attribute
|    Written By: Jeremy Worley, John Salas, and Steve Kubica
|          Date: Sep 28, 1993 10:28
------------------------------------------------------------*/
/* ARGSUSED */
static int
_position_get(kobject obj, int assoc, int attr, kaddr clientData,
	      kva_list *list)
{
   int *tmp = NULL;
   int *pos;
   int  dim;
   int  i;

   dim = kaps_get_segment_dimension(obj, (char *)clientData);
   pos = kaps_get_segment_position(obj, (char *)clientData);

   if (dim == 0 || pos == NULL)
      return FALSE;
   
   for (i = 0; i < dim; i++)
      if ((tmp = kva_arg(*list, int *)) != NULL) *tmp = pos[i];
   
   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _position_set
|       Purpose: set the position 
|    Written By: Jeremy Worley, John Salas, and Steve Kubica
|          Date: Sep 28, 1993 10:28
------------------------------------------------------------*/
/* ARGSUSED */
static int
_position_set(kobject obj, int assoc, int attr, kaddr clientData,
	      kva_list *list)
{
   int pos[KDMS_MAX_DIM] = {0, 0, 0, 0, 0};
   int dim;
   int i;

   dim = kaps_get_segment_dimension(obj, (char *)clientData);

   for (i = 0; i < dim; i++)
      pos[i] = kva_arg(*list, int);

   /* -- set the position ... -- */
   if (!kaps_set_segment_position(obj, (char *)clientData, pos))
      return FALSE;
   
   /* -- ... and reinitialize the increment state -- */
   if (!kaps_set_segment_inc_state(obj, (char *)clientData, 0, 0, 0))
      return FALSE;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _position_match
|       Purpose: match the position attribute
|    Written By: Steve Kubica & John Salas
|          Date: Dec 27, 1993
------------------------------------------------------------*/
/* ARGSUSED */
static int
_position_match(kobject obj1, kobject obj2, int assoc, int attrib,
		kaddr clientData1, kaddr clientData2)
{
   int *pos1 = NULL;
   int *pos2 = NULL;
   int  dim;
   int  i;

   /* -- verify that the segments are present -- */
   if (!kdms_query_segment(obj1, (char *)clientData1) ||
       !kdms_query_segment(obj2, (char *)clientData2))
      return FALSE;

   dim = kaps_get_segment_dimension(obj1, (char *)clientData1);

   pos1 = kaps_get_segment_position(obj1, (char *)clientData1);
   pos2 = kaps_get_segment_position(obj2, (char *)clientData2);

   if (pos1 == NULL || pos2 == NULL)
      return FALSE;
   
   for (i = 0; i < dim; i++)
      if (pos1[i] != pos2[i])
	 return FALSE;
   
   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _position_copy
|       Purpose: copy the position attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_position_copy(kobject obj1, kobject obj2, int assoc, int attrib,
	       kaddr clientData1, kaddr clientData2)
{
   int *pos = NULL;

   /* -- verify that the segments are present -- */
   if (!kdms_query_segment(obj1, (char *)clientData1) ||
       !kdms_query_segment(obj2, (char *)clientData2))
      return FALSE;

   pos = kaps_get_segment_position(obj1, (char *)clientData1);
   if (pos == NULL)
      return FALSE;
   
   return kaps_set_segment_position(obj2, (char *)clientData2, pos);
}

/*-----------------------------------------------------------
|  Routine Name: (static) _position_query
|       Purpose: query the position attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_position_query(kobject obj, int assoc, int attrib, kaddr clientData,
		int *num_args, int *arg_size, int *data_type,
		int *permanent)
{
   /* verify that the segment is present */
   if (!kdms_query_segment(obj, (char *)clientData))
      return FALSE;

   /*
    * note that the args_size and num_args are reversed on purpose.
    * The reason is that KPDS_*_POSITION is an n-tuple of arguments
    * whereas internal position is an array.
    */
   return (kdms_query_attribute(obj, (char *) clientData, _INTERNAL_POSITION,
				arg_size, num_args, data_type, permanent));
}

/*-----------------------------------------------------------
|  Routine Name: (static) _position_print
|       Purpose: print the position attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_position_print(kobject obj, int assoc, int attrib, kaddr clientData,
		kfile * outfile)
{
   int *pos = NULL;
   int  dim;
   int  i;
   

   /* -- verify that the segment is present -- */
   if (!kdms_query_segment(obj, (char *)clientData))
      return FALSE;

   dim = kaps_get_segment_dimension(obj, (char *)clientData);
   pos = kaps_get_segment_position(obj, (char *)clientData);
   if (pos == NULL)
      return FALSE;

   if (outfile)
      for (i = 0; i < dim; i++)
      {
	 kfprintf(outfile, " %d ", pos[i]);
      }

   return TRUE;
}

/*********** ---------------------------------------------------------------
 ***********  KPDS_OFFSET
 *********** --------------------------------------------------------------- */

/*-----------------------------------------------------------
|  Routine Name: (static) _offset_get
|       Purpose: get the offset attribute
|    Written By: Jeremy Worley, John Salas, and Steve Kubica
|          Date: Sep 28, 1993 10:28
------------------------------------------------------------*/
/* ARGSUSED */
static int
_offset_get(kobject obj, int assoc, int attr, kaddr clientData, kva_list *list)
{
   int *tmp = NULL;
   int *off;
   int  dim;
   int  i;

   dim = kaps_get_segment_dimension(obj, (char *)clientData);
   off = kaps_get_segment_offset(obj, (char *)clientData);

   if (dim == 0 || off == NULL)
      return FALSE;
   
   for (i = 0; i < dim; i++)
      if ((tmp = kva_arg(*list, int *)) != NULL) *tmp = off[i];
   
   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _offset_set
|       Purpose: set the offset attribute
|    Written By: Jeremy Worley, John Salas, and Steve Kubica
|          Date: Sep 28, 1993 10:28
------------------------------------------------------------*/
/* ARGSUSED */
static int
_offset_set(kobject obj, int assoc, int attr, kaddr clientData, kva_list *list)
{
   int off[KDMS_MAX_DIM] = {0, 0, 0, 0, 0};
   int dim;
   int i;

   dim = kaps_get_segment_dimension(obj, (char *)clientData);

   for (i = 0; i < dim; i++)
      off[i] = kva_arg(*list, int);

   /* -- set the offset -- */
   if (!kaps_set_segment_offset(obj, (char *)clientData, off))
      return FALSE;
   
   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _offset_match
|       Purpose: match the offset attribute
|    Written By: Steve Kubica & John Salas
|          Date: Dec 27, 1993
------------------------------------------------------------*/
/* ARGSUSED */
static int
_offset_match(kobject obj1, kobject obj2, int assoc, int attrib,
	      kaddr clientData1, kaddr clientData2)
{
   int *off1 = NULL;
   int *off2 = NULL;
   int  dim;
   int  i;


   /* -- verify that the segments are present -- */
   if (!kdms_query_segment(obj1, (char *)clientData1) ||
       !kdms_query_segment(obj2, (char *)clientData2))
      return FALSE;

   dim = kaps_get_segment_dimension(obj1, (char *)clientData1);

   off1 = kaps_get_segment_offset(obj1, (char *)clientData1);
   off2 = kaps_get_segment_offset(obj2, (char *)clientData2);

   if (off1 == NULL || off2 == NULL)
      return FALSE;
   
   for (i = 0; i < dim; i++)
      if (off1[i] != off2[i])
	 return FALSE;
   
   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _offset_copy
|       Purpose: copy the offset attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_offset_copy(kobject obj1, kobject obj2, int assoc, int attrib,
	     kaddr clientData1, kaddr clientData2)
{
   int *off = NULL;


   /* -- verify that the segments are present -- */
   if (!kdms_query_segment(obj1, (char *)clientData1) ||
       !kdms_query_segment(obj2, (char *)clientData2))
      return FALSE;

   off = kaps_get_segment_offset(obj1, (char *)clientData1);
   if (off == NULL)
      return FALSE;
   
   return(kaps_set_segment_offset(obj2, (char *)clientData2, off));
}

/*-----------------------------------------------------------
|  Routine Name: (static) _offset_query
|       Purpose: query the offset attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_offset_query(kobject obj, int assoc, int attrib, kaddr clientData,
	      int *num_args, int *arg_size, int *data_type,
	      int *permanent)
{
   /* -- verify that the segment is present -- */
   if (!kdms_query_segment(obj, (char *)clientData))
      return FALSE;

   /*
    * note that the args_size and num_args are reversed on purpose.
    * The reason is that KPDS_*_OFFSET is an n-tuple of arguments
    * whereas internal position is an array.
    */
   return (kdms_query_attribute(obj, clientData, _INTERNAL_OFFSET,
				arg_size, num_args, data_type, permanent));}

/*-----------------------------------------------------------
|  Routine Name: (static) _offset_print
|       Purpose: print the offset attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_offset_print(kobject obj, int assoc, int attrib, kaddr clientData,
	      kfile * outfile)
{
   int *off = NULL;
   int  dim;
   int  i;
   
   /* -- verify that the segment is present -- */
   if (!kdms_query_segment(obj, (char *)clientData))
      return FALSE;

   dim = kaps_get_segment_dimension(obj, (char *)clientData);
   off = kaps_get_segment_offset(obj, (char *)clientData);
   if (off == NULL)
      return FALSE;

   if (outfile)
      for (i = 0; i < dim; i++)
      {
	 kfprintf(outfile, " %d ", off[i]);
      }

   return TRUE;
}

/*********** ---------------------------------------------------------------
 ***********  KPDS_COMPLEX_CONVERT
 *********** --------------------------------------------------------------- */

/*-----------------------------------------------------------
|  Routine Name: (static) _complex_convert_get
|       Purpose: get the complex convert attribute
|    Written By: Jeremy Worley, John Salas, and Steve Kubica
|          Date: Apr 28, 1994 16:07
------------------------------------------------------------*/
/* ARGSUSED */
static int
_complex_convert_get(kobject obj, int assoc, int attr, kaddr clientData,
		     kva_list *list)
{
   int *ret_cc;
   int  cc;


   /* verify that the segment is present */
   if (!kdms_query_segment(obj, (char *)clientData))
      return FALSE;

   if (!kaps_get_attributes(obj, (char *) clientData, 
			    KDMS_COMPLEX_CONVERT, &cc, NULL))
      return FALSE;
   
   ret_cc = kva_arg(*list, int *);
   if (ret_cc != NULL) *ret_cc = cc;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _complex_convert_set
|       Purpose: set the complex convert attribute
|    Written By: Jeremy Worley, John Salas, and Steve Kubica
|          Date: Apr 28, 1994 16:07
------------------------------------------------------------*/
/* ARGSUSED */
static int
_complex_convert_set(kobject obj, int assoc, int attr, kaddr clientData,
		     kva_list *list)
{
   /* verify that the segment is present */
   if (!kdms_query_segment(obj, (char *)clientData))
      return FALSE;

   if (!kaps_set_attributes(obj, (char *) clientData,
			    KDMS_COMPLEX_CONVERT, kva_arg(*list, int), NULL))
      return FALSE;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _complex_convert_match
|       Purpose: match the complex convert attribute
|    Written By: Steve Kubica 
|          Date: Apr 28, 1994 16:06
------------------------------------------------------------*/
/* ARGSUSED */
static int
_complex_convert_match(kobject obj1, kobject obj2, int assoc, int attrib,
		       kaddr clientData1, kaddr clientData2)
{
   int cc1;
   int cc2;


   /* verify that the segments are present */
   if (!kdms_query_segment(obj1, (char *)clientData1) ||
       !kdms_query_segment(obj2, (char *)clientData2))
      return FALSE;

   if (!kaps_call_get_handler(_complex_convert_get, obj1, assoc, attrib, 
			      clientData1, &cc1))
      return FALSE;

   if (!kaps_call_get_handler(_complex_convert_get, obj2, assoc, attrib, 
			      clientData2, &cc2))
      return FALSE;

   return (cc1 == cc2);
}

/*-----------------------------------------------------------
|  Routine Name: (static) _complex_convert_copy
|       Purpose: copy the complex convert attribute
|    Written By: Steve Kubica
|          Date: Apr 28, 1994 16:01
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_complex_convert_copy(kobject obj1, kobject obj2, int assoc, int attrib,
		      kaddr clientData1, kaddr clientData2)
{
   int cc;
   

   /* verify that the segments are present */
   if (!kdms_query_segment(obj1, (char *)clientData1) ||
       !kdms_query_segment(obj2, (char *)clientData2))
      return FALSE;

   if (!kaps_call_get_handler(_complex_convert_get, obj1, assoc, attrib,
			      clientData1, &cc))
      return FALSE;

   if (!kaps_call_set_handler(_complex_convert_set, obj2, assoc, attrib,
			      clientData2, cc))
      return FALSE;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _complex_convert_query
|       Purpose: query the complex convert attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_complex_convert_query(kobject obj, int assoc, int attrib, kaddr clientData,
		       int *num_args, int *arg_size, int *data_type,
		       int *permanent)
{
   /* verify that the segment is present */
   if (!kdms_query_segment(obj, (char *)clientData))
      return FALSE;

   if (num_args)
      *num_args = 1;
   if (arg_size)
      *arg_size = 1;
   if (data_type)
      *data_type = KINT;
   if (permanent)
      *permanent = FALSE;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _complex_convert_print
|       Purpose: print the complex convert attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_complex_convert_print(kobject obj, int assoc, int attrib, kaddr clientData,
		       kfile * outfile)
{
   int cc;
   

   /* verify that the segment is present */
   if (!kdms_query_segment(obj, (char *)clientData))
      return FALSE;

   if (!kaps_call_get_handler(_complex_convert_get, obj, assoc, attrib,
			      clientData, &cc))
      return FALSE;

   if (outfile)
   {
      switch (cc)
      {
	 case KIMAGINARY:
	    kfprintf(outfile, "KIMAGINARY");
	    break;
	 case KMAGNITUDE:
	    kfprintf(outfile, "KMAGNITUDE");
	    break;
	 case KPHASE:
	    kfprintf(outfile, "KPHASE");
	    break;
	 case KREAL:
	    kfprintf(outfile, "KREAL");
	    break;
	 case KLOGMAG:
	    kfprintf(outfile, "KLOGMAG");
	    break;
	 case KLOGMAGP1:
	    kfprintf(outfile, "KLOGMAGP1");
	    break;
	 case KLOGMAGSQ:
	    kfprintf(outfile, "KLOGMAGSQ");
	    break;
	 case KLOGMAGSQP1:
	    kfprintf(outfile, "KLOGMAGSQP1");
	    break;
	 default:
	    kfprintf(outfile, "%d (invalid)", cc);
      }
   }

   return TRUE;
}

/*********** ---------------------------------------------------------------
 ***********  KPDS_INTERPOLATE
 *********** --------------------------------------------------------------- */

/*-----------------------------------------------------------
|  Routine Name: (static) _interpolate_get
|       Purpose: get the interpolate attribute
|    Written By: Jeremy Worley, John Salas, and Steve Kubica
|          Date: Sep 28, 1993 10:28
------------------------------------------------------------*/
/* ARGSUSED */
static int
_interpolate_get(kobject obj, int assoc, int attr, kaddr clientData,
		 kva_list *list)
{
   int *ret_inp;
   int  inp;


   /* verify that the segment is present */
   if (!kdms_query_segment(obj, (char *)clientData))
      return FALSE;

   if (!kaps_get_attributes(obj, (char *) clientData, 
			    KDMS_INTERPOLATE, &inp, NULL))
      return FALSE;
   
   ret_inp = kva_arg(*list, int *);
   if (ret_inp != NULL) *ret_inp = inp;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _interpolate_set
|       Purpose: set the intepolate attribute
|    Written By: Jeremy Worley, John Salas, and Steve Kubica
|          Date: Sep 28, 1993 10:28
------------------------------------------------------------*/
/* ARGSUSED */
static int
_interpolate_set(kobject obj, int assoc, int attr, kaddr clientData,
		 kva_list *list)
{
   /* verify that the segment is present */
   if (!kdms_query_segment(obj, (char *)clientData))
      return FALSE;

   if (!kaps_set_attributes(obj, (char *) clientData,
			    KDMS_INTERPOLATE, kva_arg(*list, int), NULL))
      return FALSE;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _interpolate_match
|       Purpose: match the interpolate attribute
|    Written By: Steve Kubica & John Salas
|          Date: Dec 27, 1993
------------------------------------------------------------*/
/* ARGSUSED */
static int
_interpolate_match(kobject obj1, kobject obj2, int assoc, int attrib,
		   kaddr clientData1, kaddr clientData2)
{
   int inp1;
   int inp2;


   /* verify that the segments are present */
   if (!kdms_query_segment(obj1, (char *)clientData1) ||
       !kdms_query_segment(obj2, (char *)clientData2))
      return FALSE;

   if (!kaps_call_get_handler(_interpolate_get, obj1, assoc, attrib, 
			      clientData1, &inp1))
      return FALSE;

   if (!kaps_call_get_handler(_interpolate_get, obj2, assoc, attrib, 
			      clientData2, &inp2))
      return FALSE;
   
   return (inp1 == inp2);
}

/*-----------------------------------------------------------
|  Routine Name: (static) _interpolate_copy
|       Purpose: copy the interpolate attribute
|    Written By: Steve Kubica
|          Date: Apr 28, 1994 16:02
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_interpolate_copy(kobject obj1, kobject obj2, int assoc, int attrib,
		  kaddr clientData1, kaddr clientData2)
{
   int inp;
   

   /* verify that the segments are present */
   if (!kdms_query_segment(obj1, (char *)clientData1) ||
       !kdms_query_segment(obj2, (char *)clientData2))
      return FALSE;

   if (!kaps_call_get_handler(_interpolate_get, obj1, assoc, attrib,
			      clientData1, &inp))
      return FALSE;

   if (!kaps_call_set_handler(_interpolate_set, obj2, assoc, attrib,
			      clientData2, inp))
      return FALSE;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _interpolate_query
|       Purpose: query the interpolate attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_interpolate_query(kobject obj, int assoc, int attrib, kaddr clientData,
		   int *num_args, int *arg_size, int *data_type,
		   int *permanent)
{
   /* verify that the segment is present */
   if (!kdms_query_segment(obj, (char *)clientData))
      return FALSE;

   if (num_args)
      *num_args = 1;
   if (arg_size)
      *arg_size = 1;
   if (data_type)
      *data_type = KINT;
   if (permanent)
      *permanent = FALSE;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _interpolate_print
|       Purpose: print the interpolate attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_interpolate_print(kobject obj, int assoc, int attrib, kaddr clientData,
		   kfile * outfile)
{
   int inp;


   /* verify that the segment is present */
   if (!kdms_query_segment(obj, (char *)clientData))
      return FALSE;

   if (!kaps_call_get_handler(_interpolate_get, obj, assoc, attrib,
			      clientData, &inp))
      return FALSE;

   if (outfile)
   {
      switch (inp)
      {
	 case KNONE:
	    kfprintf(outfile, "KNONE");
	    break;
	 case KPAD:
	    kfprintf(outfile, "KPAD");
	    break;
	 case KZERO_ORDER:
	    kfprintf(outfile, "KZERO_ORDER");
	    break;
	 default:
	    kfprintf(outfile, "%d (invalid)", inp);
      }
   }

   return TRUE;
}

/*********** ---------------------------------------------------------------
 ***********  KPDS_NORM_MIN
 *********** --------------------------------------------------------------- */

/*-----------------------------------------------------------
|  Routine Name: (static) _norm_min_get
|       Purpose: get the normalization minimum attribute
|    Written By: Jeremy Worley, John Salas, and Steve Kubica
|          Date: Sep 28, 1993 10:28
------------------------------------------------------------*/
/* ARGSUSED */
static int
_norm_min_get(kobject obj, int assoc, int attr, kaddr clientData,
	      kva_list *list)
{
   double *ret_min;
   double  min;


   /* verify that the segment is present */
   if (!kdms_query_segment(obj, (char *)clientData))
      return FALSE;

   if (!kaps_get_attributes(obj, (char *) clientData,
			    KDMS_NORM_MIN, &min, NULL))
      return FALSE;
   
   ret_min = kva_arg(*list, double *);
   if (ret_min != NULL) *ret_min = min;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _norm_min_set
|       Purpose: set the normalization minimum attribute
|    Written By: Jeremy Worley, John Salas, and Steve Kubica
|          Date: Sep 28, 1993 10:28
------------------------------------------------------------*/
/* ARGSUSED */
static int
_norm_min_set(kobject obj, int assoc, int attr, kaddr clientData,
	      kva_list *list)
{
   /* verify that the segment is present */
   if (!kdms_query_segment(obj, (char *)clientData))
      return FALSE;

   if (!kaps_set_attributes(obj, (char *) clientData,
			    KDMS_NORM_MIN, kva_arg(*list, double), NULL))
      return FALSE;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _norm_min_match
|       Purpose: match the normalization minimum
|    Written By: Steve Kubica & John Salas
|          Date: Dec 27, 1993
------------------------------------------------------------*/
/* ARGSUSED */
static int
_norm_min_match(kobject obj1, kobject obj2, int assoc, int attrib,
		kaddr clientData1, kaddr clientData2)
{
   double min1;
   double min2;


   /* verify that the segments are present */
   if (!kdms_query_segment(obj1, (char *)clientData1) ||
       !kdms_query_segment(obj2, (char *)clientData2))
      return FALSE;

   if (!kaps_call_get_handler(_norm_min_get, obj1, assoc, attrib, 
			      clientData1, &min1))
      return FALSE;

   if (!kaps_call_get_handler(_norm_min_get, obj2, assoc, attrib, 
			      clientData2, &min2))
      return FALSE;

   return (min1 == min2);
}

/*-----------------------------------------------------------
|  Routine Name: (static) _norm_min_copy
|       Purpose: copy the normalization minimum attribute
|    Written By: Steve Kubica
|          Date: Apr 28, 1994 17:46
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_norm_min_copy(kobject obj1, kobject obj2, int assoc, int attrib,
	       kaddr clientData1, kaddr clientData2)
{
   double min;


   /* verify that the segments are present */
   if (!kdms_query_segment(obj1, (char *)clientData1) ||
       !kdms_query_segment(obj2, (char *)clientData2))
      return FALSE;

   if (!kaps_call_get_handler(_norm_min_get, obj1, assoc, attrib,
			      clientData1, &min))
      return FALSE;

   if (!kaps_call_set_handler(_norm_min_set, obj2, assoc, attrib,
			      clientData2, min))
      return FALSE;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _norm_min_query
|       Purpose: query the normalization minimum attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_norm_min_query(kobject obj, int assoc, int attrib, kaddr clientData,
		int *num_args, int *arg_size, int *data_type,
		int *permanent)
{
   /* verify that the segment is present */
   if (!kdms_query_segment(obj, (char *)clientData))
      return FALSE;

   if (num_args)
      *num_args = 1;
   if (arg_size)
      *arg_size = 1;
   if (data_type)
      *data_type = KDOUBLE;
   if (permanent)
      *permanent = FALSE;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _norm_min_print
|       Purpose: print the normalization minimum attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_norm_min_print(kobject obj, int assoc, int attrib, kaddr clientData,
		kfile * outfile)
{
   double min;


   /* verify that the segment is present */
   if (!kdms_query_segment(obj, (char *)clientData))
      return FALSE;

   if (!kaps_call_get_handler(_norm_min_get, obj, assoc, attrib,
			      clientData, &min))
      return FALSE;

   if (outfile)
      kfprintf(outfile, "%g", min);

   return TRUE;
}

/*********** ---------------------------------------------------------------
 ***********  KPDS_NORM_MAX
 *********** --------------------------------------------------------------- */

/*-----------------------------------------------------------
|  Routine Name: (static) _norm_max_get
|       Purpose: get the normalization maximum attribute
|    Written By: Jeremy Worley, John Salas, and Steve Kubica
|          Date: Sep 28, 1993 10:28
------------------------------------------------------------*/
/* ARGSUSED */
static int
_norm_max_get(kobject obj, int assoc, int attr, kaddr clientData,
	      kva_list *list)
{
   double *ret_max;
   double  max;


   /* verify that the segment is present */
   if (!kdms_query_segment(obj, (char *)clientData))
      return FALSE;

   if (!kaps_get_attributes(obj, (char *) clientData,
			    KDMS_NORM_MAX, &max, NULL))
      return FALSE;
   
   ret_max = kva_arg(*list, double *);
   if (ret_max != NULL) *ret_max = max;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _norm_max_set
|       Purpose: set the normalization maximum attribute
|    Written By: Jeremy Worley & John Salas
|          Date: Sep 28, 1993 10:28
------------------------------------------------------------*/
/* ARGSUSED */
static int
_norm_max_set(kobject obj, int assoc, int attr, kaddr clientData,
	      kva_list *list)
{
   /* verify that the segment is present */
   if (!kdms_query_segment(obj, (char *)clientData))
      return FALSE;

   if (!kaps_set_attributes(obj, (char *) clientData,
			    KDMS_NORM_MAX, kva_arg(*list, double), NULL))
      return FALSE;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _norm_max_match
|       Purpose: match the normalization maximum attribute
|    Written By: Steve Kubica & John Salas
|          Date: Dec 27, 1993
------------------------------------------------------------*/
/* ARGSUSED */
static int
_norm_max_match(kobject obj1, kobject obj2, int assoc, int attrib,
		kaddr clientData1, kaddr clientData2)
{
   double max1;
   double max2;


   /* verify that the segments are present */
   if (!kdms_query_segment(obj1, (char *)clientData1) ||
       !kdms_query_segment(obj2, (char *)clientData2))
      return FALSE;

   if (!kaps_call_get_handler(_norm_max_get, obj1, assoc, attrib, 
			      clientData1, &max1))
      return FALSE;

   if (!kaps_call_get_handler(_norm_max_get, obj2, assoc, attrib, 
			      clientData2, &max2))
      return FALSE;

   return (max1 == max2);
}

/*-----------------------------------------------------------
|  Routine Name: (static) _norm_max_copy
|       Purpose: copy the normalization maximum attribute
|    Written By: Steve Kubica
|          Date: Apr 28, 1994 17:47
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_norm_max_copy(kobject obj1, kobject obj2, int assoc, int attrib,
	       kaddr clientData1, kaddr clientData2)
{
   double max;
   

   /* verify that the segments are present */
   if (!kdms_query_segment(obj1, (char *)clientData1) ||
       !kdms_query_segment(obj2, (char *)clientData2))
      return FALSE;

   if (!kaps_call_get_handler(_norm_max_get, obj1, assoc, attrib,
			      clientData1, &max))
      return FALSE;

   if (!kaps_call_set_handler(_norm_max_set, obj2, assoc, attrib,
			      clientData2, max))
      return FALSE;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _norm_max_query
|       Purpose: query the normalization maximum attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_norm_max_query(kobject obj, int assoc, int attrib, kaddr clientData,
		int *num_args, int *arg_size, int *data_type,
		int *permanent)
{
   /* verify that the segment is present */
   if (!kdms_query_segment(obj, (char *)clientData))
      return FALSE;

   if (num_args)
      *num_args = 1;
   if (arg_size)
      *arg_size = 1;
   if (data_type)
      *data_type = KDOUBLE;
   if (permanent)
      *permanent = FALSE;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _norm_max_print
|       Purpose: print the normalization maximum attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_norm_max_print(kobject obj, int assoc, int attrib, kaddr clientData,
		kfile * outfile)
{
   double max;


   /* verify that the segment is present */
   if (!kdms_query_segment(obj, (char *)clientData))
      return FALSE;

   if (!kaps_call_get_handler(_norm_max_get, obj, assoc, attrib,
			      clientData, &max))
      return FALSE;

   if (outfile)
      kfprintf(outfile, "%g", max);

   return TRUE;
}

/*********** ---------------------------------------------------------------
 ***********  KPDS_PAD_VALUE
 *********** --------------------------------------------------------------- */

/*-----------------------------------------------------------
|  Routine Name: (static) _pad_value_get
|       Purpose: get the pad value
|    Written By: Jeremy Worley & John Salas
|          Date: Sep 28, 1993 10:28
------------------------------------------------------------*/
/* ARGSUSED */
static int
_pad_value_get(kobject obj, int assoc, int attr, kaddr clientData,
	       kva_list *list)
{
   double *ret_pd_real;   
   double *ret_pd_imag;
   double  pd_real;
   double  pd_imag;


   /* verify that the segment is present */
   if (!kdms_query_segment(obj, (char *)clientData))
      return FALSE;

   if (!kaps_get_attributes(obj, (char *) clientData,
			    KDMS_PAD_VALUE, &pd_real, &pd_imag, NULL))
      return FALSE;

   /* if were dealing with the mask segment then return it as an int */
   if (kstrcmp((char *)clientData, KPDS_SEGMENT_MASK) == 0)
   {
      double *ret_pd;
      
      ret_pd = kva_arg(*list, double *); 
      if (ret_pd != NULL) *ret_pd = (double) pd_real;
   }
   else
   {
      ret_pd_real = kva_arg(*list, double *);
      ret_pd_imag = kva_arg(*list, double *);

      if (ret_pd_real != NULL) *ret_pd_real = pd_real;
      if (ret_pd_imag != NULL) *ret_pd_imag = pd_imag;
   }
   
   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _pad_value_set
|       Purpose: set the pad value
|    Written By: Jeremy Worley & John Salas
|          Date: Sep 28, 1993 10:28
------------------------------------------------------------*/
/* ARGSUSED */
static int
_pad_value_set(kobject obj, int assoc, int attr, kaddr clientData,
	       kva_list *list)
{
   /* verify that the segment is present */
   if (!kdms_query_segment(obj, (char *)clientData))
      return FALSE;

   if (kstrcmp((char *)clientData, KPDS_SEGMENT_MASK) == 0)
   {
      if (!kaps_set_attributes(obj, (char *) clientData, KDMS_PAD_VALUE, 
			       (double) kva_arg(*list, double), 0.0, 
			       NULL))
	 return FALSE;
   }
   else
   {
      double a = kva_arg(*list, double);
      double b = kva_arg(*list, double);
      
      
      if (!kaps_set_attributes(obj, (char *) clientData,
			       KDMS_PAD_VALUE, a, b, NULL))
	 return FALSE;
   }

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _pad_value_match
|       Purpose: match the pad values
|    Written By: Steve Kubica
|          Date: Dec 27, 1993
------------------------------------------------------------*/
/* ARGSUSED */
static int
_pad_value_match(kobject obj1, kobject obj2, int assoc, int attrib,
		 kaddr clientData1, kaddr clientData2)
{
   double pd_real1;
   double pd_imag1;
   double pd_real2;
   double pd_imag2;


   /* verify that the segments are present */
   if (!kdms_query_segment(obj1, (char *)clientData1) ||
       !kdms_query_segment(obj2, (char *)clientData2))
      return FALSE;

   if (!kaps_call_get_handler(_pad_value_get, obj1, assoc, attrib, 
			      clientData1, &pd_real1, &pd_imag1))
      return FALSE;

   if (!kaps_call_get_handler(_pad_value_get, obj2, assoc, attrib, 
			      clientData2, &pd_real2, &pd_imag2))
      return FALSE;

   return ((pd_real1 == pd_real2) && (pd_imag1 == pd_imag2));
}

/*-----------------------------------------------------------
|  Routine Name: (static) _pad_value_copy
|       Purpose: copy the pad value attribute
|    Written By: Steve Kubica
|          Date: Apr 28, 1994 17:47
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_pad_value_copy(kobject obj1, kobject obj2, int assoc, int attrib,
		kaddr clientData1, kaddr clientData2)
{
   double pd_real;
   double pd_imag;


   /* verify that the segments are present */
   if (!kdms_query_segment(obj1, (char *)clientData1) ||
       !kdms_query_segment(obj2, (char *)clientData2))
      return FALSE;

   if (!kaps_call_get_handler(_pad_value_get, obj1, assoc, attrib,
			      clientData1, &pd_real, &pd_imag))
      return FALSE;

   if (!kaps_call_set_handler(_pad_value_set, obj2, assoc, attrib,
			      clientData2, pd_real, pd_imag))
      return FALSE;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _pad_value_query
|       Purpose: query the pad value attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_pad_value_query(kobject obj, int assoc, int attrib, kaddr clientData,
		 int *num_args, int *arg_size, int *data_type,
		 int *permanent)
{
   /* verify that the segment is present */
   if (!kdms_query_segment(obj, (char *)clientData))
      return FALSE;

   if (kstrcmp((char *)clientData, KPDS_SEGMENT_MASK) == 0)
   {
      if (num_args)
	 *num_args = 1;
      if (data_type)
	 *data_type = KDOUBLE;
   }
   else
   {
      if (num_args)
	 *num_args = 2;
      if (data_type)
	 *data_type = KDOUBLE;
   }

   if (arg_size)
      *arg_size = 1;
   if (permanent)
      *permanent = FALSE;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _pad_value_print
|       Purpose: print the pad value attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_pad_value_print(kobject obj, int assoc, int attrib, kaddr clientData,
		 kfile * outfile)
{
   double pd_real;
   double pd_imag;


   /* verify that the segment is present */
   if (!kdms_query_segment(obj, (char *)clientData))
      return FALSE;

   if (!kaps_call_get_handler(_pad_value_get, obj, assoc, attrib,
			      clientData, &pd_real, &pd_imag))
      return FALSE;

   if (kstrcmp((char *)clientData, KPDS_SEGMENT_MASK) == 0)
   {
      if (outfile)
	 kfprintf(outfile, "%g", pd_real);
   }
   else
   {
      if (outfile)
	 kfprintf(outfile, "(%g, %g)", pd_real, pd_imag);
   }

   return TRUE;
}

/*********** ---------------------------------------------------------------
 ***********  KPDS_SCALE_FACTOR
 *********** --------------------------------------------------------------- */

/*-----------------------------------------------------------
|  Routine Name: (static) _scale_factor_get
|       Purpose: get the scale factor attribute
|    Written By: Jeremy Worley & John Salas
|          Date: Sep 28, 1993 10:28
------------------------------------------------------------*/
/* ARGSUSED */
static int
_scale_factor_get(kobject obj, int assoc, int attr, kaddr clientData,
		  kva_list *list)
{
   double *ret_sc;
   double  sc;


   /* verify that the segment is present */
   if (!kdms_query_segment(obj, (char *)clientData))
      return FALSE;

   if (!kaps_get_attributes(obj, (char *) clientData,
			    KDMS_SCALE_FACTOR, &sc, NULL))
      return FALSE;
   
   ret_sc = kva_arg(*list, double *);
   if (ret_sc != NULL) *ret_sc = sc;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _scale_factor_set
|       Purpose: set the scale factor attribute
|    Written By: Jeremy Worley & John Salas
|          Date: Sep 28, 1993 10:28
------------------------------------------------------------*/
/* ARGSUSED */
static int
_scale_factor_set(kobject obj, int assoc, int attr, kaddr clientData,
		  kva_list *list)
{
   /* verify that the segment is present */
   if (!kdms_query_segment(obj, (char *)clientData))
      return FALSE;

   if (!kaps_set_attributes(obj, (char *) clientData,
			    KDMS_SCALE_FACTOR, kva_arg(*list, double), NULL))
      return FALSE;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _scale_factor_match
|       Purpose: match the scale factor attributes
|    Written By: Steve Kubica & John Salas
|          Date: Dec 27, 1993
------------------------------------------------------------*/
/* ARGSUSED */
static int
_scale_factor_match(kobject obj1, kobject obj2, int assoc, int attrib,
		    kaddr clientData1, kaddr clientData2)
{
   double sc1;
   double sc2;


   /* verify that the segments are present */
   if (!kdms_query_segment(obj1, (char *)clientData1) ||
       !kdms_query_segment(obj2, (char *)clientData2))
      return FALSE;

   if (!kaps_call_get_handler(_scale_factor_get, obj1, assoc, attrib, 
			      clientData1, &sc1))
      return FALSE;

   if (!kaps_call_get_handler(_scale_factor_get, obj2, assoc, attrib, 
			      clientData2, &sc2))
      return FALSE;

   return (sc1 == sc2);
}

/*-----------------------------------------------------------
|  Routine Name: (static) _scale_factor_copy
|       Purpose: copy the scale factor attribute
|    Written By: Steve Kubica
|          Date: Apr 28, 1994 17:47
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_scale_factor_copy(kobject obj1, kobject obj2, int assoc, int attrib,
		   kaddr clientData1, kaddr clientData2)
{
   double sc;
   

   /* verify that the segments are present */
   if (!kdms_query_segment(obj1, (char *)clientData1) ||
       !kdms_query_segment(obj2, (char *)clientData2))
      return FALSE;

   if (!kaps_call_get_handler(_scale_factor_get, obj1, assoc, attrib,
			      clientData1, &sc))
      return FALSE;

   if (!kaps_call_set_handler(_scale_factor_set, obj2, assoc, attrib,
			      clientData2, sc))
      return FALSE;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _scale_factor_query
|       Purpose: query the scale factor attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_scale_factor_query(kobject obj, int assoc, int attrib, kaddr clientData,
		    int *num_args, int *arg_size, int *data_type,
		    int *permanent)
{
   /* verify that the segment is present */
   if (!kdms_query_segment(obj, (char *)clientData))
      return FALSE;

   if (num_args)
      *num_args = 1;
   if (arg_size)
      *arg_size = 1;
   if (data_type)
      *data_type = KDOUBLE;
   if (permanent)
      *permanent = FALSE;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _scale_factor_print
|       Purpose: print the scale factor attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_scale_factor_print(kobject obj, int assoc, int attrib, kaddr clientData,
		    kfile * outfile)
{
   double sc;


   /* verify that the segment is present */
   if (!kdms_query_segment(obj, (char *)clientData))
      return FALSE;

   if (!kaps_call_get_handler(_scale_factor_get, obj, assoc, attrib,
			      clientData, &sc))
      return FALSE;

   if (outfile)
      kfprintf(outfile, "%g", sc);

   return TRUE;
}

/*********** ---------------------------------------------------------------
 ***********  KPDS_SCALE_OFFSET
 *********** --------------------------------------------------------------- */

/*-----------------------------------------------------------
|  Routine Name: (static) _scale_offset_get
|       Purpose: get the scale offset
|    Written By: Jeremy Worley & John Salas
|          Date: Sep 28, 1993 10:28
------------------------------------------------------------*/
/* ARGSUSED */
static int
_scale_offset_get(kobject obj, int assoc, int attr, kaddr clientData,
		  kva_list *list)
{
   double *ret_sc_real;   
   double *ret_sc_imag;
   double  sc_real;
   double  sc_imag;


   /* verify that the segment is present */
   if (!kdms_query_segment(obj, (char *)clientData))
      return FALSE;

   if (!kaps_get_attributes(obj, (char *) clientData,
			    KDMS_SCALE_OFFSET, &sc_real, &sc_imag, NULL))
      return FALSE;

   ret_sc_real = kva_arg(*list, double *);
   ret_sc_imag = kva_arg(*list, double *);

   if (ret_sc_real != NULL) *ret_sc_real = sc_real;
   if (ret_sc_imag != NULL) *ret_sc_imag = sc_imag;
   
   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _scale_offset_set
|       Purpose: set the scale offset attribute
|    Written By: Jeremy Worley & John Salas
|          Date: Sep 28, 1993 10:28
------------------------------------------------------------*/
/* ARGSUSED */
static int
_scale_offset_set(kobject obj, int assoc, int attr, kaddr clientData,
		  kva_list *list)
{
   double a = kva_arg(*list, double);
   double b = kva_arg(*list, double);
   

   /* verify that the segment is present */
   if (!kdms_query_segment(obj, (char *)clientData))
      return FALSE;

   if (!kaps_set_attributes(obj, (char *)clientData,
			    KDMS_SCALE_OFFSET, a, b, NULL))
	 return FALSE;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _scale_offset_match
|       Purpose: match the scale offset attribute
|    Written By: Steve Kubica & John Salas
|          Date: Dec 27, 1993
------------------------------------------------------------*/
/* ARGSUSED */
static int
_scale_offset_match(kobject obj1, kobject obj2, int assoc, int attrib,
		    kaddr clientData1, kaddr clientData2)
{
   double sc_real1;
   double sc_imag1;
   double sc_real2;
   double sc_imag2;


   /* verify that the segments are present */
   if (!kdms_query_segment(obj1, (char *)clientData1) ||
       !kdms_query_segment(obj2, (char *)clientData2))
      return FALSE;

   if (!kaps_call_get_handler(_scale_offset_get, obj1, assoc, attrib, 
			      clientData1, &sc_real1, &sc_imag1))
      return FALSE;

   if (!kaps_call_get_handler(_scale_offset_get, obj2, assoc, attrib, 
			      clientData2, &sc_real2, &sc_imag2))
      return FALSE;

   return ((sc_real1 == sc_real2) && (sc_imag1 == sc_imag2));
}

/*-----------------------------------------------------------
|  Routine Name: (static) _scale_offset_copy
|       Purpose: copy the scale offset attribute
|    Written By: Steve Kubica
|          Date: Apr 28, 1994 17:47
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_scale_offset_copy(kobject obj1, kobject obj2, int assoc, int attrib,
		   kaddr clientData1, kaddr clientData2)
{
   double sc_real;
   double sc_imag;


   /* verify that the segments are present */
   if (!kdms_query_segment(obj1, (char *)clientData1) ||
       !kdms_query_segment(obj2, (char *)clientData2))
      return FALSE;

   if (!kaps_call_get_handler(_scale_offset_get, obj1, assoc, attrib,
			      clientData1, &sc_real, &sc_imag))
      return FALSE;

   if (!kaps_call_set_handler(_scale_offset_set, obj2, assoc, attrib,
			      clientData2, sc_real, sc_imag))
      return FALSE;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _scale_offset_query
|       Purpose: query the scale offset attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_scale_offset_query(kobject obj, int assoc, int attrib, kaddr clientData,
		    int *num_args, int *arg_size, int *data_type,
		    int *permanent)
{
   /* verify that the segment is present */
   if (!kdms_query_segment(obj, (char *)clientData))
      return FALSE;

   if (num_args)
      *num_args = 2;
   if (arg_size)
      *arg_size = 1;
   if (data_type)
      *data_type = KDOUBLE;
   if (permanent)
      *permanent = FALSE;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _scale_offset_print
|       Purpose: print the scale offset attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_scale_offset_print(kobject obj, int assoc, int attrib, kaddr clientData,
		    kfile * outfile)
{
   double sc_real;
   double sc_imag;


   /* verify that the segment is present */
   if (!kdms_query_segment(obj, (char *)clientData))
      return FALSE;

   if (!kaps_call_get_handler(_scale_offset_get, obj, assoc, attrib,
			      clientData, &sc_real, &sc_imag))
      return FALSE;

   if (outfile)
      kfprintf(outfile, "(%g, &g)", sc_real, sc_imag);

   return TRUE;
}

/*********** ---------------------------------------------------------------
 ***********  KPDS_SCALING
 *********** --------------------------------------------------------------- */

/*-----------------------------------------------------------
|  Routine Name: (static) _scaling_get
|       Purpose: get the scaling attribute
|    Written By: Jeremy Worley & John Salas
|          Date: Sep 28, 1993 10:28
------------------------------------------------------------*/
/* ARGSUSED */
static int
_scaling_get(kobject obj, int assoc, int attr, kaddr clientData,
	     kva_list *list)
{
   int *ret_sc;
   int  sc;


   /* verify that the segment is present */
   if (!kdms_query_segment(obj, (char *)clientData))
      return FALSE;

   if (!kaps_get_attributes(obj, (char *) clientData,
			    KDMS_SCALING, &sc, NULL))
      return FALSE;
   
   ret_sc = kva_arg(*list, int *);
   if (ret_sc != NULL) *ret_sc = sc;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _scaling_set
|       Purpose: set the scaling attribute
|    Written By: Jeremy Worley & John Salas
|          Date: Sep 28, 1993 10:28
------------------------------------------------------------*/
/* ARGSUSED */
static int
_scaling_set(kobject obj, int assoc, int attr, kaddr clientData,
	     kva_list *list)
{
   /* verify that the segment is present */
   if (!kdms_query_segment(obj, (char *)clientData))
      return FALSE;

   if (!kaps_set_attributes(obj, (char *) clientData,
			    KDMS_SCALING, kva_arg(*list, int), NULL))
      return FALSE;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _scaling_match
|       Purpose: match the scaling attribute
|    Written By: Steve Kubica & John Salas
|          Date: Dec 27, 1993
------------------------------------------------------------*/
/* ARGSUSED */
static int
_scaling_match(kobject obj1, kobject obj2, int assoc, int attrib,
	       kaddr clientData1, kaddr clientData2)
{
   int sc1;
   int sc2;


   /* verify that the segments are present */
   if (!kdms_query_segment(obj1, (char *)clientData1) ||
       !kdms_query_segment(obj2, (char *)clientData2))
      return FALSE;

   if (!kaps_call_get_handler(_scaling_get, obj1, assoc, attrib, 
			      clientData1, &sc1))
      return FALSE;

   if (!kaps_call_get_handler(_scaling_get, obj2, assoc, attrib, 
			      clientData2, &sc2))
      return FALSE;

   return (sc1 == sc2);
}

/*-----------------------------------------------------------
|  Routine Name: (static) _scaling_copy
|       Purpose: copy the scaling attribute
|    Written By: Steve Kubica
|          Date: Apr 28, 1994 17:48
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_scaling_copy(kobject obj1, kobject obj2, int assoc, int attrib,
	      kaddr clientData1, kaddr clientData2)
{
   int sc;
   

   /* verify that the segments are present */
   if (!kdms_query_segment(obj1, (char *)clientData1) ||
       !kdms_query_segment(obj2, (char *)clientData2))
      return FALSE;

   if (!kaps_call_get_handler(_scaling_get, obj1, assoc, attrib,
			      clientData1, &sc))
      return FALSE;

   if (!kaps_call_set_handler(_scaling_set, obj2, assoc, attrib,
			      clientData2, sc))
      return FALSE;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _scaling_query
|       Purpose: query the scaling attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_scaling_query(kobject obj, int assoc, int attrib, kaddr clientData,
	       int *num_args, int *arg_size, int *data_type,
	       int *permanent)
{
   /* verify that the segment is present */
   if (!kdms_query_segment(obj, (char *)clientData))
      return FALSE;

   if (num_args)
      *num_args = 1;
   if (arg_size)
      *arg_size = 1;
   if (data_type)
      *data_type = KINT;
   if (permanent)
      *permanent = FALSE;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _scaling_print
|       Purpose: print the scaling attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_scaling_print(kobject obj, int assoc, int attrib, kaddr clientData,
	       kfile * outfile)
{
   int sc;


   /* verify that the segment is present */
   if (!kdms_query_segment(obj, (char *)clientData))
      return FALSE;

   if (!kaps_call_get_handler(_scaling_get, obj, assoc, attrib,
			      clientData, &sc))
      return FALSE;

   if (outfile)
   {
      switch (sc)
      {
	 case KNONE:
	    kfprintf(outfile, "KNONE");
	    break;
	 case KNORMALIZE:
	    kfprintf(outfile, "KNORMALIZE");
	    break;
	 case KSCALE:
	    kfprintf(outfile, "KPHASE");
	    break;
	 default:
	    kfprintf(outfile, "%d (invalid)", sc);
      }
   }

   return TRUE;
}

/*********** ---------------------------------------------------------------
 ***********  KPDS_HISTORY
 *********** --------------------------------------------------------------- */

/*-----------------------------------------------------------
|  Routine Name: (static) _history_get
|       Purpose: get the history attribute
|    Written By: Steve Kubica
|          Date: Sep 28, 1993 10:28
------------------------------------------------------------*/
/* ARGSUSED */
static int
_history_get(kobject obj, int assoc, int attr, kaddr clientData,
	     kva_list *list)
{
   char **ret_hist;
   char  *hist;


   if (!kdms_get_attribute(obj, NULL, _INTERNAL_HISTORY, &hist))
      return FALSE;
   
   ret_hist = kva_arg(*list, char **);
   if (ret_hist != NULL) *ret_hist = hist;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _history_set
|       Purpose: set the history attribute
|    Written By: Steve Kubica
|          Date: Sep 28, 1993 10:28
------------------------------------------------------------*/
/* ARGSUSED */
static int
_history_set(kobject obj, int assoc, int attr, kaddr clientData,
	     kva_list *list)
{
   char *hist;
   char *new_hist;
   int   mode;
   

   hist = kva_arg(*list, char *);

   /* if the history environment variable is set, then use history */
   if (kgetenv("KHOROS_HISTORY"))
   {
      /* get the history mode */
      if (!kdms_get_attribute(obj, NULL, KPDS_HISTORY_MODE, &mode))
	  return FALSE;
   
      /* if the history is NULL, then assume we are zeroing it out */
      if (hist != NULL)
      {
	 /* either append the new history or replace it */
	 if (mode == KAPPEND_HISTORY)
	 {
	    char *old_hist;
	    
	    /* 	get the old history */
	    if (!kdms_get_attribute(obj, NULL, _INTERNAL_HISTORY, &old_hist))
	       return FALSE;
	    
	    /* append the new history to the old -- separated by a '\n' */
	    new_hist = kstring_3cat(old_hist, "\n", hist, NULL);
	 }
	 else
	 {
	    /* simply make our own copy of the history */
	    new_hist = kstring_copy(hist, NULL);
	 }
      }
      else
	 new_hist = NULL;

      /* set the new history */
      if (!kdms_set_attribute(obj, NULL, _INTERNAL_HISTORY, new_hist))
	 return FALSE;

      /* and tidy up the string */
      kfree (new_hist);
   }

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _history_match
|       Purpose: match the history attribute
|    Written By: Steve Kubica
|          Date: Dec 27, 1993
------------------------------------------------------------*/
/* ARGSUSED */
static int
_history_match(kobject obj1, kobject obj2, int assoc, int attrib,
	       kaddr clientData1, kaddr clientData2)
{
   char *hist1;
   char *hist2;


   if (!kdms_get_attribute(obj1, NULL, _INTERNAL_HISTORY, &hist1))
      return FALSE;

   if (!kdms_get_attribute(obj2, NULL, _INTERNAL_HISTORY, &hist2))
      return FALSE;

   if (kstrcmp(hist1, hist2) == 0)
      return TRUE;
   else
      return FALSE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _history_copy
|       Purpose: copy the history attribute
|    Written By: Steve Kubica
|          Date: Apr 28, 1994 17:48
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_history_copy(kobject obj1, kobject obj2, int assoc, int attrib,
	      kaddr clientData1, kaddr clientData2)
{
   char *hist;
   

   if (!kdms_get_attribute(obj1, NULL, _INTERNAL_HISTORY, &hist))
      return FALSE;

   if (!kdms_set_attribute(obj2, NULL, _INTERNAL_HISTORY, hist))
      return FALSE;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _history_query
|       Purpose: query the history attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_history_query(kobject obj, int assoc, int attrib, kaddr clientData,
	       int *num_args, int *arg_size, int *data_type,
	       int *permanent)
{
   if (num_args)
      *num_args = 1;
   if (arg_size)
      *arg_size = 1;
   if (data_type)
      *data_type = KSTRING;
   if (permanent)
      *permanent = TRUE;

   if (kgetenv("KHOROS_HISTORY"))
      return TRUE;
   else 
      return FALSE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _history_print
|       Purpose: print the history attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_history_print(kobject obj, int assoc, int attrib, kaddr clientData,
	       kfile * outfile)
{
   char *hist;


   if (!kdms_get_attribute(obj, NULL, _INTERNAL_HISTORY, &hist))
      return FALSE;

   if (outfile)
      kfprintf(outfile, "%s", hist);

   return TRUE;
}

/*********** ---------------------------------------------------------------
 ***********  KPDS_MINMAX
 *********** --------------------------------------------------------------- */

/*-----------------------------------------------------------
|  Routine Name: (static) _minmax_get
|       Purpose: get the min and max for the current region size
|    Written By: Jeremy Worley & John Salas
|          Date: Sep 28, 1993 10:28
------------------------------------------------------------*/
/* ARGSUSED */
static int
_minmax_get(kobject obj, int assoc, int attr, kaddr clientData,
	     kva_list *list)
{
   double *ret_smin;
   double *ret_smax;
   double  smin = 0.0;
   double  smax = 0.0;
   double *data = NULL;
   kobject tmp;
   int i;
   int w;
   int h;
   int d;
   int t;
   int e;
   int n;
   int j;
   
   /* verify that the segment is present */
   if (!kdms_query_segment(obj, (char *)clientData))
      return FALSE;

   tmp = kdms_reference(obj);

   /* set a friendly presentation type */
   kdms_set_attribute(tmp, (char *) clientData, KDMS_DATA_TYPE, KDOUBLE);

   if (kstrcmp((char *) clientData, KPDS_SEGMENT_VALUE) == 0) 
   {
      kpds_get_attribute(tmp, KPDS_VALUE_OPTIMAL_REGION_SIZE,
			  &w, &h, &d, &t, &e, &n);
      kpds_set_attributes(tmp, KPDS_VALUE_REGION_SIZE, w, h, d, t, e,
			   KPDS_VALUE_POSITION, 0,0,0,0,0, NULL);
			     
      if ((data = kpds_get_data(tmp, KPDS_VALUE_REGION, NULL)) == NULL) 
      {
	 kdms_close(tmp);
	 return FALSE;
      }
      
      for (j = 0; j < n; j++)
      {
	 if ((data = kpds_get_data(tmp, KPDS_VALUE_REGION, data)) == NULL) 
	 {
	    kdms_close(tmp);
	    return FALSE;
	 }
	 if (j == 0)
	    smin = smax = data[0];
	 
	 for (i = 0; i < w*h*d*t*e; i++)
	 {
	    smin = kmin(data[i],smin);
	    smax = kmax(data[i],smax);
	 }
      }
      
      kfree(data);
   }
   else if (kstrcmp((char *) clientData, KPDS_SEGMENT_MAP) == 0)
   {
      if ((data = kpds_get_data(tmp, KPDS_MAP_REGION, NULL)) == NULL)
      {
	 kdms_close(tmp);
	 return FALSE;
      }
      
      smin = smax = data[0];

      kpds_get_attribute(tmp, KPDS_MAP_REGION_SIZE, &w, &h, &d, &t, &e);

      for (i = 1; i < w*h*d*t*e; i++)
      {
	 smin = kmin(data[i],smin);
	 smax = kmax(data[i],smax);
      }

      kfree(data);
   }
   else if (kstrcmp((char *) clientData, KPDS_SEGMENT_MASK) == 0)
   {
      if ((data = kpds_get_data(tmp, KPDS_MASK_REGION, NULL)) == NULL)
      {
	 kdms_close(tmp);
	 return FALSE;
      }
      
      smin = smax = data[0];

      kpds_get_attribute(tmp, KPDS_MASK_REGION_SIZE, &w, &h, &d, &t, &e);

      for (i = 1; i < w*h*d*t*e; i++)
      {
	 smin = kmin(data[i],smin);
	 smax = kmax(data[i],smax);
      }

      kfree(data);
   }
   else if (kstrcmp((char *) clientData, KPDS_SEGMENT_LOCATION) == 0)
   {
      int grid;
   
   
      /* get the current grid type -- KRECTILINEAR or KCURVILINEAR */
      if (!kpds_get_attribute(obj, KPDS_LOCATION_GRID, &grid))
	 return(FALSE);
      
      /* 
       *  if the grid type is rectlinear, then we need to get the 
       *  location points from the width, height, and depth segments.
       */ 
      if (grid == KRECTILINEAR) 
      {
	 double *data_w = NULL;
	 double *data_h = NULL;
	 double *data_d = NULL;

	 
	 data_w = kpds_get_data(tmp, KPDS_LOCATION_WIDTH_REGION, NULL);
	 data_h = kpds_get_data(tmp, KPDS_LOCATION_HEIGHT_REGION, NULL);
	 data_d = kpds_get_data(tmp, KPDS_LOCATION_DEPTH_REGION, NULL);
	 
	 if (data_w == NULL || data_h == NULL || data_d)
	 {
	    kdms_close(tmp);
	    return FALSE;
	 }

	 smin = smax = data_w[0];
	 
	 kpds_get_attribute(tmp, KPDS_LOCATION_REGION_SIZE, &w, &h, &d, &e);

	 for (i = 1; i < w; i++)
	 {
	    smin = kmin(data_w[i],smin);
	    smin = kmin(data_h[i],smin);
	    smin = kmin(data_d[i],smin);

	    smax = kmax(data_w[i],smax);
	    smax = kmax(data_h[i],smax);
	    smax = kmax(data_d[i],smax);
	 }
	    
	 kfree(data_w);  
	 kfree(data_h);  
	 kfree(data_d);  
      }
      else if (grid == KCURVILINEAR) 
      {
	 if ((data = kpds_get_data(tmp, KPDS_LOCATION_REGION, NULL)) == NULL)
	 {
	    kdms_close(tmp);
	    return FALSE;
	 }
      
	 smin = smax = data[0];
	 
	 kpds_get_attribute(tmp, KPDS_LOCATION_REGION_SIZE, &w, &h, &d, &e);

	 for (i = 1; i < w*h*d*e; i++)
	 {
	    smin = kmin(data[i],smin);
	    smax = kmax(data[i],smax);
	 }
      }

      kfree(data);
   }
   else if (!kstrcmp((char *)clientData, KPDS_SEGMENT_TIME) == 0)
   {
      if ((data = kpds_get_data(tmp, KPDS_TIME_REGION, NULL)) == NULL)
      {
	 kdms_close(tmp);
	 return FALSE;
      }
      
      smin = smax = data[0];

      kpds_get_attribute(tmp, KPDS_TIME_REGION_SIZE, &t);

      for (i = 1; i < t; i++)
      {
	 smin = kmin(data[i],smin);
	 smax = kmax(data[i],smax);
      }

      kfree(data);
   }

   ret_smin = kva_arg(*list, double *);
   ret_smax = kva_arg(*list, double *);

   if (ret_smin != NULL) *ret_smin = smin;
   if (ret_smax != NULL) *ret_smax = smax;

   return TRUE;
}

/*********** ---------------------------------------------------------------
 ***********  KPDS_INCREMENT_SIZE
 *********** --------------------------------------------------------------- */

/*-----------------------------------------------------------
|  Routine Name: (static) _increment_size_get
|       Purpose: get the increment_size attribute
|    Written By: Jeremy Worley
|          Date: 
------------------------------------------------------------*/
/* ARGSUSED */
static int
_increment_size_get(kobject obj, int assoc, int attr, kaddr clientData,
	      kva_list *list)
{
   
   int *size;
   int *isize;
   int *order;
   int dim;
   int i;
   int *args[KDMS_MAX_DIM];

   /* verify that the segment is present */
   if (!kdms_query_segment(obj, (char *)clientData))
      return FALSE;

   kdms_get_attributes(obj, (char *)clientData, _INTERNAL_INCREMENT_SIZE,
		       &isize, KDMS_SIZE, &size, KDMS_DIMENSION, &dim,
		       KDMS_INDEX_ORDER, &order, NULL);

   for (i = 0 ; i < dim; i++)
       args[i] = kva_arg(*list, int *);

   if (kstrcmp((char *)clientData, KPDS_SEGMENT_VALUE) == 0 || 
       kstrcmp((char *)clientData, KPDS_SEGMENT_MASK) == 0)
   {
      if (args[0] != NULL)
	 *(args[0]) = isize[KWID] > 0 ? isize[KWID] : size[KWID];

      if (args[1] != NULL)
	 *(args[1]) = isize[KHGT] > 0 ? isize[KHGT] : size[KHGT];

      if (args[2] != NULL)
	 *(args[2]) = isize[KDEP] > 0 ? isize[KDEP] : size[KDEP];

      if (args[3] != NULL)
	 *(args[3]) = isize[KTIM] > 0 ? isize[KTIM] : size[KTIM];

      if (args[4] != NULL)
	 *(args[4]) = isize[KELM] > 0 ? isize[KELM] : size[KELM];

   }
   else if (kstrcmp((char *)clientData, KPDS_SEGMENT_MAP) == 0)
   {
      if (args[0] != NULL)
	 *(args[0]) = isize[KMAP_WID] > 0 ? isize[KMAP_WID] : size[KMAP_WID];

      if (args[1] != NULL)
	 *(args[1]) = isize[KMAP_HGT] > 0 ? isize[KMAP_HGT] : size[KMAP_HGT];

      if (args[2] != NULL)
	 *(args[2]) = isize[KMAP_DEP] > 0 ? isize[KMAP_DEP] : size[KMAP_DEP];

      if (args[3] != NULL)
	 *(args[3]) = isize[KMAP_TIM] > 0 ? isize[KMAP_TIM] : size[KMAP_TIM];

      if (args[4] != NULL)
	 *(args[4]) = isize[KMAP_ELM] > 0 ? isize[KMAP_ELM] : size[KMAP_ELM];

   }
   else if (kstrcmp((char *)clientData, KPDS_SEGMENT_LOCATION) == 0)
   {
      if (args[0] != NULL)
	 *(args[0]) = isize[KWID] > 0 ? isize[KWID] : size[KWID];

      if (args[1] != NULL)
	 *(args[1]) = isize[KHGT] > 0 ? isize[KHGT] : size[KHGT];

      if (args[2] != NULL)
	 *(args[2]) = isize[KDEP] > 0 ? isize[KDEP] : size[KDEP];

      if (args[3] != NULL)
	 *(args[3]) = isize[KDEP] > 0 ? isize[KDEP] : size[KDEP];

   }
   else if (kstrcmp((char *)clientData, KPDS_SEGMENT_TIME) == 0)
   {
      if (args[0] != NULL)
	 *(args[0]) = isize[0] > 0 ? isize[0] : size[0];
   }
   else
      return (FALSE);

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _increment_size_set
|       Purpose: set the increment_size 
|    Written By: Jeremy Worley
|          Date: 
------------------------------------------------------------*/
/* ARGSUSED */
static int
_increment_size_set(kobject obj, int assoc, int attr, kaddr clientData,
	      kva_list *list)
{
   int i;
   int *isize;
   int dim;
   
   int vorder[KDMS_MAX_DIM] = {KWIDTH, KHEIGHT, KDEPTH, KTIME, KELEMENTS};
   int morder[KDMS_MAX_DIM] = {KMAP_WIDTH, KMAP_HEIGHT, KMAP_DEPTH, 
			       KMAP_TIME, KMAP_ELEMENTS};
   int lorder[KDMS_MAX_DIM] = {KWIDTH, KHEIGHT, KDEPTH, KDIMENSION};
   int args[KDMS_MAX_DIM];
   int *order;
   
   /* verify that the segment is present */
   if (!kdms_query_segment(obj, (char *)clientData))
      return FALSE;

   kdms_get_attributes(obj, (char *)clientData, _INTERNAL_INCREMENT_SIZE,
		       &isize, KDMS_DIMENSION, &dim,
		       KDMS_INDEX_ORDER, &order, NULL);

   for (i = 0 ; i < dim; i++)
       args[i] = kva_arg(*list, int);

   if (kstrcmp((char *)clientData, KPDS_SEGMENT_VALUE) == 0 ||
       kstrcmp((char *)clientData, KPDS_SEGMENT_MASK) == 0)
   {
      for (i = 0 ; i < dim; i++)
	 isize[i] = _kdms_get_order(order[i], vorder, args, dim, 1);
   }
   else if (kstrcmp((char *)clientData, KPDS_SEGMENT_MAP) == 0)
   {
      for (i = 0 ; i < dim; i++)
	 isize[i] = _kdms_get_order(order[i], morder, args, dim, 1);
   }
   else if (kstrcmp((char *)clientData, KPDS_SEGMENT_LOCATION) == 0)
   {
      for (i = 0 ; i < dim; i++)
	 isize[i] = _kdms_get_order(order[i], lorder, args, dim, 1);
   }
   else if (kstrcmp((char *)clientData, KPDS_SEGMENT_TIME) == 0)
      isize[0] = args[0];
   else
      return (FALSE);
   
   return (TRUE);
}

/*-----------------------------------------------------------
|  Routine Name: (static) _increment_size_match
|       Purpose: match the increment_size attribute
|    Written By: Jeremy Worley
|          Date: 
------------------------------------------------------------*/
/* ARGSUSED */
static int
_increment_size_match(kobject obj1, kobject obj2, int assoc, int attrib,
		kaddr clientData1, kaddr clientData2)
{
   int *isize1 = NULL;
   int *isize2 = NULL;
   int  dim;
   int  i;


   /* verify that the segments are present */
   if (!kdms_query_segment(obj1, (char *)clientData1) ||
       !kdms_query_segment(obj2, (char *)clientData2))
      return FALSE;

   kdms_get_attribute(obj1, (char *)clientData1, _INTERNAL_INCREMENT_SIZE, 
		      &isize1);
   kdms_get_attributes(obj2, (char *)clientData2, _INTERNAL_INCREMENT_SIZE, 
		       &isize2, KDMS_DIMENSION, &dim, NULL);
   
   for (i = 0; i < dim; i++)
      if (isize1[i] != isize2[i])
	 return FALSE;
   
   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _increment_size_copy
|       Purpose: copy the increment_size attribute
|    Written By: Jeremy Worley
|          Date: 
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_increment_size_copy(kobject obj1, kobject obj2, int assoc, int attrib,
	       kaddr clientData1, kaddr clientData2)
{
   int *isize = NULL;


   /* verify that the segments are present */
   if (!kdms_query_segment(obj1, (char *)clientData1) ||
       !kdms_query_segment(obj2, (char *)clientData2))
      return FALSE;

   kdms_get_attribute(obj1, clientData1, _INTERNAL_INCREMENT_SIZE, &isize);
   kdms_set_attribute(obj2, clientData2, _INTERNAL_INCREMENT_SIZE, isize);

   return (TRUE);
}

/*-----------------------------------------------------------
|  Routine Name: (static) _increment_size_query
|       Purpose: query the increment_size attribute
|    Written By: Jeremy Worley
|          Date: 
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_increment_size_query(kobject obj, int assoc, int attrib, kaddr clientData,
		int *num_args, int *arg_size, int *data_type,
		int *permanent)
{
   /* verify that the segment is present */
   if (!kdms_query_segment(obj, (char *)clientData))
      return FALSE;

   /*
    * note that in this call, the arg_size and num_args are reversed
    * so so that it returns the multi-argument form.  The internal
    * version of this attribute is a single array of size dim, whereas
    * the kpds version of this attribute is dim arguments..  Reversing
    * the arguments in this manner results in returning the correct
    * information. 
    */
   return(kdms_query_attribute(obj, (char *)clientData, 
			       _INTERNAL_INCREMENT_SIZE, arg_size, num_args,
			       data_type, permanent));
}

/*-----------------------------------------------------------
|  Routine Name: (static) _increment_size_print
|       Purpose: print the increment_size attribute
|    Written By: Jeremy Worley
|          Date: 
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_increment_size_print(kobject obj, int assoc, int attrib, kaddr clientData,
		kfile * outfile)
{
   int vorder[KDMS_MAX_DIM] = {KWIDTH, KHEIGHT, KDEPTH, KTIME, KELEMENTS};
   int morder[KDMS_MAX_DIM] = {KMAP_WIDTH, KMAP_HEIGHT, KMAP_DEPTH, 
			       KMAP_TIME, KMAP_ELEMENTS};
   int lorder[KDMS_MAX_DIM] = {KWIDTH, KHEIGHT, KDEPTH, KDIMENSION};
   int *isize = NULL;
   int *size = NULL;
   int *order = NULL;
   int tmp1[KDMS_MAX_DIM];
   int tmp2[KDMS_MAX_DIM];
   int  dim;
   int  i;
   

   /* verify that the segment is present */
   if (!kdms_query_segment(obj, (char *)clientData))
      return FALSE;

   kdms_get_attributes(obj, (char *)clientData, KDMS_DIMENSION, &dim, 
		       KDMS_SIZE, &size, _INTERNAL_INCREMENT_SIZE, &isize,
		       KDMS_INDEX_ORDER, &order,
		       NULL);

   for (i = 0 ; i < dim; i++)
      tmp1[i] = isize[i] > 0 ? isize[i] : size[i];
   
   if (kstrcmp((char *)clientData, KPDS_SEGMENT_VALUE) == 0 ||
       kstrcmp((char *)clientData, KPDS_SEGMENT_VALUE) == 0)
   {
      for (i = 0 ; i < dim; i++)
	 tmp2[i] = _kdms_get_order(order[i], vorder, tmp1, dim, 1);
   }
   else if (kstrcmp((char *)clientData, KPDS_SEGMENT_MAP) == 0)
   {
      for (i = 0 ; i < dim; i++)
	 tmp2[i] = _kdms_get_order(order[i], morder, tmp1, dim, 1);
   }
   else if (kstrcmp((char *)clientData, KPDS_SEGMENT_LOCATION) == 0)
   {
      for (i = 0 ; i < dim; isize++)
	 tmp2[i] = _kdms_get_order(order[i], lorder, tmp1, dim, 1);
   }
   else if (kstrcmp((char *)clientData, KPDS_SEGMENT_TIME) == 0)
      tmp2[0] = tmp1[0];
   else
      return (FALSE);

   for (i = 0; i < dim; i++)
      kfprintf(outfile, " %d ", tmp2[i]);
   
   return TRUE;
}

/*
 *    ==================================================================
 *    Initialization Routine
 *    ==================================================================
 */

/*-----------------------------------------------------------
|
|  Routine Name: kaps_init
|
|       Purpose: 
|
|		 We don't need:	KDMS_NAME, KDMS_ARCHITECTURE
|		 because they are #defined to KDMS_*,
|		 have exactly the same arguments, and are
|		 object level attributes.
|
|         Input: none
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley, John Salas, and Steve Kubica
|          Date: Sep 28, 1993 08:10
| Modifications:
|
------------------------------------------------------------*/
void
kaps_init(void)
{
   int zero[KDMS_MAX_DIM] = {0, 0, 0, 0, 0};
   int minusone[KDMS_MAX_DIM] = {-1, -1, -1, -1, -1};

   if (_polymorphic_model_initialized)
      return;

   _polymorphic_model_initialized = TRUE;

   /* initalize handy tokenized versions of the polymorphic segment names */
   _kaps_init_segment_tokens();


   kdms_define_attribute(KDMS_OBJECT, KDMS_COMMENT, 1, 1, KSTRING,
			 TRUE, TRUE, NULL);

   kdms_define_attribute(KDMS_OBJECT, KDMS_DATE, 1, 1, KSTRING,
			 TRUE, TRUE, NULL);

   kdms_define_attribute(KDMS_OBJECT, KPDS_MAPPING_MODE, 1, 1, KINT,
			 FALSE, FALSE, KUNMAPPED);

   kdms_define_attribute(KDMS_OBJECT, KPDS_MASKED_VALUE_PRESENTATION, 1, 1, 
			 KINT, TRUE, TRUE, KUSE_ORIGINAL);

   kdms_define_attribute(KDMS_OBJECT, KPDS_MASK_SUBSTITUTE_VALUE, 2, 1, 
			 KDOUBLE, TRUE, TRUE, 1.0, 0.0);

   /* 
    *   axis assignment 
    */
   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KPDS_AXIS_ASSIGNMENT, NULL,
			       _axis_get, _axis_set, NULL, NULL, NULL, NULL);


   /* 
    *   position attributes
    */
   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KPDS_VALUE_POSITION, _pds_seg_value,
			       _position_get,   _position_set, 
			       _position_match, _position_copy, 
			       _position_query, _position_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KPDS_MASK_POSITION, _pds_seg_mask,
			       _position_get,   _position_set, 
			       _position_match, _position_copy, 
			       _position_query, _position_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KPDS_MAP_POSITION, _pds_seg_map,
			       _position_get,   _position_set, 
			       _position_match, _position_copy, 
			       _position_query, _position_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KPDS_LOCATION_POSITION, _pds_seg_location,
			       _position_get,   _position_set, 
			       _position_match, _position_copy, 
			       _position_query, _position_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KPDS_TIME_POSITION, _pds_seg_time,
			       _position_get,   _position_set, 
			       _position_match, _position_copy, 
			       _position_query, _position_print);

   /*
    *   position offsets
    */
   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KPDS_MASK_OFFSET, _pds_seg_mask,
			       _offset_get,   _offset_set, 
			       _offset_match, _offset_copy, 
			       _offset_query, _offset_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KPDS_VALUE_OFFSET, _pds_seg_value,
			       _offset_get,   _offset_set, 
			       _offset_match, _offset_copy, 
			       _offset_query, _offset_print);
   
   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KPDS_MAP_OFFSET, _pds_seg_map,
			       _offset_get,   _offset_set, 
			       _offset_match, _offset_copy, 
			       _offset_query, _offset_print);
   
   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KPDS_LOCATION_OFFSET, _pds_seg_location,
			       _offset_get,   _offset_set, 
			       _offset_match, _offset_copy, 
			       _offset_query, _offset_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KPDS_TIME_OFFSET, _pds_seg_time,
			       _offset_get,   _offset_set, 
			       _offset_match, _offset_copy, 
			       _offset_query, _offset_print);
   
   /* 
    *   increment_size attributes
    */
   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KPDS_VALUE_INCREMENT_SIZE, _pds_seg_value,
			       _increment_size_get,   _increment_size_set, 
			       _increment_size_match, _increment_size_copy, 
			       _increment_size_query, _increment_size_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KPDS_MASK_INCREMENT_SIZE, _pds_seg_mask,
			       _increment_size_get,   _increment_size_set, 
			       _increment_size_match, _increment_size_copy, 
			       _increment_size_query, _increment_size_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KPDS_MAP_INCREMENT_SIZE, _pds_seg_map,
			       _increment_size_get,   _increment_size_set, 
			       _increment_size_match, _increment_size_copy, 
			       _increment_size_query, _increment_size_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KPDS_LOCATION_INCREMENT_SIZE, _pds_seg_location,
			       _increment_size_get,   _increment_size_set, 
			       _increment_size_match, _increment_size_copy, 
			       _increment_size_query, _increment_size_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KPDS_TIME_INCREMENT_SIZE, _pds_seg_time,
			       _increment_size_get,   _increment_size_set, 
			       _increment_size_match, _increment_size_copy, 
			       _increment_size_query, _increment_size_print);

   /*
    *   internal position information
    */

   /* -- value position -- */
   kdms_define_attribute(KPDS_SEGMENT_VALUE, _INTERNAL_POSITION, 1, 5, KINT,
			 FALSE, FALSE, zero);

   /* -- mask position -- */
   kdms_define_attribute(KPDS_SEGMENT_MASK, _INTERNAL_POSITION, 1, 5, KINT,
			 FALSE, FALSE, zero);

   /* -- map position -- */
   kdms_define_attribute(KPDS_SEGMENT_MAP, _INTERNAL_POSITION, 1, 5, KINT,
			 FALSE, FALSE, zero);

   /* -- time position -- the 2 is a kludge -- */
   kdms_define_attribute(KPDS_SEGMENT_TIME, _INTERNAL_POSITION, 1, 2, KINT,
			 FALSE, FALSE, zero);

   /* -- location position for uniform rectilinear and curvilinear -- */
   kdms_define_attribute(KPDS_SEGMENT_LOCATION, _INTERNAL_POSITION, 1, 4, KINT,
			 FALSE, FALSE, zero);
   kdms_define_attribute(KPDS_SEGMENT_WIDTH, _INTERNAL_POSITION, 1, 4, KINT,
			 FALSE, FALSE, zero);
   kdms_define_attribute(KPDS_SEGMENT_UNIFORM, _INTERNAL_POSITION, 1, 4, KINT,
			 FALSE, FALSE, zero);


   /*
    *   internal offset information
    */

   /* -- value offset -- */
   kdms_define_attribute(KPDS_SEGMENT_VALUE, _INTERNAL_OFFSET, 1, 5, KINT,
			 FALSE, FALSE, zero);

   /* -- mask offset -- */
   kdms_define_attribute(KPDS_SEGMENT_MASK, _INTERNAL_OFFSET, 1, 5, KINT,
			 FALSE, FALSE, zero);

   /* -- map offset -- */
   kdms_define_attribute(KPDS_SEGMENT_MAP, _INTERNAL_OFFSET, 1, 5, KINT,
			 FALSE, FALSE, zero);

   /* -- time offset -- the 2 is a kludge -- */
   kdms_define_attribute(KPDS_SEGMENT_TIME, _INTERNAL_OFFSET, 1, 2, KINT,
			 FALSE, FALSE, zero);

   /* -- location offset for uniform rectilinear and curvilinear -- */
   kdms_define_attribute(KPDS_SEGMENT_LOCATION, _INTERNAL_OFFSET, 1, 4, KINT,
			 FALSE, FALSE, zero);
   kdms_define_attribute(KPDS_SEGMENT_WIDTH, _INTERNAL_OFFSET, 1, 4, KINT,
			 FALSE, FALSE, zero);
   kdms_define_attribute(KPDS_SEGMENT_UNIFORM, _INTERNAL_OFFSET, 1, 4, KINT,
			 FALSE, FALSE, zero);

   /*
    *   autoincrement state
    */
   kdms_define_attribute(KDMS_ALL_SEGMENTS, _INTERNAL_INC_STATE, 3, 1, KINT,
			 FALSE, FALSE, 0, 0, 0);

   /*
    *   autoincrement-size variable.
    */
   kdms_define_attribute(KDMS_ALL_SEGMENTS, _INTERNAL_INCREMENT_SIZE, 1, 5, 
			 KINT, FALSE, FALSE, minusone);

   /*
    *   complex convert attributes 
    */
   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KPDS_VALUE_COMPLEX_CONVERT, _pds_seg_value,
			       _complex_convert_get,   _complex_convert_set, 
			       _complex_convert_match, _complex_convert_copy, 
			       _complex_convert_query, _complex_convert_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KPDS_MAP_COMPLEX_CONVERT,_pds_seg_map,
			       _complex_convert_get,   _complex_convert_set, 
			       _complex_convert_match, _complex_convert_copy, 
			       _complex_convert_query, _complex_convert_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KPDS_LOCATION_COMPLEX_CONVERT,_pds_seg_location,
			       _complex_convert_get,   _complex_convert_set, 
			       _complex_convert_match, _complex_convert_copy, 
			       _complex_convert_query, _complex_convert_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KPDS_TIME_COMPLEX_CONVERT, _pds_seg_time,
			       _complex_convert_get,   _complex_convert_set, 
			       _complex_convert_match, _complex_convert_copy, 
			       _complex_convert_query, _complex_convert_print);

   /*
    *   interpolate attributes 
    */
   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KPDS_MASK_INTERPOLATE, _pds_seg_mask,
			       _interpolate_get,   _interpolate_set, 
			       _interpolate_match, _interpolate_copy, 
			       _interpolate_query, _interpolate_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KPDS_VALUE_INTERPOLATE, _pds_seg_value,
			       _interpolate_get,   _interpolate_set, 
			       _interpolate_match, _interpolate_copy, 
			       _interpolate_query, _interpolate_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KPDS_MAP_INTERPOLATE, _pds_seg_map,
			       _interpolate_get,   _interpolate_set, 
			       _interpolate_match, _interpolate_copy, 
			       _interpolate_query, _interpolate_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KPDS_LOCATION_INTERPOLATE, _pds_seg_location,
			       _interpolate_get,   _interpolate_set, 
			       _interpolate_match, _interpolate_copy, 
			       _interpolate_query, _interpolate_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KPDS_TIME_INTERPOLATE, _pds_seg_time,
			       _interpolate_get,   _interpolate_set, 
			       _interpolate_match, _interpolate_copy, 
			       _interpolate_query, _interpolate_print);

   /*
    *   normalization minimum attributes 
    */
   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KPDS_VALUE_NORM_MIN, _pds_seg_value,
			       _norm_min_get,   _norm_min_set, 
			       _norm_min_match, _norm_min_copy, 
			       _norm_min_query, _norm_min_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KPDS_MAP_NORM_MIN, _pds_seg_map,
			       _norm_min_get,   _norm_min_set, 
			       _norm_min_match, _norm_min_copy, 
			       _norm_min_query, _norm_min_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KPDS_LOCATION_NORM_MIN, _pds_seg_location,
			       _norm_min_get,   _norm_min_set, 
			       _norm_min_match, _norm_min_copy, 
			       _norm_min_query, _norm_min_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KPDS_TIME_NORM_MIN, _pds_seg_time,
			       _norm_min_get,   _norm_min_set, 
			       _norm_min_match, _norm_min_copy, 
			       _norm_min_query, _norm_min_print);

   /*
    *   normalization maximum attributes 
    */
   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KPDS_VALUE_NORM_MAX, _pds_seg_value,
			       _norm_max_get,   _norm_max_set, 
			       _norm_max_match, _norm_max_copy, 
			       _norm_max_query, _norm_max_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KPDS_MAP_NORM_MAX, _pds_seg_map,
			       _norm_max_get,   _norm_max_set, 
			       _norm_max_match, _norm_max_copy, 
			       _norm_max_query, _norm_max_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KPDS_LOCATION_NORM_MAX, _pds_seg_location,
			       _norm_max_get,   _norm_max_set, 
			       _norm_max_match, _norm_max_copy, 
			       _norm_max_query, _norm_max_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KPDS_TIME_NORM_MAX, _pds_seg_time,
			       _norm_max_get,   _norm_max_set, 
			       _norm_max_match, _norm_max_copy, 
			       _norm_max_query, _norm_max_print);

   /*
    *   pad value attributes 
    */
   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KPDS_MASK_PAD_VALUE, _pds_seg_mask,
			       _pad_value_get,   _pad_value_set, 
			       _pad_value_match, _pad_value_copy, 
			       _pad_value_query, _pad_value_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KPDS_VALUE_PAD_VALUE, _pds_seg_value,
			       _pad_value_get,   _pad_value_set, 
			       _pad_value_match, _pad_value_copy, 
			       _pad_value_query, _pad_value_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KPDS_MAP_PAD_VALUE, _pds_seg_map,
			       _pad_value_get,   _pad_value_set, 
			       _pad_value_match, _pad_value_copy, 
			       _pad_value_query, _pad_value_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KPDS_LOCATION_PAD_VALUE, _pds_seg_location,
			       _pad_value_get,   _pad_value_set, 
			       _pad_value_match, _pad_value_copy, 
			       _pad_value_query, _pad_value_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KPDS_TIME_PAD_VALUE, _pds_seg_time,
			       _pad_value_get,   _pad_value_set, 
			       _pad_value_match, _pad_value_copy, 
			       _pad_value_query, _pad_value_print);

   /*
    *   scale factor attributes 
    */
   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KPDS_VALUE_SCALE_FACTOR, _pds_seg_value,
			       _scale_factor_get,   _scale_factor_set, 
			       _scale_factor_match, _scale_factor_copy, 
			       _scale_factor_query, _scale_factor_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KPDS_MAP_SCALE_FACTOR, _pds_seg_map,
			       _scale_factor_get,   _scale_factor_set, 
			       _scale_factor_match, _scale_factor_copy, 
			       _scale_factor_query, _scale_factor_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KPDS_LOCATION_SCALE_FACTOR, _pds_seg_location,
			       _scale_factor_get,   _scale_factor_set, 
			       _scale_factor_match, _scale_factor_copy, 
			       _scale_factor_query, _scale_factor_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KPDS_TIME_SCALE_FACTOR, _pds_seg_time,
			       _scale_factor_get,   _scale_factor_set, 
			       _scale_factor_match, _scale_factor_copy, 
			       _scale_factor_query, _scale_factor_print);

   /*
    *   scale offset attributes 
    */
   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KPDS_VALUE_SCALE_OFFSET, _pds_seg_value,
			       _scale_offset_get,   _scale_offset_set, 
			       _scale_offset_match, _scale_offset_copy,
			       _scale_offset_query, _scale_offset_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KPDS_MAP_SCALE_OFFSET, _pds_seg_map,
			       _scale_offset_get,   _scale_offset_set, 
			       _scale_offset_match, _scale_offset_copy,
			       _scale_offset_query, _scale_offset_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KPDS_LOCATION_SCALE_OFFSET, _pds_seg_location,
			       _scale_offset_get,   _scale_offset_set, 
			       _scale_offset_match, _scale_offset_copy,
			       _scale_offset_query, _scale_offset_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KPDS_TIME_SCALE_OFFSET, _pds_seg_time,
			       _scale_offset_get,   _scale_offset_set, 
			       _scale_offset_match, _scale_offset_copy,
			       _scale_offset_query, _scale_offset_print);
   /*
    *   scaling attributes 
    */
   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KPDS_VALUE_SCALING, _pds_seg_value,
			       _scaling_get,   _scaling_set, 
			       _scaling_match, _scaling_copy, 
			       _scaling_query, _scaling_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KPDS_MAP_SCALING, _pds_seg_map,
			       _scaling_get,   _scaling_set, 
			       _scaling_match, _scaling_copy, 
			       _scaling_query, _scaling_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KPDS_LOCATION_SCALING, _pds_seg_location,
			       _scaling_get,   _scaling_set, 
			       _scaling_match, _scaling_copy, 
			       _scaling_query, _scaling_print);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KPDS_TIME_SCALING, _pds_seg_time,
			       _scaling_get,   _scaling_set, 
			       _scaling_match, _scaling_copy, 
			       _scaling_query, _scaling_print);

   /*
    *   min/max attributes
    */
   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KPDS_VALUE_MINMAX, _pds_seg_value,
			       _minmax_get, NULL, NULL, NULL, NULL, NULL);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KPDS_MAP_MINMAX, _pds_seg_map,
			       _minmax_get, NULL, NULL, NULL, NULL, NULL);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KPDS_LOCATION_MINMAX, _pds_seg_location,
			       _minmax_get, NULL, NULL, NULL, NULL, NULL);

   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KPDS_TIME_MINMAX, _pds_seg_time,
			       _minmax_get, NULL, NULL, NULL, NULL, NULL);


   /*
    *   history attributes
    */
   kdms_define_quasi_attribute(KDMS_OBJECT, 
			       KPDS_HISTORY,   NULL,
			       _history_get,   _history_set, 
			       _history_match, _history_copy, 
			       _history_query, _history_print);

   kdms_define_attribute(KDMS_OBJECT, _INTERNAL_HISTORY, 1, 1, KSTRING,
			 TRUE, TRUE, NULL);

   kdms_define_attribute(KDMS_OBJECT, KPDS_HISTORY_MODE, 1, 1, KINT,
			 FALSE, FALSE, KAPPEND_HISTORY);


   /* 
    *   location attributes
    */
   kdms_define_attribute(KDMS_OBJECT, KPDS_POINT_SIZE, 5, 1, KDOUBLE,
			 TRUE, TRUE, 1.0, 1.0, 1.0, 1.0, 1.0);

   kdms_define_attribute(KDMS_OBJECT, KPDS_KERNEL_ORIGIN, 5, 1, KINT,
			 TRUE, TRUE, 0, 0, 0, 0, 0);

   /* 
    *   subobject attribute
    */
   kdms_define_attribute(KDMS_OBJECT, KPDS_SUBOBJECT_POSITION, 5, 1, KINT,
			 TRUE, TRUE, 0, 0, 0, 0, 0);


   /* 
    *   elements first attribute
    */
   kdms_define_attribute(KDMS_OBJECT, KPDS_ELEMENTS_FIRST, 1, 1, KINT,
			 FALSE, FALSE, FALSE);

   /* 
    *  multiple definitions may sometimes occur - this is okay, but
    *  we don't want the user to be confused by this so we'll reset
    *  the errno
    */
   errno = 0;

   return;
}
