#! /bin/sh

# Replace oname and file.sh in the two lines below with the name 
# of the program you are testing and the file name of this test 
# (for example, test1.sh).

object_name="karith1 & khisto"
file_name="example.sh"

# ---------------------------------------------------------------
# ADD TESTS BETWEEN the  "Add tests below this" and "End of test 
# suite" comment lines (line 60).
# ---------------------------------------------------------------

echo "running "$file_name" for "$object_name" ... process ($$)"

# report utility
# ---------------------------------------------------------------
# Usage:  report  test_num test_obj cntrl_obj test_descr tol
#
# The report function uses ksegcmp to compare the test output 
# data object ($2) against the correct result ($3), and reports 
# success or failure of the test.  If the result of ksegcmp is 
# false, status is set to 1.
#
# Input Arguments: test_num - test number
#                  test_obj - the object generated by the test
#                  cntrl_obj - known result to compare against
#                  test_descr - a string describing the test
#                  tol - tolerance allowed when comparing object 
#                        data values. Recommended tolerance 1e-5
# ---------------------------------------------------------------
report()
{
   test_num="$1"
   test_obj="$2"
   cntrl_obj="$3"
   test_descr="$4"
   tol="$5"

   ksegcmp -i1 "$test_obj" -i2 "$cntrl_obj" -tol "$tol" -sat 1 -s -rt 1 -all
   if [ $? = 0 ]
   then
      echo "Passed:  Test  $test_num --> "$test_descr""
   else
      echo "FAILED:  Test  $test_num --> "$test_descr""
      status=1
   fi
}

# Initialize status to be success (ie. 0).  Upon failure, status 
# is initialized to 1.  Set the tmpdir variable.  Script will 
# use /tmp if $TMPDIR is not defined.
status=0
tmpdir=${TMPDIR:-/tmp}

# ---------------------------------------------------------------
#  Add tests below this
# ---------------------------------------------------------------

# ---------------------------------------------------------------
#  Example  1
#  This example demonstrates using the ascii import routines to 
#  build test and answer data sets.  The karith1 -abs option is 
#  being tested for float data.  For examples of using the ascii 
#  import routines to build more complex data objects, see 
#  test1.sh for knormal in the datamanip toolbox.  Appending $$ 
#  (process number) to the temporary file names ensures that 
#  tests won't collide, and is also useful for finding 
#  testsuites that are leaving files behind.
# ---------------------------------------------------------------

# Set the test number and the description of the test
test_num=1
descr="Example 1:  absolute value of float data"

# Create a temporary ascii file containing the test data values
cat <<EOF > $tmpdir/asc.$$
-1 1
57 -1e20
65536 -1.12345e-38
EOF

# Import the data as a 2x1x1x1x3 float data set using kasc2val
kasc2val -i1 $tmpdir/asc.$$ -wsize 2 -esize 3 -type float -o $tmpdir/data.$$

# Perform karith1 operation
karith1 -i $tmpdir/data.$$ -o $tmpdir/out.$$ -abs

# Create a temporary ascii file containing the result values
cat <<EOF > $tmpdir/asc.$$
1 1
57 1e20
65536 1.12345e-38
EOF

kasc2val -i1 $tmpdir/asc.$$ -wsize 2 -esize 3 -type float -o $tmpdir/result.$$

# Compare the results of the karith1 operation against the 
# correct results using the report utility
report $test_num "$tmpdir/out.$$" "$tmpdir/result.$$" "$descr" "1e-5"

# Remove the temporary files created by this test
/bin/rm -f $tmpdir/out.$$ $tmpdir/result.$$ $tmpdir/asc.$$ 
/bin/rm -f $tmpdir/data.$$

# ---------------------------------------------------------------
#  Example  2
#  This example demonstrates using the kecho program to determine
#  the path to the datamanip toolbox.  The karith2 program is 
#  tested using a file in the $DATAMANIP/data/testdata directory,
#  and the result is compared against a file containing the 
#  correct result.  The datamanip toolbox data/testdata 
#  directory contains a set of very small files that are easy to
#  verify using kprdata.
# ---------------------------------------------------------------

# Set the test number and the description of the test
test_num=2
descr="Example 2:  data multiplied, then divided by 2"

# set path for test data
tbpath=`kecho -tb DATAMANIP -echo path`
dpath="$tbpath/data/testdata"

# Perform test operations
karith2 -i1 $dpath/3x4.float.viff -o $tmpdir/out.$$ -mul -real 2
karith2 -i1 $tmpdir/out.$$ -o $tmpdir/out.$$ -div -real 2

# Compare data against the known result
report $test_num "$tmpdir/out.$$" "$dpath/3x4.float.viff" "$descr" "1e-5"

# Remove temporary files created by this test
/bin/rm -f $tmpdir/out.$$

# ---------------------------------------------------------------
#  Example  3
#  This example demonstrates using kstats to verify the results 
#  of a program.  kecho is used to determine the path to the 
#  sampledata toolbox.  The mean of the result is computed using 
#  kstats and is compared against the known mean value.  In this 
#  example, only the mean is computed.  It is often easier to 
#  compute all of the statistics (kstats -all), and save the 
#  verified results in a binary file (-o) in the test suite 
#  subdirectory.  This file can then be used directly when 
#  calling the report utility.
#
#  Computing the statistics is useful when testing larger data 
#  sets, and when testing programs where the results from run to 
#  run may vary, for example, in the noise generation routines.
#
#  For more test suite examples that use kstats, see the knoise, 
#  kgnoise, and kstats testsuites in the datamanip toolbox.
# ---------------------------------------------------------------

# Set the test number and the description of the test
test_num=3
descr="Example 3:  ball.xv histogram equalized"

# set path for sampledata images directory
tbpath=`kecho -tb SAMPLEDATA -echo path`
dpath="$tbpath/data/images"

# Histogram equalize ball.xv
khistops -equalize -i $dpath/ball.xv -o $tmpdir/out.$$ -whole

# Caluculate the mean of the result
kstats -i $tmpdir/out.$$ -o $tmpdir/mean.$$ -mean

# Create an ascii file containing the known mean value 
cat <<EOF > $tmpdir/asc.$$
128.971
EOF

# Compare the test mean against the known mean
report $test_num "$tmpdir/mean.$$" "$tmpdir/asc.$$" "$descr" "1e-3"

# Remove the temporary files created by this test
/bin/rm -f $tmpdir/asc.$$ $tmpdir/out.$$ $tmpdir/mean.$$

# ---------------------------------------------------------------
#  End of test suite
# ---------------------------------------------------------------

exit $status
