 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>> 
   >>>> 	Library Routine for kwiener
   >>>> 
   >>>>  Private: 
   >>>> 
   >>>>   Static: 
   >>>>   Public: 
   >>>> 	lkwiener
   >>>> 
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"

/* -library_includes */
kdcomplex
lkwiener_rest_calc(kdcomplex, kdcomplex, double);
/* -library_includes_end */


/****************************************************************
* 
*  Routine Name: lkwiener - perform Wiener filtering (restoration) in Fourier frequency domain
* 
*       Purpose: Perform restoration of data in the spatial domain by
*                Wiener filtering in the frequency domain.  The idea is
*                to deconvolve the blurring function from the blurred data.
*                A few problems exist in performing this operation, as
*                described in great detail in any good image processing book.
*
*		 A full implementation of a Wiener filter requires detailed 
*		 knowledge of the signal and noise power spectra. Many times 
*		 this information is simply unavailable or is very poorly known.
*		 In such situations, it is convenient to assign some constant 
*		 value to the SNR and use that in the Wiener filter
*		 implentation. This is a somewhat extreme restriction of the 
*		 Wiener filter, but it makes the filter very easy to use and 
*		 can give much better results than the pseudoinverse filter 
*		 in lkinverse().
*
*		 This routine implements the filter described in K.R. Castleman,
*		 "Digital Image Processing", Prentice-Hall 1979, 
*		 ISBN 0-13-212365-7, p281 Eq. 2 with the variable SNR term 
*		 given as the Weiner parameter constant.
*
*         Input: in_obj - source data object (FFT)
*                psg_obj - PSF data object (FFT)
*                wiener - Wiener parameter
*
*        Output: out_obj - filtered output data object (FFT)
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: Restrictions on data or input as applicable
*    Written By: Scott Wilson
*          Date: Apr 08, 1995
*      Verified: 
*  Side Effects: 
* Modifications: 
****************************************************************/
/* -library_def */
int lkwiener(kobject in_obj, kobject psf_obj, double wiener, kobject out_obj)
/* -library_def_end */

/* -library_code */
{
        char *lib = "kdatamanip", *rtn = "lkwiener";
        int w,h,d,t,e;                  /* Size of whole value segment */
        int wr,hr,dr,tr,er;             /* Size of value segment region */
        int itype,otype;                /* Data type of input and output */
        int i,j,nrgns;
        klist *objlist=NULL;
        kdcomplex *data_in,*data_psf;
 
        /* Make sure we have valid objects */
        if (in_obj == KOBJECT_INVALID)
          {
            kerror(lib, rtn, "Bogus input object");
            return(FALSE);
          }
        if (psf_obj == KOBJECT_INVALID)
          {
            kerror(lib, rtn, "Bogus PSF object");
            return(FALSE);
          }
        if (out_obj == KOBJECT_INVALID)
          {
            kerror(lib, rtn, "Bogus output object");
            return(FALSE);
          }
 
        /* Make sure there is a value segment */
        if (!kpds_query_value(in_obj))
          {
            kerror(lib,rtn,"No value data in source object to operate on.");
            (void) kpds_close_object(in_obj);
            return(FALSE);
          }
        if (!kpds_query_value(psf_obj))
          {
            kerror(lib,rtn,"No value data in PSF object to operate on.");
            (void) kpds_close_object(psf_obj);
            return(FALSE);
          }
        /* Make sure there is no location segment */
        if (kpds_query_location(in_obj))
          {
            kerror(lib,rtn,"Object with location segment not supported.");
            (void) kpds_close_object(in_obj);
 
            return(FALSE);
          }
        if (kpds_query_location(psf_obj))
          {
            kerror(lib,rtn,"Object with location segment not supported.");
            (void) kpds_close_object(psf_obj);
            return(FALSE);
          }
        /* Make sure there is no map segment */
        if (kpds_query_map(in_obj))
          {
            kerror(lib,rtn,"Object with map segment not supported.");
            (void) kpds_close_object(in_obj);
            return(FALSE);
          }
        if (kpds_query_location(psf_obj))
          {
            kerror(lib,rtn,"Object with map segment not supported.");
            (void) kpds_close_object(psf_obj);
            return(FALSE);
          }
 
        /* Reference the input objects to avoid side effects, then add the
           references to the list of goodies to be autmatically free'd on
           error. */
        KCALL((in_obj = kpds_reference_object(in_obj)));
        objlist = klist_add(objlist,in_obj,"KOBJECT");
        KCALL((psf_obj = kpds_reference_object(psf_obj)));
        objlist = klist_add(objlist,psf_obj,"KOBJECT");
 
        /* See how big the data is */
        KCALL(kpds_get_attribute(in_obj,KPDS_VALUE_SIZE,&w,&h,&d,&t,&e));
 
        /* Make sure the PSF is the same size as the source object */
        KCALL(kpds_match_attribute(in_obj,psf_obj,KPDS_VALUE_SIZE));
 
        /* Set the output data type according to the precision of the
           PSF input data */
        KCALL(kpds_get_attribute(psf_obj,KPDS_VALUE_DATA_TYPE,&itype));
        if (itype == KDCOMPLEX || itype == KDOUBLE) otype = KDCOMPLEX;
        else otype = KCOMPLEX;
 
        /* Copy all attributes from the input object to the output
           object.  Then force the datatype to that above. */
        KCALL(kpds_copy_object_attr(in_obj,out_obj));
        KCALL(kpds_set_attribute(out_obj,KPDS_VALUE_DATA_TYPE, otype));
 
        /* Reference the output object */
        KCALL((out_obj = kpds_reference_object(out_obj)));
        objlist = klist_add(objlist,out_obj,"KOBJECT");
 
        /* Set the presentation data types */
        KCALL(kpds_set_attribute(in_obj,KPDS_VALUE_DATA_TYPE, KDCOMPLEX));
        KCALL(kpds_set_attribute(psf_obj,KPDS_VALUE_DATA_TYPE, KDCOMPLEX));
        KCALL(kpds_set_attribute(out_obj,KPDS_VALUE_DATA_TYPE, KDCOMPLEX));
 
        /* Get the optimal region size and info. Set both input objects
           and the output object to match. */
        KCALL(kpds_get_attribute(in_obj,KPDS_VALUE_OPTIMAL_REGION_SIZE,
                                            &wr,&hr,&dr,&tr,&er,&nrgns));
        KCALL(kpds_set_attribute(in_obj,KPDS_VALUE_REGION_SIZE,
                                            wr,hr,dr,tr,er));
        KCALL(kpds_set_attribute(psf_obj,KPDS_VALUE_REGION_SIZE,
                                            wr,hr,dr,tr,er));
        KCALL(kpds_set_attribute(out_obj,KPDS_VALUE_REGION_SIZE,
                                            wr,hr,dr,tr,er));
        KCALL(kpds_set_attribute(in_obj,KPDS_VALUE_POSITION,0,0,0,0,0));
        KCALL(kpds_set_attribute(psf_obj,KPDS_VALUE_POSITION,0,0,0,0,0));
        KCALL(kpds_set_attribute(out_obj,KPDS_VALUE_POSITION,0,0,0,0,0));
 
        /* Allocate buffers for the data */
        KCALL(!((data_in = (kdcomplex *)kmalloc(wr*hr*dr*tr*er*
                                              sizeof(kdcomplex))) == NULL));
        objlist = klist_add(objlist,data_in,"KMALLOC");
        KCALL(!((data_psf = (kdcomplex *)kmalloc(wr*hr*dr*tr*er*
                                              sizeof(kdcomplex))) == NULL));
        objlist = klist_add(objlist,data_psf,"KMALLOC");
 
        /* Loop thru, reading the regions, processing, and writing the result */
        for (i=0; i<nrgns; i++)
          {
            kpds_get_data(in_obj,KPDS_VALUE_REGION,(kaddr)data_in);
            kpds_get_data(psf_obj,KPDS_VALUE_REGION,(kaddr)data_psf);
            for (j=0; j<wr*hr*dr*tr*er; j++)
              data_in[j] = lkwiener_rest_calc(data_in[j],data_psf[j],wiener);
            KCALL(kpds_put_data(out_obj,KPDS_VALUE_REGION,(kaddr)data_in));
          }
 
        (void)klist_free(objlist,(kfunc_void)lkcall_free);

	return TRUE;
}

kdcomplex
lkwiener_rest_calc(kdcomplex data_in, kdcomplex data_psf, double wiener)
  {
    /* This implements the filter described in K.R. Castleman,
       "Digital Image Processing", Prentice-Hall 1979, ISBN 0-13-212365-7,
       p281 Eq. 2 with the variable SNR term given as the Weiner parameter
       constant. */

    double mag;
    kdcomplex upper,lower,g;
    kdcomplex dcwiener;
    kdcomplex result;

    dcwiener.r = wiener;
    dcwiener.i = 0.0;
    upper = kdcconj(data_psf);
    mag = kdcmag(data_psf);
    lower.r = mag*mag;
    lower.i =0.0;
    g = kdcdiv(upper,kdcadd(lower,dcwiener));
    result = kdcmult(data_in,g);
    return(result);
  }
/* -library_code_end */
