 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>> 
   >>>> 	Library Routine for kreplace
   >>>> 
   >>>>  Private: 
   >>>> 
   >>>>   Static: 
   >>>>   Public: 
   >>>> 	lkreplace
   >>>> 
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"

/* -library_includes */
/* -library_includes_end */


/****************************************************************
* 
*  Routine Name: lkreplace - replace all occurrences of X with Y in data object
* 
*       Purpose: lkreplace replaces each occurence of the value old_val
*		 in the object with the value new_val. The operations 
*		 occur on the map segment if one is present in the source 
*		 object otherwise the operations occur on the value segment.
*
*		 The tolerance values allow the value that is being 
*		 replaced to fall within a given range.  the upper_tol 
*		 and lower_tol values will not be used unless the 
*		 corresponding flags are TRUE.
*
*		 The gating functionality will be removed in a future
*		 release.  
*
*		 Currently, lkreplace explicitly checks the validity 
*		 mask, and will not perform the operation if the masked
*		 data is invalid.  This will also be updated in a 
*		 future release to act more like lkarith1, where the 
*		 operation will still be performed, but the mask presentation
*		 attributes are used.
*
*         Input: src_obj   - source object whose values are substituted
*                gate_obj  - operation gating object
*                old_val   - value to be replaced
*                new_val   - value to be substituted 
*                upper_tol_flag - flag to indicate validity of upper tolerance 
*                upper_tol - upper tolerance allowed in old_val
*                lower_tol_flag - flag to indicate validity of lower tolerance
*                lower_tol - lower tolerance allowed in old_val
*
*        Output: dest_obj  - destination object
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Donna Koechner, Ashish Malhotra
*          Date: Apr 08, 1995
*      Verified: 
*  Side Effects: 
* Modifications: 
****************************************************************/
/* -library_def */
int lkreplace(
		 kobject src_obj,
                 kobject gate_obj,
                 double  old_val,
                 double  new_val,
                 int     upper_tol_flag,
                 double  upper_tol,
                 int     lower_tol_flag,
                 double  lower_tol,
                 kobject dest_obj)
/* -library_def_end */

/* -library_code */
{
        int mask_exists,map_exists,val_exists;
        int gate_val_exists = -1,gate_mask_exists = -1;
        int gate_map_exists = -1;
        int w,h,d,t,e;
        int type;
        int process_value_only   = FALSE;
        int process_map_only     = FALSE;
        char *info, *prim, *datasize, *datatype;
        int line_size, num_lines;
        int count,j;
        int error = FALSE;
        double *data = NULL, *gate_data = NULL;
        unsigned char *mask_data = NULL; 
        double low_val,high_val;


        if ( (src_obj = kpds_reference_object(src_obj)) == NULL)
        {
           kerror("kdatamanip","lkreplace",
                  "Failed to reference source object.");
           return FALSE;
        }

        if (gate_obj)
        {
           if ((gate_obj = kpds_reference_object(gate_obj)) == NULL)
           {
              (void) kpds_close_object(src_obj);
              kerror("kdatamanip","lkreplace",
                     "Failed to reference gating object.");
              return FALSE;
           }
        }

        val_exists = kpds_query_value(src_obj);
        mask_exists = kpds_query_mask(src_obj);
        if (gate_obj) 
        {
           gate_val_exists = kpds_query_value(gate_obj);
           gate_map_exists = kpds_query_map(gate_obj);
           gate_mask_exists = kpds_query_mask(gate_obj);
        }

        map_exists = kpds_query_map(src_obj);

       /*
        * The following section checks for illegal combination of segments
        * in the input objects. If an illegal segmetn is found an error
        * message is generated. 
        */

        if (!val_exists && !map_exists)
        {
           kerror("kdatamanip","lkreplace",
                  "Unable to find value or map data in source object.");
           (void) kpds_close_object(src_obj);
           if (gate_obj) (void) kpds_close_object(gate_obj);
           return FALSE; 
        }

        if (gate_obj && !val_exists && map_exists && !gate_map_exists)
        {
           kerror("kdatamanip","lkreplace", 
                  "Gate object needs map data if source object %s",
                  "contains only map data.");
           (void) kpds_close_object(src_obj);
           if (gate_obj) (void) kpds_close_object(gate_obj);
           return FALSE;
        }

        if (gate_obj && val_exists && !map_exists && !gate_val_exists)
        {
           kerror("kdatamanip","lkreplace", 
                  "Gate object needs value data if source object %s",
                  "contains only value data.");
           (void) kpds_close_object(src_obj);
           if (gate_obj) (void) kpds_close_object(gate_obj);
           return FALSE;
        }
 
        if (gate_obj && val_exists && gate_mask_exists)
        {
           kerror("kdatamanip","lkreplace", 
                  "Gate object should not have the mask segment.");
           (void) kpds_close_object(src_obj);
           if (gate_obj) (void) kpds_close_object(gate_obj);
           return FALSE;
        }

        if (val_exists && map_exists && gate_obj)
        {
           map_exists = FALSE;
           process_value_only = TRUE;
           kpds_set_attribute(src_obj, KPDS_MAPPING_MODE, KMAPPED);
           if (kpds_query_map(dest_obj)) 
              (void) kpds_destroy_map(dest_obj); 
        }
        else 
         if (map_exists) 
           process_map_only = TRUE;
        else 
           process_value_only = TRUE;
 
        if (process_value_only)
        {
           prim =  KPDS_VALUE_LINE;
           info = KPDS_VALUE_LINE_INFO;
           datasize = KPDS_VALUE_SIZE;
           datatype = KPDS_VALUE_DATA_TYPE;
        }
        else if (process_map_only)
        {
           prim = KPDS_MAP_LINE;
           info = KPDS_MAP_LINE_INFO;
           datasize = KPDS_MAP_SIZE;
           datatype = KPDS_MAP_DATA_TYPE;
        }
	else
	{
           kerror("kdatamanip","lkreplace", 
                  "Undefined processing mode (map only or value only).");
           (void) kpds_close_object(src_obj);
           if (gate_obj) (void) kpds_close_object(gate_obj);
           return FALSE;
	}

        kpds_get_attribute(src_obj,datasize,&w,&h,&d,&t,&e);
       
        if (gate_obj)
           kpds_set_attribute(gate_obj,datasize,w,h,d,t,e);

        kpds_set_attribute(dest_obj,datasize,w,h,d,t,e);

        kpds_get_attribute(src_obj,datatype,&type);

        kpds_set_attribute(dest_obj,datatype,type);

        if ( (dest_obj = kpds_reference_object(dest_obj)) == NULL)
        {
           kerror("kdatamanip","lkreplace",
                  "Failed to reference destination object.");
           return FALSE;
        }

        kpds_set_attribute(src_obj, datatype, KDOUBLE);
        kpds_set_attribute(dest_obj, datatype, KDOUBLE);
        if (gate_obj)
           kpds_set_attribute(gate_obj, datatype, KDOUBLE);

        kpds_get_attribute(src_obj,info,&line_size,&num_lines);

        high_val = (upper_tol_flag) ? (old_val+upper_tol) : old_val;
        low_val  = (lower_tol_flag) ? (old_val-lower_tol) : old_val;

        for (count = 0; count < num_lines; count++)
        {
           if (gate_obj)
              gate_data=(double *)kpds_get_data(gate_obj,prim,(kaddr)gate_data);

           if (mask_exists && val_exists) 
              mask_data = (unsigned char *) kpds_get_data(src_obj, 
                                   KPDS_MASK_LINE, (kaddr) mask_data);

           data = (double *) kpds_get_data(src_obj, prim, (kaddr) data);

           if (!gate_obj && !mask_exists)
           {
              for (j=0; j < line_size; j++)
                  data[j] = ((data[j] >=low_val) && (data[j]<=high_val)) ?
                            new_val : data[j]; 
           }
           else
           {
              for (j=0; j < line_size; j++)
              {
                 if ( (!gate_obj||gate_data[j]) && (!mask_exists||mask_data[j]))
                    data[j] = ((data[j] >=low_val) && (data[j]<=high_val)) ?
                              new_val : data[j];
              }
           }

           if (!kpds_put_data(dest_obj, prim, (kaddr) data))
           {
              kerror("kdatamanip","lkreplace",
                     "Failed to put line in output object");
              error = TRUE;
              break;
           }
        }
        (void) kpds_close_object(src_obj);
        (void) kpds_close_object(dest_obj);
        if (gate_obj) (void) kpds_close_object(gate_obj);

        kfree(gate_data);
        kfree(mask_data);
        kfree(data);
        
        if (error)
           return(FALSE);
        else
	   return TRUE;
}
/* -library_code_end */
