/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Error, Warning and Info Utilities
   >>>>
   >>>>  Private:
   >>>>   Static:
   >>>>		    _knotify_initialize()
   >>>>		    _kformat_message()
   >>>>		    _kformat_tty_message()
   >>>>             kerror_default()
   >>>>             kwarn_default()
   >>>>             kprompt_default()
   >>>>             kchoose_default()
   >>>>             kquery_default()
   >>>>             kinfo_default()
   >>>>             kannounce_default()
   >>>>   Public:
   >>>>             kset_errorhandler()
   >>>>             kset_warnhandler()
   >>>>             kset_announcehandler()
   >>>>             kset_prompthandler()
   >>>>             kset_choosehandler()
   >>>>             kset_infohandler()
   >>>>
   >>>>             kerror()
   >>>>             kwarn()
   >>>>             kprompt()
   >>>>             koverwrite()
   >>>>             kquery()
   >>>>             kchoose()
   >>>>             kinfo()
   >>>>             kannounce()
   >>>>             
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"


static int khoros_notify;
static int khoros_debug;
static knotify info_handler     = NULL;
static knotify announce_handler = NULL;
static knotify warn_handler     = NULL;
static knotify error_handler    = NULL;
static knotify prompt_handler   = NULL;
static knotify choose_handler   = NULL;
static knotify query_handler    = NULL;

static int in_use = FALSE;
static char *tempstr = NULL;

/*-----------------------------------------------------------
|
|  Routine Name: _knotify_initialize - initializes the notify level
|				       and error handlers
|
|       Purpose: This routine will initialize
|		 and re-format it according to the tty size.
|	  Input:
|        Output:
|       Returns:
|    Written By: Mark Young
|          Date: Sep 26, 1992 12:18
| Modifications:
|
------------------------------------------------------------*/

static void _knotify_initialize(void)
{
	char *str;
	static int initialized = FALSE;

	if (initialized == TRUE)
	   return;
	else
	   initialized = TRUE;

	tempstr = kmalloc(25*KLENGTH);
	if ((str = kgetenv("KHOROS_NOTIFY")) != NULL)
	{
	   if (!kstrcasecmp(str,"VERBOSE") || !kstrcasecmp(str,"KVERBOSE"))
	      khoros_notify = KVERBOSE;
	   else if (!kstrcasecmp(str,"DEBUG") || !kstrcasecmp(str,"KDEBUG"))
	      khoros_notify = KDEBUG;
	   else if (!kstrcasecmp(str,"QUIET") || !kstrcasecmp(str,"KQUIET"))
	      khoros_notify = KQUIET;
	   else if (!kstrcasecmp(str,"STANDARD")||!kstrcasecmp(str,"KSTANDARD"))
	      khoros_notify = KSTANDARD;
	   else if (!kstrcasecmp(str,"HOSTILE")||!kstrcasecmp(str,"KHOSTILE"))
	      khoros_notify = KHOSTILE;
	   else if (!kstrcasecmp(str,"SYSLIB")||!kstrcasecmp(str,"KSYSLIB"))
	      khoros_notify = KSYSLIB;
	   else if (!kstrcasecmp(str,"XVLIB")||!kstrcasecmp(str,"KXVLIB"))
	      khoros_notify = KXVLIB;
	   else
	   {
	      khoros_notify = KSTANDARD;
	      kinfo(KFORCE, "_knotify_initialize:  Unknown KHOROS_NOTIFY \
environment setting '%s'.  Setting khoros notify to STANDARD.", str);
	   }
	}
	else
	   khoros_notify = KSTANDARD;

	if ((str = kgetenv("KHOROS_DEBUG")) != NULL)
	   khoros_debug = TRUE;
	else
	   khoros_debug = FALSE;
}

/*-----------------------------------------------------------
|
|  Routine Name: _kformat_tty_message - format the message string for output 
|				       to the tty
|
|       Purpose: This routine will take the message string passed in
|		 and re-format it according to the tty size.
|
|	  Input: message - name of message in which error occured
|	         nl_spacing - special spacing to be added when a
|		              newline is added to the message during formatting 
|
|        Output:
|       Returns: A formated message string 
|
|    Written By: Tom Sauer & Steven Jorgensen
|          Date: Aug 16, 1992 15:20
| Modifications:
|
------------------------------------------------------------*/
static char *_kformat_tty_message(
   char *message,
   char *nl_spacing)
{

     char *new_message = NULL;
     char *ptr = message;
     char *last = ptr;
     char *tmp1, *tmp2;
     char temp[KLENGTH];
     int   cnt = 0;

     if (kstrlen(message) <= 80)
       return(kstrdup(message));


     while ( *ptr != '\0' )
      {
	if ( *ptr == '\n')
	  {
	    tmp1 = kstring_ncat(new_message, last, -1, ptr - last + 1, NULL);
	    kfree(new_message);
	    if (nl_spacing != NULL)
	      {
		ksprintf(temp, "%s", nl_spacing);
		tmp2 = kstring_cat(tmp1, temp, NULL);
		kfree(tmp1);
	      }
	    else
	      tmp2 = tmp1;
	    new_message = tmp2;
	    tmp1 = NULL;
	    tmp2 = NULL;
	    last = ptr +1;
	    cnt = -1;
	  }
	else if (cnt >= 80)
	  {
	    tmp2 = ptr;
	    while (tmp2 > last && (! isspace(*tmp2)))
	      tmp2--;
	    if (tmp2 != last)
	      {
		tmp1 = kstring_ncat(new_message, last, -1, tmp2-last, NULL);
		last = tmp2 + 1;
	      }
	    else
	      {
		tmp1 = kstring_ncat(new_message, last, -1, cnt, NULL);
		last = ptr+1;
	      }
	    if (nl_spacing != NULL)
	      {
		ksprintf(temp, "\n%s", nl_spacing);
		tmp2 = kstring_cat(tmp1, temp, NULL);
	      }
	    else
	      tmp2 = kstring_cat(tmp1, "\n", NULL);
	    kfree(tmp1);
	    kfree(new_message);
	    new_message = tmp2;
	    cnt = -1;
	  }
	ptr++;
	cnt++;
      }
    if (kstrlen(last))
      {
	tmp1 = kstring_cat(new_message, last, NULL);
	kfree(new_message);
	new_message = tmp1;
      }
    return(new_message);
  }

/*-----------------------------------------------------------
|
|  Routine Name: _kformat_message - format the message into the string
|				   
|
|       Purpose: This routine formats the message into the supplied
|		 message array.
|
|         Input: message - message to format
|		 args    - the argument list to be formatted
|
|        Output: string  - the formatted string
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: Aug 14, 1992
| Modifications:
|
------------------------------------------------------------*/
static int _kformat_message(
   char    *message,
   kva_list args,
   char    *string)
{
	if (message == NULL)
	   return(FALSE);

	/*
	 *  Call kvsprintf() to do the actual formating of the arguments
	 *  into the temporary character array.  If the number of arguments
	 *  printed does not equal 0 then call kfputs() to write the string
	 *  to the transport.  The return value of kfprintf() represents
	 *  the number of values actually written to the transport device.
	 */
	if (kvsprintf((char *) string, message, args) != 0)
	   return(TRUE);
	else
	   return(FALSE);
}

/*-----------------------------------------------------------
|
|  Routine Name: kerror_default - default error handler for kerror
|
|       Purpose: This routine acts as the default error handler
|		 for the public error handler kerror.  If the
|		 value sent to kset_errorhandler is NULL, the
|		 error handler used is this routine.
|
|         Input: toolbox  - name of toolbox in which error occured
|		 program  - name of program in which error occured
|		 library  - name of library in which error occured
|		 routine  - name of routine in which error occured
|		 category - name of category in which error occured
|		 message  - name of message in which error occured
|
|        Output:
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Danielle Argiro & Steven Jorgensen
|          Date: Jul 30, 1992
| Modifications:
|
------------------------------------------------------------*/
static int kerror_default(
   char *toolbox,
   char *program,
   char *library,
   char *routine,
   char *category,
   char *message)
{
	char *temp_message;

	(void) fprintf(stderr, "\n");
	if (toolbox != NULL)
	   (void) fprintf(stderr, "Toolbox:  %s\n", toolbox);
	if (program != NULL)
	   (void) fprintf(stderr, "Program:  %s\n", program);
	if (library != NULL)
	   (void) fprintf(stderr, "Library:  %s\n", library);
	if (routine != NULL)
	   (void) fprintf(stderr, "Routine:  %s\n", routine);
	if (category != NULL)
	   (void) fprintf(stderr, "Category: %s\n", category);
	if (message != NULL)
	{
	   temp_message = _kformat_tty_message(message, NULL);
	   (void) fprintf(stderr, "\n%s\n", temp_message);
	   kfree(temp_message);
	}
	(void) fprintf(stderr, "\n\n");
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: kwarn_default - default warn handler for kwarn
|
|       Purpose: This routine acts as the default warn handler
|		 for the public warn handler kwarn.  If the
|		 value sent to kset_warnhandler is NULL, the
|		 warn handler used is this routine.
|
|         Input: toolbox  - name of toolbox in which warning occured
|		 program  - name of program in which warning occured
|		 library  - name of library in which warning occured
|		 routine  - name of routine in which warning occured
|		 message  - name of message in which warning occured
|
|        Output:
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: Jan 4, 1994
| Modifications:
|
------------------------------------------------------------*/
static int kwarn_default(
   char *toolbox,
   char *program,
   char *library,
   char *routine,
   char *message)
{
	char *temp_message;

	(void) fprintf(stderr, "\n");
	if (toolbox != NULL)
	   (void) fprintf(stderr, "Toolbox: %s\n", toolbox);
	if (program != NULL)
	   (void) fprintf(stderr, "Program: %s\n", program);
	if (library != NULL)
	   (void) fprintf(stderr, "Library: %s\n", library);
	if (routine != NULL)
	   (void) fprintf(stderr, "Routine: %s\n", routine);
	if (message != NULL)
	{
	   temp_message = _kformat_tty_message(message, NULL);
	   (void) fprintf(stderr, "%s\n", temp_message);
	   kfree(temp_message);
	}
	(void) fprintf(stderr, "\n\n");
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: kinfo_default - default information handler for kinfo
|
|       Purpose: This routine is the default information handler
|		 for the kinfo routine.  If the
|		 value sent to kset_infohandler is NULL, the
|		 information handler used is this routine.
|
|         Input: message      - name of message in which error occured
|
|        Output:
|       Returns: TRUE (1) on success, FALSE (0) on fail.
|
|    Written By: Danielle Argiro, Steven Jorgensen, and Mark Young
|          Date: Oct 29, 1992
| Modifications:
|
------------------------------------------------------------*/
static int kinfo_default(
   char *message)
{
	char *temp_message;

	temp_message = _kformat_tty_message(message, NULL);
	(void) fprintf(stderr, "%s\n", temp_message);
	kfree(temp_message);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: kannounce_default - default announce handler for kannounce
|
|       Purpose: This routine is the default announce handler
|		 for the kannounce routine.  If the
|		 value sent to kset_announcehandler is NULL, the
|		 announce handler used is this routine.
|
|         Input: toolbox  - name of toolbox in which announce occured
|		 program  - name of program in which announce occured
|		 library  - name of library in which announce occured
|		 routine  - name of routine in which announce occured
|		 message  - name of message in which announce occured
|        Output:
|       Returns: TRUE (1) on success, FALSE (0) on fail.
|
|    Written By: Danielle Argiro, Steven Jorgensen, and Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/
static int kannounce_default(
   char *toolbox,
   char *program,
   char *library,
   char *routine,
   char *message)
{
	char *temp_message;

	if (!program) program = "(unknown)";
	temp_message = _kformat_tty_message(message, NULL);
	(void) fprintf(stderr, "%s: %s\n", program, temp_message);
	kfree(temp_message);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: kprompt_default - default warning handler for kprompt
|
|       Purpose: This routine is the default handler
|		 for the kprompt routine.  If the
|		 value sent to kset_prompthandler is NULL, the
|		 warning handler used is this routine.
|
|         Input: yes_repsonse - name of "yes" string in prompt
|		 no_repsonse  - name of "no" string in prompt
|		 default_val  - 1 indicates yes_response is default, 0 means
|				no_response is default.
|		 message      - name of message in which error occured
|
|        Output:
|        Returns: if prompt is TRUE, (1) on yes_response, (0) on
|			  no_response. Otherwise the default value is
|			  returned on an error 
|
|    Written By: Steven Jorgensen
|          Date: Jul 30, 1992 10:07
| Modifications:
|
------------------------------------------------------------*/
static int kprompt_default(
   char *yes_response,
   char *no_response,
   int  default_val,
   char *message)
{

    int   tty_fd, valid_response = FALSE, value, index = 0, numread;
    char *prompt;
    char *yes, *no;
    char  buffer[4*KLENGTH];

    if (( tty_fd = kopen("/dev/tty", KOPEN_RDWR, 0666)) < 0)
      {
        kfprintf(kstderr, "kprompt_prompt: Cannot open channel to terminal\n");
        kfprintf(kstderr, "                Returning default value\n");
        return(default_val);
      }

    prompt = _kformat_tty_message(message, NULL);

    ksprintf(buffer, "%s (%s/%s) [%s]: ", prompt, yes_response,
		no_response, ((default_val == 1) ? yes_response : no_response));
    value = default_val;

    yes = kstring_upper(yes_response, NULL);
    no = kstring_upper(no_response, NULL);

	/* 
	 * find the minumum unique string of the yes and no prompts 
	 */
    while (yes[index] == no[index])
      index++;
    index++;

	/* 
	 * Loop and redisplay the prompt message until a valid response
	 * is received
	 */

    while ( valid_response == FALSE )
      {
	kwrite(tty_fd, buffer, kstrlen(buffer));
	if ((numread = kread(tty_fd, buffer, KLENGTH)) == -1)
	  {
	    kfprintf(kstderr, "kprompt_default: Cannot read from terminal\n");
            kfprintf(kstderr, "                Returning default value\n");
	    break;
	  }
	kinfo(KDEBUG,"numread = %d", numread);
	if (numread == 0)
	  buffer[0] = '\0';
	else
	  {
	    buffer[numread -1] = '\0';
	    (void) kstring_cleanup(buffer, buffer);
	    kinfo(KDEBUG, "buffer == \"%s\"\n", buffer);
	    if (buffer[0] == '\0')
	      break;

	    (void) kstring_upper(buffer, buffer);
	    if (kstrncmp(buffer, yes, index) == 0)
	      {
		value = 1;
		break;
	      }

	    if (kstrncmp(buffer, no, index) == 0)
	      {
		value = 0;
		break;
	      }
	  }

	kinfo(KFORCE, "\n\nInvalid Response - Please enter \"%s\" or \"%s\"\n",
	      yes_response, no_response);
        ksprintf(buffer, "%s (%s/%s) [%s]: ", prompt, yes_response,
		no_response, ((default_val == 1) ? yes_response : no_response));
      }
    
    kfree(prompt);
    kfree(yes);
    kfree(no);
    kclose(tty_fd);
    return(value);
  }

/*-----------------------------------------------------------
|
|  Routine Name: kchoose_default - default handler for kchoose
|
|       Purpose: This routine is the default handler
|		 for the kchoose routine.  If the
|		 value sent to kset_choosehandler is NULL, the
|		 warning handler used is this routine.
|
|         Input: list_of_options - an array of strings containing the items
|                                  to select from.
|                num_options   -  The number of items in the list_of_options
|                default_index - The index number to the default item.
|                message       - grammatically correct, clear explanation of
|                                the error that occurred. This can be formatted
|                                like a (void) printf statement.
|
|        Output: return_string  - string that holds the selected item.
|                                If it's NULL, it kmallocs the space necessary,
|                                and returns the string.
|                return_index  - This is the index of the item selected.
|
|        Returns: TRUE upon success and FALSE upon failure.
|
|    Written By: Tom Sauer
|          Date: Oct 29, 1992 09:46
| Modifications:
|
------------------------------------------------------------*/
static int
kchoose_default(
   char **list_of_options,
   int    num_options,
   int    default_index,
   int   *return_index,
   char **return_string,
   char  *message)
{
   int    tty_fd;
   int    valid_response = FALSE;
   int    response;
   int    numread;
   int    i;
   char  *prompt;
   char   buffer[10*KLENGTH];
   char   buffer1[10*KLENGTH];
   char   buffer2[10*KLENGTH];
   char   temp[10*KLENGTH];
   char   list_num[10*KLENGTH];
   char  *out_buffer;
   char  *temp_message;
   char  *temp_space;


   *return_index = default_index;

   if (( tty_fd = kopen("/dev/tty", KOPEN_RDWR, 0666)) < 0)
   {
      kfprintf(kstderr, "kchoose_prompt: Cannot open channel to terminal\n");
      kfprintf(kstderr, "                Returning default value\n");
      if (*return_string == NULL) 
	 *return_string = kstrdup(list_of_options[*return_index-1]);
      else 
	 kstrcpy(*return_string, list_of_options[*return_index-1]); 
      return(TRUE); 
   }

   prompt = _kformat_tty_message(message, NULL);

   ksprintf(buffer, "%s:\n\t0. Cancel\n", prompt);
   *buffer1 = '\0';
   for (i = 0; i < num_options; i++)
   {
      sprintf(list_num,"%d. ", i+1);
      temp_space = kcalloc(1, (kstrlen(list_num) + 9) * sizeof(char));
      temp_space = (char *) memset(temp_space, ' ', (kstrlen(list_num) + 8));
      temp_message = _kformat_tty_message(list_of_options[i], temp_space);
      ksprintf(temp, "\t%d. %s\n", i+1, temp_message);
      kstrcat(buffer1, temp);
      kfree(temp_message);
      kfree(temp_space);
   }

   ksprintf(buffer2, "  Enter Selection (0 - %d) [%d]: ",
	    num_options, default_index);

   out_buffer = kstring_3cat(buffer, buffer1, buffer2, NULL);

   /*
   * Loop and redisplay the prompt message until a valid response
   * is received
   */

   while ( valid_response == FALSE )
   {
      kwrite(tty_fd, out_buffer, kstrlen(out_buffer));
      if ((numread = kread(tty_fd, buffer, KLENGTH)) == -1)
      {
	 kfprintf(kstderr, "kchoose_default: Cannot read from terminal\n");
	 kfprintf(kstderr, "                Returning default value\n");
	 break;
      }
      kinfo(KDEBUG,"numread = %d", numread);
      if (numread == 0)
	 buffer[0] = '\0';
      else
      {
	 buffer[numread -1] = '\0';
	 (void) kstring_cleanup(buffer, buffer);
	 kinfo(KDEBUG, "buffer == \"%s\"\n", buffer);
	 if (buffer[0] == '\0')
	    break;

	 if ((ksscanf(buffer, " %d", &response)) > 0)
	 {
	    if ( response >= 0 && response <= num_options)
	    {
	       *return_index = response;
	       break;    
	    }
	 }
      }

      kinfo(KFORCE, "\n\nInvalid Response - Please enter a value between 0 "
	    "and %d\n\n",
	    num_options);
   }

   kfree(prompt);
   kfree(out_buffer);
   kclose(tty_fd);

   if (*return_index == 0)
   {
      if (*return_string != NULL)
	 **return_string = '\0';
   }
   else
   {
      if (*return_string == NULL)
	 *return_string = kstrdup(list_of_options[*return_index-1]);
      else
	 kstrcpy(*return_string, list_of_options[*return_index-1]);
   }

   return(TRUE);
}

/************************************************************
*
*  Routine Name: kset_errorhandler - set the error handling routine used by
*				     kerror()
*
*       Purpose: Sets the error handler routine to be used by the kerror()
*		 reporting facility.  When set to NULL, the default, the
*		 different error handler is set the routine will be
*		 called with the following information:
*
*
*	  error_handler(toolbox, program, library, routine, category, message)
*	  ! char *toolbox   - name of toolbox
*         ! char *program   - name of program
*         ! char *library   - name of library
*         ! char *routine   - name of routine 
*         ! char *category  - name of error message category
*	  !                   ($BOOTSTRAP/include/knotify.h)
*         ! char *message   - grammatically correct, clear explanation of
*         !                   the error that occurred
*
*	  Input: new_handler - the error handler to be called.  Specify
*			       xvu_error() for the Khoros pop-up error
*			       message (xvroutines & hybrid routines only),
*			       NULL for the Khoros default error handler which
*			       prints to the tty, or your own error handler.
*        Output:
*       Returns: the previously installed error handler, or NULL
*		 if the default error handler was installed.
*
*  Restrictions: 
*    Written By: Mark Young
*          Date: Jul 16, 1992
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

knotify kset_errorhandler(
   knotify new_handler)
{
	knotify old_handler = error_handler;

	_knotify_initialize();
	error_handler = new_handler;
	return(old_handler);
}

/************************************************************
*
*  Routine Name: kset_warnhandler - set the warn handling routine used by
*				    kwarn()
*
*       Purpose: Sets the warning handler routine to be used by the kwarn()
*		 reporting facility.  When set to NULL, the default, the
*		 different warn handler is set the routine will be
*		 called with the following information:
*
*
*	  warn_handler(toolbox, program, library, routine, message)
*	  ! char *toolbox   - name of toolbox
*         ! char *program   - name of program
*         ! char *library   - name of library
*         ! char *routine   - name of routine 
*         ! char *message   - grammatically correct, clear explanation of
*         !                   the warn that occurred
*
*	  Input: new_handler - the error handler to be called.  Specify
*			       xvu_warn() for the Khoros pop-up error message
*			       (xvroutines & hybrid routines only), NULL for
*			       the Khoros default error handler which prints
*			       to the tty, or your own error handler.
*        Output:
*       Returns: the previously installed error handler, or NULL
*			 if the default error handler was installed.
*
*  Restrictions: 
*    Written By: Mark Young
*          Date: Jan 4, 1994
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

knotify kset_warnhandler(
   knotify new_handler)
{
	knotify old_handler = warn_handler;

	_knotify_initialize();
	warn_handler = new_handler;
	return(old_handler);
}

/****************************************************************
*
*  Routine Name: kset_announcehandler - set the announce handling routine
*				        used by kannounce()
*
*       Purpose: Sets the announce handler routine to be used by the kannounce()
*		 reporting facility.  When set to NULL, the default, the
*		 announce handler normally dumps the report.  If a
*		 different announce handler is set the routine will be
*		 called with the following information:
*
*	  announce_handler(toolbox, program, library, routine, message)
*	  ! char *toolbox   - name of toolbox
*         ! char *program   - name of program
*         ! char *library   - name of library
*         ! char *routine   - name of routine 
*         ! char *message   - grammatically correct, clear explanation of
*         !                   the error that occurred
*
*	  Input: new_handler - the announce handler to be called.
*        Output:
*	Returns: the previously installed announce handler, or NULL
*		          if the default announce handler was installed.
*
*  Restrictions: 
*    Written By: Mark Young
*          Date: Jul 12, 1994
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

knotify kset_announcehandler(
   knotify new_handler)
{
	knotify old_handler = announce_handler;

	_knotify_initialize();
	announce_handler = new_handler;
	return(old_handler);
}


/***************************************************************
*
*  Routine Name: kset_prompthandler - set the prompt handling routine used
*				      by kprompt()
*
*       Purpose: Sets the prompt handler routine to be used by the kprompt()
*		 reporting facility.  When set to NULL, the default, the
*		 prompt handler will communicate through kstderr and kstdin.
*		 If a different prompt handler is set the routine will be
*		 called with the following information:
*
*
*	  prompt_handler(yes_response, no_response, default, message)
*	  ! char *yes_response - string to put in the affirmative field of a
*	  !                      prompt
*	  ! char *no_response  - string to put in the negative field of a prompt
*	  ! char *default      - string to put in the default field of a prompt
*         ! char *message      - grammatically correct, clear explanation of
*         !                      the prompt message.
*
*	  Input: new_handler - the prompt handler to be called.  Specify
*			       xvu_prompt() for the Khoros pop-up prompt
*			       window (xvroutines & hybrid routines only), NULL
*		  	       for the Khoros default prompt handler which
*		               prints to the tty, or your own prompt handler.
*
*        Output:
*       Returns: the previously installed warning handler, or NULL
*		 if the default warning handler was installed.
*
*  Restrictions: 
*    Written By: Steven Jorgensen
*          Date: Jul 29, 1992
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

knotify kset_prompthandler(
   knotify new_handler)
{
	knotify old_handler = prompt_handler;

	_knotify_initialize();
	prompt_handler = new_handler;
	return(old_handler);
}

/***************************************************************
*
*  Routine Name: kset_choosehandler - set the choose handling routine used
*				      by kchoose()
*
*       Purpose: Sets the choose handler routine to be used by the kchoose()
*		 facility.  When set to NULL, the default, the
*		 choose handler will communicate through kstderr and kstdin.
*		 If a different choose handler is set the routine will be
*		 called with the following information:
*
*
*         choose_handler(list_of_options, default_index, return_index,
*         !              return_string, message ) );
*         ! list_of_options - an array of strings containing the items to
*	  !                   select from.
*         ! num_options     - The number of items in the list_of_options
*         ! default_index   - The index number to the default item.
*         ! return_string   - string that holds the selected item.  If it is
*         !                   NULL, it kmallocs the space necessary, and
*	  !                   returns the string.
*         ! return_index    - This is the index of the item selected.
*         ! message         - grammatically correct, clear explanation of
*         !                   the error that occurred. This can be formatted
*         !                   like a (void) printf statement.
*
*	  Input: new_handler - the choose handler to be called.  Specify
*			       xvu_choose() for the Khoros pop-up choice window
*		               (xvroutines & hybrid routines only), NULL for
*			       the Khoros default choice handler which prints
*			       to the tty, or your own choice handler.
*
*        Output:
*       Returns: the previously installed warning handler, or NULL
*		 if the default warning handler was installed.
*
*  Restrictions: 
*    Written By: Tom Sauer
*          Date: Oct 29, 1992
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

knotify kset_choosehandler(
   knotify new_handler)
{
	knotify old_handler = choose_handler;

	_knotify_initialize();
	choose_handler = new_handler;
	return(old_handler);
}

/****************************************************************
*
*  Routine Name: kset_infohandler - set the infomation handling routine
*				    used by kinfo()
*
*       Purpose: Sets the information handler routine to be used by the kinfo()
*		 reporting facility.  When set to NULL, the default, the
*		 information handler normally reports to standard error.  If a
*		 different info handler is set the routine will be
*		 called with the following information:
*
*	  info_handler(toolbox, program, message )
*	  ! char *toolbox   - name of toolbox
*         ! char *program   - name of program
*	  ! char *message   - message to give the user
*
*	  Input: new_handler - the info handler to be called. Specify xvu_info()
*                              for the Khoros pop-up info message (xvroutines & 
*                              hybrid routines only), NULL for the default 
*                              info handler which prints to the tty, or your own
*                              info handler.
*        Output:
*	Returns: the previously installed information handler, or NULL
*		 if the default information handler was installed.
*
*  Restrictions: 
*    Written By: Steven Jorgensen
*          Date: Jul 29, 1992
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

knotify kset_infohandler(
   knotify new_handler)
{
	knotify old_handler = info_handler;

	_knotify_initialize();
	info_handler = new_handler;
	return(old_handler);
}

/************************************************************
*
*  Routine Name: kverror - print error messages in a standardized format
*
*       Purpose: kverror produces standardized error messages for
*                Khoros library routines and applications.  It should be
*		 called in EVERY instance of error messaging by EVERY Khoros
*		 function, subroutine, or main program.
*		 If library is NULL then the program name will be used
*		 as the source of the error.
*
*         Input: library   - name of library
*                routine   - name of routine 
*                format    - grammatically correct, clear explanation of
*                            the error that occurred
*		 list      - an argument list which is used as the inputs to
*			     the format string.
*
*        Output:
*	Returns: TRUE if the error was successfully acknowledge,
*		 otherwise if the message was not acknowledged FALSE
*		 is returned.
*
*  Restrictions: 
*    Written By: Danielle Argiro, Steven Jorgensen, and Mark Young
*          Date: Jul 29, 1992
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int kverror(
   char    *library,
   char    *routine,
   char    *format,
   kva_list args)
{
	char *category = NULL;
	knotify handler;

	_knotify_initialize();
	if (in_use == TRUE)
	   return(TRUE);

	if (errno == 0 && format == NULL)
	{
	   errno = KNULL_PARAMETER;
	   return(kerror("kutils", "kerror", 
	      "You must provide a non-NULL 'message' or 'errno' must be set"));
	}

	/*
	 *  Format the message with the variable arguments...
	 */
	in_use = TRUE;
        if (format != NULL && _kformat_message(format, args, tempstr) == TRUE)
            format = tempstr;

	if (error_handler != NULL)
	   handler = error_handler;
	else
	   handler = kerror_default;

	if (errno != 0)
	   category = kerrno_lookup(errno);

	errno = 0;

	if (!(*handler)( kprog_get_toolbox(), kprog_get_program(), library,
			 routine, category, format ))
	{
	   in_use = FALSE;
	   if ( handler != kerror_default )
	       return(kerror_default(kprog_get_toolbox(), kprog_get_program(),
				     library, routine, category, format ));

	   return(FALSE);
	}
	in_use = FALSE;
	return(TRUE);
}

/************************************************************
*
*  Routine Name: kerror - print error messages in a standardized format
*
*       Purpose: kerror produces standardized error messages for
*                Khoros library routines and applications.  It should be
*		 called in EVERY instance of error messaging by EVERY Khoros
*		 function, subroutine, or main program.
*		 If library is NULL then the program name will be used
*		 as the source of the error.
*
*         Input: library   - name of library (NULL if not applicable)
*                routine   - name of routine 
*                format    - grammatically correct, clear explanation of
*                            the error that occurred.  This can be formatted
*			     like a printf statement
*
*        Output:
*	Returns: TRUE if the error was successfully acknowledge,
*		          otherwise if the message was not acknowledged FALSE
*		          is returned.
*
*  Restrictions: 
*    Written By: Danielle Argiro, Steven Jorgensen, and Mark Young
*          Date: May 24, 1993
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/
int kerror(
   char *library,
   char *routine,
   char *format,
   kvalist)
{
	kva_list args;
	int val;

	kva_start(args, format);
	val = kverror(library, routine, format, args);
	kva_end(args);
	return(val);
}

/************************************************************
*
*  Routine Name: kvwarn - print warning messages in a standardized format
*
*       Purpose: kvwarn produces standardized warning messages for
*                Khoros library routines and applications.  It should be
*		 called in EVERY instance of warning messaging by EVERY Khoros
*		 function, subroutine, or main program.
*		 If library is NULL then the program name will be used
*		 as the source of the warning.
*
*         Input: library   - name of library (NULL if not applicable)
*                routine   - name of routine 
*                format    - grammatically correct, clear explanation of
*                            the warning that occurred
*		 list      - an argument list which is used as the inputs to
*			     the format string.
*
*        Output:
*	Returns: TRUE if the warning was successfully acknowledge,
*		          otherwise if the message was not acknowledged FALSE
*		          is returned.
*
*  Restrictions: 
*    Written By: Danielle Argiro, Steven Jorgensen, and Mark Young
*          Date: Jan 4, 1994
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int kvwarn(
   char    *library,
   char    *routine,
   char    *format,
   kva_list args)
{
	knotify handler;

	_knotify_initialize();
	if (khoros_notify < KSTANDARD || in_use == TRUE)
	   return(TRUE);

	if (format == NULL)
	{
	   errno = KNULL_PARAMETER;
	   return(kerror("kutils", "kwarn", 
	      "You must provide a non-NULL 'message'"));
	}

	/*
	 *  Format the message with the variable arguments...
	 */
	in_use = TRUE;
        if (format != NULL && _kformat_message(format, args, tempstr) == TRUE)
           format = tempstr;

	if (warn_handler != NULL)
	   handler = warn_handler;
	else
	   handler = kwarn_default;

	if (!(*handler)(kprog_get_toolbox(), kprog_get_program(), library,
			 routine, format))
	{
	   in_use = FALSE;
	   if (handler != kwarn_default)
	      return(kwarn_default(kprog_get_toolbox(), kprog_get_program(),
				   library, routine, format));
	   return(FALSE);
	}
	in_use = FALSE;
	return(TRUE);
}

/************************************************************
*
*  Routine Name: kwarn - print warning messages in a standardized format
*
*       Purpose: kwarn produces standardized warning messages for
*                Khoros library routines and applications.  It should be
*		 called in EVERY instance of warning messaging by EVERY Khoros
*		 function, subroutine, or main program.
*		 If library is NULL then the program name will be used
*		 as the source of the warning.
*
*         Input: library   - name of library (NULL if not applicable)
*                routine   - name of routine 
*                format    - grammatically correct, clear explanation of
*                            the warning that occurred.  This can be formatted
*			     like a printf statement
*
*        Output:
*	Returns: TRUE if the error was successfully acknowledge,
*		          otherwise if the message was not acknowledged FALSE
*		          is returned.
*
*  Restrictions: 
*    Written By: Danielle Argiro, Steven Jorgensen, and Mark Young
*          Date: Jan 4, 1994
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int kwarn(
   char *library,
   char *routine,
   char *format,
   kvalist)
{
	kva_list args;
	int val;

	kva_start(args, format);
	val = kvwarn(library, routine, format, args);
	kva_end(args);
	return(val);
}

/************************************************************
*
*  Routine Name: kvannounce - report or announce a message in a standardized
*			      format
*
*       Purpose: kvannounce produces standardized messages for
*                Khoros library routines and applications.  It should be
*		 called when the programmer wants give generalized updates
*		 or reports to the user.  This typically used by the larger
*		 applications, such as cantata, composer, craftsman, etc
*		 to give progress or status reports to the user.
*
*         Input: library   - name of library
*                routine   - name of routine 
*                format    - grammatically correct, clear explanation of
*                            what should be announced to the user
*		 args      - an argument list which is used as the inputs to
*			     the format string.
*	Returns: TRUE if the announcement was successfully delivered,
*		 otherwise if the message was not delivered FALSE is returned.
*
*  Restrictions: 
*    Written By: Danielle Argiro, Steven Jorgensen, and Mark Young
*          Date: Jul 12, 1994
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int kvannounce(
   char    *library,
   char    *routine,
   char    *format,
   kva_list args)
{
	knotify handler;

	_knotify_initialize();
	if (in_use == TRUE)
	   return(TRUE);

	if (format == NULL)
	{
	   errno = KNULL_PARAMETER;
	   return(kerror("kutils", "kannounce", 
	      "You must provide a non-NULL 'message'"));
	}

	/*
	 *  Format the message with the variable arguments...
	 */
	if (format != NULL && _kformat_message(format, args, tempstr) == TRUE)
	   format = tempstr;

	in_use = TRUE;
	if (announce_handler != NULL)
	   handler = announce_handler;
	else
	   handler = kannounce_default;

	if (!(*handler)(kprog_get_toolbox(), kprog_get_program(), library,
			 routine, format))
	{
	   in_use = FALSE;
	   if (handler != kannounce_default)
	      return(kannounce_default(kprog_get_toolbox(), kprog_get_program(),
				   library, routine, format));
	   return(FALSE);
	}
	in_use = FALSE;
	return(TRUE);
}

/************************************************************
*
*  Routine Name: kannounce - report or announce a message in a standardized
*			     format
*
*       Purpose: kannounce produces standardized messages for
*                Khoros library routines and applications.  It should be
*		 called when the programmer wants give generalized updates
*		 or reports to the user.  This typically used by the larger
*		 applications, such as cantata, composer, craftsman, etc
*		 to give progress or status reports to the user.
*
*         Input: library   - name of library
*                routine   - name of routine 
*                format    - grammatically correct, clear explanation of
*                            what should be announced to the user
*	Returns: TRUE if the announcement was successfully delivered,
*		          otherwise if the message was not delivered FALSE
*		          is returned.
*
*  Restrictions: 
*    Written By: Danielle Argiro, Steven Jorgensen, and Mark Young
*          Date: Jul 12, 1994
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int kannounce(
   char *library,
   char *routine,
   char *format,
   kvalist)
{
	kva_list args;
	int val;

	kva_start(args, format);
	val = kvannounce(library, routine, format, args);
	kva_end(args);
	return(val);
}

/************************************************************
*
*  Routine Name: kvprompt - request an acknowledgement from the user
*
*       Purpose: kvprompt will call the specified prompt handler to request
*		 or demand an acknowledgement from the user. This utility
*		 can operate in several different modes. 
*
*		 If the notify_type variable is set to KFORCE, then the
*		 prompt will always appear regardless of the setting
*		 of the environment variable KHOROS_NOTIFY.  
*
*		 If the notify_type variable is set to KSTANDARD and the
*		 user has the environment variable KHOROS_NOTIFY set to
*		 either STANDARD or VERBOSE the prompt will appear.
*
*		 And finally, if the notify_type variable is set to KVERBOSE
*		 and the environment variable KHOROS_NOTIFY set to VERBOSE,
*		 the prompt will appear.
*
*		 Here is a summary table:
*
*		 notify_type = FORCE    always prompt, ignore the setting of 
*		 !			the environment variable KHOROS_NOTIFY
*
*		 notify_type = STANDARD  only prompt when the environment 
*		 !	          	 variable KHOROS_NOTIFY is set to
*		 !			 STANDARD or VERBOSE.
*
*		 notify_type = VERBOSE   only prompt when the environment 
*                !                       variable KHOROS_NOTIFY is set to
*                !                       VERBOSE 
*
*         Input: notify_type  - the notify level specified by the programmer
*			        KFORCE, KSTANDARD, KVERBOSSE
*		 yes_response - name of "yes" response string ("Yes" if NULL)
*		 no_response  - name of "no" response string ("No" if NULL)
*		 default_val  - the default value to list when prompting
*                format       - grammatically correct, clear explanation of
*                               the error that occurred. This can be formatted
*				like a printf statement.
*
*        Output:
*	Returns: TRUE if the prompt was successfully acknowledged, 
*			  otherwise if the message was not acknowledged FALSE 
*			  is returned. In the event and error occurs the
*			  default value is returned.
*  Restrictions: 
*    Written By: Danielle Argiro, Steven Jorgensen, and Mark Young
*          Date: Jul 29, 1992
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int kvprompt(
   int      notify_type,
   char    *yes_response,
   char    *no_response,
   int      default_val,
   char    *format,
   kva_list args)
{
	char *default_yes = "Yes";
	char *default_no = "No";
	char *yes, *no;

	_knotify_initialize();
	if (notify_type < khoros_notify)
	   return(default_val);

	if (format == NULL)
	{
	   errno = KNULL_PARAMETER;
	   return( kerror("kutils", "kprompt", 
             "you must provide a non-null 'message' to kprompt") );
	}

	yes_response = (yes_response == NULL) ? default_yes : yes_response;
	no_response = (no_response == NULL) ? default_no : no_response;

	yes = kstring_upper(yes_response, NULL);
	no = kstring_upper(no_response, NULL);
	if (kstrcmp(yes,no) == 0)
        {
	   errno = KINVALID_PARAMETER;
	   kerror("kutils", "kprompt",
	       "Yes and No response must be unique\n");
	   return(default_val);
	}
	kfree(yes);
	kfree(no);

	/*
	 *  Format the message with the variable arguments...
	 */
	if (_kformat_message(format, args, tempstr) == TRUE)
	   format = tempstr;

	if (prompt_handler == NULL)
	   (void) kset_prompthandler(kprompt_default);

	return((*prompt_handler)(yes_response, no_response, default_val,
                               format));
}

/************************************************************
*
*  Routine Name: kprompt - request an acknowledgement from the user
*
*       Purpose: kprompt will call the specified prompt handler to request
*		 or demand an acknowledgement from the user. This utility
*		 can operate in several different modes. 
*
*		 If the notify_type variable is set to KFORCE, then the
*		 prompt will always appear regardless of the setting
*		 of the environment variable KHOROS_NOTIFY.  
*
*		 If the notify_type variable is set to KSTANDARD and the
*		 user has the environment variable KHOROS_NOTIFY set to
*		 either STANDARD or VERBOSE the prompt will appear.
*
*		 And finally, if the notify_type variable is set to KVERBOSE
*		 and the environment variable KHOROS_NOTIFY set to VERBOSE,
*		 the prompt will appear.
*
*		 Here is a summary table:
*
*		 notify_type = FORCE    always prompt, ignore the setting of 
*		 !			the environment variable KHOROS_NOTIFY
*
*		 notify_type = STANDARD  only prompt when the environment 
*		 !	          	 variable KHOROS_NOTIFY is set to
*		 !			 STANDARD or VERBOSE.
*
*		 notify_type = VERBOSE   only prompt when the environment 
*                !                       variable KHOROS_NOTIFY is set to
*                !                       VERBOSE 
*
*         Input: notify_type  - the notify level specified by the programmer
*			        KFORCE, KSTANDARD, KVERBOSSE
*		 yes_response - name of "yes" response string ("Yes" if NULL)
*		 no_response  - name of "no" response string ("No" if NULL)
*		 default_val  - the default value to list when prompting
*                format       - grammatically correct, clear explanation of
*                               the error that occurred. This can be formatted
*				like a printf statement.
*
*        Output:
*	Returns: TRUE if the prompt was successfully acknowledged, 
*			  otherwise if the message was not acknowledged FALSE 
*			  is returned. In the event and error occurs the
*			  default value is returned.
*
*  Restrictions: 
*
*    Written By: Danielle Argiro, Steven Jorgensen, and Mark Young
*          Date: May 24, 1993
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int kprompt(
   int  notify_type,
   char *yes_response,
   char *no_response,
   int  default_val,
   char *format,
   kvalist)
{
	kva_list args;
	int val;

	kva_start(args, format);
	val = kvprompt(notify_type, yes_response, no_response, default_val,
			format, args);
	kva_end(args);
	return(val);
}

/************************************************************
*
*  Routine Name: koverwrite - request an acknowledgement for overwriting files
*
*       Purpose: koverwrite will call the specified prompt handler 
*	         to request or demand an acknowledgement from the user. 
*		 This utility will as the user if it is ok to overwrite the
*		 file in question, and can operate in several different modes. 
*
*		 If the notify_type variable is set to KFORCE, then the
*		 prompt will always appear regardless of the setting
*		 of the environment variable KHOROS_NOTIFY.  
*
*		 If the notify_type variable is set to KSTANDARD and the
*		 user has the environment variable KHOROS_NOTIFY set to
*		 either STANDARD or VERBOSE the prompt will appear.
*
*		 And finally, if the notify_type variable is set to KVERBOSE
*		 and the environment variable KHOROS_NOTIFY set to VERBOSE,
*		 the prompt will appear.
*
*		 Here is a summary table:
*
*		 notify_type = FORCE     always prompt, ignore the setting of 
*		 !                       the environment variable KHOROS_NOTIFY
*
*		 notify_type = STANDARD  only prompt when the environment 
*		 !                       variable KHOROS_NOTIFY is set to
*		 !                       STANDARD or VERBOSE.
*
*		 notify_type = VERBOSE   only prompt when the environment 
*                !                       variable KHOROS_NOTIFY is set to
*                !                       VERBOSE 
*
*         Input: notify_type  - the notify level specified by the programmer
*			        KFORCE, KSTANDARD, KVERBOSSE
*                filename     - the filename in question to overwrite
*
*        Output:
*	Returns: TRUE if the prompt was successfully acknowledged, 
*		 otherwise if the message was not acknowledged FALSE 
*		 is returned. In the event of an error TRUE
*		 is returned. If the file does not exist, 
*		 TRUE is returned.
*  Restrictions: 
*    Written By: Tom Sauer
*          Date: Nov 03, 1992
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int koverwrite(
   int  notify_type,
   char *filename)
{
	if (kstrcmp(filename, "-") != 0 && kstrcmp(filename, "#") != 0 &&
	    kaccess(filename, R_OK) == 0)
	{
	   return(kprompt(notify_type, "Yes", "No", 1, "Overwrite '%s'?",
			filename));
	}
	else
	   return(TRUE);
}

/************************************************************
*
*  Routine Name: kvchoose - prompt the user to select from a list of items
*
*       Purpose: kvchoose will call the specified choose handler to request
*                the user to make a selection from a list of items. 
*                This utility can operate in several different modes.
*
*                If the notify_type variable is set to KFORCE, then the
*                prompt will always appear regardless of the setting
*                of the environment variable KHOROS_NOTIFY.
*
*                If the notify_type variable is set to KSTANDARD and the
*                user has the environment variable KHOROS_NOTIFY set to
*                either STANDARD or VERBOSE the prompt will appear.
*
*                And finally, if the notify_type variable is set to KVERBOSE
*                and the environment variable KHOROS_NOTIFY set to VERBOSE,
*                the prompt will appear.
*
*                Here is a summary table:
*
*                notify_type = FORCE     always prompt, ignore the setting of
*                !                       the environment variable KHOROS_NOTIFY
*
*                notify_type = STANDARD  only prompt when the environment
*                !                       variable KHOROS_NOTIFY is set to
*                !                       STANDARD or VERBOSE.
*
*                notify_type = VERBOSE   only prompt when the environment
*                !                       variable KHOROS_NOTIFY is set to
*
*         Input: notify_type  - the notify level specified by the programmer
*                               KFORCE, KSTANDARD, KVERBOSSE
*                list_of_options - an array of strings containing the items
*                                  to select from.
*                num_options   - The number of items in the list_of_options.
*                default_index - The index number to the default item, 
*                                must start at 1.
*                format       - grammatically correct, clear explanation of
*                               the error that occurred. This can be formatted
*                               like a printf statement.
*		 args         - an argument list which is used as the inputs
*				to the format string.
*        Output: return_string - string that holds the selected item.
*                                If it's NULL, it kmallocs the space necessary,
*                                and returns the string.
*                return_index - This is the index of the item selected.
*       Returns: return_string if it is not NULL, or a pointer to the
*                resulting kmalloc'ed string if it is NULL.  
*                NULL is returned upon error.
*  Restrictions: 
*    Written By: Tom Sauer
*          Date: Oct 28, 1992
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

char *kvchoose(
   int      notify_type,
   char   **list_of_options,
   int      num_options,
   int      default_index,
   char    *return_string,
   int     *return_index,
   char    *format,
   kva_list args)
{
	_knotify_initialize();

	if (return_index == NULL)
	{
	    kerror("kutils", "kvchoose", 
		   "*return_index argument may not be passed in as NULL"); 
	    return(NULL);
	}
	*return_index = default_index;

	if (list_of_options == NULL || num_options <= 0)
	{
	   errno = KNULL_PARAMETER;
	   kerror("kutils", "kchoose", 
             "you must provide a non-null 'list_of_options' to kchoose, or \
more than one option to choose from");
	   return( NULL);
	}

	if (notify_type < khoros_notify)
	{
	   if (return_string == NULL) 
              return(kstrdup(list_of_options[*return_index-1])); 
           else 
           { 
              kstrcpy(return_string, list_of_options[*return_index-1]); 
              return(return_string); 
           } 
        }

	if (format == NULL)
	{
	   errno = KNULL_PARAMETER;
	   kerror("kutils", "kchoose", 
               "you must provide a non-null 'format' to kchoose");
	   return(NULL);
        }

        if (default_index < 1)
	   default_index = 1;

	/*
	 *  Format the message with the variable arguments...
	 */
	if (_kformat_message(format, args, tempstr) == TRUE)
	   format = tempstr;

	if (choose_handler == NULL)
	   (void) kset_choosehandler(kchoose_default);

	(*choose_handler)(list_of_options, num_options, default_index,
			return_index, &return_string, format);

	return(return_string);
}

/************************************************************
*
*  Routine Name: kchoose - prompt the user to select from a list of items
*
*       Purpose: kchoose will call the specified choose handler to request
*                the user to make a selection from a list of items. 
*                This utility can operate in several different modes.
*
*                If the notify_type variable is set to KFORCE, then the
*                prompt will always appear regardless of the setting
*                of the environment variable KHOROS_NOTIFY.
*
*                If the notify_type variable is set to KSTANDARD and the
*                user has the environment variable KHOROS_NOTIFY set to
*                either KSTANDARD or KVERBOSE the prompt will appear.
*
*                And finally, if the notify_type variable is set to KVERBOSE
*                and the environment variable KHOROS_NOTIFY set to KVERBOSE,
*                the prompt will appear.
*
*                Here is a summary table:
*
*                notify_type = FORCE     always prompt, ignore the setting of
*                !                       the environment variable KHOROS_NOTIFY
*
*                notify_type = STANDARD  only prompt when the environment
*                !                       variable KHOROS_NOTIFY is set to
*                !                       STANDARD or VERBOSE.
*
*                notify_type = VERBOSE   only prompt when the environment
*                !                       variable KHOROS_NOTIFY is set to
*
*         Input: notify_type  - the notify level specified by the programmer
*                               KFORCE, KSTANDARD, KVERBOSSE
*                list_of_options - an array of strings containing the items
*                                  to select from.
*                num_options   - The number of items in the list_of_options.
*                default_index - The index number to the default item, 
*                                must start at 1.
*                format        - grammatically correct, clear explanation of
*                               the error that occurred. This can be formatted
*                               like a (void) printf statement.
*
*        Output: return_string - string that holds the selected item.
*                                If it's NULL, it kmallocs the space necessary,
*                                and returns the string.
*                return_index - This is the index of the item selected.
*
*       Returns: return_string if it is not NULL, or a pointer to the
*                resulting kmalloc'ed string if it is NULL.  
*                NULL is returned upon error.
*  Restrictions: 
*    Written By: Danielle Argiro, Steven Jorgensen, and Mark Young
*          Date: May 24, 1993
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

char *kchoose(
   int    notify_type,
   char **list_of_options,
   int    num_options,
   int    default_index,
   char  *return_string,
   int   *return_index,
   char  *format,
   kvalist)
{
	kva_list args;
	char *choose;

	kva_start(args, format);
	choose = kvchoose(notify_type, list_of_options, num_options,
			default_index, return_string, return_index, format,
			args);
	kva_end(args);
	return(choose);
}

/************************************************************
*
*  Routine Name: kvinfo - print information messages in a standardized format
*
*       Purpose: kvinfo produces standardized information messages for
*                Khoros library routines and Khoros applications.  It should
*		 be called in EVERY instance of information messaging by EVERY
*		 Khoros function, subroutine, or main program.
*
*		 If the notify_type variable is set to KFORCE, then the
*		 prompt will always appear regardless of the setting
*		 of the environment variable KHOROS_NOTIFY.  
*
*		 If the notify_type variable is set to KSTANDARD and the
*		 user has the environment variable KHOROS_NOTIFY set to
*		 either STANDARD or VERBOSE the prompt will appear.
*
*		 And finally, if the notify_type variable is set to KVERBOSE
*		 and the environment variable KHOROS_NOTIFY set to VERBOSE,
*		 the prompt will appear.
*
*		 Here is a summary table:
*
*		 notify_type = FORCE     always prompt, ignore the setting of 
*		 !                       the environment variable KHOROS_NOTIFY
*
*		 notify_type = STANDARD  only prompt when the environment 
*		 !                       variable KHOROS_NOTIFY is set to
*		 !                       STANDARD or VERBOSE.
*
*		 notify_type = VERBOSE   only prompt when the environment 
*                !                       variable KHOROS_NOTIFY is set to
*                !                       VERBOSE 
*
*         Input: notify_type - the notify level specified by the programmer
*			       KFORCE, KSTANDARD, KVERBOSE
*                format      - grammatically correct, clear explanation of
*                              the information. Note the message can be
*			       formatted like a printf statement.
*		 args        - an argument list which is used as the inputs
*			       to the format string
*        Output:
*	Returns: TRUE(1) if the message was successfully printed,
*		 otherwise FALSE(0) is returned.
*
*  Restrictions: 
*    Written By: Danielle Argiro, Steven Jorgensen, and Mark Young
*          Date: Jul 29, 1992
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int kvinfo(
   int  notify_type,
   char *format,
   kva_list args)
{
	knotify handler;

	_knotify_initialize();
	if (notify_type != KFORCE && (!khoros_debug || notify_type != KDEBUG)
	    && notify_type < khoros_notify || in_use == TRUE)
	{
	   return(TRUE);
	}

	if (format == NULL)
	{
	   errno = KNULL_PARAMETER;
	   return(kerror("kutils", "kinfo", 
             "you must provide a non-null 'message' to kinfo"));
	}

	/*
	 *  Format the message with the variable arguments...
	 */
	in_use = TRUE;
        if (_kformat_message(format, args, tempstr) == TRUE)
           format = tempstr;

        if (notify_type != KDEBUG && info_handler != NULL)
	   handler = info_handler;
	else
	   handler = kinfo_default;

	if (!(*handler)(format))
	{
	   in_use = FALSE;
           if (handler != kinfo_default)
	     return(kinfo_default(format));

	   return(FALSE);
	}
	in_use = FALSE;
	return(TRUE);
}

/************************************************************
*
*  Routine Name: kinfo - print information messages in a standardized format
*
*       Purpose: kinfo produces standardized information messages for
*                Khoros library routines and Khoros applications.  It should
*		 be called in EVERY instance of information messaging by EVERY
*		 Khoros function, subroutine, or main program.
*
*		 If the notify_type variable is set to KFORCE, then the
*		 prompt will always appear regardless of the setting
*		 of the environment variable KHOROS_NOTIFY.  
*
*		 If the notify_type variable is set to KSTANDARD and the
*		 user has the environment variable KHOROS_NOTIFY set to
*		 either STANDARD or VERBOSE the prompt will appear.
*
*		 And finally, if the notify_type variable is set to KVERBOSE
*		 and the environment variable KHOROS_NOTIFY set to VERBOSE,
*		 the prompt will appear.
*
*		 Here is a summary table:
*
*		 notify_type = FORCE     always prompt, ignore the setting of 
*		 !                       the environment variable KHOROS_NOTIFY
*
*		 notify_type = STANDARD  only prompt when the environment 
*		 !                       variable KHOROS_NOTIFY is set to
*		 !                       STANDARD or VERBOSE.
*
*		 notify_type = VERBOSE   only prompt when the environment 
*                !                       variable KHOROS_NOTIFY is set to
*                !                       VERBOSE 
*
*         Input: notify_type  - the notify level specified by the programmer
*			        KFORCE, KSTANDARD, KVERBOSE
*                format   - grammatically correct, clear explanation of
*                            the information. Note the message can be formatted
*			     like a printf statement.
*
*        Output:
*	Returns: TRUE if the message was successfully printed,
*		 	  otherwise FALSE is returned.
*
*  Restrictions: 
*    Written By: Danielle Argiro, Steven Jorgensen, and Mark Young
*          Date: May 24, 1993
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int kinfo(
   int  notify_type,
   char *format,
   kvalist)
{
	kva_list args;
	int val;

	kva_start(args, format);
	val = kvinfo(notify_type, format, args);
	kva_end(args);
	return(val);
}

/************************************************************
*
*  Routine Name: kset_notify - set the Khoros notify level
*
*       Purpose: This routine sets the notify level.  The possible values
*		 are KFORCE, KSTANDARD, KVERBOSE, KXVLIB, KSYSLIB, KHOSTILE,
*		 and KDEBUG.
*
*         Input: notify_type  - the notify level specified by the programmer
*			        KFORCE, KSTANDARD, KVERBOSE, KXVLIB, KSYSLIB,
*				KHOSTILE, KDEBUG
*        Output:
*	Returns: the current notify value before overriding it with the new
*		 value
*  Restrictions: 
*    Written By: Mark Young
*          Date: Dec 5, 1993
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int kset_notify(
   int  notify_type)
{
	int val;

	_knotify_initialize();
	val = khoros_notify;
	khoros_notify = notify_type;

	return(val);
}

/************************************************************
*
*  Routine Name: kget_notify - get the Khoros notify level
*
*       Purpose: This routine gets the notify level.  The possible values
*		 are KFORCE, KSTANDARD, KVERBOSE, KXVLIB, KSYSLIB, KHOSTILE,
*		 and KDEBUG.
*
*         Input:
*        Output:
*	Returns: the current notify value
*  Restrictions: 
*    Written By: Mark Young
*          Date: Dec 5, 1993
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int kget_notify(void)
{
	_knotify_initialize();
	return(khoros_notify);
}
