/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>         General Khoros Transport Routines
   >>>>
   >>>>  Private:
   >>>>
   >>>>   Public:
   >>>>			 ktempnam()
   >>>>			 krename()
   >>>>			 kopen()
   >>>>			 kclose()
   >>>>			 kread()
   >>>>			 kwrite()
   >>>>			 kaccess()
   >>>>			 klseek()
   >>>>			 ktell()
   >>>>			 kdup()
   >>>>			 kdup2()
   >>>>			 kunlink()
   >>>>			 kcreat()
   >>>>			 ktouch()
   >>>>			 kflock()
   >>>>
   >>>>			 kpopen()
   >>>>			 kpclose()
   >>>>			 kpinfo()
   >>>>
   >>>>			 ksystem()
   >>>>			 kspawn()
   >>>>			 kexecvp()
   >>>>			 kfileno()
   >>>>			 kexit()
   >>>>			 kmakecommand
   >>>>
   >>>>			 kgethostname()
   >>>>			 kgetdescriptors()
   >>>>			 kgetarchitecture()
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"


/**************************************************************
*  
*  Routine Name: ktempnam - create a name for a temporary khoros transport
*       Purpose: This module is used to create temporary files.
*		 The application program calls "ktempnam"
*	         with a template filename which is used to help
*	         identify the application routine.  The type of
*	         temporary created depends on the identifier.  If
*	         a unique temporary shared memory key is desired
*	         then the template would look like:
*  
*  		 !       "shm=XXXX"
*  
*	         If a file was desired then either
*  
*  		 !       "XXXXXXXXX"
*  		 !       "file=XXXX"
*  
*	         would work.  The convention should be as follows,
*	         an identifier followed by an "=" equals indicator,
*	         and finally followed by an optional template.
*         Input: machine  - optional machine name in which to create
*		            the template file.
*  		 template - template of the file to be created.
*       Returns: a unique temporary filename on success, NULL otherwise
*    Written By: Mark Young  
*          Date: Jul 08, 1992 16:24
* Modifications:
*   Declaration: char *ktempnam(
*		 !   char *machine,
*		 !   char *template)
**************************************************************/
/*ARGSUSED*/
char *ktempnam(
   char *machine,
   char *template)
{
	TransportInformation *routines;
	char	*identifier, *filename, itemp[KLENGTH], ftemp[KLENGTH],
		result[KLENGTH];


	/*
	 *  According to the transport routines
	 */
	identifier = ktransport_identifier(template, itemp);
	if ((routines = ktransport_routines(identifier)) == NULL)
	   return(NULL);

	/*
	 *  Get the filename component
	 */
	if ((filename = ktransport_token(template, ftemp)) == NULL)
	   return(NULL);

	if (routines->tempnam == NULL)
	   return(NULL);

	if (routines->tempnam(identifier, filename, result) == -1)
	   return(NULL);
	else
	   return(kstrdup(result));
}


/**************************************************************
*  
*  Routine Name: krename - rename a khoros transport from path1 to path2
*       Purpose: krename() is used to move the contents from one
*		 file to another.  If the two files are standard
*		 UNIX files then krename() simply use the system
*		 rename(), but if this fails then the khoros transport
*		 mechanisms are used to copy the data from the
*		 first file to the second, and then the first
*		 is unlinked using kunlink().
*         Input: path1 - the old khoros transport name
*  		 path2 - the new khoros transport name
*	Returns: 0 on success, -1 on failure and sets errno to
*		 indicate the error
*    Written By: Mark Young  
*          Date: Jan 20, 1993 21:16
**************************************************************/

int krename(
   char *path1,
   char *path2)
{
	int   status;
	kfile *file1, *file2;
	char  name1[KLENGTH], name2[KLENGTH];


	/*
	 *  kfullpath() is used to expand the path before opening it...
	 */
	if (!kfullpath(path1, NULL, name1) ||
	    !kfullpath(path2, NULL, name2))
	{
	   errno = ENOTDIR;
	   return(-1);
	}

	/*
	 *  First see if we can't rename() path1 to path2 using
	 *  the system rename function.
	 */
	if (rename(name1, name2) == 0)
	   return(0);
	else
	   errno = 0;

	/*
	 *  Since that didn't work then try and copy the data from
	 *  the first file to the second.
	 */
	if ((file1 = kfile_open(name1, KOPEN_RDONLY, 0666)) == NULL)
	   return(-1);

	if ((file2 = kfile_open(name2, KOPEN_WRONLY|KOPEN_CREAT|KOPEN_TRUNC,
				0666)) == NULL)
	{
	   kfile_close(file1);
	   return(-1);
	}

	status = kfile_copydata(kfileno(file1), kfileno(file2), 0);
	(void) kfile_close(file1);
	(void) kfile_close(file2);

	/*
	 *  If we have successfully transfered the contents from file1
	 *  to file2, then go ahead and unlink file1.
	 */
	if (status != -1)
	{
	   kunlink(name1);
	   return(0);
	}
	else
	{
	   kunlink(name2);
	   return(-1);
	}
}


/**************************************************************
*  
*  Routine Name: kopen - open or create a file for reading and/or writing
*       Purpose: This function is a replacement for the system "open"
*	         call.  The only difference is that kopen() supports
*	         more than just files, it supports other data transports
*	         as well.  The path should be in the form of an identifier,
*	         followed by an "=" equal indicator, and then the transport
*	         token. ie) a Shared memory path would look like:
*  
*  		 !       "shm=1032"
*  
*	         If a file was desired then either
*  
*  		 !      "/usr/tmp/vadd1235"
*  		 !      "file=/usr/tmp/vadd1235"
*
*		 will work.
*         Input: path - is the string containing the path name to the
*		        desired file to be open.  The path name identifies
*		        the file to be opened, just like the regular UNIX
*		        open() command.
*	        flags - how the file is to be opened.
*  
*  		!	KOPEN_RDONLY - Open file for only reading
*  		!	KOPEN_WRONLY - Open file for only writing
*  		!	KOPEN_RDWR   - Open file for both reading & writing
*               !
*  		!	KOPEN_NONBLOCK - Whether to block when reading and
*  		!		         writing to the file.
*  		!	KOPEN_APPEND - Append to the file (writing).
*  		!	KOPEN_TRUNC  - Truncate the file for writing.
*  		!	KOPEN_EXCL   - Error if file exists
*  		!	KOPEN_CREAT  - the file exists, this flag has no effect.
*		!                   Otherwise, the file is created, and the
*		!		   owner ID of the file is set to the effective
*		!		   user ID of the process.
*  
*	         mode - the permissions to be used when creating a new
*		        file.
*       Returns: the kfile id on success, -1 otherwise
*    Written By: Mark Young  
*          Date: Jul 08, 1992 16:24
* Modifications:
*  
**************************************************************/

int kopen(
   char *path,
   int  flags,
   int  mode)
{
	kfile *file;

	if ((file = kfile_open(path, flags, mode)) == NULL)
	{
	   return(-1);
	}
	return(file->id);
}


/**************************************************************
*  
*  Routine Name: kclose - close and delete a transport descriptor
*  
*  Purpose: This function is a replacement for the system "close"
*	   call.  The only difference is that kclose() supports
*	   more than just files, it supports other data transports
*	   as well, such as shared memory, pipes, files, etc.
*  
*  
*    Input: id - the file to be close which was opened earlier
*		 with kopen().
*  
*   Output: none
*
*  Returns: -1 or 0 depending on whether the operation failed or succeeded.
*  
*  Restrictions:
*    Written By: Mark Young  
*          Date: Jul 08, 1992 16:24
*      Verified:
*  Side Effects:
* Modifications:
*  
**************************************************************/

int kclose(
   int id)
{
	kfile	*file;


	/*
	 *  Get the file entry associated with the id
	 */
	if ((file = kfile_checkid(id)) == NULL)
	   return(-1);

	return(kfile_close(file));
}


/**************************************************************
*  
*  Routine Name: kread - read input from a transport descriptor
*  
*  Purpose: This function is a replacement for the system "read"
*	   call.  The only difference is that kread() supports
*	   more than just files, it supports other data transports
*	   as well, such as shared memory, pipes, files, etc.
*  
*	   The routine will read nbytes into the character array
*	   "buf".  If not all nbytes can be read then the kread()
*	   routine returns the number of bytes actually read.
*  
*  
*   Input: id - the file id to be read which was opened earlier
*		 with kopen().
*  
*	    buf - the buffer to read the data into
*  
*	    nbytes - the number of bytes to be read
*  
*   Output: none
*
*  Returns: the number of bytes read, 0 when end
*	    of file is encountered, or -1 when an error
*	    is encountered.
*
*  Restrictions:
*    Written By: Mark Young  
*          Date: Jul 08, 1992 16:24
*      Verified:
*  Side Effects:
* Modifications:
*  
**************************************************************/

int kread(
   int   id,
   char  *buf,
   int   nbytes)
{
	int	num;
	kfile	*file;


	/*
	 *  Get the file entry associated with the id
	 */
	if ((file = kfile_checkid(id)) == NULL)
	   return(-1);

	/*
	 *  Call the "read()" routine to read from the transport
	 */
	num = kfile_read(file, buf, nbytes);
	return(num);
}


/**************************************************************
*  
*  Routine Name: kwrite - write output to a transport descriptor
*  
*  Purpose: This function is a replacement for the system "write"
*	   call.  The only difference is that kwrite() supports
*	   more than just files, it supports other data transports
*	   as well, such as shared memory, pipes, files, etc.
*  
*	   The routine will write nbytes from the character array
*	   "buf" to the appropriate transport mechanism specified by
*	   id.  If not all nbytes can be read then the kread()
*	   routine returns the number of bytes actually read.
*  
*  
*   Input: id - the file id to be read which was opened earlier
*		 with kopen().
*  
*	    buf - the buffer to write the data from
*  
*	    nbytes - the number of bytes to be written
*  
*  Output:  none
*
*  Returns: The number of bytes written, or -1 when
*	    an error is encountered.
*  
*  Restrictions:
*    Written By: Mark Young  
*          Date: Jul 08, 1992 16:24
*      Verified:
*  Side Effects:
* Modifications:
*  
**************************************************************/

int kwrite(
   int   id,
   char  *buf,
   int   nbytes)
{
	int	num;
	kfile	*file;


	/*
	 *  Get the file entry associated with the id
	 */
	if ((file = kfile_checkid(id)) == NULL)
	   return(-1);

	/*
	 *  Call the "write()" routine to write to the transport
	 */
	num = kfile_write(file, buf, nbytes);
	return(num);
}


/**************************************************************
*  
*  Routine Name: kaccess - determine accessibility of file
*  
*  Purpose: This function is a replacement for the system "access"
*	   call.  The only difference is that kaccess() checks
*	   more than just files, it will also check other data
*	   transports as well, such as shared memory, pipes, files,
*	   etc.
*  
*	   The routine will check to see if the token specified by
*	   the transport is either readable (R_OK), writeable (W_OK),
*	   or executable (X_OK), or accessible (F_OK) any possible
*	   combination by or'ing the flags together.
*  
*  Input: path - is the string containing the path name to the
*		   desired file to be accessed.  The path name ident-
*		   ifies the file/token to be tested, just like the
*		   regular UNIX access() command.
*  
*	   mode - the mode in which kaccess() should test for the
*		   file.  The mode is comprised of an or'ing of the
*		   following flags:
*  
*  	   !	R_OK - the file readable
*  	   !	W_OK - the file writeable
*  	   !	X_OK - the file executable
*  	   !	F_OK - the file and directory leading to the
*  	   !	       file accessible.
*  
*   Output: none
*
*  Returns: return -1 for failure otherwise 0 for success
*  
*  Restrictions:
*    Written By: Mark Young  
*          Date: Jul 08, 1992 16:24
*      Verified:
*  Side Effects:
* Modifications:
*  
**************************************************************/

int kaccess(
   char *path,
   int  mode)
{
	TransportInformation *routines;

	int	check;
	char	*identifier, *filename, name[KLENGTH], temp[KLENGTH];


	/*
	 *  kfullpath() is used to expand the path before opening it...
	 */
	if (!kfullpath(path, NULL, name))
	   return(-1);

	/*
	 *  According to the transport routines
	 */
	identifier = ktransport_identifier(name, temp);
	if ((routines = ktransport_routines(identifier)) == NULL)
	   return(-1);

	/*
	 *  Get the filename component
	 */
	if ((filename = ktransport_token(name, temp)) == NULL)
	   return(-1);

	if (routines->access != NULL)
	   check = routines->access(name, filename, mode);
	else if (routines->open == NULL)
	   check = access(filename, mode);
	else
	   check = access(filename, mode);

	if (check == -1) errno = ENOENT;
	return(check);
}


/**************************************************************
*  
*  Routine Name: kunlink - remove a filename from a directory entry
*  
*  Purpose: This function is used to unlink the kfile.  Depending
*	   on the different transport being used the behavior
*	   will be different.  For a file it unlinks the file, but
*	   for a shared memory segment it initializes it before
*	   using the shmctl() to destroy it.
*  
*  
*   Input:  path - the path to the object ot be unlinked.
*  
*   Output: none
*
*  Returns: -1 or 0 depending on whether the
*	    operation failed or succeeded.
*  
*  Restrictions:
*    Written By: Mark Young  
*          Date: Jul 08, 1992 16:24
*      Verified:
*  Side Effects:
* Modifications:
*  
**************************************************************/

int kunlink(
   char *path)
{
	TransportInformation *routines;
	char	*identifier, *filename, name[KLENGTH], itemp[KLENGTH],
		ftemp[KLENGTH];


	/*
	 *  kfullpath() is used to expand the path before opening it...
	 */
	if (!kfullpath(path, NULL, name))
	   return(FALSE);

	/*
	 *  According to the transport routines
	 */
	identifier = ktransport_identifier(name, itemp);
	if ((routines = ktransport_routines(identifier)) == NULL)
	   return(-1);

	/*
	 *  Get the filename component
	 */
	if ((filename = ktransport_token(name, ftemp)) == NULL)
	   return(-1);

	if (routines->unlink == NULL)
	   return(unlink(filename));
	else
	   return(routines->unlink(identifier, filename));
}

/************************************************************
*
*  Routine Name: kcreat - routine for creating a khoros transport
*
*       Purpose: This function is used to create a khoros transport
*		 file.  Depending on the different transport being
*		 used the behavior will be different.  For a file the
*		 kcreat call simply create the file with the mode,
*		 
*         Input: path - path to file to be created
*                mode - the permissions or mode in which to create the file
*        Output:
*       Returns: open fid on success, -1 otherwise
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Oct 23, 1993
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int kcreat(
   char *path,
   int  mode)
{
	int  fid;
	char name[KLENGTH];


	/*
	 *  kfullpath() is used to expand the path before opening it...
	 */
	if (!kfullpath(path, NULL, name))
	   return(-1);

	if ((fid = kopen(name, KOPEN_WRONLY|KOPEN_CREAT|KOPEN_TRUNC, mode)) == -1)
	   return(-1);

	return(fid);
}

/************************************************************
*
*  Routine Name: ktouch - routine for touching a temporary transport
*
*       Purpose: This function is used to create a khoros transport
*		 file.  Depending on the different transport being
*		 used the behavior will be different.  For a file the
*		 ktouch call simply open the file and then close
*		 it.  This is analgous to the "touch" system call.
*		 
*         Input: path - path to file to be created
*                mode - the permissions or mode in which to touch the file
*        Output:
*       Returns: 0 on success, -1 otherwise
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Apr 27, 1994
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int ktouch(
   char *path,
   int  mode)
{
	int  fid;
	char name[KLENGTH];


	/*
	 *  kfullpath() is used to expand the path before opening it...
	 */
	if (!kfullpath(path, NULL, name))
	   return(-1);

	if ((fid = kopen(name, KOPEN_WRONLY|KOPEN_CREAT, mode)) == -1)
	   return(-1);

	return(kclose(fid));
}


/**************************************************************
*  
*  Routine Name: kflock - apply or remove an advisory lock on an open transport
*			  descriptofiler
*  
*  Purpose: This function is used to lock the transport. Depending
*	   on the different transport being used the behavior
*	   will be different.  For a file it calls flock() to
*	   lock the file, but for a shared memory segment it
*	   it calls shmctl() to lock & unlock the shared memory
*	   segment.
*  
*  
*  Input: id	 - the id of the object to be flock.
*  
*	  operation - the mode in which kflock() should lock the
*		   file.  The operation is comprised of an or'ing of the
*		   following flags:
*  
*  	  !	KLOCK_SH - lock file shareable (when reading)
*  	  !	KLOCK_EX - lock file execlusive (when writing)
*  	  !	KLOCK_NB - lock file for no block (don't block)
*  	  !	KLOCK_UN - unlock the file (free previous lock)
*  
*  Output: none
*
* Returns: -1 or 0 depending on whether the
*	   operation failed or succeeded.
*  
*  Restrictions:
*    Written By: Mark Young  
*          Date: Jul 08, 1992 16:24
*      Verified:
*  Side Effects:
* Modifications:
*  
**************************************************************/

int kflock(
   int id,
   int operation)
{
	kfile	*file;
	TransportInformation *routines;


	/*
	 *  Get the file entry associated with the id
	 */
	if ((file = kfile_checkid(id)) == NULL)
	   return(-1);

	/*
	 *  Call the "lock()" routine to close the transport
	 */
	routines = file->routines;
	if (routines->lock != NULL)
	   return(routines->lock(file, operation));

	return(0);
}


/**************************************************************
*  
*  Routine Name: klseek - move read/write pointer of a transport descriptor
*  
*  Purpose: This function is used to seek the transport. Depending
*	   on the different transport being used the behavior
*	   will be different.  For a file it calls lseek() to
*	   lock the file, but for a shared memory segment it
*	   it simply resets the internal offset pointer.
*  
*  
*  Input:	id	 - the id of the object to be seeked.
*  		offset - the offset in which to seek
*  		whence - the control of how the offset will be applied
*  
*  
*   Output: none
*
*  Returns: -1 or the new seeked position within the transport
*  
*  Restrictions:
*    Written By: Mark Young  
*          Date: Jul 08, 1992 16:24
*      Verified:
*  Side Effects:
* Modifications:
*  
**************************************************************/

int klseek(
   int id,
   int offset,
   int whence)
{
	kfile	*file;


	/*
	 *  Get the file entry associated with the id
	 */
	if ((file = kfile_checkid(id)) == NULL)
	   return(-1);

	/*
	 *  Call the "kfile_seek()" routine to seek on the transport
	 */
	return(kfile_seek(file, offset, whence));
}


/**************************************************************
*  
*  Routine Name: ktell - report the position of the read/write pointer
*  
*  Purpose: This function is used to tell the current position
*	   within the transport. Depending on the different
*	   transport being used the behavior will be different.
*	   For a file it calls tell() to locate the offset
*	   within the file, but for a shared memory segment it
*	   it simply indicates the internal offset.
*  
*  
*   Input:  id	 - the id of the object to be told.
*  
*   Output: none
*
*  Returns: -1 or the current offset depending on whether the
*	    operation failed or succeeded.
*  
*  Restrictions:
*    Written By: Mark Young  
*          Date: Jul 08, 1992 16:24
*      Verified:
*  Side Effects:
* Modifications:
*  
**************************************************************/

int ktell(
   int id)
{
	kfile	*file;


	/*
	 *  Get the file entry associated with the id
	 */
	if ((file = kfile_checkid(id)) == NULL)
	   return(-1);

	/*
	 *  Call the "kfile_tell()" routine to tell on the transport
	 */
	return(kfile_tell(file));
}


/************************************************************
*
*  Routine Name: kdup - duplicate an existing khoros transport descriptor
*
*       Purpose: kdup() is used to duplicate an existing khoros transport
*		 descriptor.  The "id" is integer index in the process's
*		 transport descriptor table.  The new descriptor returned
*		 kdup() will be the lowest table entry id.  
*
*         Input: id - the existing khoros transport descriptor to be dup'ed
*        Output: 
*       Returns: the newly dup'ed descriptor on success, or -1 upon failure
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Apr 28, 1993 22:13
*      Verified:
*  Side Effects: If too many descriptors are active then errno will be
*		 set to EMFILE and -1 will be returned.  If id is not
*		 a valid or active descriptor then errno will be set to
*		 EBADF and -1 will be returned.
* Modifications:
*
*************************************************************/

int kdup(
   int id)
{
	int newid;


	/*
	 *  Get the file entry associated with the id
	 */
	if (!kfile_checkid(id))
	   return(-1);

	newid = kfile_dup(id, -1);
	return(newid);
}


/************************************************************
*
*  Routine Name: kdup2 - specifically duplicate an existing khoros transport
*			 descriptor
*
*       Purpose: kdup2() is used to duplicate an existing khoros transport
*		 descriptor.  The "id1" is integer index in the process's
*		 transport descriptor table.  The new descriptor returned
*		 will the specific one specified by "id2".  If id2 is active
*		 then it will be closed (kclose) before being dup'ed.
*
*         Input: id1 - the existing khoros transport descriptor to be dup'ed
*		 id2 - the descriptor to be used for the newly dup'ed descriptor
*        Output: 
*       Returns: the newly dup'ed descriptor on success, or -1 upon failure
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Apr 28, 1993 22:13
*      Verified:
*  Side Effects: If too many descriptors are active then errno will be
*		 set to EMFILE and -1 will be returned.  If id is not
*		 a valid or active descriptor then errno will be set to
*		 EBADF and -1 will be returned.
* Modifications:
*
*************************************************************/

int kdup2(
   int id1,
   int id2)
{
	int newid;


	/*
	 *  Get the file entry associated with the id
	 */
	if (!kfile_checkid(id1))
	   return(-1);

	(void) kclose(id2);
	newid = kfile_dup(id1, id2);
	return(newid);
}


/**************************************************************
*  
*  Routine Name: kpopen - open a pipe (for I/O) from or to a process
*  
*  Purpose: open a pipe (for I/O) from or to a process
*    Input: command - command to send pipe output to or from
*	    type    - open status either 'r' or 'w'
*  Returns: The open file transport pointer
*    Written By: Mark Young  
*          Date: Jul 08, 1992 16:24
* Modifications:
*   Declaration: kfile *kpopen(
*                !   char *command,
*                !   char *type)
**************************************************************/

static int *popen_table = NULL;

kfile *kpopen(
   char *command,
   char *type)
{
	kfile *file;
	int   pid, pipes[2];
	char  filename[KLENGTH], temp[KLENGTH], *args[MAX_ARGS];

        /*
	 *  Do some error checking to make sure that we have a valid command
	 *  and mode to work with.
	 */
	if (type == NULL || command == NULL)
	{
	   errno = EINVAL;
	   return(NULL);
	}

	if ((int) tolower(type[0]) != 'r' && (int) type[0] != 'w')
	{
	   errno = EINVAL;
	   return(NULL);
	}

	/*
	 *  Now that we have a valid mode, we need to create a pipe in order
	 *  for the calling routine to either read or write to the child
	 *  process.
	 */
	if (pipe(pipes) == -1)
	   return(NULL);

	kcommand_to_args(command, args);
#if KVFORK_DEF
	if ((pid = vfork()) == 0)
#else
	if ((pid = fork()) == 0)
#endif
	{
           /*
            *  this is the child.  We should never return after
            *  calling
            */
           if ((int) tolower(type[0]) == 'r' && pipes[1] != 1)
	   {
	      dup2(pipes[1], 1);
	      if (type[0] == 'R') dup2(pipes[1], 2);
	   }

           if ((int) type[0] == 'w' && pipes[0] != 0)
              dup2(pipes[0], 0);

	   close(pipes[0]); close(pipes[1]);
	   execvp(args[0], args);
	   kfprintf(kstderr,"\nUnable to execute the following \
command:\n\n  %s", command);
	   _exit(KEXIT_FAILURE);
	}
	kstrcpy(temp, type);

	/*
	 *  Kludge:  Need to redefine the fact the 'R' for stdout/stderr
	 *  colides with extended kfopen() flags.  Use 'Z' for stdout/stderr
	 *  and extended kfopen() flags.  (MY Jun 01, 1994)
	 */
	if ((temp[0] = tolower(temp[0])) == 'r')
	   close(pipes[1]), pipes[1] = -1;
	else
	   close(pipes[0]), pipes[0] = -1;

	/*
	 *  Re-open transport as a Pipe
	 */
	ksprintf(filename,"pipe=[%d,%d]", pipes[0], pipes[1]);
	file = kfopen(filename, temp);

	/*
	 *  Malloc the popen table
	 */
	if (!popen_table)
	   popen_table = (int *) kcalloc(100, sizeof(int));

	popen_table[kfileno(file)] = pid;
	return(file);
}


/**************************************************************
*  
*  Routine Name: kpclose - close a pipe (for I/O) from or to a process
*  Purpose: Close a pipe (for I/O) from or to a process
*    Input: file - tranport pointer to close
*  Returns: 0 on success, -1 on failure
*    Written By: Mark Young  
*          Date: Jul 08, 1992 16:24
* Modifications:
*   Declaration: int kpclose(
*                !   kfile *file)
**************************************************************/
/*ARGSUSED*/
int kpclose(
   kfile *file)
{
	int entry, pid, status = 0;

	/*
	 *  Make sure the entry is within our table.  If not then it
	 *  has probably been closed already.
	 */
	if ((file = kfile_checkfile(file)) == NULL)
	{
	   errno = EBADF;
	   return(-1);
	}
	entry = kfileno(file);
	pid = popen_table[entry]; popen_table[entry] = 0;

	kfile_close(file);
	waitpid(pid, &status, 0);
	return(status);
}

/**************************************************************
*  
*  Routine Name: kpinfo - gets the associated process id
*
*  Purpose: Gets the pipe process id 
*    Input: file - tranport pointer to close
*  Returns: process id on success, -1 on failure
*    Written By: Mark Young  
*          Date: Jan 20, 1995
* Modifications:
*   Declaration: int kpinfo(
*                !   kfile *file)
**************************************************************/
/*ARGSUSED*/
int kpinfo(
   kfile *file)
{
	int entry, pid;

	/*
	 *  Make sure the entry is within our table.  If not then it
	 *  has probably been closed already.
	 */
	if ((file = kfile_checkfile(file)) == NULL)
	{
	   errno = EBADF;
	   return(-1);
	}
	entry = kfileno(file);
	pid = popen_table[entry];
	return(pid);
}

/**************************************************************
*  
*  Routine Name: ksystem - issue a shell command
*  
*  Purpose: This function is a replacement for the "system" library
*	   call.  The only difference is that ksystem() supports
*	   more than just files, it supports other data transports
*	   in order to allow remote execution of a job.
*  
*  
*  Input:  command - the command to be systemed
*  
*  Output:  none
*
*  Returns: The job status
*
*  Restrictions:
*    Written By: Mark Young  
*          Date: Jul 08, 1992 16:24
*      Verified:
*  Side Effects:
* Modifications:
*
**************************************************************/

int ksystem(
   char *command)
{
	char	*identifier, temp[KLENGTH], *args[MAX_ARGS];


	/*
	 *  According to the transport routines
	 */
	kstrcpy(temp, command);
	kcommand_to_args(temp, args);
	identifier = ktransport_identifier(args[0], temp);
	if (!ktransport_routines(identifier))
	   return(-1);

	/*
	 *  If there is no remote exec routine that simply call system() to
	 *  execute the command.
	if (routines->remote == NULL || routines->remote->exec == NULL)
	   return(system(command));
	else
	   remote = routines->remote;
	 */

	/*
	 *  Exec the command...
	if (remote->exec(args[0], args) == -1)
	   return(127);
	else
	   return(0);
	 */
	return(system(command));
}

/************************************************************
*
*  Routine Name: kspawn - spawn a command
*
*       Purpose: This function is used to spawn a command.  Similar to
*		 "ksystem" call, the kspawn() supports   more than just
*		 files, it supports other data transports in order to
*		 allow remote execution of a job.
*
*         Input: command - the command to be spawn
*        Output: 
*       Returns: the sub-process id on success, -1 on failure
*
*  Restrictions: 
*    Written By: Mark Young
*          Date: Apr 14, 1994
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int kspawn(
   char *command)
{
	int   pid;
	char  *args[MAX_ARGS];


	/*
	 *  According to the transport routines
	 */
	kcommand_to_args(command, args);

#if KVFORK_DEF
	if ((pid = vfork()) == 0)
#else
	if ((pid = fork()) == 0)
#endif
	{
	   kexecvp(args[0], args);
	   kfprintf(kstderr,"\nUnable to execute the following \
command:\n\n  %s", command);
	   _exit(KEXIT_FAILURE);
	}
	return(pid);
}

/**************************************************************
*  
*  Routine Name: kexecvp - execute a command
*  
*  Purpose: This function is a replacement for the system "execvp"
*	   call.  The only difference is that kexecvp() supports
*	   executing processes on remote machines.
*  
*	   The routine will execute another process (specified by
*	   arg0) and replace the current core image with the
*	   specified core.  For example, the command:
*  
*  	   !	"vfileinfo -i1 file"
*  
*	   would be specified as:
*  
*  	   !		arg0 = "vfileinfo" 
*          !
*  	   !		args[0] = "vfileinfo" 
*  	   !		args[1] = "-i1" 
*  	   !		args[2] = "file" 
*  	   !		args[3] = NULL
*  
*	   If the command is to be executed on a different machine such
*	   as "borris" then the same command would look like:
*  
*  	   !		"vfileinfo@borris -i1 file"
*  
*	   would be specified as:
*  
*  	   !		arg0 = "vfileinfo@borris" 
*          !
*  	   !		args[0] = "vfileinfo@borris" 
*  	   !		args[1] = "-i1" 
*  	   !		args[2] = "file" 
*  	   !		args[3] = NULL
*  
*  
*	   If the command is to be executed on a different machine such
*	   as "borris", and the file to be read is on "natasha" then
*	   the command would look like this:
*  
*  	   !		"vfileinfo@borris -i1 file@natasha"
*  
*	   would be specified as:
*  
*  	   !		arg0 = "vfileinfo@borris" 
*          !
*  	   !		args[0] = "vfileinfo@borris" 
*  	   !		args[1] = "-i1" 
*  	   !		args[2] = "file@natasha" 
*  	   !		args[3] = NULL
*  
*  
*         Input: arg0 - the routine to be executed
*	         args - the arguments, include arg0, in which execute
*		        the command
*       Returns: Note: this call overlays the calling process with the named
*	         file, then transfers control to the new core image.  If the
*	         command is successfully executed then there will be no return
*	         from the new process.  If the core cannot be loaded or found
*  	         then we return with an error of -1.
*    Written By: Mark Young  
*          Date: Jul 08, 1992 16:24
* Modifications:
**************************************************************/

int kexecvp(
   char *arg0,
   char *args[])
{
	char	*identifier, temp[KLENGTH];


	/*
	 *  Make a quick check to see if we are to be execing this locally
	 *  or remotely.  If local then simply call execvp()
	 */
	if (kstrchr(arg0, '@') == NULL)
	{
	   if (!kfullpath(arg0, NULL, temp))
	      kstrcpy(temp, arg0);
	   execvp(temp, args);
	   return(-1);
	}

	/*
	 *  According to the transport routines
	 */
	identifier = ktransport_identifier(arg0, temp);
	if (!ktransport_routines(identifier))
	   return(-1);

	/*
	 *  Should never return...
	if (routines->remote == NULL || routines->remote->exec == NULL)
	   return(-1);

	remote = routines->remote;
	if (remote->exec(arg0, args) == -1)
	   return(-1);
	else
	   _exit(0);
	 */
	return(-1);
}


/**************************************************************
*  
*  Routine Name: kfileno - return the transport descriptor
*  
*  Purpose: This function is used to retrieve the corresponding
*	   kfile entry descriptor, which can be used with the
*	   (int fid) routines like: kread(), kwrite().
*  
*  Input: file - the kfile structure to be referenced.
*  
*  Output: none
*
*  Returns: The entry descriptor (or -1 if it doesn't exist)
*
*  Restrictions:
*    Written By: Mark Young  
*          Date: Jul 08, 1992 16:24
*      Verified:
*  Side Effects:
* Modifications:
*
**************************************************************/
int kfileno(
   kfile *file)
{
	/*
	 *  Make sure the entry is within our table.  If not then it
	 *  has probably been closed already.
	 */
	if ((file = kfile_checkfile(file)) == NULL)
	{
	   errno = EBADF;
	   return(-1);
	}
	else
	   return(file->id);
}

/**************************************************************
*  
*  Routine Name: kexit_handler - adds an kexit handler
*  
*  Purpose: kexit_handler adds an handler so that upon call of kexit(),
*	    the handler is called.  kexit_handler() can be called
*	    several times, in which case the termination handlers
*	    will be called in order of their instantiation.  The handlers
*	    should be declared as follows:
*	    !
*	    ! void handler(
*	    !    int status,
*	    !    kaddr client_data)
*	    !
*	    
*  
*  Input: handler - the handler to be called
*	  client_data - the data to passed when calling the routine
*  Output:
*  Returns: TRUE on success, FALSE on failure
*
*  Restrictions:
*    Written By: Mark Young
*          Date: May 17, 1993 15:49
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int kexit_handler(
*		 !   kfunc_void handler,
*		 !   kaddr      client_data)
*
**************************************************************/

typedef struct _khandler
{
	kfunc_void handler;
	kaddr      client_data;
} khandler;

static klist *kexit_list = NULL;


int kexit_handler(
   kfunc_void handler,
   kaddr      client_data)
{
	khandler *temp;


	if (!handler || (temp = (khandler *) kcalloc(1,
		sizeof(khandler))) == NULL)
	{
	   return(FALSE);
	}
	temp->handler = handler;
	temp->client_data = client_data;
	kexit_list = klist_add(kexit_list, (kaddr) temp, NULL);
	return(TRUE);
}

/**************************************************************
*  
*  Routine Name: kexit - terminate a process
*  
*  Purpose: closes any Khoros transports left open and
*           exits the system with the status passed in.
*  
*  Input: status - exit status of either KEXIT_SUCCESS or KEXIT_FAILURE
*  Output:
*  Returns:
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Aug 14, 1992 16:56
*      Verified:
*  Side Effects:
* Modifications: Converted from khoros_close() in Khoros 1.0 (MY)
*
**************************************************************/

void kexit(
   kexit_status status)
{
	khandler *tmp;
	kaddr client_data;
	kfunc_void handler;
	klist *list = kexit_list;
	char  command[KLENGTH], *args[MAX_ARGS];


	while (list != NULL)
	{
	   tmp = (khandler *) klist_identifier(list);
	   handler = tmp->handler;
	   client_data = tmp->client_data;

	   list = klist_next(list);
	   handler(status, client_data);
	}

	knegotiate_transport(FALSE);
	if (kgetenv("KHOROS_REXEC"))
	{
	   kfprintf(kstderr,"please enter new command...");
	   kfgets(command, KLENGTH, kstdin);
	   kcommand_to_args(command, args);
	   kfile_closeall();
	   klist_free(kexit_list, NULL);
	   if (args[0])
	      kexecvp(args[0], args);
	   exit(KEXIT_FAILURE);
	}
	else
	{
	   kfile_closeall();
	   klist_free(kexit_list, NULL);
	   exit(status);
	}
}

/**************************************************************
*  
*  Routine Name: kmakecommand - creates a khoros command string
*  
*       Purpose: This function make a proper command that the transport
*	         mechanisms will interpet as command@machine.  If a
*	         command of "ls -al" is given and they wish to execute
*	         that command on the machine "borris".  Then a new string
*	         will that looks like "ls@borris -al" will be returned.
*  
*  
*         Input: machine   - the machine which we wish to find out about
*  
*	         exec_type - the exec'er to use when executing the command
*  
*	         command   - the command to be executed
*  
*        Output:
*	Returns: A kmalloc'ed string containing the proper form
*	         of the command@machine syntax
*  
*  Restrictions:
*    Written By:  Mark Young & Becky Bishop
*          Date:  Jul 08, 1992 16:22
*      Verified:
*  Side Effects:
* Modifications:
*  
**************************************************************/

char *kmakecommand(
   char *machine,
   char *exec_type,
   char *command)
{
	int  indx;
	char *temp, *mtemp, *ctemp, *etemp, string[KLENGTH];


	if (machine == NULL)
	   mtemp = NULL;
	else if ((mtemp = kstring_cleanup(machine, string)) != NULL)
	   mtemp = kstrcat("@", mtemp);

	if (command == NULL)
	   ctemp = NULL;
	else
	   ctemp = kstring_cleanup(command, NULL);

	if (exec_type == NULL)
	   etemp = NULL;
	else if ((etemp = kstring_cleanup(exec_type, string)) != NULL)
	   etemp = kstrcat(etemp, "=");

	if (mtemp == NULL && etemp == NULL)
	   return(kstrdup(ctemp));
	else if (ctemp == NULL)
	{
	   kfree(mtemp);
	   kfree(etemp);
	   return(NULL);
	}

	if ((temp = kstrchr(ctemp, ' ')) != NULL ||
	    (temp = kstrchr(ctemp, '\t')) != NULL)
	{
	   indx = temp - ctemp;
	   if (etemp != NULL)
	   {
	      kstrcpy(string, etemp);
	      kstrncat(string, ctemp, indx);
	      indx += kstrlen(etemp);
	   }
	   else
	      kstrncpy(string, ctemp, indx);

	   string[indx] = '\0';
	   temp = kstring_3cat(string, mtemp, temp, NULL);
	}
	else
	{
	   /*
	    *  Since this is a single argument command, simply concatenate
	    *  
	    */
	   temp = kstring_3cat(etemp, ctemp, mtemp, NULL);
	}
	kfree(mtemp);
	kfree(ctemp);
	kfree(etemp);
	return(temp);
}


/**************************************************************
*  
*  Routine Name: kgetdescriptors - get true UNIX file descriptors
*  
*  Purpose: This function is used to retrieve the input and
*	   output file descriptor associated with a transport.
*	   Note:  this routine should not be used without
*		   extreme caution, since certain transports
*		   don't have file descriptors (such as shared
*		   memory).
*  
*  Input: file - the kfile structure to be referenced.
*  
*  Output: inum - the input descriptor if exists (or if non exists minus 1)
*	   onum - the output descriptor if exists (or if non exists minus 1)
*
*  Returns: TRUE on success, FALSE on failure
*
*  Restrictions:
*    Written By: Mark Young  
*          Date: Jul 08, 1992 16:24
*      Verified:
*  Side Effects:
* Modifications:
*
**************************************************************/

int kgetdescriptors(
   kfile  *file,
   int    *inum,
   int    *onum)
{
	int	status;
	TransportInformation *routines;


	/*
	 *  Make sure the entry is within our table.  If not then it
	 *  has probably been closed already.
	 */
	if ((file = kfile_checkfile(file)) == NULL)
	{
	   errno = EBADF;
	   return(-1);
	}


	/*
	 *  Call the "filedescriptor()" routine to see if a input and output
	 *  file descriptor exist
	 */
	routines = file->routines;
	if (routines->descriptors != NULL)
	{
	   status = routines->descriptors(file, inum, onum);
	}
	else
	{
	   if (inum != NULL) *inum = -1;
	   if (onum != NULL) *onum = -1;
	   status = -1;
	}
	return(status);
}


/**************************************************************
*
*  Routine Name: kgetarchitecture - return the machine architecture
*  
*  Purpose: Same as the Khoros routine kmach_type(), except that it 
*	   supports all machines available for distributed processing.
*  
*  Input: machine  - machine to inquire about
*  
*  Output: description - string describing the type of machine
*
*  Returns: The machine architecture number
*
*  Restrictions:
*    Written By: Mark Young  
*          Date: Jul 08, 1992 16:24
*      Verified:
*  Side Effects:
* Modifications:
*
**************************************************************/

int kgetarchitecture(
   char *machine,
   char *description)
{
	int	type;
	char	temp[KLENGTH], *identifier;


	/*
	 *  According to the transport routines
	 */
	identifier = ktransport_machines(machine, temp);
	if (!ktransport_routines(identifier))
	   return(-1);

	/*
	 *  Check the remote machtype
	if (routines->remote != NULL)
	{
	   remote = routines->remote;
	   if (remote->machtype != NULL)
	      type = remote->machtype(machine, description);
	   else
	      return(-1);
	}
	else
	  */
	{
	   type = kmach_type(description);
	}
	return(type);
}


/**************************************************************
*
*  Routine Name: kgethostname - get the current hostname
*  
*  Purpose:  This routine is the same as the system gethostname()
*	     call except that it will get the hostname depending on
*	     the machine name is specified.
*  
*  Input:   machine - the machine in which to get the official hostname
*	    namelen - the size of the "hostname" buffer
*  
*  Output:  hostname - the current hostname string
*
*  Returns: 0 on success, and -1 on failure
*
*  Restrictions:
*    Written By: Mark Young  
*          Date: Jul 08, 1992 16:24
*      Verified:
*  Side Effects:
* Modifications:
*
**************************************************************/

int kgethostname(
   char *machine,
   char *hostname,
   int  namelen)
{
	char	temp[KLENGTH], *identifier;


	/*
	 *  According to the transport routines
	 */
	identifier = ktransport_machines(machine, temp);
	if (!ktransport_routines(identifier))
	   return(-1);

/*
	if (routines->remote != NULL)
	{
	   remote = routines->remote;
	   if (remote->gethostname != NULL)
	   {
	      if (remote->gethostname(machine, hostname, namelen) == -1)
		 return(-1);
	   }
	}
	else
 */
	{
	   if (gethostname(hostname, namelen) == -1)
	      return(-1);
	}
	return(0);
}
