/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Machine Definition Service Routines
   >>>>
   >>>>  Private:
   >>>>            kmach_sizeof()
   >>>>            kmach_type()
   >>>>            kmach_order()
   >>>>            kmach_format()
   >>>>            kmach_description()
   >>>>             
   >>>>   Static:
   >>>>             
   >>>>   Public:
   >>>>             
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"

/*
 * FYI:  Heres a rundown of the machines that we currently support.
 *
 *	DECstation		IEEE	LITENDIAN
 *	Encoer			IEEE	LITENDIAN
 *	Sequent			IEEE	LITENDIAN
 *	NS32000			IEEE	LITENDIAN
 *	IBM RT			IEEE	BIGENDIAN
 *	IBM032			IEEE	BIGENDIAN
 *	MC680x0			IEEE	BIGENDIAN
 *	MC68k32			IEEE	BIGENDIAN
 *	Sony News		IEEE	BIGENDIAN
 *	Sun			IEEE	BIGENDIAN
 *	Vax			VAX	LITENDIAN
 *	Mips Big Endian		IEEE	BIGENDIAN
 *	MIPS			IEEE	BIGENDIAN
 *	Silicon Graphics	IEEE	BIGENDIAN
 *	NeXT			IEEE	BIGENDIAN
 *	hp9000			IEEE	BIGENDIAN
 *	RISC 6000		IEEE	BIGENDIAN
 *	Cray			CRAY	BIGENDIAN
 *	Convex			IEEE	BIGENDIAN
 *	MacII			IEEE	BIGENDIAN
 *	Data General		IEEE	BIGENDIAN
 *	386/486			IEEE	LITENDIAN
 *	Motorola 88000		IEEE	BIGENDIAN
 *	luna			IEEE	BIGENDIAN
 *	Alpha			ALPHA	LITENDIAN
 *	Apollo			IEEE	BIGENDIAN
 */

/*
 * static tables.  These tables are used to determine the size of a
 * specified datatype given its format.  Currently, there are five
 * formats supported: unknown, generic 32bit ieee, vax, alpha ieee,
 * and 64bit cray with cyber floating point.
 */
static unsigned int bytesize[]   = {0, 1, 1, 1, 1};
static unsigned int shortsize[]  = {0, 2, 2, 2, 8};
static unsigned int intsize[]   =  {0, 4, 4, 4, 8};
static unsigned int longsize[]   = {0, 4, 4, 8, 8};
static unsigned int floatsize[]  = {0, 4, 4, 4, 8};
static unsigned int doublesize[] = {0, 8, 8, 8, 8};

/*
 * these macros return the bits that correspond to the format portion
 * of a machtype byte and the order portion of the machtype byte.
 */
#define FORMAT(a) (a & 0x1f)
#define ORDER(a)  (a & 0xe0)

/*----------------------------------------------------------------
|
|  Routine Name: kmach_sizeof() 
|
|       Purpose: kmach_sizeof() returns the size of a data type
|                on a particular architecture.  For example,
|                given the type KFLOAT and the architecture
|                KFORMAT_CRAY, this routine will return the value
|                8, which the number of bytes in a single
|                Cray float.
|
|         Input: dtype - data type as defined in kdefines.h
|                arch  - machine architecture as defined in kdefines.h
|
|        Output: 
|       Returns: The number of bytes given valid input, 
|                or 0 given invalid input.
|
|        Status: 
|
|  Restrictions: The sky's the limit with this one, baby!
|
|    Written By: Jeremy Worley  
|
|          Date: Thu Jun 25 11:47:37 MDT 1992
|
| Modifications:
|
------------------------------------------------------------------*/

int kmach_sizeof(
   int arch,
   int dtype)
{
   if(arch < 0)
      return(0);

    if(arch == KMACH_UNKNOWN)
       kinfo(KDEBUG,"kmach_sizeof was called with an unknown architecture.  "
	     "Returning 0.\n");

    switch(dtype)
    {
       case KBYTE:
       case KUBYTE:   
	  return(bytesize[FORMAT(arch)]);
       case KSHORT:
       case KUSHORT:  
	  return(shortsize[FORMAT(arch)]);
       case KLONG:
       case KULONG:   
	  return(longsize[FORMAT(arch)]);
       case KINT:      
       case KUINT:    
	  return(intsize[FORMAT(arch)]);
       case KFLOAT:   
	  return(floatsize[FORMAT(arch)]);
       case KDOUBLE:  
	  return(doublesize[FORMAT(arch)]);
       case KCOMPLEX: 
	  return(2 * floatsize[FORMAT(arch)]);
       case KDCOMPLEX:
	  return(2 * doublesize[FORMAT(arch)]);
       default: 
	 return(0);
   }
}

/*------------------------------------------------------------------
|
|  Routine Name: kmach_type()
|        
|       Purpose: returns the machine type or entry.
|
|         Input: None
|
|        Output: description - if not NULL then return the description of
|			       the machine entry.
|       Returns: the machine entry number for the current local
|			  architecture
|
|        Status: Private Routine
|
|  Restrictions: This routine is to be used only for its intended 
|                purpose.
|
|    Written By: Jeremy Worley 
|
|          Date: Jul 23, 1992 12:17
|
| Modifications: 
|       Jeremy Worley Fri Jun 26 15:46:01 MDT 1992
|              Minor face-lift for Khoros 2.0
|
-------------------------------------------------------------------*/

int kmach_type(
   char *description)
{
   if (description != NULL)
      kstrcpy(description, kmach_description(KMACH_LOCAL));

   return(KMACH_LOCAL);
}

/*------------------------------------------------------------------
|
|  Routine Name: kmach_order()
|        
|       Purpose: returns the word ordering for a given machine.
|                this routine probably ought to be called machorder(),
|                but that name was stolen by the above routine to 
|                return the machtype. Hmmm.
|
|         Input: type - type of machine...returned by machorder().
|
|        Output: 
|       Returns: a number corresponding to KORDER_BIGENDIAN or
|                KORDER_LITENDIAN from the table machorder[] in
|                machdefs.h
|
|        Status: Private Routine
|
|  Restrictions: This routine is to be used only for its intended 
|                purpose.
|
|    Written By: Jeremy Worley 
|
|          Date: Tue Sep 24 16:20:33 MDT 1991
|
| Modifications: 
|       Jeremy Worley Fri Jun 26 15:46:01 MDT 1992
|              Minor face-lift for Khoros 2.0
|
-------------------------------------------------------------------*/

int kmach_order(
   int type)
{
  int i;

  if(type == KMACH_UNKNOWN)
  {
     kinfo(KDEBUG,"kmach_order was called with an unknown architecture.  "
	   "Returning KORDER_UNKNOWN.\n");
     return(KORDER_UNKNOWN);
  }

  if (type < 0)
    return(KORDER_UNKNOWN);

  return(ORDER(type));
}

/*------------------------------------------------------------------
|
|  Routine Name: kmach_format()
|        
|       Purpose: returns the float format for a given machine.
|
|         Input: type - type of machine...returned by kget_machtype().
|
|        Output: 
|       Returns: a number corresponding to float point format
|                in machdefs.h
|
|        Status: Private Routine
|
|  Restrictions: This routine is to be used only for its intended 
|                purpose.
|
|    Written By: Jeremy Worley 
|
|          Date: Tue Sep 24 16:20:33 MDT 1991
|
| Modifications: 
|       Jeremy Worley Fri Jun 26 15:46:01 MDT 1992
|              Minor face-lift for Khoros 2.0
|
-------------------------------------------------------------------*/

int kmach_format(
   int type)
{
  int i;

  if(type == KMACH_UNKNOWN){
     kinfo(KDEBUG,"kmach_format was called with an unknown architecture.  "
	   "Returning KFORMAT_UNKNOWN.\n");
     return(KFORMAT_UNKNOWN);
  }

  return(FORMAT(type));
}

/*-----------------------------------------------------------
|
|  Routine Name: kmach_description - return the machine architecture
|				     description
|
|       Purpose: This routine returns the description given an machine
|		 architecture type.  The string returned is an internal
|		 pointer to the architecture description and should not
|		 be modified or freed.
|
|         Input: type - the machine architecture type
|       Returns: returns the description of the machine type, or NULL on
|		 failure.
|
|    Written By: Mark Young
|          Date: May 19, 1994
|
|  Restrictions: dont free the thing it returns.
| Modifications:
|
------------------------------------------------------------*/
char *kmach_description(
   int type)
{
   int i;
 
   if (type < 0)
      return(NULL);
 
   /*
    * right now, only five or so unique types really exist, or at least,
    * have had Khoros ported to them.  This routine isn't perfect,
    * but it'll reliably deal with all of the machines that have been
    * ported to so far.
    */
   switch(type)
   {
      case KORDER_BIG_ENDIAN | KFORMAT_IEEE:
	 return("Big Endian IEEE");
      case KORDER_LITTLE_ENDIAN | KFORMAT_IEEE_ALPHA:
	 return("Little Endian DEC Alpha IEEE");
      case KORDER_LITTLE_ENDIAN | KFORMAT_IEEE:
	 return("Little Endian IEEE");
      case KORDER_LITTLE_ENDIAN | KFORMAT_VAX:
	 return("Little Endian VAX");
      case KORDER_BIG_ENDIAN | KFORMAT_CRAY:
	 return("Big Endian Cray/Cyber");
      case KMACH_UNKNOWN:
	 return("Unknown");
      default:
	 return("Unrecognized");
   }
}
