/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            errno facility
   >>>>
   >>>>  Private:
   >>>>   Static:
   >>>>   Public:
   >>>>             
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"


static int	num_errors  = 0;
static kerrlist *error_list = NULL;


/*
 * System Error Messages
 */
struct _syserrs {
	int  kerrno;
	char *message;
};

static struct _syserrs syserrs[] = 
{
  { EPERM,		"Not owner" },
  { ENOENT,		"No such file or directory" },
  { ESRCH,		"No such process" },
  { EINTR,		"Interrupted system call" },
  { EIO,		"I/O error" },
  { ENXIO,		"No such device or address" },
  { E2BIG,		"Arg list too long" },
  { ENOEXEC,		"Exec format error" },
  { EBADF,		"Bad file number" },
  { ECHILD,		"No children" },
  { EAGAIN,		"No more processes" },
  { ENOMEM,		"Not enough core" },
  { EACCES,		"Permission denied" }, 
  { EFAULT,		"Bad address" },
#if KOPSYS_LOCAL != KOPSYS_FREEBSD
  { EBUSY,		"Mount device busy" },
#endif
  { EEXIST,		"File exists" },
  { EXDEV,		"Cross-device link" },
  { ENODEV,		"No such device" },
  { ENOTDIR,		"Not a directory" },
  { EISDIR,		"Is a directory" },
  { EINVAL,		"Invalid argument" }, 
  { ENFILE,		"File table overflow" },
  { EMFILE,		"Too many open files" }, 
  { ENOTTY,		"Not a typewriter" },
  { EFBIG,		"File too large" },
  { ENOSPC,		"No space left on device" },
  { ESPIPE,		"Illegal seek" },
  { EROFS,		"Read-only file system" },
  { EMLINK,		"Too many links" },
  { EPIPE,		"Broken pipe" },
  { EDOM,		"Argument too large" },
  { ERANGE,		"Result too large" },
  { EDEADLK,		"Deadlock condition" }, 
  { ENOTEMPTY,		"Directory not empty" },
/*
  { ENOLCK,		"No record locks available" },
  { ENOSYS,		"Function not implemented" },
 */
};

static int num_syserrs = knumber(syserrs);


/************************************************************
*
*  Routine Name: kerrno_init_errors - initialize errors to be used with khoros
*				      errno
*
*       Purpose: kerrno_init_errors initialize errors to be used with the
*		 khoros error reporting facility. 
*
*         Input: errlist   - the array of errno-message string pairs to
*			     be added to the global errno list.
*		 num_errs - number of errno's in the errlist array
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*    Written By: Mark Young
*          Date: Oct 30, 1993
*************************************************************/
int kerrno_init_errors(
   kerrlist *errlist,
   int	    num_errs)
{
	int	 i;
	kerrlist *tmplist;
	static int next_errno = 0;


	if (!error_list)
	{
	   /*
	    *  This list is different because the system errno is a bunch
	    *  of errnos.
	    */
	   error_list = (kerrlist *) kcalloc(num_syserrs, sizeof(kerrlist));
	   for (i = 0; i < num_syserrs; i++)
	   {
	      error_list[i].message = syserrs[i].message;
	      error_list[i].kerrno  = &syserrs[i].kerrno;
	      if (syserrs[i].kerrno > next_errno)
		 next_errno = syserrs[i].kerrno;
	   }
	   num_errors = num_syserrs;
	}

	if ((tmplist = krealloc((kaddr) error_list, (num_errors + num_errs) *
			sizeof(kerrlist))) == NULL)
	{
	   return(FALSE);
	}

	for (i = 0; i < num_errs; i++)
	{
	   tmplist[i+num_errors] = errlist[i];
	   *tmplist[i+num_errors].kerrno = ++next_errno;
	}
	num_errors += num_errs;
	error_list  = tmplist;

	return(TRUE);
}


/************************************************************
*
*  Routine Name: kerrno_lookup - lookup the error message associated
*				 with a errno.
*
*       Purpose: This routine finds the message string associated with an errno.
*         Input: errnum - the error number to be looked up
*       Returns: a pointer to the error message associated with the errnum or
*		 NULL if the error number does not exist.
*    Written By: Mark Young
*          Date: Oct 30, 1993 
*************************************************************/

char *kerrno_lookup(
   int errnum)
{
	int i;

	if (!error_list)
	   kerrno_init_errors(NULL, 0);

	for (i = 0; i < num_errors; i++)
	{
	   if (*error_list[i].kerrno == errnum)
	      return(error_list[i].message);
	}
	return(NULL);
}


/************************************************************
*
*  Routine Name: kerrno_check - check to see if an errno is within a given
*				error list.
*
*       Purpose: This routine looks up a errno to see if it is within a 
*		 supplied list of errors.
*
*         Input: errnum - the error number to be looked up
*		 errlist - the error list to check against
*		 num_errs - the number of errors
*       Returns: TRUE (1) if the errno is a member of the errlist passed in,
*		 FALSE (0) othersize.
*    Written By: Mark Young
*          Date: Oct 30, 1993 
*************************************************************/

int kerrno_check(
   int	    errnum,
   kerrlist *errlist,
   int      num_errs)
{
	int i;

	if (!error_list)
	   kerrno_init_errors(errlist, num_errs);

	for (i = 0; i < num_errs; i++)
	{
	   if (*errlist[i].kerrno == errnum)
	      return(TRUE);
	}
	return(FALSE);
}
