/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>         UNIX Pipe Transport Routines
   >>>>
   >>>>  Private:
   >>>>			shm_tempnam
   >>>>			shm_open	
   >>>>			shm_close
   >>>>			shm_read
   >>>>			shm_write
   >>>>			shm_lseek
   >>>>			shm_unlink
   >>>>			shm_access
   >>>>			shm_lock
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#if KOPSYS_LOCAL == KOPSYS_OSF
#undef _POSIX_SOURCE
#endif

#include "internals.h"	


#if !defined(KSHM_DEF)
TransportInformation shm_transport[] = {NULL};
#else

#if (KOPSYS_LOCAL == KOPSYS_SOLARIS || KOPSYS_LOCAL == KOPSYS_SUNOS) && KSTDC_DEF
#define ushort unsigned short
#define ulong  unsigned long
#define uint   unsigned int
#define uchar  unsigned char

#define u_short ushort
#define u_int   uint
#define u_long  ulong
#define u_char  uchar
#endif

#if KOPSYS_LOCAL == KOPSYS_DYNIX
typedef struct _quad { long val[2]; } quad;
typedef unsigned char   unchar;
typedef unsigned short  ushort;
typedef unsigned int    uint;
typedef unsigned long   ulong;
#endif /* KOPSYS_DYNIX */

#include <sys/ipc.h>
#include <sys/shm.h>


#if KOPSYS_LOCAL == KOPSYS_SUNOS
char *shmat();
#endif

static int shm_tempnam	PROTO((char *, char *, char *));
static int shm_open	PROTO((char *, int, int, kfile *));
static int shm_close	PROTO((kfile *));
static int shm_read	PROTO((kfile *, char *, int));
static int shm_write	PROTO((kfile *, char *, int));
static int shm_lseek	PROTO((kfile *, int, int));
static int shm_access	PROTO((char *, char *, int));
static int shm_unlink	PROTO((char *, char *));
static int shm_lock	PROTO((kfile *, int));

TransportInformation shm_transport[] =
{
    {
	"Shared Memory",
	"shm",
	TRUE,
	shm_tempnam,
	shm_open,
	shm_close,
	shm_read,
	shm_write,
	shm_lseek,
	shm_access,
	shm_unlink,
	shm_lock,
	NULL,
    }
};

/*
 *  Internal resource structures for "standard file" transport
 */
typedef struct
{
	int     shmid;
	int     key;
	int     size;
	int     offset;
	int     shmflag;
	char	*addr;
} ResourceStruct;



/*-------------------------------------------------------------------*
|  
|		  Private Shared Memory Routines
|  
--------------------------------------------------------------------*/

/*ARGSUSED*/
static int shm_reopen(
   kfile *file,
   int   num_bytes)
{
	int	status = 0;
	char	*temp, *addr = NULL;
	ResourceStruct *resources = (ResourceStruct *) file->resources;


	addr = (char *) kmalloc(resources->size);
	if (addr != NULL)
	   kmemcpy(addr, resources->addr, resources->size);

	if (resources->addr != NULL)
	{
	   status = shmdt(resources->addr);
	   if (shmctl(resources->shmid, IPC_RMID, NULL) < 0)
	   {
	      kfree(addr);
	      return(-1);
	   }
	}

	if ((resources->shmid = shmget(resources->key, num_bytes,
			resources->shmflag)) < 0)
	{
	   kfree(addr);
	   return(-1);
	}

	/*
	 *  Attach to file the appropriate memory map space...
	 */
        if ((temp = shmat(resources->shmid, 0, SHM_RND)) == (char *) -1)
	{
	   kfree(addr);
           return(-1);
	}

	/*
	 *  copy data back into the new shared memory segment.
	 */
	if (addr != NULL)
	   kmemcpy(temp, addr, resources->size);

	resources->addr = temp;
	resources->size = num_bytes;
	/*ksetbuffer(file, temp, num_bytes);*/

	kfree(addr);
	return(status);
}


/*ARGSUSED*/
static int shm_initialize(
   int key,
   int shmflg,
   int *size)
{
	int	shmid;
	struct  shmid_ds stats;


	/*
	 *  Get the shared memory id.  We pass in DefaultPageSize just in case
	 *  we are going to be creating this segment.
	 */
	if ((shmid = shmget(key, 1024, shmflg)) < 0)
	   return(-1);

	/*
	 *  Get the size of the segment in case anyone wants to know
	 */
	if (size != NULL)
	{
	   if (shmctl(shmid, IPC_STAT, &stats) < 0)
	      return(-1);

	   *size = stats.shm_segsz;
	}
	return(shmid);
}



/*-------------------------------------------------------------------*
|  
|		  Shared Memory Routines
|  
--------------------------------------------------------------------*/

/*-------------------------------------------------------------
|  
|  Routine Name: shm_tempnam
|  
|       Purpose: This function initializes a tempnam for a "file"
|	         type transport.  Given a request with a filename of
|	         the following syntax:
|  
|		  	"shm=XXXXXXXXXXX"
|		  	     ...etc....
|  
|         Input:  identifier - the transport identifier
|		  template   - the filename template to be modeled
|        Output:  result     - the newly created transport name
|   Called From:  internal routine called from ktempnam()
|    Written By:  Mark Young
|  
-------------------------------------------------------------*/
/*ARGSUSED*/
static int shm_tempnam(
   char *identifier,
   char *template,
   char *result)
{
	int  shmkey;
	char path[KLENGTH], *filename;


	/*
	 *  Make a tempnam from the template
	 */
	(void) kstrcpy(path, template);
	if ((filename = ktmpnam(path)) == NULL || creat(filename, 0666) == -1)
	   return(-1);

	/*
	 *  Initialize the Shared memory id to something unqiue by using
	 *  tempnam and the ftok() routine.
	 */
	if ((shmkey = ftok(filename, 0)) == -1)
	{
	   /*
	    *  Opps...  We failed to create a unique key.  Error and return
	    *  failure.
	    */
	   kinfo(KDEBUG,"shm_tempnam: Unable to get unique shared memory key.");
	   return(-1);
	}

	/*
	 *  Initialize and create
	 */
	if (shm_initialize(shmkey, 0664 | IPC_CREAT, NULL) < -1)
	   return(-1);

	/*
	 *  Store the unique key so that we can use it later.
	 */
	if (identifier == NULL)
	   (void) kstrcpy(result, filename);
	else
	   (void) sprintf(result,"%s=%s", identifier, filename);

	return(0);
}

/*-------------------------------------------------------------
|  
|  Routine Name: 	shm_open
|  
|       Purpose: This function opens a "shared memory" connection.  It is the
|	         internal driver to open a file, which is called by
|	         the user when they call kopen().  kopen() calls when
|	         it gets an open request with a filename of the following
|	         syntax:
|  
|		  	"shm=XXXXXXXXX"
|         Input: 
|        Output:  returns 0 or -1 if an error occurs
|   Called From:  internal routine called from kopen()
|    Written By:  Mark Young
|  
-------------------------------------------------------------*/
/*ARGSUSED*/
static int shm_open(
   char  *path,
   int   flags,
   int   mode,
   kfile *file)
{
	char	  *addr;
	ResourceStruct *resources;
	char	  filename[KLENGTH];
	int	  size, shmid, shmflag, shmkey, prot = SHM_RND;


	/*
	 *  Get shared memory (shmat) protections from the open permission
	 *  flags...
	 */
	if (!kfile_iswrite(file))
	{
	   prot &= SHM_RDONLY;
	   mode = 0400;
	}
	else
	   mode = 0664;

	/*
	 *  Get the size of the current segment.  If the file is to be
	 *  created, etc...
	 */
	shmflag = mode;
	if (flags & KOPEN_EXCL) shmflag |= IPC_EXCL;
	if (flags & KOPEN_CREAT) shmflag |= IPC_CREAT;

	/*
	 *  Open the shared memory segment...
	 */
	if (ksscanf(path, "%d", &shmkey) == 0)
	{
	   shmkey = -1;
	   if (ksscanf(path, "%s", filename) == 1 &&
	       (shmkey = ftok(filename, 0)) == -1)
	   {
	      if (kfile_iswrite(file) && access(filename, F_OK) == -1)
	      {
		 creat(filename, 0666);
	         shmkey = ftok(filename, 0);
	      }
	   }
	}

	if (shmkey == -1 || (shmid = shm_initialize(shmkey,shmflag,&size)) < 0)
	   return(kfile_rename(file, "file="));

	/*
	 *  Attach to file the appropriate memory map space...
	 */
	if ((addr = shmat(shmid, 0, prot)) == (char *) -1)
	   return(kfile_rename(file, "file="));

	if ((resources = (ResourceStruct *) kcalloc(1,
			sizeof(ResourceStruct))) == NULL)
	{
	   kinfo(KSYSLIB, "shm_open: Not enough memory to kcalloc (%d) bytes \
for the khoros shm structure.\n", sizeof(ResourceStruct));
	   return(-1);
	}
	resources->addr    = addr;
	resources->shmid   = shmid;
	resources->shmflag = shmflag;
	resources->key     = shmkey;
	resources->size    = (flags & KOPEN_TRUNC) ? 0 : size;

        if (resources->size > 0 && !kfile_iswrite(file))
	{
	   ksetbuffer(file, resources->addr, resources->size);
	   resources->offset = size;
	}
	else
	   resources->offset = (flags & KOPEN_APPEND) ? size : 0;

	file->resources = (kaddr) resources;
	return(0);
}

/*-------------------------------------------------------------
|  
|  Routine Name: shm_close
|  
|       Purpose: This function closes a "shared memory".  It is the internal
|	         driver to close a data shm segment, which is called by the user
|	         when they call kclose().  kclose() calls the "shared memory"
|	         internal drivers by the nature of the transport, which is
|	         dictated by the initial kopen().
|  
|         Input:  file - the kfile structure.
|        Output:  returns whether we were able to close the file
|   Called From:  internal routine called from kclose()
|    Written By:  Mark Young
|  
-------------------------------------------------------------*/
/*ARGSUSED*/
static int shm_close(
   kfile *file)
{
	char	  *addr;
	int	  status;
	ResourceStruct *resources = (ResourceStruct *) file->resources;


	/*
	 *  Simply call shmdt() to detach from shared memory segment
	 */
	addr = resources->addr;

	status = shmdt(addr);
	return(status);
}

/*-------------------------------------------------------------
|  
|  Routine Name: shm_read
|  
|       Purpose: This function reads a "shared memory".  It is the internal
|	         driver to read data from a shm, which is called by the user
|	         when they call kread().  kread() calls the "shared memory"
|	         internal drivers by the nature of the transport, which is
|	         dictated by the kopen().
|  
|         Input:  file   - the kfile structure.
|		  ptr    - the pointer to store the data into.
|		  nbytes - the number of bytes to read.
|        Output:  returns the number of bytes read from the shm
|   Called From:  internal routine called from kread()
|    Written By:  Mark Young
|  
-------------------------------------------------------------*/
/*ARGSUSED*/
static int shm_read(
   kfile *file,
   char  *ptr,
   int   nbytes)
{
	char	*addr;
	int	size, offset;
	struct	shmid_ds stats;
	ResourceStruct *resources = (ResourceStruct *) file->resources;


	/*
	 *  Simply call kmemcpy() since "kfile" is just a regular 
	 *  memory pointer.
	 */
	addr   = resources->addr;
	size   = resources->size;
	offset = resources->offset;

	/*
	 *  Make sure that we aren't trying to read more bytes than we
	 *  currently have.  If so then try re-opening the shared memory
	 *  segment to see if it's gotten bigger.
	 */
	if (offset+nbytes > size)
	{
           /*
            *  Get the size of the segment....
            */
           if (shmctl(resources->shmid, IPC_STAT, &stats) < 0)
              return(-1);

           resources->size = stats.shm_segsz;
	}

	if (offset == size)
	{
	   return(EOF);
	}
	else if (addr == ptr || addr+offset == ptr)
	{
	   if (offset+nbytes > size)
	      nbytes = size-offset;
	}
	else if (offset+nbytes > size)
	{
	   kmemcpy(ptr, addr+offset, size-offset);
	   nbytes = size-offset;
	}
	else
	   kmemcpy(ptr, addr+offset, nbytes);

	resources->offset = offset+nbytes;
	return(nbytes);
}

/*-------------------------------------------------------------
|  
|  Routine Name: shm_write
|  
|       Purpose: This function writes to a "shared memory".  It is the internal
|	         driver to write data from the supplied data array to a shm,
|	         which is called by the user when they call kwrite().  kwrite()
|	         calls the "shared memory" internal drivers by the nature of
|	         the transport, which is dictated by the kopen().
|  
|         Input:  file   - the kfile structure.
|		  ptr    - the pointer to store the data into.
|		  nbytes - the number of bytes to read.
|        Output:  returns the number of bytes written to the kfile
|   Called From:  internal routine called from kread()
|    Written By:  Mark Young
|  
-------------------------------------------------------------*/
/*ARGSUSED*/
static int shm_write(
   kfile *file,
   char  *ptr,
   int   nbytes)
{
	char	*addr;
	int	size, offset;
	ResourceStruct *resources = (ResourceStruct *) file->resources;


	/*
	 *  Simply call kmemcpy() since "kfile" is just a regular 
	 *  memory pointer.
	 */
	addr   = resources->addr;
	size   = resources->size;
	offset = file->position;

	/*
	 *  Make sure that we aren't trying to read more bytes than we
	 *  currently have.  If so then try re-opening the file to see
	 *  if it's gotten bigger.
	 */
	if (offset+nbytes > size)
	{
	   shm_reopen(file, offset+nbytes);
	   addr   = resources->addr;
	   size   = resources->size;
	}

	if (offset == size)
	{
	   return(EOF);
	}
	else if (addr == ptr || addr+offset == ptr)
	{
	   if (offset+nbytes > size)
	      nbytes = size-offset;
	}
	else if (offset+nbytes > size)
	{
	   kmemcpy(addr+offset, ptr, size-offset);
	   nbytes = size-offset;
	}
	else
	   kmemcpy(addr+offset, ptr, nbytes);

	resources->offset = offset+nbytes;
	return(nbytes);
}

/*-------------------------------------------------------------
|  
|  Routine Name: shm_lseek
|  
|       Purpose: This function is used to do a "lseek".  It is the internal
|	         driver to rewind to a specific point so that the data can be
|	         skipped or re-read.  This is called when the user calls
|	         klseek().  klseek() calls the "shared memory" internal drivers
|	         by the nature of the transport, which is dictated by the
|		 kopen().
|  
|         Input:  file   - the kfile structure.
|		  offset - the offset in which to seek
|		  whence - the control of how the offset will be applied
|        Output:  returns the number of bytes written to the kfile
|   Called From:  internal routine called from kread()
|    Written By:  Mark Young
|  
-------------------------------------------------------------*/
/*ARGSUSED*/
static int shm_lseek(
   kfile *file,
   int   offset,
   int   whence)
{
	ResourceStruct *resources = (ResourceStruct *) file->resources;


	/*
	 *  Simply call lseek() since "kfile" is just a regular shm
	 */
	if (whence == KSEEK_SET)
	   resources->offset = offset;
	else if (whence == KSEEK_CUR)
	   resources->offset += offset;
	else if (whence == KSEEK_END)
	   resources->offset = resources->size + offset;
	else
	{
	   errno = EINVAL;
	   return(-1);
	}

	if (resources->offset > resources->size)
	/*EMPTY*/
	{
	   /* not sure as of yet */
	}
	return(resources->offset);
}

/*-------------------------------------------------------------
|  
|  Routine Name: shm_access
|  
|       Purpose: This function initializes a tempnam for a "shm"
|	         type transport.  Given a request with a filename of
|	         the following syntax:
|  
|		  	"shm=XXXXXXXXXX"
|  
|         Input: 
|        Output:  returns 0 or -1 depending whether we
|		  sucessfully created the template
|   Called From:  internal routine called from kaccess()
|    Written By:  Mark Young
|  
-------------------------------------------------------------*/
/*ARGSUSED*/
static int shm_access(
   char *identifier,
   char *path,
   int  mode)
{
	int   shmkey;
	char  filename[KLENGTH], *name = NULL;

	/*
	 *  Retrieve the shared memory key from the filename.  If it's not
	 *  a valid key then
	 */
	if (ksscanf(path, "%d", &shmkey) != 1)
	{
	   if (ksscanf(path, "%s", filename) == 0 ||
	       (shmkey = ftok(filename, 0)) == -1)
	   {
	      errno = ENOTDIR;
	      return(-1);
	   }
	   name = filename;
	}

	/*
	 *  Check to see if the key (filename) exists
	 */
	if (shm_initialize(shmkey, 0400, NULL) < 0 && name == NULL)
	   return(-1);
	else
	   return(0);
}

/*-------------------------------------------------------------
|  
|  Routine Name: shm_unlink
|  
|       Purpose: This function initializes a tempnam for a "shm"
|	         type transport.  Given a request with a filename of
|	         the following syntax:
|  
|		  	"shm=XXXXXXXXXX"
|         Input: 
|        Output:  returns 0 or -1 depending whether we
|		  sucessfully created the template
|   Called From:  internal routine called from kunlink()
|    Written By:  Mark Young
|  
-------------------------------------------------------------*/
/*ARGSUSED*/
static int shm_unlink(
   char *identifier,
   char *path)
{
	char  filename[KLENGTH];
	int   shmid, shmkey, status = -1;


	/*
	 *  Retrieve the shared memory key from the filename.  If it's not
	 *  a valid key then
	 */
	if (ksscanf(path, "%d", &shmkey) != 1)
	{
	   if (ksscanf(path, "%s", filename) == 0 ||
	       (shmkey = ftok(filename, 0)) == -1)
	   {
	      errno = ENOTDIR;
	      return(-1);
	   }
	   status = unlink(filename);
	}

	/*
	 *  Retrieve the shared memeory id so that we can delete it
	 *  from shared memory.
	 */
	if ((shmid = shm_initialize(shmkey, 0400, NULL)) < 0)
	   return(status);

	if (shmctl(shmid, IPC_RMID, NULL) < 0)
	   return(-1);

	return(0);
}

/*-------------------------------------------------------------
|  
|  Routine Name: shm_lock
|  
|       Purpose: This function locks a "shm" transport.  Given
|	         a request with a filename of the following syntax:
|  
|		  	"shm=key"
|         Input: 
|        Output:  returns 0 or -1 depending whether we
|		  sucessfully created the template
|   Called From:  internal routine called from kflock()
|    Written By:  Mark Young
|  
-------------------------------------------------------------*/
/*ARGSUSED*/
static int shm_lock(
   kfile *file,
   int   operation)
{
	ResourceStruct *resources = (ResourceStruct *) file->resources;


	return(-1);
}

#endif  /* KSHM_DEF */
/* don't add after the endif */
