/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>              Data Manipulation Utilities
   >>>>
   >>>>  Private:
   >>>>		    kdata_normalize()
   >>>>   Static:
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"


/*-----------------------------------------------------------
|
|  Routine Name: kdata_normalize
|
|       Purpose: This routine accepts as input a vector
|		 of data of any type and attempts to
|		 normalize it to a specified maximum
|		 value.  The input sequence is scanned for
|		 the maximum magnitude value.  A scale factor
|		 is then computed and all values are scaled
|		 accordingly.
|
|         Input: idata - the input data sequence.
|		 num   - number of input and output data
|	   		 values.
|		 dtype - data type of input and output
|			 data.
|		 rnorm - desired real component of magnitude
|			 of the largest output value.
|		 inorm - desired imaginary component of
|			 magnitude of largest output value.
|
|        Output: odata - the output data.
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|  Restrictions:
|    Written By: Jeremy Worley
|          Date: Oct 05, 1992 11:10
|      Verified:
|  Side Effects:
| Modifications:
|
|-----------------------------------------------------------*/

int 
kdata_normalize(kaddr idata, int num, int dtype, double rnorm, double inorm, kaddr odata)
{
   int i;
   register double rmax, imax;

   switch (dtype) {
   case KBIT:
      /* the easy case...normalization doesn't make sense, so return */
      return (TRUE);
   case KBYTE:
      {
	 unsigned char *tmp = (unsigned char *) idata;

         rmax = (double) tmp[0];
         imax = 0.0;
         for (i = 1; i < num; i++)
	    rmax = (rmax < (double) tmp[i]) ? (double) tmp[i] : rmax;
      }
      break;
   case KUBYTE:
      {
	 unsigned char *tmp = (unsigned char *) idata;

	 rmax = (double) tmp[0];
	 imax = 0.0;
	 for (i = 1; i < num; i++)
	    rmax = (rmax < (double) tmp[i]) ? (double) tmp[i] : rmax;
      }
      break;
   case KSHORT:
      {
	 short *tmp = (short *) idata;

	 rmax = (double) tmp[0];
	 imax = 0.0;
	 for (i = 1; i < num; i++)
	    rmax = (rmax < (double) tmp[i]) ? (double) tmp[i] : rmax;
      }
      break;
   case KUSHORT:
      {
	 unsigned short *tmp = (unsigned short *) idata;

	 rmax = (double) tmp[0];
	 imax = 0.0;
	 for (i = 1; i < num; i++)
	    rmax = (rmax < (double) tmp[i]) ? (double) tmp[i] : rmax;
      }
      break;
   case KLONG:
      {
	 long *tmp = (long *) idata;

	 rmax = (double) tmp[0];
	 imax = 0.0;
	 for (i = 1; i < num; i++)
	    rmax = (rmax < (double) tmp[i]) ? (double) tmp[i] : rmax;
      }
      break;
   case KULONG:
      {
	 unsigned long *tmp = (unsigned long *) idata;

	 rmax = (double) tmp[0];
	 imax = 0.0;
	 for (i = 1; i < num; i++)
	    rmax = (rmax < (double) tmp[i]) ? (double) tmp[i] : rmax;
      }
      break;
   case KINT:
      {
	 int *tmp = (int *) idata;

	 rmax = (double) tmp[0];
	 imax = 0.0;
	 for (i = 1; i < num; i++)
	    rmax = (rmax < (double) tmp[i]) ? (double) tmp[i] : rmax;
      }
      break;
   case KUINT:
      {
	 unsigned int *tmp = (unsigned int *) idata;

	 rmax = (double) tmp[0];
	 imax = 0.0;
	 for (i = 1; i < num; i++)
	    rmax = (rmax < (double) tmp[i]) ? (double) tmp[i] : rmax;
      }
      break;
   case KFLOAT:
      {
	 float *tmp = (float *) idata;

	 rmax = (double) tmp[0];
	 imax = 0.0;
	 for (i = 1; i < num; i++)
	    rmax = (rmax < (double) tmp[i]) ? (double) tmp[i] : rmax;
      }
      break;
   case KDOUBLE:
      {
	 double *tmp = (double *) idata;

	 rmax = tmp[0];
	 imax = 0.0;
	 for (i = 1; i < num; i++)
	    rmax = (rmax < tmp[i]) ? tmp[i] : rmax;
      }
      break;
   case KCOMPLEX:
      {
	 kcomplex *tmp = (kcomplex *) idata;
	 double a, b;

	 rmax = (double) kcreal(tmp[0]);
	 imax = (double) kcimag(tmp[0]);
	 for (i = 1; i < num; i++) {
	    a = (double) kcreal(tmp[i]);
	    b = (double) kcimag(tmp[i]);
	    if (sqrt(a * a + b * b) > sqrt(rmax * rmax + imax * imax)) {
	       rmax = a;
	       imax = b;
	    }
	 }
      }
      break;
   case KDCOMPLEX:
      {
	 kdcomplex *tmp = (kdcomplex *) idata;
	 double a, b;

	 rmax = kdcreal(tmp[0]);
	 imax = kdcimag(tmp[0]);
	 for (i = 1; i < num; i++) {
	    a = kdcreal(tmp[i]);
	    b = kdcimag(tmp[i]);
	    if (sqrt(a * a + b * b) > sqrt(rmax * rmax + imax * imax)) {
	       rmax = a;
	       imax = b;
	    }
	 }
      }
      break;
   default:
      errno = KINVALID_DATATYPE;
      kerror("kmath", "kdata_normalize()",
	 "The output data type is not recognized (%d).",dtype);
      return (FALSE);
   }

   /*
    * Call kdata_scale with the appropriate parameters.  This is wrong for
    * kcomplex...perhaps I should use a kcomplex division to compute the
    * scale factors.
    */
   rmax = (rmax != 0) ? rnorm / rmax : rnorm;
   imax = (imax != 0) ? inorm / imax : inorm;

   return (kdata_scale(idata, num, dtype, rmax, imax, 0.0, 0.0, odata));
}

