/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            libm compatibility functions
   >>>>
   >>>>  Private:
   >>>>			__kfmod
   >>>>			__kasinh
   >>>>			__kacosh
   >>>>			__katanh
   >>>>			__kexpm1
   >>>>			__klog1p
   >>>>			__kexp2
   >>>>			__kexp10
   >>>>			__klog2
   >>>>			__kcbrt
   >>>>			__kgamma
   >>>>			__ktrunc
   >>>>   Static:
   >>>>             
   >>>>   Public:
   >>>>             
   >>>>             
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"
#include <float.h>

/*-----------------------------------------------------------
|
|  Routine Name: __kfmod - floating point modulo operation
|
|       Purpose: computes the double precision floating point
|		 point modulo function.
|
|         Input: a - first argument of modulo function
|		 b - second argument of modulo function
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Tait Cyrus
|          Date: Aug 01, 1991 15:30
| Modifications:
|
------------------------------------------------------------*/

double __kfmod(
   double a,
   double b)
{
   if (b == 0) return(0.0);
   
   if(a > 0.0){
      if (b > 0.0)return( a - (floor(a / b) * b) );

      return( a - (ceil(a / b) * b) );
   }

   if (b > 0.0) return( a - (ceil(a / b) * b) );

   return( a - (floor(a / b) * b) );
}

/*-----------------------------------------------------------
|
|  Routine Name: __kasinh - arc hyperbolic sine
|
|       Purpose: This function serves as a replacement for the
|		 system asinh in the event that it does not
|		 exist on the current machine.
|
|         Input: x - double argument to asinh
|
|        Output: 
|
|       Returns: The arc hyperbolic sine of the input argument
|
|    Written By: Jeremy Worley
|          Date: Mar 27, 1993 11:41
| Modifications:
|
------------------------------------------------------------*/

double __kasinh(double x)
{
   return(klog(x + ksqrt(x*x + 1)));
}


/*-----------------------------------------------------------
|
|  Routine Name: __kacosh - arc hyperbolic cosine
|
|       Purpose: This function serves as a replacement for the
|		 system acosh in the event that it does not
|		 exist on the current machine.
|
|         Input: x - double argument to acosh
|
|        Output: 
|
|       Returns: The arc hyperbolic cosine of the input argument
|
|    Written By: Jeremy Worley
|          Date: Mar 27, 1993 11:41
| Modifications:
|
------------------------------------------------------------*/

double __kacosh(double x)
{
   if (x < 1.0) {
      kinfo(KSTANDARD,"kacosh():  Domain error: argument = %g",x);
      return(KMAXFLOAT);
   }

   return(klog(x + ksqrt(x*x - 1)));
}

/*-----------------------------------------------------------
|
|  Routine Name: __katanh - arc hyperbolic tangent
|
|       Purpose: This function serves as a replacement for the
|		 system tangent in the event that it does not
|		 exist on the current machine.
|
|         Input: x - double argument to atanh
|
|        Output: 
|
|       Returns: The arc hyperbolic tangent of the input argument
|
|    Written By: Jeremy Worley
|          Date: Mar 27, 1993 11:41
| Modifications:
|
------------------------------------------------------------*/

double __katanh(double x)
{
   if(x <= -1.0 || x >= 1.0){
      kinfo(KSTANDARD,"katanh():  Domain error: argument = %g",x);
      return(KMAXFLOAT);
   }

   return(0.5 * klog((1.0 + x)/(1.0 - x)));
}


/*-----------------------------------------------------------
|
|  Routine Name: __kexpm1 - e^x - 1
|
|       Purpose: This function serves as a replacement for the
|		 system expm1 in the event that it does not
|		 exist on the current machine.
|
|         Input: x - double argument to expm1
|
|        Output: 
|
|       Returns: The exponential - 1 of the input argument
|
|    Written By: Jeremy Worley
|          Date: Mar 27, 1993 11:41
| Modifications:
|
------------------------------------------------------------*/

double __kexpm1(double x)
{
   return(kexp(x) - 1.0);
}


/*-----------------------------------------------------------
|
|  Routine Name: __klog1p - log(1 + x)
|
|       Purpose: This function serves as a replacement for the
|		 system log1p in the event that it does not
|		 exist on the current machine.
|
|         Input: x - double argument to log1p
|
|        Output: 
|
|       Returns: The log of 1 plus the input argument
|
|    Written By: Jeremy Worley
|          Date: Mar 27, 1993 11:41
| Modifications:
|
------------------------------------------------------------*/

double __klog1p(double x)
{
   return(klog(x + 1.0));
}


/*-----------------------------------------------------------
|
|  Routine Name: __kexp2 - base 2 exponential
|
|       Purpose: This function serves as a replacement for the
|		 system exp2 in the event that it does not
|		 exist on the current machine.
|
|         Input: x - double argument to exp2
|
|        Output: 
|
|       Returns: The base 2 exponential of the input argument
|
|    Written By: Jeremy Worley
|          Date: Mar 27, 1993 11:41
| Modifications:
|
------------------------------------------------------------*/

double __kexp2(double x)
{
   return(kexp(x * KLN2));
}


/*-----------------------------------------------------------
|
|  Routine Name: __kexp10 - base 10 exponential
|
|       Purpose: This function serves as a replacement for the
|		 system exp10 in the event that it does not
|		 exist on the current machine.
|
|         Input: x - double argument to exp10
|
|        Output: 
|
|       Returns: The base 10 exponential of the input argument
|
|    Written By: Jeremy Worley
|          Date: Mar 27, 1993 11:41
| Modifications:
|
------------------------------------------------------------*/

double __kexp10(double x)
{
   return(kexp(x * KLN10));
}


/*-----------------------------------------------------------
|
|  Routine Name: __klog2 - base 2 logarithm
|
|       Purpose: This function serves as a replacement for the
|		 system log2 in the event that it does not
|		 exist on the current machine.
|
|         Input: x - double argument to log2
|
|        Output: 
|
|       Returns: The base 2 logarithm of the input argument
|
|    Written By: Jeremy Worley
|          Date: Mar 27, 1993 11:41
| Modifications:
|
------------------------------------------------------------*/

double __klog2(double x)
{
   return(klog(x) * KLOG2E);
}


/*-----------------------------------------------------------
|
|  Routine Name: __kcbrt - cube root
|
|       Purpose: This function serves as a replacement for the
|		 system cbrt in the event that it does not
|		 exist on the current machine.
|
|         Input: x - double argument to cbrt
|
|        Output: 
|
|       Returns: The cube root of the input argument
|
|    Written By: Jeremy Worley
|          Date: Mar 27, 1993 11:41
| Modifications:
|
------------------------------------------------------------*/

double __kcbrt(double x)
{
   int sgn;
   
   if (x > -FLT_EPSILON && x < FLT_EPSILON)
      return(0.0);

   sgn = (x > 0) ? 1 : -1;
   
   return(sgn * kexp(klog(kabs(x)) * 1.0 / 3.0));
}

/*-----------------------------------------------------------
|
|  Routine Name: __kgamma()
|
|       Purpose: This function implements a log gamma 
|		 function.  A weak attempt but its a start.
|
|                 reference:  Wong, Computational Methods in 
|                 Physics & Engineering Prentice-Hall, 1992.  
|                 Pages 196 - 203.  This is an 8-term polynomial
|                 approximation to Gamma(x).  This function is 
|                 said to accurate to |e| <= 3e-7.  Computes gamma 
|                 in range 0-1 of fractional part of argument, 
|                 then uses recurrance relation Gamma(1+x) = xGamma(x).
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Mar 29, 1993 08:41
| Modifications:
|
------------------------------------------------------------*/

double 
__kgamma(double x)
{
   double f, z, g, y;
   int i;
   
   static double b[] = { -0.577191652, 0.988205891,-0.897056937, 0.918206857, 
			 -0.756704078, 0.482199394,-0.193527818, 0.035868343};
   
   if (x <= 0.0)
   {
      kinfo(KSTANDARD,"Invalid argument (%g) to gamma.\n");
      return(0.0);
   }
   
   /*
    * reduce the argument to the range [0,1]
    */
   z = x - 1.0;
   f = 1.0;
   
   /*
    * if z < 0, use Gamma(x) = x^{-1} Gamma(1+x)
    */
   if (z < 0.0)
   {
      f = 1.0/x;
      z += 1.0;
   }
   /*
    * if z > 1, use Gamma(1+x) = xGamma(x) to reduce the argument
    * to < 1.  UIse f=z(z-1)(z-2) ... to keep track of the factor.
    */
   else if (z >= 1.0)
   {
      do
      {
	 f *= z;
	 z -= 1.0;
      }
      while (z >= 1.0);
   }
   
   /*
    * step 3 - evaluate G(1+z)
    */
   g = y = 1.0;

   for (i = 0; i < 8; i++)
   {
      y *= z;
      g += b[i] * y;
   }
   
   return (klog(f * g));
}

/*-----------------------------------------------------------
|
|  Routine Name: __ktrunc()
|
|       Purpose: This function implements a portable trunc
|		 function
|
|         Input: x - number to truncate
|
|        Output: 
|
|       Returns: the truncated version of x.
|
|    Written By: Jeremy Worley
|          Date: Mar 31, 1993 13:18
| Modifications:
|
------------------------------------------------------------*/

double __ktrunc(double x)
{
  return((double)((int)x));
}

/*-----------------------------------------------------------
|
|  Routine Name: __ksrandom()
|
|       Purpose: This function sets the random generator seed.
|
|         Input: seed - seed number
|
|        Output: 
|
|       Returns: void
|
|    Written By: Jeremy Worley
|          Date: Mar 31, 1993 13:18
| Modifications:
|
------------------------------------------------------------*/

void __ksrandom(int seed)
{
  (void)ksrand48(seed);
  return;
}


/*-----------------------------------------------------------
|
|  Routine Name: __krandom()
|
|       Purpose: This function implements random number generator
|
|         Input: 
|
|        Output: 
|
|       Returns: A random number in the range of 0..MAXLINT.
|
|    Written By: Jeremy Worley
|          Date: Mar 31, 1993 13:18
| Modifications:
|
------------------------------------------------------------*/

long __krandom(void)
{
   return(klrand48());
}
