/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>              Data Manipulation Utilities
   >>>>
   >>>>  Private:
   >>>>		    kdata_sample()
   >>>>   Static:
   >>>>		    _interp1 (macro)
   >>>>             _data_sample_1st
   >>>>             _data_sample_zero_order
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"

#include <float.h>
/*
 * a floating point indexer
 */
#define _do_index(stride, offset) (int)(i * (stride + FLT_EPSILON) + offset)

#if 0
#define _do_zero_interp(out, in, type)            \
      {                                           \
	 register type *optr = (type *) out;               \
	 register type *iptr = (type *) in;                \
	 for (i = 0; i < num_elem; i++)           \
	    *optr++ = iptr[_do_index(istride, ioffset)]; \
      }
#endif

#define _do_zero_interp(out, in, type)                \
      {                                               \
	 register type *optr = (type *) out;          \
	 register type *iptr = (type *) in;           \
         register float f = ioffset;                  \
	 for (i = 0; i < num_elem; i++, f += istride + FLT_EPSILON) \
	    *optr++ = iptr[(int)f];                   \
      }

/*-----------------------------------------------------------
|
|  Routine Name: _interp1
|
|       Purpose: This macro is a first order linear
|		 interpolation routine used by kdata_sample.
|
|         Input: data - the data to be re-sampled.
|		 i1   - the integer first index.
|		 i2   - the integer next  index.
|		 f    - the double intermediate index.
|		 num  - number of elements in data.
|		 type - data type (i.e. float, double)
|
|        Output: 
|
|       Returns: Returns a first order approx. of data[f].
|
|    Written By: Jeremy Worley
|          Date: Oct 02, 1992 17:30
| Modifications:
|
------------------------------------------------------------*/

#define _interp1(data,i1,f,i2,num,type) \
      (type)(((int)f < (int)(num - 1)) ? ((double)data[i1] + (((double)data[i1] - \
         (double)data[i2])/((double)i1 - (double)i2)) * ((double)f - (double)i1)) : \
         data[num-1])

/*-----------------------------------------------------------
|
|  Routine Name: _data_sample_1st
|
|       Purpose: This routine is used to subsample or
|		 supersample a float data set by 1st order
|		 interpolation.
|
|		 This function assumes that the input and
|		 output pointers have been checked and
|		 are valid, and that the input and output
|		 dimensions are positive.
|
|         Input: idata - input data vector
|                inum  - the number of elements in idata
|		 onum  - the desired number elements in
|			 odata.
|		 dtype - data type of incoming and outgoing
|			 data vectors.
|
|        Output: odata - the output data vector
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Oct 01, 1992 14:22
| Modifications:
|
------------------------------------------------------------*/

static int 
_data_sample_1st(char *idata, int inum, int onum, int dtype, char *odata)
{
   int i;			/* an index for a loop */
   register double f = 0.0;	/* a float index for sampling */
   register double delta;	/* a increment factor for samplint */
   int size;

   /*
    * determine the size of the requested data type for the current machine.
    * If the current machine is unknown, or the data type is illegal, return
    * a fail without issuing an error message.
    */
   if (dtype == KBIT)
      size = 0;
   else if ((size = kdata_size(dtype)) == 0)
      return (FALSE);

   /*
    * if the number of input data points is equal to the number of output
    * data points, then there is really nothing to be done.  This is ok,
    * though, so we will simply copy the data from the input to the output
    * and return
    */
   if (inum == onum) {
      if (size != 0)
	 memcpy(odata, idata, inum * size);
      else
	 memcpy(odata, idata, ((inum >> 3) + ((inum % 8) != 0)));

      return (TRUE);
   }

   /*
    * compute a floating point "skip factor".  If this value is less than 1.0
    * then we are supersampling (or a proper interpolation), otherwise we are
    * subsampling.
    */
   delta = (double) (inum) / (double) (onum);

   /*
    * depending on data type, call the first order interpolation with the
    * appropriate flags.
    */
   switch (dtype) {
   case KBIT:
      {
	 unsigned char *optr = (unsigned char *) odata;
	 unsigned char *iptr = (unsigned char *) idata;
	 unsigned char mask, maskout;

	 for (i = 0; i < (onum >> 3) + ((onum % 8) != 0); i++)
	    optr[i] = 0;
	 for (i = 0; i < onum; i++) {
	    mask = 0x80 >> ((int) (i * delta) % 8);
	    maskout = 0x80 >> (i % 8);
	    optr[(int) (i / 8)] |= (unsigned char)
	       ((iptr[(int) (i * delta / 8)] & mask) ? maskout : 0x0);
	 }
      }
      break;
   case KBYTE:
      {
	 for (i = 0, f = 0.0; i < onum; i++, f += delta) {
	    odata[i] = _interp1(idata, (int) f, f,
	       kmin((int) (f + 1), (inum - 1)), inum,
	       char);
	 }
      }
      break;
   case KUBYTE:
      {
	 unsigned char *_idata = (unsigned char *) idata;
	 unsigned char *_odata = (unsigned char *) odata;

	 for (i = 0, f = 0.0; i < onum; i++, f += delta) {
	    _odata[i] = _interp1(_idata, (int) f, f,
	       kmin((int) (f + 1), (inum - 1)), inum,
	       unsigned char);
	 }
      }
      break;
   case KSHORT:
      {
	 short *_idata = (short *) idata;
	 short *_odata = (short *) odata;

	 for (i = 0, f = 0.0; i < onum; i++, f += delta) {
	    _odata[i] = _interp1(_idata, (int) f, f,
	       kmin((int) (f + 1), (inum - 1)), inum,
	       short);
	 }
      }
      break;
   case KUSHORT:
      {
	 unsigned short *_idata = (unsigned short *) idata;
	 unsigned short *_odata = (unsigned short *) odata;

	 for (i = 0, f = 0.0; i < onum; i++, f += delta) {
	    _odata[i] = _interp1(_idata, (int) f, f,
	       kmin((int) (f + 1), (inum - 1)), inum,
	       unsigned short);
	 }
      }
      break;
   case KLONG:
      {
	 long *_idata = (long *) idata;
	 long *_odata = (long *) odata;

	 for (i = 0, f = 0.0; i < onum; i++, f += delta) {
	    _odata[i] = _interp1(_idata, (int) f, f,
	       kmin((int) (f + 1), (inum - 1)), inum,
	       long);
	 }
      }
      break;
   case KULONG:
      {
	 unsigned long *_idata = (unsigned long *) idata;
	 unsigned long *_odata = (unsigned long *) odata;

	 for (i = 0, f = 0.0; i < onum; i++, f += delta) {
	    _odata[i] = _interp1(_idata, (int) f, f,
	       kmin((int) (f + 1), (inum - 1)), inum,
	       unsigned long);
	 }
      }
      break;
   case KINT:
      {
	 int *_idata = (int *) idata;
	 int *_odata = (int *) odata;

	 for (i = 0, f = 0.0; i < onum; i++, f += delta) {
	    _odata[i] = _interp1(_idata, (int) f, f,
	       kmin((int) (f + 1), (inum - 1)), inum,
	       int);
	 }
      }
      break;
   case KUINT:
      {
	 unsigned int *_idata = (unsigned int *) idata;
	 unsigned int *_odata = (unsigned int *) odata;

	 for (i = 0, f = 0.0; i < onum; i++, f += delta) {
	    _odata[i] = _interp1(_idata, (int) f, f,
	       kmin((int) (f + 1), (inum - 1)), inum,
	       unsigned int);
	 }
      }
      break;
   case KFLOAT:
      {
	 float *fidata = (float *) idata;
	 float *fodata = (float *) odata;

	 for (i = 0, f = 0.0; i < onum; i++, f += delta) {
	    fodata[i] = _interp1(fidata, (int) f, f,
	       kmin((int) (f + 1), (inum - 1)), inum,
	       float);
	 }
      }
      break;
   case KDOUBLE:
      {
	 double *_idata = (double *) idata;
	 double *_odata = (double *) odata;

	 for (i = 0, f = 0.0; i < onum; i++, f += delta) {
	    _odata[i] = _interp1(_idata, (int) f, f,
	       kmin((int) (f + 1), (inum - 1)), inum,
	       double);
	 }
      }
      break;
   case KCOMPLEX:
      {
	 kcomplex *_idata = (kcomplex *) idata;
	 kcomplex *_odata = (kcomplex *) odata;

	 for (i = 0, f = 0.0; i < onum; i++, f += delta) {
	    if (f < inum - 1)
	       _odata[i] = kcadd(_idata[(int) f],
		  kcmult(kcdiv(kcsub(_idata[(int) f], _idata[(int) (f + 1)]),
			kccomp((float) (((int) f - (int) (f + 1))), 0.0)),
		     kccomp((float) (f - (float) (int) f), 0.0)));
	    else
	       _odata[i] = _idata[inum - 1];
	 }
      }
      break;
   case KDCOMPLEX:
      {
	 kdcomplex *_idata = (kdcomplex *) idata;
	 kdcomplex *_odata = (kdcomplex *) odata;

	 for (i = 0, f = 0.0; i < onum; i++, f += delta) {
	    if (f < inum - 1)
	       _odata[i] = kdcadd(_idata[(int) f],
		  kdcmult(kdcdiv(kdcsub(_idata[(int) f], _idata[(int) (f + 1)]),
			kdccomp((double) (((int) f - (int) (f + 1))), 0.0)),
		     kdccomp((double) (f - (double) (int) f), 0.0)));
	    else
	       _odata[i] = _idata[inum - 1];
	 }
      }
   default:
      return (FALSE);
   }
   return (TRUE);
}
/*-----------------------------------------------------------
|
|  Routine Name: _data_sample_pad
|
|       Purpose: This routine is used to to pad data instead
|		 of interpolating it.
|
|		 This function assumes that the input and
|		 output pointers have been checked and
|		 are valid, and that the input and output
|		 dimensions are positive.
|
|         Input: idata - input data vector
|                inum  - the number of elements in idata
|		 onum  - the desired number elements in
|			 odata.
|		 dtype - data type of incoming and outgoing
|			 data vectors.
|
|        Output: odata - the output data vector
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Oct 01, 1992 14:22
| Modifications:
|
------------------------------------------------------------*/

static int 
_data_sample_pad(char *idata, int inum, int onum, int dtype, double rpad,
    double ipad, char *odata)
{
   int i;			/* an index for a loop */
   register double f = 0.0;	/* a float index for sampling */
   register double delta;	/* a increment factor for samplint */
   int size;

   /*
    * determine the size of the requested data type for the current machine.
    * If the current machine is unknown, or the data type is illegal, return
    * a fail without issuing an error message.
    */
   if (dtype == KBIT)
      size = 0;
   else if ((size = kdata_size(dtype)) == 0)
      return (FALSE);

   /*
    * if the number of input data points is equal to the number of output
    * data points, then there is really nothing to be done.  This is ok,
    * though, so we will simply copy the data from the input to the output
    * and return
    */
   if (inum == onum) {
      if (size != 0)
	 memcpy(odata, idata, inum * size);
      else
	 memcpy(odata, idata, ((inum >> 3) + ((inum % 8) != 0)));

      return (TRUE);
   }

   /*
    * compute a floating point "skip factor".  If this value is less than 1.0
    * then we are supersampling (or a proper interpolation), otherwise we are
    * subsampling.
    */
   delta = (double) (inum) / (double) (onum);

   /*
    * depending on data type, call the first order interpolation with the
    * appropriate flags.
    */
   switch (dtype) {
   case KBIT:
      {
	 unsigned char *optr = (unsigned char *) odata;
	 unsigned char *iptr = (unsigned char *) idata;
	 unsigned char mask, maskout;

	 for (i = 0; i < (onum >> 3) + ((onum % 8) != 0); i++)
	    optr[i] = 0;
	 for (i = 0; i < onum; i++) {
	    maskout = 0x80 >> (i % 8);
            if(i<inum){
	       mask = 0x80 >> (i % 8);
	       optr[(int) (i / 8)] |= (unsigned char)
	          ((iptr[(int) (i / 8)] & mask) ? maskout : 0x0);
            } else {
               optr[(int) (i / 8)] |= (unsigned char)
		  ((rpad == 0.0) ? 0x0 : maskout);
            }
	 }
      }
      break;
   case KBYTE:
      {
	 for (i = 0, f = 0.0; i < onum; i++, f += delta) {
	    if(i < inum){
               odata[i] = idata[i];
            } else {
               odata[i] = (char)rpad;
            }
	 }
      }
      break;
   case KUBYTE:
      {
	 unsigned char *_idata = (unsigned char *) idata;
	 unsigned char *_odata = (unsigned char *) odata;

	 for (i = 0, f = 0.0; i < onum; i++, f += delta) {
	    if(i < inum){
               _odata[i] = _idata[i];
            } else {
               _odata[i] = (unsigned char)rpad;
            }
	 }
      }
      break;
   case KSHORT:
      {
	 short *_idata = (short *) idata;
	 short *_odata = (short *) odata;

	 for (i = 0, f = 0.0; i < onum; i++, f += delta) {
	    if(i < inum){
               _odata[i] = _idata[i];
            } else {
               _odata[i] = (short)rpad;
            }
	 }
      }
      break;
   case KUSHORT:
      {
	 unsigned short *_idata = (unsigned short *) idata;
	 unsigned short *_odata = (unsigned short *) odata;

	 for (i = 0, f = 0.0; i < onum; i++, f += delta) {
	    if(i < inum){
               _odata[i] = _idata[i];
            } else {
               _odata[i] = (unsigned short)rpad;
            }
	 }
      }
      break;
   case KLONG:
      {
	 long *_idata = (long *) idata;
	 long *_odata = (long *) odata;

	 for (i = 0, f = 0.0; i < onum; i++, f += delta) {
	    if(i < inum){
               _odata[i] = _idata[i];
            } else {
               _odata[i] = (long)rpad;
            }
	 }
      }
      break;
   case KULONG:
      {
	 unsigned long *_idata = (unsigned long *) idata;
	 unsigned long *_odata = (unsigned long *) odata;

	 for (i = 0, f = 0.0; i < onum; i++, f += delta) {
	    if(i < inum){
               _odata[i] = _idata[i];
            } else {
               _odata[i] = (unsigned long)rpad;
            }
	 }
      }
      break;
   case KINT:
      {
	 int *_idata = (int *) idata;
	 int *_odata = (int *) odata;

	 for (i = 0, f = 0.0; i < onum; i++, f += delta) {
	    if(i < inum){
               _odata[i] = _idata[i];
            } else {
               _odata[i] = (int)rpad;
            }
	 }
      }
      break;
   case KUINT:
      {
	 unsigned int *_idata = (unsigned int *) idata;
	 unsigned int *_odata = (unsigned int *) odata;

	 for (i = 0, f = 0.0; i < onum; i++, f += delta) {
	    if(i < inum){
               _odata[i] = _idata[i];
            } else {
               _odata[i] = (unsigned int)rpad;
            }
	 }
      }
      break;
   case KFLOAT:
      {
	 float *_idata = (float *) idata;
	 float *_odata = (float *) odata;

	 for (i = 0, f = 0.0; i < onum; i++, f += delta) {
	    if(i < inum){
               _odata[i] = _idata[i];
            } else {
               _odata[i] = (float)rpad;
            }
	 }
      }
      break;
   case KDOUBLE:
      {
	 double *_idata = (double *) idata;
	 double *_odata = (double *) odata;

	 for (i = 0, f = 0.0; i < onum; i++, f += delta) {
	    if(i < inum){
               _odata[i] = _idata[i];
            } else {
               _odata[i] = rpad;
            }
	 }
      }
      break;
   case KCOMPLEX:
      {
	 kcomplex *_idata = (kcomplex *) idata;
	 kcomplex *_odata = (kcomplex *) odata;

	 for (i = 0, f = 0.0; i < onum; i++, f += delta) {
	    if(i < inum){
               _odata[i] = _idata[i];
            } else {
               _odata[i] = kccomp((float)rpad,(float)ipad);
            }
	 }
      }
      break;
   case KDCOMPLEX:
      {
	 kdcomplex *_idata = (kdcomplex *) idata;
	 kdcomplex *_odata = (kdcomplex *) odata;

	 for (i = 0, f = 0.0; i < onum; i++, f += delta) {
	    if(i < inum){
               _odata[i] = _idata[i];
            } else {
               _odata[i] = kdccomp((double)rpad,(double)ipad);
            }
	 }
      }
   default:
      return (FALSE);
   }
   return (TRUE);
}



/*-----------------------------------------------------------
|
|  Routine Name: _data_sample_zero_order
|
|       Purpose: This routine casts data from a unsigned byte
|                representation into any of the other known
|                data types.
|
|		 Input and output strides are provided to
|		 allow for arbitrary indexing through the
|		 input and output vectors.  Both of these
|		 strides are of type double because this
|		 allows a primitive sub- or super-sampling
|		 capability.
|
|         Input: in         - unsigned byte data
|                num_elem   - number of signed bytes in
|                             input and output vectors.
|                odatatype - desired output data type.
|		 istride    - a double that describes the
|			      stride or skip factor to be
|			      used when indexing through
|			      the input.
|		 ostride    - a double that describes the
|			      stride or skip factor to be
|			      used when indexing through
|			      the output.
|
|        Output: out        - vector of data after it
|                             has been cast to the new
|                             data type.
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Aug 26, 1992 16:31
| Modifications:
|
------------------------------------------------------------*/

static int 
_data_sample_zero_order(
  char *in, 
  int num_elem, 
  int datatype, 
  int convert, 
  double istride, 
  double ioffset,
  char *out)
{
   register int i;
   unsigned char mask;

   switch (datatype) {
   case KBIT:
      {
	 unsigned char *optr = (unsigned char *) out;
	 unsigned char *iptr = (unsigned char *) in;
	 mask = 0x80;
	 for (i = 0; i < (num_elem >> 3) + ((num_elem % 8) != 0); i++)
	    optr[i] = 0;
	 for (i = 0; i < num_elem; i++) {
	    mask = 0x80 >> (i % 8);
	    optr[i / 8] |= 
	       (iptr[_do_index(istride, ioffset)] == 0) ? 0x0 : mask;
	 }
      }
      break;
   case KBYTE:
      _do_zero_interp(out, in, char)
      break;
   case KUBYTE:
      _do_zero_interp(out, in, unsigned char)
      break;
   case KSHORT:
      _do_zero_interp(out, in, short)
      break;
   case KUSHORT:
      _do_zero_interp(out, in, unsigned short)
      break;
   case KLONG:
      _do_zero_interp(out, in, long)
      break;
   case KULONG:
      _do_zero_interp(out, in, unsigned long)
      break;
   case KINT:
      _do_zero_interp(out, in, int)
      break;
   case KUINT:
      _do_zero_interp(out, in, unsigned int)
      break;
   case KFLOAT:
      _do_zero_interp(out, in, float)
      break;
   case KDOUBLE:
      _do_zero_interp(out, in, double)
      break;
   case KCOMPLEX:
      _do_zero_interp(out, in, kcomplex)
      break;
   case KDCOMPLEX:
      _do_zero_interp(out, in, kdcomplex)
      break;
   default:
      errno = KINVALID_DATATYPE;
      kerror("kmath", "kdata_cast()",
	 "The output data type is not recognized (%d).", datatype);
      return (FALSE);
   }
   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: kdata_sample
|
|       Purpose: This routine is used to perform an N-d
|		 interpolation or resample of the input
|		 data.  This allows you to simulate both
|		 sub-sampling and super-sampling.
|
|         Input: idata - input data to be re-sampled
|                inum  - the number of elements in idata
|		 onum  - the desired number elements in
|			 odata.
|		 dtype - data type of both the incoming and
|			 the outgoing data vectors.
|		 convert - complex convert mode.
|		 order - desired order of interpolation.
|
|        Output: odata - the output data vector
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|  Restrictions:
|    Written By: Jeremy Worley
|          Date: Oct 02, 1992 17:35
|      Verified:
|  Side Effects:
| Modifications:
|
------------------------------------------------------------*/

int 
kdata_sample(
   kaddr	idata, 
   int		inum, 
   int		onum, 
   int		dtype, 
   int		order,
   int		convert,
   double	rpad,
   double	ipad, 
   double       ioffset,
   double       ooffset,
   kaddr	odata)
{

   /*
    * Call the appropriate routine based on the order of interpolation
    * desired.  NOTE:  Only 0 order and 1st order interpolation are currently
    * present.
    */
   switch (order) {
   case KPAD: 
       return(_data_sample_pad(idata,inum,onum,dtype,rpad,ipad,odata));
   case KZERO_ORDER:  
       return (_data_sample_zero_order(idata, onum, dtype, convert, 
				       (double) inum / (double) onum,
				       ioffset, odata));
   case KFIRST_ORDER:  
       return (_data_sample_1st(idata, inum, onum, dtype, odata));
   default:
      return (FALSE);
   }
}

/*-----------------------------------------------------------
|
|  Routine Name: kdata_resample
|
|       Purpose: This routine is used to perform an N-d
|		 interpolation or resample of the input
|		 data.  This allows you to simulate both
|		 sub-sampling and super-sampling.
|
|         Input: idata - input data to be re-sampled
|                inum  - the number of elements in idata
|		 onum  - the desired number elements in
|			 odata.
|		 factor - scale factor
|		 dtype - data type of both the incoming and
|			 the outgoing data vectors.
|		 convert - complex convert mode.
|		 order - desired order of interpolation.
|
|        Output: odata - the output data vector
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|  Restrictions:
|    Written By: Jeremy Worley
|          Date: Oct 02, 1992 17:35
|      Verified:
|  Side Effects:
| Modifications:
|
------------------------------------------------------------*/

int 
kdata_resample(
   kaddr	idata, 
   int		inum, 
   int		onum, 
   int		dtype, 
   int		order,
   int		convert,
   double       factor,
   double	rpad,
   double	ipad, 
   double       ioffset,
   double       ooffset,
   kaddr	odata)
{

   /*
    * Call the appropriate routine based on the order of interpolation
    * desired.  NOTE:  Only 0 order and 1st order interpolation are currently
    * present.
    */
   switch (order) {
   case KPAD: 
       return(_data_sample_pad(idata,inum,onum,dtype,rpad,ipad,odata));
   case KZERO_ORDER:  
      return (_data_sample_zero_order(idata, onum, dtype, convert, factor,
				      ioffset, odata));
   case KFIRST_ORDER:  
       return (_data_sample_1st(idata, inum, onum, dtype, odata));
   default:
      return (FALSE);
   }
}
