/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>       Primitive Matrix and Vector Processing
   >>>>
   >>>>  Private:
   >>>>
   >>>>   Static:
   >>>>            _mult3x3()
   >>>>            _mult4x4()
   >>>>            _inverse3x3()
   >>>>   Public:
   >>>>             kdmatrix_inner_prod()
   >>>>             kdmatrix_vector_prod()
   >>>>             kdmatrix_princ_axis()
   >>>>             kdmatrix_multiply()
   >>>>             kdmatrix_clear()
   >>>>             kdmatrix_identity()
   >>>>             kdmatrix_inverse()
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"


/************************************************************
*
*  Routine Name: kdmatrix_inner_prod() - compute the inner product of
*                           two vectors.
*
*       Purpose: kdmatrix_inner_prod() computes the inner product of two 
*                vectors.
*
*         Input: x - first vector.
*                y - second vector.
*                n - number of components in each vector.
*
*        Output: 
*                RETURNS: The double precision inner product
*                of the two vectors.
*
*  Restrictions: 
*    Written By: Scott Wilson
*          Date: Jan 04, 1991 16:02
*      Verified:
*  Side Effects:
* Modifications: Converted from inner in Khoros 1.0 (JW)
*
*************************************************************/

double kdmatrix_inner_prod(
   double *x,
   double *y,
   int    n)
{
  int i;
  double a;

  a = 0;
  for (i=0; i<n; i++) a += x[i]*y[i];
  return(a);
}

/************************************************************
*
*  Routine Name: kdmatrix_vector_prod() - compute the matrix-vector
*                            product.
*
*       Purpose: kdmatrix_vector_prod() computes a matrix-vector product.
*
*         Input: a    - input matrix stored in 1D array of doubles
*                x    - input vector
*                rows - number of rows in matrix a as well as
*                       the number of elements in x.
*                cols - number of columns in matrix a
*
*        Output: y    - the output vector containing the matrix-vector
*		 	product
*                RETURNS: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Scott Wilson
*          Date: Jan 04, 1991 16:20
*      Verified:
*  Side Effects:
* Modifications: Converted from matvec() in Khoros 1.0 (JW)
*
*************************************************************/

int kdmatrix_vector_prod(
   double *a,
   double *x,
   int    rows,
   int    cols,
   double *y)
{
  int i;

  for (i=0; i<rows; i++) y[i] = kdmatrix_inner_prod(a+i*cols,x,cols);

  return(TRUE);
}

/************************************************************
*
*  Routine Name: kdmatrix_princ_axis - obtain the principle
*                                  axis of a covariance
*                                  matrix.
*
*       Purpose: kdmatrix_princ_axis() obtains the principal axis 
*                (the eigenvector associated with the largest 
*                eigenvalue) for the covariance matrix "a" of 
*                size n by n. Put the principal eigenvector 
*                in the place pointed to by "y". 
*
*                The power iteration method to obtain the 
*                dominant eigenvalue and its associated 
*                eigenvector as described in Gollub and 
*                VanLoan, MATRIX COMPUTATIONS, pp 209.    
*
*         Input: a - input matrix stored in 1D array of doubles
* 		 n - size of matrix (assumed square. n x n).
*                
*        Output: y - the output vector (axis)
*
*       Returns: TRUE (1) on success, otherwise it returns FALSE (0)
*		 if this function is unable to allocate sufficient 
*		 memory for the operation.
*
*  Restrictions: Since a is a covariance matrix it is 
*                symmetric, real, and very likely to be 
*                positive definate. It is positive semidefinate 
*                for sure, and may also be diagonal dominant.
*
*                Other possible difficulties: The major hitch 
*                in this technique is that the convergence is 
*                proportional to lambda(2)/lambda(1) where 
*                lambda(1) is the dominant eigenvector. If 
*                these eigenvalues are closely spaced then we 
*                won't get a decent eigenvector (it will have 
*                an incorrect direction).
*
*                Fortunately, when using the principal axis to
*                split a cluster and there are two very
*                strongly dominant axes with the same
*                ellipticity, then we can split on any 
*                combination of those axis and reduce the 
*                cluster variances greatly.
*
*                The 10 iterations used in the code have been 
*                found to be satisfactory for all of the data 
*                so far encountered unless it is an ugly special case.
*
*    Written By: Scott Wilson
*          Date: Jan 04, 1991 16:30
*      Verified:
*  Side Effects:
* Modifications: Converted from get_princ_axis() in Khoros 1.0
*                (JW)
*
*************************************************************/

int kdmatrix_princ_axis(
   double *a,
   int    n,
   double *y)
{
  double *b,*c,lambda;
  int i,j;

  if((c = (double *)kmalloc(n*sizeof(double)))==NULL){
     kerror("kmath","kget_princ_axis",
            "Unable to allocate sufficient memory for this operation");
     return(FALSE);
  }

  if((b = (double *)kmalloc(n*sizeof(double)))==NULL){
     kerror("kmath","kget_princ_axis",
            "Unable to allocate sufficient memory for this operation");
     kfree(c);
     return(FALSE);
  }
    
  for(i=0; i<n; i++) c[i] = 1.0;
   
  for(j=0; j<10; j++){
     kdmatrix_vector_prod(a,c,n,n,b);
     lambda = sqrt(kdmatrix_inner_prod(b,b,n)); 
     for (i=0; i<n; i++) c[i] = b[i]/lambda;
  }
    
  for (i=0; i<n; i++) y[i] = c[i];

  kfree(c);
  kfree(b);

  return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: _mult3x3 - multiply two 3x3 matrices
|
|       Purpose: _mult3x3 multiplies two 3x3 matrices
|                together and returns the result
|
|         Input: matrix1   - first matrix
|                matrix2   - second matrix
|
|        Output: outmat - resulting matrix
|                RETURNS: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: Jul 15, 1992 15:54
| Modifications: 
|
------------------------------------------------------------*/

static int _mult3x3(
   double *outmat,
   double *matrix1,
   double *matrix2)
{
   double tmp_matrix[9];
   int i,j;

   for(i = 0; i < 3; i++){
       for(j = 0; j < 3; j++){
           tmp_matrix[i*3+j]  = matrix1[i*3]    * matrix2[j] +
                                matrix1[i*3 + 1] * matrix2[j+3] +
                                matrix1[i*3 + 2] * matrix2[j+6];
       }
   }

  /*
   *  Copy the temporary matrix to the final matrix (outmatrix).  We 
   *  could have put this directly into the final matrix but this enables
   *  the user to use the output matrix as either matrix1 or matrix2.
   *  ie)
   *      matrix1 = matrix1 * matrix2
   */
        
   memcpy((char *)outmat,(char *)tmp_matrix, 9*sizeof(double));
   return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: _mult4x4 - multiply two 4x4 matrices
|
|       Purpose: _mult4x4 multiplies two 4x4 matrices
|                together and returns the result
|
|         Input: matrix1   - first matrix
|                matrix2   - second matrix
|
|        Output: outmat - resulting matrix
|                RETURNS: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: Jul 15, 1992 15:54
| Modifications: 
|
------------------------------------------------------------*/

static int _mult4x4(
   double *outmat,
   double *matrix1,
   double *matrix2)
{
   double tmp_matrix[16];
   int i,j;

   for(i = 0; i < 4; i++){
       for(j = 0; j < 4; j++){
           tmp_matrix[i*4 + j]  = matrix1[i*4] * matrix2[j] +
                                  matrix1[i*4 + 1] * matrix2[j + 4] +
                                  matrix1[i*4 + 2] * matrix2[j + 8] +
                                  matrix1[i*4 + 3] * matrix2[j + 12];
       }
   }

  /*
   *  Copy the temporary matrix to the final matrix (outmatrix).  We 
   *  could have put this directly into the final matrix but this enables
   *  the user to use the output matrix as either matrix1 or matrix2.
   *  ie)
   *      matrix1 = matrix1 * matrix2
   */
   memcpy( (char *) outmat, (char *) tmp_matrix, 16*sizeof(double));
   return(TRUE);
}

/************************************************************
*
*  Routine Name: kdmatrix_multiply() - multiply two matrices
*
*       Purpose: kdmatrix_multiply() multiplies two arbitrary
*                matrices.  The input matrices are expected
*                to be organized in a 1 dimensional array
*                as consecutive rows.  The result is 
*                stored in the same format and has the
*                dimensions rows1 * cols2. The result is returned in 
*                outmat.
*
*         Input: matrix1 - input matrix stored in 1D array of doubles
*                rows1   - number of rows in the first 
*                          matrix.
*                cols1   - number of columns in the first
*                          matrix.
*                matrix2 - input matrix stored in 1D array of doubles
*                rows2   - number of rows in the second
*                          matrix.
*                cols2   - number of columns in the second
*                          matrix.
*
*        Output: outmat  - the output matrix.  Its 
*                          dimension will be rows1 * cols2.
*       RETURNS: TRUE (1) on success, otherwise it returns FALSE (0)
*		 and kerrno is set to KLIMITATION if the matrices are
*		 not 3x3 or 4x4.
*
*  Restrictions: This function really only works on when the
*                two matrices are either 3x3 or 4x4.
*    Written By: Jeremy Worley
*          Date: Jul 15, 1992 16:22
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int kdmatrix_multiply(
   double *matrix1,
   int    rows1,
   int    cols1,
   double *matrix2,
   int    rows2,
   int    cols2,
   double *outmat)
{
  if(rows1 == rows2 && cols1 == cols2){
     if(rows1 == 3 && cols1 == 3){
        _mult3x3(outmat,matrix1,matrix2);
     }else if(rows1 == 4 && cols1 == 4){
        _mult4x4(outmat,matrix1,matrix2);
     }
     return(TRUE);
  }else{
     errno = KLIMITATION;
     kerror("kmath","kdmatrix_multiply",
 "This routine is presently only capable of multiplying 3x3 and 4x4 matrices");
     return(FALSE);
  }
}

/************************************************************
*
*  Routine Name: kdmatrix_clear() - zeros a matrix
*
*       Purpose: kdmatrix_clear() clears a matrix by setting
*                all of its values to 0.0.
*
*         Input: rows   - number of rows in the matrix.
*                cols   - number oc columns in the matrix.
*
*        Output: matrix - the cleared matrix.
*       RETURNS: TRUE (1) on success, otherwise it returns FALSE (0)
*		 and kerrno is set to KINVALID_PARAMETER if the
*		 dimensions of the input matrix are not positive
*		 and non-zero.
*
*  Restrictions: 
*    Written By: Jeremy Worley
*          Date: Jul 15, 1992 17:01
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int kdmatrix_clear(
   int    rows,
   int    cols,
   double *matrix)
{
   int i;

   if(rows<=0 || cols<=0){
errno = KINVALID_PARAMETER;
      kerror("kmath","kdmatrix_clear",
         "The dimensions of the input matrix must be positive and non-zero");
      return(FALSE);
   }

   for (i = 0; i < rows*cols; i++)matrix[i] = 0.0;
   return(TRUE);
}

/************************************************************
*
*  Routine Name: kdmatrix_identity() - set matrix to identity
*
*       Purpose: kdmatrix_identity() sets an input matrix
*                to the identity matrix.
*
*         Input: rows   - the number of rows in the matrix.
*                cols   - the number of columns in the 
*                         matrix.
*
*        Output: matrix - output matrix stored in 1D array of doubles
*
*       Returns: TRUE (1) on success, otherwise this function returns
*		 FALSE (0) and kerrno is set to KINTERNAL if the internal
*		 call to kdmatrix_clear failed.
*
*  Restrictions: 
*    Written By: Jeremy Worley
*          Date: Jul 15, 1992 17:05
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int kdmatrix_identity(
   int    rows,
   int    cols,
   double *matrix)
{
   int i,j;

   if(!kdmatrix_clear(rows,cols,matrix)){
errno = KINTERNAL;
      kerror("kmath","kdmatrix_identity",
         "An internal call to kdmatrix_clear failed.");
      return(FALSE);
   }

   for(i=0;i<rows;i++){
       for(j=0;j<cols;j++){
           if(i==j)matrix[i*cols + j] = 1.0;
       }
   }

   return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: _inverse3x3() - computes the inverse of a 
|                3x3 matrix.
|
|       Purpose: _inverse3x3() computes the inverse of the
|                input matrix and places it in outmatrix.
|
|         Input: matrix - the input matrix.
|
|        Output: outmatrix - the output matrix.
|                RETURNS: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: Jul 15, 1992 17:19
| Modifications:
|
------------------------------------------------------------*/

static int _inverse3x3(
   double *outmatrix,
   double *matrix)
{
  double tmp[9], det;

  det =   matrix[0]*matrix[5]*matrix[7]
        - matrix[0]*matrix[4]*matrix[8]
        + matrix[3]*matrix[1]*matrix[8]
        - matrix[3]*matrix[2]*matrix[7]
        - matrix[6]*matrix[2]*matrix[5]
        + matrix[6]*matrix[2]*matrix[4];

   if(det <= 0.0){ 
errno = KNUMERIC;
      kerror("kmath","kdmatrix_inverse",
         "The input matrix is singular and cannot be inverted.");
      return(FALSE);
   }

   tmp[0] = (matrix[5]*matrix[7] - matrix[4]*matrix[8])/det;
   tmp[1] = (matrix[1]*matrix[8] - matrix[2]*matrix[7])/det;
   tmp[2] = (matrix[2]*matrix[4] - matrix[1]*matrix[5])/det;

   tmp[3] = (matrix[3]*matrix[8] - matrix[5]*matrix[6])/det;
   tmp[4] = (matrix[2]*matrix[6] - matrix[0]*matrix[8])/det;
   tmp[5] = (matrix[0]*matrix[5] - matrix[2]*matrix[3])/det;

   tmp[6] = (matrix[4]*matrix[6] - matrix[3]*matrix[7])/det;
   tmp[7] = (matrix[0]*matrix[7] - matrix[1]*matrix[6])/det;
   tmp[8] = (matrix[1]*matrix[3] - matrix[0]*matrix[4])/det;

   memcpy((char *) outmatrix, (char *) tmp, 9*sizeof(double));
   return(TRUE);
}

/************************************************************
*
*  Routine Name: kdmatrix_inverse() - inverts a matrix.
*
*       Purpose: kdmatrix_inverse() - returns the inverse of
*                the input matrix.
*
*         Input: matrix - input matrix stored in 1D array of doubles
*                order  - order of the matrix.
*
*        Output: outmatrix - the inverted matrix stored in 1D array
*	      		     of doubles.
*
*       Returns: TRUE (1) on success, otherwise it returns FALSE (0)
*		 and kerrno is set to KNUMERIC if the input matrix
*		 is singular and cannot be inverted.
*
*  Restrictions: 
*    Written By: Jeremy Worley and Wes Bethel
*          Date: Jul 15, 1992 17:12
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int kdmatrix_inverse(
   double *matrix,
   int    order,
   double *outmatrix)
{
   if(order==3){
      if(!_inverse3x3(outmatrix,matrix)){
         /* error message was generated by _inverse3x3 */
         return(FALSE);
      }
   }
   else
   {
      double *wmatrix;
      double *work;
      double *det;
      int    *pivot;
      int     info;
      

      /* -- allocate space for work -- */
      wmatrix = kmalloc(order*order*sizeof(double));
      work    = kmalloc(order*2*sizeof(double));
      det     = kmalloc(order*2*sizeof(double));
      pivot   = kmalloc(order*sizeof(int));

      /* -- make a working copy of the matrix -- */
      kmemcpy((char *)wmatrix,(char *)matrix, order*order*sizeof(double));

      /* -- determine the inverse -- */
      info = klin_dgefa(wmatrix, order, order, pivot);
      if (info != 0)
      {
         /* -- tidy up -- */
         kfree(work);
         kfree(det);
         kfree(pivot);
         kfree(wmatrix);
	 return FALSE;  /* -- not really an error -- */
      }

      klin_dgedi(wmatrix, order, order, pivot, KLIN_INVERSE, work, det);

      /* -- return the output -- */ 
      kmemcpy((char *)outmatrix, (char *)wmatrix, order*order*sizeof(double));
      
      /* -- tidy up -- */
      kfree(work);
      kfree(det);
      kfree(pivot);
      kfree(wmatrix);
   }

   return(TRUE);
}
