/*
 * Khoros: $Id$
 */
 
#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif
 
/*
 * $Log$
 */
 
/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */
 
 
/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>  Double Precision General Matrix and Vector Processing
   >>>>
   >>>>	         - adapted from linpack and blas -
   >>>>
   >>>>   Static:
   >>>>			_klin_is_amax()
   >>>>  Private:
   >>>>
   >>>>   Public:
   >>>>			kblas_dscal() 
   >>>>			kblas_daxpy() 
   >>>>			kblas_dswap() 
   >>>>			klin_dgefa()  
   >>>>			klin_dgedi()  
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"

/*-----------------------------------------------------------
|
|  Routine Name: (static) _kblas_is_amax
|
|       Purpose: This routine finds the index of sx which
|	 	 has the maximum absolute value. 
|
|		 The functioning of this code for 
|		 incx != 1 appears to be questionable.
|
|         Input: sx   - linear array to search
|		 n    - number of bins in array to check
|		 incx - increment to use when searching
|        Output: none
|
|       Returns: the index into the array where the maximum absolute 
|		 value is stored on success, -1 on failure
|		 (This actually appears to be the bin number).
|    Written By: Jack Dongarra (adapted from blas by Steve Kubica)
|          Date: Aug 04, 1994 11:44
| Modifications:
|
------------------------------------------------------------*/
static int
_kblas_is_amax(
   double *sx,
   int    n,
   int    incx)
{
   double smax;
   int   i;
   int   ix = 0;
   int   return_value = 0;
   

   if (n < 1)
      return -1;

   if (n == 1)
      return 0;

   if (incx != 1)
   {
      /* -- code for increment not equal to 1 -- */
      smax = kabs(sx[0]);
      ix += incx;
      for (i = 1; i < n; i++, ix += incx)
	 if (kabs(sx[ix]) > smax)
	 {
	    return_value = i; 
	    smax = kabs(sx[ix]);
	 }
   }
   else
   {
      /* -- code for both increments equal to 1 -- */
      smax = kabs(sx[0]);
      for (i = 1; i < n; i++)
	 if (kabs(sx[i]) > smax)
	 {
	     return_value = i;
	     smax = kabs(sx[i]);
	 }
   }

   return return_value;
}

/************************************************************
*
*  Routine Name: kblas_dscal - scale a double vector
*
*       Purpose: This routine multiplies a vector by a constant.
*	 	 It uses unrolled loops for increments == 1.
*
*         Input: sx   - linear array to scale
*		 sa   - scalar multiplier for sx array
*		 n    - number of bins in array to scale
*		 incx - increment to use on sx array
*
*        Output: sx   - computation is done in place
*       Returns: none
*  Restrictions:
*    Written By: Jack Dongarra (adapted from blas by Steve Kubica)
*          Date: Aug 04, 1994 17:28
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: void kblas_dscal(
*	       	!    double *sx,
*	       	!    double  sa,
*	       	!    int    n,
*	       	!    int    incx)
*
*************************************************************/
void
kblas_dscal(
   double *sx,
   double  sa,
   int    n,
   int    incx)
{
   int nincx;
   int i;
   int ix;
   int iy;
   int m;
   
   
   if (n < 0) 
      return;
 
   if (incx != 1)
   {
      /* -- code for increment not equal to 1 -- */
      nincx = n * incx;
      for (i = 0; i < nincx; i += incx)
      {
	 sx[i] = sa * sx[i];
      }
   }
   else
   {
      /* -- code for increment equal to 1 -- */
      m = n % 5;
      if (m != 0)
      {
	 for (i = 0; i < m; i++)
	    sx[i] = sa*sx[i];
	 if (n < 5)
	    return;
      }
      for (i = m; i < n; i += 5)
      {
	 sx[i]   = sa*sx[i];
	 sx[i+1] = sa*sx[i+1];
	 sx[i+2] = sa*sx[i+2];
	 sx[i+3] = sa*sx[i+3];
	 sx[i+4] = sa*sx[i+4];
      }
   }
   
   return;
}

/************************************************************
*
*  Routine Name: kblas_daxpy - add two double vectors while scaling one
*
*       Purpose: This routine multiplies a vector by a constant
*		 and then adds it to another vector. The calculation
*		 will be : SY = SY + sa*SX
*	 	 It uses unrolled loops for increments == 1.
*
*         Input: sx   - linear array to process
*		 sy   - linear array to add to add to sx*sa
*		 sa   - scalar multiplier for sx array
*		 n    - number of bins in array to operate on
*		 incx - increment to use on sx array
*		 incy - increment to use on sy array
*        Output: sy   - computation is done in place
*
*      Returns: none
*  Restrictions:
*    Written By: Jack Dongarra (adapted from blas by Steve Kubica)
*          Date: Aug 04, 1994 17:30
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: void kblas_dscal(
*	       	!    double *sx,
*	       	!    double *sy,
*	       	!    double  sa,
*	       	!    int    n,
*	       	!    int    incx,
*	       	!    int    incy)
*
*************************************************************/
void
kblas_daxpy(
   double *sx,
   double *sy,
   double  sa,
   int    n,
   int    incx,
   int    incy)
{
   int i;
   int ix = 0;
   int iy = 0;
   int m;
   

   if (n < 0 || sa == 0.0)
      return;

   if (incx != 1 || incy != 1)
   {
      /* -- code for unequal increments or equal increments != 1 -- */
   
      if (incx < 0) 
	 ix = (-n+1)*incx;
      if (incy < 0) 
	 iy = (-n+1)*incy;
      
      for (i = 0; i < n; i++, ix += incx, iy += incy)
	 sy[iy] = sy[iy] + sa*sx[ix];
   }
   else
   {
      /* -- code for increments == 1 -- */

      /* -- clean-up loop -- */

      m = n % 4;
      if (m != 0)           
      {
	 for (i = 0; i < m; i++)
	    sy[i] = sy[i] + sa*sx[i];
	 if (n < 4)
	    return;
      }
      for (i = m; i < n; i+=4) 
      {
	 sy[i]   = sy[i]   + sa*sx[i];
	 sy[i+1] = sy[i+1] + sa*sx[i+1];
	 sy[i+2] = sy[i+2] + sa*sx[i+2];
	 sy[i+3] = sy[i+3] + sa*sx[i+3];
      }
   }
   
   return;
}

/************************************************************
*
*  Routine Name: kblas_dswap - swap two double vectors
*
*       Purpose: This routine interchanges two vectors.
*	 	 It uses unrolled loops for increments == 1.
*
*         Input: sx   - linear array to search
*		 n    - number of bins in array to swap
*		 incx - increment to use when searching
*        Output: sx and sy are swapped
*
*       Returns: none
*  Restrictions:
*    Written By: Jack Dongarra (adapted from blas by Steve Kubica)
*          Date: Aug 04, 1994 17:33
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: void kblas_dswap(
*	       	!    double *sx,
*	       	!    double *sy,
*	       	!    int    n,
*	       	!    int    incx,
*	       	!    int    incy)
*
*************************************************************/
void
kblas_dswap(
   double *sx,
   double *sy,
   int    n,
   int    incx,
   int    incy)
{
   double stemp;
   int i;
   int ix = 0;
   int iy = 0;
   int m;

   
   if (n < 0)
      return;
   
   if (incx != 1 || incy != 1)
   {
      /* -- code for unequal increments of equal increments != 1 -- */
      
      if (incx < 0) 
	 ix = (-n+1)*incx;
      if (incy < 0) 
	 iy = (-n+1)*incy;

      for (i = 0; i < n; i++, ix += incx, iy += incy)
      {
	 stemp = sx[ix];    sx[ix] = sy[iy];    sy[iy] = stemp;
      }
   }
   else
   {
      /* -- code for both increments equal to 1 -- */
	 
      m = n % 3;
      if (m != 0)           
      {
	 for (i = 0; i < m; i++)
	 {
	    stemp = sx[i];    sx[i] = sy[i];    sy[i] = stemp;
	 }
	 if (n < 3)
	    return;
      }
      for (i = m; i < n; i+=3) 
      {
	 stemp = sx[i];    sx[i]   = sy[i];      sy[i]   = stemp;
	 stemp = sx[i+1];  sx[i+1] = sy[i+1];    sy[i+1] = stemp;
	 stemp = sx[i+2];  sx[i+2] = sy[i+2];    sy[i+2] = stemp;
      }
   }
   
   return;
}

/************************************************************
*
*  Routine Name: klin_dgefa - factors a double matrix by gaussian elimination.
*
*       Purpose: This routine is usually called by klin_sgeco, but
*		 it can be called directly with a saving in time if
*		 the rcond is not needed.  
*
*         Input: matrix - input matrix stored in a 1D array of
*			  doubles in row-major order such that
*		!              m[i][j] = a[i * cols+j]
*
*		 rows   - the number of rows in matrix.
*		 cols   - the number of cols in matrix.
*
*        Output: pivot  - the pivot vector 
*
*       Returns: 0 implies a normal value; a value of k means that U(k,k) == 0
*			  This is not an error condition for this routine,
*			  but it does indicate that klin_sgesl or 
*			  klin_dgedi will divide by zero if called.  
*			  Use the argument rcond in klin_segco for a reliable 
*		   	  indication of singularity.
*
*  Restrictions: 
*    Written By: Cleve Moler (adapted from linpack by Steve Kubica)
*
*          Date: Aug 04, 1994 11:00
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration: int klin_dgefa(
*		!    double *matrix,
*		!    int    rows,
*		!    int    cols,
*		!    int   *pivot)
*
*************************************************************/
int
klin_dgefa(
   double *matrix,
   int    rows,
   int    cols,
   int   *pivot)
{
   int i;
   int j;
   int k;
   int nm1;
   int info = 0;
   int l;
   double t;
   
   /* -- gaussian elimination with partial pivoting -- */

   if (cols > 1) 
   {
      for (k = 0; k < cols-1; k++)
      {
	 /* -- find l = pivot index -- */
	 l = _kblas_is_amax(&(matrix[k*rows+k]),cols-k-1,1) + k;
	 pivot[k] = l;
      
	 /* -- 0 pivot implies that this column is already triangularized -- */
	 if (matrix[l*rows+k] != 0.0)
	 {
	    /* -- interchange if necessary -- */
	    if (l != k)
	    {
	       t = matrix[k*rows+l];
	       matrix[k*rows+l] = matrix[k*rows+k];
	       matrix[k*rows+k] = t;
	    }
	    
	    /* -- compute multipliers -- */
	    t = -1.0/matrix[k*rows+k];
	    kblas_dscal(&(matrix[k*rows+k+1]),t,cols-k-1,1);
	    
	    /* 	-- row elimination with column indexing -- */
	    for (j = k+1; j < cols; j++)
	    {
	       t = matrix[j*rows+l];
	       if (l != k)
	       {
		  matrix[j*rows+l] = matrix[j*rows+k];
		  matrix[j*rows+k] = t;
	       }
	       kblas_daxpy(&(matrix[k*rows+k+1]),
			   &(matrix[j*rows+k+1]),t,cols-k-1,1,1);
	    }
	 }
	 else
	    info = k;
      }
   }
   
   pivot[cols-1] = cols - 1;
   
   if (matrix[(cols-1)*rows+(cols-1)] == 0.0)
      info = cols;

   return info;
}

/************************************************************
*
*  Routine Name: klin_dgedi - computes the determinate and 
*			      inverse of a matrix
*
*       Purpose: This routine will compute the determinant 
*		 and inverse of an NxN matrix using the factors 
*		 computed by klin_sgeco or klin_dgefa.
*
*		 The determinate functionality is not well tested.
*
*         Input: matrix- input matrix stored in a 1D array of
*			 doubles in row-major order such that
*		!             m[i][j] = a[i * cols+j]
*
*		 rows  - the number of rows in matrix.
*		 cols  - the number of columns in matrix.
*
*		 pivot - the pivot vector from klin_dgefa
*
*		 job  - defines what to compute:  the inverse, the
*			determinate, or both of the matrix.  
*			KLIN_INVERSE to compute inverse
*		        KLIN_DETERMINATE to compute determinate
*		        KLIN_INVERSE | KLIN_DETERMINATE to compute both
*
*        Output: matrix - inverse of the original matrix if requested,
*			  othewise unchanged.
*		 det    - determinate of original matrix if requested,
*			  otherwise not referenced.
*			    determinate = det(1)*10.0**det(2)
*			    with 1.0 < kabs(det(1)) < 10.0 or det(1) == 0.0
*
*       Returns: TRUE (1) all the time
*
*  Restrictions: A division by zero will occur if the input factor contains
*	         a zero on the diagonal and the inverse is requested.
*	         It will not occur if the subroutines are called correctly
*	         and if klin_sgeco has set rcond > 0.0 or dgefa has set
*	         info < 0.
*
*    Written By: Cleve Moler (adapted from linpack by Steve Kubica)
*
*          Date: Aug 04, 1994 11:00
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration:
*
*************************************************************/
int
klin_dgedi(
   double *matrix,
   int    rows,
   int    cols,
   int   *pivot,
   int    job,
   double *work,
   double *det)
{
   double t;
   double ten;
   int   i;
   int   j;
   int   k;
   int   kb;
   int   kp1;
   int   l;


   /* -- compute determinate -- */
   if (job & KLIN_DETERMINATE)
   {
      det[0] = 1.0;
      det[1] = 0.0;
      ten = 10.0; 
      
      for (i = 0; i < cols; i++)
      {
         if (pivot[i] != i)
            det[0] = -det[0];

         det[0] = matrix[i*rows+i]*det[0];
         
         if (det[0] != 0.0)
	 {
	    while (kabs(det[0]) < 1) 
	    {
	       det[0] = det[0]/ten;
	       det[1] = det[1] + 1.0;
	    }
	    
	    while (kabs(det[0]) >= ten) 
	    {
	       det[0] = det[0]/ten;
	       det[1] = det[1] + 1.0;
	    }
	 }
      }
   }
   
   /* -- compute inverse(u) -- */
   if (job & KLIN_INVERSE)
   {
      for (k = 0; k < cols; k++)
      {
         matrix[k*rows+k] = 1.0/matrix[k*rows+k];
         t = -matrix[k*rows+k];
         kblas_dscal(&(matrix[k*rows]),t,k,1);

         if (k < cols)
         {
            for (j = k+1; j < cols; j++)
            {
               t = matrix[j*rows+k];
               matrix[j*cols+k] = 0.0;
               kblas_daxpy(&(matrix[k*rows]),&(matrix[j*rows]),t,k+1,1,1);
            }
         }
      }

      /* -- form inverse(u)*inverse(1) -- */
      if (cols > 1)
      {
	 for (kb = 0; kb < cols; kb++)
	 {
	    k = cols - kb - 1;
	    kp1 = k + 1;
	    for (i = kp1; i < cols; i++)
	    {
	       work[i] = matrix[k*rows+i];
	       matrix[k*rows+i] = 0.0;
	    }
	    for (j = kp1; j < cols; j++)
	    {
	       t = work[j];
	       kblas_daxpy(&(matrix[j*rows]),&(matrix[k*rows]),t,cols,1,1);
	    }
	    l = pivot[k];
	    if (l != k)
	       kblas_dswap(&(matrix[k*rows]), 
			   &(matrix[l*rows]),cols,1,1);
	 }
      }
   }

   return TRUE;
}
