/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Complex libm compatibility
   >>>>
   >>>>  Private:
   >>>>             
   >>>>   Static:
   >>>>             
   >>>>   Public:
   >>>>			kdcsqrt()
   >>>>			kdcexp()
   >>>>			kdclog()
   >>>>			kdcpow()
   >>>>			kdccos()
   >>>>			kdcsin()
   >>>>			kdctan()
   >>>>			kdccosh()
   >>>>			kdcsinh()
   >>>>			kdctanh()
   >>>>
   >>>>			kcsqrt()
   >>>>			kcexp()
   >>>>			kclog()
   >>>>			kcpow()
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"


/************************************************************
*
*  Routine Name: kdcsqrt() - calculate the double precision 
*                            complex square root of a double precision 
*                            complex number.
*
*       Purpose: kdcsqrt() returns the double precision 
*                complex square root of the kdcomplex argument
*                passed as input.
*
*         Input: a - kdcomplex number
*
*        Output: 
*                
*       Returns: The kdcomplex square root of the input 
*                argument.
*
*  Restrictions: This function returns a structure, not a pointer to a structure.  This is allowed in ANSI C.  However, if you are using an older C compiler that is not fully ANSI compliant, you may have problems.
*
*    Written By: Jeremy Worley  
*
*          Date: Jul  2, 1992 11:10
*
*      Verified:
*
*  Side Effects:
*
* Modifications:
*
*************************************************************/

kdcomplex kdcsqrt(
   kdcomplex a)
{
    double ar,ai,r,w;
    kdcomplex c;

    if(a.r==0.0 && a.i == 0.0){
       c.r = 0.0;
       c.i = 0.0;
    }else{
       ar = fabs(a.r);
       ai = fabs(a.i);

       if(ar >= ai){
          r = ai/ar;
          w = sqrt(ar) * sqrt(0.5*(1.0+sqrt(1.0+r*r)));
       }else{
          r = ar/ai;
          w = sqrt(ai) * sqrt(0.5*(r+sqrt(1.0+r*r)));
       }
       if(a.r >= 0.0){
          c.r = w;
          c.i = a.i/(2.0*w);
       }else{
          c.i = (a.i >= 0) ? w : -w;
          c.r = a.i/(2*c.i);
       }
    }
    return(c);
} 

/***********************************************************
*
*  Routine Name: kdcexp - double complex exponential function
*
*       Purpose: This function returns the double complex exponential
*		 of the input argument.
*
*         Input: x - a double complex number.
*
*        Output: 
*
*       Returns: a double complex exponential of the input argument
*
*  Restrictions: 
*    Written By: Jeremy Worley
*          Date: Mar 05, 1993 12:11
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

kdcomplex kdcexp(kdcomplex x)
{
  double ex;
  kdcomplex y;

  if(x.r == 0.0){
     y.r = y.i = 0.0;
     return(y);
  }
   
  ex = exp(x.r);
  y.r = ex * cos(x.i);
  y.i = ex * sin(x.i);
 
  return(y);
}

/************************************************************
*
*  Routine Name: kdclog - double complex natural logarithm
*
*       Purpose: This function returns the double complex
*		 natural logarithm of the input argument.
*
*         Input: x - double complex argument.
*
*        Output: 
*
*       Returns: a double complex logarithm of the input argument.
*
*  Restrictions: 
*    Written By: Jeremy Worley
*          Date: Mar 05, 1993 12:19
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

kdcomplex kdclog(kdcomplex x)
{
  kdcomplex y;

  if(x.r == 0.0 && x.i == 0.0){
     y.r = y.i = 0.0;
     return(y);
  }

  y.r = log(sqrt(x.r*x.r + x.i*x.i));
  y.i = atan2(x.i,x.r);
  return(y);
}

/************************************************************
*
*  Routine Name: kdcpow() - calculate the double precision 
*                           complex power of two double precision 
*                           complex number.
*
*       Purpose: kdcpow() returns the double precision 
*                complex power of two kdcomplex arguments
*                passed in as input.
*
*         Input: x - kdcomplex number
*		 y - kdcomplex power
*
*        Output: 
*                
*       Returns: The kdcomplex power of the two input argument.
*
*  Restrictions: This function returns a structure, not a pointer to a structure.  This is allowed in ANSI C.  However, if you are using an older C compiler that is not fully ANSI compliant, you may have problems.  
*
*    Written By: Jeremy Worley  
*
*          Date: Mar 04, 1993 23:02
*
*      Verified:
*
*  Side Effects:
*
* Modifications:
*
*************************************************************/

kdcomplex kdcpow(kdcomplex x, kdcomplex y)
{
   return(kdcexp(kdcmult(kdclog(x),y)));
}

/************************************************************
*
*  Routine Name: kdccos - double complex cosine
*
*       Purpose: kdccos computes the double precision complex
*		 cosine of the input argument
*
*         Input: a - double complex value to take complex cosine
*		     of.
*
*        Output: 
*
*       Returns: the double precision complex cosine of a.
*
*  Restrictions: 
*    Written By: Jeremy Worley
*          Date: Apr 04, 1993 09:10
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

kdcomplex kdccos(kdcomplex a)
{
  return(kdccomp(kcos(a.r)*kcosh(a.i), -ksin(a.r)*ksinh(a.i)));
}


/************************************************************
*
*  Routine Name: kdcsin - double complex sine
*
*       Purpose: kdcsin computes the double precision complex
*		 sine of the input argument
*
*         Input: a - double complex value to take complex sine
*		     of.
*
*        Output: 
*
*       Returns: the double precision complex sine of a.
*
*  Restrictions: 
*    Written By: Jeremy Worley
*          Date: Apr 04, 1993 09:10
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

kdcomplex kdcsin(kdcomplex a)
{
  return(kdccomp(ksin(a.r)*kcosh(a.i), kcos(a.r)*ksinh(a.i)));
}


/************************************************************
*
*  Routine Name: kdctan - double complex tangent
*
*       Purpose: kdctan computes the double precision complex
*		 tangent of the input argument
*
*         Input: a - double complex value to take complex tangent
*		     of.
*
*        Output: 
*
*       Returns: the double precision complex tangent of a.
*
*  Restrictions: 
*    Written By: Jeremy Worley
*          Date: Apr 04, 1993 09:10
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

kdcomplex kdctan(kdcomplex a)
{
  double r = 2*a.r, i = 2*a.i;
  double d = kcos(r) + kcosh(i);
  return(kdccomp(ksin(r)/d, ksinh(i)/d));
}

/************************************************************
*
*  Routine Name: kdccosh - double complex hyperbolic cosine
*
*       Purpose: kdccosh computes the double precision complex
*		 hyperbolic cosine of the input argument
*
*         Input: a - double complex value to take complex hyperbolic
*		     cosine of.
*
*        Output: 
*
*       Returns: the double precision complex hyperbolic cosine of a.
*
*  Restrictions: 
*    Written By: Jeremy Worley
*          Date: Apr 04, 1993 09:10
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

kdcomplex kdccosh(kdcomplex a)
{
  return(kdccos(kdccomp(-a.i,a.r)));
}


/************************************************************
*
*  Routine Name: kdcsinh - double complex hyperbolic sine
*
*       Purpose: kdcsinh computes the double precision complex
*		 hyperbolic sine of the input argument
*
*         Input: a - double complex value to take complex hyperbolic
*		     sine of.
*
*        Output: 
*
*       Returns: the double precision complex hyperbolic sine of a.
*
*  Restrictions: 
*    Written By: Jeremy Worley
*          Date: Apr 04, 1993 09:10
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

kdcomplex kdcsinh(kdcomplex a)
{
  kdcomplex x = kdcsin(kdccomp(-a.i,a.r));
  return(kdccomp(x.i,-x.r));
}

/************************************************************
*
*  Routine Name: kdctanh - double complex hyperbolic tangent
*
*       Purpose: kdctanh computes the double precision complex
*		 hyperbolic tangent of the input argument
*
*         Input: a - double complex value to take complex hyperbolic
*		     tangent of.
*
*        Output: 
*
*       Returns: the double precision complex hyperbolic tangent of a.
*
*  Restrictions: 
*    Written By: Jeremy Worley
*          Date: Apr 04, 1993 09:10
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

kdcomplex kdctanh(kdcomplex a)
{
  kdcomplex x = kdctan(kdccomp(-a.i,a.r));
  return(kdccomp(x.i,-x.r));
}


/************************************************************
*
*  Routine Name: kcsqrt() - calculate the complex square root
*                           of a complex number.
*
*       Purpose: kcsqrt() returns the complex square root of
*                the complex argument passed as input.
*
*         Input: a - complex number
*
*        Output: 
*                
*       Returns: The complex square root of the input 
*                argument.
*
*  Restrictions: This function returns a structure, not a pointer to a structure.  This is allowed in ANSI C.  However, if you are using an older C compiler that is not fully ANSI compliant, you may have problems.
*
*    Written By: Jeremy Worley  
*
*          Date: Jul  2, 1992 11:10
*
*      Verified:
*
*  Side Effects:
*
* Modifications:
*
*************************************************************/

kcomplex kcsqrt(
   kcomplex a)
{
    float ar,ai,r,w;
    kcomplex c;

    if(a.r==0.0 && a.i == 0.0){
       c.r = 0.0;
       c.i = 0.0;
    }else{
       ar = (float)fabs((double)a.r);
       ai = (float)fabs((double)a.i);

       if((double)ar >= (double)ai){
          r = (float)((double)ai/(double)ar);
          w = (float)(sqrt((double)ar)*
                      sqrt((0.5*(1.0+sqrt(1.0+(double)r*(double)r)))));
       }else{
          r = (float)((double)ar/(double)ai);
          w = (float)(sqrt((double)ai)*
                    sqrt(0.5*((double)r+(float)sqrt(1.0+(double)r*(double)r))));
       }
       if(a.r >= 0.0){
          c.r = w;
          c.i = a.i/(2.0*w);
       }else{
          c.i = (float) (((double)a.i >= 0) ? (double)w : -(double)w);
          c.r = (float) ((double)a.i)/(2*(double)c.i);
       }
    }
    return(c);
} 

/***********************************************************
*
*  Routine Name: kcexp - complex exponential function
*
*       Purpose: This function returns the complex exponential
*		 of the input argument.
*
*         Input: x - a complex number.
*
*        Output: 
*
*       Returns: a complex exponential of the input argument
*
*  Restrictions: 
*    Written By: Jeremy Worley
*          Date: Mar 05, 1993 12:11
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

kcomplex kcexp(kcomplex x)
{
  float ex;
  kcomplex y;

  if(x.r == 0.0){
     y.r = y.i = 0.0;
     return(y);
  }
   
  ex = exp((double) x.r);
  y.r = ex * cos((double) x.i);
  y.i = ex * sin((double) x.i);
 
  return(y);
}

/************************************************************
*
*  Routine Name: kclog - complex natural logarithm
*
*       Purpose: This function returns the complex
*		 natural logarithm of the input argument.
*
*         Input: x - complex argument.
*
*        Output: 
*
*       Returns: a complex logarithm of the input argument.
*
*  Restrictions: 
*    Written By: Jeremy Worley
*          Date: Mar 05, 1993 12:19
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

kcomplex kclog(kcomplex x)
{
  kcomplex y;

  if(x.r == 0.0 && x.i == 0.0){
     y.r = y.i = 0.0;
     return(y);
  }

  y.r = log(sqrt((double) x.r*x.r + x.i*x.i));
  y.i = atan2((double) x.i, (double) x.r);
  return(y);
}

/************************************************************
*
*  Routine Name: kcpow() - calculate the complex power of two
*                          complex number.
*
*       Purpose: kcpow() returns the complex power of two
*		 kcomplex arguments passed in as input.
*
*         Input: x - kcomplex number
*		 y - kcomplex power
*
*        Output: 
*                
*       Returns: The kcomplex power of the two input argument.
*
*  Restrictions: This function returns a structure, not a pointer to a
*                structure.  This is allowed in ANSI C.  However, if
*                you are using * an older C compiler that is not fully
*                ANSI compliant, you may have * problems.
*
*    Written By: Jeremy Worley  
*
*          Date: Mar 04, 1993 23:02
*
*      Verified:
*
*  Side Effects:
*
* Modifications:
*
*************************************************************/

kcomplex kcpow(kcomplex x, kcomplex y)
{
   return(kcexp(kcmult(kclog(x),y)));
}
