/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>	           Form Deletion/Cleanup Routines             <<<<
   >>>>                                                       <<<<
   >>>>  Private:                                             <<<<
   >>>>                kvf_destroy_subform()                  <<<<
   >>>>                kvf_free_line_info_strings()           <<<<
   >>>>                kvf_delete_live_sel_link()             <<<<
   >>>>                kvf_delete_scroll_link()               <<<<
   >>>>                kvf_destroy_control()                  <<<<
   >>>>                kvf_destroy_guide()                    <<<<
   >>>>                kvf_destroy_sel_list()                 <<<<
   >>>>                kvf_destroy_selection()                <<<<
   >>>>   Public:                                             <<<<
   >>>>                kvf_destroy_form()                     <<<<
   >>>>   Static:                                             <<<<
   >>>>                free_callback_list()                   <<<<
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"


static void free_callback_list PROTO((klist *));

/************************************************************
*
*  Routine Name: kvf_destroy_form - destroy and free form tree
*
*       Purpose: Destroys the form tree and frees all memory associated 
*                with the pointer returned by kvf_create_form().  
*                To be used only when the caller is totally finished 
*                with the form tree.  
*
*         Input: form - pointer to the form tree being destroyed
*        Output: none
*       Returns: none
*  Restrictions: none
*    Written By: Danielle Argiro
*          Date: Jul 16, 1992 
*      Verified:
*  Side Effects: none
* Modifications: 
*
*************************************************************/

void kvf_destroy_form(
   kform *form)
{
	ksubform *subform; 
	ksubform *old_subform;

	if (form == NULL) return;
	
	kvf_delete_kformstructs(form);

	if (form->master != NULL)
	{
	    subform = form->master->subform_list;
	    while (subform != NULL)
	    {
		old_subform = subform;
		subform = subform->next;
		kvf_destroy_subform(old_subform);
	    }
	    kvf_destroy_control(form->master);
	}
	else kvf_destroy_subform(form->subform);

	kfree(form->control_line);
	kfree(form->control_comment);
	kfree(form->uis_location);

	kvf_delete_entry((char *) form);
	kfree(form);
}




/*------------------------------------------------------------
|
|  Routine Name: kvf_destroy_subform
|
|       Purpose: Destroys the subform tree associated with the 
|                subform pointer.  
|
|         Input: subform - the subform pointer
|        Output: none
|       Returns: nothing
|    Written By: Danielle Argiro and Mark Young
|          Date: May 6, 1992
| Modifications: Converted from Khoros 1.0 (DA)
|
-------------------------------------------------------------*/

void kvf_destroy_subform(
   ksubform *subform)
{
	kguide *guide, *old_guide;

	if (subform == NULL) return;

	if (subform->guidepane != NULL)
	{
	    guide = subform->guidepane->guide_list;
	    while (guide != NULL)
	    {
		old_guide = guide;
		guide = guide->next;
	        kvf_destroy_guide(old_guide);
	    }
	    kvf_destroy_control(subform->guidepane);
	}
	else if (subform->guide != NULL)
	    kvf_destroy_guide(subform->guide);

	kfree(subform->line);
	kfree(subform->comment);
	kfree(subform->control_line);
	kfree(subform->control_comment);

	kvf_delete_entry((char *) subform);
	kfree(subform);
}

/*-----------------------------------------------------------
|
|  Routine Name: kvf_destroy_control
|
|       Purpose: destroys the structure associated with a master,
|		 guidepane, or pane.  
|         Input: control - the pointer to control panel structure
|        Output: none
|       Returns: nothing
|          Date: May 7, 1992
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/

void kvf_destroy_control(
   kcontrol *control)
{
	int i;

	kfree(control->control_line);
	kfree(control->control_comment);

	if (control->sel_list != NULL)
	   kvf_destroy_sel_list(control->sel_list);

	if (control->sel_names != NULL)
	{
	    for (i = 0; i < control->sel_count; i++)
               kfree(control->sel_names[i]);
            kfree(control->sel_names);
	}

	kfree(control);
}

/*-----------------------------------------------------------
|
|  Routine Name: kvf_destroy_guide
|
|       Purpose: destroys the structure associated with a guide
|         Input: guide - the pointer to guide structure
|        Output: none
|       Returns: nothing
|          Date: May 7, 1992
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
void kvf_destroy_guide(
   kguide *guide)
{
	kfree(guide->line);
	kfree(guide->control_line);
	kfree(guide->comment);

	if (guide->pane != NULL) 
	    kvf_destroy_control(guide->pane);

	kfree(guide);
}

/*-----------------------------------------------------------
|
|  Routine Name: kvf_destroy_sel_list
|
|       Purpose: destroys a selection list
|         Input: selection - the header of the selection list
|        Output: none
|       Returns: nothing
|          Date: May 7, 1992
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/

void kvf_destroy_sel_list(
   kselection *sel_list)
{
	kselection *old_sel;
	kselection *selection = sel_list;

	while (selection != NULL)
	{
	    old_sel = selection;
	    selection = selection->next;
	    kvf_destroy_selection(old_sel);
	}
}


/*-----------------------------------------------------------
|
|  Routine Name: kvf_destroy_selection
|
|       Purpose: destroys a selection
|         Input: selection - the selection to be destroyed
|        Output: none
|       Returns: nothing
|          Date: May 7, 1992
|    Written By: Danielle Argiro
| Modifications:
|
------------------------------------------------------------*/
void kvf_destroy_selection(
   kselection *selection)
{
	kselection   *toggle,  *old_toggle;
	kselection   *group,   *old_group;
	kform_struct *submenu, *old_submenu;

	/* free memory associated with toggle */
	toggle = selection->toggle_next;
	while (toggle != NULL)
	{
	    old_toggle = toggle;
	    toggle = toggle->next;
	    kvf_destroy_selection(old_toggle);
	}

	/* free memory associated with group */
	group = selection->group_next;
	while (group != NULL)
	{
	    old_group = group;
	    group = group->next;
	    kvf_destroy_selection(old_group);
	}

	/* free memory associated with submenu */
	submenu = selection->submenu_next;
        while (submenu != NULL)
	{
	    old_submenu = submenu;
	    submenu = submenu->next;
	    kfree(old_submenu);
	}

	/* free memory associated with selection itself */
	kfree(selection->extensions);
       	kfree(selection->line);
       	kfree(selection->comment);
	kfree(selection->submenu_buttons);

	kvf_unlink_formstruct(selection->back_kformstruct);
        kfree(selection);
	
}

/*-----------------------------------------------------------
|
|  Routine Name: kvf_delete_kformstructs()
|
|       Purpose: deletes the list of kform_struct's
|         Input: none
|        Output: none
|       Returns: nothing
|          Date: Jul 20, 1992 
|    Written By: Danielle Argiro 
| Modifications: 
|
------------------------------------------------------------*/

void kvf_delete_kformstructs(
   kform *form)
{
	kform_struct *kformstruct, *old_kformstruct;
	kform   *compare_form = NULL;

	kformstruct = kformstruct_hdr;
	while (kformstruct != NULL)
	{
	    old_kformstruct = kformstruct;
	    kformstruct = kformstruct->next;
	    switch(old_kformstruct->type)
	    {
		case KSELECTION: 
	             compare_form = old_kformstruct->Selptr->back_form;
		     break;

		case KCONTROL:
		case KMASTER:
		case KGUIDEPANE:
		case KPANE:
	             compare_form = old_kformstruct->Controlptr->back_form;
		     break;
		
		case KGUIDE:
	             compare_form = old_kformstruct->Guideptr->back_form;
		     break;

		case KSUBFORM:
	             compare_form = old_kformstruct->Subformptr->back_form;
		     break;

		case KFORM:
	             compare_form = old_kformstruct->Formptr;
		     break;
	   }

	    if (compare_form == form) 
	    {
		klist_free(old_kformstruct->callback_list, free_callback_list);
		kfree(old_kformstruct);
	    }
	}
	kformstruct_hdr = NULL;
	kformstruct_tail = NULL;
}

/*-----------------------------------------------------------
|
|  Routine Name: free_callback_list
|
|       Purpose: Frees the callback_info structure which is 
|                the client_data of the kformstruct->callback_list.
|                kformstruct->callback list is used to keep a list
|                of the callbacks when xvf_add_gui_callback() has been used
|                to add a callback to the GUI item, which will be
|                fired when an attribute of that GUI item changes.
|
|         Input: list - pointer to the kformstruct->callback_list node
|        Output: TRUE
|       Returns: nothing
|          Date: Oct 6, 1993
|    Written By: Danielle Argiro
| Modifications:
|
------------------------------------------------------------*/

static void free_callback_list(
   klist *list)
{
	kaddr call_info;

	call_info = klist_clientdata(list);
	kfree(call_info);
}


/*-----------------------------------------------------------
|
|  Routine Name: kvf_free_line_info_strings
|
|       Purpose: Frees all strings contained in a Line_Info struct
| 		
|         Input: line_info - pointer to the Line_Info struct
|        Output: none
|       Returns: nothing
|          Date: Jul 6, 1992 
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/

void kvf_free_line_info_strings(
   Line_Info *line_info)
{
	int i;

	kfree(line_info->title);
	kfree(line_info->description);
	kfree(line_info->variable);
	kfree(line_info->routine);
	kfree(line_info->help_file);
	kfree(line_info->filename);
	kfree(line_info->string_val);
	kfree(line_info->string_def);
	kfree(line_info->file_def);
	kfree(line_info->literal);
	kfree(line_info->lib_call);

	for (i = 0; i < line_info->val_labelnum; i++)
		kfree(line_info->val_labels[i]);
	line_info->val_labelnum = 0;
	kfree(line_info->val_labels);
}
