/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>		Attribute Utilities			      <<<<
   >>>>                                                       <<<<
   >>>>  Private:                                             <<<<
   >>>>	     		kvf_init_attributes()	              <<<<
   >>>>	     		kvf_check_attributes()	              <<<<
   >>>>	     		kvf_vset_attributes()		      <<<<
   >>>>	     		kvf_vget_attributes()		      <<<<
   >>>>   Static:                                             <<<<
   >>>>	     		kvf_attribute_error_mesg()	      <<<<
   >>>>   Public:                                             <<<<
   >>>>	     		kvf_set_attributes()		      <<<<
   >>>>	     		kvf_get_attributes()		      <<<<
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<  */

#include "internals.h"


static void kvf_attribute_error_mesg PROTO((char *, kform_struct *, char *));

/************************************************************
*
*  Routine Name: kvf_set_attributes - set attributes of GUI items / UIS lines
*
*       Purpose: Sets attributes associated with nodes in the form tree
*                when xvobjects for the GUI have not yet been created.
*
*         Input: kformstruct  - the generic kform_struct associated
*                               with a form tree node
*                attribute, value - variable number of attribute/value pairs
*                                   list of pairs; MUST be terminated by NULL.
*        Output: 
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*  Restrictions:
*    Written By: Danielle Argiro & Mark Young
*          Date: March 22, 1994
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int kvf_set_attributes(
   kform_struct *kformstruct,
   kvalist)
{
	kva_list list;

	/*
         * sanity check
         */
        if (kformstruct == NULL)
        {
            kerror("kforms", "kvf_set_attributes",
                   "kformstruct passed in is NULL");
            return(FALSE);
        }

	/*
	 *  Start at the beginning of the list again, and on this 2nd
         *  pass, let kvf_vset_attributes() actually do the work 
         *  of setting the attributes.
	 */
        kva_start(list, kformstruct);
        return(kvf_vset_attributes(kformstruct, list));
}


/************************************************************
*
*  Routine Name: kvf_get_attributes - get attributes of GUI items / UIS lines
*
*       Purpose: Gets attributes associated with nodes in the form tree
*                when xvobjects for the GUI have not yet been created.
*
*         Input: kformstruct      -  the generic kform_struct associated
*                                    with a form tree node
*                attribute, value - variable number of attribute/value pairs
*                                   list of pairs; MUST be terminated by NULL.
*        Output:
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*  Restrictions:
*    Written By: Danielle Argiro & Mark Young
*          Date: March 22, 1994
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int kvf_get_attributes(
   kform_struct *kformstruct,
   kvalist)
{
        kva_list list;
	
	/*
         * sanity check
         */
        if (kformstruct == NULL)
        {
            kerror("kforms", "kvf_get_attributes",
                   "kformstruct passed in is NULL");
            return(FALSE);
        }

	/*
	 *  Start at the beginning of the list again, and on this 2nd
         *  pass, let kvf_vget_attributes() actually do the work 
         *  of getting the attributes.
	 */
        kva_start(list, kformstruct);
        return(kvf_vget_attributes(kformstruct, list));
}

/************************************************************
*
*  Routine Name: kvf_vset_attributes - set attributes of GUI items / UIS lines using existing variable argument list
*
*       Purpose: Sets attributes associated with nodes in the form tree
*                when xvobjects for the GUI have not yet been created.
*
*         Input: kformstruct  - the generic kform_struct associated
*                               with a form tree node
*                attribute, value - variable number of attribute/value pairs
*                                   list of pairs; MUST be terminated by NULL.
*        Output: 
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*  Restrictions:
*    Written By: Danielle Argiro & Mark Young
*          Date: March 23, 1994
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int kvf_vset_attributes(
   kform_struct *kformstruct,
   kva_list list)
{
	double  tmpdbl;
	kaddr   tmpstruct;
	AttributeEntry *entry;
	kaddr   calldata = NULL;
	char    *attribute, *tmpstr;
	int     i, status, token, tmpint;
	float   tmpflt;
	kselection **save_extensions = NULL;

	/*
	 *  for each attribute/value resource pair...
	 */
	status = TRUE;
	while ((attribute = kva_arg(list, char *)) != NULL)
        {
	    /* make sure the attribute is valid */
	    if (!(kvf_check_attributes("kvf_set_attribute(s)",
                                      kformstruct, attribute)))
                 return(FALSE);

	    /*
	     *  find the attribute in the attribute table
	     */
	    token = kstring_to_token(attribute);
	    entry = NULL;
	    for (i = 0; i < kvf_attribute_num; i++)
	    {
		if (token == kvf_attribute_table[i].token)
		{
		    entry = &kvf_attribute_table[i];
	  	    break;
		}
	    }
	    /*
	     *  shouldn't be possible to get here with kvf_check_attributes,
	     *  but you never know.
	     */
	    if (entry == NULL)
	    {
		kerror("kforms", "kvf_vset_attributes",
		       "Unknown attribute '%s'. Unable to continue setting the rest of the attributes list.", attribute);
		return(FALSE);
	    }

	    switch(entry->data_type)
	    {	
		case KINT:
	             tmpint = kva_arg(list, int);
		     calldata = (kaddr) &tmpint;
		     break;

		case KSTRING:
	             tmpstr = kva_arg(list, char *);
		     calldata = (kaddr) &tmpstr;
		     break;

		case KSTRUCT:
	             tmpstruct = kva_arg(list, kaddr);
		     calldata = (kaddr) &tmpstruct;
		     break;

		case KFLOAT:
	             tmpflt = (float) kva_arg(list, double);
		     calldata = (kaddr) &tmpflt;
		     break;
	
		case KDOUBLE:
	             tmpdbl = kva_arg(list, double);
		     calldata = (kaddr) &tmpdbl;
		     break;
	
	        default:
		     kerror("kforms", "kvf_vset_attributes",
			    "Unknown attribute data type");
		     break;
	    }

	    status = (*entry->set_attr_routine)
		     (kformstruct, attribute, calldata);

	    if (kstrcmp(attribute, KVF_DELETE) == 0)
	       return(status);
	
           /*
            *  special case for selections with extensions array to support
            *  setting of attributes menuform selections, making
            *  sure they are reflected by the same attribute in the
            *  corresponding selection on the front panel
            */
            if ((kformstruct->type == KSELECTION)        &&
                (kformstruct->Selptr->extensions != NULL))
            {
                /* only attributes that "travel" are "value" attributes */
                if ((kstrcmp(attribute, KVF_FILE_NAME)  == 0) ||
                    (kstrcmp(attribute, KVF_FILE_DEF)   == 0) ||
                    (kstrcmp(attribute, KVF_STRING_VAL) == 0) ||
                    (kstrcmp(attribute, KVF_STRING_DEF) == 0) ||
                    (kstrcmp(attribute, KVF_TOGGLE_NUM) == 0) ||
                    (kstrcmp(attribute, KVF_TOGGLE_VAL) == 0) ||
                    (kstrcmp(attribute, KVF_LOGIC_VAL)  == 0) ||
                    (kstrcmp(attribute, KVF_LIST_VAL)   == 0) ||
                    (kstrcmp(attribute, KVF_CYCLE_VAL)  == 0) ||
                    (kstrcmp(attribute, KVF_INT_VAL)    == 0) ||
                    (kstrcmp(attribute, KVF_INT_DEF)    == 0) ||
                    (kstrcmp(attribute, KVF_FLOAT_VAL)  == 0) ||
                    (kstrcmp(attribute, KVF_DOUBLE_VAL) == 0) ||
                    (kstrcmp(attribute, KVF_LITERAL) == 0)    ||
                    (kstrcmp(attribute, KVF_OPTSEL) == 0))
                {
                    kselection  *extension;
                    int          restore_link = FALSE;

		    for (i = 0; i < kformstruct->Selptr->extension_num; i++)
		    {
                        extension = kformstruct->Selptr->extensions[i];
                        if (extension->extensions != NULL)
		        {
                            restore_link = TRUE;
			    save_extensions = extension->extensions;
                            extension->extensions = NULL;
		        }
    
                        switch(entry->data_type)
                        {
                            case KINT:
                                 status =
                                 kvf_set_attribute(extension->back_kformstruct,
                                                      attribute, tmpint);
                                 break;
    
                            case KSTRING:
                                 status =
                                 kvf_set_attribute(extension->back_kformstruct,
                                                   attribute, tmpstr);
                                 break;
    
                            case KSTRUCT:
                                 status =
                                 kvf_set_attribute(extension->back_kformstruct,
                                                   attribute, tmpstruct);
                                 break;
    
                            case KFLOAT:
                                 status =
                                 kvf_set_attribute(extension->back_kformstruct,
                                                   attribute, tmpflt);
                                 break;
    
                            case KDOUBLE:
                                 status =
                                 kvf_set_attribute(extension->back_kformstruct,
                                                   attribute, tmpdbl);
                                 break;
    
                            default:
                                 kerror("xvforms", "xvf_vset_attributes",
                                         "Unknown attribute data type");
                                 break;
                        }
    
                        extension->modified = TRUE;
    
                        if (restore_link)
                            extension->extensions = save_extensions;
		    }
                }
            }

        }
	return(status);

}


/************************************************************
*
*  Routine Name: kvf_vget_attributes - get attributes of GUI items / UIS lines using existing variable argument list
*
*       Purpose: Gets attributes associated with nodes in the form tree
*                when xvobjects for the GUI have not yet been created.
*
*         Input: kformstruct      -  the generic kform_struct associated
*                                    with a form tree node
*                attribute, value - variable number of attribute/value pairs
*                                   list of pairs; MUST be terminated by NULL.
*        Output:
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*  Restrictions:
*    Written By: Danielle Argiro & Mark Young
*          Date: March 23, 1994
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int kvf_vget_attributes(
   kform_struct *kformstruct,
   kva_list list)
{
        int     i, token, status;
	char    *attribute;
        kaddr   calldata;
        AttributeEntry *entry;

        /*
         *  for each attribute/value resource pair...
         */
	status = TRUE;
        while ((attribute = kva_arg(list, char *)) != NULL)
        {
	    /* make sure the attribute is valid */
            if (!(kvf_check_attributes("kvf_set_attribute(s)",
                                      kformstruct, attribute)))
                 return(FALSE);

            /*
             *  find the attribute in the attribute table
             */
            token = kstring_to_token(attribute);
            entry = NULL;
            for (i = 0; i < kvf_attribute_num; i++)
            {
                if (token == kvf_attribute_table[i].token)
                {
                    entry = &kvf_attribute_table[i];
                    break;
                }
            }

            /*
             *  shouldn't be possible to get here with kvf_check_attributes,
             *  but you never know.
             */
            if (entry == NULL)
            {
                kerror("kforms", "kvf_vget_attributes",
                       "Unknown attribute '%s'. Unable to continue getting the rest of the attributes list.", attribute);
                return(FALSE);
            }

	    calldata = kva_arg(list, kaddr);

            if (!(*entry->get_attr_routine)(kformstruct, attribute, calldata))
		status = FALSE;
        }

	return(status);
}

/*------------------------------------------------------------
|
|  Routine Name: kvf_init_attributes
|
|       Purpose: Initializes the tokens associated with the
|                kforms attributes.
|
|         Input: none
|        Output: none
|          Date: March 23, 1994
|    Written By: Danielle Argiro & Mark Young
| Modifications:
|
-------------------------------------------------------------*/

void kvf_init_attributes(void)
{
	int i;

        for (i = 0; i < kvf_attribute_num; i++)
        {
            kvf_attribute_table[i].token =
                 kstring_to_token(kvf_attribute_table[i].attribute);
        }
}


/*------------------------------------------------------------
|
|  Routine Name: kvf_check_attributes
|
|       Purpose: Given the parameter list that was passed to 
|                kvf_set_attributes or kvf_get_attributes,
|                checks to make sure that the attribute(s) given 
|                apply to the GUI item in question.
|
|         Input: routine     - name of routine checking 
|                kformstruct - generic kform_struct
|                attribute   - attribute to check
|        Output: TRUE if combination is OK, FALSE otherwise
|          Date: Jan 31, 1994
|    Written By: Danielle Argiro
| Modifications:
|
-------------------------------------------------------------*/
int kvf_check_attributes(
   char         *routine,
   kform_struct *kformstruct,
   char         *attribute)
{
	int status = TRUE;

	/*
	 * activate attribute for most GUI items, a few exceptions 
	 */
	if (kstrcmp(attribute,  KVF_ACTIVATE) == 0) 
	{
	    switch(kformstruct->flag)
	    {
	       case KUIS_STARTFORM:
       	       case KUIS_STARTSUBMENU:
               case KUIS_STARTSUBFORM:
               case KUIS_STARTGUIDE:
               case KUIS_STARTPANE:
               case KUIS_SUBFORMBUTTON:
               case KUIS_MASTERACTION:
               case KUIS_GUIDEBUTTON:
               case KUIS_QUIT:
               case KUIS_SUBFORMACTION:
               case KUIS_HELP:
               case KUIS_INPUTFILE:
               case KUIS_OUTPUTFILE:
               case KUIS_STDIN:
               case KUIS_STDOUT:
               case KUIS_INTEGER:
               case KUIS_FLOAT:
               case KUIS_DOUBLE:
               case KUIS_LOGICAL:
               case KUIS_FLAG:
               case KUIS_CYCLE:
               case KUIS_LIST:
               case KUIS_DISPLAYLIST:
               case KUIS_STRING:
               case KUIS_STRINGLIST:
               case KUIS_ROUTINE:
               case KUIS_TOGGLE:
               case KUIS_PANEACTION:
               case KUIS_BLANK:
               case KUIS_WORKSPACE:
                    status = TRUE;
		    break;

               default:
		    kvf_attribute_error_mesg(routine, kformstruct, attribute);
                    status = FALSE;
		    break;
	    }
	}

	/*
	 * selected attribute for most GUI items, a few exceptions 
	 */
	if (kstrcmp(attribute,  KVF_SELECTED) == 0)
	{
	    switch(kformstruct->flag)
            {
	       case KUIS_STARTFORM:
       	       case KUIS_STARTSUBMENU:
               case KUIS_STARTSUBFORM:
               case KUIS_STARTPANE:
               case KUIS_SUBFORMBUTTON:
               case KUIS_MASTERACTION:
               case KUIS_GUIDEBUTTON:
               case KUIS_QUIT:
               case KUIS_SUBFORMACTION:
               case KUIS_INPUTFILE:
               case KUIS_OUTPUTFILE:
               case KUIS_INTEGER:
               case KUIS_FLOAT:
               case KUIS_DOUBLE:
               case KUIS_LOGICAL:
               case KUIS_FLAG:
               case KUIS_CYCLE:
               case KUIS_LIST:
               case KUIS_DISPLAYLIST:
               case KUIS_STRING:
               case KUIS_STRINGLIST:
               case KUIS_ROUTINE:
               case KUIS_TOGGLE:
               case KUIS_PANEACTION:
                    status = TRUE;
                    break;

               default:
                    kvf_attribute_error_mesg(routine, kformstruct, attribute);
                    status = FALSE;
                    break;
            }
	}

	/*
	 * modified attribute for all selections
	 */
	if (kstrcmp(attribute,  KVF_MODIFIED) == 0)
	{
	    switch(kformstruct->flag)
            {
               case KUIS_MASTERACTION:
               case KUIS_HELP:
               case KUIS_QUIT:
               case KUIS_SUBFORMACTION:
               case KUIS_INPUTFILE:
               case KUIS_OUTPUTFILE:
               case KUIS_STDIN:
               case KUIS_STDOUT:
               case KUIS_INTEGER:
               case KUIS_FLOAT:
               case KUIS_DOUBLE:
               case KUIS_LOGICAL:
               case KUIS_FLAG:
               case KUIS_CYCLE:
               case KUIS_LIST:
               case KUIS_BLANK:
               case KUIS_DISPLAYLIST:
               case KUIS_STRING:
               case KUIS_STRINGLIST:
               case KUIS_ROUTINE:
               case KUIS_TOGGLE:
               case KUIS_PANEACTION:
               case KUIS_GROUP:
               case KUIS_MUTINCL:
               case KUIS_MUTEXCL:
                    status = TRUE;
                    break;

               default:
                    kvf_attribute_error_mesg(routine, kformstruct, attribute);
                    status = FALSE;
                    break;
            }
	}
	/*
	 *  optional attributesfor all selections 
         */
	else if (kstrcmp(attribute, KVF_OPTIONAL) == 0)
	{
	    switch(kformstruct->flag)
	    {
	       case KUIS_INPUTFILE:
	       case KUIS_OUTPUTFILE:
	       case KUIS_STDIN:
	       case KUIS_STDOUT:
	       case KUIS_INTEGER:
	       case KUIS_FLOAT:
	       case KUIS_DOUBLE:
	       case KUIS_LOGICAL:
	       case KUIS_FLAG:
	       case KUIS_CYCLE:
	       case KUIS_LIST:
	       case KUIS_DISPLAYLIST:
	       case KUIS_STRING:
	       case KUIS_STRINGLIST:
	       case KUIS_TOGGLE:
	       case KUIS_MUTEXCL:
	       case KUIS_MUTINCL:
	       case KUIS_GROUP:
                    status = TRUE;
		    break;

	       default:
		    kvf_attribute_error_mesg(routine, kformstruct, attribute);
                    status = FALSE;
		    break;
	    }
	}
	/*
	 *  optional & opt_sel attributes for all selections 
         */
	else if (kstrcmp(attribute, KVF_OPTSEL)   == 0)
	{
	    switch(kformstruct->flag)
	    {
	       case KUIS_INPUTFILE:
	       case KUIS_OUTPUTFILE:
	       case KUIS_STDIN:
	       case KUIS_STDOUT:
	       case KUIS_INTEGER:
	       case KUIS_FLOAT:
	       case KUIS_DOUBLE:
	       case KUIS_LOGICAL:
	       case KUIS_FLAG:
	       case KUIS_CYCLE:
	       case KUIS_LIST:
	       case KUIS_DISPLAYLIST:
	       case KUIS_STRING:
	       case KUIS_STRINGLIST:
	       case KUIS_TOGGLE:
                    status = TRUE;
		    break;

	       default:
		    kvf_attribute_error_mesg(routine, kformstruct, attribute);
                    status = FALSE;
		    break;
	    }
	}

	/*
         *  literal attribute for selections w/ text parameter boxes
         */
        else if (kstrcmp(attribute, KVF_LITERAL) == 0)
        {
            switch(kformstruct->flag)
            {
               case KUIS_STDIN:
               case KUIS_STDOUT:
               case KUIS_INPUTFILE:
               case KUIS_OUTPUTFILE:
               case KUIS_INTEGER:
               case KUIS_FLOAT:
               case KUIS_DOUBLE:
               case KUIS_STRING:
               case KUIS_STRINGLIST:
               case KUIS_TOGGLE:
                    status = TRUE;
                    break;

               default:
                    kvf_attribute_error_mesg(routine, kformstruct, attribute);
                    status = FALSE;
                    break;
            }
        }

	/*
         * live attribute for all selections except AnsInfile, AnsOutfile
         */
        else if (kstrcmp(attribute, KVF_LIVE) == 0)
        {
            switch(kformstruct->flag)
            {
               case KUIS_INPUTFILE:
               case KUIS_OUTPUTFILE:
               case KUIS_INTEGER:
               case KUIS_FLOAT:
               case KUIS_DOUBLE:
               case KUIS_LOGICAL:
               case KUIS_FLAG:
               case KUIS_CYCLE:
               case KUIS_LIST:
               case KUIS_DISPLAYLIST:
               case KUIS_STRING:
               case KUIS_STRINGLIST:
               case KUIS_TOGGLE:
                    status = TRUE;
                    break;

               default:
                    kvf_attribute_error_mesg(routine, kformstruct, attribute);
                    status = FALSE;
                    break;
            }
        }

        /*
         * description for most GUI items
         */
        else if (kstrcmp(attribute, KVF_DESCRIPTION) == 0)
        {
            switch(kformstruct->flag)
            {
               case KUIS_MASTERACTION:
               case KUIS_SUBFORMACTION:
               case KUIS_HELP:
               case KUIS_INPUTFILE:
               case KUIS_OUTPUTFILE:
               case KUIS_INTEGER:
               case KUIS_FLOAT:
               case KUIS_DOUBLE:
               case KUIS_LOGICAL:
               case KUIS_FLAG:
               case KUIS_CYCLE:
               case KUIS_LIST:
               case KUIS_DISPLAYLIST:
               case KUIS_STRING:
               case KUIS_STRINGLIST:
               case KUIS_ROUTINE:
               case KUIS_TOGGLE:
               case KUIS_PANEACTION:
               case KUIS_WORKSPACE:
                    status = TRUE;
                    break;

               default:
                    kvf_attribute_error_mesg(routine, kformstruct, attribute);
                    status = FALSE;
                    break;
            }
        }

	/* 
	 * title attribute for on most GUI items, a few exceptions 
	 */
	else if (kstrcmp(attribute, KVF_TITLE) == 0)
	{
            switch(kformstruct->flag)
            {
               case KUIS_STARTFORM:
               case KUIS_STARTMASTER:
               case KUIS_STARTSUBMENU:
               case KUIS_STARTSUBFORM:
               case KUIS_STARTPANE:
               case KUIS_SUBFORMBUTTON:
               case KUIS_MASTERACTION:
               case KUIS_GUIDEBUTTON:
               case KUIS_QUIT:
               case KUIS_SUBFORMACTION:
               case KUIS_HELP:
               case KUIS_INPUTFILE:
               case KUIS_OUTPUTFILE:
               case KUIS_STDIN:
               case KUIS_STDOUT:
               case KUIS_INTEGER:
               case KUIS_FLOAT:
               case KUIS_DOUBLE:
               case KUIS_LOGICAL:
               case KUIS_FLAG:
               case KUIS_CYCLE:
               case KUIS_LIST:
               case KUIS_DISPLAYLIST:
               case KUIS_STRING:
               case KUIS_STRINGLIST:
               case KUIS_ROUTINE:
               case KUIS_TOGGLE:
               case KUIS_BLANK:
               case KUIS_PANEACTION:
                    status = TRUE;
                    break;

               default:
                    kvf_attribute_error_mesg(routine, kformstruct, attribute);
                    status = FALSE;
                    break;
	    }
	}

        /*
         * can change the literal on any selection with a text parameter box
         */
        else if (kstrcmp(attribute, KVF_LITERAL) == 0)
        {
            switch(kformstruct->flag)
            {
               case KUIS_INPUTFILE:
               case KUIS_OUTPUTFILE:
               case KUIS_INTEGER:
               case KUIS_FLOAT:
               case KUIS_DOUBLE:
               case KUIS_STRING:
               case KUIS_STRINGLIST:
               case KUIS_TOGGLE:
                    status = TRUE;
                    break;

               default:
                    status = FALSE;
            }
        }


	/* 
         * can change the button title on any GUI item with a button 
         */
	else if ((kstrcmp(attribute, KVF_BUTTONTITLE) == 0) ||
		 (kstrcmp(attribute, KVF_BUTTONX) == 0)     ||
		 (kstrcmp(attribute, KVF_BUTTONY) == 0)     ||
		 (kstrcmp(attribute, KVF_BUTTONWIDTH) == 0) ||
		 (kstrcmp(attribute, KVF_BUTTONHEIGHT) == 0))
	{
	    switch(kformstruct->flag)
            {
	       case KUIS_MASTERACTION:
	       case KUIS_SUBFORMACTION:
	       case KUIS_SUBFORMBUTTON:
	       case KUIS_GUIDEBUTTON:
	       case KUIS_ROUTINE:
	       case KUIS_HELP:
	       case KUIS_QUIT:
	       case KUIS_PANEACTION:
                    status = TRUE;
		    break;

	       default:
                    status = FALSE;
	    }
	}

        /*
         * can use the KVF_DELETE attribute to delete any GUI item
        else if (kstrcmp(attribute, KVF_DELETE) == 0)
        {
            switch(kformstruct->flag)
            {
               case KUIS_STARTSUBMENU:
               case KUIS_STARTSUBFORM:
               case KUIS_STARTPANE:
               case KUIS_SUBFORMBUTTON:
               case KUIS_MASTERACTION:
               case KUIS_GUIDEBUTTON:
               case KUIS_QUIT:
               case KUIS_SUBFORMACTION:
               case KUIS_HELP:
               case KUIS_INPUTFILE:
               case KUIS_OUTPUTFILE:
               case KUIS_INTEGER:
               case KUIS_FLOAT:
               case KUIS_DOUBLE:
               case KUIS_LOGICAL:
               case KUIS_FLAG:
               case KUIS_CYCLE:
               case KUIS_LIST:
               case KUIS_DISPLAYLIST:
               case KUIS_STRING:
               case KUIS_STRINGLIST:
               case KUIS_ROUTINE:
               case KUIS_TOGGLE:
               case KUIS_PANEACTION:
                    status = TRUE;
                    break;

               default:
                    status = FALSE;
		    break;
            }
        }
         */

        /*
         * KVF_VARIABLE attribute for any GUI item w/ variable associated
         */
        else if (kstrcmp(attribute, KVF_VARIABLE) == 0)
        {
            switch(kformstruct->flag)
            {
               case KUIS_STARTFORM:
               case KUIS_STARTSUBMENU:
               case KUIS_STARTSUBFORM:
               case KUIS_STARTPANE:
               case KUIS_SUBFORMBUTTON:
               case KUIS_MASTERACTION:
               case KUIS_GUIDEBUTTON:
               case KUIS_QUIT:
               case KUIS_SUBFORMACTION:
               case KUIS_HELP:
               case KUIS_INPUTFILE:
               case KUIS_OUTPUTFILE:
               case KUIS_STDIN:
               case KUIS_STDOUT:
               case KUIS_INTEGER:
               case KUIS_FLOAT:
               case KUIS_DOUBLE:
               case KUIS_LOGICAL:
               case KUIS_FLAG:
               case KUIS_CYCLE:
               case KUIS_LIST:
               case KUIS_DISPLAYLIST:
               case KUIS_STRING:
               case KUIS_STRINGLIST:
               case KUIS_ROUTINE:
               case KUIS_TOGGLE:
               case KUIS_PANEACTION:
                    status = TRUE;
                    break;

               default:
                    status = FALSE;
            }
        }


	/* 
	 * filename, default filename on InputFile, OutputFile selections 
	 */
        else if ((kstrcmp(attribute, KVF_FILE_NAME) == 0) ||
		 (kstrcmp(attribute, KVF_FILE_DEF)  == 0))
	{
	    if ((kformstruct->flag == KUIS_INPUTFILE)     ||
		(kformstruct->flag == KUIS_OUTPUTFILE)    ||
		(kformstruct->flag == KUIS_STDIN)         ||
                (kformstruct->flag == KUIS_STDOUT)        ||
		(kformstruct->flag == KUIS_TOGGLE))       
	    {
		/* toggle ok - special case for cantata */
                status = TRUE;
	    }
	    else 
	    {
	        kvf_attribute_error_mesg(routine, kformstruct, attribute);
                status = FALSE;
	    }
	}

	/* 
	 * all these attributes are specific to Integer selections 
	 */
	else if ((kstrcmp(attribute, KVF_INT_UPPER)        == 0) ||
		 (kstrcmp(attribute, KVF_INT_LOWER)        == 0) ||
		 (kstrcmp(attribute, KVF_INT_DEF)          == 0) ||
		 (kstrcmp(attribute, KVF_INT_VAL)          == 0))
	{
	    if (kformstruct->flag == KUIS_INTEGER)
	    {
                status = TRUE;
	    }
	    else 
	    {
	        kvf_attribute_error_mesg(routine, kformstruct, attribute);
                status = FALSE;
	    }
	}

	/* 
	 * all these attributes are specific to Float selections 
	 */
	else if ((kstrcmp(attribute, KVF_FLOAT_UPPER)        == 0) ||
		 (kstrcmp(attribute, KVF_FLOAT_LOWER)        == 0) ||
		 (kstrcmp(attribute, KVF_FLOAT_DEF)          == 0) ||
		 (kstrcmp(attribute, KVF_FLOAT_VAL)          == 0))
	{
	    if (kformstruct->flag == KUIS_FLOAT)
	    {
                status = TRUE;
	    }
	    else 
	    {
	        kvf_attribute_error_mesg(routine, kformstruct, attribute);
                status = FALSE;
	    }
	}

	/* 
	 * all these attributes are specific to Double selections 
	 */
	else if ((kstrcmp(attribute, KVF_DOUBLE_UPPER)        == 0) ||
		 (kstrcmp(attribute, KVF_DOUBLE_LOWER)        == 0) ||
		 (kstrcmp(attribute, KVF_DOUBLE_DEF)          == 0) ||
		 (kstrcmp(attribute, KVF_DOUBLE_VAL)          == 0))
	{
	    if (kformstruct->flag == KUIS_DOUBLE)
	    {
                status = TRUE;
	    }
	    else 
	    {
	        kvf_attribute_error_mesg(routine, kformstruct, attribute);
                status = FALSE;
	    }
	}

	/*
	 * mechanism specific to Integers, Floats & Doubles
	 */
	else if (kstrcmp(attribute, KVF_MECHANISM) == 0)
	{
            if ((kformstruct->flag == KUIS_INTEGER) ||
		(kformstruct->flag == KUIS_FLOAT)   ||
                (kformstruct->flag == KUIS_DOUBLE))
            {
                status = TRUE;
            }
            else
            {
                kvf_attribute_error_mesg(routine, kformstruct, attribute);
                status = FALSE;
            }
        }

	/*
	 * precision specific to Floats & Doubles
	 */
        else if ((kstrcmp(attribute, KVF_FLOAT_PREC)  == 0) ||
		 (kstrcmp(attribute, KVF_DOUBLE_PREC) == 0))
        {
            if ((kformstruct->flag == KUIS_FLOAT)   ||
                (kformstruct->flag == KUIS_DOUBLE))
            {
                status = TRUE;
            }
            else
            {
                kvf_attribute_error_mesg(routine, kformstruct, attribute);
                status = FALSE;
            }
        }


	/*
         * all these attributes are specific to Toggle selections
         */
        else if ((kstrcmp(attribute, KVF_TOGGLE_SIZE) == 0) ||
                 (kstrcmp(attribute, KVF_TOGGLE_NUM)        == 0) ||
                 (kstrcmp(attribute, KVF_TOGGLE_TYPE)       == 0))      
        {
            if (kformstruct->flag == KUIS_TOGGLE)
	    {
                status = TRUE;
	    }
            else
            {
                kvf_attribute_error_mesg(routine, kformstruct, attribute);
                status = FALSE;
            }
        }
	else if (kstrcmp(attribute, KVF_TOGGLE_VAL) == 0)
	{
            if (kformstruct->flag == KUIS_TOGGLE)
            {
                status = TRUE;
            }
            else
            {
                kvf_attribute_error_mesg(routine, kformstruct, attribute);
                status = FALSE;
            }
	}

        /*
         * all these attributes are specific to String selections
         */
        else if ((kstrcmp(attribute, KVF_STRING_VAL) == 0) ||
                 (kstrcmp(attribute, KVF_STRING_DEF) == 0))
        {
            if ((kformstruct->flag == KUIS_STRING) ||
		(kformstruct->flag == KUIS_STRINGLIST))
	    {
                status = TRUE;
	    }
            else
            {
                kvf_attribute_error_mesg(routine, kformstruct, attribute);
                status = FALSE;
            }
        }
	else if (kstrcmp(attribute, KVF_STRING_MULTILINE) == 0) 
	{
	    if ((kformstruct->flag == KUIS_STRING) ||
                (kformstruct->flag == KUIS_STRINGLIST))
	    {
                status = TRUE;
	    }
            else
            {
                kvf_attribute_error_mesg(routine, kformstruct, attribute);
                status = FALSE;
            }
	}

	/*
         * all these attributes are specific to Logical selections
         */
        else if ((kstrcmp(attribute, KVF_LOGIC_VAL) == 0)    ||
                 (kstrcmp(attribute, KVF_LOGIC_DEF) == 0))
        {
            if ((kformstruct->flag == KUIS_LOGICAL)  ||
	        (kformstruct->flag == KUIS_MUTEXCL)  || /* logic_val used    */
	        (kformstruct->flag == KUIS_MUTINCL)  || /* with groups to    */
	        (kformstruct->flag == KUIS_GROUP))      /* indicate optional */
	    {
                status = TRUE;
	    }
            else
            {
                kvf_attribute_error_mesg(routine, kformstruct, attribute);
                status = FALSE;
            }
        }
	else if ((kstrcmp(attribute, KVF_LOGIC_1LABEL) == 0)||
                 (kstrcmp(attribute, KVF_LOGIC_0LABEL) == 0)) 
	{
            if (kformstruct->flag == KUIS_LOGICAL)
            {   
                status = TRUE;
            }
            else
            {
                kvf_attribute_error_mesg(routine, kformstruct, attribute);
                status = FALSE;
            }
        }

	/*
         * all these attributes are specific to List, DisplayList,
         * and StringList selections
         */
        else if ((kstrcmp(attribute, KVF_LIST_SIZE)       == 0) ||
                 (kstrcmp(attribute, KVF_LIST_VAL)       == 0) ||
                 (kstrcmp(attribute, KVF_LIST_INDEX)     == 0) ||
                 (kstrcmp(attribute, KVF_LIST_START)     == 0) ||
                 (kstrcmp(attribute, KVF_LIST_DELETEALL) == 0))
        {
            if ((kformstruct->flag == KUIS_LIST)        ||
                (kformstruct->flag == KUIS_DISPLAYLIST) ||
                (kformstruct->flag == KUIS_CYCLE)       ||
                (kformstruct->flag == KUIS_STRINGLIST))
	    {
                status = TRUE;
	    }
            else
            {
                kvf_attribute_error_mesg(routine, kformstruct, attribute);
                status = FALSE;
            }
        }
	else if ((kstrcmp(attribute, KVF_LIST_LABEL)  == 0)  ||
		 (kstrcmp(attribute, KVF_LIST_ADD)    == 0)  ||
		 (kstrcmp(attribute, KVF_LIST_DELETE) == 0))
        {
            if ((kformstruct->flag == KUIS_LIST)        ||
                (kformstruct->flag == KUIS_DISPLAYLIST) ||
                (kformstruct->flag == KUIS_CYCLE)       ||
                (kformstruct->flag == KUIS_STRINGLIST))
	    {
                status = TRUE;
	    }
            else
            {
                kvf_attribute_error_mesg(routine, kformstruct, attribute);
                status = FALSE;
            }
        }
	else if (kstrcmp(attribute, KVF_LIST_CONTENTS) == 0)
	{
            if ((kformstruct->flag == KUIS_LIST)        ||
                (kformstruct->flag == KUIS_DISPLAYLIST) ||
                (kformstruct->flag == KUIS_CYCLE)       ||
                (kformstruct->flag == KUIS_STRINGLIST))
	    {
                status = TRUE;
	    }
            else
            {
                kvf_attribute_error_mesg(routine, kformstruct, attribute);
                status = FALSE;
            }
	}
	else if (kstrcmp(attribute, KVF_LIST_DOUBLECLICK) == 0)
	{
            if (kformstruct->flag == KUIS_DISPLAYLIST)    
            {
                status = TRUE;
            }
            else
            {
                kvf_attribute_error_mesg(routine, kformstruct, attribute);
                status = FALSE;
            }
	}

        /*
         * print pane only applicable to panes
         */
        else if (kstrcmp(attribute, KVF_PRINT_PANE) == 0) 
        {
            if (kformstruct->type == KPANE)
            {
                status = TRUE;
	    }
            else
            {
                kvf_attribute_error_mesg(routine, kformstruct, attribute);
                status = FALSE;
            }
        }

        /*
         * print subform only applicable to subforms
         */
        else if (kstrcmp(attribute, KVF_PRINT_SUBFORM) == 0) 
        {
            if (kformstruct->type == KSUBFORM)
            {
                status = TRUE;
            }
            else
            {
                kvf_attribute_error_mesg(routine, kformstruct, attribute);
                status = FALSE;
            }
        }

	/*
	 * print UIS only applicable to panes, guidepanes, masters
	 */
        else if (kstrcmp(attribute, KVF_PRINT_UIS) == 0)
        {
            if ((kformstruct->type == KMASTER) ||
                (kformstruct->type == KGUIDEPANE) ||
		(kformstruct->type == KPANE))
            {
                status = TRUE;
            }
            else
            {
                kvf_attribute_error_mesg(routine, kformstruct, attribute);
                status = FALSE;
            }
        }



	/*
         * subform display only applicable to subforms
        else if (kstrcmp(attribute, KVF_DISPLAY_SUBFORM) == 0)
        {
            if ((kformstruct->flag == KUIS_STARTSUBFORM) ||
                (kformstruct->flag == KUIS_SUBFORMBUTTON))
            {
                status = TRUE;
            }
            else
            {
                kvf_attribute_error_mesg(routine, kformstruct, attribute);
                status = FALSE;
            }
        }
         */

        /*
         * pane display only applicable to panes
        else if (kstrcmp(attribute, KVF_DISPLAY_PANE) == 0)
        {
            if ((kformstruct->flag == KUIS_STARTPANE) ||
                (kformstruct->flag == KUIS_GUIDEBUTTON))
            {
                status = TRUE;
            }
            else
            {
                kvf_attribute_error_mesg(routine, kformstruct, attribute);
                status = FALSE;
            }
        }
         */

	/*
         * execution type is only applicable to Routine buttons
         */
	else if (kstrcmp(attribute, KVF_EXECTYPE) == 0)
	{
            if (kformstruct->flag == KUIS_ROUTINE)
            {
                status = TRUE;
            }
            else
            {
                kvf_attribute_error_mesg(routine, kformstruct, attribute);
                status = FALSE;
            }
	}
	
	/*
	 * routine only applicable to Routine buttons
	 */
        else if (kstrcmp(attribute, KVF_ROUTINE) == 0)
        {
            if (kformstruct->flag == KUIS_ROUTINE)
            {
                status = TRUE;
            }
            else
            {
                kvf_attribute_error_mesg(routine, kformstruct, attribute);
                status = FALSE;
            }
        }

	/*
	 * help path only applicable to HELP buttons
	 */
        else if (kstrcmp(attribute, KVF_HELPPATH) == 0)
        {
            if (kformstruct->flag == KUIS_HELP)
            {
                status = TRUE;
            }
            else
            {
                kvf_attribute_error_mesg(routine, kformstruct, attribute);
                status = FALSE;
            }
        }

	/* 
	 * geometry applicable to most GUI items, a few exceptions 
	 */
	else if ((kstrcmp(attribute, KVF_WIDTH)  == 0) ||
                 (kstrcmp(attribute, KVF_HEIGHT) == 0) ||
                 (kstrcmp(attribute, KVF_X)      == 0) ||
                 (kstrcmp(attribute, KVF_Y)      == 0))
	{
	    switch(kformstruct->flag)
	    {
	       case KUIS_STARTMASTER:
	       case KUIS_BLANK:
	       case KUIS_INCLUDEPANE:
	       case KUIS_INCLUDESUBFORM:
	       case KUIS_MUTEXCL:
	       case KUIS_MUTINCL:
		    kvf_attribute_error_mesg(routine, kformstruct, attribute);
                    status = FALSE;
		    break;

	       default:
                    status = TRUE;
		    break;
	    }
	}

	/*
	 * can change the title offset on some UIS lines, some exceptions 
	 */
	else if ((kstrcmp(attribute, KVF_XPOS) == 0) ||
		 (kstrcmp(attribute, KVF_YPOS) == 0))
	{
	    switch(kformstruct->flag)
	    {
	       case KUIS_STARTFORM:
	       case KUIS_STARTSUBFORM:
	       case KUIS_STARTGUIDE:
	       case KUIS_STARTPANE:
	       case KUIS_TOGGLE:
	       case KUIS_BLANK:
                    status = TRUE;
		    break;

	       default:
		    kvf_attribute_error_mesg(routine, kformstruct, attribute);
                    status = FALSE;
		    break;

	    }
	}

        /*
         * mutually exclusive subform flag applicable only on startmaster line
         */
        else if ((kstrcmp(attribute, KVF_XPOS) == 0) ||
                 (kstrcmp(attribute, KVF_YPOS) == 0))
        {
            switch(kformstruct->flag)
            {
              case KUIS_STARTMASTER:
                   status = TRUE;
		   break;

               default:
                    kvf_attribute_error_mesg(routine, kformstruct, attribute);
                    status = FALSE;
                    break;
            }
        }
	/*
         * workspace valid only on workspace UIS lines
         */
        else if (kstrcmp(attribute, KVF_WORKSPACE) == 0)
        {
            switch(kformstruct->flag)
            {
              case KUIS_WORKSPACE:
                   status = TRUE;
                   break;

               default:
                    kvf_attribute_error_mesg(routine, kformstruct, attribute);
                    status = FALSE;
                    break;
            }
        }

        /*
         * guidepane title valid only on subform UIS lines
         */
        else if (kstrcmp(attribute, KVF_GUIDEPANETITLE) == 0)
        {
            switch(kformstruct->flag)
            {
              case KUIS_STARTSUBFORM:
                   status = TRUE;
                   break;

               default:
                    kvf_attribute_error_mesg(routine, kformstruct, attribute);
                    status = FALSE;
                    break;
            }
        }
	return(status);
}

/*------------------------------------------------------------
|
|  Routine Name: kvf_attribute_error_mesg
|
|       Purpose: Given an kformstruct and an attribute that
|                cannot be used together with kvf_set/get_attribute(s),
|                calls kerror() with an appropriate error message
|
|         Input: kformstruct - generic kform_struct
|                attribute   - the attribute
|        Output: none
|          Date: Jan 31, 1994
|    Written By: Danielle Argiro
| Modifications:
|
-------------------------------------------------------------*/

static void kvf_attribute_error_mesg(
   char         *routine_name,
   kform_struct *kformstruct,
   char         *attribute)
{
	char temp[KLENGTH];

	ksprintf(temp, "Incorrect call to %s; attribute %s cannot be resolved with kformstruct of type %s", 
		routine_name, attribute, kvf_ascii_typeflag(kformstruct->flag));
	errno = KCALL;
	kerror("kforms", routine_name, temp);
}

