/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>             Attribute Actions For Toggles              <<<<
   >>>>                                                       <<<<
   >>>>  Private:                                             <<<<
   >>>>	     		kvf_get_toggle_size()	              <<<<
   >>>>	     		kvf_set_toggle_size()	              <<<<
   >>>>	     		kvf_get_toggle_num()		      <<<<
   >>>>	     		kvf_set_toggle_num()		      <<<<
   >>>>	     		kvf_get_toggle_val()		      <<<<
   >>>>	     		kvf_set_toggle_val()		      <<<<
   >>>>	     		kvf_get_toggle_type()		      <<<<
   >>>>                                                       <<<<
   >>>>	     		kvf_begin_toggle_member()	      <<<<
   >>>>	     		kvf_switch_toggle_value()	      <<<<
   >>>>                                                       <<<<
   >>>>	     		kvf_get_toggle_contents()	      <<<<
   >>>>	     		kvf_get_toggle_contents_desc()	      <<<<
   >>>>   Static:                                             <<<<
   >>>>	     		toggle_member_num()		      <<<<
   >>>>	     		set_buffer_with_toggle_val()	      <<<<
   >>>>   Public:                                             <<<<
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<  */

#include "internals.h"


static int  toggle_member_num           PROTO((kselection *));
static int  set_buffer_with_toggle_val  PROTO((kselection *, char *));


/*-----------------------------------------------------------
|
|       Routine: toggle_member_num
|       Purpose: Counts the number of members in a toggle.
|
|         Input: toggle_start - header of the toggle (-T) line
|        Output: 
|       Returns: number of members in the toggle
|          Date: March 20, 1993
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
static int toggle_member_num(
    kselection *toggle_start)
{
	int        count = 0;
	kselection *toggle;

	if (toggle_start->toggle_next == NULL)
	    return(0);

	toggle = toggle_start->toggle_next;
	while (toggle != NULL)
	{
	    count++;
	    toggle = toggle->next;
	}
	return(count);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_get_toggle_size
|       Purpose: Gets the number of toggle values by counting
|                the selections that occur directly after 
|                the (-T) UIS line of a toggle selection
|                menuform field <= current selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_TOGGLE_SIZE ("kvf_toggle_size")
|        Output: calldata    - passes back number of toggle members
|
|          Date: March 20, 1993
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_toggle_size(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int *size;

        /* set the pointer to be returned */
        size = (int *) calldata;

	*size = toggle_member_num(kformstruct->Selptr);
	return(TRUE);
}


/*-----------------------------------------------------------
|
|       Routine: kvf_set_toggle_size
|       Purpose: Sets the number of toggle values by adding
|                or subtracting selections directly after
|                the (-T) UIS line of a toggle selection
|                menuform field => current selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_TOGGLE_SIZE ("kvf_toggle_member_num")
|                calldata    - provides number of toggle members
|        Output: none
|
|          Date: March 20, 1993
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_toggle_size(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int	     type, i, old_size, toggle_num;
	int          *new_size;
	kselection   *member, *old_member, *save_member, *toggle_start;
	
	new_size = (int *) calldata;

	/*
	 *  see how many toggle members there were before
	 */
	old_size = toggle_member_num(kformstruct->Selptr);
	
	/*
	 * same number of toggle members now as there was before: return
	 */
	if (old_size == *new_size) return(TRUE);
	
	/*
	 *  fewer toggle members now than before;  need
         *  to delete the excess toggle members
	 */
	if (*new_size < old_size)
	{
	    /*
	     * if the toggle was set to one of the members that are about
             * to be deleted, reset the toggle to the first member
             */
	    kvf_get_attribute(kformstruct, KVF_TOGGLE_NUM, &toggle_num);
	    if (toggle_num > *new_size)
		kvf_set_attribute(kformstruct, KVF_TOGGLE_NUM, 1);

	    /*
             *  seek past the last of the old members that will 
             *  remain part of the toggle; set the new end of the list
	     */
	    member = kformstruct->Selptr->toggle_next;
	    for (i = 1; i < *new_size+1; i++)
	    {
		save_member = member;
		member = member->next;
		if (i == *new_size) 
		    save_member->next = NULL;
	    }

	    /*
             *  delete the outdated members of the toggle
             */
	    old_member = member;
	    while (member != NULL)
	    {
		member = member->next;
		kvf_destroy_selection(old_member);
		old_member = member;
	    }
	}

	/*
	 * more toggle members than before;  need to
         * add in the requisite number of additional toggle members
         * and map them to the displayed GUI
	 */
	else if (*new_size > old_size)
	{
	    kvf_get_attribute(kformstruct, KVF_TOGGLE_TYPE, &type);

	    toggle_start = kformstruct->Selptr;

	    for (i = old_size+1; i <= *new_size; i++)
	    {
		member = kvf_begin_toggle_member(toggle_start, type, i);
	        if (type == KUIS_INTEGER)
	            kvf_set_attribute(kformstruct, KVF_INT_GT0_SEL, 1);
		else if (type == KUIS_FLOAT)
	            kvf_set_attribute(kformstruct, KVF_FLOAT_GT0_SEL, 1);
		else if (type == KUIS_DOUBLE)
	            kvf_set_attribute(kformstruct, KVF_DOUBLE_GT0_SEL, 1);
	    }
	}

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_get_toggle_num
|       Purpose: Gets the number (the index) of the toggle member 
|                that is currently selected.
|                menuform field <= current selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_TOGGLE_NUM ("kvf_toggle_num")
|        Output: calldata    - passes back number of selected toggle member
|
|          Date: March 20, 1993
|    Written By: Danielle Argiro
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_toggle_num(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
        Line_Info    line_info;
        int          *toggle_num;

        /* set the pointer to be returned */
        toggle_num = (int *) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

        if (line_info.toggle_val > 0)
            *toggle_num = line_info.toggle_val;
        else *toggle_num = line_info.toggle_def;

        kvf_free_line_info_strings(&line_info);
        return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_toggle_num
|       Purpose: Sets the value of a toggle selection to the member
|                with the number (the index) specified.
|                menuform field => current selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_TOGGLE_NUM ("kvf_toggle_num")
|                calldata    - provides number of member to be selected
|        Output: none
|
|          Date: March 20, 1993
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_toggle_num(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int          n;
        int          *toggle_num;
        kselection   *selection, *toggle;
	char         temp[KLENGTH];

        /* the value to which KVF_TOGGLE_NUM is to be set */
        toggle_num  = (int *) calldata;

	selection = kformstruct->Selptr;
	if (selection->back_toggle != NULL)
	     toggle = selection->back_toggle->toggle_next;
	else toggle = selection->toggle_next;

        n = 1;
        while (toggle != NULL)
        {
            /*
             * if its number is the same as the one requested,
             * reverse its colors, turning it on, and set it
             * to be selected
             */
            if (*toggle_num == n)
            {
                if (!(kvf_switch_toggle_value(toggle)))
                    return(FALSE);
		else return(TRUE);
            }
            n++;
            toggle = toggle->next;
        }
	ksprintf(temp, 
		"Illegal value provided for toggle '%s'; legal_values include:",
		ktoken_to_string(selection->var_token));
	kvf_append_toggle_values_to_buffer(selection, temp);
	kerror("kforms", "kvf_set_toggle_num", temp);
	return(FALSE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_get_toggle_val
|       Purpose: Gets the value of the toggle member that is
|                currently selected.  We don't know what data type
|                that value will be, so we put the value into a
|                string.
|                menuform field <= current selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_TOGGLE_VAL ("kvf_toggle_val")
|        Output: calldata    - passes back string representation
|                              of the toggle value
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_toggle_val(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	kselection   *toggle_member;
        char         **value_string;
	char         buffer[KLENGTH];
	int          count, toggle_num;


        /* set the pointer to be returned */
        value_string = (char **) calldata;

	if (!(kvf_get_attribute(kformstruct, KVF_TOGGLE_NUM, &toggle_num)))
	    return(FALSE);

	toggle_member = kformstruct->Selptr->toggle_next;
	count = 1;
	while ((count != toggle_num) && (toggle_member != NULL))
	{
	    count++;
	    toggle_member = toggle_member->next;
	}
	if (toggle_member == NULL)
	{
	    kerror("kforms", "kvf_get_toggle_val",
		   "Can't resolve current toggle num with toggle member list");
	    return(FALSE);
	}

	if ((toggle_member->type == KUIS_FLAG)  ||
                 (toggle_member->type == KUIS_LOGICAL))
        {
            sprintf(buffer, "%d", count);
        }
	else if (!(set_buffer_with_toggle_val(toggle_member, buffer)))
	{
            kerror("kforms", "kvf_get_toggle_val", "Unsupported toggle type");
	    return(FALSE);
	}

	*value_string = kstrdup(buffer);
        return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_toggle_val
|       Purpose: Sets the value of a toggle to the value  
|                specified.  We don't know what data type that 
|                value will be, so the value comes in as a string.
|                menuform field => current selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_TOGGLE_VAL ("kvf_toggle_val")
|                calldata    - provides string representation of 
|                              of the toggle value
|        Output: none
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_toggle_val(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	char         **value_string;
	int          found, toggle_type;
        int          status;
        char         buffer[KLENGTH];
        kselection   *toggle_member;

        /* string representing value to which KVF_TOGGLE_VAL is to be set */
        value_string  = (char **) calldata;

	if (*value_string == NULL)
	{
	    kerror("kforms", "kvf_set_toggle_val",
		   "Can't set toggle value to NULL");
	    return(FALSE);
	}

	toggle_member = kformstruct->Selptr->toggle_next;
	toggle_type   = toggle_member->type;

	/*
	 * changing the value of a logical or flag toggle is the
         * same procedure as changing the toggle int.
	 */
	if ((toggle_type == KUIS_LOGICAL) ||
	    (toggle_type == KUIS_FLAG))
	{
	    status = kvf_set_attribute(kformstruct, KVF_TOGGLE_NUM, 
				       atoi(*value_string));
	    return(status);
	}

	found = FALSE;
	while ((toggle_member != NULL) && (!found))
	{
	    if (!(set_buffer_with_toggle_val(toggle_member, buffer)))
	    {
		kerror("kforms", "kvf_change_toggleval",
		       "Unsupported toggle type");
		return(FALSE);
	    }
	    if (kstrncasecmp(buffer, *value_string,
			     kstrlen(*value_string)) == 0) 
		found = TRUE;
	    else toggle_member = toggle_member->next;
	}

	if (!found)
	{
	    ksprintf(buffer, "Illegal value provided for toggle; legal values include: ");
	    kvf_append_toggle_values_to_buffer(kformstruct->Selptr, buffer);
	    kerror("kforms", "kvf_change_toggleval", buffer);
	    return(FALSE);
	}

	if (!(kvf_switch_toggle_value(toggle_member)))
	     return(FALSE);

	return(TRUE);
}



/*-----------------------------------------------------------
|
|       Routine: kvf_get_toggle_type
|       Purpose: Gets the type of the toggle (InputFile, OutputFile,
|                Logical, Flag, Integer, Float, or String)
|                menuform field <= current selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_TOGGLE_TYPE ("kvf_toggle_type")
|        Output: calldata    - sets number of toggle member currently selected
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_toggle_type(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
        int          *toggle_type;

        /* set the pointer to be returned */
        toggle_type = (int *) calldata;

	/* default type is FLAG */ 
	if (kformstruct->Selptr->toggle_next == NULL) 
	     *toggle_type = KUIS_FLAG;  
	else *toggle_type = kformstruct->Selptr->toggle_next->type;

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_begin_toggle_member
|       Purpose: Creates a new toggle member of the type 
|                specified at the end of the toggle list.
|
|         Input: toggle_start - header of toggle list (-T selection)
|                type         - type of toggle to be created.
|                toggle_count - number of the toggle member (in order)
|
|        Output: none
|       Returns: TRUE on success, FALSE on failure
|          Date: April 7, 1993
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
kselection *kvf_begin_toggle_member(
    kselection *toggle_start,
    int        type,
    int        toggle_count)
{
	char       *line = NULL;
	char       temp[KLENGTH];
	Line_Info  line_info;
	kselection *new_toggle;
	
        kvf_clear_line_info(&line_info);

        line_info.typeflag    = type;

        line_info.activate    = 1;
        line_info.optional    = 1;
        line_info.selected    = 0;
        if (toggle_count == 1)
            line_info.opt_sel = 1;
        else line_info.opt_sel= 0;
        line_info.live        = 0;

        line_info.width       = 10.0;
        line_info.height      = 1.0;
        line_info.x           = 1.0;
        line_info.y           = (float) toggle_count;

        line_info.variable    = kstrdup("dummy");

	if (type == KUIS_FLAG)
	{
	    sprintf(temp, "Choice%d", toggle_count);
            line_info.title       = kstrdup(temp);
	    (void) kstring_lower(temp, temp);
            line_info.description = kstrdup(temp);
	}
	else if (type == KUIS_LOGICAL)
	{
	    line_info.logical_def   = 0;
	    line_info.val_labels = (char **) kcalloc(1, 2*sizeof(char *));
	    if (line_info.val_labels == NULL)
	    {
		kerror("kforms", "kvf_begin_toggle_member",
		       "Unable to allocate strings for toggle member labels");
		return(FALSE);
	    }
	    line_info.val_labels[0] = kstrdup("False");
	    line_info.val_labels[1] = kstrdup("TRUE");
	    sprintf(temp, "Choice%d", toggle_count);
	    line_info.title         = kstrdup(temp);
	    (void) kstring_lower(temp, temp);
	    line_info.description   = kstrdup(temp);
	}
	else if (type == KUIS_INTEGER)
	{
	    line_info.lower_int = 0;
	    line_info.upper_int = 0;
	    line_info.int_def   = toggle_count;
            line_info.exec_type = 1;
	    sprintf(temp, "%d", toggle_count);
	    line_info.title       = kstrdup(temp);
	    line_info.description = kstrdup(temp);
	}
	else if (type == KUIS_FLOAT)
	{
            line_info.lower_float = 0.0;
            line_info.upper_float = 0.0;
            line_info.float_def   = (float) toggle_count;
            line_info.exec_type   = 1;
	    sprintf(temp, "%f", (float) toggle_count);
	    line_info.title       = kstrdup(temp);
	    line_info.description = kstrdup(temp);
        }
	else if (type == KUIS_DOUBLE)
	{
            line_info.lower_double = 0.0;
            line_info.upper_double = 0.0;
            line_info.double_def   = (float) toggle_count;
            line_info.exec_type    = 1;
	    sprintf(temp, "%f", (double) toggle_count);
	    line_info.title       = kstrdup(temp);
	    line_info.description = kstrdup(temp);
        }
	else if (type == KUIS_STRING)
	{
	    sprintf(temp, "String%d", toggle_count);
	    line_info.string_def  = kstrdup(temp);
	    line_info.title       = kstrdup(temp);
	    (void) kstring_lower(temp, temp);
	    line_info.description = kstrdup(temp);
	}
	else if (type == KUIS_INPUTFILE)
	{
	    sprintf(temp, "Infile%d", toggle_count);
	    line_info.file_def   = kstrdup(temp);
	    line_info.title      = kstrdup(temp);
	    (void) kstring_lower(temp, temp);
	    line_info.description = kstrdup(temp);
	}
	else if (type == KUIS_OUTPUTFILE)
	{
	    sprintf(temp, "Outfile%d", toggle_count);
	    line_info.file_def   = kstrdup(temp);
	    line_info.title      = kstrdup(temp);
	    (void) kstring_lower(temp, temp);
	    line_info.description = kstrdup(temp);
	}

        kvf_gen_deparse(&line_info, &line);
        kvf_free_line_info_strings(&line_info);

	new_toggle = kvf_create_new_toggle_member(toggle_start, line);
	kfree(line);

	return(new_toggle);
}

/*-----------------------------------------------------------
|
|  Routine Name: kvf_switch_toggle_value
|
|       Purpose: Changes the current value of a toggle to
|                the member specified
|
|         Input: selection - pointer to the Toggle selectoin
|        Output:
|          Date: March 22, 1994
|    Written By: Danielle Argiro
| Modifications:
|
-------------------------------------------------------------*/
int kvf_switch_toggle_value(
    kselection *selection)
{
        kselection *start_toggle;
        int        int_def, precision;
        float      float_def;
        double     double_def;
        char       *string_def = NULL;
        kselection *old_selection;
        Line_Info  line_info;
        char       temp[KLENGTH], scale[KLENGTH];

        /*
         * if selection is not part of a toggle, that's an error
         */
        if (selection->back_toggle == NULL)
        {
            errno = KINTERNAL;
            kerror("kforms", "kvf_switch_toggle_value",
                   "trying to switch toggle value to non-toggle-member");
            return(FALSE);
        }
        start_toggle = selection->back_toggle;

        /*
         * search for the toggle member that was selected before
         */
        old_selection = start_toggle->toggle_next;
        while ((old_selection->opt_selected != TRUE) &&
               (old_selection->next != NULL))
            old_selection = old_selection->next;

        if ((old_selection->next == NULL) &&
            (!old_selection->opt_selected))
            return(FALSE);

        /*
         * if old selection same as new selection, can abort
         */
        if (old_selection == selection) return(TRUE);

        /*
         * switch off the old toggle member
         */
        if (old_selection->opt_selected)
        {
            /* change the UIS line to be not selected */
            kvf_set_attribute(old_selection->back_kformstruct, KVF_OPTSEL,  FALSE);
            old_selection->opt_selected = FALSE;
        }

        /*
         * set the new toggle member to be selected
         */
        kvf_set_attribute(selection->back_kformstruct, KVF_OPTSEL,  TRUE);
        selection->opt_selected = TRUE;

        /*
         * set the new value of the controlling Toggle selection
         */
        kvf_clear_line_info(&line_info);
        kvf_parse_toggle_line(start_toggle->line, &line_info);
        start_toggle->modified = TRUE;

        line_info.toggle_val = selection->toggle_num;
        line_info.opt_sel    = TRUE;
        line_info.toggle_def = selection->toggle_num;

        if ((selection->type == KUIS_FLAG) ||
            (selection->type == KUIS_LOGICAL))
        {
            sprintf(temp, "%d", line_info.toggle_val);
        }
        else if (selection->type == KUIS_INTEGER)
        {
            kvf_get_attribute(selection->back_kformstruct, 
			      KVF_INT_DEF, &int_def);
            sprintf(temp, "%d", int_def);
        }
        else if (selection->type == KUIS_FLOAT)
        {
            kvf_get_attributes(selection->back_kformstruct,
                               KVF_FLOAT_DEF,  &float_def,
                               KVF_FLOAT_PREC, &precision,
                               NULL);
            if (precision == 0)
                sprintf(scale, "%%g");
            else if (precision == -1)
                sprintf(scale, "%%f");
            else sprintf(scale, "%%.%df", precision);
            sprintf(temp, scale, float_def);
        }
        else if (selection->type == KUIS_DOUBLE)
        {
            kvf_get_attributes(selection->back_kformstruct,
                               KVF_DOUBLE_DEF,  &double_def,
                               KVF_DOUBLE_PREC, &precision,
                               NULL);
            if (precision == 0)
                sprintf(scale, "%%g");
            else if (precision == -1)
                sprintf(scale, "%%f");
            else sprintf(scale, "%%.%df", precision);
            sprintf(temp, scale, double_def);
        }
        else if ((selection->type == KUIS_INPUTFILE) ||
                 (selection->type == KUIS_OUTPUTFILE))
        {
            kvf_get_attribute(selection->back_kformstruct, KVF_FILE_DEF,
                              &string_def);
            kfree(string_def);
        }
        else if (selection->type == KUIS_STRING)
        {
            kvf_get_attribute(selection->back_kformstruct, KVF_STRING_DEF,
                              &string_def);
            kfree(string_def);
        }
        kvf_deparse_toggle_line(&line_info, &start_toggle->line);
        kvf_free_line_info_strings(&line_info);

        return(TRUE);

}

/*-----------------------------------------------------------
|
|       Routine: kvf_get_toggle_contents
|       Purpose: Gets an array of strings containing the toggle
|                values for each toggle member.
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_TOGGLE_CONTENTS ("kvf_toggle_contents")
|        Output: calldata    - passes back array of strings with the
|                              toggle values
|          Date: March 30, 1994
|    Written By: Danielle Argiro
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_toggle_contents(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int        i, toggle_size, toggle_type;
	char       ***contents;
	kselection *selection, *toggle;
	char       buffer[KLENGTH];

	contents = (char ***) calldata;
	selection = kformstruct->Selptr;

	kvf_get_attributes(kformstruct, 
			   KVF_TOGGLE_SIZE, &toggle_size,
			   KVF_TOGGLE_TYPE, &toggle_type,
			   NULL);
	*contents = (char **) kcalloc((unsigned)toggle_size, 
					sizeof(char *));
	i = 0;
	toggle = selection->toggle_next;
	while (toggle != NULL)
	{
	    if ((toggle_type == KUIS_FLAG) ||
		(toggle_type == KUIS_LOGICAL))
		sprintf(buffer, "%d", i+1);
	    else 
	    {
		if (!(set_buffer_with_toggle_val(toggle, buffer)))
		{
		    kerror("kforms", "kvf_get_toggle_contents",
		           "Cannot get toggle value(s)");
		    return(FALSE);
		}
	    }
	    (*contents)[i] = kstrdup(buffer);
	    toggle = toggle->next; i++;
	}
	return(TRUE);
	
}

/*-----------------------------------------------------------
|
|       Routine: kvf_get_toggle_contents_desc
|       Purpose: Gets an array of strings containing the descriptions
|                for each toggle member.
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_TOGGLE_CONTENTS_DESC 
|                             ("kvf_toggle_contents_desc")
|        Output: calldata    - passes back array of strings with the
|                              toggle member descriptions
|
|          Date: March 30, 1994
|    Written By: Danielle Argiro
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_toggle_contents_desc(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
        int        i, toggle_size;
        char       *description;
        char       ***descriptions;
        kselection *selection, *toggle;

        descriptions = (char ***) calldata;
        selection = kformstruct->Selptr;

        if (!(kvf_get_attribute(kformstruct, KVF_TOGGLE_SIZE, 
				&toggle_size)))
	    return(FALSE);

        *descriptions = (char **) kcalloc((unsigned)toggle_size, 
				           sizeof(char *));
	if (*descriptions == NULL)
	{
	    kerror("kforms", "kvf_get_toggle_contents_desc",
		   "Unable to allocate descriptions array");
	    return(FALSE);
	}

        i = 0;
        toggle = selection->toggle_next;
        while (toggle != NULL)
        {
            if (!(kvf_get_attribute(toggle->back_kformstruct, 
			            KVF_DESCRIPTION, &description)))
		    return(FALSE);
            (*descriptions)[i] = description;
            toggle = toggle->next; i++;
        }
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: set_buffer_with_toggle_val
|       Purpose: Given a buffer, fills it with the string representation
|                of the current toggle value, regardless of the toggle type.
|
|         Input: toggle_member - member of toggle which is currently selected
|                buffer        - buffer in which to put value
|        Output:
|       Returns: TRUE on success, FALSE on failure
|          Date: March 30, 1994
|    Written By: Danielle Argiro
| Modifications:
|
------------------------------------------------------------*/
static int set_buffer_with_toggle_val(
    kselection *toggle_member,
    char       *buffer)
{
	int     intdef;
        float   floatdef;
        double  doubledef;
	char    *string_def;

	if ((toggle_member->type == KUIS_INPUTFILE) ||
            (toggle_member->type == KUIS_OUTPUTFILE))
        {
            if (!(kvf_get_attribute(toggle_member->back_kformstruct,
                                  KVF_FILE_DEF, &string_def)))
		return(FALSE);
            sprintf(buffer, "%s", string_def);
	    kfree(string_def);
        }

        else if (toggle_member->type == KUIS_STRING)
        {
            if (!(kvf_get_attribute(toggle_member->back_kformstruct,
                                  KVF_STRING_DEF, &string_def)))
		return(FALSE);
            sprintf(buffer, "%s", string_def);
	    kfree(string_def);
        }

        else if (toggle_member->type == KUIS_FLOAT)
        {
            if (!(kvf_get_attribute(toggle_member->back_kformstruct,
                                    KVF_FLOAT_DEF,  &floatdef)))
		return(FALSE);
            sprintf(buffer, "%g", floatdef);
        }

        else if (toggle_member->type == KUIS_DOUBLE)
        {
            if (!(kvf_get_attribute(toggle_member->back_kformstruct,
                                    KVF_DOUBLE_DEF,  &doubledef)))
		return(FALSE);
            sprintf(buffer, "%g", doubledef);
        }

        else if (toggle_member->type == KUIS_INTEGER)
        {
            kvf_get_attribute(toggle_member->back_kformstruct,
                              KVF_INT_DEF, &intdef);
            sprintf(buffer, "%d", intdef);
        }

	else 
	{
	    kerror("kforms", "set_buffer_with_toggle_val",
		   "Unexpected toggle type");
	    return(FALSE);
	}
	return(TRUE);
}
