/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $GUI/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>	        Attribute Actions For Strings & Files         <<<<
   >>>>                                                       <<<<
   >>>>  Private:                                             <<<<
   >>>>	     		kvf_get_file_name()		      <<<<
   >>>>	     		kvf_set_file_name()		      <<<<
   >>>>	     		kvf_get_file_check()		      <<<<
   >>>>	     		kvf_set_file_check()		      <<<<
   >>>>	     		kvf_get_file_def()		      <<<<
   >>>>	     		kvf_set_file_def()		      <<<<
   >>>>	     		kvf_get_string_val()		      <<<<
   >>>>	     		kvf_set_string_val()		      <<<<
   >>>>	     		kvf_get_string_def()		      <<<<
   >>>>	     		kvf_set_string_def()		      <<<<
   >>>>	     		kvf_get_string_multiline()	      <<<<
   >>>>	     		kvf_set_string_multiline()	      <<<<
   >>>>   Static:                                             <<<<
   >>>>   Public:                                             <<<<
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<  */

#include "internals.h"


/*-----------------------------------------------------------
|
|       Routine: kvf_get_file_name
|       Purpose: Gets the value of the "filename" field for a selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_FILE_NAME ("kvf_file_name")
|        Output: calldata    - passes back value of filename
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_file_name(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	Line_Info    line_info;
	int          num;
	char         **filename;
	kselection   *toggle;

        /* set the pointer to be returned */
        filename = (char **) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

	/* special case: cantata needs the file name from toggle selections */
        if (kformstruct->Selptr->type == KUIS_TOGGLE)
        {
            toggle = kformstruct->Selptr->toggle_next;
            num = 1;
            while (num < line_info.toggle_val)
            {
                toggle = toggle->next;
                num++;
            }
            if ((toggle->type == KUIS_INPUTFILE) ||
                (toggle->type == KUIS_OUTPUTFILE))
            {
                kvf_free_line_info_strings(&line_info);
                kvf_gen_parse(toggle->line, &line_info);
                *filename = kstrdup(line_info.file_def);
		kvf_free_line_info_strings(&line_info);
		return(TRUE);
            }
            else
            {
		*filename = NULL;
		return(FALSE);
            }
        }
	else if (line_info.filename == NULL)
	    *filename = NULL;
	else *filename = kstrdup(line_info.filename);

	kvf_free_line_info_strings(&line_info);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_file_name
|       Purpose: Sets the "filename" field of the selection
|                according to the value of the "filename" selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_FILE_NAME ("kvf_file_name")
|                calldata    - provides string value of filename
|        Output: none
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
-------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_file_name(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	char         mesg[KLENGTH];
	char         temp[KLENGTH];
	char         **filename;
	Line_Info    line_info;

	/* the value to which KVF_FILE_NAME is to be set */
        filename  = (char **) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

	if ((line_info.typeflag == KUIS_INPUTFILE) && 
	    (line_info.file_check != FALSE)        &&
	    (kstrlen(*filename) > 0))
	{
	    if ((kaccess(*filename, R_OK)) == -1)
	    {
	        if (kvf_context == KVF_CLUI)
		   ksprintf(temp, "CLUI argument '%s'", line_info.variable);
	        else ksprintf(temp, "GUI selection '%s'", line_info.title);

		if ((kaccess(*filename, F_OK)) == -1)
                   ksprintf(mesg, 
			    "File '%s' specified for %s does not exist.", 
			    *filename, temp);
                else ksprintf(mesg, 
			      "File '%s' specified for %s is not readable.",
                              *filename, temp);
		kerror("kforms", "kvf_set_file_name", mesg);
		return(FALSE);
	    }
	}

        kfree(line_info.filename);
        if (*filename == NULL)
            line_info.filename = NULL;
        else line_info.filename = kstrdup(*filename);
	
        kvf_gen_deparse_kformstruct(kformstruct, &line_info);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_get_file_check
|       Purpose: Gets the value of the "file check" field for a selection.
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_FILE_CHECK ("kvf_file_check")
|        Output: calldata    - passes back value of filename
|
|          Date: November 29, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_file_check(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
        Line_Info    line_info;
        int          *file_check;

        /* set the pointer to be returned */
        file_check = (int *) calldata;

        if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
            return(FALSE);

        *file_check = line_info.file_check;

        kvf_free_line_info_strings(&line_info);
        return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_file_check
|       Purpose: Sets the "filecheck" field of the selection,
|                This routine was written specially for xvlang,
|                in cases where cantata is dealing with operators
|                that use basenames for files with predefined postfixes,
|                and do their own file checking;  it can be used
|                to suppress file checking (set file_check = 0).
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_FILE_CHECK ("kvf_file_check")
|                calldata    - provides boolean value of file check
|        Output: none
|
|          Date: November 29, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
-------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_file_check(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int         *file_check;
	Line_Info    line_info;

	/* the value to which KVF_FILE_CHECK is to be set */
        file_check  = (int *) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

        line_info.file_check = *file_check;

        kvf_gen_deparse_kformstruct(kformstruct, &line_info);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_get_file_def
|       Purpose: Gets the value of the "file_def" field for a 
|                file selection.
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_FILE_DEF ("kvf_file_def")
|        Output: calldata    - passes back default filename 
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_file_def(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	Line_Info    line_info;
	char         **file_def;

        /* set the pointer to be returned */
        file_def = (char **) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

	if (line_info.file_def == NULL)
	    *file_def = NULL;
	else *file_def = kstrdup(line_info.file_def);

	kvf_free_line_info_strings(&line_info);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_file_def
|       Purpose: Sets the "file_def" field of the selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_FILE_DEF ("kvf_file_def")
|                calldata    - provides default filename of file selection
|        Output: none
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
-------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_file_def(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	char         **file_def;
	Line_Info    line_info;
	kselection   *selection;

	/* the value to which KVF_FILE_DEF is to be set */
        file_def  = (char **) calldata;

	selection = kformstruct->Selptr;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

	if (kstrcmp(line_info.file_def, *file_def) == 0)
	{
	    kvf_free_line_info_strings(&line_info);
	    return(TRUE);
	}

        kfree(line_info.file_def);
        if (*file_def == NULL)
            line_info.file_def = NULL;
        else line_info.file_def = kstrdup(*file_def);

        kvf_gen_deparse_kformstruct(kformstruct, &line_info);

        if (!(kvf_set_attribute(kformstruct, KVF_FILE_NAME, *file_def)))
	    return(FALSE);

	if (selection->back_toggle != NULL)
	{
            if (!(kvf_set_attribute(kformstruct, KVF_TITLE, *file_def)))
	        return(FALSE);
	}

	return(TRUE);
}
/*-----------------------------------------------------------
|
|       Routine: kvf_get_string_val
|       Purpose: Gets the value of the "string_val" field for a 
|                string selection.
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_STRING_VAL ("kvf_string_val")
|        Output: calldata    - passes back value of string selection
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_string_val(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	Line_Info    line_info;
	char         **string_val;

        /* set the pointer to be returned */
        string_val = (char **) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

	if (line_info.string_val == NULL)
	    *string_val = NULL;
	else *string_val = kstrdup(line_info.string_val);

	kvf_free_line_info_strings(&line_info);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_string_val
|       Purpose: Sets the "filename" field of the selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_STRING_VAL ("kvf_string_val")
|                calldata    - provides string value of string selection
|        Output: none
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
-------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_string_val(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	char         **string_val;
	Line_Info    line_info;

	/* the value to which KVF_STRING_VAL is to be set */
        string_val  = (char **) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

        kfree(line_info.string_val);
	if (*string_val == NULL)
            line_info.string_val = NULL;
	else line_info.string_val = kstrdup(*string_val);

        kfree(line_info.literal);
        if (kstrlen(*string_val) > 0)
            line_info.literal = kstrdup(*string_val);
        else line_info.literal = NULL;

        kvf_gen_deparse_kformstruct(kformstruct, &line_info);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_get_string_def
|       Purpose: Gets the value of the "string_def" field for a 
|                string selection.
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_STRING_DEF ("kvf_string_def")
|        Output: calldata    - passes back default of string selection
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_string_def(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	Line_Info    line_info;
	char         **string_def;

        /* set the pointer to be returned */
        string_def = (char **) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

	if (line_info.string_def == NULL)
	    *string_def = NULL;
	else *string_def = kstrdup(line_info.string_def);

	kvf_free_line_info_strings(&line_info);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_string_def
|       Purpose: Sets the "string_def" field of the selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_STRING_DEF ("kvf_string_def")
|                calldata    - provides default string of string selection
|        Output: none
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
-------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_string_def(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	char          **string_def;
	Line_Info    line_info;
 	kselection   *selection;

	/* the value to which KVF_STRING_DEF is to be set */
        string_def  = (char **) calldata;

	selection = kformstruct->Selptr;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);
        kfree(line_info.string_def);

	if (*string_def == NULL)
            line_info.string_def = NULL;
	else line_info.string_def = kstrdup(*string_def);

        kvf_gen_deparse_kformstruct(kformstruct, &line_info);

        if (!(kvf_set_attribute(kformstruct, KVF_STRING_VAL, *string_def)))
	    return(FALSE);

	if (selection->back_toggle != NULL)
	{
            if (!(kvf_set_attribute(kformstruct, KVF_TITLE, *string_def)))
	        return(FALSE);
	}

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_get_string_multiline
|       Purpose: Gets the implied value of the "multiline" attribute
|                as determined by the height of a string selection.
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_STRING_MULTILINE ("kvf_string_multiline")
|        Output: calldata    - passes back TRUE for multiline, FALSE otherwise.
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_string_multiline(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	Line_Info    line_info;
	int          *multiline;

        /* set the pointer to be returned */
        multiline = (int *) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

	if (line_info.height > 1.0)
	    *multiline = TRUE;
	else *multiline = FALSE;

	kvf_free_line_info_strings(&line_info);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_string_multiline
|       Purpose: Sets the implied value of the "multiline" attribute
|                and sets the height of the string selection accordingly.
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_STRING_MULTILINE ("kvf_string_multiline")
|                calldata    - provides default string of string selection
|        Output: none
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
-------------------------------------------------------------*/ 
/* ARGSUSED */ 
int kvf_set_string_multiline( 
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int          *multiline;
	Line_Info    line_info;

	/* the value to which KVF_MULTILINE is to be set */
        multiline  = (int *) calldata;

	if (!(kvf_gen_parse_kformstruct(kformstruct, &line_info)))
	    return(FALSE);

	if (!(*multiline))
	    line_info.height = 1.0;

        kvf_gen_deparse_kformstruct(kformstruct, &line_info);

	return(TRUE);
}
