 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>      Various Private Print Routines For the kexpr Library
   >>>>
   >>>>  Private:
   >>>>             print_symlist()
   >>>>
   >>>>            _kexpr_print_recursive()
   >>>>            kexpr_print_symbol()
   >>>>            kexpr_print_symlist()
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"
#include "y.tab.h"

/*-----------------------------------------------------------
|
|  Routine Name: print_symlist - print out symbols
|
|       Purpose: print_symlist() is called to actually print out
|		 the symbols.  print_symlist() is called recursively
|		 just like eval_symlist() to print out the symbols
|		 that are used to represent an expression.
|
|         Input: file     - the file stream to print to
|                symlist  - the current symbol list in which to start
|			    printing symbols
|                abort_when_undefined - whether to abort when an undefined
|					variable is found
|
|        Output: 
|                RETURNS: the current symbol list position
|
|    Written By: Mark Young  
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

static SymbolList *print_symlist(
   kfile      *file,
   SymbolList *symlist,
   int        abort_when_undefined)
{
	int	   i;
	Symbol	   *symbol, *temp;
	char	   error[KLENGTH];


	/*
	 *  If the current symlist is NULL then the endlist will become
	 *  the beginning of the symbol list, so we just return it.
	 */
	if (symlist == NULL)
	{
	   ksprintf(error,"Error! end of symbol list encountered, but more \
expected.");
	   kexpr_error(error);
	}

	symbol = symlist->symbol;
	switch (symbol->type)
	{
	   case NUMBER:
		kfprintf(file,"%g", symbol->Value);
		symlist = symlist->next;
		break;

	   case STRING:
		kfprintf(file,"%s", symbol->String);
		symlist = symlist->next;
		break;

	   case VARIABLE:
	   case CONSTANT:
	   case EXPRESSION:
		kfprintf(file,"%s", symbol->name);
		symlist = symlist->next;
		break;

	   case UFUNCTION:
	   case FUNCTION:
		{
	           register int num = symbol->num;

		   kfprintf(file,"%s(", symbol->name);
		   symlist = symlist->next;
	           for (i = 0; i < num; i++)
		   {
		       symlist=print_symlist(file,symlist,abort_when_undefined);

		       if (i+1 < num)
		          kfprintf(file,",");
		   }

		   kfprintf(file,")");
		   break;
		}

	   case OPERATOR:
		symlist = symlist->next;

		if (symbol->Operator == STRCAT)
		{
		   temp = symlist->symbol;
		   if (temp->type != STRING && temp->type != OPERATOR)
		      kfprintf(file,"$");
		   symlist = print_symlist(file, symlist, abort_when_undefined);

		   temp = symlist->symbol;
		   if (temp->type != STRING && temp->type != OPERATOR)
		      kfprintf(file,"$");
		   symlist = print_symlist(file, symlist, abort_when_undefined);
		}
		else
		{
		   kfprintf(file,"(");
		   symlist = print_symlist(file, symlist, abort_when_undefined);

		   kfprintf(file,"%s", symbol->name);
		   symlist = print_symlist(file, symlist, abort_when_undefined);

		   kfprintf(file,")");
		}
		break;

	   case UOPERATOR:
		kfprintf(file,"%s", symbol->name);
		symlist = symlist->next;
		symlist = print_symlist(file, symlist, abort_when_undefined);
		break;

	   case INSTRUCTION:
	        if (symbol->Instruction == IF)
		{
		   kfprintf(file,"(");
		   symlist = symlist->next;
		   symlist = print_symlist(file, symlist, abort_when_undefined);
		   kfprintf(file,") ? (");
		   symlist = print_symlist(file, symlist, abort_when_undefined);
		   kfprintf(file,") : (");
		   symlist = print_symlist(file, symlist, abort_when_undefined);
		   kfprintf(file,")");
		}
		break;

	   case UNDEFINED:
		if (abort_when_undefined)
		{
		   ksprintf(error,"Error! Unknown variable or constant '%s'",
				symbol->name);
		   kexpr_error(error);
		}
		kfprintf(file,"%s", symbol->name);
		symlist = symlist->next;
		break;

	   default:
		ksprintf(error,"Error!  Unknown symbol type '%d'",symbol->type);
		kexpr_error(error);
		break;
	}
	return(symlist);
}

/*-----------------------------------------------------------
|
|  Routine Name: _kexpr_recursive - is the expression recursive
|
|       Purpose: _kexpr_recursive() - is used to indicate whether
|		 an expression is recursive or not.  This is helpful
|		 when trying to recurse a symbol list and you may
|		 recursively print the same symbol list.
|
|         Input: symbol  - the symbol which is to be checked
|                symlist - the symbol list which we check the symbol
|			   against
|
|        Output: RETURNS: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young  
|          Date: Thu Jun 25 1992
| Modifications:
|
------------------------------------------------------------*/

static int _kexpr_recursive(
   Symbol *symbol,
   Symbol *symlist)
{
	return(FALSE);
}

/*-----------------------------------------------------------
|
|  Routine Name: kexpr_print_symbol - print out symbol definition
|
|       Purpose: kexpr_print_symbol() -s called when we want to print
|		 out a symbol. 
|
|        Input:  file     - the file stream to print to
|                symbol   - the current symbol to be printed
|                abort_when_undefined - whether to abort when an undefined
|					variable is found
|
|        Output: prints to a file the symbol & symlist
|
|    Written By: Mark Young  
|          Date: Thu Jun 25 1992
| Modifications:
|
------------------------------------------------------------*/

void kexpr_print_symbol(
   kfile  *file,
   Symbol *symbol,
   int    abort_when_undefined)
{
	int	   i;
	Symbol     *arg, result;
	SymbolList list, *symlist;
	char       error[KLENGTH];


	/*
	 *  If the current symbol is NULL then print out an error
	 *  and abort.
	 */
	if (symbol == NULL)
	{
	   ksprintf(error,"Error! NULL symbol list encountered.\n");
	   kexpr_error(error);
	}

 	if (symbol->type == VARIABLE)
	{
	   kexpr_eval_symlist(symbol->symlist, FALSE, &result);

	   kfprintf(file,"# Dependant expression '%s'\n", symbol->name);
	   kfprintf(file,"# current evaluated value = %g\n", result.Value);
	   if (_kexpr_recursive(symbol, NULL))
	   {
	      kfprintf(file,"# recursive base value = %g\n",
			result.Value, symbol->Value);
	   }
	   kfprintf(file,"  %s = ", symbol->name);
	   (void) print_symlist(file, symbol->symlist, abort_when_undefined);
	   kfprintf(file,"\n\n");
	}
	else if (symbol->type == EXPRESSION)
	{
	   kfprintf(file,"# Simple expression '%s'\n", symbol->name);
	   if (symbol->symlist->symbol->type != NUMBER)
	   {
	      list.symbol = symbol;
	      list.next   = NULL;
	      kexpr_eval_symlist(&list, FALSE, &result);
	      kfprintf(file,"# current evaluated value = %g\n", result.Value);
	   }
	   kfprintf(file,"%s = ", symbol->name);
	   (void) print_symlist(file, symbol->symlist, abort_when_undefined);
	   kfprintf(file,"\n\n");
	}
	else if (symbol->type == UFUNCTION)
	{
	   register int num = symbol->num;

	   kfprintf(file,"# User defined function '%s()'\n", symbol->name);
	   kfprintf(file,"  %s(", symbol->name);
	   symlist = symbol->symlist;
	   for (i = 0; i < num; i++)
	   {
	      arg = symlist->symbol; 
	      kfprintf(file,"%s", arg->name);

	      if (i+1 < num)
	         kfprintf(file,",");

	      symlist = symlist->next;
	   }
	   kfprintf(file,") = ");
	   (void) print_symlist(file, symlist, abort_when_undefined);
	   kfprintf(file,"\n\n");
	}
	else if (symbol->type == FUNCTION)
	{
	   register int num = symbol->num;

	   kfprintf(file,"# Pre-defined function '%s()'\n", symbol->name);
	   kfprintf(file,"  %s(", symbol->name);
	   symlist = symbol->symlist;
	   for (i = 0; i < num; i++)
	   {
	      kfprintf(file,"arg%1d", i);

	      if (i+1 < num)
	         kfprintf(file,",");
	   }
	   kfprintf(file,")\n\n");
	}
	else if (symbol->type == UNDEFINED)
	{
	   kfprintf(file,"# Currently Undefined symbol '%s'\n", symbol->name);
	   list.symbol = symbol;
	   list.next   = NULL;
	   (void) print_symlist(file, &list, abort_when_undefined);
	   kfprintf(file,"\n\n");
	}
	else
	{
	   kfprintf(file,"# Unknown symbol type\n");
	   list.symbol = symbol;
	   list.next   = NULL;
	   (void) print_symlist(file, &list, abort_when_undefined);
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: kexpr_print_symlist - print out symbol list definition
|
|       Purpose: kexpr_print_symbol() -s called when we want to print
|		 out a symbol list.
|
|        Input:  file     - the file stream to print to
|                symlist  - the current symbol list to be printed
|                abort_when_undefined - whether to abort when an undefined
|					variable is found
|
|        Output: prints to a file the symlist definition
|
|    Written By: Mark Young  
|          Date: Thu Jun 25 1992
| Modifications:
|
------------------------------------------------------------*/

void kexpr_print_symlist(
   kfile      *file,
   SymbolList *symlist,
   int        abort_when_undefined)
{
	char  error[KLENGTH];


	/*
	 *  If the current symlist is NULL then the endlist will become
	 *  the beginning of the symbol list, so we just return it.
	 */
	if (symlist == NULL)
	{
	   ksprintf(error,"Error! NULL symbol list encountered.\n");
	   kexpr_error(error);
	}
	kfprintf(file,"expression: ");
	(void) print_symlist(file, symlist, abort_when_undefined);
	kfprintf(file,"\n");
}
