/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */



/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>       Routines to generate GUI Info structure         <<<<
   >>>>            for xvroutines in form_info.h              <<<<
   >>>>                                                       <<<<
   >>>>  Private:                                             <<<<
   >>>>              kgen_gui_include_file()                  <<<<
   >>>>   Static:                                             <<<<
   >>>>              gen_subform_struct()                     <<<<
   >>>>              gen_driver_decls()                       <<<<
   >>>>              gen_doroutine_decls()                    <<<<
   >>>>   Public:                                             <<<<
   >>>>                                                       <<<<
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"

static int 
gen_subform_struct    PROTO((kfile *, ksubform *, char *, char *));

static int 
gen_selection_structs PROTO((kfile *, kselection *));

static void 
gen_driver_decls      PROTO((kfile *, kform *, char *, char *));

static void 
gen_doroutine_decls   PROTO((kfile *, kselection *, char *, char *));



/*--------------------------------------------------------------
|
|  Routine Name: kgen_gui_include_file
|
|       Purpose: Creates the forminfo.h file, in which is the
|                FormInfo structure which will be used to pass 
|                values input by the user to the GUI back to the
|                application.  
|
|         Input: form         - pointer to the form tree
|		 object       - the software object being generated.
|                program_name - name of software object being generated.
|                levels       - level of extraction [-l] 
|
|        Output: Returns TRUE if the form_info.h file was created
|                successfully, FALSE otherwise
|
|          Date: May 09, 1992
|    Written By: Danielle Argiro 
| Modifications:
|
-------------------------------------------------------------*/

int kgen_gui_include_file(
   kform   *form,
   kobject program,
   char    *program_name,
   int     levels)
{
        kstring    routine = "kgen_gui_include_file()";
	kfile 	   *file; 
	kobject    toolbox;
	char       *form_var, *subform_var, *copyright; 
	char       *sf_variables[100]; 
	char       purpose[KLENGTH], temp[KLENGTH]; 
	int        user_refused, sf_count = 0, master_quit = 0;
	ksubform   *subform;
	kselection *selection;

        /*
	 *  first of all, if 'form_info.h' exists, see if they want to over-
	 *  write it.  If not, return immediately.
	 */
	file = kgen_open_file(program, "form_info.h", &user_refused);
	if (user_refused) return(TRUE);
	else if (file == NULL) 
	{
	    kerror(KCODEGEN, "kgen_gui_include_file",
                   "Could not create file 'form_info.h'");
            return(FALSE);
	}
	
	/* begin file with RCS header */
	kgen_rcs_hhdr(file);

	/*
	 * generate the copyright
	 */
	if (kcms_get_attribute(program, KCMS_PARENT, &toolbox))
	{
	   copyright = kcms_get_copyright(toolbox, KCMS_TB_COPYRIGHT_LONG,
					  KCMS_LANG_C);
	   if (copyright != NULL)
	      kfprintf(file, "%s\n", copyright);
	   kfree(copyright);
	}

	/* get the variable off the -F line */

	form_var = ktoken_to_string(form->var_token);

	/* print the include file header */
	ksprintf(purpose, "Form Info Structure for %s Form", program_name);
	ksprintf(temp, "conductor -name %s -l %d", program_name, levels);
	kgen_inclfile_hdr(file, purpose, temp, NULL);

	/* print ifdef's */
	kfprintf(file, "#ifndef _form_info_h_\n");
	kfprintf(file, "#define _form_info_h_\n\n");

	/* write to file the necessary #includes */
	kfprintf(file,"#include <xvinclude.h>\n\n");

	/* print stdized comment preceding includes */
        kgen_inclfile_incl_comm(file);

	/* print stdized comment preceding defines */
        kgen_inclfile_def_comm(file);

	/* print stdized comment preceding typedefs */
	kgen_inclfile_typedef_comm(file);
	    
	if (form->master != NULL)
        {
            subform = form->master->subform_list;
            selection = form->master->sel_list;
        }
        else
        {
            subform = form->subform;
            selection = NULL;
        }

	/*
	 * collect variable program_names from subforms & save for 
         * later declaration. generate appropriate structures for each subform.
	 */
        while (subform != NULL)
	{
	    sf_variables[sf_count] = ktoken_to_string(subform->var_token);

	    if (!(gen_subform_struct(file, subform, form_var,
				     sf_variables[sf_count])))
		return(FALSE);
	    sf_count++;
	    subform = subform->next;
	}

	ksprintf(temp, " * Form Info Struct for '%s' form", form_var);
	kfprintf(file, "/*\n%s\n */\n", temp);
	kfprintf(file, "typedef struct _gui_info_struct {\n\n");

	/*  
	 * declare variables for selections on master 
         */
	if (!(gen_selection_structs(file, selection)))
	    return(FALSE);

	/* 
	 * declare variables for subforms
 	 */
	if (form->master != NULL)
            subform = form->master->subform_list;
        else subform = form->subform;

	while (subform != NULL)
        {
	     subform_var = ktoken_to_string(subform->var_token);
    	     kfprintf(file, "%s_%s  *%s;\n", form_var,
                      subform_var, subform_var);
	     kfprintf(file, "int %s_selected;\n", subform_var);
             kfprintf(file, "kform_struct *%s_struct;\n\n",
                      subform_var);
	     subform = subform->next;
        }

	if (master_quit == TRUE)
	{
           kfprintf(file, "int quit;\n");
           kfprintf(file, "kform_struct *quit_struct;\n\n");
        }

	if (form->master != NULL)
	    kfprintf(file, "\nkform_struct *%s_struct;\n\n", form_var);

	kfprintf(file, "\n } gui_info_struct; \n\n");

	/* print stdized comment preceding global variable declarations */
        kgen_inclfile_global_comm(file);

	/* print stdized comment preceding macros*/
        kgen_inclfile_macro_comm(file);

	/* print stdized comment preceding routine defs */
	kgen_inclfile_routine_comm(file);
        gen_driver_decls(file, form, form_var, program_name);

	/* end #ifdefs */
	kfprintf(file, "#endif /* _form_info_h_ */ \n");
	kfprintf(file, "/* Don't add after this point */\n");
	kfclose(file);

        kannounce(KCODEGEN, routine, "done generating 'form_info.h'");

	return(TRUE);

} /* end kgen_gui_include_file */



/*--------------------------------------------------------------
|
|  Routine Name: gen_subform_struct
|
|       Purpose: Generates the structure needed for a subform
|
|         Input: file        - open stream to form_info.h file
|                subform     - pointer to subform
|                form_var    - program_name of the form variable
|                subform_var - program_name of the subform variable
|
|	 Output: Returns TRUE if the SubformInfo structures 
|                were successfully created in form_info.h,
|                FALSE otherwise.
|
|          Date: May 09, 1992
|    Written By: Danielle Argiro
| Modifications:
|
-------------------------------------------------------------*/

static int gen_subform_struct(
   kfile    *file,
   ksubform *subform,
   char     *form_var,
   char     *subform_var)
{
	int	   spacenum, varsize, i, j; 
	int        panecount = 0, guide_quit = 0;
	char	   *pn_variables[50], temp[KLENGTH];
	kguide     *guide;
	kselection *selection;

	if (subform->guidepane != NULL)
        {
            guide = subform->guidepane->guide_list;
            selection = subform->guidepane->sel_list;
        }
        else
        {
            guide = subform->guide;
            selection = NULL;
        }
        while (guide != NULL)
	{
            pn_variables[panecount] = ktoken_to_string(guide->pane->var_token);

	    if ((guide->pane != NULL) &&
		(guide->pane->sel_list != NULL) &&
		(kgen_valid_selections(guide->pane->sel_list)))
	    {
	        ksprintf(temp, " * Pane Info Struct for '%s' Pane", 
                         pn_variables[panecount]);
	        kfprintf(file, "/*\n%s\n */\n", temp);
                kfprintf(file, "typedef struct _%s_%s{\n\n",
			 subform_var, pn_variables[panecount]);
                gen_selection_structs(file, guide->pane->sel_list);
                kfprintf(file, "} %s_%s;\n\n",
			 subform_var, pn_variables[panecount]);
	    }
	    guide = guide->next;
	    panecount++;
	} 

	/* print the first line of the SubformInfo definition */
	ksprintf(temp, " * Subform Info Struct for '%s' Subform", subform_var);
	kfprintf(file, "/*\n%s\n */\n", temp);
	kfprintf(file, "typedef struct _%s_%s {\n\n", form_var, subform_var);

	/*  
	 * declare variables for selections on guide pane 
         */
	if (!(gen_selection_structs(file, selection)))
	    return(FALSE);


	/* declare variables for panes */
	if (subform->guidepane != NULL)
	    guide = subform->guidepane->guide_list;
	else guide = subform->guide;

	i = 0;
	while (guide != NULL)
	{
	    /* want nice spacing */
	    varsize  = kstrlen(subform_var) + kstrlen(pn_variables[i]) + 2;
	    spacenum = kmax(varsize, 11);

	    /* 
	     *  if they had selections on pane that created a need for a
             *  creation of a Pane Info structure for that pane, then need
             *  a pointer to the Pane Info structure and a xxx_selected flag
             *  declared from its Subform Info structure.
	     */
            if ((guide->pane != NULL) &&
		(guide->pane->sel_list != NULL) &&
		(kgen_valid_selections(guide->pane->sel_list)))
	    {
	        ksprintf(temp, "%s_%s ", subform_var, pn_variables[i]);
	        for (j = 0; j < (spacenum-kstrlen(subform_var)-kstrlen(pn_variables[i])-1); j++)
		    temp[j+varsize] = ' ';
	        ksprintf(&(temp[j+varsize]), "*%s;", pn_variables[i]);
	        kfprintf(file, "%s", temp);
 	        kfprintf(file, "          /* ptr to Pane Info struct for pane '%s' */\n",
		     pn_variables[i]);
	
	        /* boolean TRUE if this subform selected */ 
	        ksprintf(temp, "int ");
	        for (j = 0; j < (spacenum-4); j++)
	    	    temp[j+4] = ' ';
	        ksprintf(&(temp[j+4]), "%s_selected;", pn_variables[i]);
	        kfprintf(file, "%s", temp);
 	        kfprintf(file, "          /* TRUE if input was from pane '%s' */\n",
	    	         pn_variables[i]);
	    }

	    /* 
	     *  whether or not a PaneInfo structure was generated for the
	     *  pane, need a generic kformstruct associated with the pane
             *  for the application to send to xvf_set_attributes() if desired
             */ 
	    ksprintf(temp, "kform_struct ");
	    for (j = 0; j < (spacenum-13); j++)
		temp[j+13] = ' ';
	    ksprintf(&(temp[j+13]), "*%s_struct;", pn_variables[i]);
	    kfprintf(file, "%s", temp);
 	    kfprintf(file, "          /* pass to xvf_set_attributes() */\n\n");
	    guide = guide->next;
	    i++;
        }

	if (guide_quit)
        {
	    kfprintf(file, "int        quit;\n");
            kfprintf(file, "kform_struct *quit_struct;\n");
        }

	kfprintf(file, "}  %s_%s;\n\n", form_var, subform_var);

	return(TRUE);
}



/*--------------------------------------------------------------
|
|  Routine Name: gen_selection_structs
|
|       Purpose: Generates the structures needed for each selection
|
|         Input: file      - open stream to "form_init.h" file
|                selection - pointer to header of selection list
|
|	 Output: Returns TRUE if structures needed for selections 
|                were successfully created in form_info.h, FALSE 
|                otherwise.
|          Date: May 09, 1992
|    Written By: Danielle Argiro
| Modifications:
|
-------------------------------------------------------------*/

static int gen_selection_structs(
   kfile      *file,
   kselection *selection)
{
	char *variable, *title;
	int  live, optional, doubleclick;

        while (selection != NULL)
	{
	    switch (selection->type) {

	      case KUIS_INPUTFILE:
		    kvf_get_attributes(selection->back_kformstruct,
				       KVF_LIVE,     &live,
				       KVF_OPTIONAL, &optional,
				       NULL);
		    variable = ktoken_to_string(selection->var_token);
		    kfprintf(file, "char         *%s;         ", variable);
		    kfprintf(file, "/* filename for '%s'*/\n", variable);
		    kfprintf(file, "kform_struct *%s_struct;  ", variable);
		    kfprintf(file, "/* pass to xvf_set_attributes() */\n");
		    if (live)
		    {
			kfprintf(file, "int          %s_selected; ", variable);
		        kfprintf(file, "/* TRUE on <cr> for '%s' */\n",
				 variable);
		    }
		    if (optional)
		    {
			kfprintf(file, "int          %s_optsel;   ", variable);
			kfprintf(file, "/* TRUE if optional box of '%s' is ON */\n",
				      variable);
		    }
		    kfprintf(file, "\n");
		    break;

	       case KUIS_STDIN:
		    variable = ktoken_to_string(selection->var_token);
		    kfprintf(file, "kform_struct *%s_struct; ", variable);
		    kfprintf(file, "/* pass to xvf_set_attributes() */\n\n");
		    break;

	       case KUIS_OUTPUTFILE:
		    variable = ktoken_to_string(selection->var_token);
		    kvf_get_attributes(selection->back_kformstruct,
				       KVF_LIVE,     &live,
				       KVF_OPTIONAL, &optional,
				       NULL);
		    kfprintf(file, "char         *%s;         ", variable);
		    kfprintf(file, "/* filename for '%s'*/\n", variable);
		    kfprintf(file, "kform_struct *%s_struct;  ", variable);
		    kfprintf(file, "/* pass to xvf_set_attributes() */\n");
		    if (live)
		    {
			kfprintf(file, "int          %s_selected; ", variable);
		        kfprintf(file, "/* TRUE on <cr> for '%s' */\n",
				 variable);
		    }
		    if (optional)
		    {
			kfprintf(file, "int          %s_optsel;   ", variable);
			kfprintf(file, "/* TRUE if optional box of '%s' is ON */\n",
				      variable);
		    }
		    kfprintf(file, "\n");
		    break;

	       case KUIS_STDOUT:
		    variable = ktoken_to_string(selection->var_token);
		    kfprintf(file, "kform_struct *%s_struct; ", variable);
		    kfprintf(file, "/* pass to xvf_set_attributes() */\n\n");
		    break;

	       case KUIS_STRING:
	       case KUIS_STRINGLIST:
		    variable = ktoken_to_string(selection->var_token);
		    kvf_get_attributes(selection->back_kformstruct,
				       KVF_LIVE,     &live,
				       KVF_OPTIONAL, &optional,
				       NULL);
		    kfprintf(file, "char         *%s;         ", variable);
		    kfprintf(file, "/* string for '%s'*/\n", variable);
		    kfprintf(file, "kform_struct *%s_struct;  ", variable);
		    kfprintf(file, "/* pass to xvf_set_attributes() */\n");
		    if (live)
		    {
			kfprintf(file, "int          %s_selected; ", variable);
		        kfprintf(file, "/* TRUE on <cr> for '%s' */\n",
				 variable);
		    }
		    if (optional)
		    {
			kfprintf(file, "int          %s_optsel;   ", variable);
			kfprintf(file, "/* TRUE if optional box of '%s' is ON */\n",
				      variable);
		    }
		    kfprintf(file, "\n");
		    break;

	       case KUIS_INTEGER:
		    variable = ktoken_to_string(selection->var_token);
		    kvf_get_attributes(selection->back_kformstruct,
				       KVF_LIVE,     &live,
				       KVF_OPTIONAL, &optional,
				       NULL);
		    kfprintf(file, "int          %s;          ", variable);
		    kfprintf(file, "/* int for '%s'*/\n", variable);
		    kfprintf(file, "kform_struct *%s_struct;  ", variable);
		    kfprintf(file, "/* pass to xvf_set_attributes() */\n");
		    if (live)
		    {
			kfprintf(file, "int          %s_selected; ", variable);
		        kfprintf(file, "/* TRUE on <cr> for '%s' */\n",
				 variable);
		    }
		    if (optional)
		    {
			kfprintf(file, "int          %s_optsel;   ", variable);
			kfprintf(file, "/* TRUE if optional box of '%s' is ON */\n",
				      variable);
		    }
		    kfprintf(file, "\n");
		    break;

	       case KUIS_TOGGLE:
		    variable = ktoken_to_string(selection->var_token);
		    kvf_get_attributes(selection->back_kformstruct,
				       KVF_LIVE,     &live,
				       KVF_OPTIONAL, &optional,
				       NULL);
		    if ((selection->toggle_next->type == KUIS_INTEGER) ||
			(selection->toggle_next->type == KUIS_LOGICAL)   ||
			(selection->toggle_next->type == KUIS_FLAG))
		    {
		        kfprintf(file, "int          %s_val;      ", variable);
		    }
		    else if (selection->toggle_next->type == KUIS_FLOAT)
		    {
		        kfprintf(file, "float        %s_val;      ", variable);
		    }
		    else if (selection->toggle_next->type == KUIS_DOUBLE)
		    {
		        kfprintf(file, "double       %s_val;      ", variable);
		    }
		    else if ((selection->toggle_next->type == KUIS_STRING) ||
			     (selection->toggle_next->type == KUIS_INPUTFILE) ||
			     (selection->toggle_next->type == KUIS_OUTPUTFILE))
		    {
		        kfprintf(file, "char        *%s_val;      ", variable);

		    }
		    else
		    {
			errno = KINTERNAL;
			kerror(KCODEGEN, "gen_selection_structs",
			       "Toggle member of unsupported type");
		        return(FALSE);
		    }
		    kfprintf(file, "/* value of toggle '%s'*/\n", variable);
		    kfprintf(file, "int          %s_num;      ", variable);
		    kfprintf(file, "/* number of toggle member selected (1-N) */\n");
		    kfprintf(file, "kform_struct *%s_struct;  ", variable);
		    kfprintf(file, "/* pass to xvf_set_attributes() */\n");
		    if (live)
		    {
			kfprintf(file, "int          %s_selected; ", variable);
		        kfprintf(file, "/* TRUE on <cr> for '%s' */\n",
				 variable);
		    }
		    if (optional)
		    {
			kfprintf(file, "int          %s_optsel;   ", variable);
			kfprintf(file, "/* TRUE if optional box of '%s' is ON */\n",
				      variable);
		    }
		    kfprintf(file, "\n");
		    break;

	       case KUIS_LOGICAL:
		    variable = ktoken_to_string(selection->var_token);
		    kvf_get_attributes(selection->back_kformstruct,
				       KVF_LIVE,     &live,
				       KVF_OPTIONAL, &optional,
				       NULL);
		    kfprintf(file, "int          %s;          ", variable);
		    kfprintf(file, "/* boolean value of '%s'*/\n", variable);
		    kfprintf(file, "char         *%s_label;   ", variable);
		    kfprintf(file, "/* label assoc w/ value of '%s'*/\n", variable);
		    kfprintf(file, "kform_struct *%s_struct;  ", variable);
		    kfprintf(file, "/* pass to xvf_set_attributes() */\n");
		    if (live)
		    {
			kfprintf(file, "int          %s_selected; ", variable);
		        kfprintf(file, "/* TRUE on <cr> for '%s' */\n",
				 variable);
		    }
		    if (optional)
		    {
			kfprintf(file, "int          %s_optsel;   ", variable);
			kfprintf(file, "/* TRUE if optional box of '%s' is ON */\n",
				      variable);
		    }
		    kfprintf(file, "\n");
		    break;
	      
	       case KUIS_FLAG:
		    variable = ktoken_to_string(selection->var_token);
		    kvf_get_attribute(selection->back_kformstruct,
				      KVF_LIVE, &live);
		    kfprintf(file, "int           %s;         ", variable);
		    kfprintf(file, "/* TRUE if the box for '%s' is ON */\n", variable);
		    kfprintf(file, "kform_struct *%s_struct;  ", variable);
		    kfprintf(file, "/* pass to xvf_set_attributes() */\n");
		    if (live)
		    {
		        kfprintf(file, "int          %s_selected; ", variable);
		        kfprintf(file, "/* TRUE for value change of '%s' */\n\n",
		  	         variable);
		    }
		    kfprintf(file, "\n");
		    break;

	       case KUIS_CYCLE:
		    variable = ktoken_to_string(selection->var_token);
		    kvf_get_attributes(selection->back_kformstruct,
				       KVF_LIVE,     &live,
				       KVF_OPTIONAL, &optional,
				       NULL);
		    kfprintf(file, "int          %s;          ", variable);
		    kfprintf(file, "/* int value of '%s'*/\n", variable);
		    kfprintf(file, "char         *%s_label;   ", variable);
		    kfprintf(file, "/* label representing int value of '%s'*/\n",
			     variable);
		    kfprintf(file, "kform_struct *%s_struct;  ", variable);
		    kfprintf(file, "/* pass to xvf_set_attributes() */\n");
		    if (live)
		    {
			kfprintf(file, "int          %s_selected; ", variable);
		        kfprintf(file, "/* TRUE on <cr> for '%s' */\n",
				 variable);
		    }
		    if (optional)
		    {
			kfprintf(file, "int          %s_optsel;   ", variable);
			kfprintf(file, "/* TRUE if optional box of '%s' is ON */\n",
				      variable);
		    }
		    kfprintf(file, "\n");
		    break;
	      
	       case KUIS_LIST:
	       case KUIS_DISPLAYLIST:
		    variable = ktoken_to_string(selection->var_token);
		    kvf_get_attributes(selection->back_kformstruct,
				       KVF_LIVE,     &live,
				       KVF_OPTIONAL, &optional,
				       NULL);
		    if (selection->type == KUIS_DISPLAYLIST)
			kvf_get_attribute(selection->back_kformstruct,
                                          KVF_LIST_DOUBLECLICK, &doubleclick);
		    else doubleclick = FALSE;

		    kfprintf(file, "int          %s;          ", variable);
		    kfprintf(file, "/* int value of '%s'*/\n", variable);
		    kfprintf(file, "char         *%s_label;   ", variable);
		    kfprintf(file, "/* label representing int value of '%s'*/\n",
			     variable);
		    kfprintf(file, "kform_struct *%s_struct;  ", variable);
		    kfprintf(file, "/* pass to xvf_set_attributes() */\n");
		    if (live)
		    {
			kfprintf(file, "int          %s_selected; ", variable);
		        kfprintf(file, "/* TRUE on <cr> for '%s' */\n",
				 variable);
		    }
		    if (optional)
		    {
			kfprintf(file, "int          %s_optsel;   ", variable);
			kfprintf(file, "/* TRUE if optional box of '%s' is ON */\n",
				      variable);
		    }
		    if (doubleclick)
		    {
			kfprintf(file, "int          %s_dblclick;   ", variable);
			kfprintf(file, "/* TRUE for double click on list */\n");
		    }
		    kfprintf(file, "\n");
		    break;
	      
	       case KUIS_FLOAT:
	       case KUIS_DOUBLE:
		    variable = ktoken_to_string(selection->var_token);
		    kvf_get_attributes(selection->back_kformstruct,
				       KVF_LIVE,     &live,
				       KVF_OPTIONAL, &optional,
				       NULL);
		    if (selection->type == KUIS_FLOAT)
		    { 
		       kfprintf(file, "float        %s;          ", variable);
		       kfprintf(file, "/* float value of '%s'*/\n", variable);
		    }
		    else
		    { 
		       kfprintf(file, "double       %s;          ", variable);
		       kfprintf(file, "/* double value of '%s'*/\n", variable);
		    }
		    kfprintf(file, "kform_struct *%s_struct;  ", variable);
		    kfprintf(file, "/* pass to xvf_set_attributes() */\n");
		    if (live)
		    {
			kfprintf(file, "int          %s_selected; ", variable);
		        kfprintf(file, "/* TRUE on <cr> for '%s' */\n",
				 variable);
		    }
		    if (optional)
		    {
			kfprintf(file, "int          %s_optsel;   ", variable);
			kfprintf(file, "/* TRUE if optional box of '%s' is ON */\n",
				      variable);
		    }
		    kfprintf(file, "\n");
		    break;
	       
	       case KUIS_ROUTINE:
		    kfprintf(file, "int          routine;          ");
		    kfprintf(file, "/* TRUE when user clicks on routine button */\n");
		    kfprintf(file, "kform_struct *routine_struct;  ");
		    kfprintf(file, "/* pass to xvf_set_attributes() */\n\n");
		    break;

	       case KUIS_MASTERACTION:
		    variable = ktoken_to_string(selection->var_token);
		    kfprintf(file, "int          %s;          ", variable);
		    kfprintf(file, "/* TRUE when user clicks on '%s'*/\n", variable);
		    kfprintf(file, "kform_struct *%s_struct;  ", variable);
		    kfprintf(file, "/* pass to xvf_set_attributes() */\n\n");
		    break;

	       case KUIS_SUBFORMACTION:
		    variable = ktoken_to_string(selection->var_token);
		    kfprintf(file, "int          %s;          ", variable);
		    kfprintf(file, "/* TRUE when user clicks on '%s'*/\n", variable);
		    kfprintf(file, "kform_struct *%s_struct;  ", variable);
		    kfprintf(file, "/* pass to xvf_set_attributes() */\n\n");
		    break;

	       case KUIS_PANEACTION:
		    variable = ktoken_to_string(selection->var_token);
		    kfprintf(file, "int          %s;          ", variable);
		    kfprintf(file, "/* TRUE when user clicks on '%s'*/\n", variable);
		    kfprintf(file, "kform_struct *%s_struct;  ", variable);
		    kfprintf(file, "/* pass to xvf_set_attributes() */\n\n");
		    break;

	       case KUIS_WORKSPACE:
		    variable = ktoken_to_string(selection->var_token);
		    kfprintf(file, "xvobject     %s;          ", variable);
		    kfprintf(file, "/* address of workspace '%s'*/\n",
			     variable);
		    kfprintf(file, "kform_struct *%s_struct;  ", variable);
		    kfprintf(file, "/* pass to xvf_set_attributes() */\n\n");
		    break;

	       case KUIS_QUIT:
		    variable = ktoken_to_string(selection->var_token);
		    kvf_get_attribute(selection->back_kformstruct,
				      KVF_TITLE, &title);
		    kfprintf(file, "int          quit;         ");
		    kfprintf(file, "/* TRUE when user clicks on '%s' button */\n",
			     title);
		    kfprintf(file, "kform_struct *quit_struct; ");
		    kfprintf(file, "/* pass to xvf_set_attributes() */\n\n");
		    break;

	       case KUIS_BLANK:
		    variable = ktoken_to_string(selection->var_token);
		    kfprintf(file, "kform_struct *%s_struct; ", variable);
		    kfprintf(file, "/* pass to xvf_set_attributes() */\n\n");
		    break;

	       case KUIS_HELP:
		    variable = ktoken_to_string(selection->var_token);
		    kfprintf(file, "kform_struct *%s_struct; ", variable);
		    kfprintf(file, "/* pass to xvf_set_attributes() */\n\n");
		    break;

	       case KUIS_STARTSUBMENU:
		    variable = ktoken_to_string(selection->var_token);
                    kfprintf(file, "kform_struct *%s_struct; ", variable);
                    kfprintf(file, "/* submenu button, pass to xvf_set_attributes() */\n\n");
                    break;

	       case KUIS_MUTEXCL:
	       case KUIS_MUTINCL:
	       case KUIS_GROUP:
		    if (!(gen_selection_structs(file, 
					    selection->group_next)))
			return(FALSE);
		    break;

	       default:
		   kerror(KCODEGEN, "gen_selection_structs", 
		          "Unexpected line '%s';\n ignoring line\n", 
			  selection->line);
		   break;

	   }  /* end switch */

	   selection = selection->next;

	}  /* end while */ 

	return(TRUE);

}  /* end gen_selection_structs */

/*--------------------------------------------------------------
|
|  Routine Name: gen_driver_decls
|
|       Purpose: Generates declarations for each of the GUI subform
|                drivers and pane drivers at the end of the form_info.h file.
|
|         Input: file         - open stream to "form_init.h" file
|                kform     - pointer to the form tree
|                form_var     - variable name on [-F] line
|                program_name - name of program being generated
|
|	 Output: none
|          Date: Jan 20, 1993
|    Written By: Danielle Argiro
| Modifications:
|
-------------------------------------------------------------*/

static void gen_driver_decls(
    kfile *file,
    kform *form,
    char  *form_var,
    char  *program_name)
{
	ksubform  *subform;
	kguide    *guide;
	char      *subform_var, *pane_var;


	if (form->master != NULL)
	{
	    subform = form->master->subform_list;
	    gen_doroutine_decls(file, form->master->sel_list,
				"g""ui", "info_struct"); 
	}
	else subform = form->subform;


	kfprintf(file, "void _xvf_free_%s PROTO((gui_info_struct *));\n", 
		 form_var);
	kfprintf(file, "void _xvf_get_%s  PROTO((gui_info_struct *));\n\n", 
		 form_var);
	kfprintf(file, "int  _xvf_init_%s PROTO((kform *, gui_info_struct *));\n", 
		 form_var);
	kfprintf(file, "void run_%s       PROTO((kform *, ksubform *, kaddr));\n\n", 
		 form_var);

	while (subform != NULL)
	{
	    subform_var = ktoken_to_string(subform->var_token);
	    kfprintf(file, "void run_%s      PROTO((%s_%s *));\n", 
		     subform_var, form_var, subform_var);
	    kfprintf(file, "void _xvf_get_%s PROTO((%s_%s *));\n", 
		     subform_var, form_var, subform_var);

	    if (subform->guidepane != NULL)
	    {
		guide = subform->guidepane->guide_list;
		gen_doroutine_decls(file, subform->guidepane->sel_list,
				    form_var, subform_var); 
	    }
	    else guide = subform->guide;

	    kfprintf(file, "\n");

	    while (guide != NULL)
	    {
		if ((guide->pane != NULL) && 
                    (guide->pane->sel_list != NULL) &&
		    (kgen_valid_selections(guide->pane->sel_list)))
		{
	            pane_var = ktoken_to_string(guide->pane->var_token);
	            kfprintf(file, "void run_%s      PROTO((%s_%s *));\n", 
		             pane_var, subform_var, pane_var);
	            kfprintf(file, "void _xvf_get_%s PROTO((%s_%s *));\n", 
		             pane_var, subform_var, pane_var);

		    gen_doroutine_decls(file, guide->pane->sel_list,
					subform_var, pane_var); 
		    kfprintf(file, "\n");
		}
		guide  = guide->next;
	    }
	    subform = subform->next;
	}
	kfprintf(file, "\n\n\n");
}

/*--------------------------------------------------------------
|
|  Routine Name: gen_doroutine_decls
|
|       Purpose: Generates declarations for each of the do routines
|                that would get generated for a particular selection list.
|
|         Input: file         - open stream to "form_init.h" file
|                sel_list     - header of selection list
|                control      - variable name of pane, subform, or form.
|                back_var     - variable name on succeeding control:
|                               subform, form, or program name, respectively.
|	 Output: none
|          Date: Jan 20, 1993
|    Written By: Danielle Argiro
| Modifications:
|
-------------------------------------------------------------*/
static void gen_doroutine_decls(
    kfile      *file,
    kselection *sel_list,
    char       *back_var, 
    char       *control)
{
	kselection *selection;
	char       *first_part;
	char       *variable;
	int        live, doubleclick;

	if (sel_list == NULL) return;


	selection = sel_list;
	while (selection != NULL)
	{
	    if (selection->back_control->type == KMASTER)
		first_part = ktoken_to_string(selection->back_form->var_token);
	    else first_part = control;

	    switch (selection->type) {

		case KUIS_MASTERACTION:
		case KUIS_SUBFORMACTION:
		case KUIS_PANEACTION:
		     variable = ktoken_to_string(selection->var_token);
	             kfprintf(file, "void %s_%s	PROTO((%s_%s *));\n",
		              first_part, variable, back_var, control);
		     break;

		case KUIS_INPUTFILE:
		case KUIS_OUTPUTFILE:
		case KUIS_INTEGER:
		case KUIS_FLOAT:
		case KUIS_DOUBLE:
		case KUIS_STRING:
		case KUIS_STRINGLIST:
		case KUIS_LOGICAL:
		case KUIS_FLAG:
		case KUIS_CYCLE:
		case KUIS_LIST:
		case KUIS_DISPLAYLIST:
		case KUIS_TOGGLE:
		     variable = ktoken_to_string(selection->var_token);
		     kvf_get_attribute(selection->back_kformstruct,
				       KVF_LIVE, &live);
		     if (live)
	                 kfprintf(file, "void %s_%s	PROTO((%s_%s *));\n",
		                  first_part, variable, back_var, control);

		     if ((live) && (selection->type == KUIS_DISPLAYLIST))
		     {
			kvf_get_attribute(selection->back_kformstruct,
                                          KVF_LIST_DOUBLECLICK, &doubleclick);
		        if (doubleclick)
			    kfprintf(file, "void %s_%s_dblclick     PROTO((%s_%s *));\n",
                                 first_part, variable, back_var, control);
		     }
		     break;

		case KUIS_MUTINCL:
                case KUIS_MUTEXCL:
		     gen_doroutine_decls(file, selection->group_next,
    					 back_var, control);
		     break;

		default:
		     break;
	    }
	    selection = selection->next;
	}
}
