/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>              Routines to Generate *.1 file            <<<<
   >>>>                                                       <<<<
   >>>>  Private:                                             <<<<
   >>>>                kgen_clui_generate_man1file()          <<<<
   >>>>   Static:                                             <<<<
   >>>>                print_man1_req_args()                  <<<<
   >>>>                print_man1_opt_args()                  <<<<
   >>>>                print_man1_req_group_args()            <<<<
   >>>>                print_man1_opt_group_args()            <<<<
   >>>>   Public:                                             <<<<
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"

static void print_man1_req_args       PROTO((kfile *));
static void print_man1_opt_args       PROTO((kfile *));
static void print_man1_req_group_args PROTO((kfile *, kselection *, int));
static void print_man1_opt_group_args PROTO((kfile *, kselection *, int));
static void print_man1_single_arg     PROTO((kfile *, kselection *));



/*---------------------------------------------------------------
|
|  Routine Name: kgen_clui_generate_man1file
|
|       Purpose: Creates the man1 file 
|
|         Input: program   -  program object
|                prog_spec -  pointer to the internal PS struct
|                opath     -  path to program object
|                tbname   -  name of toolbox
|		 oname     -  name of program object being generated
|		 bname     -  name of the program binary being generated
|
|        Output: Returns TRUE on success, FALSE on failure
|    Written By: Danielle Argiro
|          Date: Jul 23, 1992 
| Modifications: 
|
--------------------------------------------------------------*/

int  kgen_clui_generate_man1file(
   kobject   program,
   char    **prog_spec,
   char     *opath,
   char     *tbname,
   char     *oname,
   char     *bname)
{
        kstring   routine = "kgen_clui_generate_man1file()";
        kobject   toolbox;
	int       object_type;
	time_t    clck;
	struct tm *tmval;
	kfile     *file;
        char      *filepath, *copyright; 
	kobject   file_object = NULL;
	char      temp[KLENGTH], date[KLENGTH], directory[KLENGTH];
	kstring   short_description = NULL;
	int       force_flag;
	int       isnewfile         = FALSE;


	/*
	 * get file object representing man1 page
	 */
	if (!kcms_get_attributes(program,
				 KCMS_PARENT,           &toolbox,
				 KCMS_CMOBJ_GEN_MAN1,   &file_object,
				 KCMS_CMOBJ_SHORT_DESC, &short_description,
				 KCMS_CMOBJ_TYPE,       &object_type,
				 KCMS_END)
	    || !kcms_query_bit(program, KCMS_CMOBJ_FLAGS, KCMS_BIT_CMOBJ_FORCE,
			       &force_flag))
	    return(FALSE);

	/* 
	 * 1st time man1 page has been created - create file object, so that 
         * kcms registers the man1 page in the kcms database
         */
	if (file_object == NULL)
        {
	    ksprintf(temp, "%s/man/%s.1", opath, oname);
            file_object = kcms_create_fileobj(program, temp, NULL,
                                               KCMS_FOBJ_TYPE_MAN,
                                               KCMS_FOBJ_SUBTYPE_MAN1,
                                               KCMS_FOBJ_GEN_MAN1,
                                               KCMS_FOBJ_ACCESS_RDWR);
            if (file_object == NULL)
                return(FALSE);
            isnewfile = TRUE;
            filepath = kstrdup(temp);
	}

	/*
	 * the man1 file has already been created & added to kcms database;
         * simply get the path to the man1 page.
	 */
	else if (!kcms_get_attribute(file_object, KCMS_PATH, &filepath))
            return(FALSE);

	/*
	 *  Add the manpage to the toolbox manpage database
	 */
	if (filepath != NULL)
	{
	   kdbm   *dbm;
	   kdatum key, data;
	
	   ksprintf(temp, "$%s/repos/db/manpage", tbname);
	   if ((dbm = kdbm_open(temp, O_WRONLY | O_CREAT, 0666)) != NULL)
	   {
	      key.dptr   = oname;
  	      key.dsize  = kstrlen(key.dptr) + 1;
	      data.dptr  = filepath;
	      data.dsize = kstrlen(data.dptr) + 1;
	      kdbm_store(dbm, key, data, KDBM_REPLACE);
	      kdbm_close(dbm);
	   }
	}

	/*
	 *  script object kludge: only want to generate a man page initially.
         *  never want to re-gen a man page, as we can't parse the info for
         *  a man page for a script object, and don't want to over-write it.
	 */
	if ((file_object == NULL) && (object_type == KCMS_SCRIPT))
	    return(TRUE);

	/*
	 *  if somehow the database GHOST_MAN1 key got deleted from the kcms
         *  database, but the man1 file really is there, don't want to 
         *  clobber it!
	 */
        if ((isnewfile) && (kaccess(filepath, R_OK) == 0))
        {
            kerror(KCODEGEN, "kgen_clui_generate",
                   "WARNING!  CMS database appears to have been corrupted;  cms dbm key GHOST_MAN1 is missing, but the %s man1 file does exist.  Man1 file will NOT be updated. Adding GHOST_MAN1 cms key to database; next run of ghostwriter should be OK.", oname);
	    return(FALSE);
        }

        /*
         * get the directory to the help file.  if the directory does
         * not exist, it will get created by kmake_dir();  if kmake_dir()
         * cannot create it, it will return the ENOENT error.
         */
        (void) kdirname(filepath, directory);
        if (!kmake_dir(directory, 0777) && errno != ENOENT)
        {
             kerror(KCODEGEN, "kgen_clui_generate_man1file",
                    "Can't create directory %s in which to deposit man1 file",
                    directory);
             return(FALSE);
        }

	/*
	 *  see if man1 file already exists.  If so, prompt to over-write,
	 *  return TRUE if the answer is NO
	 */
	if (!force_flag &&
	    koverwrite(kget_notify(), filepath) == FALSE)
         return(TRUE);

	/*
	 *  create & open man1 file  
	 */
        if (kaccess(filepath, R_OK) != 0)
	   isnewfile = TRUE;
	if ((file = kfopen(filepath, "w"))== NULL)
	{
	     kerror(KCODEGEN, "kgen_clui_generate_man1file",
	            "Could not create file '%s'", filepath);
	     return(FALSE);
	}
	
	/*
	 * print out first line of man1 page with time & date
	 */
	clck = time(NULL);
	tmval = localtime(&clck);
	if (tmval != NULL)
	{
	     strftime( date, KLENGTH, "%h %d, %Y", tmval);
	}
	
	kfprintf(file, ".TH \"%s\" \"%s\" \"COMMANDS\" \"\" \"%s\"\n",
	         oname, kstring_upper(tbname, NULL), date);
	
	/* 
	 * print out name and short description
	 */
	kfprintf(file, ".SH PROGRAM\n");
	if (object_type == KCMS_LIBRARY)
	    kfprintf(file, "%s \\- ", oname);
	else kfprintf(file, "%s \\- ", bname);

	if (kstrlen(short_description) > 0)
	    kfprintf(file, "%s\n", short_description);
	else
	    kfprintf(file, "\n");

	/*
	 * the .syntax macro generates an interpretive syntax statement
	 */
	kfprintf(file, ".syntax %s %s\n", tbname, oname);
	
	/* 
	 * print out description
	 */
	kfprintf(file, ".SH DESCRIPTION\n");
	if (kstrlen(prog_spec[MAN1_LONGDESC]) > 0) 
	    kfprintf(file, "%s\n", prog_spec[MAN1_LONGDESC]);
	
	/* 
	 * print out required arguments
	 */
	kfprintf(file, ".SH \"REQUIRED ARGUMENTS\"\n");
	if (kgen_req_num > 0)
	    print_man1_req_args(file);
	else kfprintf(file, "none\n.sp\n");
	
	/* 
	 * print out optional arguments
	 */
	kfprintf(file, ".SH \"OPTIONAL ARGUMENTS\"\n");
	if (kgen_opt_num > 0)
	    print_man1_opt_args(file);
	else kfprintf(file, "none\n.sp\n");
	
	/* 
	 * print out examples
	 */
	kfprintf(file, ".SH EXAMPLES\n");
	if (kstrlen(prog_spec[MAN1_EXAMPLES]) > 0) 
	    kfprintf(file, "%s\n", prog_spec[MAN1_EXAMPLES]);
	
	/* 
	 * print out "see also"
	 */
	kfprintf(file, ".SH \"SEE ALSO\"\n");
	
	if (kstrlen(prog_spec[MAN1_SEEALSO]) > 0) 
	   kfprintf(file, "%s\n", prog_spec[MAN1_SEEALSO]);
	
	/* 
	 * print out restrictions
	 */
	kfprintf(file, ".SH RESTRICTIONS \n");
	if (kstrlen(prog_spec[MAN1_RESTRICTIONS]) > 0) 
	    kfprintf(file, "%s\n", prog_spec[MAN1_RESTRICTIONS]);
	
	/*
	 * print out references
	 */
	kfprintf(file, ".SH REFERENCES \n");
	if (kstrlen(prog_spec[MAN1_REFERENCES]) > 0) 
            kfprintf(file, "%s\n", prog_spec[MAN1_REFERENCES]);

	/*
	 * print the copyright 
	 */
	kfprintf (file, ".SH COPYRIGHT\n");
	copyright = kcms_get_copyright(toolbox, KCMS_TB_COPYRIGHT_SHORT,
				       KCMS_LANG_NONE);
	if (copyright != NULL) 
	    kfprintf(file, "%s\n", copyright);
	kfree(copyright);
	
	
	kgen_generate_whatis(program, bname, short_description);
	
	/*
         *  that's the end...
         */
	kfclose(file);
	kannounce(KCODEGEN, routine, "done generating %s man page", oname);
	kinfo(KHOSTILE, "your %s man page is *really* sad!", oname);
	return(TRUE);
}


/*-----------------------------------------------------------
|
|  Routine Name: print_man1_req_args
|
|       Purpose: Prints the required arguments to the man1 file 
|
|         Input: file     - open stream to man1 page
|
| 	 Output: none
|    Written By: Danielle Argiro
|          Date: Sept 2, 1993
| Modifications: Converted from Khoros 1.0 (DA)
|
--------------------------------------------------------------*/

static void print_man1_req_args(
   kfile *file)
{
	int        indx;
	kselection *selection;

	/* 
	 * print out required arguments 
	 */
	indx = 0;
	while (indx < kgen_req_num)
	{
	    selection = kgen_req_sels[indx];

	    if ((selection->type == KUIS_MUTEXCL) ||
                (selection->type == KUIS_MUTINCL) ||
                (selection->type == KUIS_GROUP)) 
		print_man1_req_group_args(file, selection, FALSE);
	    else print_man1_single_arg(file, selection);

	    indx++;
	}

        kfprintf(file, ".sp\n");
}

/*-----------------------------------------------------------
|
|  Routine Name: print_man1_opt_args
|
|       Purpose: Prints the optional arguments to the man1 file 
|
|         Input: file     - open stream to man1 page
|
|        Output: none
|    Written By: Danielle Argiro
|          Date: Sept 2, 1993
| Modifications: Converted from Khoros 1.0 (DA)
|
--------------------------------------------------------------*/

static void print_man1_opt_args(
   kfile *file)
{
	int        indx;
	kselection *selection;

	/* 
	 * print out optional arguments 
	 */
	indx = 0;
	while (indx < kgen_opt_num)
	{
	    selection = kgen_opt_sels[indx];

	    if ((selection->type == KUIS_MUTEXCL) ||
                (selection->type == KUIS_MUTINCL) ||
                (selection->type == KUIS_GROUP))
                print_man1_opt_group_args(file, selection, FALSE);
            else print_man1_single_arg(file, selection);

	    indx++;
	}
  	kfprintf(file, ".sp\n");

}

/*-----------------------------------------------------------
|
|  Routine Name: print_man1_req_group_args
|
|       Purpose: Prints the required ME/MI group arguments to the man1 file
|
|         Input: file        - open stream to man1 page
|                group_start - beginning of group list (-C or -B line)
|                nested      - TRUE if nested group
|        Output: none
|    Written By: Danielle Argiro
|          Date: Sept 2, 1993
| Modifications: 
|
--------------------------------------------------------------*/

static void print_man1_req_group_args(
    kfile      *file,
    kselection *group_start,
    int        nested)
{
	int        optional;
	kselection *groupmember;

	if (!nested)
	{
	    if (group_start->type == KUIS_MUTEXCL)
                kfprintf(file, 
		    "Mutually Exclusive Group; you must specify ONE of:\n");
            else if (group_start->type == KUIS_MUTINCL)
		kfprintf(file, 
                    "Mutually Inclusive Group; you must specify ALL of:\n");
            else kfprintf(file, 
                    "Group; you must specify AT LEAST ONE of:\n");
	}
	else
	{
	    if (group_start->type == KUIS_MUTEXCL)
                kfprintf(file, 
		    ".IP \"\tONE OF the Mutually Exclusive Group:\"\n");
            else if (group_start->type == KUIS_MUTINCL)
		kfprintf(file, 
		    ".IP \"\tALL OF the Mutually Inclusive Group:\"\n");
            else kfprintf(file, 
		    ".IP \"\tAT LEAST ONE OF the Group:\"\n");
	}

	groupmember = group_start->group_next;
        while (groupmember != NULL)
        {
	    if (groupmember->type == KUIS_BLANK)
	    {
		groupmember = groupmember->next;
		continue;
	    }
	    if (groupmember->group_next == NULL)
	    {
	        print_man1_single_arg(file, groupmember);

		if (groupmember->next != NULL) 
		{
		    if (!nested)
		    {
		    	if (group_start->type == KUIS_MUTEXCL)
		            kfprintf(file, ".IP OR\n");
		        else if (group_start->type == KUIS_MUTINCL)
		            kfprintf(file, ".IP AND\n");
			else kfprintf(file, ".IP AND/OR\n");
		    }
		    else
		    {
		    	if (group_start->type == KUIS_MUTEXCL)
		            kfprintf(file, ".IP \"\tOR\"\n");
		        else if (group_start->type == KUIS_MUTINCL)
		            kfprintf(file, ".IP \"\tAND\"\n");
		        else kfprintf(file, ".IP \"\tAND/OR\"\n");
		    }
		}
	    }
	    else
	    {
		kvf_get_attribute(group_start->back_kformstruct,
				  KVF_OPTIONAL, &optional);
		if (!optional)
		    print_man1_req_group_args(file, groupmember, TRUE);
		else print_man1_opt_group_args(file, groupmember, TRUE);
	    }

            groupmember = groupmember->next;
        }
}

/*-----------------------------------------------------------
|
|  Routine Name: print_man1_opt_group_args
|
|       Purpose: Prints the optional ME/MI group arguments to the man1 file
|
|         Input: file        - open stream to man1 page
|                group_start - beginning of group list (-C or -B line)
|                nested      - TRUE if nested group
|        Output: none
|    Written By: Danielle Argiro
|          Date: Sept 2, 1993
| Modifications: 
|
--------------------------------------------------------------*/

static void print_man1_opt_group_args(
    kfile      *file,
    kselection *group_start,
    int        nested)
{
	kselection *groupmember;
	char       *group_def_var = NULL, *group_def_val = NULL;
	int        optional, optsel;

	if (!nested)
	{
	    if (group_start->type == KUIS_MUTEXCL)
                kfprintf(file, 
		         "Mutually Exclusive Group; if desired, specify ONE of:\n");
            else if (group_start->type == KUIS_MUTINCL)
		kfprintf(file, 
                          "Mutually Inclusive Group; if desired, specify ALL of:\n");
	    else kfprintf(file,
                          "Group; specify AT LEAST ONE of: \n");

	}
	else
	{
	    if (group_start->type == KUIS_MUTEXCL)
                kfprintf(file, 
		         ".IP \"\tONE OF the Mutually Exclusive Group:\"\n");
            else if (group_start->type == KUIS_MUTINCL)
		kfprintf(file, 
		         ".IP \"\tALL OF the Mutually Inclusive Group:\"\n");
            else kfprintf(file, 
		         ".IP \"\tAT LEAST ONE OF the Group:\"\n");
	}

	kfprintf(file, ".br\n");

        groupmember = group_start->group_next;
        while (groupmember != NULL)
        {
	    if (groupmember->type == KUIS_BLANK)
	    {
		groupmember = groupmember->next;
		continue;
	    }
	    if (groupmember->group_next == NULL)
	    {
	        print_man1_single_arg(file, groupmember);
	        kvf_get_attribute(groupmember->back_kformstruct, 
				  KVF_OPTSEL, &optsel);
                if (optsel)
                {
                    if ((groupmember->type != KUIS_INPUTFILE) &&
                        (groupmember->type != KUIS_OUTPUTFILE))
                        group_def_val = kgen_ascii_def_val(groupmember);
                    else group_def_val = kstrdup(" ");
                }

		if (groupmember->next != NULL)
		{
		    if (!nested)
		    {
		        if (group_start->type == KUIS_MUTEXCL)
		            kfprintf(file, ".IP OR\n");
		        else if (group_start->type == KUIS_MUTINCL)
		            kfprintf(file, ".IP AND\n");
			else kfprintf(file, ".IP AND/OR\n");
 		    }
		    else
		    {
		    	if (group_start->type == KUIS_MUTEXCL)
		            kfprintf(file, ".IP \"\tOR\"\n");
		        else if (group_start->type == KUIS_MUTINCL)
		            kfprintf(file, ".IP \"\tAND\"\n");
			else kfprintf(file, ".IP \"\tAND/OR\"\n");
		    }
		}
	    }
	    else
	    {
	        kvf_get_attribute(group_start->back_kformstruct,
                                  KVF_OPTIONAL, &optional);
                if (!optional)
                    print_man1_req_group_args(file, groupmember, TRUE);
                else print_man1_opt_group_args(file, groupmember, TRUE);
	    }

            if ((group_start->type == KUIS_MUTEXCL) && 
                (group_def_var != NULL) &&
                (group_def_val != NULL))
            {
                kfprintf(file, ".sp\n");
                kfprintf(file, "Default argument of");
		kfprintf(file, " mutually exclusive group:\n");
                kfprintf(file, ".br\n");
                kfprintf(file, "-%s %s\n", group_def_var, group_def_val);
            }
            groupmember = groupmember->next;
        }
}


/*-----------------------------------------------------------
|
|  Routine Name: print_man1_single_arg
|
|       Purpose: Prints info on a single argument to the man1 file
|
|         Input: file      - open stream to man1 page
|                selection - selection corresponding to argument
|        Output: none
|    Written By: Danielle Argiro
|          Date: March 30, 1994
| Modifications:
|
--------------------------------------------------------------*/

static void print_man1_single_arg(
    kfile      *file,
    kselection *selection)
{
	char   *variable, *description;
	char   *ascii_default; 
	char   *typestring;
	int    optional, toggle_type;

	kvf_get_attributes(selection->back_kformstruct,
			   KVF_DESCRIPTION, &description,
			   KVF_OPTIONAL,    &optional,
			   NULL);
       	
	if (description == NULL) 
	    description = kstrdup("no description");
	variable = ktoken_to_string(selection->var_token);
	kfprintf(file, ".IP -%s 7\n", variable);
	
	if (selection->type == KUIS_TOGGLE)
        {
                kvf_get_attribute(selection->back_kformstruct,
                                  KVF_TOGGLE_TYPE, &toggle_type);
                if ((toggle_type == KUIS_LOGICAL) || (toggle_type == KUIS_FLAG))
                    typestring = kstrdup("integer");
                else typestring = kgen_ascii_datatype(toggle_type);
        }
        else typestring = kgen_ascii_datatype(selection->type);

	kfprintf(file, "type: %s\n.br\n", typestring);
	kfprintf(file, "desc: ");
	if (description != NULL)
	    kfprintf(file, "%s\n.br\n", description);
	else kfprintf(file, "\n.br\n");

	if ((selection->type != KUIS_FLAG) && (optional))
	{
	    ascii_default = kgen_ascii_def_val(selection);
	    kfprintf(file, "default: %s\n.br\n", ascii_default);
	    kfree(ascii_default);
	}
	
	if ((selection->type == KUIS_INTEGER) ||
	    (selection->type == KUIS_FLOAT)   ||
	    (selection->type == KUIS_DOUBLE))
            kgen_clui_print_bounds(file, selection, KGEN_CLUI_MAN1_SYNTAX);

	else if (selection->type == KUIS_TOGGLE)
        {
	    kfprintf(file, "allowed values: \n.br\n");
            kgen_clui_print_toggle_vals(file, selection,
                                        KGEN_CLUI_MAN1_SYNTAX, FALSE);
            kfprintf(file, "\n.br\n");
        }
}
