/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Routines for creation of toolbox manual
   >>>>
   >>>>  Private:
   >>>>
   >>>>   Static:
   >>>>		kcms_create_chapter
   >>>>
   >>>>   Public:
   >>>>         kcms_create_manual
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"
#include <kcms/private.h>

static int kcms_create_chapter PROTO((kobject, kstring, kstring, kstring, int));


/************************************************************
* Routine Name:	kcms_create_manual - create a manual in given toolbox
*
* Purpose:	This routine is used to generate a template manual in
*		a toolbox.
*		The manual is created in the directory $TOOLBOX/manual.
*
*		The manual created contains three initial chapters:
*		introduction, glossary, and index.
*		Each chapter is in a separate subdirectory of the manual
*		directory.
*
* Input:	toolbox     - the toolbox to create the manual for.
*		manual_name - The name of the manual to create.
*			      The first letter of the name will be converted
*			      to upper-case,
*			      with the remained of the manual name forced
*			      to be lower-case.
*
* Returns:	TRUE (1) if the manual was successfully created,
*		FALSE (0) otherwise.
*
* Written By:	Neil Bowers
* Date:		19-aug-94
*************************************************************/
int
kcms_create_manual(
   kobject  toolbox,
   kstring  manual_name)
{
   kstring  routine           = "kcms_create_manual()";
   char     man_name[KLENGTH];
   char     dirpath[KLENGTH];
   kstring  tbpath;
   kstring  tbname;


   if (!kcms_get_attributes(toolbox,
			    KCMS_PATH,  &tbpath,
			    KCMS_NAME,  &tbname,
			    KCMS_END))
      return FALSE;

   if (kstring_lower(manual_name, man_name) == NULL)
      return FALSE;

   man_name[0] = toupper(man_name[0]);

   /*-- create the $TOOLBOX/manual directory if it doesn't exist ------*/
   ksprintf(dirpath, "%s/manual", tbpath);
   if (kaccess(dirpath, F_OK) == -1
       && kmkdir(dirpath, 0777) == -1 && errno != EEXIST)
   {
      kerror(KCMS, routine, "Unable to create manual directory:\n\t%s\n",
	     dirpath);
      return FALSE;
   }

   ksprintf(dirpath, "%s/manual/%s", tbpath, man_name);
   if (kaccess(dirpath, F_OK) == 0)
   {
      kerror(KCMS, routine, "The manual directory for toolbox `%s' "
	     "already contains an entry `%s'.\n", tbname, man_name);
      return FALSE;
   }

   kannounce(KCMS, routine, "Creating manual $%s/manual/%s",
	     tbname, manual_name);

   /*-- create the toolbox manual directory ---------------------------*/
   if (kmkdir(dirpath, 0777) == -1 && errno != EEXIST)
   {
      kerror(KCMS, routine, "Unable to create manual directory:\n\t%s\n",
	     dirpath);
      return FALSE;
   }

   /*-- create the default chapters -----------------------------------*/
   /*-- then we can create Imakefile & Makefile in manual directory ---*/

   return kcms_create_chapter(toolbox, man_name, "1.introduction", "manual",
			      KCMS_TEMPLATE_MANUAL)
       && kcms_create_chapter(toolbox, man_name, "index", "mindex",
			      KCMS_TEMPLATE_INDEX)
       && kcms_create_chapter(toolbox, man_name, "glossary", "mglossary",
			      KCMS_TEMPLATE_GLOSSARY)
       && kcms_dir_imakeandmake(dirpath, tbname, 1);
}

/*-----------------------------------------------------------
| Routine Name:	kcms_create_chapter - create a chapter for a toolbox manual
| Purpose:	This function creates a chapter in an existing manual,
|		for a specific toolbox.
|
| Input:	toolbox      - The toolbox which contains the manual.
|		manual_name  - The name of the manual to create the chapter.
|		chapter_name - The name of the chapter to create.
|		chapter_type - A string which gives the type of chapter to
|			       create.  This can be "manual" (for a regular
|			       manual chapter), "index", or "glossary".
|		template_id  - The identifier for the template to copy
|			       into the new chapter.
|
| Returns:	TRUE (1) if the path does correspond to an object,
|		FALSE (0) otherwise.
| Written By:	Neil Bowers
| Date:		20-aug-94
------------------------------------------------------------*/
static int
kcms_create_chapter(
   kobject  toolbox,
   kstring  manual_name,
   kstring  chapter_name,
   kstring  chapter_type,
   int      template_id)
{
   kstring  routine = "kcms_create_chapter()";
   char     chapter[KLENGTH];
   char     command[KLENGTH];
   char     fullpath[KLENGTH];
   char     path[KLENGTH];
   char     file_path[KLENGTH];
   kstring  tbpath;
   kstring  tbname;
   kstring  template_path;


   if (!kcms_get_attributes(toolbox,
			    KCMS_PATH, &tbpath,
			    KCMS_NAME, &tbname,
			    KCMS_END))
      return FALSE;

   ksprintf(chapter, "%c.%s", manual_name[0], chapter_name);
   ksprintf(path, "%s/manual/%s/%s", tbpath, manual_name, chapter);
   if (kfullpath(path, NULL, fullpath) == NULL)
   {
      kerror(KCMS, routine, "Could determine explicit path to chapter `%s' "
	     "in manual `%s' of toolbox `%s'.\n"
	     "\tPath: %s\n",
	     chapter, manual_name, tbname, path);
      return FALSE;
   }

   /*-- create the directory -----------------------------------------*/
   if (kmkdir(path, 0777) == -1 && errno != EEXIST)
   {
      kerror(KCMS, routine,
             "Unable to create directory for chapter `%s' in manual `%s'.\n"
             "\tToolbox: %s\n"
             "\tPath:    %s\n",
	     chapter, manual_name, tbname, path);
      return FALSE;
   }

   /*-- keep the user informed what's going on -----------------------*/
   kannounce(KCMS, routine, "Creating chapter `%s' in manual `%s'.",
	     chapter, manual_name);

   /*-- copy template .man file to chapter directory -----------------*/
   template_path = kcms_query_template(toolbox, template_id);
   ksprintf(file_path, "%s/%s.man", path, chapter);
   if (!ksedfile(template_path, file_path, FALSE, KFILE_UPDATE, NULL,
		 "#TOOLBOX-NAME#", tbname,
		 "#manual-name#",  manual_name,
		 "#chapter-name#", chapter,
		 NULL))
      kwarn(KCMS, routine, "The template file could not be copied into "
	    "chapter `%s' of manual `%s' in toolbox `%s':\n\n"
	    "\tToolbox:     %s\n"
	    "\tManual:      %s\n"
	    "\tChapter:     %s\n"
	    "\tTemplate:    %s\n"
	    "\tDestination: %s\n"
	    "This operation will continue, but the chapter may well be "
	    "incomplete.",
	    chapter, manual_name, tbname, tbname, manual_name,
	    chapter, template_path, file_path);

   /*-- generate Imakefile and Makefile for chapter ------------------*/
   ksprintf(command,
	    "cd %s ; "
	    "kgenimake -tb %s -%s ; "
	    "kgenmake", fullpath, tbname, chapter_type);
   ksystem(command);

   return TRUE;
}
