
/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Image Display Utility Routines
   >>>>
   >>>>   Static:
   >>>>             _kghost_check_progstatus()
   >>>>             _kghost_check_five()
   >>>>  Private:
   >>>>             kghost_get_progspec()
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"
#include "cmobjP.h"

#define KSPFILE	6

#define KCMS_NO_EXTRA -1

#define BNAME   0
#define ONAME   1

struct file_struct
{
   char *file_desc;
   int file_ident;
   int fo_ident;
};

static struct file_struct ftbl[KSPFILE] =
{
   {"main program", KCMS_CMOBJ_GEN_MAIN,  KCMS_FOBJ_GEN_MAIN},
   {"include file", KCMS_CMOBJ_GEN_INCL,  KCMS_FOBJ_GEN_INCL},
   {"library file", KCMS_CMOBJ_GEN_LFILE, KCMS_FOBJ_GEN_LFILE},
   {"man1 file",    KCMS_CMOBJ_GEN_MAN1,  KCMS_FOBJ_GEN_MAN1},
   {"man3 file",    KCMS_CMOBJ_GEN_MAN3,  KCMS_FOBJ_GEN_MAN3},
   {"help file",    KCMS_CMOBJ_GEN_HELP,  KCMS_FOBJ_GEN_HELP}
};

struct key_struct
{
   char *begin;
   char *end;
   int   begextra;
};

struct prog_struct
{
   int   prog_id;			/* defined in ghost.h */
   struct key_struct key[KSPFILE];	/* begin and end keys for all file
					   files */
   char *block_type;		/* string indicating section for errors */
   int   checkerr;		/* TRUE = call _kghost_missing_key_warning */
   int   strip;			/* TRUE = call kstrip_header or _strip_h */
   int   cleanup;		/* TRUE = call kstring_cleanup */
   int   oneline;		/* TRUE = replace '\n' with ' ' */
};

static struct prog_struct prog_info[] =
{
   {AUTHORS,
    {
       {
	  "^\\|\\s*Written By:",
	  "^\\|\\s*Date:",
	  KCMS_NO_EXTRA
       },

       {
	  "^\\s*>>>>\\s*Written By:",
	  "^\\s*>>>>\\s*Modifications:",
	  KCMS_NO_EXTRA
       },

       {
	  "^\\*\\s*Written By:",
	  "^\\*\\s*Date:",
	  KCMS_NO_EXTRA
       },

       {NULL, NULL, KCMS_NO_EXTRA},

       {NULL, NULL, KCMS_NO_EXTRA},

       {NULL, NULL, KCMS_NO_EXTRA}
    },
    "'authors' section", TRUE, TRUE, TRUE, FALSE},

   {MAN1_LONGDESC,
    {
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {
	  "^\\.SH DESCRIPTION\\s*$",
	  "^\\.SH \"?REQUIRED ARGUMENTS\"?\\s*$",
	  KCMS_NO_EXTRA
       },

       {NULL, NULL, KCMS_NO_EXTRA},

       {
	  "^\\.section\\s*\\d*\\s*DESCRIPTION\\s*$",
	  "^\\.section\\s*\\d\\s*\"PANE ARGUMENTS\"\\s*$",
	  KCMS_NO_EXTRA
       }
    },
    "long program description", TRUE, FALSE, TRUE, FALSE},

   {MAN1_EXAMPLES,
    {
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},

       {
	  "^\\.SH EXAMPLES\\s*$",
	  "^\\.SH \"?SEE ALSO\"?\\s*$",
	  KCMS_NO_EXTRA
       },

       {NULL, NULL, KCMS_NO_EXTRA},
       {
	  "^\\.section\\s*\\d*\\s*EXAMPLES\\s*$",
	  "^\\.section\\s*\\d*\\s*\"?SEE ALSO\"?\\s*$",
	  KCMS_NO_EXTRA
       }
    },
    "program examples", TRUE, FALSE, TRUE, FALSE},

   {MAN1_RESTRICTIONS,
    {
       {NULL, NULL, KCMS_NO_EXTRA},

       {NULL, NULL, KCMS_NO_EXTRA},

       {NULL, NULL, KCMS_NO_EXTRA},

       {
	  "^\\.SH RESTRICTIONS\\s*$",
	  "^\\.SH REFERENCES\\s*$",
	  KCMS_NO_EXTRA},

       {NULL, NULL, KCMS_NO_EXTRA},

       {
	  "^\\.section\\s*\\d*\\s*RESTRICTIONS\\s*$",
	  "^\\.section\\s*\\d\\s*REFERENCES\\s*$",
	  KCMS_NO_EXTRA
       }
    },
    "restrictions on program usage", TRUE, FALSE, TRUE, FALSE},

   {MAN1_REFERENCES,
    {
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},

       {
	  "^\\.SH REFERENCES\\s*$",
	  "^\\.SH COPYRIGHT\\s*$",
	  KCMS_NO_EXTRA
       },

       {NULL, NULL, KCMS_NO_EXTRA},

       {
	  "^\\.section\\s*\\d*\\s*REFERENCES\\s*$",
	  "^\\.section\\s*\\d\\s*COPYRIGHT\\s*$",
	  KCMS_NO_EXTRA
       }
    },
    "references for program algorithms", TRUE, FALSE, TRUE, FALSE},

   {MAN1_SEEALSO,
    {
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {
	  "^\\.SH \"?SEE ALSO\"?\\s*$",
	  "^\\.SH RESTRICTIONS\\s*$",
	  KCMS_NO_EXTRA
       },

       {NULL, NULL, KCMS_NO_EXTRA},

       {
	  "^\\.section\\s*\\d*\\s*\"?SEE ALSO\"?\\s*$",
	  "^\\.section\\s*\\d*\\s*RESTRICTIONS\\s*$",
	  KCMS_NO_EXTRA
       }
    },
    "'see also' section", TRUE, FALSE, TRUE, FALSE},

   {USAGEADD,
    {
       {
	  "^\\s*/\\* -usage_additions \\*/\\s*$",
	  "^\\s*/\\* -usage_additions_end \\*/\\s*$",
	  KCMS_NO_EXTRA
       },

       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA}
    },
    "additions to usage statement", TRUE, FALSE, FALSE, FALSE},

   {INCLUDE_INCLUDES,
    {
       {NULL, NULL, KCMS_NO_EXTRA},
       {
	  "^\\s*\\Q/* -include_includes */\\E\\s*$",
	  "^\\s*\\Q/* -include_includes_end */\\E\\s*$",
	  KCMS_NO_EXTRA
       },

       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA}},
    "includes", TRUE, FALSE, FALSE, FALSE},

   {INCLUDE_DEFINES,
    {
       {NULL, NULL, KCMS_NO_EXTRA},
       {
	  "^\\s*\\Q/* -include_defines */\\E\\s*$",
	  "^\\s*\\Q/* -include_defines_end */\\E\\s*$",
	  KCMS_NO_EXTRA
       },

       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA}},
    "defines", TRUE, FALSE, FALSE, FALSE},

   {INCLUDE_TYPEDEFS,
    {
       {NULL, NULL, KCMS_NO_EXTRA},
       {
	  "^\\s*\\Q/* -include_typedefs */\\E\\s*$",
	  "^\\s*\\Q/* -include_typedefs_end */\\E\\s*$",
	  KCMS_NO_EXTRA
       },

       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA}},
    "typedefs", TRUE, FALSE, FALSE, FALSE},

   {INCLUDE_VARIABLES,
    {
       {NULL, NULL, KCMS_NO_EXTRA},
       {
	  "^\\s*\\Q/* -include_variables */\\E\\s*$",
	  "^\\s*\\Q/* -include_variables_end */\\E\\s*$",
	  KCMS_NO_EXTRA
       },

       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA}},
    "variables", TRUE, FALSE, FALSE, FALSE},

   {INCLUDE_MACROS,
    {
       {NULL, NULL, KCMS_NO_EXTRA},
       {
	  "^\\s*\\Q/* -include_macros */\\E\\s*$",
	  "^\\s*\\Q/* -include_macros_end */\\E\\s*$",
	  KCMS_NO_EXTRA
       },
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA}},
    "macros", TRUE, FALSE, FALSE, FALSE},

   {INCLUDE_ROUTINES,
    {
       {NULL, NULL, KCMS_NO_EXTRA},
       {
	  "^\\s*\\Q/* -include_routines */\\E\\s*$",
	  "^\\s*\\Q/* -include_routines_end */\\E\\s*$",
	  KCMS_NO_EXTRA
       },
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA}},
    "routines", TRUE, FALSE, FALSE, FALSE},

   {INCLUDE_ADDITIONS,		/* don't check begin key */
    {
       {NULL, NULL, KCMS_NO_EXTRA},
       {
	  "^\\s*/\\* -include_additions \\*/\\s*$",
	  "^\\s*/\\* -include_additions_end \\*/\\s*$",
	  KCMS_NO_EXTRA
       },
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA}
    },
    "additions", FALSE, FALSE, FALSE, FALSE
   },

   {MAIN_VARIABLES,
    {
       {
	  "^\\s*/\\* -main_variable_list \\*/\\s*$",
	  "^\\s*/\\* -main_variable_list_end \\*/\\s*$",
	  KCMS_NO_EXTRA
       },
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA}},
    "variables", TRUE, FALSE, FALSE, FALSE},

   {MAIN_GETARGS,
    {
       {
	  "^\\s*/\\* -main_get_args_call \\*/\\s*$",
	  "^\\s*/\\* -main_get_args_call_end \\*/\\s*$",
	  KCMS_NO_EXTRA
       },
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA}},
    "get arguments call", TRUE, FALSE, FALSE, FALSE},

   {MAIN_BEFORELIB,
    {
       {
	  "^\\s*/\\* -main_before_lib_call \\*/\\s*$",
	  "^\\s*/\\* -main_before_lib_call_end \\*/\\s*$",
	  KCMS_NO_EXTRA
       },
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA}},
    "code to go before lib call", TRUE, FALSE, FALSE, FALSE},

   {MAIN_LIBCALL,
    {
       {
	  "^\\s*/\\* -main_library_call \\*/\\s*$",
	  "^\\s*/\\* -main_library_call_end \\*/\\s*$",
	  KCMS_NO_EXTRA
       },
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA}},
    "library call", TRUE, FALSE, FALSE, FALSE},

   {MAIN_AFTERLIB,
    {
       {
	  "^\\s*/\\* -main_after_lib_call \\*/\\s*$",
	  "^\\s*/\\* -main_after_lib_call_end \\*/\\s*$",
	  KCMS_NO_EXTRA
       },
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA}},
    "code to go after library call", TRUE, FALSE, FALSE, FALSE},

   {FREEHANDADD,
    {
       {
	  "^\\s*/\\* -free_handler_additions \\*/\\s*$",
	  "^\\s*/\\* -free_handler_additions_end \\*/\\s*$",
	  KCMS_NO_EXTRA
       },
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA}},
    "code to go in the exit handler", TRUE, FALSE, FALSE, FALSE},

   {LIBRARY_SHORTDESC,
    {
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {
	  "^\\*\\s*Routine Name:\\s*l%s\\s*-*",
	  "^\\*\\s*Purpose:",
	  ONAME
       },
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA}},
    "short library routine description", TRUE, TRUE, TRUE, TRUE},

   {LIBRARY_LONGDESC,
    {
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {
	  "^\\s*\\*\\s*Purpose:",
	  "^\\s*\\*\\s*Input:",
	  KCMS_NO_EXTRA
       },
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA}},
    "library header 'purpose' section", TRUE, FALSE, TRUE, FALSE},

   {LIBRARY_INPUT,
    {
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {
	  "^\\s*\\*\\s*Input:",
	  "^\\s*\\*\\s*Output:",
	  KCMS_NO_EXTRA
       },
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA}},
    "library header 'input' section", TRUE, FALSE, TRUE, FALSE},

   {LIBRARY_OUTPUT,
    {
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {
	  "^\\s*\\*\\s*Output:",
	  "^\\s*\\*\\s*Returns:",
	  KCMS_NO_EXTRA
       },
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA}},
    "library header 'output' section", TRUE, FALSE, TRUE, FALSE},

   {LIBRARY_RETURNS,
    {
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {
	  "^\\s*\\*\\s*Returns:",
	  "^\\s*\\*\\s*Restrictions:",
	  KCMS_NO_EXTRA
       },
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA}},
    "library header 'returns' section", TRUE, FALSE, TRUE, FALSE},

   {LIBRARY_RESTRICT,
    {
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {
	  "^\\s*\\*\\s*Restrictions:",
	  "^\\s*\\*\\s*Written By:",
	  KCMS_NO_EXTRA
       },
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA}},
    "library header 'restrictions' section", TRUE, FALSE, TRUE, FALSE},

   {LIBRARY_VERIFY,
    {
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {
	  "^\\s*\\*\\s*Verified:",
	  "^\\s*\\*\\s*Side Effects:",
	  KCMS_NO_EXTRA
       },
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA}},
    "library header 'verify' section", TRUE, FALSE, TRUE, FALSE},

   {LIBRARY_SIDEEFF,
    {
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {
	  "^\\s*\\*\\s*Side Effects:",
	  "^\\s*\\*\\s*Modifications:",
	  KCMS_NO_EXTRA
       },
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA}},
    "library header 'side effects' section", TRUE, FALSE, TRUE, FALSE},

   {LIBRARY_MODS,
    {
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {
	  "^\\s*[|*]\\s*Modifications:",
	  "^\\s*\\*{3,}/$",
	  KCMS_NO_EXTRA
       },
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA}},
    "library header 'modifications' section", TRUE, FALSE, TRUE, FALSE},

   {LIBRARY_DEF,
    {
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {
	  "^\\s*/\\* -library_def \\*/\\s*$",
	  "^\\s*/\\* -library_def_end \\*/\\s*$",
	  KCMS_NO_EXTRA
       },
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA}},
    "library definition", TRUE, FALSE, FALSE, FALSE},

   {LIBRARY_CODE,
    {
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {
	  "^\\s*/\\* -library_code \\*/\\s*$",
	  "^\\s*/\\* -library_code_end \\*/\\s*$",
	  KCMS_NO_EXTRA
       },
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA}},
    "library code body", TRUE, FALSE, FALSE, FALSE},

   {LIBRARY_INCLUDES,
    {
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {
	  "^\\s*/\\* -library_includes \\*/\\s*$",
	  "^\\s*/\\* -library_includes_end \\*/\\s*$",
	  KCMS_NO_EXTRA
       },
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA}},
    "includes", TRUE, FALSE, FALSE, FALSE},

   {MAN3_SHORTDESC,
    {
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {
	  "^%s \\\\-\\s*",
	  "^\\.SH DESCRIPTION\\s*$",
	  ONAME
       },
       {NULL, NULL, KCMS_NO_EXTRA}},
    "library short description", TRUE, FALSE, TRUE, FALSE},

   {MAN3_LONGDESC,
    {
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {
	  "^\\.SH DESCRIPTION\\s*$",
	  "^\\.SH \"LIST OF LIBRARY FUNCTIONS\"\\s*$",
	  KCMS_NO_EXTRA
       },

       {NULL, NULL, KCMS_NO_EXTRA}
    },
    "library long description", TRUE, FALSE, TRUE, FALSE},

   {MAN3_ADDINFO,
    {
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {
	  "^\\.SH \"ADDITIONAL INFORMATION\"\\s*$",
	  "^\\.SH \"LOCATION OF SOURCE FILES:\"\\s*$",
	  KCMS_NO_EXTRA
       },
       {NULL, NULL, KCMS_NO_EXTRA}},
    "library additional information", TRUE, FALSE, TRUE, FALSE},

   {MAN3_SEEALSO,
    {
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {
	  "^\\.SH \"SEE ALSO\"\\s*$",
	  "^\\.SH \"SEE MANUAL\"\\s*$",
	  KCMS_NO_EXTRA
       },
       {NULL, NULL, KCMS_NO_EXTRA}},
    "library see also", TRUE, FALSE, TRUE, FALSE},

   {MAN3_MANUAL,
    {
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {NULL, NULL, KCMS_NO_EXTRA},
       {
	  "^\\.SH \"SEE MANUAL\"\\s*$",
	  "^\\.SH COPYRIGHT\\s*$",
	  KCMS_NO_EXTRA
       },
       {NULL, NULL, KCMS_NO_EXTRA}},
    "library manual reference", TRUE, FALSE, TRUE, FALSE}
};

/*-----------------------------------------------------------
| Routine Name:	_kghost_init_prog_spec - initialize an objects prog spec array
|
| Purpose:	This routine mallocs an array large enough to hold the
|		prog_spec and the prog_valid arrays in the object's internal
|		structure.  If the prog_spec is already initialized, this
|		routine will do nothing and return TRUE.
|
| Input:	prog - object that (potentially) needs prog_spec array space
| Output:	
| Returns:	TRUE (1) on success or already initialized, FALSE (0) otherwise
| Written By:	Steven Jorgensen
| Date:		Apr 14, 1993 01:05
------------------------------------------------------------*/
static int
_kghost_init_prog_spec(
   kobject  prog)
{
   kstring  routine = "_kghost_init_prog_spec";


   if (prog->prog_spec != NULL)
   {
      kfree(prog->prog_spec);
      prog->prog_spec = NULL;
   }

   prog->prog_spec = (char **)kcalloc(MAX_PROGSPEC_SIZE, sizeof(char *));
   if (prog->prog_spec == NULL)
   {
      kerror(KCMS, routine, "Cannot calloc prog_spec");
      errno = KCMS_CANNOTMALLOCOBJ;
      return FALSE;
   }
   prog->prog_valid = (int *)kcalloc(MAX_PROGSPEC_SIZE, sizeof(int));
   if (prog->prog_valid == NULL)
   {
      kerror(KCMS, routine, "Cannot calloc prog_valid");
      kfree(prog->prog_spec);
      errno = KCMS_CANNOTMALLOCOBJ;
      return FALSE;
   }
   prog->prog_cnt = MAX_PROGSPEC_SIZE;
   return TRUE;
}

/*-----------------------------------------------------------
| Routine Name:	_kghost_missing_key_warning - print a warning for a missing key
|
| Purpose:	This routine makes a call to kinfo to print an
|		a warning message about a missing key for the user
|
| Input:	key	   - The key that was not found
|		block_type - The section that the key was trying to find.
|		location   - The file the key was supposed to be in.
|		status	   - The status returned by the string parser.
|
| Written By:	Neil Bowers
| Date:		29-jun-94
------------------------------------------------------------*/
static void
_kghost_missing_key_warning(
   kstring  key,
   kstring  block_type,
   kstring  location,
   int      status)
{
   /*-- safety check: was there really an error? ----------------------*/
   if (status != KPARSE_NOKEY && status != KPARSE_NOEND)
      return;

   kinfo(KFORCE, "Unable to find %s key '%s'; marking the %s of the "
	 "text block for %s to go into the %s. "
	 "This part of the generated code will be left empty.",
	 (status == KPARSE_NOKEY ? "begin" : "end"),
	 key,
	 (status == KPARSE_NOKEY ? "beginning" : "ending"),
	 block_type, location);
}

/*-----------------------------------------------------------
| Routine Name:	_kghost_strip_hfile - string source code '|' garbage out of
|		a string
|
| Purpose:	This routine removes the '>>>[ \t]*' that follows the
|		'\n' in text that was retrieved from a hfile header.
|
| Input:	str - string to clean up
| Returns:	The cleaned up string
|
| Written By:	Steven Jorgensen
| Date:		Oct 22, 1992 22:47
------------------------------------------------------------*/
static char *
_kghost_strip_hfile(
   kstring  str)
{
   char     new[2 * KLENGTH];
   kstring  sstr      = str;
   int      i         = 0;
   int      lastwhite = TRUE;


   if (str == NULL)
      return NULL;

   while (((unsigned int)*sstr) != '\0')
   {
      if ((((unsigned int)*sstr) == '>') && (lastwhite == TRUE))
      {
	 sstr++;
	 while ((((unsigned int)*sstr) != '\0') &&
		((((unsigned int)*sstr) == ' ') ||
		 (((unsigned int)*sstr) == '\t') ||
		 (((unsigned int)*sstr) == '>')))
	    sstr++;
      }
      else
      {
	 if ((lastwhite == TRUE) && (!isspace(((unsigned int)*sstr))))
	    lastwhite = FALSE;
	 if (((unsigned int)*sstr) == '\n')
	    lastwhite = TRUE;
	 new[i++] = *sstr++;
      }
   }
   new[i] = (char)'\0';
   return kstring_cleanup(new, NULL);
}

/*-----------------------------------------------------------
| Routine Name:	kghost_get_prog_info - get the prog information
|
| Purpose:	This function reads a block of text between two keys from the
|		an open file and returns it in a contiguous block of memory.
|		This routine is a front end to the general parser, and services
|		the ghost routines specific needs.
|
| Input:	file    - open stream to a file
|		id      - the index into the prog array
|		file_id - the index indicating the file.  One of:
|		! KCMS_FOBJ_GEN_MAIN	oname.c
|		! KCMS_FOBJ_GEN_INCL	oname.h
|		! KCMS_FOBJ_GEN_LFILE	loname.h
|		! KCMS_FOBJ_GEN_MAN1	man1/bname
|		! KCMS_FOBJ_GEN_HELP	help/oname/oname.hlp
|		names    - an array with the oname and bname
|
| Output:	status - returns the status of the textblock search.
|		may be one of:
|		! KPARSE_OK,		{ valid text block found }
|		! KPARSE_NOKEY,	{ begin key not found    }
|		! KPARSE_NOEND,	{ end key not found    }
|		! KPARSE_PARTKEY	{ only part of begin key found }
|		! KPARSE_PARTEND	{ only part of end key found }
|		! KPARSE_SYNTAXKEY	{ syntax error on begin key }
|		! KPARSE_SYNTAXEND	{ syntax error on end key }
|		! KPARSE_DATAERR	{ an error occurred while parsing}
|
| Returns:	a pointer to a block of characters that was read from
|		a file on success, or NULL on failure.
| Written By:	Steven Jorgensen
| Date:		Feb 22, 1993 22:04
------------------------------------------------------------*/
static char *
kghost_get_prog_info(
   kfile  *file,
   int     id,
   int     file_id,
   char  **names)
{
   char     begin_key[KLENGTH];
   kstring  tmp;
   kstring  tmp1;
   int      status;


   if (file == NULL)
      return NULL;
   if (prog_info[id].key[file_id].begin == NULL)
      return NULL;
   if (prog_info[id].key[file_id].begextra == KCMS_NO_EXTRA)
      kstrcpy(begin_key, prog_info[id].key[file_id].begin);
   else
      ksprintf(begin_key, prog_info[id].key[file_id].begin,
	       names[prog_info[id].key[file_id].begextra]);
   krewind(file);
   tmp = kparse_file_scan(file, begin_key, prog_info[id].key[file_id].end,
			  KIGNORE_CASE, NULL, NULL, &status);
   if (status == KPARSE_OK)
   {
      if (prog_info[id].strip == TRUE &&
	  (file_id == KCMS_FOBJ_GEN_MAIN || file_id == KCMS_FOBJ_GEN_LFILE))
	 tmp1 = kstrip_header(tmp);
      else if (prog_info[id].strip == TRUE && (file_id == KCMS_FOBJ_GEN_INCL))
	 tmp1 = _kghost_strip_hfile(tmp);
      else if (prog_info[id].cleanup == TRUE)
	 tmp1 = kstring_cleanup(tmp, NULL);
      else
      {
	 tmp1 = tmp;
	 tmp = NULL;
      }
      kfree(tmp);
      if (prog_info[id].oneline == TRUE)
      {
	 tmp = kstring_replace(tmp1, "\n", " ", NULL);
	 kfree(tmp1);
	 return tmp;
      }
      return tmp1;
   }
   kfree(tmp);
   if (prog_info[id].checkerr == TRUE && status == KPARSE_NOKEY)
      _kghost_missing_key_warning(begin_key, prog_info[id].block_type,
				  ftbl[file_id].file_desc, KPARSE_NOKEY);
   else if (status == KPARSE_NOEND)
      _kghost_missing_key_warning(prog_info[id].key[file_id].end,
				  prog_info[id].block_type,
				  ftbl[file_id].file_desc, KPARSE_NOEND);
   return NULL;
}

/*-----------------------------------------------------------
| Routine Name:	_kghost_check_five - compare five fields and allow user
|		to choose correct one if different
|
| Purpose:	This routine takes the five different entries
|		and compares them.  If they are different, then the
|		user is prompted as to which one to use.
|
| Input:	one     - string containing the first reference
|		two     - string containing the second reference
|		three   - string containing the third reference
|		four    - string containing the fourth reference
|		five    - string containing the fifth reference
|		force   - whether force is specified or not
|		progidx - index into the prog spec table
|
| Output:	
|
| Returns:	a pointer to the correct string
|
| Written By:	Steven Jorgensen
| Date:		Feb 22, 1993 23:25
------------------------------------------------------------*/
static kbool
_kghost_check_five(
   kstring  one,
   kstring  two,
   kstring  three,
   kstring  four,
   kstring  five,
   int      force,
   int      progidx,
   char   **field)
{
   int     cnt = 0;
   int     i;
   int     keep[KSPFILE];
   int     findx[KSPFILE];
   char   *list1[KSPFILE];
   char   *list2[KSPFILE];
   char   *nlist1[KSPFILE];
   char   *tmp1;
   char   *tmp2;
   kbool   retvalue = TRUE;
   char    prompt[KLENGTH];


   *field = NULL;		/* initialize return string to NULL */
   if (one != NULL)
   {
      list1[cnt] = kstrdup(one);
      findx[cnt] = KCMS_FOBJ_GEN_MAN1;
      cnt++;
   }
   if (two != NULL)
   {
      list1[cnt] = kstrdup(two);
      findx[cnt] = KCMS_FOBJ_GEN_HELP;
      cnt++;
   }
   if (three != NULL)
   {
      list1[cnt] = kstrdup(three);
      findx[cnt] = KCMS_FOBJ_GEN_INCL;
      cnt++;
   }
   if (four != NULL)
   {
      list1[cnt] = kstrdup(four);
      findx[cnt] = KCMS_FOBJ_GEN_MAIN;
      cnt++;
   }
   if (five != NULL)
   {
      list1[cnt] = kstrdup(five);
      findx[cnt] = KCMS_FOBJ_GEN_LFILE;
      cnt++;
   }
   if (cnt == 0)
      if (cnt == 0)
	 return TRUE;

   if (cnt <= 1)
   {
      *field = list1[0];
      return TRUE;
   }

   /* cnt is 2 or more, so initialize the lists */
   for (i = 0; i < cnt; i++)
   {
      tmp1 = kstring_replace(list1[i], " ", NULL, NULL);
      tmp2 = kstring_replace(tmp1, "\t", NULL, NULL);
      list2[i] = kstring_replace(tmp2, "\n", NULL, NULL);
      nlist1[i] = NULL;
      keep[i] = FALSE;
      kfree(tmp1);
      kfree(tmp2);
   }
   for (i = cnt; i < KSPFILE; i++)
   {
      findx[i] = KCMS_NO_EXTRA;
      list1[i] = kstrdup(list1[0]);
      list2[i] = kstrdup(list2[0]);
      nlist1[i] = NULL;
      keep[i] = FALSE;
   }

   /* determine which of the fields are unique */
   cnt = 1;
   nlist1[0] = list1[0];

   if (kstrcmp(list2[1], list2[0]) != 0)
   {
      keep[1] = TRUE;
      nlist1[cnt] = list1[1];
      findx[cnt] = findx[1];
      cnt++;
   }
   else
      kfree(list1[1]);

   if (((keep[1] != TRUE) || (kstrcmp(list2[2], list2[1]) != 0)) &&
       (kstrcmp(list2[2], list2[0]) != 0))
   {
      keep[2] = TRUE;
      nlist1[cnt] = list1[2];
      findx[cnt] = findx[2];
      cnt++;
   }
   else
      kfree(list1[2]);

   if (((keep[2] != TRUE) || (kstrcmp(list2[3], list2[2]) != 0)) &&
       ((keep[1] != TRUE) || (kstrcmp(list2[3], list2[1]) != 0)) &&
       (kstrcmp(list2[3], list2[0]) != 0))
   {
      keep[3] = TRUE;
      nlist1[cnt] = list1[3];
      findx[cnt] = findx[3];
      cnt++;
   }
   else
      kfree(list1[3]);

   if (((keep[3] != TRUE) || (kstrcmp(list2[4], list2[3]) != 0)) &&
       ((keep[2] != TRUE) || (kstrcmp(list2[4], list2[2]) != 0)) &&
       ((keep[1] != TRUE) || (kstrcmp(list2[4], list2[1]) != 0)) &&
       (kstrcmp(list2[4], list2[0]) != 0))
   {
      keep[4] = TRUE;
      nlist1[cnt] = list1[4];
      findx[cnt] = findx[4];
      cnt++;
   }
   else
      kfree(list1[4]);

   for (i = 0; i < KSPFILE; i++)
      kfree(list2[i]);

   /* if there is only one unique one, return it */
   if (cnt == 1)
   {
      *field = list1[0];
      return TRUE;
   }

   /* otherwise, let the user choose the correct one */
   switch (cnt)
   {
      case 2:
	 ksprintf(prompt, "the %s and the %s",
		  ftbl[findx[0]].file_desc,
		  ftbl[findx[1]].file_desc);
	 break;
      case 3:
	 ksprintf(prompt, "the %s, the %s, and the %s",
		  ftbl[findx[0]].file_desc,
		  ftbl[findx[1]].file_desc,
		  ftbl[findx[2]].file_desc);
	 break;
      case 4:
	 ksprintf(prompt, "the %s, the %s, the %s, and the %s",
		  ftbl[findx[0]].file_desc,
		  ftbl[findx[1]].file_desc,
		  ftbl[findx[2]].file_desc,
		  ftbl[findx[3]].file_desc);
	 break;
      case 5:
	 ksprintf(prompt,
		  "the %s, the %s, the %s, the %s, and the %s",
		  ftbl[findx[0]].file_desc,
		  ftbl[findx[1]].file_desc,
		  ftbl[findx[2]].file_desc,
		  ftbl[findx[3]].file_desc,
		  ftbl[findx[4]].file_desc);
	 break;
   }
   if (force == TRUE)
   {
      kinfo(KSTANDARD, "Force is set to TRUE, and the %s does not match "
	    "between %s.  Choosing default: '%s' in the %s\n",
	    prog_info[progidx].block_type, prompt, nlist1[0],
	    ftbl[findx[0]].file_desc);
      *field = kstrdup(nlist1[0]);
   }
   else
   {
      *field = kchoose(KSTANDARD, nlist1, cnt, 1, NULL, &i,
		       "The %s between %s are not the same.  Please "
		       "select the text section listed below that you "
		       "want to appear in %s. (Note: selecting 'Cancel' "
		       "implies you want the %s to be empty)",
		       prog_info[progidx].block_type, prompt, prompt,
		       prog_info[progidx].block_type);
      retvalue = (*field != NULL);
   }

   for (i = 0; i < KSPFILE; i++)
      kfree(nlist1[0]);
   return retvalue;
}

/*-----------------------------------------------------------
| Routine Name:	kghost_get_progspec - file out a prog specification definition
|
| Purpose:	This routine reads the various source and documentation files
|		to aquire the program specification information required by
|		the ghostwriter to generate new main and library code and
|		documentation.  It allocates and returns
|		the prog_spec array of character blocks, each of which is
|		referenced by an index with a #define'd name
|		that indicates the key from which it came.
|
| Input:	prog - program object to get the prog file of
| Output:	
| Returns:	a pointer to the prog spec array or NULL on error
|
| Written By:	Steven Jorgensen & Danielle Argiro
| Date:		Jan 07, 1993 16:42
------------------------------------------------------------*/
char **
kghost_get_progspec(
   kobject  prog)
{
   kfile     *file[KSPFILE];
   char     **prog_spec;
   kstring    names[2];
   kstring    prog_stuff[KSPFILE];
   kstring    path;
   int        i;
   int        j;
   int        force;
   kobject    fobj[KSPFILE];


   for (i = 0; i < KSPFILE; i++)
   {
      prog_stuff[i] = NULL;
      fobj[i] = NULL;
      file[i] = NULL;
   }

   if (_kghost_init_prog_spec(prog) == FALSE)
      return NULL;

   prog_spec = prog->prog_spec;

   path = NULL;
   for (i = 0; i < KSPFILE; i++)
   {
      if (!kcms_get_attribute(prog, ftbl[i].file_ident, &fobj[i]))
      {
	 errno = KCMS_METHODERR;
	 return NULL;
      }
      /*-- NEILB - what if one the file object is NULL?	--*/
      /*-- for example, i==2 --> library file for lroutine	--*/
      /*-- this is null for objects other than lkroutine	--*/

      if (fobj[i] == NULL)
	 continue;

      /*-- end of NEILB --*/

      if (!kcms_get_attribute(fobj[i], KCMS_PATH, &path))
      {
	 errno = KCMS_OK;
	 continue;
      }
      if (kaccess(path, F_OK) == 0)
      {
	 if ((file[i] = kfopen(path, "r")) == NULL)
	 {
	    kerror(KCMS, "kghost_get_progspec",
		   "Unable to open '%s' to read %s", path,
		   ftbl[i].file_desc);
	    errno = KCMS_METHODERR;
	    return NULL;
	 }
      }
   }

   if (kcms_get_attribute(prog, KCMS_NAME, &names[ONAME]) == FALSE)
      return NULL;
   if (kcms_get_attribute(prog, KCMS_CMOBJ_BNAME, &names[BNAME]) == FALSE)
      return NULL;

   if (!kcms_query_bit(prog, KCMS_CMOBJ_FLAGS, KCMS_BIT_CMOBJ_FORCE, &force))
      return NULL;

   for (i = 0; i < MAX_PROGSPEC_SIZE; i++)
   {
      if (prog->prog_valid[i] == FALSE)
      {
	 for (j = 0; j < KSPFILE; j++)
	 {
	    kfree(prog_stuff[j]);
	    prog_stuff[j] = kghost_get_prog_info(file[j], i,
						 j, names);
	 }
	 if (i > LIBRARY_INCLUDES)
	    prog_spec[i] = kstrdup(prog_stuff[KCMS_FOBJ_GEN_MAN3]);
	 else if (!_kghost_check_five(
					prog_stuff[KCMS_FOBJ_GEN_MAN1],
					prog_stuff[KCMS_FOBJ_GEN_HELP],
					prog_stuff[KCMS_FOBJ_GEN_INCL],
					prog_stuff[KCMS_FOBJ_GEN_MAIN],
					prog_stuff[KCMS_FOBJ_GEN_LFILE],
					force, i,
					&prog_spec[i]))
	    return NULL;

	 prog->prog_valid[i] = TRUE;
      }
   }

   for (i = 0; i < KSPFILE; i++)
   {
      kfree(prog_stuff[i]);
      kfclose(file[i]);
   }

   return prog_spec;
}

/*-----------------------------------------------------------
| Routine Name:	kghost_get_progitem - get one prog tag field from all places
|		it exists.
|
| Purpose:	This routine is similar in function as get_prog_spec;
|		however, this routine only gets the specified prog
|		tag field from each file it exists in, resolves
|		possible differences, and selects or prompts the
|		correct string to return to the calling routine.
|
| Input:	prog	-	kobject pointer of the program user wants
|		to get information on
|		pindx	-	prog spec index indicating which prog tag
|		field to get.
| Output:	none
|
| Returns:	The string inside the field requested, NULL on error or on
|		empty.
|
| Written By:	Steven Jorgensen
| Date:		Apr 12, 1993 02:50
------------------------------------------------------------*/
char *
kghost_get_progitem(
   kobject  prog,
   int      pindx)
{
   kfile   *file;
   int      force;
   int      i;
   kstring  path;
   kstring  names[2];
   kstring  prog_stuff[KSPFILE];
   kobject  fobj;


   if (pindx < 0 || pindx > MAX_PROGSPEC_SIZE)
      return NULL;

   if (_kghost_init_prog_spec(prog) == FALSE)
      return NULL;

   if (kcms_get_attribute(prog, KCMS_NAME, &names[ONAME]) == FALSE)
      return NULL;
   if (kcms_get_attribute(prog, KCMS_CMOBJ_BNAME, &names[BNAME]) == FALSE)
      return NULL;

   path = NULL;
   for (i = 0; i < KSPFILE; i++)
   {
      prog_stuff[i] = NULL;
      if (prog_info[pindx].key[i].begin != NULL)
      {
	 if (kcms_get_attribute(prog, ftbl[i].file_ident, &fobj)
	     == FALSE)
	 {
	    errno = KCMS_METHODERR;
	    return NULL;
	 }
	 if (fobj == NULL || !kcms_get_attribute(fobj, KCMS_PATH, &path))
	 {
	    errno = KCMS_OK;
	    continue;
	 }
	 if (kaccess(path, F_OK) == 0)
	 {
	    if ((file = kfopen(path, "r")) == NULL)
	    {
	       kerror(KCMS, "kghost_get_progitem",
		      "Unable to open '%s' to read %s",
		      path, ftbl[i].file_desc);
	       errno = KCMS_METHODERR;
	       return NULL;
	    }
	    prog_stuff[i] = kghost_get_prog_info(file, pindx, i, names);
	    kfclose(file);
	 }
      }
   }
   if (!kcms_query_bit(prog, KCMS_CMOBJ_FLAGS, KCMS_BIT_CMOBJ_FORCE, &force))
      return NULL;
   if (!_kghost_check_five(prog_stuff[KCMS_FOBJ_GEN_MAN1],
			   prog_stuff[KCMS_FOBJ_GEN_HELP],
			   prog_stuff[KCMS_FOBJ_GEN_INCL],
			   prog_stuff[KCMS_FOBJ_GEN_MAIN],
			   prog_stuff[KCMS_FOBJ_GEN_LFILE],
			   force, pindx, &prog->prog_spec[pindx]))
      return NULL;
   prog->prog_valid[pindx] = TRUE;
   return prog->prog_spec[pindx];
}

/*-----------------------------------------------------------
| Routine Name:	kghost_set_progitem - set one prog tag field in all places
|		it exists.
|
| Purpose:	This routine sets the specified prog tag field in
|		each file it exists in.
|
| Input:	prog   - kobject pointer of the program user wants to get
|		information on
|		pindx  - prog spec index indicating which prog tag field to
|		get.
|		field  - string section to replace the section with
|		ignore - file number to ignore
| Output:	
|
| Returns:	TRUE (1) on success, FALSE (0) otherwise
|
| Written By:	Steven Jorgensen
| Date:		Apr 24, 1993 15:46
------------------------------------------------------------*/
int
kghost_set_progitem(
   kobject  prog,
   int      pindx,
   kstring  field,
   int      ignore)
{
   kstring   routine = "kghost_set_progitem()";
   kobject   fobj;
   kfile    *file;
   int       i;
   int       status1;
   int       status2;
   kstring   path;
   kstring   names[2];
   kstring   key1;
   kstring   key2;
   kstring   begin;
   kstring   end;
   char      compose_key[KLENGTH];
   char      begin_key[KLENGTH];


   if (pindx < 0 || pindx > MAX_PROGSPEC_SIZE
       || !_kghost_init_prog_spec(prog))
   {
      return FALSE;
   }

   if (!kcms_get_attributes(prog,
			   KCMS_NAME,        &names[ONAME],
			   KCMS_CMOBJ_BNAME, &names[BNAME],
			   KCMS_END))
      return FALSE;

   path = NULL;
   for (i = 0; i < KSPFILE; i++)
   {
      if (i == ignore)
	 continue;
      if (prog_info[pindx].key[i].begin != NULL)
      {
	 if (kcms_get_attribute(prog, ftbl[i].file_ident, &fobj)
	     == FALSE)
	 {
	    errno = KCMS_METHODERR;
	    return FALSE;
	 }
	 if (fobj == NULL || !kcms_get_attribute(fobj, KCMS_PATH, &path))
	 {
	    errno = KCMS_OK;
	    continue;
	 }
	 if (kaccess(path, F_OK) == 0)
	 {
	    if ((file = kfopen(path, "r")) == NULL)
	    {
	       kerror(KCMS, "kghost_set_progitem",
		      "Unable to open '%s' to read %s",
		      path, ftbl[i].file_desc);
	       errno = KCMS_METHODERR;
	       return FALSE;
	    }
	    if (prog_info[pindx].key[i].begextra == KCMS_NO_EXTRA)
	       kstrcpy(compose_key,
		       prog_info[pindx].key[i].begin);
	    else
	       ksprintf(compose_key, prog_info[pindx].key[i].begin,
			names[prog_info[pindx].key[i].begextra]);
	    if (prog_info[pindx].key[i].begextra ==
		KCMS_NO_EXTRA)
	       kstrcpy(begin_key, prog_info[pindx].key[i].begin);
	    else
	       ksprintf(begin_key, prog_info[pindx].key[i].begin,
			names[prog_info[pindx].key[i].begextra]);

	    begin = kparse_file_scan(file, KPARSE_BOF,
				     begin_key, KIGNORE_CASE, NULL, &key1,
				     &status1);
	    end = kparse_file_scan(file,
				   prog_info[pindx].key[i].end,
				   KPARSE_EOF, KIGNORE_CASE, &key2,
				   NULL, &status2);
	    kfclose(file);
	    if (status1 != KPARSE_OK ||
		status2 != KPARSE_OK)
	    {
	       kerror(KCMS, routine,
		      "Unable to replace tag section in file '%s'.", path);
	       errno = KCMS_METHODERR;
	       return FALSE;
	    }
	    if ((file = kfopen(path, "w")) == NULL)
	    {
	       kerror(KCMS, "kghost_set_progitem",
		      "Unable to open '%s' to write %s",
		      path, ftbl[i].file_desc);
	       errno = KCMS_METHODERR;
	       return FALSE;
	    }
	    kfputs(begin, file);
	    kfputs(key1, file);
	    if (field != NULL)
	       kfprintf(file, "\n%s\n%s\n", field, key2);
	    else
	       kfprintf(file, "\n%s\n", key2);
	    kfputs(end, file);
	    kfclose(file);
	    kfree(key1);
	    kfree(key2);
	    kfree(begin);
	    kfree(end);
	 }
      }
   }
   prog->prog_valid[pindx] = TRUE;
   prog->prog_spec[pindx] = kstrdup(field);
   return TRUE;
}

/*-----------------------------------------------------------
| Routine Name:	kghost_update_file - get all the fields from the specified, and
|		propagate to all the others affected.
|
| Purpose:	This routine takes the specified input file, and
|
| Input:	object  - program object that is affected
|		filenum - filenumber to update
| Output:	
| Returns:	TRUE (1) on success, FALSE (0) otherwise
| Written By:	Steven Jorgensen
| Date:		May 28, 1993 13:33
------------------------------------------------------------*/
int
kghost_update_file(
   kobject  object,
   int      filenum)
{
   int       i;
   kstring   info;
   kstring   path;
   kstring   names[2];
   kobject   fobj;
   kfile    *file;


   if (_kghost_init_prog_spec(object) == FALSE)
      return FALSE;

   if (!kcms_get_attributes(object,
			    KCMS_NAME,        &names[ONAME],
			    KCMS_CMOBJ_BNAME, &names[BNAME],
			    KCMS_END))
      return FALSE;

   path = NULL;
   if (!kcms_get_attribute(object, ftbl[filenum].file_ident, &fobj))
      return FALSE;
   if (!kcms_get_attribute(fobj, KCMS_PATH, &path))
      return FALSE;
   if (kaccess(path, F_OK) != 0)
   {
      errno = KCMS_PERMISSION;
      return FALSE;
   }
   if ((file = kfopen(path, "r")) == NULL)
   {
      kerror(KCMS, "kghost_get_progspec",
	     "Unable to open '%s' to read %s", path,
	     ftbl[filenum].file_desc);
      return FALSE;
   }
   for (i = 0; i < MAX_PROGSPEC_SIZE; i++)
   {
      if (prog_info[i].key[filenum].begin == NULL)
	 continue;
      info = kghost_get_prog_info(file, i, filenum, names);
      if (kghost_set_progitem(object, i, info, filenum) == FALSE)
      {
	 errno = KCMS_METHODERR;
	 kfclose(file);
	 return FALSE;
      }
      kfree(info);
   }
   kfclose(file);
   return TRUE;
}
