/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Khoros clui get interface routines
   >>>>
   >>>>	 Static:
   >>>>  Private:
   >>>>             kclui_prompt_value()
   >>>>
   >>>>             kclui_prompt_flag()
   >>>>             kclui_prompt_int()
   >>>>             kclui_prompt_long()
   >>>>             kclui_prompt_short()
   >>>>             kclui_prompt_float()
   >>>>             kclui_prompt_double()
   >>>>             kclui_prompt_string()
   >>>>             kclui_prompt_infile()
   >>>>             kclui_prompt_outfile()
   >>>>             kclui_prompt_logical()
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"


/*-----------------------------------------------------------
|
|  Routine Name: kclui_prompt_value - prompt the user for a value
|
|       Purpose: This routines is used to get a value from a form
|		 structure.  The value is pulled from the form structure,
|		 for the given variable name.  If the variable is not
|		 defined failure is returned and the errno XXXX is set.
|
|         Input: pane - the pane to search for the value
|                name - the name of the flag
|                attribute     - the attribute associated with the selection
|                prompt_string - the string to prompt the user
|                type          - the data type (ie. KINT, etc) of the return
|				 value
|
|        Output: return_value - the pointer to the value to be returned
|		 flag	      - the flag as to whether the user specified it
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young & Danielle Argiro
|          Date: Mar 28, 1994
| Modifications:
|
------------------------------------------------------------*/

int kclui_prompt_value(
   kform *pane,
   char  *name,
   char  *attribute,
   char  *prompt_string,
   int   type,
   kaddr return_value,
   int   *flag)
{
	kselection   *selection;
	kform_struct *formstruct;

	int     intval;
	float   floatval;
	double  doubleval;
	kfile   *file;
	char    temp[KLENGTH];
	int     present, done = FALSE;


	/*
	 *  Initialize the presence flag to FALSE
	 */
	if (flag)
	   *flag = FALSE;

	if (!kclui_get_value(pane, name, attribute, return_value, &present))
	   return(FALSE);
	else if (present)
	{
	   if (flag) *flag = present;
	   return(TRUE);
	}

	/*
	 *  Find the selection, if not then return FALSE for failure
	 */
	if ((selection = kvf_variable_sel_search(pane, name)) == NULL)
	   return(FALSE);

	formstruct = selection->back_kformstruct;
	file = kfopen("/dev/tty", "w+");
	if (!file)
	{
	   kerror("kclui", "kclui_prompt_value", "Unable to open /dev/tty for \
interactive prompting of variable '%s'", name);
	   return(FALSE);
	}

	do
	{
	   kfprintf(file, "%s", prompt_string);
	   if (!kfgets(temp, KLENGTH, file) || !kstring_cleanup(temp, temp))
	   {
	      kfclose(file);
	      return(TRUE);
	   }

	   switch (type)
	   {
	      case KINT:
		   if ((done = ksscanf(temp, "%d", &intval)) == 1)
		   {
		      if (selection->type == KUIS_INTEGER)
		      {
		          if (!kvf_set_attribute(formstruct, KVF_LITERAL, temp))
                              return(FALSE);
		      }
		      if (!kvf_set_attribute(formstruct, attribute, intval))
			return(FALSE);
		      if (return_value) *((int *) return_value) = intval;
		   }
		   break;

	      case KFLOAT:
		   if ((done = ksscanf(temp, "%f", &floatval)) == 1)
		   {
		      if (selection->type == KUIS_FLOAT)
		      {
		          if (!kvf_set_attribute(formstruct, KVF_LITERAL, temp))
                              return(FALSE);
		      }
		      if (!kvf_set_attribute(formstruct, attribute, floatval))
			return(FALSE);
		      if (return_value) *((float *) return_value) = floatval;
		   }
		   break;

	      case KDOUBLE:
		   if ((done = ksscanf(temp, "%lg", &doubleval)) == 1)
		   {
		      if (selection->type == KUIS_DOUBLE)
		      {
		          if (!kvf_set_attribute(formstruct, KVF_LITERAL, temp))
                              return(FALSE);
		      }
		      if (!kvf_set_attribute(formstruct, attribute, doubleval))
			return(FALSE);
		      if (return_value) *((double *) return_value) = doubleval;
		   }
		   break;

	      case KSTRING:
		   if ((selection->type == KUIS_STRING)    ||
		       (selection->type == KUIS_INPUTFILE) ||
		       (selection->type == KUIS_OUTPUTFILE))
		   {
		       if (!kvf_set_attribute(formstruct, KVF_LITERAL, temp))
                           return(FALSE);
		   }
		   if (!kvf_set_attribute(formstruct, attribute, temp))
		      return(FALSE);
		   if (return_value) *((kstring *)return_value) = kstrdup(temp);
		   done = TRUE;
		   break;

	      case KLOGICAL:
		   if (!kstrcasecmp(temp,"yes") ||  !kstrcasecmp(temp,"y") ||
		       !kstrcasecmp(temp,"true") || !kstrcasecmp(temp,"t") ||
		       !kstrcmp(temp,"1"))
		   {
		      done = TRUE;
		      intval = TRUE;
		   }
		   else if (!kstrcasecmp(temp,"no") || !kstrcasecmp(temp,"n") ||
		       !kstrcasecmp(temp, "false") || !kstrcasecmp(temp, "f") ||
		       !kstrcmp(temp,"0"))
		   {
		      done = TRUE;
		      intval = FALSE;
		   }

		   if (done)
		   {
		      if (!kvf_set_attribute(formstruct, attribute, intval))
			return(FALSE);
		      if (return_value) *((int *) return_value) = intval;
		   }
		   break;
	      default:
	           kerror("kclui", "kclui_prompt_value", "Internal Error! \
Unknown data type for interactive prompting.");
		   kfclose(file);
		   return(FALSE);
	   }

	   if (!done)
	   {
	      kerror("kclui", "kclui_prompt_value", "Invalid input \
'%s'.  Variable '%s' must be followed by a %s input.", temp, name,
			kdefine_to_datatype(type));
	   }
	} while (!done);

	/*
	 *  If we've gotten this far, then it means that the presence flag is
	 *  TRUE.  Meaning that the user entered something via interactive
	 *  prompting.
	 */
	kfclose(file);
	if (flag) *flag = TRUE;
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: kclui_prompt_flag - prompt flag argument
|
|       Purpose: This routines is used to get a flag argument
|		 from a form structure.  The flag is pulled from the
|		 form structure, for the given variable name.  If the
|		 variable is not defined failure is returned and
|		 the errno XXXX is set.
|
|         Input: pane - the pane to search for the flag
|                name - the name of the flag
|                prompt_string - the string to prompt the user
|
|        Output: return_flag - the pointer to the flag to be returned
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young & Danielle Argiro
|          Date: Mar 28, 1994
|  Side Effects: 
| Modifications: 
|
------------------------------------------------------------*/

int kclui_prompt_flag(
   kform *pane,
   char  *name,
   char  *prompt_string,
   int   *return_flag)
{
	return(kclui_prompt_value(pane, name, KVF_OPTSEL, prompt_string,
			          KLOGICAL, (kaddr) return_flag, NULL));
}

/*-----------------------------------------------------------
|
|  Routine Name: kclui_prompt_int - prompt int argument
|
|       Purpose: This routines is used to get a int argument
|		 from a form structure.  The int is pulled from the
|		 form structure, for the given variable.  If the
|		 variable is not defined failure is returned and
|		 the errno XXXX is set.
|
|		 The presence flag is used to indicate whether the
|		 argument was acutally specified or the default value
|		 is being passed back.
|
|         Input: pane - the pane to search for the int
|                name - the name of the int
|                prompt_string - the string to prompt the user
|
|        Output: return_int    - the pointer to the int value being returned
|                flag - whether the variable was specified or not
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young & Danielle Argiro
|          Date: Mar 28, 1994
|  Side Effects: 
| Modifications: 
|
------------------------------------------------------------*/

int kclui_prompt_int(
   kform *pane,
   char  *name,
   char  *prompt_string,
   int   *return_int,
   int   *flag)
{
	return(kclui_prompt_value(pane, name, KVF_INT_VAL, prompt_string,
			          KINT, (kaddr) return_int, flag));
}

/*-----------------------------------------------------------
|
|  Routine Name: kclui_prompt_long - prompt long argument
|
|       Purpose: This routines is used to get a long argument
|		 from a form structure.  The long is pulled from the
|		 form structure, for the given variable.  If the
|		 variable is not defined failure is returned and
|		 the errno XXXX is set.
|
|		 The presence flag is used to indicate whether the
|		 argument was acutally specified or the default value
|		 is being passed back.
|
|         Input: pane - the pane to search for the long
|                name - the name of the long
|                prompt_string - the string to prompt the user
|
|        Output: return_long    - the pointer to the long value being returned
|                flag - whether the variable was specified or not
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young & Danielle Argiro
|          Date: Mar 28, 1994
|  Side Effects: 
| Modifications: 
|
------------------------------------------------------------*/

int kclui_prompt_long(
   kform *pane,
   char  *name,
   char  *prompt_string,
   long  *return_long,
   int   *flag)
{
	int intval;

	if (!kclui_prompt_value(pane, name, KVF_INT_VAL, prompt_string, 
			        KINT, (kaddr) &intval, flag))
	   return(FALSE);

	if (return_long)
	   *return_long = intval;

	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: kclui_prompt_short - prompt short argument
|
|       Purpose: This routines is used to get a short argument
|		 from a form structure.  The short is pulled from the
|		 form structure, for the given variable.  If the
|		 variable is not defined failure is returned and
|		 the errno XXXX is set.
|
|		 The presence flag is used to indicate whether the
|		 argument was acutally specified or the default value
|		 is being passed back.
|
|         Input: pane - the pane to search for the short
|                name - the name of the short
|                prompt_string - the string to prompt the user
|
|        Output: return_short    - the pointer to the short value being returned
|                flag - whether the variable was specified or not
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young & Danielle Argiro
|          Date: Mar 28, 1994
|  Side Effects: 
| Modifications: 
|
------------------------------------------------------------*/

int kclui_prompt_short(
   kform *pane,
   char  *name,
   char  *prompt_string,
   short *return_short,
   int   *flag)
{
	int intval;

	if (!kclui_prompt_value(pane, name, KVF_INT_VAL, prompt_string, 
			        KINT, (kaddr) &intval, flag))
	   return(FALSE);

	if (return_short)
	   *return_short = intval;

	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: kclui_prompt_float - prompt float argument
|
|       Purpose: This routines is used to get a float argument
|		 from a form structure.  The float is pulled from the
|		 form structure, for the given variable.  If the
|		 variable is not defined failure is returned and
|		 the errno XXXX is set.
|
|		 The presence flag is used to indicate whether the
|		 argument was acutally specified or the default value
|		 is being passed back.
|
|         Input: pane - the pane to search for the float
|                name - the name of the float
|                prompt_string - the string to prompt the user
|
|        Output: return_float  - the pointer to the float value being returned
|                flag - whether the variable was specified or not
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young & Danielle Argiro
|          Date: Mar 28, 1994
|  Side Effects: 
| Modifications: 
|
------------------------------------------------------------*/

int kclui_prompt_float(
   kform *pane,
   char  *name,
   char  *prompt_string,
   float *return_float,
   int   *flag)
{
	return(kclui_prompt_value(pane, name, KVF_FLOAT_VAL, prompt_string,
			          KFLOAT, (kaddr) return_float, flag));
}

/*-----------------------------------------------------------
|
|  Routine Name: kclui_prompt_double - prompt double argument
|
|       Purpose: This routines is used to get a double argument
|		 from a form structure.  The double is pulled from the
|		 form structure, for the given variable.  If the
|		 variable is not defined failure is returned and
|		 the errno XXXX is set.
|
|		 The presence flag is used to indicate whether the
|		 argument was acutally specified or the default value
|		 is being passed back.
|
|         Input: pane - the pane to search for the double
|                name - the name of the double
|                prompt_string - the string to prompt the user
|
|        Output: return_double - the pointer to the double value being returned
|                flag - whether the variable was specified or not
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young & Danielle Argiro
|          Date: Mar 28, 1994
|  Side Effects: 
| Modifications: 
|
------------------------------------------------------------*/

int kclui_prompt_double(
   kform *pane,
   char   *name,
   char   *prompt_string,
   double *return_double,
   int   *flag)
{
	return(kclui_prompt_value(pane, name, KVF_DOUBLE_VAL, prompt_string,
			KDOUBLE, (kaddr) return_double, flag));
}

/*-----------------------------------------------------------
|
|  Routine Name: kclui_prompt_string - prompt string argument
|
|       Purpose: This routines is used to get a string argument
|		 from a form structure.  The string is pulled from the
|		 form structure, for the given variable.  If the
|		 variable is not defined failure is returned and
|		 the errno XXXX is set.
|
|		 The presence flag is used to indicate whether the
|		 argument was acutally specified or the default value
|		 is being passed back.
|
|         Input: pane - the pane to search for the string
|                name - the name of the string
|                prompt_string - the string to prompt the user
|
|        Output: return_string - the pointer to the string value being returned
|                flag - whether the variable was specified or not
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young & Danielle Argiro
|          Date: Mar 28, 1994
|  Side Effects: 
| Modifications: 
|
------------------------------------------------------------*/

int kclui_prompt_string(
   kform   *pane,
   char    *name,
   char    *prompt_string,
   kstring *return_string,
   int     *flag)
{
	return(kclui_prompt_value(pane, name, KVF_STRING_VAL, prompt_string,
			          KSTRING, (kaddr) return_string, flag));
}

/*-----------------------------------------------------------
|
|  Routine Name: kclui_prompt_infile - prompt infile argument
|
|       Purpose: This routines is used to get a infile argument
|		 from a form structure.  The infile is pulled from the
|		 form structure, for the given variable.  If the
|		 variable is not defined failure is returned and
|		 the errno XXXX is set.
|
|		 The presence flag is used to indicate whether the
|		 argument was acutally specified or the default value
|		 is being passed back.
|
|         Input: pane - the pane to search for the infile
|                name - the name of the infile
|                prompt_string - the string to prompt the user
|
|        Output: return_infile - the pointer to the infile value being returned
|                flag - whether the variable was specified or not
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young & Danielle Argiro
|          Date: Mar 28, 1994
|  Side Effects: 
| Modifications: 
|
------------------------------------------------------------*/

int kclui_prompt_infile(
   kform   *pane,
   char    *name,
   char    *prompt_string,
   kstring *return_infile,
   int     *flag)
{
	return(kclui_prompt_value(pane, name, KVF_FILE_NAME, prompt_string,
			          KSTRING, (kaddr) return_infile, flag));
}

/*-----------------------------------------------------------
|
|  Routine Name: kclui_prompt_outfile - prompt outfile argument
|
|       Purpose: This routines is used to get a outfile argument
|		 from a form structure.  The outfile is pulled from the
|		 form structure, for the given variable.  If the
|		 variable is not defined failure is returned and
|		 the errno XXXX is set.
|
|		 The presence flag is used to indicate whether the
|		 argument was acutally specified or the default value
|		 is being passed back.
|
|         Input: pane - the pane to search for the outfile
|                name - the name of the outfile
|                prompt_string - the string to prompt the user
|
|        Output: return_outfile - the pointer to the outfile value being
|				  returned
|                flag  - whether the variable was specified or not
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young & Danielle Argiro
|          Date: Mar 28, 1994
|  Side Effects: 
| Modifications: 
|
------------------------------------------------------------*/

int kclui_prompt_outfile(
   kform   *pane,
   char    *name,
   char    *prompt_string,
   kstring *return_outfile,
   int     *flag)
{
	return(kclui_prompt_value(pane, name, KVF_FILE_NAME, prompt_string,
			          KSTRING, (kaddr) return_outfile, flag));
}

/*-----------------------------------------------------------
|
|  Routine Name: kclui_prompt_logical - prompt logical argument
|
|       Purpose: This routines is used to get a logical argument
|		 from a form structure.  The logical is pulled from the
|		 form structure, for the given variable.  If the
|		 variable is not defined failure is returned and
|		 the errno XXXX is set.
|
|		 The presence flag is used to indicate whether the
|		 argument was acutally specified or the default value
|		 is being passed back.
|
|         Input: pane - the pane to search for the logical
|                name - the name of the logical
|                prompt_string - the string to prompt the user
|
|        Output: return_logical - the pointer to the logical value being
|				  returned
|                flag  - whether the variable was specified or not
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young & Danielle Argiro
|          Date: Mar 28, 1994
|  Side Effects: 
| Modifications: 
|
------------------------------------------------------------*/

int kclui_prompt_logical(
   kform *pane,
   char  *name,
   char  *prompt_string,
   int   *return_logical,
   int   *flag)
{
	return(kclui_prompt_value(pane, name, KVF_LOGIC_VAL, prompt_string,
			          KLOGICAL, (kaddr) return_logical, flag));
}
