/*
 *----------------------------------------------------------------------
 *
 * Copyright 1991, University of New Mexico.  All rights reserved.
 * Permission to copy and modify this software and its documen-
 * tation only for internal use in your organization is hereby
 * granted, provided that this notice is retained thereon and
 * on all copies.  UNM makes no representations as to the sui-
 * tability and operability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.
 * 
 * UNM DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FIT-
 * NESS.  IN NO EVENT SHALL UNM BE LIABLE FOR ANY SPECIAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY OTHER DAMAGES WHAT-
 * SOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER
 * IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PER-
 * FORMANCE OF THIS SOFTWARE.
 * 
 * No other rights, including, for example, the right to redis-
 * tribute this software and its documentation or the right to
 * prepare derivative works, are granted unless specifically
 * provided in a separate license agreement.
 *---------------------------------------------------------------------
 */

#include "unmcopyright.h"        /* Copyright 1991 by UNM */

/*>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>  <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
 >>>>
 >>>>         File Name: lvringint.c
 >>>>
 >>>>      Program Name: vringint
 >>>>
 >>>> Date Last Updated: Mon Apr 15 22:17:07 1991 
 >>>>
 >>>>          Routines: lvringint - the library call for vringint
 >>>>
 >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>   <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<*/


#include "vinclude.h"


/* -library_includes */
/* -library_includes_end */


/****************************************************************
*
* Routine Name: lvringint - library call for vringint
*
* Purpose:
*    
*    Compute ring integrals
*    
*    
* Input:
*    
*    image          pointer to xvimage structure to be processed
*    
*    x              X coordinate of the point where the radius is zero
*    
*    y              Y coordinate of the point where the radius is zero
*    
*    rmin           radius of the minimum size ring to integrate
*    
*    rmax           radius of the maximum size ring to integrate
*    
*    step           radius step to use in incrementing  from  rmin  to
*                   rmax
*    
*    ang            angular  interval  (step  size)  for   integration
*                   (degrees)
*    
*    type           selectes line integral(0) or ring integral(1) cal-
*                   culation
*    
*    
* Output:
*    
*    image          points to the result image.  The data storage type
*                   is  FLOAT with the number of rows equal to one and
*                   the number of columns equal to 360/ang. Note  that
*                   this  image  overwrites  the  input  image, so the
*                   input image must be saved if it is desired to keep
*                   it for later reference.
*    
*    Return Value:  1 on success, 0 on failure.
*    
*    
*
* Written By: Scott Wilson
*    
*    
****************************************************************/


/* -library_def */
int
lvringint(image,x,y,rmin,rmax,step,ang,type)
struct xvimage *image;
float x,y,rmin,rmax,step,ang;
int type;
/* -library_def_end */

/* -library_code */
  {
    int nc,nr,np;
    float *f,*data,finterp(),cinterp();
    char *malloc();
    float tot,r,theta,t;
    int i,j;
    unsigned char *c;
    float a;

    nc = image->number_of_cols;
    nr = image->number_of_rows;
    np = (rmax-rmin)/step;
    a = 3.1415926*ang/180.0;

    f = (float *)malloc(np*sizeof(float));
    if (f == NULL)
      {
        fprintf(stderr,"lvringint: Unable to allocate new data array\n");
        return(0);
      }

    if (image->data_storage_type == VFF_TYP_FLOAT)
      {
        for (i=0; i<np; i++)
          {
            r = rmin+i*step;
            tot = 0.0;
            for (j=0; j<(int)(360.0/ang); j++)
              {
                theta = j*a;
                t = finterp((float *)(image->imagedata),
                            nr,nc,r,theta,x,y)*r*a;
                if (type != 0) t *= step;
                tot += t;
              }
            f[i] = tot;
          }
      }
    else if (image->data_storage_type == VFF_TYP_1_BYTE)
      {
        for (i=0; i<np; i++)
          {
            r = rmin+i*step;
            tot = 0.0;
            for (j=0; j<(int)(360.0/ang); j++)
              {
                theta = j*a;
                t = cinterp(image->imagedata,
                              nr,nc,r,theta,x,y)*r*a;
                if (type != 0) t *= step;
                tot += t;
              }
            f[i] = tot;
          }
      }
    else
      {
        fprintf(stderr,"lvringint: Unsupported data type\n");
        free(f);
        return(0);
      }

    free(image->imagedata);
    image->imagedata = (char *)f;
    image->data_storage_type = VFF_TYP_FLOAT;
    image->row_size = np;
    image->col_size = 1;
    return(1);
  }

float finterp(f,nr,nc,r,theta,x,y)
float *f,r,theta,x,y;
int nr,nc;
  {
    int x1,x2,x3,x4;
    int y1,y2,y3,y4;
    float z,z1,z2;
    float f1,f2,f3,f4;

    float xloc,yloc;
    double cos(),sin();

    xloc = x+r*cos((double)theta);
    yloc = y+r*sin((double)theta);
 
    x1 = floor(xloc);
    y1 = floor(yloc);
    x2 = x1+1;
    y2 = y1;
    x3 = x1+1;
    y3 = y1+1;
    x4 = x1;
    y4 = y1+1;

    if (x1 < 0 || x1 >= nc-2 || y1 < 0 || y1 >= nr-2) return(0.0);

    f1 = f[y1*nc+x1];
    f2 = f[y2*nc+x2];
    f3 = f[y3*nc+x3];
    f4 = f[y4*nc+x4];

    z1 = f1+(f2-f1)*(xloc-x1);
    z2 = f4+(f3-f4)*(xloc-x1);
    z = z1+(z2-z1)*(yloc-y1);
    return(z);
  }
    
float cinterp(c,nr,nc,r,theta,x,y)
unsigned char *c;
float r,theta,x,y;
int nr,nc;
  {
    int x1,x2,x3,x4;
    int y1,y2,y3,y4;
    float z,z1,z2;
    float f1,f2,f3,f4;

    float xloc,yloc;
    double cos(),sin();

    xloc = x+r*cos((double)theta);
    yloc = y+r*sin((double)theta);
 
    x1 = floor(xloc);
    y1 = floor(yloc);
    x2 = x1+1;
    y2 = y1;
    x3 = x1+1;
    y3 = y1+1;
    x4 = x1;
    y4 = y1+1;

    if (x1 < 0 || x1 >= nc-2 || y1 < 0 || y1 >= nr-2) return(0.0);

    f1 = c[y1*nc+x1];
    f2 = c[y2*nc+x2];
    f3 = c[y3*nc+x3];
    f4 = c[y4*nc+x4];

    z1 = f1+(f2-f1)*(xloc-x1);
    z2 = f4+(f3-f4)*(xloc-x1);
    z = z1+(z2-z1)*(yloc-y1);
    return(z);
  }
/* -library_code_end */
