/* copyright 1987, 1988, 1989 Phil Andrews, Pittsburgh Supercomputing Center */
#include <stdio.h>
#include <ctype.h>
#include "defs.h"
/* macros for generic TEKTRONIX processing */
#define eot	'\004'	/* end of file */
#define ctrl_e	'\005'	/* enquire tty status */
#define ctrl_g	'\007'	/* ring bell */
#define ctrl_h	'\010'	/* back space */
#define lf	'\012'	/* line feed, end of record */
#define ctrl_k	'\013'	/* reverse of line feed */
#define ctrl_l	'\014'	/* esc-ctrl_l clear screen */
#define ctrl_m	'\015'	/* leave graphics mode and do <CR> */
#define ctrl_n	'\016'	/* esc-ctrl_n APL character set */
#define ctrl_o	'\017'	/* esc-ctrl_o ASCII character set */
#define ctrl_w	'\027'	/* make hard copy */
#define ctrl_v	'\026'	/* padding */
#define can	'\030'	/* the can command to end graphics */
#define ctrl_z	'\032'	/* enter GIN mode */
#define esc	'\033'	/* prefix character */
#define g_on	'\035'	/* enter graphics mode */
#define g_off	'\037'	/* leave graphics mode */
#define SUPER_SURFACE -1
#define MAX_RGB 100.4999

#define byte_size 8
/* now the fancier stuff (200 series), these are strings */
#define sel_code	"\033%!"	/* select the tty code */
#define sel_fill 	"\033MP"	/* select the fill pattern */
#define set_cmap	"\033TG"	/* set the colour map */
#define set_cmode	"\033TM"	/* set the colour mode */
#define beg_panel 	"\033LP"	/* begin panel */
#define end_panel 	"\033LE"	/* end panel */
#define g_text	 	"\033LT"	/* write graphics text */
#define l_index	 	"\033ML"	/* sets the line colour index */
#define l_style	 	"\033MV"	/* sets the line style */
#define t_index	 	"\033MT"	/* sets the text colour index */
#define tek_save 	"\033JV"	/* save terminal settings */
#define tek_load 	"\033JL"	/* load terminal settings */
#define tek_syntax 	"\033#!"	/* syntax mode command */
#define rep_set 	"\033IQ"	/* report terminal settings */
#define begin_pixel 	"\033RU"	/* begin pixel operations */
#define set_pixel_view 	"\033RS"	/* set the pixel viewport */
#define set_pixel_beam 	"\033RH"	/* set the pixel beam position */
#define raster_write 	"\033RP"	/* write out the pixels */

#define hi_y 040
#define hi_x 040
#define lo_y 0140
#define lo_x 0100

#define lo_mask 037
#define hi_mask 0140
#define t_shift 5
/* variables global to this module */
static int xmax_tek_vec = 1023;	/* maximum addresses (will also be set externally) */
static int ymax_tek_vec = 767;
static hi_res = 0;		/* do we use extended addressing ? */
static int pxl_per_in;
static char tint_string[4] = "";
static int old_l_colour = -1;
static int old_f_colour = -1;
static int max_colours, bit_planes;		/* maximum no. of colours */
static int first_page;
#define max_str 128
static char tty_buffer[max_str + 1];	/* for replies from the tty */
static int my_inds[3];
static int base_index;

#define tek_com(str1, str2) out_string(str1); out_string(str2);
static int tek_y_max;	/* max y value */
static float cosr, sinr;
#define newx(xin, yin) ((int) (cosr * xin - sinr * yin))
#define newy(xin, yin) ((int) (cosr * yin + sinr * xin))
static int yt1_old, yt2_old, xt1_old;	/* used for compression */

/* store these pointers to allow lookups into the CGM data structures */
static struct one_opt 		*popt;	/* the command line options, in only */
static struct mf_d_struct 	*pc1;	/* the class 1 elements, in only */
static struct pic_d_struct 	*pc2;	/* the class 2 elements, in only */
static struct control_struct	*pc3;	/* the class 3 elements, in only */
static struct attrib_struct	*pc5;	/* the class 5 elements, in only */
static struct info_struct 	*dptr;	/* device info in and out */

static int text_colour = -1;
static int state_level;
static enum dev_type {tek4010, tek4014, tek4107, tek4205, tek4207}
	this_dev = tek4010;
#define check_dev(instr, dev) if (0==strcmp(instr, "dev")) this_dev = dev;
#define series_41 (this_dev == tek4107)
#define series_42 ((this_dev == tek4205) || (this_dev == tek4207))
#define simple_series ((!series_42) && (!series_41))
extern unsigned char *allocate_mem();
/* first general tektronix utility routines */
/* start tekronix graphics format */
start_vec()
{
	outc(g_on);
	return(1);
}
/* end tekronix graphics format */
end_vec()
{
	outc(g_off);
	return(1);
}
/* general colour index routine */
static int get_colour(cptr)
struct rgbi_struct *cptr;
{
int col_index;

	if (pc2->c_s_mode == i_c_mode) {	/* indexed */
	    col_index = (cptr->ind >= max_colours) ? 1 : cptr->ind;
	}
	else if (pc2->c_s_mode == d_c_mode) {	/* direct colour */
	    col_index = int_rgb(cptr->red, cptr->green, cptr->blue, my_inds);
	    col_index += base_index;
	}

	return(col_index);
}
/* write out tektronix graphics here, use compression, assume i/o bound */
put_tek(x1, y1)
int x1, y1;
{
int xin, yin;
char xt1, xt2, yt1, yt2, extra_byte;

	xin = newx(x1, y1);
	yin = newy(x1, y1);

	if (xin < 0) xin = 0;
	if (yin < 0) yin = 0;

	if (xin >= xmax_tek_vec) xin = xmax_tek_vec;
	if (yin >= ymax_tek_vec) yin = ymax_tek_vec;

	if (hi_res)	{
	    extra_byte =  (xin % 4) & ( (yin % 4) << 2);
	    xin >>= 2;
	    yin >>= 2;
	} else extra_byte = 0;

	yt1 = hi_y + (yin / 32);
	yt2 = lo_y + (yin % 32);

	xt1 = hi_x + (xin / 32);
	xt2 = lo_x + (xin % 32);

	if (yt1 != yt1_old) outc(yt1_old = yt1);

	if (extra_byte) outc(lo_y & extra_byte);

	if (xt1 != xt1_old) {
	    outc(yt2_old = yt2);
	    outc(xt1_old = xt1);
	} else if (yt2 != yt2_old) outc(yt2_old = yt2);

	outc(xt2);

	return(1);
}
/* encode an integer into a string for 4200 series ttys, returns ptr */
char *code_int(int_in)
int int_in;
{
int lo_i, hi_i1, hi_i2, no_args, quot1, quot2;

	lo_i = (int_in < 0) ? 32 - int_in % 16 : 48 + int_in % 16;
	if (int_in < 0) int_in = -int_in;
	quot1 = int_in / 16;
	if (!quot1) {
	    tint_string[0] = lo_i;
	    tint_string[1] = 0;
	    return(tint_string);
	}

	hi_i1 = 64 + quot1 % 64;
	quot2 = quot1 / 64;
	if (!quot2) {
	    tint_string[0] = hi_i1;
	    tint_string[1] = lo_i;
	    tint_string[2] = 0;
	    return(tint_string);
	}

	if (quot2 > 63) {
	    fprintf(stderr, "trouble, int_in too large (%d) !\n", int_in);
	    quot2 %= 64;
	}
	hi_i2 = 64 + quot2;
	tint_string[0] = hi_i2;
	tint_string[1] = hi_i1;
	tint_string[2] = lo_i;
	tint_string[3] = 0;
	return(tint_string);
}

/* function to output a line, end points at (x1, y1), (x2, y2) represented
   by nx bits which may be on or off */
#define round(x) ((int) ((x) + 0.499))
void put_line(x1, y1, x2, y2, dat_ptr, nx)
int x1, x2, y1, y2, nx;
char *dat_ptr;
{
static unsigned char bit_on[byte_size] = {128, 64, 32, 16, 8, 4, 2, 1};
int i, pen;
float dx, dy;

	dx = (x2 - x1); dx /= nx;	/* note not (nx - 1) */
	dy = (y2 - y1); dy /= nx;

	start_vec();
	pen = 0;
	for (i = 0; i < nx; ++i) {
	    if ((pen && !(dat_ptr[i / byte_size] & bit_on[i % byte_size])) ||
		(!pen && (dat_ptr[i / byte_size] & bit_on[i % byte_size]))) {

		put_tek(x1 + round(dx * i), y1 + round(dy * i));
		if (pen) start_vec();
		pen = (!pen);
	    }
	}
	/* take care of end of line */
	if (pen) put_tek(x2, y2);
	return;
}
/* now the CGM specific routines */
/* begin a tektronix device */
tek_begin(comment, file_name, prog_name)
char *comment, *file_name, *prog_name;
{
double atan();
int surface_no, i;
	first_page = 1;
	++state_level;
	if (state_level > 0) return(1);
	
	pxl_per_in = dptr->pxl_in + 0.5;
	hi_res = (pxl_per_in == 400);
	if (hi_res) {
	    xmax_tek_vec = 4 * 1024 - 1;
	    ymax_tek_vec = 4 * 768 - 1;
	    dptr->c_height = dptr->c_width = 4 * 12;
	} else {
	    xmax_tek_vec = 1024 - 1;
	    ymax_tek_vec = 768 - 1;
	    dptr->c_height = dptr->c_width = 12;
	}
	yt1_old = yt2_old = xt1_old = -1;
	if (simple_series) return(1);
	if (series_41) {
	    tek_com(sel_code, code_int(0));	/* put it in TEK mode */
	    return(1);
	}
/* now smarter stuff */

	tek_com(tek_syntax, code_int(1));	/* store command mode */
	tek_com(sel_code, code_int(0));	/* put it in TEK mode */
	save_env(state_level);	    	/* save the current settings */
	tek_com(set_cmode, code_int(1));	/* colour mode */
	out_string(code_int(1));
	out_string(code_int(1));

	/* can we talk to the device ? */
	if (!popt[(int) screen].set) return(1);
	/* how many bit planes do we have ? */
	tek_com(rep_set, "RD");
	fb();	/* flush the buffer */
	tty_buffer[0] = '\0';
	if ((NULL == fgets(tty_buffer, max_str, stdin)) || 
	    (2 != sscanf(tty_buffer+2, " %d %d", &surface_no, &bit_planes))) {
	    fprintf(stderr, "trouble getting bit_planes [%s]\n", tty_buffer);
	} else {
	    max_colours = 1;
	    for (i=0; i<bit_planes; ++i) max_colours *= 2;
	    if (popt[(int) debug].set) fprintf(stderr, 
	    "string = %s, surface = %d, bit_planes = %d, max_colours = %d\n",
		tty_buffer, surface_no, bit_planes, max_colours);
	}
	return(1);
}
/* send a string in Hollerith form */
static tek_string(instr)
char *instr;
{
	out_string(code_int(strlen(instr)));
	out_string(instr);
	return(1);
}
/* save terminal environment */
static save_env(env_no)
int env_no;
{
char buffer[10];	/* should be plenty */

	if ((env_no < 0) || (env_no > 9)) {
	    fprintf(stderr, "illegal env_no in save %d\n", env_no);
	    env_no = 0;
	}
	out_string(tek_save);
	tek_string("ENV");			/* environment */
	out_string(code_int(-1));		/* all settings */
	tek_string("TO");
	sprintf(buffer, "M%d:", env_no);
	tek_string(buffer);			/* pseudofile */
	return(1);
}
/* load saved terminal environment */
static load_env(env_no)
int env_no;
{
char c, buffer[15];	/* should be plenty */

	if ((env_no < 0) || (env_no > 9)) {
	    fprintf(stderr, "illegal env_no in load %d\n", env_no);
	    env_no = 0;
	}
	out_string(tek_load);
	sprintf(buffer, "M%d:ENV", env_no);
	tek_string(buffer);
	fb();		/* flush the buffer */
	/* the tty will send back the model number */
	if (popt[(int) screen].set) {
 	    while ((c=getchar()) != lf); 
	}
	return(1);
}
/* delete saved terminal environment */
static delete_env(env_no)
int env_no;
{
char buffer[10];	/* should be plenty */

	if ((env_no < 0) || (env_no > 9)) {
	    fprintf(stderr, "illegal env_no in delete %d\n", env_no);
	    env_no = 0;
	}
	out_string(tek_save);
	tek_string("ENV");			/* environment */
	out_string(code_int(0));		/* delete mode */
	tek_string("TO");
	sprintf(buffer, "M%d:", env_no);
	tek_string(buffer);			/* pseudofile */
	return(1);
}
/* start a page */
tek_bpage(pic_name, xoffset, yoffset, rotation, rb, gb, bb, page_no,
    xsize, ysize)
char *pic_name;
float rotation;
int xoffset, yoffset, page_no;
float rb, gb, bb;	/* background colour values */
int xsize, ysize;
{

	cosr = 1.0;
	sinr = 0.0;
	if (state_level > 0) return(1);
	tek_y_max = yoffset;
	outc(esc);
	outc(ctrl_l);

	return(1);

}
/* start a page */
t2_begin(pic_name, xoffset, yoffset, rotation, rb, gb, bb, page_no,
    xsize, ysize)
char *pic_name;
float rotation;
int xoffset, yoffset, page_no;
float rb, gb, bb;	/* background colour values */
int xsize, ysize;
{
int back_index;

	cosr = 1.0;
	sinr = 0.0;
	if (state_level > 0) return(1);
	tek_y_max = yoffset;
	outc(esc);
	outc(ctrl_l);

	back_index = 0;
	if (pc2->c_s_mode == d_c_mode) {	/* direct colour */
	    base_index = 1;
	    get_cinds(max_colours - base_index, my_inds);
	    load_dcmap(my_inds, rb, gb, bb);
	} else if (pc2->c_s_mode == i_c_mode) {	/* indexed colour */
	    /* fill out the colour map */
	    t2_ctab(0, pc1->max_c_index, pc5->ctab); /* colour tab */
	    base_index = 0;
	}
	if (first_page) {
	    first_page = 0;
	}
	return(1);
}
/* end a page */
tek_epage(no_copies)
int no_copies;
{
	start_vec();		/* move cleanly */
	put_tek(0, 0);		/* go to (0,0) */
	end_vec();		/* out of graphics mode */
	if ((simple_series) && (popt[(int) copies].set)) {
	    outc(esc);
	    outc(ctrl_w);
	}
	fb();			/* flush the buffer */
/*	if (series_42) load_env(state_level + 1); */
	return(1);
}
/* end the metafile */
tek_end()
{
	end_vec(); fb();
	if ((series_42) && (state_level == 0)) { /* restore environment */
	    load_env(state_level);
	    delete_env(state_level);
	    if (!first_page) delete_env(state_level + 1);
	    tek_com(tek_syntax, code_int(2));	/* restore command mode */
	}
	--state_level;
	return(1);
}
/* set text on a tektronix device */
tek_text(x, y, final, buffer)
int x, y, final;
char *buffer;
{
char *c_ptr;
int s_len, new_text_colour;

	start_vec();
	put_tek(x, y);
	end_vec();
	if (!simple_series) {
	    if (text_colour != (new_text_colour = 
		    get_colour(&(pc5->text_colour)))) {
		tek_com(t_index, code_int(text_colour = new_text_colour));
	    }
	    out_string(g_text);
	    s_len = strlen(buffer);
	    out_string(code_int(s_len));
	}
	out_string(buffer);

	return(1);
}
/* do line between a series of points */
tek_pline(no_pairs, x1_ptr, y1_ptr)
int no_pairs, *x1_ptr, *y1_ptr;
{
int i, j, lwpxls, xdelta, ydelta, pdelta, new_l_colour;
double cos(), sin(), theta, atan2(), ctheta, stheta;

	if (no_pairs <= 0) return(1);
	if ((!simple_series) && 
	    ((new_l_colour = get_colour(&pc5->line_colour)) != old_l_colour)) {
	    old_l_colour = new_l_colour;
	    tek_com(l_index, code_int(old_l_colour));
	}

	switch (pc2->l_w_s_mode) {
case absolute: lwpxls = pc5->line_width.i; break;
case scaled: lwpxls = pc5->line_width.r + 0.5; break;
	}
	if (lwpxls < 1) lwpxls = 1;

	/* simple case first */
	if (lwpxls == 1) {	/* single pxl width */
	    start_vec();	/* go to start */
	    put_tek(x1_ptr[0], y1_ptr[0]);
	    for (i=1;i<no_pairs;++i) 
		draw_line(x1_ptr[i-1], y1_ptr[i-1], x1_ptr[i], y1_ptr[i],
		    pc5->line_type);
	    return(1);
	}

/* now multiple line width */
	for (i=1;i<no_pairs;++i) {
	    if (x1_ptr[i] != x1_ptr[i-1]) {
		theta = atan2((double) (y1_ptr[i] - y1_ptr[i-1]),
		    (double)(x1_ptr[i] - x1_ptr[i-1]));
		ctheta = cos(theta);
		stheta = sin(theta);
	    } else {
		ctheta = 0.0;
		stheta = 1.0;
	    }
	    /* loop for the line width */
	    for (j=0; j<lwpxls; ++j) {
		pdelta = j - (lwpxls / 2);
		xdelta = stheta * pdelta;	/* should increase these */
		ydelta = ctheta * pdelta;

		start_vec();	/* go to start */
		put_tek(x1_ptr[i-1] + xdelta, y1_ptr[i-1] + ydelta);
		draw_line(x1_ptr[i-1] + xdelta, y1_ptr[i-1] + ydelta, 
		x1_ptr[i] + xdelta, y1_ptr[i] + ydelta, pc5->line_type);
	    }
	}
	return(1);
}
/* plot a set of lines between alternate points */
tek_dpline(no_pairs, x1_ptr, y1_ptr)
int no_pairs, *x1_ptr, *y1_ptr;
{
int i, new_l_colour;

	if ( no_pairs <= 1 ) return(1);
	if ((!simple_series) && 
	    ((new_l_colour = get_colour(&pc5->line_colour)) != old_l_colour)) {
	    old_l_colour = new_l_colour;
	    tek_com(l_index, code_int(old_l_colour));
	}

	for (i=0; i < no_pairs ; i+=2) {
	    start_vec();
	    put_tek(x1_ptr[i], y1_ptr[i]);
	    draw_line(x1_ptr[i], y1_ptr[i], x1_ptr[i+1], y1_ptr[i+1], 
		pc5->line_type);
	}
	return(1);
}
/* function draws line from (x0, y0) to (x1, y1) assuming already at (x0,y0) */
/* and in graphics mode, looks at line_type to tailor line */
draw_line(x0, y0, x1, y1, line_type)
int x0, y0, x1, y1;
enum line_enum line_type;
{
double sqrt();
int  x2, y2;
float rx, ry, rxd, ryd, rxd1, ryd1, l_length, da_length, do_length;
/* now lengths in inches */
#define DASH_LENGTH 0.1
#define DOT_LENGTH 0.025

	/* take care of the simple case */
	if (((x1==x0) && (y1==y0)) || (line_type == solid_l)) {
	    put_tek(x1, y1); 
	    return(1);
	}
	/* now the more interesting stuff */
	l_length = sqrt((double) ((x1-x0)*(x1-x0) + (y1-y0)*(y1-y0)));
	if (l_length < 1) l_length = 1;
	da_length = DASH_LENGTH * pxl_per_in;
	do_length = DOT_LENGTH * pxl_per_in;

	switch (line_type) {
case dash:
	    rxd = (x1 - x0) * da_length / l_length;
	    ryd = (y1 - y0) * da_length / l_length;
	    x2 = rx = x0 + rxd;
	    y2 = ry = y0 + ryd;
	    while (((x0<=x1)==(x2<=x1)) && ((y0<=y1)==(y2<=y1))) {
		put_tek(x2, y2);
		x2 = rx += rxd;
		y2 = ry += ryd;
		if (!(((x0<=x1)==(x2<=x1)) && ((y0<=y1)==(y2<=y1)))) break;
		start_vec();
		put_tek(x2, y2);
		x2 = rx += rxd;
		y2 = ry += ryd;
	    }
	    start_vec();
	    put_tek(x1, y1);	/* make sure we're at the right point */
	    break;
/* dot */
case dot_l:
	    rxd = (x1 - x0) * do_length / l_length;
	    ryd = (y1 - y0) * do_length / l_length;
	    x2 = rx = x0 + rxd;
	    y2 = ry = y0 + ryd;
	    while (((x0<=x1)==(x2<=x1)) && ((y0<=y1)==(y2<=y1))) {
		put_tek(x2, y2);
		x2 = rx += rxd;
		y2 = ry += ryd;
		if (!(((x0<=x1)==(x2<=x1)) && ((y0<=y1)==(y2<=y1)))) break;
		start_vec();
		put_tek(x2, y2);
		x2 = rx += rxd;
		y2 = ry += ryd;
	    }
	    start_vec();
	    put_tek(x1, y1);	/* make sure we're at the right point */
	    break;
/* dash dot */
case dash_dot:
	    rxd = (x1 - x0) * (da_length + do_length) / l_length;
	    ryd = (y1 - y0) * (da_length + do_length) / l_length;
	    rxd1 = rxd * do_length / (da_length + do_length);
	    ryd1 = ryd * do_length / (da_length + do_length);
	    rxd = rxd * da_length / (da_length + do_length);
	    ryd = ryd * da_length / (da_length + do_length);
	    x2 = rx = x0 + rxd;
	    y2 = ry = y0 + ryd;
	    while (((x0<=x1)==(x2<=x1)) && ((y0<=y1)==(y2<=y1))) {
		put_tek(x2, y2);
		x2 = rx += rxd;
		y2 = ry += ryd;
		if (!(((x0<=x1)==(x2<=x1)) && ((y0<=y1)==(y2<=y1)))) break;
		start_vec();
		put_tek(x2, y2);
		x2 = rx += rxd;
		y2 = ry += ryd;
		put_tek(x2, y2);
		x2 = rx += rxd1;
		y2 = ry += ryd1;
		if (!(((x0<=x1)==(x2<=x1)) && ((y0<=y1)==(y2<=y1)))) break;
		start_vec();
		put_tek(x2, y2);
		x2 = rx += rxd1;
		y2 = ry += ryd1;
	    }
	    start_vec();
	    put_tek(x1, y1);	/* make sure we're at the right point */
	    break;
/* dash dot dot */
case dash_d_d:
	    rxd = (x1 - x0) * (da_length + do_length) / l_length;
	    ryd = (y1 - y0) * (da_length + do_length) / l_length;
	    rxd1 = rxd * do_length / (da_length + do_length);
	    ryd1 = ryd * do_length / (da_length + do_length);
	    rxd = rxd * da_length / (da_length + do_length);
	    ryd = ryd * da_length / (da_length + do_length);
	    x2 = rx = x0 + rxd;
	    y2 = ry = y0 + ryd;
	    while (((x0<=x1)==(x2<=x1)) && ((y0<=y1)==(y2<=y1))) {
		put_tek(x2, y2);
		x2 = rx += rxd;
		y2 = ry += ryd;
		if (!(((x0<=x1)==(x2<=x1)) && ((y0<=y1)==(y2<=y1)))) break;
		start_vec();
		put_tek(x2, y2);
		x2 = rx += rxd;
		y2 = ry += ryd;
		put_tek(x2, y2);
		x2 = rx += rxd1;
		y2 = ry += ryd1;
		if (!(((x0<=x1)==(x2<=x1)) && ((y0<=y1)==(y2<=y1)))) break;
		start_vec();
		put_tek(x2, y2);
		x2 = rx += rxd1;
		y2 = ry += ryd1;
		put_tek(x2, y2);
		x2 = rx += rxd1;
		y2 = ry += ryd1;
		if (!(((x0<=x1)==(x2<=x1)) && ((y0<=y1)==(y2<=y1)))) break;
		start_vec();
		put_tek(x2, y2);
		x2 = rx += rxd1;
		y2 = ry += ryd1;
	    }
	    start_vec();
	    put_tek(x1, y1);	/* make sure we're at the right point */
	    break;
	}

	return(1);
}
#undef DASH_LENGTH
#undef DOT_LENGTH
/* do a polygon, may not be able to fill it */
tek_pgon(no_pairs, x1_ptr, y1_ptr)
int no_pairs, *x1_ptr, *y1_ptr;
{
int i;
	if (no_pairs <= 0) return(1);

	start_vec();
	for (i=0;i<no_pairs;++i) put_tek(x1_ptr[i], y1_ptr[i]);
	put_tek(x1_ptr[0], y1_ptr[0]);

	return(1);
}
/* a (filled) polygon */
t2_pgon(no_pairs, x1_ptr, y1_ptr)
int no_pairs, *x1_ptr, *y1_ptr;
{
int i, want_edge, new_f_colour;

	if (no_pairs <= 0) return(1);

	want_edge = (pc5->edge_vis == on) || (pc5->int_style == hollow);

	switch (pc5->int_style) {
	case hollow :	break;	/* don't try to fill these */
	case empty :	break;
	
	case pattern :		/* no pattern fill yet */
	case hatch :		/* no hatch fill yet */
	case solid_i :			
	    new_f_colour = get_colour(&pc5->fill_colour);
	    if (new_f_colour != old_f_colour) {
	        tek_com(sel_fill, code_int(-new_f_colour));
		old_f_colour = new_f_colour;
	    }
	    out_string(beg_panel);
	    put_tek(x1_ptr[0], y1_ptr[0]);
	    out_string(code_int(1));
	    start_vec();
	    for (i=1;i<no_pairs;++i) put_tek(x1_ptr[i], y1_ptr[i]);
	    end_vec();
	    out_string(end_panel);
	}

	/* may want visible edge */
	if (want_edge){	
	    if (pc5->edge_colour.ind != old_l_colour) {
		old_l_colour = pc5->edge_colour.ind % max_colours;
		tek_com(l_index, code_int(old_l_colour));
	    }
	    start_vec();
	    for (i=0;i<no_pairs;++i) put_tek(x1_ptr[i], y1_ptr[i]);
	    put_tek(x1_ptr[0], y1_ptr[0]);
	}

	return(1);
}

/* do a rectangle, assume it should be filled, try to make it look nice */
#define direction(a, b) ((b > a) ? 1 : (b == a) ? 0 : -1)
tek_rectangle(x1, y1, x2, y2)
int x1, x2, y1, y2;
{
int stepx, stepy;

	start_vec();
	stepx = direction(x1, x2);
	stepy = direction(y1, y2);

	/* first ensure a nice clean outer edge if not a degenerate case */
	if ((x1 != x2) && (y1 != y2)) {
	    put_tek(x1, y1);
	    put_tek(x1, y2);
	    put_tek(x2, y2);
	    put_tek(x2, y1);
	    put_tek(x1, y1);
	    x1 += stepx;
	    x2 -= stepx;
	    y2 -= stepy;	/* get ready to spiral */
	    if ((x1 != x2) && (y1 != y2)) put_tek(x1, y1);
	}

	/* now fill it up */
	while ((direction(x1, x2) == stepx) && (direction(y1, y2) == stepy)){ 
	    /* take care of degenerate cases */
	    /* single point */
	    if ((x1 == x2) && (y1 == y2)) {
	        put_tek(x1, y1);
	        return(1);
	    }
	    /* straight line */
	    if ((x1 == x2) || (y1 == y2)) {
	    	put_tek(x1, y1);
	    	put_tek(x2, y2);
	    	return(1);
	    }
	    /* spiral fill */
	    put_tek(x1, y2);
	    put_tek(x2, y2);
	    put_tek(x2, y1 += stepy);
	    put_tek(x1 += stepx, y1);
	    x2 -= stepx;
	    y2 -= stepy;
	}
	return(1);
}
/* set marker at a series of points */
tek_pmarker(no_pairs, x1_ptr, y1_ptr)
int no_pairs, *x1_ptr, *y1_ptr;
{
int i, x_adjust = 0, y_adjust = 0, new_text_colour;
	if (no_pairs <= 0) return(1);

	if (!simple_series) {
	    if (text_colour != (new_text_colour = 
		    get_colour(&pc5->mk_colour))) {
		tek_com(t_index, code_int(text_colour = new_text_colour));
	    }
	}

	switch (pc5->mk_type) {
case 1 : x_adjust = - 0.25 * dptr->c_height ;
		y_adjust = - 0.7 * dptr->c_height ;
case 2 : x_adjust = - 0.3 * dptr->c_height ;
		y_adjust = -0.3 * dptr->c_height; break;
case 3 : x_adjust = - 0.25 * dptr->c_height ;
		y_adjust = -0.45 * dptr->c_height; break;
case 4 : x_adjust = - 0.25 * dptr->c_height ;
		y_adjust = -0.3 * dptr->c_height; break;
case 5 : x_adjust = - 0.35 * dptr->c_height ;
		y_adjust = -0.3 * dptr->c_height; break;
default: fprintf(stderr, "illegal marker type ! = %d\n", pc5->mk_type);
	   x_adjust = - dptr->c_height / 2; y_adjust = - dptr->c_height / 2; 
}

	for (i=0;i<no_pairs;++i) {
	    start_vec();
	    put_tek(x1_ptr[i] + x_adjust, y1_ptr[i] + y_adjust);
	    end_vec();
	    if (series_42) {
	    	out_string(g_text);
		out_string(code_int(1));
	    }
	    outc(c_mark[pc5->mk_type]);
	}
	return(1);
}
/* for cell arrays */
tek_carray(cp, cq, cr, nx, ny, col_prec, dat_ptr, rep_mode, no_bytes)
int *cp;	/* first corner */
int *cq;	/* the diagonal to cp p*/
int *cr;	/* first row is from cp to cr */
int nx;		/* number of elements in x rows (i.e., parallel to cp-cr) */
int ny;		/* number of rows */
int col_prec;	/* the colour precision (how many possible colour values */
char *dat_ptr;	/* pointer to the body of information */
int rep_mode;	/* the representation mode; 0, run length encoded, 1 normal */
int no_bytes;	/* number of data bytes */
/* without compression there should be ny * row_size bytes after dat_ptr, 
   the first nx * c_size bits of each row should contain information */
{
enum cs_enum c_s_mode;	/* the colour selection mode, indexed or direct */
char *new_ptr = NULL, *bit_ptr = NULL;
float *grey_ptr;
int row_size;	/* how many bytes in a row of data (allows for rounding) */
int c_size;	/* how many bits to specify one colour value */
float dx, dy, stepx, stepy;
int nsteps, x1, x2, y1, y2, i, row_no, new_size;
void put_line();

	c_s_mode = pc2->c_s_mode;	/* start with this, may change */
	switch (c_s_mode) {
case d_c_mode:	c_size = 3 * col_prec; break;
case i_c_mode:	c_size = col_prec; break;
default:	fprintf(stderr, "illegal colour selection mode ! (%d)\n",
		    (int) c_s_mode); break;
	}
	/* now get the row size in bytes allowing for rounding */
	row_size = (nx * c_size + byte_size - 1) / byte_size;	/* check */
	row_size = (row_size % 2) ? row_size + 1 : row_size; 	/* round up */

	/* expand a compressed raster */
	if (rep_mode == 0) {
	    new_ptr = (char *) allocate_mem(ny * row_size, 1);
	    if (!new_ptr) return(2);
	    (void) c_expand(dat_ptr, nx, ny, row_size, c_size, new_ptr);
	    dat_ptr = new_ptr;	/* carry on as before */
	}

/* take care of problems */

	if ((col_prec != 1) || (c_s_mode != i_c_mode)){	/* need to condense */
	    new_size = (nx + byte_size - 1) / byte_size;	/* check */
	    new_size = (new_size % 2) ? new_size + 1 : new_size; 

	    bit_ptr =  (char *) allocate_mem(ny * new_size, 1);
	    if (!bit_ptr) return(2);

	    bit_array(dat_ptr, nx, ny, row_size, bit_ptr, c_s_mode, col_prec,
	    pc5->ctab, new_size);

	    col_prec = 1;		/* single bits now */
	    c_s_mode = i_c_mode;	/* indexed mode */
	    dat_ptr = bit_ptr;	
	    row_size = new_size;
	    c_size = 1;			/* carry on as before */
	}
/* now confident we have packed list single precision cell array */
/* but we may have to expand/shrink/rotate/skew */

	/* we'll move the second point towards Q one pixel at a time in the
	   the slowest direction (to try to get a nice outline) */


	dx = cq[0] - cr[0];
	dy = cq[1] - cr[1];
	if (dx * dx > dy * dy) {
	    stepx = (dx > 0) ? 1.0 : -1.0;
	    stepy = dy / dx ;
	    nsteps = abs(cq[0] - cr[0]) + 1;
	} else {
	    stepx = (dy) ? dx / dy : dx;
	    stepy = (dy) ? ((dy > 0) ? 1.0 : -1.0) : 0;
	    nsteps = abs(cq[1] - cr[1]) + 1;
	}
	start_vec();
	for (i = 0; i < nsteps; ++i) {
	    x1 = cp[0] + (i * stepx);
	    x2 = cr[0] + (i * stepx);
	    y1 = cp[1] + (i * stepy);
	    y2 = cr[1] + (i * stepy);
	    row_no = (i * ny) / nsteps;
	    put_line(x1, y1, x2, y2, dat_ptr + row_no * row_size, nx);
	}
	if (new_ptr) free(new_ptr);
	if (bit_ptr) free(bit_ptr);
	return(1);
}
/* handle a bit array here */
/* three corners are at p, q, r, may be a parallelogram */
tek_barray(p, q, r, nx, ny, bit_ptr)
int p[2], q[2], r[2], nx, ny;
char *bit_ptr;
{
int height, width, offset, i, j, k, l, no_vecs, x0, y0, pen;
float xmag, ymag, ctheta;
double fabs();
int varray[4096];	/* storage for vectors */
	width  = q[0] - p[0];
	height = q[1] - p[1];

	if ((!nx) || (!ny) || (!width) || (!height)) return(1);

	xmag = width;  xmag /= nx;
	ymag = height; ymag /= ny;

	offset = q[0] - r[0];
	ctheta = offset; ctheta /= height;

	if ( (fabs(xmag) < 1) || (fabs(ymag) < 1)) {	/* can't handle it easily */
	    start_vec();
	    put_tek(p[0], p[1]);
	    put_tek(p[0] + offset, q[1]);
	    put_tek(q[0], q[1]);
	    put_tek(r[0], r[1]);
	    put_tek(p[0], p[1]);
	    return(1);
	}

/* do a simple algorithm to do the cell array row by row */

	y0 = p[1];
	for (i = 0; i < ny; ++i){
	    /* first store the vectors necesary to reproduce the row */
	    k = 0;
	    pen = 0;
	    for (l=0; l<nx; ++l) {
		if (*(bit_ptr++) ^ pen) {
		    varray[k++] = l * xmag;
		    pen = (!pen);
		}
	    }
	    /* is the pen down ? */
	    if (pen) varray[k++] = l * xmag -1;
	    no_vecs = k;
	    /* now spit out the vectors */
	    for (j=0; j<ymag; ++j) {
		x0 = p[0] + ctheta * (y0 - p[1]);
		for (k=1; k<no_vecs; k+=2) {
		    start_vec();
		    put_tek(x0 + varray[k-1], y0);
		    put_tek(x0 + varray[k], y0);
		}
		++y0;
	    }
	}
	return(1);
}
/* for cell arrays */
t2_carray(cp, cq, cr, nx, ny, col_prec, dat_ptr, rep_mode, no_bytes)
int *cp;	/* first corner */
int *cq;	/* the diagonal to cp p*/
int *cr;	/* first row is from cp to cr */
int nx;		/* number of elements in x rows (i.e., parallel to cp-cr) */
int ny;		/* number of rows */
int col_prec;	/* the colour precision (how many possible colour values */
unsigned char *dat_ptr;	/* pointer to the body of information */
int rep_mode;	/* the representation mode; 0, run length encoded, 1 normal */
int no_bytes;	/* number of data bytes */
{
unsigned char *new_ptr = NULL, *start_ptr;
unsigned int new_bytes, i, j, row_size, bdone, c_size, new_size;
unsigned int x1, x2, y1, y2, c_index;
int out_flag;
int my_flag = 0;
int want_rmode = 1;
int want_cp;
int want_csmode = 0;
int new_nx, new_ny, no_bits;
int bytes_needed, bits_row, bytes_row, first_index, no_entries;
int px, py, qx, qy, rx, ry;	/* the local co-ord system */
#define lctab_size 2
static float lctab[3 * lctab_size] = {1.0, 1.0, 1.0, 0.0, 0.0, 0.0};
#define ALU_MODE 11
#define MAX_ENTRIES 10000
#define MAX_XPXL 640
#define MAX_YPXL 480

	/* pixel are in a different coordinate system */
	px = (cp[0] * MAX_XPXL) / (pxl_per_in * dptr->x_size);
	py = (cq[1] * MAX_YPXL) / (pxl_per_in * dptr->y_size);
	qx = (cr[0] * MAX_XPXL) / (pxl_per_in * dptr->x_size);;
	qy = (cr[1] * MAX_YPXL) / (pxl_per_in * dptr->y_size);;
	rx = (cq[0] * MAX_XPXL) / (pxl_per_in * dptr->x_size);;
	ry = (cp[1] * MAX_YPXL) / (pxl_per_in * dptr->y_size);;
	want_cp = 4;
	first_index =  (pc2->c_s_mode == d_c_mode) ? 1 : 0;
	/* the rows and columns we need */
	new_nx = (rx > px) ? 1 + rx - px : 1 + px - rx;
	new_ny = (qy > py) ? 1 + qy - py : 1 + py - qy;
	
	bits_row = new_nx * want_cp;
	bytes_row = (bits_row + byte_size - 1) / byte_size;
	bytes_needed = bytes_row * new_ny;

	if (!(new_ptr = (unsigned char *) allocate_mem(bytes_needed, 1)))
	    return(2);

/* basically only need the bottom left hand corner, nx and ny */
	x1 = (px < rx) ? px : rx;
	y1 = (py < qy) ? py : qy;
	x2 = x1 + new_nx - 1;	/* for safety */
	y2 = y1 + new_ny - 1;
	/* lets make the cell array fit */
	new_carray(NULL, nx, ny, col_prec, rep_mode, 
	    0, dat_ptr, no_bytes, pc2->c_s_mode,
	    pc5->ctab, pc1->max_c_index, px, py, qx, qy,
	    rx, ry, new_nx, new_ny,
	    want_cp, want_rmode, bits_row, new_ptr, bytes_needed, 
	    want_csmode, lctab, lctab_size, my_flag, &out_flag, 
	    first_index, my_inds, &(pc1->c_v_extent));
	/* at this point we are sure of a normal cell array in cgm format; */

	/* get the terminal ready */
	no_bits = 6;
	tek_com(begin_pixel, code_int(SUPER_SURFACE));
	out_string(code_int(ALU_MODE));
	out_string(code_int(no_bits));
	out_string(set_pixel_view);
	/* only works in hi_res mode (Tektronix bug) */
	if (x1 < x2) ++x2;	/* needs more room (Tektronix bug) */
	else ++x1;
	put_tek(x1, y1);
	put_tek(x2 + 1 , y2);

	/* now ready to write the pixels */
	start_ptr = new_ptr;		/* mark our place */
	no_entries = new_nx + 1;	/* one for the line terminator */
	for (i=0; i<new_ny; ++i) {	/* one row at a time */
	    tek_com(raster_write, code_int(new_nx));
	    out_string(code_int(no_entries));
	    for (j=0; j<new_nx; ++j) {
		c_index = (j % 2) ? (new_ptr[j / 2] & 15) + ' ' : 
		    ((new_ptr[j / 2] >> 4) & 15) + ' ';
		outc(c_index);
	    }
	    outc(96);	/* terminate the line */
	    new_ptr += bytes_row;
	}

	if (start_ptr) (void) free(start_ptr);
	return(1);
}

/* colour map for terminals that can handle it */
t2_ctab(beg_index, no_entries, ctab)
int beg_index, 	/* beginning index */
no_entries; 	/* number of entries to add starting at beg_index */
float *ctab;	/* direct colour array, *(ctab + i*3) is the red
		   entry for the i'th index, followed by g and b */
{
int ret = 1, i, j;
float rmag;

	if ((beg_index > max_colours) || (beg_index < 0)) return(2);

	if ((beg_index + no_entries) > max_colours) {
	    no_entries = max_colours - beg_index;
	    ret = 2;
	}
	tek_com(set_cmap, code_int(SUPER_SURFACE));
	out_string(code_int(no_entries * 4));
	for (i=beg_index; i<(beg_index+no_entries); ++i) {
	    out_string(code_int(i));
	    for (j=0; j<3; ++j) {
		rmag = ctab[i*3 + j];
		if (rmag < 0.0) {
		    fprintf(stderr, "illegal rgb value (%d,%d) %f\n",
			i, j, rmag);
		    rmag = 0.0;
		}
		if (rmag > 1.0) {
		    fprintf(stderr, "illegal rgb value (%d,%d) %f\n",
			i, j, rmag);
		    rmag = 1.0;
		}
		out_string(code_int((int) (MAX_RGB * rmag)));
	    }
	}
	return(ret);
}
/* load the DC colour map */
static load_dcmap(ninds, rb, gb, bb)
int *ninds;
float rb, gb, bb;
{
int i, j, k, no_entries, c_index, ind1;
#define no_colours 3
float *col_tab[no_colours], top0, top1, top2;

	no_entries = ninds[0] * ninds[1] * ninds[2] + 1;
	ind1 = base_index;

	/* establish bounds */
	top0 = (ninds[0] > 1) ? ninds[0] - 1.0 : 1.0;
	top1 = (ninds[1] > 1) ? ninds[1] - 1.0 : 1.0;
	top2 = (ninds[2] > 1) ? ninds[2] - 1.0 : 1.0;

	/* prepare the terminal */
	tek_com(set_cmap, code_int(SUPER_SURFACE));
	out_string(code_int(no_entries * 4));
	/* first set the background colour */
	c_index = 0;
	out_string(code_int(c_index));
	out_string(code_int((int) (MAX_RGB * rb)));
	out_string(code_int((int) (MAX_RGB * gb)));
	out_string(code_int((int) (MAX_RGB * bb)));
	/* now move over the info */
	for (i=0; i<ninds[2]; ++i) {
	    for (j=0; j<ninds[1]; ++j) {
		for (k=0; k<ninds[0]; ++k) {
		    c_index = ind_ijk(ninds, i, j, k);
		    out_string(code_int(c_index));
		    out_string(code_int((int) (MAX_RGB * k /top0)));
		    out_string(code_int((int) (MAX_RGB * j /top1)));
		    out_string(code_int((int) (MAX_RGB * i /top2)));
		}
	    }
	}
	return(1);
}
#undef SUPER_SURFACE
#undef MAX_RGB
/* this is the routine that sets everything up */
/* the initialising routine for the postscript module */
void tek_setup(opt, dev_info, c1, c2, c3, c5, delim, mfdesc, pdesc, mfctrl,
	gprim, attr, escfun, extfun, ctrl, pargc, argv)

struct one_opt 		*opt;	/* the command line options, in only */
struct info_struct *dev_info;	/* device info to fill out, out only */
struct mf_d_struct 	*c1;	/* the class 1 elements, in only */
struct pic_d_struct 	*c2;	/* the class 2 elements, in only */
struct control_struct	*c3;	/* the class 3 elements, in only */
struct attrib_struct	*c5;	/* the class 5 elements, in only */
int (*delim[])();     		/* delimiter functions */
int (*mfdesc[])();     		/* metafile descriptor functions */
int (*pdesc[])();      		/* page descriptor functions */
int (*mfctrl[])();     		/* control functions */
int (*gprim[])();      		/* graphical primitives */
int (*attr[])();       		/* the attribute functions */
int (*escfun[])();     		/* the escape functions */
int (*extfun[])();     		/* the external functions */
int (*ctrl[])();       		/* external controller functions */
int *pargc;			/* pointer to argc */
char **argv;			/* the main argv */	
{
int i;
#define dev_length 20
char dev_name[dev_length + 1], buffer[dev_length + 1];

	/* store the command line argument pointer */
	popt = opt;
	/* store the device info pointer */
	dptr = dev_info;
	/* fill out the device info structure */
	dev_info->pxl_in 	= 100.0;
	dev_info->ypxl_in 	= 100.0;
	dev_info->x_size 	= 10.23;
	dev_info->y_size 	= 7.67;
	dev_info->x_offset	= 0.0;
	dev_info->y_offset	= 0.0;
	dev_info->c_height	= 12;
	dev_info->c_width	= 12;
	dev_info->d_l_width	= 1;				
	dev_info->d_e_width	= 1;				
	dev_info->d_m_size	= dev_info->c_height;	/* marker size */
	strcpy(dev_info->out_name, ".tek");
	dev_info->capability	= 	no_cr 
					/* + char_text + string_text */;
	dev_info->rec_size	= 80;

	/* store the CGM data structure pointers */
	pc1 = c1;
	pc2 = c2;
	pc3 = c3;
	pc5 = c5;

	/* now fill out the function pointer arrays for CGM */
	/* the delimiter functions */
	delim[(int) B_Mf] 	= tek_begin;
	delim[(int) E_Mf]	= tek_end;
	delim[(int) B_Pic_Body]	= tek_bpage;
	delim[(int) E_Pic]	= tek_epage;

	/* the default graphical primitives */
	gprim[(int) PolyLine]	= tek_pline;
	gprim[(int) Dis_Poly]	= tek_dpline;
	gprim[(int) PolyMarker]	= tek_pmarker;
	gprim[(int) Text]	= tek_text;
	gprim[(int) Polygon]	= tek_pgon;
	gprim[(int) Cell_Array]	= tek_carray;
	gprim[(int) Rectangle]	= tek_rectangle;

	/* lets see exactly what device we're working for */
	/* now see if we asked for a device */
	if (opt[(int) device].set) 
	    strncpy(dev_name, opt[(int) device].val.str, dev_length);
	else if (opt[(int) screen].set) 
	    strncpy(dev_name, opt[(int) screen].val.str, dev_length);
	dev_name[dev_length - 1] = 0;	/* safety */

	/* for compatibility with earlier versions: */
	if (dev_name[0] == '4') {	/* old tektronix form */
	    strcpy(buffer, dev_name);
	    sprintf(dev_name, "tek%s", buffer);
	}

	/* and convert to lower case */
	for (i=0; i<strlen(dev_name); ++i)
	    dev_name[i] = (isupper(dev_name[i])) ? tolower(dev_name[i]):
		 dev_name[i];

	/* figure out what device we are on */
	check_dev(dev_name,tek4010);
	check_dev(dev_name,tek4014);
	check_dev(dev_name,tek4107);
	check_dev(dev_name,tek4205);
	check_dev(dev_name,tek4207);

	/* may override some of the defaults */
	if (series_42) {
	    /* delimiters */
	    delim[(int) B_Pic_Body]	= t2_begin; 
	    /* primitives */
	    gprim[(int) Polygon]	= t2_pgon;
	    /* the attributes */
	    attr[(int) ColTab] 	= t2_ctab;
	    gprim[(int) Cell_Array]	= t2_carray;
	    max_colours = 16;	/* for a start */
	} else if (series_41) {
	    /* delimiters */
	    delim[(int) B_Pic]	= t2_begin; 
	    /* primitives */
	    gprim[(int) Polygon]	= t2_pgon;
	    /* the attributes */
	    attr[(int) ColTab] 	= t2_ctab;
	    gprim[(int) Cell_Array]	= t2_carray;
	    max_colours = 16;
	}
	if (simple_series) {
	    dev_info->capability |= no_colour;
	} else {
	    dev_info->pxl_in 	= 400.0;
	    dev_info->ypxl_in 	= 400.0;
	}
	state_level = -1;

}
