/* printer.c */
/* Copyright 1988 John M. Sullivan.  See main program for details. */
/**** functions to print data structures ****/

/***** External functions
void fprint_face(fp,f),fprint_wing(fp,f),fprint_fe(fp,f):
    three ways to print a facet_edge
void fprint_corner(fp,c), fprint_site(fp,s):
    print out these objects for debugging
void fprint_grads(fp,s): print out all gradients for site s
*****/
/* Note: header file defines print_...(x) as fprint_...(stdout,x) */

/***** local functions *****/
static void fprint_loc(),fprint_vloc();

#include "vcs.h"

static void
fprint_loc(fp,x)
file fp;
site x;
{
    if (x)
	fprint_point(fp,x->loc);
    else
	fprint_point(fp,NULL);
}

static void
fprint_vloc(fp,x)
file fp;
corner x;
{
    if (x)
    {
	if (infinite(x)) putc('~',fp);
	fprint_point(fp,x->loc);
    }
    else fprint_point(fp,NULL);
}

void
fprint_corner(fp,c)
file fp;
corner c;
{
    int i;

    fprint_vloc(fp,c); fputs(" between ",fp);
    for (i=0; i<4; i++)
	fprint_loc(fp,c->cell[i]); fputs((i<3)? ", " : "\n",fp);
}

void
fprint_site(fp,s)
file fp;
site s;
{
    fprint_loc(fp,s);
    fprintf(fp,", color %d, color-volume %.3lf\n",s->color,s->volume);
}

void
fprint_grads(fp,s)
file fp;
site s;
{
    clr k;
    char c = '\t';

    fprint_site(fp,s);
    if (s->color>0)
    {
	fprintf(fp,"Volume gradients:\n");
	for_colors(k)
	{
	    register point p = vol_grad(k,s);
	    if ((p->x != 0.) || (p->y != 0.) || (p->z != 0.))
	    {
		fprintf(fp,"%d: ",k); fprint_point(fp,p); putc(c,fp);
		c = '\t' + '\n' - c;
	    }
	}
	if (c=='\n')
	    putc(c,fp);
	fprintf(fp,"area gradient:\t");
	fprint_point(fp,area_grad(s)); fprintf(fp,"\n\n");
    }
}

void
fprint_wing(fp,w)
file fp;
facet_edge w;
{
    fprint_loc(fp,w->cell_2_); putc('|',fp);
    fprint_fe(fp,w);
}

void
fprint_fe(fp,f)
file fp;
facet_edge f;
{
    fprint_loc(fp,f->cell[0]); putc('|',fp);
    fprint_loc(fp,f->cell[1]); putc('\t',fp);
    fprint_vloc(fp,f->vert[0]); fputs("->",fp);
    fprint_vloc(fp,f->vert[1]);
    DBG(3, fprintf(fp," %x",(int)f));
    putc('\n',fp);
}

void
fprint_face(fp,f)
file fp;
facet_edge f;
{
    facet_edge fstart = f;

    fprint_loc(fp,f->cell[0]); putc('|',fp);
    fprint_loc(fp,f->cell[1]); fputs(":\n",fp);

    for_edges(f,e)
	if (infinite(e->vert[0]))
	    fstart = e;
    end_edges(f,e)

    for_edges(fstart,e)
	fprint_vloc(fp,e->vert[0]); fputs("->",fp);
    end_edges(fstart,e)

    fprint_vloc(fp,fstart->prevedge->vert[1]); putc('\n',fp);

    if (f->fi)
    {
	fputs("\twith unit normal ",fp); fprint_point(fp,unit_normal(f));
	fprintf(fp,", area %lf, height %lf\n",f->fi->area,f->fi->height);
    }
}

void
fprint_volumes(fp)
file fp;
{
    clr k;

    fputs("Volumes:\n",fp);
    for_colors(k)
	fprintf(fp,"%d: %.3lf%c", k, Volumes[k],(k&3? '\t':'\n'));
    fputs(fp,"\n");
}

void
fprint_volume_summary(fp)
file fp;
{
    clr k;
    real sx = 0., sx2 = 0., mn = HUGE, mx = 0.;
    int s1 = 0;
    real ave;

    for_colors(k)
    {
	s1++;
	sx += Volumes[k];
	sx2 += Volumes[k]*Volumes[k];
	mx = max(Volumes[k],mx);
	mn = min(Volumes[k],mn);
    }
    ave = sx/s1;
    fprintf(fp,"Total boundary area is %.4lf\n",Total_Area);
    if (Wulff_Count)
	fprintf(fp,"Total Wulff energy is %.4lf\n",Total_Energy);
    if (Flag_volume)
	return;
    fprintf(fp,"Volume information: %d regions, total volume %.2f\n",s1,sx);
    fprintf(fp,"\tAverage %.3f, std. dev. %.5f, ", ave, sqrt(sx2/s1-ave*ave));
    fprintf(fp,"Range %.3f to %.3f\n", mn, mx);
    if (Flag_kelvin || Max_color == 1)
	fprintf(fp,"Area^3/Volume^2 for average region: %f\n",
		Total_Area*Total_Area*Total_Area/(sx*sx*s1));
    if (Max_color < 5)
	fprint_volumes(fp);
}

void
eprint_face(fe)
facet_edge fe;
{
    fprint_face(stderr,fe);
}

site
third_edge_cell(fe)
facet_edge fe;
{
    site s,poss[4];
    int i,j,k=0;
    for (i=0;i<2;i++)
	for (j=0;j<4;j++)
	    if ((s = fe->vert[i]->cell[j]) != fe->cell[0] && s != fe->cell[1])
		poss[k++] = s;

    for(i=0;i<4;i++)
	for(j=0;j<4;j++)
	    if (i!=j && poss[i]==poss[j])
		return poss[i];
}

void
fprint_num(fp,s)
file fp;
site s;
{
    fprintf(fp,"%d ",s-Sites);
}

static file Fp;

void
print_jean_face(fe,s0)
facet_edge fe;
site s0;
{
    facet_edge e;
    site t;
    int d;

    t = other_cell(fe,s0); print_num(t);
    d = dot(unit_normal(fe),s0->loc) > dot(unit_normal(fe),t->loc);
    for_edges(fe,e)
	fprint_num(Fp,third_edge_cell(e));
    end_edges_dir(fe,e,d)
    fprintf(Fp,"\n");
}

void
write_jean(fp)
file fp;
{

    Fp = fp;
    trace_faces_of_cell(Sites,print_jean_face);
    trace_faces_of_cell(Sites,no_op);
}


static int NF, NV, NE;
static corner *Verts;

/* Clears seq numbers in preparation for number_vf() below */
static void
clear_vf(e)		/* trace_all_edges(clear_vf) */
facet_edge e;
{
    NE++;
    e->vert[0]->seq_num = 0;
    e->vert[1]->seq_num = 0;
    e->fi->seq_num = 0;
}

/* Assign seq numbers to all corners & faces; count corners, faces, edges */
static void
number_faces(f)		/* trace_all_faces(number_faces) */
facet_edge f;
{
    clr k0, k1;

    if(f->fi->seq_num == 0) {
	if((k0 = f->cell[0]->color) == (k1 = f->cell[1]->color)
		|| f->fi->area == HUGE)
	    return;
	if (Flag_torus && (k0 <= 0 && k1 <= 0 || Torus_Same && k0 == -k1))
	    return;
	f->fi->seq_num = ++NF;
    }
}

static void
number_verts(f)
facet_edge f;
{
    if(f->fi->seq_num != 0) {
	for_edges(f, e)
	    if(e->vert[0]->seq_num == 0)
		e->vert[0]->seq_num = ++NV;
	end_edges(f, e)
    }
}

/* Build array of vertices */
static void
enumerate_v(f)		/* trace_all_faces(enumerate_v) */
facet_edge f;
{
    register facet_edge e;

    for_edges(f, e)
	int seq = e->vert[0]->seq_num;
	if(seq <= 0 || seq > NV) {
	    if(seq != -1 && seq != 0)
		fprintf(stderr,
			"write_pixar: enumerate_v: vert %x seq = %d; NV = %d\n",
		    e, seq, NV);
	    continue;
	}
	Verts[seq-1] = e->vert[0];
    end_edges(f, e)
}

static void
write_off_face(f)		/* trace_all_facets(write_off_face) */
facet_edge f;
{
    register facet_edge e;
    int nfv;
    clr k0, k1, mn, mx;

    if(f->fi->seq_num != 0) {
	/* We've already decided (in number_faces) that this face will be drawn */
	nfv = 0;
	for_edges(f, e)
	    nfv++;
	end_edges(f, e)
	fprintf(Fp, "%d\t", nfv);
	for_edges(f, e)
	    fprintf(Fp, "%d ", e->vert[0]->seq_num-1);
	end_edges(f, e)

	k0 = f->cell[0]->color; if(k0 < 0) k0 = 0;
	k1 = f->cell[1]->color; if(k1 < 0) k1 = 0;
	mx = max(k0,k1);  mn = k0+k1-mx;
	fprintf(Fp, "\t%d\n", mx*(mx-1)/2 + mn);
    }
}

void
write_pixar(fp)
file fp;
{
    int i;

    Fp = fp;
    NV = NF = NE = 0;
    trace_all_edges(clear_vf);
    trace_all_faces(number_faces);
    trace_all_faces(number_verts);

    fprintf(Fp,"OFF\n");
    fprintf(Fp, "%d %d %d\n", NV, NF, NE);
    Verts = (corner *) malloc(NV * sizeof(corner));
    trace_all_faces(enumerate_v);
    for(i=0; i<NV; i++) {
	point p = Verts[i]->loc;
	if (Flag_torus)
	    fprintf(Fp,"%g %g %g\n",p->x-.5*Torus->x,
		    p->y-.5*Torus->y,p->z-.5*Torus->z);
	else
	    fprintf(Fp,"%g %g %g\n",p->x,p->y,p->z);
    }
    fputc('\n', Fp);
    trace_all_faces(write_off_face);
    free(Verts);
    fflush(Fp);
}

static void
number_ve(e)
facet_edge e;
{
    static E = 0;
    static V = 0;
    int j;

    if (e->cell[0]->color <= 0 && e->cell[1]->color <= 0 && e->cell_2_->color <= 0)
	return;
    if (e->cell[0]->color == e->cell[1]->color
     && e->cell[0]->color == e->cell_2_->color)
	return;
    e->seq_num = ++E;
    for (j=0; j<2; j++)
	if (!e->vert[j]->seq_num)
	{
	    e->vert[j]->seq_num = ++V;
	    fprintf(Fp,"%d\t%f\t%f\t%f\n",V,
		    e->vert[j]->loc->x,e->vert[j]->loc->y,e->vert[j]->loc->z);
	}
    e->nextfacet->seq_num = (e->nextfacet->vert[0] == e->vert[0])? E : -E;
    e->prevfacet->seq_num = (e->prevfacet->vert[0] == e->vert[0])? E : -E;
}

static void
print_eken(e)
facet_edge e;
{
    if (e->seq_num)
	fprintf(Fp,"%d\t%d\t%d\n",e->seq_num,
		e->vert[0]->seq_num,e->vert[1]->seq_num);
}

static void
print_feken(e)
facet_edge e;
{
    facet_edge starte;
    bool d;
    struct point a[1];

    if (!e->seq_num)
	return;
    fprintf(Fp,"%d\t",e->seq_num);

    starte = e;
    for_facets(e,f)
	if (!f->fi->seq_num)
	    starte = f;
    end_facets(e,f)

    area_x(e->cell[0]->loc,e->cell[1]->loc,e->cell_2_->loc, a);
    d = dot_q(e->vert[0]->loc,a) < dot_q(e->vert[1]->loc,a);

    for_facets(starte,f)
	unless (f->fi->seq_num)
	    continue;
	if (f->vert[0] != e->vert[0])
	    putc('-',Fp);
	fprintf(Fp,"%d\t",f->fi->seq_num);
    end_facets_dir(starte,f,d)
    fprintf(Fp,"\n");
}

static void
number_f(f)
facet_edge f;
{
    static F=0;

    if (f->cell[0]->color <= 0 && f->cell[1]->color <= 0)
	return;
    if (f->cell[0]->color == f->cell[1]->color)
	return;
    f->fi->seq_num = ++F;
    fprintf(Fp,"%d\t",F);
    for_edges(f,e)
	fprintf(Fp,"%d\t",e->seq_num);
    end_edges(f,e)
    fprintf(Fp,"\n");
}

static clr K;

static void
print_bken(f)
facet_edge f;
{
    int i;
    site s,t;

    if (f->cell[0]->color == f->cell[1]->color)
	return;
    for (i=0; i<2; i++)
	if (f->cell[i]->color == K)
	    break;
    if (i==2)
	return;
    s = f->cell[i]; t = f->cell[1-i];
    if (dot(unit_normal(f),s->loc) > dot(unit_normal(f),t->loc))
	putc('-',Fp);
    fprintf(Fp,"%d\t",f->fi->seq_num);
}

write_ken(fp)
file fp;
{
    int k=1;

    Fp = fp;
    if (Flag_torus)
	fprintf(fp,"XTORUS\nperiods\n%f 0 0\n0 %f 0\n0 0 %f\n",
		Torus->x,Torus->y,Torus->z);
    fprintf(fp,"vertices\n");
    trace_all_edges(number_ve);
    fprintf(fp,"\nedges\n");
    trace_all_edges(print_eken);
    fprintf(fp,"\nfaces\n");
    trace_all_faces(number_f);
    fprintf(fp,"\nfacet-edges\n");
    trace_all_edges(print_feken);
    fprintf(fp,"\nbodies\n");
    for_colors(K)
    {
	fprintf(fp,"%d\t",k++);
	trace_all_faces(print_bken);
	fprintf(fp,"volume %f\n",Volumes[K]);
    }
}

static void
number_tetra(e)
facet_edge e;
{
    static V = 0;
    int j;
    corner c;

    for (j=0; j<2; j++)
    {
	if (!(c=e->vert[j])->seq_num && !infinite(c))
	    c->seq_num = ++V;
    }
}

#define vnum(c,i) ((c)->cell[i]-Sites+1)
#define tnum(c) (abs((c)->seq_num))
static site Vert;

static void
facing_tet(fe,c)
corner c;
facet_edge fe;
{
    int j = (fe->vert[0]==c? 1 : 0);
    corner c1 = fe->vert[j];

    for (j=0; j<4; j++)
	if (c1->cell[j]==Vert)
	    return;
    
    fprintf(Fp,"%d ",tnum(c1));
}

static void
pat_tetra(e)
facet_edge e;
{
    int j;

    for (j=0; j<2; j++)
	if (e->vert[j]->seq_num > 0)
	{
	    int i;
	    corner c = e->vert[j];

	    fprintf(Fp,"%d %d %d %d ",vnum(c,0),vnum(c,1),vnum(c,2),vnum(c,3));
	    for (i=0; i<4; i++)
	    {
		Vert = c->cell[i];
		trace_4wings_of_vertex(c,facing_tet);
	    }
	    fprintf(Fp,"\n");
	    c->seq_num = - c->seq_num;
	}
}

write_pat(fp)
file fp;
{
    site s;
    Fp = fp;
    fprintf(fp,"%d\n",N_Sites);
    for_sites(s)
    {
	fprintf(fp,"%.3f %.3f %.3f %d\n",
	    s->loc->x,s->loc->y,s->loc->z,s->color);
    }
    fprintf(fp,"%d\n",N_DTet);
    trace_all_edges(number_tetra);
    trace_all_edges(pat_tetra);
}
