/* $Id$
 *
 * graphics.c: graphics procedures
 */

/**************************************************************************
 *     Copyright (C) 1990 by Mark B. Phillips and Robert R. Miner	  *
 * 									  *
 * Permission to use, copy, modify, and distribute this software, its	  *
 * documentation, and any images it generates for any purpose and without *
 * fee is hereby granted, provided that					  *
 * 									  *
 * (1) the above copyright notice appear in all copies and that both that *
 *     copyright notice and this permission notice appear in supporting	  *
 *     documentation, and that the names of Mark B.  Phillips, Robert R.  *
 *     Miner, or the University of Maryland not be used in advertising or *
 *     publicity pertaining to distribution of the software without	  *
 *     specific, written prior permission.				  *
 *									  *
 * (2) Explicit written credit be given to the authors Mark B.  Phillips  *
 *     and Robert R. Miner in any publication which uses part or all of	  *
 *     any image produced by this software.				  *
 *									  *
 * This software is provided "as is" without express or implied warranty. *
 **************************************************************************/

#include <math.h>
#include "wdefs.h"
#include "window.h"
#include "internal.h"

/* Arc resolution.  Arcs are drawn with line segments this many
 * pixels long, usually. */
#define CANVAS_ARCRES		10
#define MINI_CANVAS_ARCRES	6

/* Minimum number of line segments to use when drawing arcs: */
#define CANVAS_MIN_ARCDIV	12
#define MINI_CANVAS_MIN_ARCDIV	12

/* Maximum number of line segments to use when drawing arcs: */
#define CANVAS_MAX_ARCDIV	80
#define MINI_CANVAS_MAX_ARCDIV	60

#define CANVAS_MAX_AINC		(2*M_PI/CANVAS_MIN_ARCDIV)
#define CANVAS_MIN_AINC		(2*M_PI/CANVAS_MAX_ARCDIV)
#define MINI_CANVAS_MAX_AINC	(2*M_PI/MINI_CANVAS_MIN_ARCDIV)
#define MINI_CANVAS_MIN_AINC	(2*M_PI/MINI_CANVAS_MAX_ARCDIV)

/*
 * The following lists (canvas_point_list and mini_canvas_point_list)
 * specify how points are drawn in the canvas and mini_canvas.  These
 * lists represent pairs of points in relative pixel coordinates.  The
 * routines wPoint and wMCPoint work by drawing this array of points,
 * relative to the coordintes of the point to be plotted.
 */
static int	point_list_default[] = {0,0, 1,0, -1,0, 0,1, 0,-1};
static int	*canvas_point_list = point_list_default;
static int	canvas_point_list_count =
  sizeof(point_list_default) / (2 * sizeof(int));
static int	*mini_canvas_point_list = point_list_default;
static int	mini_canvas_point_list_count =
  sizeof(point_list_default) / (2 * sizeof(int));

static int	canvas_batching=0;
static int	mini_canvas_batching=0;

/*
 * The following define the viewport coordinate system for the canvas.
 * This is the coordinate system in which the graphics procedures work.
 * (wVPX,wVPY) is the lower left corner, and wVPS is the side length.
 * These variables are set by calls to wSetViewport. The initial values
 * given here are meaningless; they are given to prevent errors from
 * occurring during window initialization, before wSetViewport has been
 * called for the first time.
 */
double wVPX=0,wVPY=0,wVPS=1, wCXF=1, wCYF=1;

/*
 * And the analogous things for the mini_canvas:
 */
double wMCVPX=0, wMCVPY=0, wMCVPS=1, wMCXF=1, wMCYF=1;

/*
 * NOTE: There are two sets of drawing procedures, one for each canvas.
 * They are exactly analogous, the only difference being the different
 * values of the various canvas parameters.  The code which implements
 * the procedures is in the file "gprocs.c"; this code depends on various
 * macros to specify these parameters.  This file #includes gprocs.c
 * twice --- once with the macros set for the main canvas, and once with
 * them set for the mini-canvas.
 */

/*-----------------------------------------------------------------------
 * Function:	wPoint, wMCPoint
 * Description:	draws a point in the canvas (wPoint) or
 *		   mini_canvas (wMCPoint)
 * Args:	(x,y) coords of point to draw
 * Author:	mbp
 * Date:	Wed Apr  4 21:16:04 1990
 * Returns:	success status
 */

/*-----------------------------------------------------------------------
 * Function:	wSegment, wMCSegment
 * Description:	Draw a line segment from (x1,y1) to (x2,y2) in the
 *		  canvas (wSegment) or mini_canvas (wMCSegment).
 * Args:	x1,y1,x2,y2
 * Returns:	success status
 */

/*-----------------------------------------------------------------------
 * Function:	wArc, wMCArc
 * Description:	Draw an arc in the canvas (wArc) or mini_canvas (wMCArc)
 * Args  IN:	(x,y): center of arc
 *		r: radius of arc
 *		ang1: initial angle in radians
 *		ang2: terminal angle in radians
 * Returns:	success status
 * Author:	mbp
 * Date:	Wed Apr  4 20:42:46 1990
 * Notes:	Angle is drawn counterclockwise from ang1 to ang2.
 */

/*-----------------------------------------------------------------------
 * Function:	wErase, wMCErase
 * Description:	Erase the canvas (wErase) or mini_canvas (wMCErase)
 * Args:	(none)
 * Returns:	success
 */

/*-----------------------------------------------------------------------
 * Function:	wCanvasBatch, wMCanvasBatch
 * Description:	turn batching on or off in the canvas (wCanvasBatch) or
 *		  mini_canvas (wMCanvasBatch)
 * Args  IN:	onoff: 1 to turn on, 0 to turn off
 * Returns:	nothing
 * Author:	mbp
 * Date:	Sat Apr 14 15:50:06 1990
 * Notes:	These are low-level procedures which turn batching
 *		on or off.  They translate directly to sunview procedure
 *		calls.  The procedure wBatch (see below) is the
 *		public batching procedure available to the hcore.
 */

#define	CLASS			extern
#define	PW			wCanvas_pw
#define WIDTH			canvas_width
#define HEIGHT			canvas_height
#define	X_COORD			CANVAS_X_COORD
#define	Y_COORD			CANVAS_Y_COORD
#define	DIMEN			CANVAS_DIMEN
#define	ARCRES			CANVAS_ARCRES
#define	AINC_MIN		CANVAS_MIN_AINC
#define	AINC_MAX		CANVAS_MAX_AINC
#define	POINT_LIST		canvas_point_list
#define	POINT_LIST_COUNT	canvas_point_list_count
#define	BATCHING		canvas_batching
#define	ARCPROC			wArc
#define POINTPROC		wPoint
#define SEGMENTPROC		wSegment
#define ERASEPROC		wErase
#define BATCHPROC		wCanvasBatch

#include "gprocs.c"

#undef	CLASS
#undef	PW
#undef	WIDTH
#undef	HEIGHT
#undef	X_COORD
#undef	Y_COORD
#undef	DIMEN
#undef	ARCRES
#undef	AINC_MIN
#undef	AINC_MAX
#undef	POINT_LIST
#undef	POINT_LIST_COUNT
#undef	BATCHING
#undef	ARCPROC
#undef	POINTPROC	
#undef	SEGMENTPROC	
#undef	ERASEPROC	
#undef	BATCHPROC

#define	CLASS			extern
#define	PW			wMCanvas_pw
#define WIDTH			mini_canvas_width
#define HEIGHT			mini_canvas_height
#define	X_COORD			MINI_CANVAS_X_COORD
#define	Y_COORD			MINI_CANVAS_Y_COORD
#define	DIMEN			MINI_CANVAS_DIMEN
#define	ARCRES			MINI_CANVAS_ARCRES
#define	AINC_MIN		MINI_CANVAS_MIN_AINC
#define	AINC_MAX		MINI_CANVAS_MAX_AINC
#define	POINT_LIST		mini_canvas_point_list
#define	POINT_LIST_COUNT	mini_canvas_point_list_count
#define	BATCHING		mini_canvas_batching
#define	ARCPROC			wMCArc
#define POINTPROC		wMCPoint
#define SEGMENTPROC		wMCSegment
#define ERASEPROC		wMCErase
#define BATCHPROC		wMCanvasBatch

#include "gprocs.c"

/*-----------------------------------------------------------------------
 * Function:	wBatch
 * Description:	signal the beginning or end of a sequence of
 *		  graphics operations.
 * Args  IN:	op: 1 means beginning, 0 means end
 * Returns:	nothing
 * Author:	mbp
 * Date:	Mon Apr 16 12:20:20 1990
 * Notes:	
 */
int
  wBatch(op)
int op;
{

  if ( (wDrawingMode == BATCH) && (op == 1) )
    wCanvasBatch(1);
  else
    wCanvasBatch(0);
}

int
  wSetCanvasViewport(x, y, s)
double x,y,s;
{
  if (s <= 0) return(-1);

  wVPX = x;
  wVPY = y;
  wVPS = s;
  wComputeCanvasFactors();
  return(0);
}

int
  wSetMiniCanvasViewport(x, y, s)
double x,y,s;
{
  if (s <= 0) return(-1);

  wMCVPX = x;
  wMCVPY = y;
  wMCVPS = s;
  wComputeMiniCanvasFactors();
  return(0);
}

int
  wComputeCanvasFactors()
{
  wCXF = canvas_width / wVPS;
  wCYF = - canvas_height / wVPS;
}

int
  wComputeMiniCanvasFactors()
{
  wMCXF = mini_canvas_width / wMCVPS;
  wMCYF = - mini_canvas_height / wMCVPS;
}
