/* $Id$
 *
 *  postscript.c
 *  
 *  PostScript generation procedures
 *
 */

/**************************************************************************
 *     Copyright (C) 1990 by Mark B. Phillips and Robert R. Miner	  *
 * 									  *
 * Permission to use, copy, modify, and distribute this software, its	  *
 * documentation, and any images it generates for any purpose and without *
 * fee is hereby granted, provided that					  *
 * 									  *
 * (1) the above copyright notice appear in all copies and that both that *
 *     copyright notice and this permission notice appear in supporting	  *
 *     documentation, and that the names of Mark B.  Phillips, Robert R.  *
 *     Miner, or the University of Maryland not be used in advertising or *
 *     publicity pertaining to distribution of the software without	  *
 *     specific, written prior permission.				  *
 *									  *
 * (2) Explicit written credit be given to the authors Mark B.  Phillips  *
 *     and Robert R. Miner in any publication which uses part or all of	  *
 *     any image produced by this software.				  *
 *									  *
 * This software is provided "as is" without express or implied warranty. *
 **************************************************************************/

/*
 * Notes on how this file works:
 *
 * PSOutput is the interface to the outside world.  It takes a function
 * argument and stores it in the static global variable (*psoutfunc) for
 * use by other procedures in this file.  (*psoutfunc) is called many
 * times in generating the the PostScript code for the current picture
 * --- once for each line of code.
 *
 * PSOutput first outputs some PostScript header code which defines the
 * current coordinate system; this uses the external viewport vars
 * vpx,vpy, and vps.  The bulk of the header code comes from the file
 * 'psheader.h', which contains one huge string ('psheader.h' is created
 * by the Makefile from the regular text file 'header.h').
 *
 * To generate the code for the picture itself, PSOutput calls
 * PSDrawPicture, which in turn calls the other PS drawing procedures in
 * this file.
 */

#include "hcore.h"
#include "hglobals.h"

static int	(*psoutfunc)() = NULL;
static char	obuf[100];

#define	RADTODEG(x)	((x)*180/M_PI)

/*-----------------------------------------------------------------------
 * Function:	PSOutput
 * Description:	generate a PostScript file of the current picture
 * Args  IN:	outfunc: function to call with PostScript output strings
 * Author:	mbp
 * Date:	Fri May 11 11:05:05 1990
 * Returns:	success
 * Notes:	This procedure generates a PostScript file and passes
 *		it to outfunc as a sequence of null-terminated strings.
 *		outfunc syntax should be:
 *		  int outfunc(s)
 *		  char *s;
 *		This should process the string s by sending it to a
 *		printer or a file.  In general outfunc will be called
 *		many times in the course of generating a single file.
 */
PSOutput(outfunc)
     int (*outfunc)();
{
  if (outfunc == NULL) return(-1);
  psoutfunc = outfunc;

  (*psoutfunc)("%!\n");
  sprintf(obuf, "/VPX %f def\n", vpx);
  (*psoutfunc)(obuf);
  sprintf(obuf, "/VPY %f def\n", vpy);
  (*psoutfunc)(obuf);
  sprintf(obuf, "/VPS %f def\n", vps);
  (*psoutfunc)(obuf);
  (*psoutfunc)(
#include "psheader.h"
);
  PSDrawPicture();
  (*psoutfunc)("\nshowpage\n");

  return(0);
}
  
/*-----------------------------------------------------------------------
 * Function:	PSPoint
 * Description:	Issue PostScript code to draw a point
 * Args  IN:	x,y: coords of point to draw
 * Returns:	nothing
 * Author:	mbp
 * Date:	Sat May 12 11:21:47 1990
 * Notes:	Uses global proc (*psoutfunc) to output PostScript
 *		string
 */
static int
  PSPoint(x, y)
double x,y;
{
  sprintf(obuf, "%f %f p\n", x, y);
  (*psoutfunc)(obuf);
}

/*-----------------------------------------------------------------------
 * Function:	PSSegment
 * Description:	Issue PostScript code to draw a line segment
 * Args  IN:	x1,y1,x2,y2: coords of endpoints of segment
 * Returns:	nothing
 * Author:	mbp
 * Date:	Sat May 12 11:21:47 1990
 * Notes:	Uses global proc (*psoutfunc) to output PostScript
 *		string
 */
static int
  PSSegment(x1, y1, x2, y2)
double x1,y1,x2,y2;
{
  sprintf(obuf, "%f %f %f %f s\n", x1, y1, x2, y2);
  (*psoutfunc)(obuf);
}

/*-----------------------------------------------------------------------
 * Function:	PSPoint
 * Description:	Issue PostScript code to draw an arc
 * Args  IN:	x,y: coords of center of arc
 *		r: radius of arc
 *		a1,a2: initial and terminal angle of arc, in radians
 * Returns:	nothing
 * Author:	mbp
 * Date:	Sat May 12 11:21:47 1990
 * Notes:	Uses global proc (*psoutfunc) to output PostScript
 *		string
 */
static int
  PSArc(x, y, r, a1, a2)
double x, y, r, a1, a2;
{
  /* PostScript arc command uses degrees, so we change from radians */
  sprintf(obuf, "%f %f %f %f %f a\n", x, y, r, RADTODEG(a1), RADTODEG(a2));
  (*psoutfunc)(obuf);
}

/*-----------------------------------------------------------------------
 * Function:	PSDrawPoint
 * Description:	draws a point in the current model
 * Args  IN:	p: a kPoint
 * Returns:	nothing
 * Author:	mbp
 * Date:	Fri May 11 11:04:33 1990
 * Notes:	Uses PS procedures above to do the actual drawing
 */

/*-----------------------------------------------------------------------
 * Function:	PSDrawSegment
 * Description:	draws a geodesic segment in the current model
 * Args  IN:	*s: a kSegment in Klein coordinates
 * Returns:	nothing
 * Author:	mbp
 * Date:	Fri May 11 11:04:54 1990
 * Notes:	Uses PS procedures above to do the actual drawing
 */

/*-----------------------------------------------------------------------
 * Function:	PSDrawEntity
 * Description:	draw an entity
 * Args  IN:	entity: the entity to be drawn
 * Returns:	success status
 * Author:	mbp
 * Date:	Fri May 11 11:04:56 1990
 * Notes:	Uses PS procedures above to do the actual drawing
 */

#define CLASS		static
#define DRAWPOINT	PSDrawPoint
#define DRAWSEGMENT	PSDrawSegment
#define DRAWENTITY	PSDrawEntity
#define DEVPOINT	PSPoint
#define DEVSEGMENT	PSSegment
#define DEVARC		PSArc

#include "drawprocs.c"

#undef CLASS
#undef DRAWPOINT
#undef DRAWSEGMENT
#undef DRAWENTITY
#undef DEVPOINT
#undef DEVSEGMENT
#undef DEVARC

/*-----------------------------------------------------------------------
 * Function:    PSDrawPicture
 * Description: Erase the canvas and draw the entire picture
 * Arguments:   (none)
 * Returns:     success status
 * Author:	mbp
 * Date:	Mon Apr 16 11:58:20 1990
 * Notes:       Uses PS procedures above to do the actual drawing
 */
static int
  PSDrawPicture()
{
  gdb_Entity entity;
  gdb_Sequence_spec spec;

  /*
   * First draw the circle at infinity (which is the x axis in UHP):
   */
  switch (model) {
  default:
  case KLEIN:
  case POINCARE:
    PSArc(0.0, 0.0, 1.0, 0.0, TWO_PI);
    break;
  case UHP:
    PSSegment(vpx, 0.0, vpx+vps, 0.0);
    break;
  }

  /*
   * Loop through the database, drawing each object
   */
  spec = GDB_FIRST;
  do {
    gdb_retrieve_entity( POINT|SEGMENT, GDB_SEQUENTIAL,
			 (char*)&spec, &entity );
    if (entity!=NULL) {
      PSDrawEntity( entity );
    }
  } while (entity!=NULL);


  return(0);
}
