/* $Id: separate.c,v 1.3 90/07/11 13:08:40 mbp Exp Locker: mbp $
 *
 * separate.c: program to weed nearby points out of an array
 *   in the hyperbolic plane
 */

/**************************************************************************
 *     Copyright (C) 1990 by Mark B. Phillips and Robert R. Miner	  *
 * 									  *
 * Permission to use, copy, modify, and distribute this software, its	  *
 * documentation, and any images it generates for any purpose and without *
 * fee is hereby granted, provided that					  *
 * 									  *
 * (1) the above copyright notice appear in all copies and that both that *
 *     copyright notice and this permission notice appear in supporting	  *
 *     documentation, and that the names of Mark B.  Phillips, Robert R.  *
 *     Miner, or the University of Maryland not be used in advertising or *
 *     publicity pertaining to distribution of the software without	  *
 *     specific, written prior permission.				  *
 *									  *
 * (2) Explicit written credit be given to the authors Mark B.  Phillips  *
 *     and Robert R. Miner in any publication which uses part or all of	  *
 *     any image produced by this software.				  *
 *									  *
 * This software is provided "as is" without express or implied warranty. *
 **************************************************************************/

/*
 * usage: separate [-epsilon <epsilon>]
 *
 * This program reads a list of points from stdin and outputs on stdout
 * a list of indices into the input list which represent an epsilon-
 * separated sublist.  (A list of points is epsilon-separated if the
 * distance between every pair of points in the list is at least
 * epsilon.)  The points are taken to lie in the projective disk (unit
 * disk) model of the hyperbolic plane, and the hyperbolic distance
 * function is used.
 *
 * The input list should be formatted as a Mathematica list of pairs of
 * numbers, i.e. like:
 *
 * 	{ {x1,y1}, {x1,y1}, {x1,y1}, ... {xn,yn} }
 *
 * The output is a Mathematica list of integers, such as
 *
 * 	{ 1, 3, 4, 7, 8, 11, 12 }
 *
 * which would indicate that the epsilon-separated sublist consists of
 * entries 1, 3, 4, 7, 8, 11, and 12 from the input list.
 *
 * If no value of epsilon is specified, 0.02 is assumed.
 *
 * This program is intended to be called by the RunThrough Mathematica
 * function.  This code is implemented in C because the corresponding
 * program written in Mathematica runs about 100 times slower.
 */

#include <stdio.h>
#include <math.h>
#include <malloc.h>

#define NEXTARG --argc; ++argv

double epsilon=0.02;
double xnew,ynew;
int inew;
double *xsaved=NULL, *ysaved=NULL;
int *isaved=NULL;
int saved_count=0;

#define SAVEBLOCKSIZE 100

int c;				/* current char from input stream */
double number;			/* current number read from stdin */

#define NextChar()		c=getchar()

#define DIGITCASES \
    case '0':  case '1':  case '2':  case '3':  case '4': \
    case '5':  case '6':  case '7':  case '8':  case '9': \
    case '-':  case '.':  case 'e':  case 'E':  case '+'  \

double ReadNumber();

#define NUMBERBUFLEN	30

/*
 * input tokens:
 */
#define LEFTBRACE	1
#define RIGHTBRACE	2
#define COMMA		3
#define NUMBER		4
#define ENDFILE		5
#define UNKNOWN		-1

#define INITIALSTATE	0
#define FINALSTATE	100


double factor, cosh1, cosh2;

main(argc, argv)
     int argc;
     char **argv;
{
  if ( ! ParseArgs(argc, argv) ) {
    /* error */
    fprintf(stderr, "WARNING: error in parsing arguments\n");
  }

  cosh1 = cosh(epsilon);
  cosh2 = cosh1*cosh1;
  
  NextChar();

  if (ParseInput())
    PrintOutput();
  else {
    /* error */
    printf("Null");
  }
}

ParseInput()
{
  int state, success;

  state = INITIALSTATE;
  
  while (state != FINALSTATE) {
    switch (state) {
      
    case /* state */ INITIALSTATE:
      switch (NextToken()) {
      case LEFTBRACE:
	state = 1;
	break;
      default:
	success = 0;
	state = FINALSTATE;
	break;
      }
      break;	  /* end state INITIALSTATE */
      
    case /* state */ 1:
      switch (NextToken()) {
      case LEFTBRACE:
	state = 2;
	break;
      default:
	success = 0;
	state = FINALSTATE;
	break;
      }
      break;	  /* end state 1 */

    case /* state */ 2:
      switch (NextToken()) {
      case NUMBER:
	xnew = number;
	state = 3;
	break;
      default:
	success = 0;
	state = FINALSTATE;
	break;
      }
      break;	  /* end state 2 */

    case /* state */ 3:
      switch (NextToken()) {
      case COMMA:
	state = 4;
	break;
      default:
	success = 0;
	state = FINALSTATE;
	break;
      }
      break;	  /* end state 3 */

    case /* state */ 4:
      switch (NextToken()) {
      case NUMBER:
	ynew = number;
	state = 5;
	break;
      default:
	success = 0;
	state = FINALSTATE;
	break;
      }
      break;	  /* end state 4 */

    case /* state */ 5:
      switch (NextToken()) {
      case RIGHTBRACE:
	ProcesNewPoint();
	state = 6;
	break;
      default:
	success = 0;
	state = FINALSTATE;
	break;
      }
      break;	  /* end state 5 */

    case /* state */ 6:
      switch (NextToken()) {
      case COMMA:
	state = 1;
	break;
      case RIGHTBRACE:
	state = 7;
	break;
      default:
	success = 0;
	state = FINALSTATE;
	break;
      }
      break;	  /* end state 6 */

    case /* state */ 7:
      switch (NextToken()) {
      case ENDFILE:
	success = 1;
	state = FINALSTATE;
	break;
      default:
	success = 0;
	state = FINALSTATE;
	break;
      }
      break;	  /* end state 7 */

    case /* state */ FINALSTATE:
	/* should never get here! */
      break;	  /* end state FINALSTATE */
    }
  }

  return(success);
}

ParseArgs(argc, argv)
     int argc;
     char *argv[];
{
  NEXTARG;
  while (argc) {

    /* -epsilon <eps> */
    if (strcmp(&(argv[0][0]), "-epsilon") == 0) {
      if (argc < 2) return(0);
      epsilon = atof(argv[1]);
      NEXTARG;
    }

    /* default (invalid argument): */
    else return(0);

    NEXTARG;
  }
  return(1);
}

/*-----------------------------------------------------------------------
 * Function:	NextToken
 * Description:	return the next token from the input stream (stdin)
 * Args:	(none)
 * Returns:	the token (one of the token constants defined above)
 * Author:	mbp
 * Date:	Thu May  3 10:12:12 1990
 * Notes:	Assumes that the global int c contains the next char
 *		to be read from stdin.
 */
NextToken()
{
  while (1) {
    switch (c) {
    case '{':
      NextChar();
      return(LEFTBRACE);
    case '}':
      NextChar();
      return(RIGHTBRACE);
    case ',':
      NextChar();
      return(COMMA);
    DIGITCASES:
      number=ReadNumber();
      return(NUMBER);
    case ' ':  case '\t': case '\n':
      /* whitespace; do nothing */
      NextChar();
      break;
    case EOF:
      return(ENDFILE);
    default:
      NextChar();
      return(UNKNOWN);
    }
  }
}

/*-----------------------------------------------------------------------
 * Function:	ReadNumber
 * Description:	read a number from stdin, returning its value
 * Args:	(none)
 * Returns:	the value of the number read, as a double
 * Author:	mbp
 * Date:	Thu May  3 10:12:12 1990
 * Notes:	Assumes that the global int c contains the next char
 *		from stdin.  Upon return c will contain the first char
 *		from stdin which is not part of the number read.
 * BUGS:	This procedure stores the number being read in an
 *		internal char buffer of length NUMBERBUFLEN.  If the
 *		actual number of digit chars read in is larger than
 *		this, it will overwrite the buffer, clobbering the stack.
 */
double
  ReadNumber()
{
  char buf[NUMBERBUFLEN+1], *bufptr;

  bufptr = buf;
  while (LookingAtDigit()) {
    *bufptr = c;
    ++bufptr;
    NextChar();
  }
  *bufptr = '\0';
  return(atof(buf));
}

/*-----------------------------------------------------------------------
 * Function:	LookingAtDigit
 * Description:	return 1 if c is a digit char, 0 otherwise
 * Args:	(none)
 * Returns:	1 or 0
 * Author:	mbp
 * Date:	Thu May  3 10:16:10 1990
 * Notes:	
 */
LookingAtDigit()
{
  switch (c) {
  DIGITCASES:
    return(1);
  default:
    return(0);
  }
}


/*-----------------------------------------------------------------------
 * Function:	ProcesNewPoint
 * Description:	process a newly read point
 * Args:	(none)
 * Returns:	nothing
 * Author:	mbp
 * Date:	Thu May  3 10:48:16 1990
 * Notes:	The new point is stored in the global double xnew, ynew.
 */
ProcesNewPoint()
{
  ++inew;
  if (CheckNewPoint())
    SaveNewPoint();
}

/*-----------------------------------------------------------------------
 * Function:	CheckNewPoint
 * Description:	check a new point to see if it should be added to the
 *		  saved list
 * Args:	(none)
 * Returns:	1 to indicate it should be saved, 0 means don't save
 * Author:	mbp
 * Date:	Thu May  3 12:01:18 1990
 * Notes:	The new point is in the global doubles xnew,ynew.
 */
CheckNewPoint()
{
  register double *x = xsaved;
  register double *y = ysaved;
  register int i;

  factor = cosh2 * (xnew*xnew + ynew*ynew - 1.0);
  for (x=xsaved,y=ysaved,i=0;
       i<saved_count;
       ++x, ++y, ++i) {
    if (TooClose(*x,*y)) {
      return(0);
    }
  }
  return(1);
}

/*-----------------------------------------------------------------------
 * Function:	SaveNewPoint
 * Description:	add the new point to the saved array, and add its
 *		  index to the index array
 * Args:	(none)
 * Returns:	nothing
 * Author:	mbp
 * Date:	Thu May  3 12:09:53 1990
 * Notes:	The new point is in the global doubles xnew,ynew, and
 *		its index is the global int inew.
 */
SaveNewPoint()
{
  if (saved_count % SAVEBLOCKSIZE == 0)
    ExpandSaveBlocks();

  xsaved[saved_count] = xnew;
  ysaved[saved_count] = ynew;
  isaved[saved_count] = inew;
  ++saved_count;
}

/*-----------------------------------------------------------------------
 * Function:	ExpandSaveBlocks
 * Description:	Expand the size of the saved arrays to make room for
 *		  more points.  When called the first time, initializes
 *		  these arrays.
 * Args:	(none)
 * Returns:	nothing
 * Author:	mbp
 * Date:	Thu May  3 12:20:31 1990
 */
ExpandSaveBlocks()
{
  static int blockcount=0;

  ++blockcount;
  if (blockcount == 1) {
    xsaved = (double*)malloc(SAVEBLOCKSIZE*sizeof(double));
    ysaved = (double*)malloc(SAVEBLOCKSIZE*sizeof(double));
    isaved = (int*)malloc(SAVEBLOCKSIZE*sizeof(int));
  }
  else {
    xsaved =
      (double*)realloc(xsaved, blockcount * SAVEBLOCKSIZE*sizeof(double));
    ysaved =
      (double*)realloc(ysaved, blockcount * SAVEBLOCKSIZE*sizeof(double));
    isaved =
      (int*)realloc(isaved, blockcount * SAVEBLOCKSIZE*sizeof(int));
  }
}

PrintOutput()
{
  int i;

  printf("{");
  if (saved_count > 0)
    printf("%1d", isaved[0]);
  for (i=1; i<saved_count; ++i)
    printf(", %1d", isaved[i]);
  printf("}\n");
}

TooClose(x,y)
     register double x,y;
{
  register double tmp = x*xnew + y*ynew - 1.0;
  return( (tmp*tmp) < (factor * (x*x + y*y - 1.0)) );
}

