/*
 * panel.c: control panel module
 */

/***************************************************************************
 *                Copyright (C) 1990 by Mark B. Phillips                   *
 *                                                                         *
 *  Permission to use, copy, modify, and distribute this software, its     *
 *  documentation, and any images it generates for any purpose and without *
 *  fee is hereby granted, provided that                                   *
 *                                                                         *
 *  (1) the above copyright notice appear in all copies and that both      *
 *      that copyright notice and this permission notice appear in         *
 *      supporting documentation, and that the names of Mark B.            *
 *      Phillips, or the University of Maryland not be used in             *
 *      advertising or publicity pertaining to distribution of the         *
 *      software without specific, written prior permission.               *
 *                                                                         *
 *  (2) Explicit written credit be given to the author Mark B. Phillips    *
 *      in any publication which uses part or all of any image produced    *
 *      by this software.                                                  *
 *                                                                         *
 * This software is provided "as is" without express or implied warranty.  *
 ***************************************************************************/

#include <math.h>
#include <stdio.h>
#include "../lgd.h"
#include "panel.h"
#include "gr.h"
#include "dpu.h"
#include GLGHEADER

static int xorg, yorg, xsize, ysize;
#define	XMIN	 0.0
#define	XMAX	10.0
#define	YMIN	 0.0
#define	YMAX	10.0

static int panel_initialized = 0;

/* Gadget callback procedures: */
static int
  RampLoOffsetProc(),
  RampHiOffsetProc(),
  NearZOffsetProc(),
  FarZOffsetProc(),
  NearClippingProc(),
  FarClippingProc(),
  ViewMessageProc();

/*
 * Gadgets:
 */

GLGSLIDER(RampLoOffsetSlider,	/* slider symbol */
	  RampLoOffsetGadget,	/* gadget symbol */
	  S_INT,		/* type int */
	  "Ramp Lo Offset: %3d",/* label string */
	  RampLoOffsetProc,	/* callback proc */
	  7.5,2.533,		/* slider center */
	  2.0,0.35,		/* slider size */
	  0,10,5,		/* min, max, current value of slider */
	  0, 0, 0, 0, 0		/* colors (get set at run-time) */
	  );

GLGSLIDER(RampHiOffsetSlider,	/* slider symbol */
	  RampHiOffsetGadget,	/* gadget symbol */
	  S_INT,		/* type int */
	  "Ramp Hi Offset: %3d",/* label string */
	  RampHiOffsetProc,	/* callback proc */
	  2.537,2.533,		/* slider center */
	  2.0,0.35,		/* slider size */
	  0,10,5,		/* min, max, current value of slider */
	  0, 0, 0, 0, 0		/* colors (get set at run-time) */
	  );

GLGSLIDER(NearZOffsetSlider,	/* slider symbol */
	  NearZOffsetGadget,	/* gadget symbol */
	  S_INT,		/* type int */
	  "Near Z Offset: %4d", /* label string */
	  NearZOffsetProc,	/* callback proc */
	  2.537,1.0,		/* slider center */
	  2.0,0.35,		/* slider size */
	  LO_SCREEN_COORD_INT,	/* min,	*/
	  HI_SCREEN_COORD_INT,	/* max, */
	  NEARZOFFSETDEFAULT,	/* and current value of slider */
	  0, 0, 0, 0, 0		/* colors (get set at run-time) */
	  );

GLGSLIDER(FarZOffsetSlider,	/* slider symbol */
	  FarZOffsetGadget,	/* gadget symbol */
	  S_INT,		/* type int */
	  "Far Z Offset: %4d", /* label string */
	  FarZOffsetProc,	/* callback proc */
	  7.5,1.0,		/* slider center */
	  2.0,0.35,		/* slider size */
	  LO_SCREEN_COORD_INT, 	/* min,	*/
	  HI_SCREEN_COORD_INT, 	/* max, */
	  FARZOFFSETDEFAULT,	/* and current value of slider */
	  0, 0, 0, 0, 0		/* colors (get set at run-time) */
	  );

/* was s_h1: */
GLGSLIDER(FarClippingSlider,	/* slider symbol */
	  FarClippingGadget,	/* gadget symbol */
	  S_DOUBLE,		/* type int */
	  "Far Clip-Plane: %7.3f",/* label string */
	  FarClippingProc,	/* callback proc */
	  7.5,4.2,		/* slider center */
	  2.0,0.35,		/* slider size */
	  0,10,5,		/* min, max, current value of slider */
	  0, 0, 0, 0, 0		/* colors (get set at run-time) */
	  );

/* was s_h2: */
GLGSLIDER(NearClippingSlider,	/* slider symbol */
	  NearClippingGadget,	/* gadget symbol */
	  S_DOUBLE,		/* type int */
	  "Near Clip-Plane: %7.3f",/* label string */
	  NearClippingProc,	/* callback proc */
	  2.537,4.2,		/* slider center */
	  2.0,0.35,		/* slider size */
	  0,10,5,		/* min, max, current value of slider */
	  0, 0, 0, 0, 0		/* colors (get set at run-time) */
	  );


#define GRMSG_STRING_LEN 80
char grmsg_string[81];
GLGMESSAGE(GrStringOutMessage,	/* message symbol */
	   GrStringOutGadget,	/* gadget symbol */
	   grmsg_string,	/* message string */
	   NULL,		/* no callback proc */
	   0.25, 9.0,		/* (x,y) coords of message */
	   0, 0			/* colors (set later) */
	   );

static char grtxt_string[41];
GLGTEXT(GrStringInText,		/* text symbol */
	GrStringInGadget,	/* gadget symbol */
	grtxt_string,		/* place to store text value string */
	sizeof(grtxt_string),	/* declared length of text value string */
	NULL,			/* callback procedure */
	0.25, 8.0,		/* (x,y) coords of text */
	0, 0, 0			/* colors */
	);

GLGMESSAGE(GrRotateMessage,	/* message symbol */
	   GrRotateGadget,	/* gadget symbol */
	   "r: Rotate",		/* message string */
	   ViewMessageProc,	/* callback proc */
	   0.3, 7.0,		/* (x,y) coords of message */
	   0, 0			/* colors (set later) */
	   );

GLGMESSAGE(GrZoomMessage,	/* message symbol */
	   GrZoomGadget,	/* gadget symbol */
	   "z: Zoom",		/* message string */
	   ViewMessageProc,	/* callback proc */
	   2.8, 6.0,		/* (x,y) coords of message */
	   0, 0			/* colors (set later) */
	   );

GLGMESSAGE(GrTranslateMessage,	/* message symbol */
	   GrTranslateGadget,	/* gadget symbol */
	   "t: Translate",	/* message string */
	   ViewMessageProc,	/* callback proc */
	   0.3, 6.0,		/* (x,y) coords of message */
	   0, 0			/* colors (set later) */
	   );

GLGMESSAGE(GrScaleMessage,	/* message symbol */
	   GrScaleGadget,	/* gadget symbol */
	   "s: Scale",		/* message string */
	   ViewMessageProc,	/* callback proc */
	   2.8, 7.0,		/* (x,y) coords of message */
	   0, 0			/* colors (set later) */
	   );

GLGMESSAGE(GrFovMessage,	/* message symbol */
	   GrFovGadget,		/* gadget symbol */
	   "f: Field of View",		/* message string */
	   ViewMessageProc,	/* callback proc */
	   5.0, 7.0,		/* (x,y) coords of message */
	   0, 0			/* colors (set later) */
	   );

GLGMESSAGE(GrTwistMessage,	/* message symbol */
	   GrTwistGadget,	/* gadget symbol */
	   "w: Twist",		/* message string */
	   ViewMessageProc,	/* callback proc */
	   5.0, 6.0,		/* (x,y) coords of message */
	   0, 0			/* colors (set later) */
	   );

GLGMESSAGE(GrResetMessage,	/* message symbol */
	   GrResetGadget,	/* gadget symbol */
	   "ESC: Reset",	/* message string */
	   ViewMessageProc,	/* callback proc */
	   8.0, 6.0,		/* (x,y) coords of message */
	   0, 0			/* colors (set later) */
	   );

GLGMESSAGE(GrHaltMessage,	/* message symbol */
	   GrHaltGadget,	/* gadget symbol */
	   "h: Halt",		/* message string */
	   ViewMessageProc,	/* callback proc */
	   8.3, 7.0,		/* (x,y) coords of message */
	   0, 0			/* colors (set later) */
	   );

GLGMESSAGE(GrWriteMessage,	/* message symbol */
	   GrWriteGadget,	/* gadget symbol */
	   ">: Write Parameter File",	/* message string */
	   NULL,		/* no callback proc */
	   0.611, 5.033,	/* (x,y) coords of message */
	   0, 0			/* colors (set later) */
	   );

GLGMESSAGE(GrReadMessage,	/* message symbol */
	   GrReadGadget,	/* gadget symbol */
	   "<: Read Parameter File",	/* message string */
	   NULL,		/* no callback proc */
	   5.630, 4.967,	/* (x,y) coords of message */
	   0, 0			/* colors (set later) */
	   );

GLGMESSAGE(GrNTSCMessage,	/* message symbol */
	   GrNTSCGadget,	/* gadget symbol */
	   "n: NTSC",		/* message string */
	   NULL,		/* callback proc */
	   8.296, 8.067,	/* (x,y) coords of message */
	   0, 0			/* colors (set later) */
	   );

static GlGadget *gadgets[] = {
  &RampLoOffsetGadget,		/* 0 */
  &RampHiOffsetGadget,		/* 1 */
  &NearZOffsetGadget,		/* 2 */
  &FarZOffsetGadget,		/* 3 */
  &FarClippingGadget,		/* 4 */
  &NearClippingGadget,		/* 5 */
  &GrStringOutGadget,		/* 6 */
  &GrStringInGadget,		/* 7 */
  &GrRotateGadget,		/* 8 */
  &GrZoomGadget,		/* 9 */
  &GrTranslateGadget,		/* 10 */
  &GrScaleGadget,		/* 11 */
  &GrFovGadget,			/* 12 */
  &GrTwistGadget,		/* 13 */
  &GrResetGadget,		/* 14 */
  &GrHaltGadget,		/* 15 */
  &GrReadGadget,		/* 15 */
  &GrWriteGadget,		/* 15 */
  &GrNTSCGadget,		/* 16 */
  NULL				/* 17 */
  };

static int NTSCGadgetIndex = sizeof(gadgets)/sizeof(GlGadget *) - 2;

/* Panel colors: */
static short bgnd;		/* background */
static short barmark;		/* barmark */
static short inbar;		/* inbar */
static short outline;		/* outline */
static short txtcol;		/* text */
static short highlight;		/* highlight */

static Colorindex CInterp();

/*-----------------------------------------------------------------------
 * Function:	GrPanelInit
 * Description:	Initialize the control panel
 * Args:	(none)
 * Returns:	the gid of the control panel
 * Author:	mbp
 * Date:	Sun Jun  3 17:17:41 1990
 * Notes:	In addition to initializing, this procedure causes the
 *		control panel to appear on the screen.
 */
long
  GrPanelInit()
{
  long current_gid, panel_gid;
 
  int dim, i;
  double wbox_low[3],wbox_high[3];
  lgd_View3 view;

  if (panel_initialized) return(GrPanelGID);
  
  /* Set limits and current values of h1 and h2 sliders; these
     are based on the current world and view */
  FarClippingSlider.umin = NearClippingSlider.umin = 
    - ( FarClippingSlider.umax = NearClippingSlider.umax = 
       2*dpu_default_clipping_distance() );
  lgd_inquire_view( &view );
  FarClippingSlider.u = view.h1;
  NearClippingSlider.u = view.h2;
  
  /* Set colormap offset slider maxes and current values */
  RampLoOffsetSlider.umax = GrRampLen;
  RampLoOffsetSlider.u = RampLoOffset;
  RampHiOffsetSlider.umax = GrRampLen;
  RampHiOffsetSlider.u = RampHiOffset;
  
  current_gid = winget();

  prefsize(540, 300);
  panel_gid = winopen("LGDcontrol");

  winset(panel_gid);
  wintitle("LGD Control Panel");
  ortho2(XMIN, XMAX, YMIN, YMAX);
  doublebuffer();
  gconfig();
  frontbuffer(FALSE);
  GlgSetWindow(XMIN, XMAX, YMIN, YMAX);

  highlight = CInterp(.1, RampLo[GR_WHITE], RampHi[GR_WHITE]);
  bgnd = CInterp(.5, RampLo[GR_WHITE], RampHi[GR_WHITE]);
  barmark = CInterp(.8, RampLo[GR_WHITE], RampHi[GR_WHITE]);
  inbar = CInterp(.1, RampLo[GR_WHITE], RampHi[GR_WHITE]);
  outline = CInterp(1., RampLo[GR_WHITE], RampHi[GR_WHITE]);
  txtcol = CInterp(1., RampLo[GR_WHITE], RampHi[GR_WHITE]);

  RampLoOffsetSlider.bgnd	= bgnd;
  RampLoOffsetSlider.barmark	= barmark;
  RampLoOffsetSlider.inbar	= inbar;
  RampLoOffsetSlider.outline	= outline;
  RampLoOffsetSlider.txtcol	= txtcol;
  
  RampHiOffsetSlider.bgnd	= bgnd;
  RampHiOffsetSlider.barmark	= barmark;
  RampHiOffsetSlider.inbar	= inbar;
  RampHiOffsetSlider.outline	= outline;
  RampHiOffsetSlider.txtcol	= txtcol;

  FarZOffsetSlider.bgnd		= bgnd;
  FarZOffsetSlider.barmark	= barmark;
  FarZOffsetSlider.inbar	= inbar;
  FarZOffsetSlider.outline	= outline;
  FarZOffsetSlider.txtcol	= txtcol;
  
  NearZOffsetSlider.bgnd	= bgnd;
  NearZOffsetSlider.barmark	= barmark;
  NearZOffsetSlider.inbar	= inbar;
  NearZOffsetSlider.outline	= outline;
  NearZOffsetSlider.txtcol	= txtcol;
  
  FarClippingSlider.bgnd	= bgnd;
  FarClippingSlider.barmark	= barmark;
  FarClippingSlider.inbar	= inbar;
  FarClippingSlider.outline	= outline;
  FarClippingSlider.txtcol	= txtcol;
  
  NearClippingSlider.bgnd	= bgnd;
  NearClippingSlider.barmark	= barmark;
  NearClippingSlider.inbar	= inbar;
  NearClippingSlider.outline	= outline;
  NearClippingSlider.txtcol	= txtcol;
  
  GrStringOutMessage.bgnd	= bgnd;
  GrStringOutMessage.txtcol	= txtcol;
  
  GrStringInText.bgnd		= bgnd;
  GrStringInText.txtcol		= txtcol;
  GrStringInText.outline	= outline;

  GrRotateMessage.bgnd 		= bgnd;
  GrRotateMessage.txtcol	= txtcol;

  GrZoomMessage.bgnd		= bgnd;
  GrZoomMessage.txtcol		= txtcol;

  GrTranslateMessage.bgnd	= bgnd;
  GrTranslateMessage.txtcol	= txtcol;

  GrScaleMessage.bgnd		= bgnd;
  GrScaleMessage.txtcol		= txtcol;

  GrFovMessage.bgnd		= bgnd;
  GrFovMessage.txtcol		= txtcol;

  GrTwistMessage.bgnd		= bgnd;
  GrTwistMessage.txtcol		= txtcol;

  GrResetMessage.bgnd		= bgnd;
  GrResetMessage.txtcol		= txtcol;

  GrHaltMessage.bgnd		= bgnd;
  GrHaltMessage.txtcol		= txtcol;

  GrReadMessage.bgnd		= bgnd;
  GrReadMessage.txtcol		= txtcol;

  GrWriteMessage.bgnd		= bgnd;
  GrWriteMessage.txtcol		= txtcol;

  if (GrHasNTSC())  {
    GrNTSCMessage.bgnd		= bgnd;
    GrNTSCMessage.txtcol	= txtcol;
  }
  else
    gadgets[NTSCGadgetIndex] = NULL;

  GlgInit(gadgets, GrPanelDraw);

  winset(current_gid);

  panel_initialized = 1;

  return(panel_gid);
}

GrSetViewActionDisplay(action)
     int action;
{
  GrRotateMessage.bgnd 		= bgnd;
  GrZoomMessage.bgnd		= bgnd;
  GrTranslateMessage.bgnd	= bgnd;
  GrScaleMessage.bgnd		= bgnd;
  GrFovMessage.bgnd		= bgnd;
  GrTwistMessage.bgnd		= bgnd;
  GrResetMessage.bgnd		= bgnd;
  GrHaltMessage.bgnd		= bgnd;
  switch (action) {
  case ROTATEVIEW:
    GrRotateMessage.bgnd = highlight;
    break;
  case ZOOMVIEW:
    GrZoomMessage.bgnd = highlight;
    break;
  case TRANSLATEVIEW:
    GrTranslateMessage.bgnd = highlight;
    break;
  case SCALEVIEW:
    GrScaleMessage.bgnd = highlight;
    break;
  case FOVVIEW:
    GrFovMessage.bgnd = highlight;
    break;
  case TWISTVIEW:
    GrTwistMessage.bgnd = highlight;
    break;
  case RESETVIEW:
    GrResetMessage.bgnd = highlight;
    break;
  case HALTVIEW:
    GrHaltMessage.bgnd = highlight;
    break;
  }
  GrPanelDraw();
}


/*-----------------------------------------------------------------------
 * Function:	GrPanelDraw
 * Description:	Draw the panel
 * Args:	(none)
 * Returns:	nothing
 * Author:	mbp
 * Date:	Sun Jun  3 17:21:22 1990
 * Notes:	erases the panel before drawing it
 */
int
  GrPanelDraw()
{
  long current_gid = winget();

  winset(GrPanelGID);

  reshapeviewport();
  getorigin(&xorg,&yorg);
  getsize(&xsize,&ysize);
  GlgSetViewport(xorg, yorg, xsize, ysize);
  frontbuffer(FALSE);
  color(bgnd);
  clear();
  GlgDrawAll();
  swapbuffers();
  winset(current_gid);
}

/*-----------------------------------------------------------------------
 * Function:	GrPanelProcessEvent
 * Description:	process an event that happened in the panel
 * Args  IN:	event: the event
 * Returns:	nothing
 * Author:	mbp
 * Date:	Sun Jun  3 17:24:14 1990
 */
int
  GrPanelProcessEvent(event)
GlgEvent event;
{
  long current_gid = winget();

  winset(GrPanelGID);
  if (!GlgProcessEvent(event) && (event.dev == KEYBD))
    GrProcessKey((int)event.val);
  winset(current_gid);
}

/*-----------------------------------------------------------------------
 * Function:	GrPanelSetHValues
 * Description:	Update the display of the 'h' sliders so that they
 *		  reflect new values
 * Args  IN:	h1,h2: the new values
 * Returns:	nothing
 * Author:	mbp
 * Date:	Sun Jun  3 17:24:52 1990
 * Notes:	This procedure may never be needed --- it's just in
 *		case the viewing transformation passed to dpu_set_view
 *		ever explicitly changes the h-values.
 */
int
  GrPanelSetHValues(h1, h2)
double h1, h2;
{
  FarClippingSlider.u = h1;
  NearClippingSlider.u = h2;
  if (panel_initialized)
    GrPanelDraw();
}

/*-----------------------------------------------------------------------
 * Function:	GrPanelSetMessage
 * Description:	Change the message displayed in the panel
 * Args  IN:	s: the new message string
 * Returns:	nothing
 * Author:	mbp
 * Date:	Mon Jun  4 01:22:42 1990
 * Notes:	s = NULL or "" means clear the message
 */
int
  GrPanelSetMessage(s)
char *s;
{
  if (s == NULL) s = "";
  strncpy(grmsg_string, s, GRMSG_STRING_LEN);
  grmsg_string[GRMSG_STRING_LEN] = '\0';
  if (panel_initialized)
    GrPanelDraw();
}

int
  GrPanelGetString(s)
char *s;
{
  int done = 0;
  GlgEvent event;

  while (!done) {

    if (qtest()) {

      GlgReadEvent(&event);

      switch (event.dev) {
      case REDRAW:
	if (event.val == GrPanelGID)
	  GrPanelDraw();
	else if (event.val == GrGraphicsGID)
	  GrRedraw();
	break;

      case INPUTCHANGE:
	GrCurrentGID = event.val;
	break;

      default:
	if (GrCurrentGID == GrPanelGID) {
	  if ( (GlgProcessEvent(event) == &GrStringInGadget) &&
	      (event.dev==KEYBD) &&
	      ( (event.val==GlgCTRL_J) || (event.val==GlgRET) ) )
	    done = 1;
	}
	break;
      }
    }
  }
  strcpy(s, GrStringInText.string);
}

/*-----------------------------------------------------------------------
 * Function:	CInterp
 * Description:	Interpolate a color index
 * Args  IN:	x: the interpolation distance (between 0.0 and 1.0)
 *		c1: low index (corresponds to x=0.0)
 *		c2: high index (corresponds to x=1.0)
 * Returns:	the interpolated index
 * Author:	mbp
 * Date:	Sun Jun  3 17:26:57 1990
 */
static Colorindex
  CInterp(x, c1, c2)
double x;
Colorindex c1,c2;
{
  return( (Colorindex) (c1 + x * (c2 - c1)) );
}

/*-----------------------------------------------------------------------
 * Function:	RampLoOffsetProc
 * Description:	callback proc for colormap lo offset slider
 * Args  IN:	n: new value
 * Returns:	nothing
 * Author:	mbp
 * Date:	Sun Jun  3 17:28:02 1990
 * Notes:	resets the lo ramp offset and redraws the graphics
 */
static int
  RampLoOffsetProc(n)
int n;
{
  RampLoOffset = n;
  GrRedraw();
}

/*-----------------------------------------------------------------------
 * Function:	RampHiOffsetProc
 * Description:	callback proc for colormap hi offset slider
 * Args  IN:	n: new value
 * Returns:	nothing
 * Author:	mbp
 * Date:	Sun Jun  3 17:28:02 1990
 * Notes:	resets the hi ramp offset and redraws the graphics
 */
static int
  RampHiOffsetProc(n)
int n;
{
  RampHiOffset = n;
  GrRedraw();
}

/*-----------------------------------------------------------------------
 * Function:	FarClippingProc
 * Description:	callback procedure for h1 slider
 * Args  IN:	val: new value
 * Returns:	nothing
 * Author:	mbp
 * Date:	Sun Jun  3 17:29:23 1990
 * Notes:	Resets current h1 and redraws the graphics
 */
static int
  FarClippingProc(val)
double val;
{
  lgd_View3 view;

  lgd_inquire_view( &view );
  view.h1 = val;
  lgd_set_view( &view );
  GrRedraw();
}

/*-----------------------------------------------------------------------
 * Function:	NearClippingProc
 * Description:	callback procedure for h2 slider
 * Args  IN:	val: new value
 * Returns:	nothing
 * Author:	mbp
 * Date:	Sun Jun  3 17:29:23 1990
 * Notes:	Resets current h2 and redraws the graphics
 */
static int
  NearClippingProc(val)
double val;
{
  lgd_View3 view;

  lgd_inquire_view( &view );
  view.h2 = val;
  lgd_set_view( &view );
  GrRedraw();
}

static int
  ViewMessageProc(mp)
GlgMessage *mp;
{
  if (mp == &GrRotateMessage)
    SetViewAction(ROTATEVIEW);
  else if (mp == &GrZoomMessage)
    SetViewAction(ZOOMVIEW);
  else if (mp == &GrTranslateMessage)
    SetViewAction(TRANSLATEVIEW);
  else if (mp == &GrScaleMessage)
    SetViewAction(SCALEVIEW);
  else if (mp == &GrFovMessage)
    SetViewAction(FOVVIEW);
  else if (mp == &GrTwistMessage)
    SetViewAction(TWISTVIEW);
  else if (mp == &GrResetMessage)
    SetViewAction(RESETVIEW);
  else if (mp == &GrHaltMessage)
    SetViewAction(HALTVIEW);
}


static int
  NearZOffsetProc(n)
int n;
{
  GrNearZOffset = n;
  GrRedraw();
}

static int
  FarZOffsetProc(n)
int n;
{
  GrFarZOffset = n;
  GrRedraw();
}

int
  GrPanelWriteFile(file)
char *file;
{
  FILE *fp;
  lgd_View3 view;

  fp = fopen(file, "w");
  if (fp == NULL) return(0);

  fprintf(fp, "%1d GrNearZOffset\n", GrNearZOffset);
  fprintf(fp, "%1d GrFarZOffset\n", GrFarZOffset);

  fprintf(fp, "%1d RampLoOffset\n", (int)RampLoOffset);
  fprintf(fp, "%1d RampHiOffset\n", (int)RampHiOffset);

  lgd_inquire_view( &view );

  fprintf(fp, "%f current_view.h1 (Far Clipping Plane)\n", view.h1);
  fprintf(fp, "%f current_view.h2 (Near Clipping Plane)\n", view.h2);

  fclose(fp);

  return(1);
}

int
  GrPanelReadFile(file)
char *file;
{
  FILE *fp;
  lgd_View3 view;
#define BUFSIZE 100
  char buf[BUFSIZE+1];

  fp = fopen(file, "r");
  if (fp == NULL) return(0);

  if (fgets(buf, BUFSIZE, fp) == NULL) return(0);
  NearZOffsetSlider.u = GrNearZOffset = atoi(buf);

  if (fgets(buf, BUFSIZE, fp) == NULL) return(0);
  FarZOffsetSlider.u = GrFarZOffset = atoi(buf);

  if (fgets(buf, BUFSIZE, fp) == NULL) return(0);
  RampLoOffsetSlider.u = RampLoOffset = atoi(buf);

  if (fgets(buf, BUFSIZE, fp) == NULL) return(0);
  RampHiOffsetSlider.u = RampHiOffset = atoi(buf);

  lgd_inquire_view( &view );

  if (fgets(buf, BUFSIZE, fp) == NULL) return(0);
  view.h1 = FarClippingSlider.u = atof(buf);

  if (fgets(buf, BUFSIZE, fp) == NULL) return(0);
  view.h2 = NearClippingSlider.u = atof(buf);

  fclose(fp);

  GrPanelDraw();
  lgd_set_view( &view );

  return(1);
#undef BUFSIZE
}

