/*
    dpu.c:	Display Processing Unit Procedures


   $Id: dpu.c,v 1.13 90/06/07 01:28:41 mbp Exp Locker: mbp $

		       This file is part of LGD

   Author: Mark Phillips
           University of Maryland
           College Park, Maryland  20742
           mbp@lakisis.umd.edu

 */

/***************************************************************************
 *                Copyright (C) 1990 by Mark B. Phillips                   *
 *                                                                         *
 *  Permission to use, copy, modify, and distribute this software, its     *
 *  documentation, and any images it generates for any purpose and without *
 *  fee is hereby granted, provided that                                   *
 *                                                                         *
 *  (1) the above copyright notice appear in all copies and that both      *
 *      that copyright notice and this permission notice appear in         *
 *      supporting documentation, and that the names of Mark B.            *
 *      Phillips, or the University of Maryland not be used in             *
 *      advertising or publicity pertaining to distribution of the         *
 *      software without specific, written prior permission.               *
 *                                                                         *
 *  (2) Explicit written credit be given to the author Mark B. Phillips    *
 *      in any publication which uses part or all of any image produced    *
 *      by this software.                                                  *
 *                                                                         *
 * This software is provided "as is" without express or implied warranty.  *
 ***************************************************************************/

/*
   This module implements a "canonical" DPU.  Although the exact
   manner in which this is done will vary from one type of graphics
   device to another, the public procedures in this file should
   perform the same function regardless of the details of the
   implementation.  These procedures are:

   1.  dpu_initialize()
   2.  dpu_begin_segment(n)
   3.  dpu_end_segment()
   4.  dpu_delete_segment(n)
   5.  dpu_set_segment_visibility(n,s)
   6.  dpu_move(v)
   7.  dpu_draw(v)
   8.  dpu_point()
   9.  dpu_set_color(n)
   10. dpu_set_style(n)
   11. dpu_update_display()
   12. dpu_save(fp)
   13. dpu_load(fp)
   14. dpu_set_view(view)
   15. dpu_inquire_view(view)
   16. dpu_get_default_view()

   These procedures should be present in every implementation of this
   module, but only the ones corresponding to the actual capabilities
   of the particular graphics device should do anything--the rest
   should be noops.
*/

/*
   This version is for the IRIS 3000
*/

/****************************************************************/
/*		EXTERNAL DEFS:					*/

#include	<stdio.h>
#include	<math.h>
#include	<malloc.h>
#include	"dpu.h"
#include	"../internal.h"
#include	"../lgd.h"
#include	"../mf.h"
#include	"../vector.h"
#include	"gr.h"

double dpu_unit_ivec[3], dpu_unit_jvec[3], dpu_unit_kvec[3];
double dpu_unit_up[3];
double dpu_focal_dist, dpu_u1, dpu_u2, dpu_v1, dpu_v2;

/****************************************************************/
/*              EXTERNAL DEFS:					*/

char *dpu_error = NULL;

/****************************************************************/
/*		INTERNAL DEFS:					*/

static mfa last_executed;	/* address of entry in MF which	*/
				/* was most recently executed	*/

static lgd_View3 current_view;	/* current viewwing transf spec */

#define		UP	0
#define		DOWN	1

#define JACK 15 

static int pt_default[] = {
  0,	-JACK,	0,	0,
  1,	2*JACK,	0,	0,
  0,	-JACK,	0,	0,
  0,	0,	-JACK,	0,
  1,	0,	2*JACK,	0,
  0,	0,	-JACK,	0,
  0,	0,	0,	-JACK,
  1,	0,	0,	2*JACK,
  0,	0,	0,	-JACK
};
static int *pt = pt_default;
static pt_count = sizeof(pt_default) / (4 * sizeof(int));

#define		MAX(x,y)	( (x) > (y) ? (x) : (y) )

static int	dpu_pen_status	=	UP;

char *dpu_internal_error=NULL;

typedef double Matrix4x4[4][4];
/*
 * M1,M2,M3,M4 are the transposes of the matrices referred to in
 * my notes of 8/11/88; i.e. Mi here is (Mi)^t in my notes.
 *
 * M0 is the matrix corresponding to the transformation which used
 * to be performed by the call to world_coord.  This is so the
 * IRIS transformation matrix can operate directly on the contents
 * of the MetaFile.
 *
 * In the following initialization, entries which are initialized
 * to 2 are ones which get changed during execution.  The others
 * always retain their initialized value of 0 or 1.
 */
static Matrix M0 = { 2, 0, 0, 0,
                     0, 2, 0, 0,
                     0, 0, 2, 0,
                     2, 2, 2, 1 };
static Matrix M1 = { 1, 0, 0, 0,
                     0, 1, 0, 0,
                     0, 0, 1, 0,
                     2, 2, 2, 1 };
static Matrix M2 = { 2, 2, 2, 0,
                     2, 2, 2, 0,
                     2, 2, 2, 0,
                     0, 0, 0, 1 };
static Matrix M3 = { 1, 0, 0, 0,
                     0, 1, 0, 0,
                     0, 0, 1, 2,
                     0, 0, 0, 1 };
static Matrix M4 = { 2, 0, 0, 0,
                     0, 2, 0, 0,
                     0, 0, 2, 0,
                     2, 2, 2, 1 };

/* NOTE: these matrices used to be Matrix4x4, which is double.
 * They are now Matrix, defined in gl.c, using float.  But I think
 * float is the same as double anyway on the IRIS!! ?? */

/****************************************************************/
/*		EXTERNAL PROCEDURES:				*/

/*-----------------------------------------------------------------------
 * Function:     dpu_initialize
 * Description:  initialize the DPU
 * Arguments:    (none)
 * Returns:      nothing
 * Notes:        This procedure may be called more than once.  It
 *               should always have the effect of leaving the DPU
 *               (and the graphics device) in a cleared state, ready
 *               to start displaying a new picture.
 *
 * IRIS NOTES:  1. I don't know what will happen if this is called
 *                 more than once.  It may not work properly.
 */
dpu_initialize()
{
  char fname[256], *homedir, *getenv();

  last_executed = mf_first_addr() - 1;
  dpu_pen_status = UP;
  dev_initialize();
  if ((homedir=getenv("HOME")) == NULL) return;
  sprintf(fname, "%s/.lgdiris4point", homedir);
  dpu_read_point_file(fname);
  dpu_read_point_file(".lgdiris4point");
}

/*-----------------------------------------------------------------------
 * Function:     dpu_begin_segment
 * Description:  begin the definition of a segment
 * Arguments IN: n: the number of the segment to begin
 * Returns:      nothing
 * Notes:        This segment will consist of all move/draw/point calls
 *               between now and the next call to "dpu_end_segment".
 */
/*ARGSUSED*/
dpu_begin_segment( n )
mfe n;
{
  /* this version is a noop */
}

/*-----------------------------------------------------------------------
 * Function:     dpu_end_segment
 * Description:  end the definition of a segment
 * Arguments:    (none)
 * Returns:      nothing
 * Notes:        This ends the segment begin by the most recent call
 *               to "dpu_begin_segment"
 */
dpu_end_segment()
{
  /* this version is a noop */
}

/*-----------------------------------------------------------------------
 * Function:     dpu_delete_segment
 * Description:  delete (erase) a segment
 * Arguments IN: n: the number of the segment to be deleted
 * Returns:      nothing
 * Notes:        n had better not be the current segment!
 */
/*ARGSUSED*/
dpu_delete_segment( n )
mfe n;
{
  /* noop */
}

/*-----------------------------------------------------------------------
 * Function:     dpu_set_segment_visibility
 * Description:  Set the visibility of a segment
 * Arguments IN: n: the number of the segment
 *               s: visibility status to set to: 1=visible, 0=invisible
 * Returns:      nothing
 * Notes:        
 */
/*ARGSUSED*/
dpu_set_segment_visibility( n, s )
mfe n; 
int s;
{
  /* noop */
}

/*-----------------------------------------------------------------------
 * Function:     dpu_move
 * Description:  move current point without drawing
 * Arguments IN: v: point to move to
 * Returns:      nothing
 */
/*ARGSUSED*/
dpu_move( v )
double v[];
{
  /* noop */
}

/*-----------------------------------------------------------------------
 * Function:     dpu_draw
 * Description:  move current point, drawing a line along the way
 * Arguments IN: v: point to move to
 * Returns:      nothing
 */
/*ARGSUSED*/
dpu_draw( v )
double v[];
{
  /* noop */
}

/*-----------------------------------------------------------------------
 * Function:     dpu_point
 * Description:  draw a point at the current point
 * Arguments:    (none)
 * Returns:      nothing
 */
/*ARGSUSED*/
dpu_point()
{
  /* noop */
}

/*-----------------------------------------------------------------------
 * Function:     dpu_set_color
 * Description:  set the color to be used by draws and points
 * Arguments IN: n: the color to use
 * Returns:      nothing
 */
/*ARGSUSED*/
dpu_set_color( n )
mfe n;
{
  /* noop */
}

/*-----------------------------------------------------------------------
 * Function:     dpu_set_style
 * Description:  set the line style to be used by draws
 * Arguments IN: n: the style to use
 * Returns:      nothing
 */
/*ARGSUSED*/
dpu_set_style( n )
mfe n;
{
  /* noop */
}

/*-----------------------------------------------------------------------
 * Function:     dpu_update_display
 * Description:  update the display -- make sure it shows the current MF
 * Arguments:    (none)
 * Returns:      nothing
 */
dpu_update_display()
{
  compute_tmat();
  redraw_mf();
  if (dpu_internal_error!=NULL) {
    dpu_error = dpu_internal_error;
    dpu_internal_error = NULL;
    return;
  }
}

/*-----------------------------------------------------------------------
 * Function:     dpu_save
 * Description:  Write current dpu state to a file
 * Arguments IN: fp: file to write to
 * Returns:      nothing
 * Notes:        fp should point to a file which has been opened
 *               for binary writing.
 */
dpu_save( fp )
     FILE *fp;
{
  int items_written;

#define	SAVEIT(x,type)	{items_written = fwrite( \
		(char*)x,sizeof(type), 1, fp );\
		if (items_written!=1) \
		{dpu_error=E_svd_bad_write; return; } }
  /*
   *********************
   * THIS IS FORMAT 2  *
   *********************
   */

  SAVEIT( &last_executed, mfa );
  SAVEIT( &current_view, lgd_View3 );
  SAVEIT( &dpu_pen_status, int );

#undef	SAVEIT
}


/*-----------------------------------------------------------------------
 * Function:     dpu_load
 * Description:  Read a dpu state from a file
 * Arguments IN: fp: file to read from
 * Returns:      nothing
 * Notes:        fp should point to a file which has been opened
 *               for binary reading.  The current dpu is re-initialized
 *               first.
 */
dpu_load( fp )
     FILE *fp;
{
  int items_read;

#define	LOADIT(x,type)	{items_read = fread( \
		(char*)x,sizeof(type), 1, fp );\
		if (items_read!=1) \
		{dpu_error=E_svd_bad_read; return; } }
  /*
   *********************
   * THIS IS FORMAT 1  *
   *********************
   */
  dpu_initialize();
  LOADIT( &last_executed, mfa );
  LOADIT( &current_view, lgd_View3 );
  LOADIT( &dpu_pen_status, int );
  compute_tmat();
  execute_mf( mf_first_addr(), last_executed );
  if (dpu_internal_error!=NULL) {
    dpu_error = dpu_internal_error;
    dpu_internal_error = NULL;
    return;
  }

#undef	LOADIT
}

/*-----------------------------------------------------------------------
 * Function:     dpu_set_view
 * Description:  set the current viewwing transformation
 * Arguments IN: *view: the viewwing specs
 * Returns:      nothing
 * Notes:        This may not actually cause the display to change;
 *               call dpu_update_display to make sure the display
 *               is current.
 */
dpu_set_view( view )
lgd_View3 *view;
{
  int hchanged = 0;

  LGD_copy_vec( current_view.eye, view->eye );
  LGD_copy_vec( current_view.focus,view->focus );
  LGD_copy_vec( current_view.up,view->up );
  dpu_focal_dist = LGD_L2dist_vec(current_view.eye, current_view.focus);
  current_view.u1 = view->u1;
  current_view.u2 = view->u2;
  current_view.v1 = view->v1;
  current_view.v2 = view->v2;

  if (current_view.h1 != view->h1) {
    current_view.h1 = view->h1;
    hchanged = 1;
  }
  if (current_view.h2 != view->h2) {
    current_view.h2 = view->h2;
    hchanged = 1;
  }
  if (hchanged)
    GrPanelSetHValues(view->h1, view->h2);
  if (dpu_internal_error!=NULL) {
    dpu_error = dpu_internal_error;
    dpu_internal_error = NULL;
  }
}

/*-----------------------------------------------------------------------
 * Function:     dpu_inquire_view
 * Description:  Get the current viewing transformation
 * Arguments OUT: *view: the viewwing transformation specs
 * Returns:      nothing
 */
dpu_inquire_view( view )
lgd_View3 *view;
{
  LGD_copy_vec( view->eye, current_view.eye );
  LGD_copy_vec( view->focus, current_view.focus );
  LGD_copy_vec( view->up, current_view.up );
  view->u1 = current_view.u1;
  view->u2 = current_view.u2;
  view->v1 = current_view.v1;
  view->v2 = current_view.v2;
  view->h1 = current_view.h1;
  view->h2 = current_view.h2;
}

/*-----------------------------------------------------------------------
 * Function:     dpu_get_default_view
 * Description:  Compute a standard default viewwing transformation
 * Arguments:    (none)
 * Returns:      nothing
 */
dpu_get_default_view( view )
lgd_View3 *view;
{
  int dimension, i;
  double wbox_low[MAXDIM], wbox_high[MAXDIM], d;
#define	WFACT	2.0

  lgd_inquire_world( &dimension, wbox_low, wbox_high );
  if (dimension==3) {
    /* Set FOCUS to center of world		*/
    for (i=0; i<3; ++i)
      view->focus[i] =
	wbox_low[i] + (wbox_high[i]-wbox_low[i]) / 2;
    /* Set EYE on line parallel to z axis	*/
    view->eye[0] = view->focus[0];
    view->eye[1] = view->focus[1];
    view->eye[2] = view->focus[2] + 2*(wbox_high[2]-wbox_low[2]);
    /* Set UP on line parallel to y axis	*/
    view->up[0] = view->focus[0];
    view->up[1] = view->focus[1] + wbox_high[1];
    view->up[2] = view->focus[2];
    /* Make square window to just include	*/
    /* all of world box.			*/
    d = MAX( wbox_high[0]-wbox_low[0],
	    wbox_high[1]-wbox_low[1] ) / 2;
    view->u1 = view->v1 =
      -(view->u2 = view->v2 = (WFACT*4*d/3) );
    view->h1 = -( view->h2 = dpu_default_clipping_distance() );
    
  }
#undef	WFACT
}

/****************************************************************/
/*		INTERNAL PROCEDURES:				*/


/*-----------------------------------------------------------------------
 * Function:     execute_mf
 * Description:  Execute a section of the MetaFile
 * Arguments IN: first: first address to execute
 *               last: last address to execute
 * Returns:      nothing
 */
static execute_mf( first, last )
     register mfa first, last;
{
  register mfe entry;
  register int *pp, *pplast;
  
  while (first<=last) {
    entry=mf_peek(first);
    if (mf_is_command(entry)) {
      if (entry==MF_begin_segment) {
	/* Check visibility:	*/
	if (mf_peek(first+VIS_OS)) {	/* (visible)	 */
	  first += lgd_header_length;
	}
	else			        /* (invisible) */
	  first += mf_peek(first+LEN_OS);
	continue;
      }
      if (entry==MF_end_segment) {	/* skip trailer	*/
	first += lgd_trailer_length;
	continue;
      }
      if (entry==MF_pen_up) {	/* raise pen	*/
	dpu_pen_status = UP;
	++first;
	continue;
      }
      if (entry==MF_pen_down) {
	/* lower pen	*/
	dpu_pen_status = DOWN;
	++first;
	continue;
      }
      if (entry==MF_point) {	/* draw a point	*/
	/* (we actually draw a little jack) */ 
	pplast = pt + 4*pt_count;
	for (pp=pt; pp<pplast; pp+=4)
	  switch (*pp) {
	  case 0:		/* rel move */
	    rmvi(*(pp+1), *(pp+2), *(pp+3));
	    break;
	  case 1:		/* rel draw */
	    rdri(*(pp+1), *(pp+2), *(pp+3));
	    break;
	  }
	++first;
	continue;
      }
      if (entry==MF_line_style) {
	/* change style	*/
	/* noop */
	first += 2;
	continue;
      }
      if (entry==MF_color) {	/* change color	*/
	dev_color((int)mf_peek(first+1));
	first += 2;
	continue;
      }
      /* Invalid entry	*/
      dpu_internal_error = E_bad_mf_entry;
      return;
    }  /* End of "if (mf_is_command..."		*/
    
    else {		/* We have a vector		*/
      /* Send vector from MF to IRIS */      
      if (dpu_pen_status==DOWN) {
	drawi((Icoord)mf_peek(first),
	      (Icoord)mf_peek(first+1),
	      (Icoord)mf_peek(first+2));
      }
      else {
	movei((Icoord)mf_peek(first),
	      (Icoord)mf_peek(first+1),
	      (Icoord)mf_peek(first+2));
      }
      first += lgd_dim;
    }
  }  /* end of while loop 				*/
}

/*-----------------------------------------------------------------------
 * Function:     flush_mf
 * Description:  flush out (execute) unexecuted part of MF
 * Arguments:    (none)
 * Returns:      nothing
 * Notes:        The part executed is that between last_executed
 *               and the current end of the MF.
 */
static flush_mf()
{
  execute_mf( last_executed+1, mf_last_addr() );
  if (dpu_internal_error==NULL)
    last_executed = mf_last_addr();
}

/*-----------------------------------------------------------------------
 * Function:     redraw_mf
 * Description:  redraw (execute) the entire MF
 * Arguments:    (none)
 * Returns:      nothing
 * Note:         This proc was made external for the IRIS version
 */
redraw_mf()
{
  last_executed = mf_first_addr() - 1;
  flush_mf();
}

/*-----------------------------------------------------------------------
 * Function:     compute_tmat()
 * Description:  Compute the current viewing transformation matrix
 * Arguments:    (none)
 * Returns:      nothing
 * Notes:        This loads the new matrix into the IRIS's stack
 */
static compute_tmat()
{
  double du, dv, dh, two_over_du, two_over_dv, two_over_dh;
  int i;

  /* Step 1: set M4 and load it into top stack position.*/

  /* We compute a different ("new") view plane window, since IRIS
   * graphics window may not be square.  This maps the original view
   * plane window into the largest square contained in the graphics
   * window.
   *
   * Note: GrAspectRatio = width/height of IRIS graphics window */
  if (GrAspectRatio > 1.0) {
    /* Graphics window is wider than it is tall, so expand horizontal
       dimensions of view plane window. */
    dpu_u1 = (  (current_view.u1 - current_view.u2) * GrAspectRatio
	      + (current_view.u2 + current_view.u1)                 )/2;
    dpu_u2 = (  (current_view.u2 - current_view.u1) * GrAspectRatio
	      + (current_view.u2 + current_view.u1)                 )/2;
    dpu_v1 = current_view.v1;
    dpu_v2 = current_view.v2;
  }
  else {
    /* Graphics window is taller than it is wide, so expand vertical
       dimensions of view plane window. */
    dpu_v1 = (  (current_view.v1 - current_view.v2) / GrAspectRatio 
	      + (current_view.v2 + current_view.v1)                 )/2;
    dpu_v2 = (  (current_view.v2 - current_view.v1) / GrAspectRatio
	      + (current_view.v2 + current_view.v1)                 )/2;
    dpu_u1 = current_view.u1;
    dpu_u2 = current_view.u2;
  }
  du = dpu_u2 - dpu_u1;
  dv = dpu_v2 - dpu_v1;
  dh = current_view.h2 - current_view.h1;
  two_over_du = 2 / du;
  two_over_dv = 2 / dv;
  two_over_dh = 2 / dh;
  M4[0][0] = two_over_du;
  M4[1][1] = two_over_dv;
  M4[2][2] = two_over_dh;
  M4[3][0] = - ( two_over_du * dpu_u1 + 1 );
  M4[3][1] = - ( two_over_dv * dpu_v1 + 1 );
  M4[3][2] = - ( two_over_dh * current_view.h1 + 1 );
  /* printf("loading M4:\n");
     printmatrix("M4 = ",M4); printf("\n"); */
  loadmatrix(M4);

  /* Step 2: set M3 and premult the stack top by it */
/* The following line has been moved to dpu_set_view:
   dpu_focal_dist = LGD_L2dist_vec(current_view.eye, current_view.focus); */
  M3[2][3] = -1.0 / dpu_focal_dist;
  /* printf("multmatrix M3:\n");
     printmatrix("M3 = ",M3); printf("\n"); */
  multmatrix(M3);		/* (Now stack top is M3 * M4) */

  /* Step 3: set M2 and premult the stack top by it */
  LGD_sub_vec( dpu_unit_kvec, current_view.eye, current_view.focus );
  LGD_unit_vec( dpu_unit_kvec );
  LGD_sub_vec( dpu_unit_up, current_view.up, current_view.focus );
  LGD_unit_vec( dpu_unit_up);
  LGD_cross_vec(dpu_unit_ivec, dpu_unit_up, dpu_unit_kvec );
  LGD_unit_vec( dpu_unit_ivec );
  LGD_cross_vec( dpu_unit_jvec,	dpu_unit_kvec,	dpu_unit_ivec );
  for (i=0; i<3; ++i) M2[i][0] = dpu_unit_ivec[i];
  for (i=0; i<3; ++i) M2[i][1] = dpu_unit_jvec[i];
  for (i=0; i<3; ++i) M2[i][2] = dpu_unit_kvec[i];
  /* printf("multmatrix M2:\n");
     printmatrix("M2 = ",M2); printf("\n"); */
  multmatrix(M2);		/* (Now stack top is M2 * M3 * M4 */

  /* Step 4: set M1 and premult the stack top by it */
  for (i=0; i<3; ++i) M1[3][i] = - current_view.focus[i];
  /* printf("multmatrix M1:\n");
     printmatrix("M1 = ",M1); printf("\n"); */
  multmatrix(M1);

  /* Step 5: set M0 and premult the stack top by it */
  for (i=0; i<3; ++i) {
    M0[i][i] = 1.0 / lgd_wscale[i];
    M0[3][i] = lgd_wbox[0][i];
  }
  /* printf("multmatrix M0:\n");
     printmatrix("M0 = ",M0); printf("\n"); */
  multmatrix(M0);

  /* Now we are done; stack top is M0 * M1 * M2 * M3 * M4 */
}

dpu_pick_segment(s, n)
int **s, *n;
{
  *s = NULL;
  *n = 0;
}


double
  dpu_default_clipping_distance()
{
  int dim, i;
  double wbox_low[3],wbox_high[3], d, hmaxsq;

  lgd_inquire_world(&dim, wbox_low, wbox_high);
  hmaxsq = 0;
  for (i=0; i<3; ++i) {
    d = wbox_high[i] - wbox_low[i];
    hmaxsq += d*d;
  }
  return(sqrt(hmaxsq));
}


int
  dpu_read_point_file(file)
char *file;
{
  FILE *fp;
#define BUFSIZE 80
#define MAXPOINTS 100
  char buf[BUFSIZE+1];
  int n, *newpt=(int*)malloc(4*MAXPOINTS*sizeof(int));

  if (newpt == NULL) return(0);
  if ( (fp = fopen(file, "r")) == NULL ) return(0);
  n = 0;
  while ((n<MAXPOINTS) && (fgets(buf, BUFSIZE, fp) != NULL))
    if (sscanf(buf, "%d %d %d %d", 
	       newpt+4*n+0, newpt+4*n+1, newpt+4*n+2, newpt+4*n+3) == 4)
      ++n;
  fclose(fp);

  realloc(newpt, 4*n*sizeof(int));
  pt = newpt;
  pt_count = n;
/*
  printf("new point data:\n");
  printpointdata();
*/
  return(1);

#undef BUFSIZE
}

printpointdata()
{
  int *pp, *pplast;
  
  pplast = pt + 4*pt_count;
  for (pp=pt; pp<pplast; pp+=4)
    switch (*pp) {
    case 0:		/* rel move */
      printrmvi(*(pp+1), *(pp+2), *(pp+3));
      break;
    case 1:		/* rel draw */
      printrdri(*(pp+1), *(pp+2), *(pp+3));
      break;
    }
}

printrmvi(x,y,z)
     int x,y,z;
{
  printf("\trmvi(%d, %d, %d)\n", x,y,z);
}


printrdri(x,y,z)
     int x,y,z;
{
  printf("\trdri(%d, %d, %d)\n", x,y,z);
}
